// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

#include <Common/Base/Types/Geometry/Aabb/hkAabbHalf.h> // needed for Codec18
#include <Common/Base/Types/Geometry/Aabb16/hkAabb16.h> // needed for CodecInt16

namespace hkcdDynamicTree
{
    ///
    /// User size indices, full precision AABB, 32 + sizeof(Index) * 3 + padding.
    ///
    template <typename INDEX_TYPE>
    struct CodecRaw
    {
        HK_DECLARE_CLASS(CodecRaw, New, Reflect, Pod);

        typedef hkAabb      PackAabb;   ///< Type of the AABB stored in the node.
        typedef hkAabb      RawAabb;    ///< Type of the AABB used for processing.
        typedef INDEX_TYPE Index;       ///< Index.

        CodecRaw() {}

        HK_INLINE void      getAabb(hkAabb& aabb) const             { aabb = m_aabb; }
        HK_INLINE void      setAabb(const hkAabb& aabb)             { m_aabb = aabb; }

        HK_INLINE Index     getChild(int index) const               { return m_children[index]; }
        HK_INLINE void      setChild(int index, Index value)        { m_children[index]=value; }

        HK_INLINE Index     getData() const                         { return getChild(1); }

        template <typename T>
        HK_INLINE void      setData(T value)                        { HK_COMPILE_TIME_ASSERT(sizeof(T)==sizeof(Index)); setChild(1,*(const Index*)&value); }

        template <typename T>
        HK_INLINE const T&  typedData() const                       { HK_COMPILE_TIME_ASSERT(sizeof(T)==sizeof(Index)); return *(const T*)&m_children[1]; }

        template <typename T>
        HK_INLINE T&        typedData()                             { HK_COMPILE_TIME_ASSERT(sizeof(T)==sizeof(Index)); return *(T*)&m_children[1]; }

        HK_INLINE Index     getParent() const                       { return m_parent; }
        HK_INLINE void      setParent(Index parent)                 { m_parent=parent; }

        HK_INLINE hkBool32  isInternal() const                      { return (hkBool32)m_children[0]; }
        HK_INLINE void      merge(const hkAabb& x,const hkAabb& y)  { m_aabb.setUnion(x,y); }
        HK_INLINE void      include(const hkAabb& x)                { m_aabb.includeAabb(x); }

    protected:

        hkAabb  m_aabb;         ///< AABB of the node.
        Index   m_parent;       ///< Parent node or next free node if this node is not allocated.
        Index   m_children[2];  ///< Children.
    };

    ///
    /// 16 bits indices, full precision, node size 32 bytes.
    ///
    struct HK_EXPORT_COMMON Codec32     
    {
        HK_DECLARE_CLASS(Codec32, New, Reflect, Pod);

        typedef hkAabb      PackAabb;   ///< Type of the AABB stored in the node.
        typedef hkAabb      RawAabb;    ///< Type of the AABB used for processing.
        typedef hkUint16    Index;      ///< Index.

        Codec32() {}

        HK_INLINE void      getAabb(hkAabb& aabb) const         { aabb.m_max.setXYZ_0(m_aabb.m_max); aabb.m_min = m_aabb.m_min;  }
#if defined(HK_PLATFORM_WIIU) && defined(HK_ENABLE_PAIRED_SINGLE_OPTS)
        HK_INLINE void      setAabb(const hkAabb& aabb)
        {
            // COM-2545. m_aabb.m_max(3) will surely be a denormal as it is a combination of two hkUint16. On WiiU
            // however, if a denormal is in the second element of a PairedSingleFloat, it will get copied to 0 which
            // is what can occur by calling 'hkVector4::setXYZ()'. Having a denormal in an hkVector4 is a specific
            // case so we instead override the behavior here, directly.
            m_aabb.m_min.setXYZ(aabb.m_min);
            m_aabb.m_max.m_quad.m_a = aabb.m_max.m_quad.m_a;
            m_aabb.m_max.m_quad.m_b[0] = aabb.m_max.m_quad.m_b[0];
        }
#else
        HK_INLINE void      setAabb(const hkAabb& aabb)         { m_aabb.m_min.setXYZ(aabb.m_min); m_aabb.m_max.setXYZ(aabb.m_max); }
#endif

#if HK_ENDIAN_LITTLE
        HK_INLINE Index     getChild(int index) const               { return ((const hkUint16*)&m_aabb.m_max(3))[index]; }
        HK_INLINE void      setChild(int index, Index value)        { ((hkUint16*)&m_aabb.m_max(3))[index]=value; }
    #else
        HK_INLINE Index     getChild(int index) const               { return ((const hkUint16*)&m_aabb.m_max(3))[1-index]; }
        HK_INLINE void      setChild(int index, Index value)        { ((hkUint16*)&m_aabb.m_max(3))[1-index]=value; }
    #endif
        HK_INLINE Index     getData() const                         { return getChild(1); }
        HK_INLINE void      setData(Index value)                    { return setChild(1,value); }
        HK_INLINE Index     getParent() const                       { return (Index)m_aabb.m_min.getInt24W(); }
        HK_INLINE void      setParent(Index parent)                 { m_aabb.m_min.setInt24W(parent); }
        HK_INLINE hkBool32  isInternal() const                      { return (hkBool32)getChild(0); }
        HK_INLINE void      merge(const hkAabb& x,const hkAabb& y)  { hkAabb aabb; aabb.setUnion(x,y); setAabb(aabb); }
        HK_INLINE void      include(const hkAabb& x)                { hkAabb aabb; hkVector4 mx; mx.setXYZ_0(m_aabb.m_max); aabb.m_min.setMin(x.m_min, m_aabb.m_min); aabb.m_max.setMax(x.m_max, mx); setAabb(aabb); }

    protected:

        hkAabb  m_aabb;         ///< AABB of the node as well as: Parent in m_min:W(24) and children in m_max:W(16|16).
    };

    ///
    /// 16 bits indices, 7 bits mantissa AABB accuracy, node size 18 bytes.
    ///
    struct Codec18
    {
        HK_DECLARE_CLASS(Codec18, New, Reflect, Pod);

        typedef hkAabbHalf  PackAabb;   ///< Type of the AABB stored in the node.
        typedef hkAabb      RawAabb;    ///< Type of the AABB used for processing.
        typedef hkUint16    Index;      ///< Index.

        Codec18() {}

        HK_INLINE void      getAabb(hkAabb& aabb) const             { m_aabb.unpackUnaligned(aabb); }
        HK_INLINE void      setAabb(const hkAabb& aabb)             { m_aabb.pack(aabb); }
        HK_INLINE Index     getChild(int index) const               { return m_aabb.m_data[6+index]; }
        HK_INLINE void      setChild(int index, Index value)        { m_aabb.m_data[6+index]=value; }
        HK_INLINE Index     getData() const                         { return getChild(1); }
        HK_INLINE void      setData(Index value)                    { return setChild(1,value); }
        HK_INLINE Index     getParent() const                       { return m_parent; }
        HK_INLINE void      setParent(Index parent)                 { m_parent=parent; }
        HK_INLINE hkBool32  isInternal() const                      { return (hkBool32)getChild(0); }
        HK_INLINE void      merge(const hkAabb& x,const hkAabb& y)  { hkAabb aabb; aabb.m_min.setMin(x.m_min,y.m_min); aabb.m_max.setMax(x.m_max,y.m_max); setAabb(aabb); }
        HK_INLINE void      include(const hkAabb& x)                { hkAabb aabb; getAabb(aabb); merge(aabb,x); }

    protected:

        hkAabbHalf  m_aabb;     ///< AABB and children in m_extras.
        Index       m_parent;   ///< Parent.
    };

    typedef CodecRaw<hkUlong> CodecRawUlong;
    typedef CodecRaw<unsigned int> CodecRawUint;

    ///
    /// Integer codec
    ///
    HK_CLASSALIGN16(struct) CodecInt16
    {
        HK_DECLARE_CLASS(CodecInt16, New, Reflect, Pod);

        HK_CLASSALIGN16(struct) IntAabb : public hkAabb16
        {
            HK_DECLARE_CLASS(IntAabb, New, Reflect, Pod);
        };

        typedef IntAabb     PackAabb;   ///< Type of the AABB stored in the node.
        typedef IntAabb     RawAabb;    ///< Type of the AABB used for processing.
        typedef hkUint32    Index;      ///< Index.

        CodecInt16() {}

        HK_INLINE void      getAabb(IntAabb& aabb) const        { aabb.set(m_aabb); }
        HK_INLINE void      setAabb(const IntAabb& aabb)        { m_aabb.set(aabb); }

        HK_INLINE Index     getChild(int index) const               { return m_children[index]; }
        HK_INLINE void      setChild(int index, Index value)        { m_children[index]=value; }

        HK_INLINE Index     getData() const                         { return getChild(1); }

        template <typename T>
        HK_INLINE void      setData(T value)                        { HK_COMPILE_TIME_ASSERT(sizeof(T)==sizeof(Index)); setChild(1,*(const Index*)&value); }

        template <typename T>
        HK_INLINE const T&  typedData() const                       { HK_COMPILE_TIME_ASSERT(sizeof(T)==sizeof(Index)); return *(const T*)&m_children[1]; }

        template <typename T>
        HK_INLINE T&        typedData()                             { HK_COMPILE_TIME_ASSERT(sizeof(T)==sizeof(Index)); return *(T*)&m_children[1]; }

        HK_INLINE Index     getParent() const                       { return m_parent; }
        HK_INLINE void      setParent(Index parent)                 { m_parent=parent; }

        HK_INLINE hkBool32  isInternal() const                          { return (hkBool32)m_children[0]; }
        HK_INLINE void      merge(const IntAabb& x,const IntAabb& y)    { m_aabb.set(x); m_aabb.includeAabb(y); }
        HK_INLINE void      include(const IntAabb& x)                   { m_aabb.includeAabb(x); }

    protected:

        IntAabb     m_aabb;         ///< AABB of the node.
        Index       m_parent;       ///< Parent node or next free node if this node is not allocated.
        Index       m_children[2];  ///< Children.
        hkUint32    m_pad;          ///< Padding.
    };
}

#include <Geometry/Internal/_Auto/TemplateTypes/hkcdDynamicTreeCodecs_Types.inl>

HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::CodecRawUlong, hkcdDynamicTree_CodecRawUlong);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::CodecRawUint, hkcdDynamicTree_CodecRawUint);

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
