// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE
#include <Geometry/Internal/Algorithms/TreeQueries/hkcdAabbTreeQueries.h>

    /// A bounding volume tree that can be dynamically updated.
    /// When adding a leaf, a handle is returned; this must be tracked if the leaf can later be removed or updated.
    /// After all leaves have been added, the tree can be rebuilt to improve query performance.
class HK_EXPORT_COMMON hkcdDynamicAabbTree : public hkReferencedObject
{
public:
    HK_DECLARE_CLASS(hkcdDynamicAabbTree, New, Reflect, BypassCtor);
    HK_RECORD_ATTR(hk::Version(2));

    class Impl;
    typedef hkUint32 TreeHandle;

    hkcdDynamicAabbTree();
    hkcdDynamicAabbTree(const hkcdDynamicAabbTree& src);
    virtual ~hkcdDynamicAabbTree();

    void operator = (const hkcdDynamicAabbTree& src);

    void swap(hkcdDynamicAabbTree& b);

        /// Allocate memory for an additional number of leaves
    hkResult reserveLeaves( int numLeaves );

        /// Add a leaf with the specified AABB and key. The key will be used during query callbacks
    TreeHandle insert( const hkAabb& aabb, hkUint32 key );

        /// Add a leaf with the specified point and key. The key will be used during query callbacks
    TreeHandle insertPoint( hkVector4Parameter point, hkUint32 key );

        /// Add a leaf with the specified AABB and key. The hkResult will be set to HK_SUCCESS if all memory allocations succeed, or to HK_FAILURE otherwise.
    TreeHandle tryInsert( const hkAabb& aabb, hkUint32 key, hkResult& res );

        /// Add a leaf with the specified point and key. The hkResult will be set to HK_SUCCESS if all memory allocations succeed, or to HK_FAILURE otherwise.
    TreeHandle tryInsertPoint( hkVector4Parameter point, hkUint32 key, hkResult& res );

        // Sets the key associated with an existing node.
    void setNodeKey( TreeHandle handle, hkUint32 key );

        /// Change the AABB of the leaf.
    void update( TreeHandle handle, const hkAabb& aabb );

        /// Remove a leaf
    void remove( TreeHandle handle );

        /// Rebuild the tree to improve query performance.
    hkResult rebuild();

        /// Do a fast, in-place optimization on the tree to improve query performance.
    void rebuildFast( int numPasses );

        /// Incremental optimization of the tree. This is suitable to be called every frame if objects have moved.
    void optimizeIncremental(int passes, int lookahead = 2);

        /// Completely clears the tree.
    void clear();

        /// Returns the leaf data for the specified handle.
    hkUint32 getLeafData( TreeHandle handle );

        /// Cast a ray through the tree. The collector's callback is triggered for each leaf in the tree that is hit
    hkBool32 castRay(hkVector4Parameter start, hkVector4Parameter end, _Inout_ hkcdAabbTreeQueries::RaycastCollector* collector) const;

        /// Cast a ray through the tree. The collector's callback is triggered for each leaf in the tree that is hit
    hkBool32 castSphere( hkVector4Parameter start, hkVector4Parameter end, hkSimdRealParameter radius, _Inout_ hkcdAabbTreeQueries::RaycastCollector* collector) const;

        /// AABB check on the tree. The collector's callback is triggered for each leaf that overlaps with the AABB.
    void queryAabb( const hkAabb& aabb, _Inout_ hkcdAabbTreeQueries::AabbCollector* collector) const;

        /// Special-case AABB query where only the leaf IDs are saved.
        /// This is about 10% faster than the collector-based version (depending on platform).
    hkResult queryAabb( const hkAabb& aabb, hkArray<hkUint32>::Temp& hits ) const;

        /// Find the closest leaf in the tree. The collector's callback is triggered for each leaf that is closer to the current smallest distance, starting at maxDistance.
    hkUint32 getClosestPoint( hkVector4Parameter point, hkSimdRealParameter maxDistance, _Inout_ hkcdAabbTreeQueries::ClosestPointCollector* collector, hkVector4& closestPointOut ) const;

        /// Find the closest leaf in the tree. The distance to the leaf will be taken as the distance to the AABB.
    hkUint32 getClosestPoint( hkVector4Parameter point, hkSimdRealParameter maxDistance, hkVector4& closestPointOut ) const;

        /// Finds all pairs of overlapping leaves in this tree.
    void getAllPairs(_Inout_  hkcdAabbTreeQueries::AllPairsCollector* collector ) const;

        /// Finds all leaves in this tree that overlap with leaves in the other tree.
    void getAllPairs(_In_ const hkcdDynamicAabbTree* otherTree, _Inout_ hkcdAabbTreeQueries::AllPairsCollector* collector ) const;

        /// Returns the size (in bytes) of the tree.
    int getMemoryFootPrint() const;

        /// Returns the number of leaves in the tree.
    int getNumLeaves() const;

        /// Accessor for the internal tree.
    _Ret_notnull_ const void* getTreePtr() const;

        /// Accessor for the internal tree.
    _Ret_notnull_ void* getTreePtr();

protected:

    hkRefPtr<Impl> m_treePtr;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
