// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

/// IfProcessChildrenResult
enum IfProcessChildrenResult
{
    IPC_NONE,
    IPC_LEFT,
    IPC_RIGHT,
    IPC_BOTH
};

/// Inplace stack of value type, with fall-back value on pop in case of overflow.
template <typename T, int CAPACITY>
struct BoundedStack
{
    HK_INLINE           BoundedStack() : m_count(0) {}
    HK_INLINE           ~BoundedStack()         { HK_ASSERT(0x28A7D5C9, m_count==0, "Stack not empty"); }
    HK_INLINE void  push(T value)           { if(m_count < CAPACITY) m_values[m_count++]=value; else ++m_count; }
    HK_INLINE T     pop(T fallBackValue)    { HK_ASSERT(0x2897D5B9, m_count>0, "Stack underflow"); if(m_count-- <= CAPACITY) return m_values[m_count]; else return fallBackValue; }
    HK_INLINE int       getSize() const         { return m_count; }

private:

    T   m_values[CAPACITY];
    int m_count;
};

/// Pair of slots.
template <typename TA, typename TB>
struct SlotPair
{
    HK_INLINE   SlotPair() {}
    HK_INLINE   SlotPair(const typename TA::Slot& a, const typename TB::Slot& b) : m_nodeA(a),m_nodeB(b) {}

    typename TA::Slot   m_nodeA;
    typename TB::Slot   m_nodeB;
};

/// Unary stackless query decl.
template <bool VALID, typename TREE, typename QUERY> struct unaryStackless {};

/// Unary stackless query not supported impl.
template <typename TREE, typename QUERY> struct unaryStackless<false,TREE,QUERY>
{
    static HK_INLINE void   evaluate(const TREE&,const typename TREE::NodeContext&, QUERY&) {}
};

/// Unary stackless query supported impl.
template <typename TREE, typename QUERY> struct unaryStackless<true,TREE,QUERY>
{
    static inline void  evaluate(const TREE& tree, const typename TREE::NodeContext& rootContext, QUERY& query)
    {
        HK_COMPILE_TIME_ASSERT(TREE::SUPPORT_STACKLESS_TRAVERSAL != 0);
        typename TREE::NodeContext  context = rootContext;
        if(context->isInternal())
        {
            const typename TREE::Index base = context.m_index;
            context.m_index = context->getChild(0);
            do
            {
                context.updateFromIndex(tree);
                if ( query.template processNode<TREE, typename TREE::NodeContext>(context) )
                {
                    if ( context.m_node->isInternal() )
                    {
                        context.m_index = tree.nextLeftNode(context.m_index, base);
                        continue;
                    }
                    else
                    {
                        query.processLeaf(context);
                    }
                }
                context.m_index = tree.nextRightNode(context.m_index, base);
                IfPopNode<QUERY>::call(query, context);
            } while ( context.m_index );
        }
        else if(query.template processNode<TREE, typename TREE::NodeContext>(context))
        {
            query.processLeaf(context);
        }
    }
};

//
// Utilities
//

// Convert any non-zero value to 1 and zero to 0.
HK_INLINE static hkUint32   zeroOrOne(hkBool32 value) { return (((hkUint32)0)-value)>>31; }

//
// Template trickery to allow for default implementation of query methods.
//

// Compile time type selection, select A if index if 0 else B.
template <typename A, typename B, int INDEX> struct SelectType {};
template <typename A, typename B> struct SelectType<A,B,0> { typedef A TYPE; };
template <typename A, typename B> struct SelectType<A,B,1> { typedef B TYPE; };

// Compile time type equality check.
template <typename A, typename B>   struct SameType         { enum { VALUE = 0 }; };
template <typename A>               struct SameType<A,A>    { enum { VALUE = 1 }; };

// Call the query processChildren method if the query define enum { HAS_PROCESS_CHILDREN }; else call processNode on each child.
template <typename Q, typename TREE>
struct IfProcessChildren
{
    template <typename CONTEXT>
    static HK_INLINE IfProcessChildrenResult    call(Q& q, _In_ const CONTEXT* c) { return (IfProcessChildrenResult) SelectType<Default,Member,Exist>::TYPE::call(q,c); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_PROCESS_CHILDREN>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename CONTEXT> static HK_INLINE int   call(Q& q, _In_reads_(2) const CONTEXT* children) { return (q.template processNode<TREE,CONTEXT>(children[0])?1:0) | (q.template processNode<TREE,CONTEXT>(children[1])?2:0); } };
    struct Member { template <typename CONTEXT> static HK_INLINE int    call(Q& q, _In_reads_(2) const CONTEXT* children) { return q.template processChildren<TREE,CONTEXT>(children); } };
    };

// Call the query processSelf method if the query define enum { HAS_PROCESS_SELF }; else return 1(true).
template <typename Q>
struct IfProcessSelf
{
    template <typename CONTEXT>
    static HK_INLINE hkBool32   call(Q& q, const CONTEXT& n) { return SelectType<Default,Member,Exist>::TYPE::call(q,n); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_PROCESS_SELF>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename CONTEXT> static HK_INLINE int   call(Q&, const CONTEXT&) { return 1; } };
    struct Member { template <typename CONTEXT> static HK_INLINE int    call(Q& q, const CONTEXT& n) { return q.processSelf(n); } };
};

// Call the query select method if the query define enum { HAS_SELECT }; else call the default implementation.
template <typename Q>
struct IfSelect
{
    template <typename CONTEXT>
    static HK_INLINE int            call(Q& q, _In_reads_(2) const CONTEXT* nodes) { return SelectType<Default,Member,Exist>::TYPE::call(q,nodes); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_SELECT>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename CONTEXT> static HK_INLINE int   call(Q&, _In_opt_ const CONTEXT*) { return 0; } };
    struct Member { template <typename CONTEXT> static HK_INLINE int    call(Q& q, _In_ const CONTEXT* nodes) { return q.select(nodes); } };
};

// Call the query popNode method if the query define enum { HAS_POP_NODE }; else call the default implementation.
template <typename Q>
struct IfPopNode
{
    template <typename A>
    static HK_INLINE void call(Q& q, const A& a) { SelectType<Default,Member,Exist>::TYPE::call(q,a); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_POP_NODE>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename A> static HK_INLINE void    call(Q&, const A&) {} };
    struct Member { template <typename A> static HK_INLINE void call(Q& q, const A& a) { q.popNode(a); } };
};

// Uses 2 way binary queries if USES_2_WAY_BINARY_QUERIES is defined, else uses 4 way.
template <typename Q>
struct UseTwoWayBinary
{
private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::USES_2_WAY_BINARY_QUERIES>*);
    template <typename U> static int    Check(...);
public:

    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };
};

// Call the query beginFront method if the query define enum { HAS_FRONT_TRACKING }; else does nothing.
template <typename Q>
struct IfBeginFront
{
    template <typename TREE>
    static HK_INLINE int call(Q& q, const TREE& t) { return SelectType<Default,Member,Exist>::TYPE::call(q,t); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_FRONT_TRACKING>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename TREE> static HK_INLINE int  call(Q&, const TREE&) { return 1; } };
    struct Member { template <typename TREE> static HK_INLINE int       call(Q& q, const TREE& t) { return q.beginFront(t); } };
};

// Call the query getFront method if the query define enum { HAS_FRONT_TRACKING }; else fetch the tree root context.
template <typename Q>
struct IfGetFront
{
    template <typename TREE, typename CONTEXT>
    static HK_INLINE void call(Q& q, int i, const TREE& t, CONTEXT& c) { SelectType<Default,Member,Exist>::TYPE::call(q,i,t,c); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_FRONT_TRACKING>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename TREE, typename CONTEXT> static HK_INLINE void   call(Q&, int, const TREE& t, CONTEXT& c) { t.initializeRootContext(c, TREE::USE_SOFTWARE_CACHE ? true : false); } };
    struct Member { template <typename TREE, typename CONTEXT> static HK_INLINE void call(Q& q, int i, const TREE& t, CONTEXT& c) { q.getFront(i,t,c); } };
};

// Call the query pushFront method if the query define enum { HAS_FRONT_TRACKING }; else does nothing.
template <typename Q>
struct IfPushFront
{
    template <typename CONTEXT>
    static HK_INLINE void call(Q& q, const CONTEXT& c, bool l) { SelectType<Default,Member,Exist>::TYPE::call(q,c,l); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_FRONT_TRACKING>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename CONTEXT> static HK_INLINE void  call(Q&, const CONTEXT&, bool) {} };
    struct Member { template <typename CONTEXT> static HK_INLINE void       call(Q& q, const CONTEXT& c, bool l) { q.pushFront(c,l); } };
};

// Call the query endFront method if the query define enum { HAS_FRONT_TRACKING }; else does nothing.
template <typename Q>
struct IfEndFront
{
    template <typename TREE>
    static HK_INLINE void call(Q& q, const TREE& t) { return SelectType<Default,Member,Exist>::TYPE::call(q,t); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_FRONT_TRACKING>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename TREE> static HK_INLINE void call(Q&, const TREE&) {} };
    struct Member { template <typename TREE> static HK_INLINE void  call(Q& q, const TREE& t) { q.endFront(t); } };
};

// Check whatever the tree has a getLeafTree method.
template <typename T>
struct HasGetLeafTree
{
private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_GET_LEAF_TREE>*);
    template <typename U> static int    Check(...);
public:
    enum { Exist = sizeof(Check<T>(0)) == sizeof(char) };
};

// Call the query filterNode method if the query define enum { HAS_NODE_FILTER }; else return 1 (true).
template <typename Q, typename TREE>
struct IfFilterNode
{
    template <typename CONTEXT>
    static HK_INLINE hkBool32   call(Q& q, const CONTEXT& c) { return SelectType<Default,Member,Exist>::TYPE::call(q,c); }

private:

    template <typename U, int>          struct Operand {};
    template <typename U> static char   Check(Operand<U,U::HAS_NODE_FILTER>*);
    template <typename U> static int    Check(...);
    enum { Exist = sizeof(Check<Q>(0)) == sizeof(char) };

    struct Default { template <typename CONTEXT> static HK_INLINE hkBool32  call(Q&, const CONTEXT&) { return 1; } };
    struct Member { template <typename CONTEXT> static HK_INLINE hkBool32       call(Q& q, const CONTEXT& c) { return q.filterNode(c); } };
};

//
// Trampolines
// Used to handle sub-tree queries and shared stacks.
//

// Abstract unary trampoline
template <typename QUERIES_TYPE, int TYPE> struct UnaryTrampoline {};

// Partial unary trampoline specialization for terminal trees.
template <typename QUERIES_TYPE> struct UnaryTrampoline<QUERIES_TYPE,0>
{
    template <typename TREE, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processLeaf(const TREE& tree, NEW_STACK&, QUERY& query, const typename TREE::NodeContext& context)
    {
        tree.processLeaf(query, context);
    }
};

// Partial unary trampoline specialization for non-terminal trees.
template <typename QUERIES_TYPE> struct UnaryTrampoline<QUERIES_TYPE,1>
{
    // Abstract forwarder (TREE is a parameter of the class instead of the method to avoid explicit specialization forbidden at class level).
    template <typename TREE, bool SHARED_STACK> struct Forwarder {};

    // Use a new stack for the sub-tree query.
    template <typename TREE> struct Forwarder<TREE, false>
    {
        template <typename SUB_TREE, typename NEW_STACK, typename QUERY>
        static HK_INLINE void   forward(const TREE& tree, const SUB_TREE& subTree, NEW_STACK&, QUERY& query, const typename TREE::NodeContext& context)
        {
            QUERIES_TYPE::unary(subTree, query);
        }
    };

    // Share the current stack for the sub-tree query.
    template <typename TREE> struct Forwarder<TREE, true>
    {
        template <typename SUB_TREE, typename NEW_STACK, typename QUERY>
        static HK_INLINE void   forward(const TREE& tree, const SUB_TREE& subTree, NEW_STACK& stack, QUERY& query, const typename TREE::NodeContext& context)
        {
            QUERIES_TYPE::unary(subTree, stack, query);
        }
    };

    // Only share the stack if the stack item type size is the same.
    template <typename TREE, typename SUB_TREE, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processSubTree(const TREE& tree, const SUB_TREE& subTree, NEW_STACK& stack, QUERY& query, const typename TREE::NodeContext& context)
    {
        const SUB_TREE* subTreeAddr = &subTree;
        if(query.enterSubTree(tree, subTreeAddr, context))
        {
            Forwarder<TREE, sizeof(typename TREE::Slot) == sizeof(typename SUB_TREE::Slot)>::forward(tree, *subTreeAddr, stack, query, context);
            query.leaveSubTree(*subTreeAddr, tree);
        }
    }

    // Extract the sub-tree type / instance and delegate to processSubTree.
    template <typename TREE, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processLeaf(const TREE& tree, NEW_STACK& stack, QUERY& query, const typename TREE::NodeContext& context)
    {
        processSubTree(tree, tree.getLeafTree(context), stack, query, context);
    }
};

// Abstract binary trampoline
template <typename QUERIES_TYPE, int TYPE_A, int TYPE_B> struct BinaryTrampoline {};

// Partial binary trampoline specialization for terminal trees.
template <typename QUERIES_TYPE> struct BinaryTrampoline<QUERIES_TYPE,0,0>
{
    template <typename TREE_A, typename TREE_B, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processLeaves(bool self, const TREE_A& treeA, const TREE_B& treeB, NEW_STACK&, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
    {
        query.processLeaves(contextA, contextB);
    }
};

// Partial binary trampoline specialization for non-terminal trees.
template <typename QUERIES_TYPE> struct BinaryTrampoline<QUERIES_TYPE,1,1>
{
    // Abstract forwarder (TREE is a parameter of the class instead of the method to avoid explicit specialization forbidden at class level).
    template <typename TREE_A, typename TREE_B, bool SHARED_STACK> struct Forwarder {};

    // Use a new stack for the sub-tree query.
    template <typename TREE_A, typename TREE_B> struct Forwarder<TREE_A, TREE_B, false>
    {
        template <typename SUB_TREE_A, typename SUB_TREE_B, typename NEW_STACK, typename QUERY>
        static HK_INLINE void   forward(bool self, const TREE_A& treeA, const TREE_B& treeB, const SUB_TREE_A& subTreeA, const SUB_TREE_B& subTreeB, NEW_STACK&, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
        {
            QUERIES_TYPE::binary(self, subTreeA, subTreeB, query);
        }
    };

    // Share the current stack for the sub-tree query.
    template <typename TREE_A, typename TREE_B> struct Forwarder<TREE_A, TREE_B, true>
    {
        template <typename SUB_TREE_A, typename SUB_TREE_B, typename NEW_STACK, typename QUERY>
        static HK_INLINE void   forward(bool self, const TREE_A& treeA, const TREE_B& treeB, const SUB_TREE_A& subTreeA, const SUB_TREE_B& subTreeB, NEW_STACK& stack, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
        {
            QUERIES_TYPE::binary(self, subTreeA, subTreeB, stack, query);
        }
    };

    // Only share the stack if the stack item type size is the same.
    template <typename TREE_A, typename TREE_B, typename SUB_TREE_A, typename SUB_TREE_B, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processSubTrees(bool self, const TREE_A& treeA, const TREE_B& treeB, const SUB_TREE_A& subTreeA, const SUB_TREE_B& subTreeB, NEW_STACK& stack, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
    {
        const SUB_TREE_A*   subTreeA_Addr = &subTreeA;
        const SUB_TREE_B*   subTreeB_Addr = &subTreeB;
        if(query.enterSubTrees(treeA, subTreeA_Addr, contextA, treeB, subTreeB_Addr, contextB))
        {
            Forwarder<TREE_A, TREE_B, sizeof(typename TREE_A::Slot) == sizeof(typename SUB_TREE_A::Slot) && sizeof(typename TREE_B::Slot) == sizeof(typename SUB_TREE_B::Slot)>::forward(self, treeA, treeB, *subTreeA_Addr, *subTreeB_Addr, stack, query, contextA, contextB);
            query.leaveSubTrees(*subTreeA_Addr, treeA, *subTreeB_Addr, treeB);
        }
    }

    // Extract the sub-tree type / instance and delegate to processSubTree.
    template <typename TREE_A, typename TREE_B, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processLeaves(bool self, const TREE_A& treeA, const TREE_B& treeB, NEW_STACK& stack, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
    {
        processSubTrees(self, treeA, treeB, treeA.getLeafTree(contextA), treeB.getLeafTree(contextB), stack, query, contextA, contextB);
    }
};

// Partial binary trampoline specialization for left non-terminal trees.
template <typename QUERIES_TYPE> struct BinaryTrampoline<QUERIES_TYPE,1,0>
{
    // Abstract forwarder (TREE is a parameter of the class instead of the method to avoid explicit specialization forbidden at class level).
    template <typename TREE_A, typename TREE_B, bool SHARED_STACK> struct Forwarder {};

    // Use a new stack for the sub-tree query.
    template <typename TREE_A, typename TREE_B> struct Forwarder<TREE_A, TREE_B, false>
    {
        template <typename SUB_TREE_A, typename SUB_TREE_B, typename NEW_STACK, typename QUERY>
        static HK_INLINE void   forward(bool self, const TREE_A& treeA, const TREE_B& treeB, const SUB_TREE_A& subTreeA, const SUB_TREE_B& subTreeB, NEW_STACK&, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
        {
            QUERIES_TYPE::binary(self, subTreeA, subTreeB, query, HK_NULL, &contextB);
        }
    };

    // Share the current stack for the sub-tree query.
    template <typename TREE_A, typename TREE_B> struct Forwarder<TREE_A, TREE_B, true>
    {
        template <typename SUB_TREE_A, typename SUB_TREE_B, typename NEW_STACK, typename QUERY>
        static HK_INLINE void   forward(bool self, const TREE_A& treeA, const TREE_B& treeB, const SUB_TREE_A& subTreeA, const SUB_TREE_B& subTreeB, NEW_STACK& stack, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
        {
            QUERIES_TYPE::binary(self, subTreeA, subTreeB, stack, query, HK_NULL, &contextB);
        }
    };

    // Only share the stack if the stack item type size is the same.
    template <typename TREE_A, typename TREE_B, typename SUB_TREE_A, typename SUB_TREE_B, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processSubTrees(bool self, const TREE_A& treeA, const TREE_B& treeB, const SUB_TREE_A& subTreeA, const SUB_TREE_B& subTreeB, NEW_STACK& stack, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
    {
        const SUB_TREE_A*   subTreeA_Addr = &subTreeA;
        const SUB_TREE_B*   subTreeB_Addr = &subTreeB;
        if(query.enterSubTrees(treeA, subTreeA_Addr, contextA, treeB, subTreeB_Addr, contextB))
        {
            Forwarder<TREE_A, TREE_B, sizeof(typename TREE_A::Slot) == sizeof(typename SUB_TREE_A::Slot) && sizeof(typename TREE_B::Slot) == sizeof(typename SUB_TREE_B::Slot)>::forward(self, treeA, treeB, *subTreeA_Addr, *subTreeB_Addr, stack, query, contextA, contextB);
            query.leaveSubTrees(*subTreeA_Addr, treeA, *subTreeB_Addr, treeB);
        }
    }

    // Extract the sub-tree type / instance and delegate to processSubTree.
    template <typename TREE_A, typename TREE_B, typename NEW_STACK, typename QUERY>
    static HK_INLINE void   processLeaves(bool self, const TREE_A& treeA, const TREE_B& treeB, NEW_STACK& stack, QUERY& query, const typename TREE_A::NodeContext& contextA, const typename TREE_B::NodeContext& contextB)
    {
        processSubTrees(self, treeA, treeB, treeA.getLeafTree(contextA), treeB, stack, query, contextA, contextB);
    }
};

//
// Query wrappers.
//

/// BaseWrapper
template <typename QUERY>
struct BaseWrapper
{
    HK_INLINE               BaseWrapper(QUERY& query) : m_query(query) {}

    template <typename PARENT_TREE, typename SUB_TREE, typename CONTEXT>
    HK_INLINE hkBool32  enterSubTree(const PARENT_TREE& parentTree, _Pre_valid_ _Outref_ const SUB_TREE*& subTree, const CONTEXT& context) const
    {
        return m_query.enterSubTree(parentTree, subTree, context);
    }

    template <typename SUB_TREE, typename PARENT_TREE>
    HK_INLINE void      leaveSubTree(const SUB_TREE& subTree, const PARENT_TREE& parentTree) const
    {
        m_query.leaveSubTree(subTree, parentTree);
    }

    template <typename PARENT_TREE_A, typename SUB_TREE_A, typename CONTEXT_A, typename PARENT_TREE_B, typename SUB_TREE_B, typename CONTEXT_B>
    HK_INLINE hkBool32  enterSubTrees(const PARENT_TREE_A& parentTreeA, _Pre_valid_ _Outref_ const SUB_TREE_A*& subTreeA, const CONTEXT_A& contextA,
        const PARENT_TREE_B& parentTreeB, _Pre_valid_ _Outref_ const SUB_TREE_B*& subTreeB, const CONTEXT_B& contextB) const
    {
        return m_query.enterSubTrees(parentTreeA, subTreeA, contextA, parentTreeB, subTreeB, contextB);
    }

    template <typename SUB_TREE_A, typename PARENT_TREE_A, typename SUB_TREE_B, typename PARENT_TREE_B>
    HK_INLINE void      leaveSubTrees(  const SUB_TREE_A& subTreeA, const PARENT_TREE_A& parentTreeA,
                                                const SUB_TREE_B& subTreeB, const PARENT_TREE_B& parentTreeB) const
    {
        m_query.leaveSubTrees(subTreeA, parentTreeA, subTreeB, parentTreeB);
    }

    template <typename CONTEXT>
    HK_INLINE hkBool32  filterNode(const CONTEXT& node) const
    {
        return IfFilterNode<QUERY,CONTEXT>::call(m_query,node);
    }

    template <typename CONTEXT_A, typename CONTEXT_B>
    HK_INLINE hkBool32  filterNodes(const CONTEXT_A& nodeA, const CONTEXT_B& nodeB) const
    {
        return (IfFilterNode<QUERY,CONTEXT_A>::call(m_query,nodeA) != hkFalse32) && (IfFilterNode<QUERY,CONTEXT_B>::call(m_query,nodeB) != hkFalse32);
    }

    QUERY&      m_query;
};

/// AabbOverlapsWithEarlyExitWrapper
template <typename QUERY>
struct AabbOverlapsWithEarlyExitWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               AabbOverlapsWithEarlyExitWrapper(QUERY& query) : BaseWrapper<QUERY>(query), m_continue(1) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const { return m_continue && this->filterNode(node) && m_aabb.overlaps(node.m_aabb); }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { m_continue = m_continue && BaseWrapper<QUERY>::m_query.processLeaf(leaf); }

    hkBool32    m_continue;
    hkAabb      m_aabb;
};

/// AabbOverlapsNoEarlyExitWrapper
template <typename QUERY>
struct AabbOverlapsNoEarlyExitWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               AabbOverlapsNoEarlyExitWrapper(QUERY& query) : BaseWrapper<QUERY>(query) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const { return this->filterNode(node) ? m_aabb.overlaps(node.m_aabb) : 0; }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeaf(leaf); }

    hkAabb      m_aabb;
};

/// AabbOverlapsNearMissAabbWrapper
template <typename QUERY>
struct AabbOverlapsNearMissAabbWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               AabbOverlapsNearMissAabbWrapper(QUERY& query) : BaseWrapper<QUERY>(query) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node)
    {
        if(this->filterNode(node))
            return m_aabb.overlapsUpdateSmallestNonExtraHitAabb(node.m_aabb, m_nearMissAabb);
        else
            return 0;
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeafNmp(leaf, &m_nearMissAabb); }

    hkAabb      m_aabb;
    hkAabb      m_nearMissAabb;
};

/// KineticAabbWrapper
template <typename QUERY>
struct KineticAabbOverlapsWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               KineticAabbOverlapsWrapper(QUERY& query) : BaseWrapper<QUERY>(query) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
            return m_aabbs[0].overlaps(node.m_aabb) | m_aabbs[1].overlaps(node.m_aabb);
        else
            return 0;
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) const
    {
        BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_aabbs[0].overlaps(leaf.m_aabb), m_aabbs[1].overlaps(leaf.m_aabb));
    }

    hkAabb              m_aabbs[2];
};

/// CoherentKineticAabbOverlapsWrapper
template <typename QUERY>
struct CoherentKineticAabbOverlapsWrapper : public KineticAabbOverlapsWrapper<QUERY>
{
    typedef KineticAabbOverlapsWrapper<QUERY> Parent;

    HK_INLINE               CoherentKineticAabbOverlapsWrapper(QUERY& query) : Parent(query) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
            return m_compound.overlaps(node.m_aabb);
        else
            return 0;
    }

    hkAabb      m_compound;
};

/// ConvexOverlapsWrapperBase
template <typename QUERY>
struct ConvexOverlapsWrapperBase : public BaseWrapper<QUERY>
{
    enum Classification { CLS_OUT=0, CLS_IN=1, CLS_CROSS=2 };

    HK_INLINE ConvexOverlapsWrapperBase(QUERY& q) : BaseWrapper<QUERY>(q) {}

    static HK_INLINE Classification classify(const hkAabb& aabb, const hkVector4& plane)
    {
        hkSimdReal  minDst, maxDst;
        hkAabbUtil::computeAabbPlaneSpan(plane, aabb, minDst, maxDst);
        if( (minDst*maxDst).isLessZero() ) return CLS_CROSS;
        if( minDst.isLessEqualZero() ) return CLS_IN;
        return CLS_OUT;
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) const { BaseWrapper<QUERY>::m_query.processLeaf(leaf); }

    const hkVector4*    m_planes;
    int                 m_numPlanes;
};

/// ConvexOverlapsWrapper
template <typename QUERY>
struct ConvexOverlapsWrapper : public BaseWrapper<QUERY>
{
    enum Classification { CLS_OUT=0, CLS_IN=1, CLS_CROSS=2 };

    HK_INLINE   ConvexOverlapsWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    static HK_INLINE Classification classify(const hkAabb& aabb, const hkVector4& plane)
    {
        hkSimdReal  minDst, maxDst;
        hkAabbUtil::computeAabbPlaneSpan(plane, aabb, minDst, maxDst);
        if( (minDst*maxDst).isLessZero() ) return CLS_CROSS;
        if( minDst.isLessEqualZero() ) return CLS_IN;
        return CLS_OUT;
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node)
    {
        if(this->filterNode(node))
    {
        int aggregateClass = 0;
        for(int i=0; i < m_numPlanes; ++i)
        {
            Classification  cls = classify(node.m_aabb, m_planes[i]);
            aggregateClass |= cls;
            if(cls == CLS_OUT) return 0;
        }

        if(HasGetLeafTree<TREE>::Exist || (aggregateClass & CLS_CROSS)) return 1;
        static_cast<const TREE*>(m_tree)->collectLeavesWithCollector(node.m_index, *this);
        }
        return 0;
    }

    template <typename TREE, typename INDEX>
    HK_INLINE void append(_In_ const TREE* tree, INDEX leaf) const
    {
        typename TREE::NodeContext context;
        tree->setContextFromNode(leaf, context, false);
        processLeaf(context);
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) const
    {
        BaseWrapper<QUERY>::m_query.processLeaf(leaf);
    }

    const void*         m_tree;
    const hkVector4*    m_planes;
    int                 m_numPlanes;
};

/// VisibilityWrapper
template <typename QUERY>
struct VisibilityWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               VisibilityWrapper(QUERY& q) : BaseWrapper<QUERY>(q), m_visible(1) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(m_visible && this->filterNode(node))
        {
            hkSimdReal  fraction = BaseWrapper<QUERY>::m_query.getFraction();
            return hkcdIntersectRayAabb(m_ray, node.m_aabb, &fraction);
        }
        return 0;
    }

    template <typename PARENT_TREE, typename SUB_TREE, typename CONTEXT>
    HK_INLINE hkBool32  enterSubTree(const PARENT_TREE& parentTree, _Pre_valid_ _Outref_ const SUB_TREE*& subTree, const CONTEXT& context)
    {
        return BaseWrapper<QUERY>::m_query.enterSubTree(parentTree, subTree, context, m_ray);
    }

    template <typename SUB_TREE, typename PARENT_TREE>
    HK_INLINE void      leaveSubTree(const SUB_TREE& subTree, const PARENT_TREE& parentTree)
    {
        return BaseWrapper<QUERY>::m_query.leaveSubTree(subTree, parentTree, m_ray);
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { if(m_visible) m_visible = m_visible && bool(BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_ray)); }

    hkcdRay     m_ray;
    bool        m_visible;
};

/// RayOverlapsWrapper
template <typename QUERY>
struct RayOverlapsWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               RayOverlapsWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
        {
            hkSimdReal  fraction = m_fraction;
            return hkcdIntersectRayAabb(m_ray, node.m_aabb, &fraction);
        }
        return 0;
    }

    template <typename PARENT_TREE, typename SUB_TREE, typename CONTEXT>
    HK_INLINE hkBool32  enterSubTree(const PARENT_TREE& parentTree, _Pre_valid_ _Outref_ const SUB_TREE*& subTree, const CONTEXT& context)
    {
        return BaseWrapper<QUERY>::m_query.enterSubTree(parentTree, subTree, context, m_ray);
    }

    template <typename SUB_TREE, typename PARENT_TREE>
    HK_INLINE void      leaveSubTree(const SUB_TREE& subTree, const PARENT_TREE& parentTree)
    {
        return BaseWrapper<QUERY>::m_query.leaveSubTree(subTree, parentTree, m_ray);
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_ray); }

    hkcdRay     m_ray;
    hkSimdReal  m_fraction;
};

/// RayCastWrapper
template <typename QUERY>
struct RayCastWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               RayCastWrapper(QUERY& q) : BaseWrapper<QUERY>(q), m_selectResult(-1) {}

    enum { HAS_PROCESS_CHILDREN, HAS_SELECT };

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
    {
        hkSimdReal  fraction = m_ray.getFraction();
        return hkcdIntersectRayAabb(m_ray, node.m_aabb, &fraction);
    }
        return 0;
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE int           processChildren(_In_reads_(2) const CONTEXT* nodes)
    {
        if(this->filterNode(nodes[0]) || this->filterNode(nodes[1]))
        {
            hkSimdReal  fraction0 = m_ray.getFraction(), fraction1 = fraction0;
            const int   ret =   (hkcdIntersectRayAabb(m_ray, nodes[0].m_aabb, &fraction0)? 1 : 0) |
                                (hkcdIntersectRayAabb(m_ray, nodes[1].m_aabb, &fraction1)? 2 : 0) ;
            if(ret == 3) m_selectResult = fraction1 < fraction0 ? 1 : 0;
            return ret;
        }
        return 0;
    }

    template <typename PARENT_TREE, typename SUB_TREE, typename CONTEXT>
    HK_INLINE hkBool32  enterSubTree(const PARENT_TREE& parentTree, _Pre_valid_ _Outref_ const SUB_TREE*& subTree, const CONTEXT& context)
    {
        return BaseWrapper<QUERY>::m_query.enterSubTree(parentTree, subTree, context, m_ray);
    }

    template <typename SUB_TREE, typename PARENT_TREE>
    HK_INLINE void      leaveSubTree(const SUB_TREE& subTree, const PARENT_TREE& parentTree)
    {
        return BaseWrapper<QUERY>::m_query.leaveSubTree(subTree, parentTree, m_ray);
    }

    template <typename CONTEXT>
    HK_INLINE int           select(_In_opt_ const CONTEXT* nodes) const { return m_selectResult; }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_ray); }

    hkcdRay     m_ray;
    int         m_selectResult;
};

/// RayCastWrapperExternal
template <typename QUERY>
struct RayCastWrapperExternal : public BaseWrapper<QUERY>
{
    HK_INLINE               RayCastWrapperExternal(QUERY& q) : BaseWrapper<QUERY>(q), m_selectResult(-1) {}

    enum { HAS_PROCESS_CHILDREN, HAS_SELECT };

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
        {
            hkSimdReal  fraction = BaseWrapper<QUERY>::m_query.getFraction();
            return hkcdIntersectRayAabb(m_ray, node.m_aabb, &fraction);
        }
        return 0;
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE int           processChildren(_In_reads_(2) const CONTEXT* nodes) const
    {
        if(this->filterNode(nodes[0]) || this->filterNode(nodes[1]))
        {
            hkSimdReal  fraction0 = BaseWrapper<QUERY>::m_query.getFraction(), fraction1 = fraction0;
            const int   ret =   (hkcdIntersectRayAabb(m_ray, nodes[0].m_aabb, &fraction0)? 1 : 0) |
                                (hkcdIntersectRayAabb(m_ray, nodes[1].m_aabb, &fraction1)? 2 : 0) ;
            if(ret == 3) m_selectResult = fraction1 < fraction0 ? 1 : 0;
            return ret;
        }
        return 0;
    }

    template <typename PARENT_TREE, typename SUB_TREE, typename CONTEXT>
    HK_INLINE hkBool32  enterSubTree(const PARENT_TREE& parentTree, _Pre_valid_ _Outref_ const SUB_TREE*& subTree, const CONTEXT& context)
    {
        return BaseWrapper<QUERY>::m_query.enterSubTree(parentTree, subTree, context, m_ray);
    }

    template <typename SUB_TREE, typename PARENT_TREE>
    HK_INLINE void      leaveSubTree(const SUB_TREE& subTree, const PARENT_TREE& parentTree)
    {
        return BaseWrapper<QUERY>::m_query.leaveSubTree(subTree, parentTree, m_ray);
    }

    template <typename CONTEXT>
    HK_INLINE int           select(_In_opt_ const CONTEXT* nodes) const { return m_selectResult; }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) const { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_ray); }

    hkcdRay         m_ray;
    mutable int     m_selectResult;
};

/// RayCastBatchWrapper
template <typename QUERY>
struct RayCastBatchWrapper : public BaseWrapper<QUERY>
{
    enum { RAY_CAPACITY = 4, HAS_PROCESS_CHILDREN, HAS_POP_NODE };

    HK_INLINE   RayCastBatchWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    HK_INLINE void initialize(_In_reads_(numRays) const hkcdRay* rays, int numRays, int baseIndex)
    {
        for(int i=0; i<numRays; ++i) m_rays[i] = *rays++;
        m_allValids =   (hkUint8) ((1<<numRays)-1);
        m_state     =   m_allValids;
        m_baseIndex =   baseIndex;

    }

    static HK_INLINE hkUint8    lsbPosition(hkUint8 bits) { return (hkUint8) (0x484c4840 >> bits) & 3; }

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkUint8       processNode(const CONTEXT& node, hkUint8 currentState, int index, hkUint8 bit)
    {
        if(node->isInternal())
        {
            for(; currentState; ++index, currentState >>= 1)
            {
                hkSimdReal  fraction = m_rays[index].getFraction();
                if(hkcdIntersectRayAabb(m_rays[index], node.m_aabb, &fraction))
                {
                    return hkUint8(currentState << index);
                }
            }
            return 0;
        }
        else
        {
            hkUint8 newState = 0;
            for(; currentState; ++index, bit <<= 1, currentState >>= 1)
            {
                if(currentState & 1)
                {
                    hkSimdReal  fraction = m_rays[index].getFraction();
                    if(hkcdIntersectRayAabb(m_rays[index], node.m_aabb, &fraction))
                    {
                        newState |= bit;
                    }
                }
            }
            return newState;
        }
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE int       processNode(const CONTEXT& node)
    {
        if(this->filterNode(node))
    {
        const hkUint8   first = lsbPosition(m_state);
        const hkUint8   state = m_state >> first;
        const hkUint8   bit = 1 << first;
        m_state = processNode<TREE, CONTEXT>(node, state, first, bit);
        }
        else
        {
            m_state = 0;
        }
        return m_state;
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE int       processChildren(_In_reads_(2) const CONTEXT* nodes)
    {
        const hkUint8   first = lsbPosition(m_state);
        const hkUint8   state = m_state >> first;
        const hkUint8   bit = 1 << first;
        const hkUint8   s0 = this->filterNode(nodes[0]) ? processNode<TREE,CONTEXT>(nodes[0], state, first, bit) : 0;
        const hkUint8   s1 = this->filterNode(nodes[1]) ? processNode<TREE,CONTEXT>(nodes[1], state, first, bit) : 0;
        const hkUint8   rc = (s0?1:0) | (s1?2:0);
        switch(rc & 3)
        {
            case    0:  break;
            case    1:  m_state = s0; break;
            case    2:  m_state = s1; break;
            case    3:  m_state = s0; m_stack.push(s1); break;
        }
        return rc;
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf)
    {
        hkUint8 index = lsbPosition(m_state);
        hkUint8 valids = m_state >> index;
        for(int globalIndex = index + m_baseIndex; valids; ++index, ++globalIndex, valids >>= 1)
        {
            if(valids & 1) BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_rays[index], globalIndex);
        }
    }

    template <typename CONTEXT>
    HK_INLINE void      popNode(const CONTEXT&) { m_state = m_stack.pop(m_allValids); }

    hkcdRay                     m_rays[RAY_CAPACITY];
    BoundedStack<hkUint8,128>   m_stack;
    int                         m_baseIndex;
    hkUint8                     m_state;
    hkUint8                     m_allValids;
};

/// SphereCastWrapper
template <typename QUERY>
struct SphereCastWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE               SphereCastWrapper(QUERY& q) : BaseWrapper<QUERY>(q), m_selectResult(-1) {}

    enum { HAS_PROCESS_CHILDREN, HAS_SELECT };

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
        {
            hkSimdReal  fraction = m_ray.getFraction();
            hkAabb      expAabb; expAabb.setExpandBy(node.m_aabb,m_radius);
            return hkcdIntersectRayAabb(m_ray, expAabb, &fraction);
        }
        return 0;
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE int           processChildren(_In_reads_(2) const CONTEXT* nodes)
    {
        if(this->filterNode(nodes[0]) || this->filterNode(nodes[1]))
        {
            hkSimdReal  fraction0 = m_ray.getFraction(), fraction1 = fraction0;
            hkAabb      expAabb0; expAabb0.setExpandBy(nodes[0].m_aabb,m_radius);
            hkAabb      expAabb1; expAabb1.setExpandBy(nodes[1].m_aabb,m_radius);
            const int   ret =   (hkcdIntersectRayAabb(m_ray, expAabb0, &fraction0)? 1 : 0) |
                                (hkcdIntersectRayAabb(m_ray, expAabb1, &fraction1)? 2 : 0) ;
            if(ret == 3) m_selectResult = fraction1 < fraction0 ? 1 : 0;
            return ret;
        }
        return 0;
    }

    template <typename PARENT_TREE, typename SUB_TREE, typename CONTEXT>
    HK_INLINE hkBool32  enterSubTree(const PARENT_TREE& parentTree, _Pre_valid_ _Outref_ const SUB_TREE*& subTree, const CONTEXT& context)
    {
        return BaseWrapper<QUERY>::m_query.enterSubTree(parentTree, subTree, context, m_ray);
    }

    template <typename SUB_TREE, typename PARENT_TREE>
    HK_INLINE void      leaveSubTree(const SUB_TREE& subTree, const PARENT_TREE& parentTree)
    {
        return BaseWrapper<QUERY>::m_query.leaveSubTree(subTree, parentTree, m_ray);
    }

    template <typename CONTEXT>
    HK_INLINE int           select(_In_opt_ const CONTEXT* nodes) const { return m_selectResult; }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_ray, m_radius); }

    hkcdRay     m_ray;
    hkSimdReal  m_radius;
    int         m_selectResult;
};

/// AabbCastWrapper
template <typename QUERY>
struct AabbCastWrapper : public BaseWrapper<QUERY>
{
    enum { HAS_PROCESS_CHILDREN, HAS_SELECT };

    HK_INLINE               AabbCastWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
    {
            hkSimdReal  fraction = m_ray.getFraction();
        hkAabb extAabb;
            extAabb.m_min.setSub(node.m_aabb.m_min, m_halfExtent);
            extAabb.m_max.setAdd(node.m_aabb.m_max, m_halfExtent);
            return hkcdIntersectRayAabb(m_ray, extAabb, &fraction);
        }
        return 0;
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_ray); }


    template <typename TREE, typename CONTEXT>
    HK_INLINE int processChildren(_In_reads_(2) const CONTEXT* nodes)
    {
        hkAabb extAabb[2];
        extAabb[0].m_min.setSub(nodes[0].m_aabb.m_min, m_halfExtent);
        extAabb[0].m_max.setAdd(nodes[0].m_aabb.m_max, m_halfExtent);
        extAabb[1].m_min.setSub(nodes[1].m_aabb.m_min, m_halfExtent);
        extAabb[1].m_max.setAdd(nodes[1].m_aabb.m_max, m_halfExtent);
        hkSimdReal  fraction0 = m_ray.getFraction(), fraction1 = fraction0;
        const int   ret =   (hkcdIntersectRayAabb(m_ray, extAabb[0], &fraction0)? (this->filterNode(nodes[0]) ? 1 : 0) : 0) |
                            (hkcdIntersectRayAabb(m_ray, extAabb[1], &fraction1)? (this->filterNode(nodes[1]) ? 2 : 0) : 0) ;
        if(ret == 3) m_selectResult = fraction1 < fraction0 ? 1 : 0;
        return ret;
    }

    template <typename CONTEXT>
    HK_INLINE int select(_In_opt_ const CONTEXT* nodes) const { return m_selectResult; }

    hkcdRay     m_ray;
    hkVector4   m_halfExtent;
    int         m_selectResult;
};

/// AabbCastWrapperExternal
template <typename QUERY>
struct AabbCastWrapperExternal : public BaseWrapper<QUERY>
{
    enum { HAS_PROCESS_CHILDREN, HAS_SELECT };

    HK_INLINE               AabbCastWrapperExternal(QUERY& q) : BaseWrapper<QUERY>(q) {}

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
{
            hkSimdReal  fraction = BaseWrapper<QUERY>::m_query.getFraction();
            hkAabb extAabb;
            extAabb.m_min.setSub(node.m_aabb.m_min, m_halfExtent);
            extAabb.m_max.setAdd(node.m_aabb.m_max, m_halfExtent);
            return hkcdIntersectRayAabb(m_ray, extAabb, &fraction);
        }
        return 0;
    }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) const { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_ray); }


    template <typename TREE, typename CONTEXT>
    HK_INLINE int processChildren(_In_reads_(2) const CONTEXT* nodes) const
    {
        hkAabb extAabb[2];
        extAabb[0].m_min.setSub(nodes[0].m_aabb.m_min, m_halfExtent);
        extAabb[0].m_max.setAdd(nodes[0].m_aabb.m_max, m_halfExtent);
        extAabb[1].m_min.setSub(nodes[1].m_aabb.m_min, m_halfExtent);
        extAabb[1].m_max.setAdd(nodes[1].m_aabb.m_max, m_halfExtent);
        hkSimdReal  fraction0 = BaseWrapper<QUERY>::m_query.getFraction(), fraction1 = fraction0;
        const int   ret =   (hkcdIntersectRayAabb(m_ray, extAabb[0], &fraction0)? (this->filterNode(nodes[0]) ? 1 : 0) : 0) |
                            (hkcdIntersectRayAabb(m_ray, extAabb[1], &fraction1)? (this->filterNode(nodes[1]) ? 2 : 0) : 0) ;
        if(ret == 3) m_selectResult = fraction1 < fraction0 ? 1 : 0;
        return ret;
    }

    template <typename CONTEXT>
    HK_INLINE int select(_In_opt_ const CONTEXT* nodes) const { return m_selectResult; }

    hkcdRay     m_ray;
    hkVector4   m_halfExtent;
    mutable int m_selectResult;
};

/// ClosestFromPointWrapper
template <typename QUERY>
struct ClosestFromPointWrapper : public BaseWrapper<QUERY>
{
    enum { HAS_PROCESS_CHILDREN, HAS_SELECT };

    HK_INLINE   ClosestFromPointWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    static HK_INLINE hkSimdReal pointAabbDistanceSquared(const hkVector4& point, const hkAabb& aabb)
    {
        hkVector4   a; a.setMax(aabb.m_min, point);
        hkVector4   b; b.setMin(aabb.m_max, a);
        return point.distanceToSquared(b);
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE int processChildren(_In_reads_(2) const CONTEXT* nodes)
    {
            const hkSimdReal    d0 = pointAabbDistanceSquared(m_point, nodes[0].m_aabb);
            const hkSimdReal    d1 = pointAabbDistanceSquared(m_point, nodes[1].m_aabb);
        const int           rt = (d0.isLessEqual(m_distanceSquared)? (this->filterNode(nodes[0])? 1 : 0) : 0) | (d1.isLessEqual(m_distanceSquared)? (this->filterNode(nodes[1])? 2 : 0) : 0);
            if(rt == 3) m_selectResult = d1 < d0 ? 1 : 0;
            return rt;
        }

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
            return pointAabbDistanceSquared(m_point, node.m_aabb).isLessEqual(m_distanceSquared);
        else
        return 0;
    }

    template <typename CONTEXT>
    HK_INLINE int select(_In_opt_ const CONTEXT* nodes) const { return m_selectResult; }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_point, m_distanceSquared); }

    hkVector4   m_point;
    hkSimdReal  m_distanceSquared;
    int         m_selectResult;
};

/// ClosestFromAabbWrapper
template <typename QUERY>
struct ClosestFromAabbWrapper : public BaseWrapper<QUERY>
{
    enum { HAS_PROCESS_CHILDREN, HAS_SELECT };

    HK_INLINE   ClosestFromAabbWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    HK_INLINE hkSimdReal    AabbAabbDistanceSquared(const hkAabb& aabb) const
    {
        hkVector4   mi; mi.setSub(aabb.m_min, m_extents);
        hkVector4   mx; mx.setAdd(aabb.m_max, m_extents);
        hkVector4   a; a.setMax(mi, m_center);
        hkVector4   b; b.setMin(mx, a);
        return m_center.distanceToSquared(b);
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE int processChildren(_In_reads_(2) const CONTEXT* nodes)
    {
        const hkSimdReal    d0 = AabbAabbDistanceSquared(nodes[0].m_aabb);
        const hkSimdReal    d1 = AabbAabbDistanceSquared(nodes[1].m_aabb);
        const int           rt = (d0.isLessEqual(m_distanceSquared)? (this->filterNode(nodes[0]) ? 1 : 0) : 0) | (d1.isLessEqual(m_distanceSquared)? (this->filterNode(nodes[1]) ? 2 : 0) : 0);
        if(rt == 3) m_selectResult = d1 < d0 ? 1 : 0;
        return rt;
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(this->filterNode(node))
            return AabbAabbDistanceSquared(node.m_aabb).isLessEqual(m_distanceSquared);
        else
            return 0;
    }

    template <typename CONTEXT>
    HK_INLINE int select(_In_opt_ const CONTEXT* nodes) const { return m_selectResult; }

    template <typename CONTEXT>
    HK_INLINE void      processLeaf(const CONTEXT& leaf) { BaseWrapper<QUERY>::m_query.processLeaf(leaf, m_aabb, m_distanceSquared); }

    hkAabb      m_aabb;
    hkVector4   m_center;
    hkVector4   m_extents;
    hkSimdReal  m_distanceSquared;
    int         m_selectResult;
};

/// AllPairsWrapper
template <typename QUERY>
struct AllPairsWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE   AllPairsWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    template <typename CTX_A, typename CTX_B>
    HK_INLINE hkBool32  processNodes(const CTX_A& nodeA, const CTX_B& nodeB) const
    {
        if(this->filterNodes(nodeA,nodeB))
            return nodeA.m_aabb.overlaps(nodeB.m_aabb);
        else
            return 0;
    }

    template <typename CTX_A, typename CTX_B>
    HK_INLINE void      processLeaves(const CTX_A& leafA, const CTX_B& leafB) { BaseWrapper<QUERY>::m_query.processLeaves(leafA, leafB); }
};

/// AllPairsWithTransformWrapper
template <typename QUERY>
struct AllPairsWithTransformWrapper : public BaseWrapper<QUERY>
{
    HK_INLINE   AllPairsWithTransformWrapper(QUERY& q) : BaseWrapper<QUERY>(q) {}

    template <typename CTX_A, typename CTX_B>
    HK_INLINE hkBool32  processNodes(const CTX_A& nodeA, const CTX_B& nodeB) const
    {
        if(this->filterNodes(nodeA,nodeB))
        {
            hkAabb  aabbA,aabbB;
            aabbA.m_min.setSub(nodeA.m_aabb.m_min, m_marginA);
            aabbA.m_max.setAdd(nodeA.m_aabb.m_max, m_marginA);
            aabbB.m_min.setSub(nodeB.m_aabb.m_min, m_marginB);
            aabbB.m_max.setAdd(nodeB.m_aabb.m_max, m_marginB);
            return hkVector4UtilInternal::overlaps6(aabbA, aabbB, m_BtoA, m_AtoB);
        }
        return 0;
    }

    template <typename CTX_A, typename CTX_B>
    HK_INLINE void      processLeaves(const CTX_A& leafA, const CTX_B& leafB) { BaseWrapper<QUERY>::m_query.processLeaves(leafA, leafB); }

    hkTransform m_AtoB;
    hkTransform m_BtoA;
    hkVector4   m_marginA;
    hkVector4   m_marginB;
};

/// AllLeavesPairs
template <typename PAIR>
struct AllLeavesPairsWrapper
{
    HK_INLINE AllLeavesPairsWrapper(hkArray<PAIR>& pairs) : m_pairsOut(pairs) {}

    template <typename CTX_A, typename CTX_B>
    HK_INLINE void  processLeaves(const CTX_A& a, const CTX_B& b) { m_pairsOut.pushBack(PAIR(a.m_index,b.m_index)); }

    hkArray<PAIR>&  m_pairsOut;
};

/// AabbExistWrapper
struct AabbExistWrapper
{
    template <typename CONTEXT>
    HK_INLINE hkBool32  processLeaf(const CONTEXT& leaf)
    {
        hkVector4Comparison cMin = m_aabb.m_min.equal(leaf.m_aabb.m_min);
        hkVector4Comparison cMax = m_aabb.m_max.equal(leaf.m_aabb.m_max);
        hkVector4Comparison cmp; cmp.setAnd( cMin, cMax );
        if(cmp.allAreSet<hkVector4ComparisonMask::MASK_XYZ>())
        {
            m_leaf = leaf.m_index;
            return 0;
        }
        return 1;
    }

    int     m_leaf;
    hkAabb  m_aabb;
};

/// LocatePointWrapper
struct LocatePointWrapper
{
    template <typename CONTEXT>
    HK_INLINE hkBool32  processLeaf(const CONTEXT& leaf)
    {
        if(leaf.m_aabb.containsPoint(m_point))
        {
            m_leaf = leaf.m_index;
            return 0;
        }
        return 1;
    }

    int         m_leaf;
    hkVector4   m_point;
};

/// OverlapExistWrapper
struct OverlapExistWrapper
{
    template <typename CONTEXT>
    HK_INLINE hkBool32  processLeaf(const CONTEXT& leaf)
    {
        if(leaf.m_aabb.overlaps(m_aabb))
    {
            m_leaf = (int) leaf.m_index;
            m_overlap = 1;
            return 0;
        }
        return 1;
    }

    hkBool32    m_overlap;
    hkAabb      m_aabb;
    int         m_leaf;
};

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
