// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
//HK_REFLECTION_PARSER_EXCLUDE_FILE

#pragma once

#include <Geometry/Collide/Types/hkcdRay.h>
#include <Geometry/Collide/Algorithms/Intersect/hkcdIntersectRayAabb.h>
#include <Geometry/Collide/Algorithms/Distance/hkcdDistanceAabbAabb.h>

#include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Math/Vector/hkVector4UtilInternal.h>

#include <Common/Base/Container/BitField/hkBitField.h>

    /// Queries flags
struct HK_EXPORT_COMMON hkcdTreeQueriesFlags
{
    enum _
    {
        NONE                        =   0,
        ALLOW_STACKLESS             =   0x0001  ///< Fallback to stackless query if out of stack space.
    };
};

    /// Querying static or dynamic trees using base queries.
    ///     Define a query type such as:
    ///
    ///     \code
    ///     struct MyQuery
    ///     {
    ///         // Required for unary queries:
    ///         hkBool32    processNode(    const Tree::NodeContext& node) const;       // Return true to dive into the branch else false to skip it.
    ///         void        processLeaf(    const Tree::NodeContext& leaf);         // Process a leaf.
    ///
    ///         // Optional for unary queries:
    ///         int         processChildren(const Tree& tree, const Tree::Index* indices);      // 'enum { HAS_SELECT };' to enable; Enable further processing of a node children depending on the return value bit 1 for child 0, bit 2 for child 1.
    ///         int         select(const Tree& tree, const Tree::Index* indices);       // 'enum { HAS_SELECT };' to enable; Allow for selection the first branch to dive in (default 0), only called if processChildren returns 3.
    ///         void        popNode( const Tree::NodeContext& leaf );                   // 'enum { HAS_POP_NODE };' to enable; A node as been removed from the stack and become the new current branch.
    ///
    ///         // Required for binary queries:
    ///         hkBool32    processNodes(   const Tree::NodeContext& nodeA,
    ///                                     const Tree::NodeContext& nodeB) const;  // Return true to dive into the branches else false to skip them.
    ///         void        processLeaves(  const Tree::NodeContext& leafA,
    ///                                     const Tree::NodeContext& leafB);            // Process a pair of leaves.
    ///
    ///     };
    ///     \endcode
    ///
    ///     Then call one the base queries methods:
    ///     - \code hkcdTreeQueries::unary<TREE,QUERY>(const TREE& myTreeInstance, QUERY& myQueryInstance)\endcode.
    ///     - \code hkcdTreeQueries::binary<TREE_A,TREE_B,QUERY>(const TREE_A& firstTreeInstance, const TREE_B& secondTreeInstance, QUERY& myQueryInstance)\endcode.
    ///     Note: processNode / processNodes is always called before processLeaf / processLeaves, allowing to compute information in the former to be used in the latter.
    ///
    /// Predefined queries specializations:
    ///     a) hkcdFixedStackTreeQueries, queries implementation using a fixed stack of 64 elements with stackless fallback for unary query.
    ///     b) hkcdForcedStackTreeQueries, queries implementation using a fixed stack of 64 elements with NO stackless fallback for unary query, reduce code size on SPU.
    ///     c) hkcdDynamicStackTreeQueries, queries implementation using a dynamic stack with an initial capacity 64 elements (Not available on SPU).
template <template <int SIZE, typename TYPE> class STACK, int SIZE, int FLAGS>
struct hkcdTreeQueries
{
    //
    // Base queries.
    //

    /// Unary query with explicit stack instance.
    template <typename TREE, typename STACK_TYPE, typename QUERY>
    static HK_NEVER_INLINE void unary(const TREE& tree, STACK_TYPE& stack, QUERY& query);

    /// Unary query.
    template <typename TREE, typename QUERY>
    static HK_INLINE void   unary(const TREE& tree, QUERY& query);

    /// Binary query with explicit stack instance.
    template <typename TREE_A, typename TREE_B, typename STACK_TYPE, typename QUERY>
    static HK_NEVER_INLINE void binary(bool self, const TREE_A& treeA, const TREE_B& treeB, STACK_TYPE& stack, QUERY& query, _In_opt_ const typename TREE_A::NodeContext* rootContextA = HK_NULL, _In_opt_ const typename TREE_B::NodeContext* rootContextB = HK_NULL);

    /// Binary query.
    template <typename TREE_A, typename TREE_B, typename QUERY>
    static HK_INLINE void   binary(bool self, const TREE_A& treeA, const TREE_B& treeB, QUERY& query, _In_opt_ const typename TREE_A::NodeContext* rootContextA = HK_NULL, _In_opt_ const typename TREE_B::NodeContext* rootContextB = HK_NULL);

    //
    // Specialized queries.
    //

    /// AABB Query.
    
    template <typename TREE, typename QUERY>
    static HK_INLINE void aabbOverlapsWithEarlyExit(const TREE& tree, const hkAabb& aabb, QUERY& query);

    /// AABB Query.
    
    
    template <typename TREE, typename QUERY>
    static HK_INLINE void aabbOverlapsNoEarlyExit(const TREE& tree, const hkAabb& aabb, QUERY& query);

    /// AABB Query and nearest miss AABB.
    /// Note: nearMissAabb MUST be initialized (ie: setFull()).
    
    template <typename TREE, typename QUERY>
    static inline void          aabbOverlapsNearMiss(const TREE& tree, const hkAabb& aabb, QUERY& query, hkAabb& nearMissAabb);

    /// kinetic AABB Query.
    
    template <typename TREE, typename QUERY>
    static inline void          kineticAabbOverlaps(const TREE& tree, const hkAabb& oldAabb, const hkAabb& newAabb, QUERY& query);

    /// Convex Query.
    
    template <typename TREE, typename QUERY>
    static inline void          convexOverlaps(const TREE& tree, const hkVector4* planes, int numPlanes, QUERY& query);

    /// RayCast Query.
    
    template <typename TREE, typename QUERY>
    static hkSimdReal rayCast(const TREE& tree, const struct hkcdRay& ray, QUERY& query);

    /// Visibility Query.
    
    template <typename TREE, typename QUERY>
    static inline hkBool32      visible(const TREE& tree, const hkcdRay& ray, QUERY& query);

    /// Ray overlaps Query.
    
    template <typename TREE, typename QUERY>
    static inline void          rayOverlaps(const TREE& tree, const hkcdRay& ray, QUERY& query);

    /// RayCast Query.
    
    template <typename TREE, typename QUERY>
    static inline hkSimdReal    rayCast(const TREE& tree, const hkVector4& from, const hkVector4& to, QUERY& query, hkSimdRealParameter fraction = hkSimdReal::getConstant(HK_QUADREAL_1));

    /// RayCast Query.
    /// Fraction is maintained by the query, thus the query needs to implement 'const hkSimdReal& getFraction() const'.
    
    template <typename TREE, typename QUERY>
    static inline void          rayCastExternal(const TREE& tree, const hkcdRay& ray, QUERY& query);

    /// RayCastBatch Query.
    
    template <typename TREE, typename QUERY>
    static inline void          rayCastBatch(const TREE& tree, _In_reads_(numRays) const struct hkcdRay* rays, int numRays, QUERY& query);

    /// RayCastBatch Query.
    
    template <typename TREE, typename QUERY>
    static inline void          rayCastFan(const TREE& tree, const hkVector4& from, _In_reads_(numRays) const hkVector4* to, int numRays, QUERY& query);

    /// RayCast Query.
    
    template <typename TREE, typename QUERY>
    static inline hkSimdReal    sphereCast(const TREE& tree, const struct hkcdRay& ray, const hkSimdReal& radius, QUERY& query);

    /// AabbCast Query.
    
    template <typename TREE, typename QUERY>
    static inline hkSimdReal    aabbCast(const TREE& tree, const hkAabb& from, const hkVector4& to, QUERY& query, hkReal maxFraction = 1.0f);

    /// AabbCast Query.
    /// Fraction is maintained by the query, thus the query needs to implement 'const hkSimdReal& getFraction() const'.
    
    template <typename TREE, typename QUERY>
    static inline void          aabbCastExternal(const TREE& tree, const hkcdRay& ray, hkVector4Parameter halfExtents, QUERY& query);

    /// Closest Query.
    
    template <typename TREE, typename QUERY>
    static inline hkSimdReal    closest(const TREE& tree, const hkVector4& point, QUERY& query, const hkSimdReal& initialDistanceSquared = hkSimdReal::getConstant(HK_QUADREAL_MAX));

    /// Closest Query.
    
    template <typename TREE, typename QUERY>
    static inline hkSimdReal    closest(const TREE& tree, const hkAabb& aabb, QUERY& query, const hkSimdReal& initialDistanceSquared = hkSimdReal::getConstant(HK_QUADREAL_MAX));

    /// Closest Query.
    
    template <typename TREE, typename QUERY, int K>
    static inline hkSimdReal    K_closest(const TREE& tree, const hkVector4& point, QUERY& query, const hkSimdReal& initialDistanceSquared = hkSimdReal::getConstant(HK_QUADREAL_MAX));

    /// AllPairs Query.
    
    template <typename TREE, typename QUERY>
    static inline void          allPairs(const TREE& tree, QUERY& query);

    /// AllPairs Query.
    
    template <typename TREE_A, typename TREE_B, typename QUERY>
    static inline void          allPairs(const TREE_A& treeA, const TREE_B& treeB, QUERY& query);

    /// AllPairs Query.
    
    template <typename TREE_A, typename TREE_B, typename QUERY>
    static inline void          allPairs(const TREE_A& treeA, const TREE_B& treeB, const hkTransform& AtoB, QUERY& query, hkReal marginA = 0.0f, hkReal marginB = 0.0f);

    /// AllLeavesPairs Query.
    
    template <typename TREE, typename PAIR>
    static inline int           allLeavesPairs(const TREE& tree, hkArray<PAIR>& pairs);

    /// AllLeavesPairs Query.
    
    template <typename TREE_A, typename TREE_B, typename PAIR>
    static inline int           allLeavesPairs(const TREE_A& treeA, const TREE_B& treeB, hkArray<PAIR>& pairs);

    /// AABB exist.
    
    template <typename TREE>
    static inline hkBool        aabbExist(const TREE& tree, const hkAabb& aabb, typename TREE::Index& leafOut);

    /// Point exist.
    
    template <typename TREE>
    static inline hkBool        pointExist(const TREE& tree, const hkVector4& point, typename TREE::Index& leafOut);

    /// Locate the first leaf containing a point.
    
    template <typename TREE>
    static inline hkBool        locatePoint(const TREE& tree, const hkVector4& point, typename TREE::Index& leafOut);

    /// Check if an AABB overlaps at least one leaf.
    template <typename TREE>
    static inline hkBool        overlapExist(const TREE& tree, const hkAabb& aabb, _Inout_opt_ typename TREE::Index* leafOut = HK_NULL);

private:

    #include <Geometry/Internal/Algorithms/TreeQueries/hkcdTreeQueriesInternals.inl>

};

//
// Tree queries stack policies
//

#include <Geometry/Internal/Algorithms/TreeQueries/hkcdTreeQueriesStacks.h>

//
// Predefined queries specializations.
//

/// Fixed stack queries with stackless fall-back for unary queries.
typedef hkcdTreeQueries<hkcdTreeQueriesStacks::FixedCpu, 64, hkcdTreeQueriesFlags::ALLOW_STACKLESS>             hkcdFixedStackTreeQueries;

/// Fixed stack queries with NO stackless fall-back for unary queries.
typedef hkcdTreeQueries<hkcdTreeQueriesStacks::FixedCpu, 64, hkcdTreeQueriesFlags::NONE>                        hkcdForcedStackTreeQueries;

/// Dynamic stack queries.
typedef hkcdTreeQueries<hkcdTreeQueriesStacks::Dynamic, 64, hkcdTreeQueriesFlags::NONE>                         hkcdDynamicStackTreeQueries;

#include <Geometry/Internal/Algorithms/TreeQueries/hkcdTreeQueries.inl>

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
