// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_REFLECTION_PARSER_EXCLUDE_FILE

#pragma once

// this: #include <Geometry/Internal/Algorithms/RayCast/hkcdRayCastTriangle.h>

#include <Geometry/Collide/Types/hkcdRay.h>
#include <Geometry/Collide/Types/hkcdRayBundle.h>

struct hkcdRayBundle;


/// Internal triangle functions, read the function comments carefully before using these functions.
namespace hkcdRayCastTriangle
{
    /// Intersect a ray with a triangle.
    /// This function ensures that a ray never passes through a closed mesh, however it might report hits slightly
    /// outside the epsilon boundary of a triangle.
    /// Triangles are considered two sided by default, but can be seen as one sided by setting the query flags.
    ///
    /// Usage: This function is optimized for performance, not for usability! So please note that:
    ///  - The endpoint of the ray is calculated by ray.start + ray.directon * fractionInOut
    ///  - fractionInOut must be initialized (e.g. by ray.getFraction()).
    ///  - ray.getFraction() is not used.
    ///  - fractionInOut and normalOut are modified regardless whether you have a hit or not.
    ///    So you need to make sure that you only modify the input fraction of the next ray-triangle test if a
    ///    previous test had a hit.
    /// Quality:
    ///  - The ray will not go through a closed mesh.
    ///  - If the start or end point is on a triangle, the ray might or might not hit.
    ///    This is due to the fact that we are working with floating point numbers and it does not make sense
    ///    to special case rare esoteric cases (like axis aligned triangles) which would make this code slower
    ///    for general usage.
    ///  - 2 rays forming a segmented path might go through a mesh if the first ray's end point and the second
    ///    ray's start point are both on a triangle.
    ///  - To account for numerical drift when you have tiled meshes, this function also inflates the triangles by
    ///    hkcdRayCastTriangle::g_tolerance (set to 0.01f by default).
    ///
    /// Note about the implementation: The implementation calculates the perfect epsilon of how to expand the edges
    /// so we do not miss collisions. However if 2 edges form a very sharp angle (as seen from the ray start point),
    /// the intersection of the 2 expanded edges (as seen from the ray start point) might be slightly outside the triangle.
    /// Since the expanded edges are used to decide whether the ray hits, the ray might hit slightly outside the triangle.
    /// Example: Your raylength is 100km, so we roughly expand the edges by 2.0f * 100km * FLOAT_EPS(=1e-7) = 2cm.
    ///          If our triangle corner is very sharp as seen from the ray start point (~1degree), the ray might hit the
    ///          triangle easily 20 or more centimeters outside the triangle.
    HK_EXPORT_COMMON hkcdRayCastResult HK_CALL safeUsingDynamicTolerance(
        const hkcdRay& ray, hkVector4Parameter vA, hkVector4Parameter vB, hkVector4Parameter vC,
        hkFlags<hkcdRayQueryFlags::Enum,hkUint32> flags,
        hkSimdReal& fractionInOut, hkVector4& nonUnitNormalOut );

    /// Same as safeUsingDynamicTolerance, but different interface, this code uses:
    /// verts[0], verts[1+triIndex], verts[2+triIndex]
    HK_EXPORT_COMMON hkcdRayCastResult HK_CALL safeUsingDynamicTolerance2(
        const hkcdRay& ray, const hkVector4* verts, int triIndex, hkFlags<hkcdRayQueryFlags::Enum, hkUint32> flags,
        hkSimdReal& fractionInOut, hkVector4& nonUnitNormalOut );

    /// Same as safeUsingDynamicTolerance, but using a quad instead of a triangle
    HK_EXPORT_COMMON hkcdRayCastResult HK_CALL safeUsingDynamicToleranceQuad(
        const hkcdRay& ray, const hkVector4* quad,  hkFlags<hkcdRayQueryFlags::Enum, hkUint32> flags,
        hkSimdReal& fractionInOut, hkVector4& nonUnitNormalOut );

    /// Intersect a ray with a triangle.
    /// Same as safeUsingDynamicTolerance() but without dynamic tolerance.
    /// Good numerical accuracy, about 30% faster than safeUsingDynamicTolerance(), but might pass through a mesh.
    /// Also the likelihood of hitting a triangle outside its area is reduced by factor of 10.
    /// Also this version ignores hkcdRayCastTriangle::g_tolerance
    HK_EXPORT_COMMON hkcdRayCastResult HK_CALL fastUsingZeroTolerance(
        const hkcdRay& ray, hkVector4Parameter vA, hkVector4Parameter vB, hkVector4Parameter vC,
        hkFlags<hkcdRayQueryFlags::Enum,hkUint32> flags,
        hkSimdReal& fractionInOut, hkVector4& nonUnitNormalOut );

    /// same as fastUsingZeroTolerance but for quads
    HK_EXPORT_COMMON hkcdRayCastResult HK_CALL fastUsingZeroToleranceQuad(
        const hkcdRay& ray, hkVector4Parameter vA, hkVector4Parameter vB, hkVector4Parameter vC,  hkVector4Parameter vD,
        hkFlags<hkcdRayQueryFlags::Enum, hkUint32> flags,
        hkSimdReal& fractionInOut, hkVector4& nonUnitNormalOut );

    /// DEPRECATED: Intersects a ray bundle with the triangle. Not so fast and bad epsilon behavior.
    HK_EXPORT_COMMON hkVector4Comparison HK_CALL hkcdSegmentBundleTriangleIntersect(
        const hkcdRayBundle& rayBundle, hkVector4Parameter vA, hkVector4Parameter vB, hkVector4Parameter vC,
        hkSimdRealParameter tolerance,
        hkFourTransposedPoints& normalsOut, hkVector4& fractionsInOut );

    /// Intersect a sphere (ray + radius) with a triangle.
    HK_EXPORT_COMMON hkBool32 HK_CALL sphereCastTriangle(
        const hkcdRay& ray, hkSimdRealParameter radius,
        hkVector4Parameter vA, hkVector4Parameter vB, hkVector4Parameter vC,
        _Inout_ hkSimdReal* fractionOut, _Inout_ hkVector4* planeOut );

    /// Inline implementation of safeUsingDynamicTolerance() and fastZeroTolerance(), only available if you have access to the .inl file
    template<bool ENABLE_DYNAMIC_TOLERANCE, bool SORT_EDGES, bool ENABLE_BEVEL_PLANE_CHECK>
    HK_INLINE hkcdRayCastResult HK_CALL _rayTriangleImpl(
        const hkcdRay& ray, hkVector4Parameter vA, hkVector4Parameter vB, hkVector4Parameter vC,
        hkFlags<hkcdRayQueryFlags::Enum,hkUint32> flags, hkSimdReal* fractionInOut, hkVector4* nonUnitNormalOut );

    /// Inline implementation of safeUsingDynamicTolerance() and fastZeroTolerance(), only available if you have access to the .inl file
    template<bool ENABLE_DYNAMIC_TOLERANCE, bool SORT_EDGES, bool ENABLE_BEVEL_PLANE_CHECK>
    HK_INLINE hkcdRayCastResult HK_CALL _rayQuadImpl(
        const hkcdRay& ray, hkVector4Parameter vA, hkVector4Parameter vB, hkVector4Parameter vC, hkVector4Parameter vD,
        hkFlags<hkcdRayQueryFlags::Enum, hkUint32> flags, hkSimdReal* fractionInOut, hkVector4* nonUnitNormalOut );

    /// When a ray is cast against triangles using safeUsingDynamicTolerance(), the triangles are virtually inflated
    /// by this tolerance. This might be necessary if your have subdivided a mesh into multiple bodies and you get small
    /// gaps between neighboring meshes due to numerical inaccuracies.
    /// The default value is 1cm, which is good for tiled meshes up to 100km away from the origin.
    /// If you have a perfect mesh, you can set this value to 0.
    HK_EXPORT_COMMON extern hkSimdReal g_tolerance;

    /// The default value for g_tolerance.
    HK_EXPORT_COMMON extern const hkSimdReal g_defaultTolerance;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
