// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_REFLECTION_PARSER_EXCLUDE_FILE

#pragma once

#include <Geometry/Collide/Types/hkcdVertex.h>
#include <Geometry/Collide/Types/hkcdRay.h>

// this include path
// #include <Geometry/Internal/Algorithms/Gsk/hkcdGskData.h>

/// Utility function to calculate the distance between 2 convex objects.
/// This function is very robust and can handle penetrating situations as well.
namespace hkcdGsk
{
        /// The cache for GSK. Stores the dimensions of the current simplices for both shapes A and B, together with the vertex indices of each simplex.
    struct HK_EXPORT_COMMON Cache
    {
        typedef hkUint8 VertexId;

        HK_INLINE void setDims(int dimA, int dimB, int userData) { m_dimAb = (hkUint8)((dimA<<6)|dimB|(userData<<2)); }
        HK_INLINE int getDimA() const { return m_dimAb>>6; }
        HK_INLINE int getDimB() const { return m_dimAb&0x3; }
        HK_INLINE hkUchar dimBisNot3_orPenetrating() const { return m_dimAb ^ ((1<<6)|(0<<2)|(3<<0));  }

        void init()
        {
            union U { hkUint8 u8[4]; hkUint32 u32; };
            U* u = reinterpret_cast<U*>(&m_vertexIds);
            u->u32 = 0;
            setDims(1,1,0);
        }

        HK_INLINE void clearStatus()
        {
            // Clear status bits 2-4
            m_dimAb = m_dimAb & 0xE3;
        }

        _Ret_notnull_ const VertexId* getVertexIdsA() const { return m_vertexIds; }
        _Ret_notnull_ const VertexId* getVertexIdsB() const { return &m_vertexIds[getDimA()]; }

        VertexId m_vertexIds[4];    // This should be aligned on a 4 byte boundary, not aligning it here because of enforced padding
        hkUint8 m_dimAb;            // dimA are bits 6-7, dimB are bits 0-1, bits 2-4 are the GetClosestPointStatus
    };


    // Closest point input and output


        /// Low level GJK and GSK output data
    struct GetClosestPointInput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(0, GetClosestPointInput);
        GetClosestPointInput()
            :   m_flags(0)
            ,   m_terminationGrowthRate(1.4f)
        {
            m_coreCollisionTolerance = hkSimdReal::getConstant(HK_QUADREAL_MAX);
        }

        enum Flags
        {
            DISABLE_PENETRATION_CHECK = 1 << 0,     ///< if set, no penetration checks will be performed
            ON_4_DIM_EARLY_OUT = 1 << 1,                ///< if set, the function returns if the sum of the dimension is 4. This can be used for a repeated GSK call during linear cast
            DISABLE_EARLY_OUT = 1<<2,           /// if you are interested that the caches are properly updated, you can set this flag. this will ensure that the cache will contain the closest distance in case of result=STATUS_PD_DISTANCE_TOO_BIG
        };

        hkTransform     m_bTa;                      ///< transforms from a to b space, not initialized in the constructor!

                                                    /// The collision tolerance plus shape radii
        hkSimdReal      m_coreCollisionTolerance;   ///< If the objects are further away than this, no results will be calculated

        hkFlags<Flags, hkUint32> m_flags;

        hkReal          m_terminationGrowthRate;    ///< Internal factor to indicate how quickly internal epsilons are increased each iteration.
    };

        /// Low level GJK and GSK output data
    struct GetClosestPointOutput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(0, GetClosestPointOutput);

        hkSimdReal getDistance() const { return m_distance; }

        hkSimdReal m_distance;
        hkVector4 m_normalInA;  ///< Normal from B to A in A.
        hkVector4 m_pointAinA;
    };

        /// GetClosestPoint output status.
    enum GetClosestPointStatus
    {
        // General success indication
        STATUS_OK_NOT_PENTRATING = 0,   ///< The objects are NOT penetrating and the returned distance is OK
        STATUS_OK_PENETRATING = 1,      ///< The objects are penetrating and the returned distance is OK

        // Penetration depth algorithm errors
        STATUS_PD_HEURISTIC_SAMPLING,   ///< The objects are probably penetrating, but GSK ran into internal numerical issues, a fallback algorithms is used, so the output is an approximation at best.
        STATUS_PD_OUT_OF_MEMORY,        ///< The objects are so deeply penetrating and have many vertices, so that the algorithms runs out of memory. There is still a valid output, but it is an approximation at best.

        STATUS_OK_FLAG,                 ///< Not a return value, but indicates the end of all OK statii

        STATUS_PD_DISTANCE_TOO_BIG,     ///< The output only has a normal and a distance , which is bigger than the collision tolerance.
        STATUS_PENETRATING,             ///< Objects are penetrating but no further details.
        STATUS_PD_UNSOLVABLE,           ///< No solution because of numerical issues, this is an internal flag and NOT returned by hkcdGsk::getClosestPoint().
    };



    // Linear cast input and output

        /// Linear cast input
    struct LinearCastInput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(0, LinearCastInput);

        /// Constructor. Receives the convex radii of the cast shapes and an epsilon value to
        /// control the stopping criteria.
        HK_INLINE LinearCastInput( hkReal convexRadiusA, hkReal convexRadiusB )
        {
            m_radiusAB = hkSimdReal::fromFloat(convexRadiusA) + hkSimdReal::fromFloat(convexRadiusB);
            m_initialFraction.setZero();
            m_accuracy = hkSimdReal::fromFloat(1e-4f);
            m_maxCastIterations = 20;
        }


        hkVector4 m_from;       ///< Linear cast start point
        hkVector4 m_direction;  ///< Linear cast un-normalized direction, effectively endPoint - m_from
        hkTransform m_bTa;      ///< Transform from A to B

        hkSimdReal m_accuracy;          ///< the accuracy of the cast, the higher the faster
        hkSimdReal m_initialFraction;   ///< jump start linear cast with an initial fraction, typically comes from the broadphase
        hkSimdReal m_radiusAB;
        int m_maxCastIterations;
    };

        /// Linear cast output
    struct LinearCastOutput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(0, LinearCastOutput);

        hkSimdReal m_fractionInOut;     ///< The fraction, must be initialized to 1.0 or a multiple of the input.m_direction.length<3>()
        hkVector4 m_pointAinA;          ///< The hit point on object A when object A is moved to the hitFraction
        hkVector4 m_normalInA;          ///< The surface normal of the hit in A space
        hkBool32 m_hasHit;
    };


        /// Ray cast input
    struct RayCastInput : public LinearCastInput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(0, RayCastInput);

        /// Constructor. Receives the convex radii of the cast shapes and an epsilon value to
        /// control the stopping criteria.
        HK_INLINE RayCastInput(hkReal convexRadiusA)
        :   LinearCastInput(convexRadiusA, 0.0f), m_flags(hkcdRayQueryFlags::NO_FLAGS)
        {
            m_bTa.setIdentity();
        }

        hkFlags<hkcdRayQueryFlags::Enum,hkUint32> m_flags;
    };

    struct RayCastOutput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(0, RayCastOutput);
        HK_INLINE RayCastOutput() {}

        hkVector4 m_normalOut;
        hkSimdReal m_fractionInOut;
    };

    /// Helper class to access shape functions
    typedef void Shape;


    struct GskWorkData
    {
        // data, using dim6 to allow for padding
        hkcdVertex m_verticesA[6];   ///< The supporting vertex of A in A space with vertex index in W
        hkcdVertex m_verticesBinA[6];///< The supporting vertex of B in A space with vertex index in W

    };

    struct GetSupportingVertexOut
    {
        hkcdVertex m_vertexAinA;    ///< The supporting vertex of A in A space with vertex index in W
        hkcdVertex m_vertexBinB;    ///< The supporting vertex of B in B space with vertex index in W
    };

    struct ShapeInterface
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_COLLIDE, hkcdGsk::ShapeInterface );

        virtual void getSupportingVertices(
            _In_ const Shape* shapeA, hkVector4Parameter directionA, _In_ const Shape* shapeB, const hkTransform* bTa,
            GetSupportingVertexOut* HK_RESTRICT output ) const = 0;

        virtual void getPlanes(
            const void* shapeA, const void* shapeB,
            hkVector4 const ** planesAOut, int& numPlanesAOut,
            hkVector4 const ** planesBOut, int& numPlanesBOut) const
        {
            // default implementation, means it is not implements
            numPlanesAOut = 0;
            numPlanesBOut = 0;
        }
    };
}

#include <Geometry/Internal/Algorithms/Gsk/hkcdGskData.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
