// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>

#include <Geometry/Collide/Algorithms/Distance/hkcdDistancePointLine.h>
#include <Geometry/Collide/DataStructures/Planar/Predicates/hkcdPlanarGeometryPredicates.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

#define DEBUG_LOG_IDENTIFIER "test.geo.collide.planarpredicate"
#include <Common/Base/System/Log/hkLog.hxx>

#define NB_PLANES 1000
#define NB_POINTS 1000

// It is assumed that all point are unique
static void generatePointsPool(hkIntVector* points)
{
    hkPseudoRandomGenerator rng(13);
    const int nbBits = hkcdPlanarGeometryPrimitives::NumBitsVertex::NUM_BITS;
    const int shiftValue = (33 - nbBits);

    for ( int p = 0 ; p < NB_POINTS ; p++)
    {
        // Get randomly placed vertices
        for (int k = 0 ; k < 3 ; k++)
        {
            points[p].setComponent(k, int(rng.getRand32() >> shiftValue));
            if ( rng.getRandReal01() < 0.5f )
            {
                points[p].setComponent(k, points[p].getComponent(k) | 0xffd00000);  // Negative value
            }
        }
        points[p].setComponent<3>(0);
    }
}

/*
int predicateSpeedTest(hkcdPlanarGeometryPredicates::Plane* planes)
{
    const int numiterations = 1000000;

    hkPseudoRandomGenerator rng(13);

    hkVector4d fvd;
    hkVector4d fvdSum;
    fvdSum.setZero();

    Log_Info( "Starting approximateIntersectionFast time measure ({}) iterations...", numiterations );
    hkStopwatch timer;
    timer.start();
    for (int t = 0 ; t < numiterations ; t++)
    {
        hkcdPlanarGeometryPredicates::Plane pla[3];
        pla[0] = planes[int(rng.getRandRange(0, NB_PLANES))];
        pla[1] = planes[int(rng.getRandRange(0, NB_PLANES))];
        pla[2] = planes[int(rng.getRandRange(0, NB_PLANES))];
        hkcdPlanarGeometryPredicates::approximateIntersectionFast(pla, fvd);
        fvdSum.add(fvd);
    }
    timer.stop();
    Log_Info( "Predicate average time: {}s.", timer.getElapsedSeconds() );
    timer.reset();

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//  Log_Info( "Starting approximateIntersection time measure ({}) iterations...", numiterations );
//  timer.start();
//  for (int t = 0 ; t < numiterations ; t++)
//  {
//      hkcdPlanarGeometryPredicates::Plane pla[3];
//      int ind[3];
//      for (int k = 0 ; k < 3 ; k++) ind[k] = int(rng.getRandRange(0, NB_PLANES));
//      if ( ind[0] == ind[1] || ind[0] == ind[2] || ind[1] == ind[2] )
//      {
//          t--;
//          continue;
//      }
//      for (int k = 0 ; k < 3 ; k++) pla[k] = planes[ind[k]];
//      hkIntVector fiv;
//      hkcdPlanarGeometryPredicates::approximateIntersection(pla, fiv);
//  }
//  timer.stop();
//  Log_Info( "Predicate average time: {}s.", timer.getElapsedSeconds() );
//  timer.reset();

    return 0;
}
*/

static hkResult getRandomPlane(hkPseudoRandomGenerator& rng, const hkIntVector* points, hkcdPlanarGeometryPredicates::Plane& planeOut)
{
    // Choose 3 unique points
    int ind[3];
    for (int k = 0 ; k < 3 ; k++)
    {
        // check not already chosen
        bool alreadyChosen;
        do
        {
            ind[k] = int(rng.getRandRange(0, NB_POINTS - 1));
            alreadyChosen = false;
            for (int l = 0 ; l < k ; l++)
            {
                if ( ind[l] == ind[k] )
                {
                    alreadyChosen = true;
                    break;
                }
            }
        } while ( alreadyChosen );
    }

    // Build a plane out of them
    hkInt64Vector4 iN;
    hkInt128 iO;
    HK_RETURN_IF_FAILED( hkcdPlanarGeometryPrimitives::computePlaneEquation(points[ind[0]], points[ind[1]], points[ind[2]], iN, iO) );

    // set plane
    planeOut.setExactEquation(iN, iO);
    return HK_SUCCESS;
}

hkResult get3PlaneCrossingAtRandomPoint(hkPseudoRandomGenerator& rng, const hkIntVector* points, int& crossingPtId, hkcdPlanarGeometryPredicates::Plane* planesOut)
{
    // Choose a ref point (ind 0), and build 3 random planes crossing at this points
    int ind[7];
    for (int k = 0 ; k < 7 ; k++)
    {
        // check not already chosen
        bool alreadyChosen;
        do
        {
            ind[k] = int(rng.getRandRange(0, NB_POINTS - 1));
            alreadyChosen = false;
            for (int l = 0 ; l < k ; l++)
            {
                if ( ind[l] == ind[k] )
                {
                    alreadyChosen = true;
                    break;
                }
            }
        } while ( alreadyChosen );
    }

    for ( int p = 0 ; p < 3 ; p++)
    {
        hkInt64Vector4 iN;
        hkInt128 iO;
        HK_RETURN_IF_FAILED( hkcdPlanarGeometryPrimitives::computePlaneEquation(points[ind[0]], points[ind[p*2 + 1]], points[ind[p*2 + 2]], iN, iO) );

        // set plane
        planesOut[p].setExactEquation(iN, iO);
    }

    crossingPtId = ind[0];
    return HK_SUCCESS;
}

hkResult get2PairsOfPlaneWithSameIntersection(hkPseudoRandomGenerator& rng, const hkIntVector* points, hkcdPlanarGeometryPredicates::Plane* pair1, hkcdPlanarGeometryPredicates::Plane* pair2)
{
    // Choose a ref point (ind 0), and build 3 random planes crossing at this points
    int ind[6];
    for (int k = 0 ; k < 6 ; k++)
    {
        // check not already chosen
        bool alreadyChosen;
        do
        {
            ind[k] = int(rng.getRandRange(0, NB_POINTS - 1));
            alreadyChosen = false;
            for (int l = 0 ; l < k ; l++)
            {
                if ( ind[l] == ind[k] )
                {
                    alreadyChosen = true;
                    break;
                }
            }
        } while ( alreadyChosen );
    }

    for ( int p = 0 ; p < 2 ; p++)
    {
        hkInt64Vector4 iN;
        hkInt128 iO;
        HK_RETURN_IF_FAILED( hkcdPlanarGeometryPrimitives::computePlaneEquation(points[ind[0]], points[ind[1]], points[ind[p + 2]], iN, iO) );

        // set plane
        pair1[p].setExactEquation(iN, iO);
    }

    for ( int p = 0 ; p < 2 ; p++)
    {
        hkInt64Vector4 iN;
        hkInt128 iO;
        HK_RETURN_IF_FAILED( hkcdPlanarGeometryPrimitives::computePlaneEquation(points[ind[0]], points[ind[1]], points[ind[p + 4]], iN, iO) );

        // set plane
        pair2[p].setExactEquation(iN, iO);
    }
    return HK_SUCCESS;
}

int approxPredicatesValidityTest(const hkIntVector* points)
{
    int numIterationFactor = 40;

    int numiterations = 10 * numIterationFactor;
    hkPseudoRandomGenerator rng(13);
    hkVector4d half;        half.setAll(hkSimdDouble64_Inv2);

    double maxErrDist = 0.0;
    double maxAbsErr = 0.0;
    int maxManErr = 0;
    hkVector4d fvd, fvdRef;
    hkIntVector fiv;
    hkIntVector planeIds;
    planeIds.set(0, 1, 2, 3);

    Log_Info( "Checking approximateIntersectionFast precision ({} iterations)", numiterations );

    for (int t = 0 ; t < numiterations ; t++)
    {
        hkcdPlanarGeometryPredicates::Plane planes[3];
        int refPtId;
        get3PlaneCrossingAtRandomPoint(rng, points, refPtId, planes);

        // Compute approx crossing point value
        hkcdPlanarGeometryPredicates::approximateIntersectionFast(planes, fvd);
        points[refPtId].convertS32ToF32(fvdRef);

        // Manhattan error
        hkIntVector vI;
        hkVector4d vdW = fvd;   vdW.add(half);
        vI.setConvertF32toS32(vdW);
        vI.setSubS32(points[refPtId], vI);
        vI.setAbsS32(vI);
        int manErr = hkMath::max2(hkMath::max2(vI.getComponent<0>(), vI.getComponent<1>()), vI.getComponent<2>());
        if ( manErr > maxManErr )
        {
            maxManErr = manErr;
        }

        // absolute error
        const double abserr = fvdRef.distanceTo(fvd).getReal();
        if ( abserr > maxAbsErr )
        {
            maxAbsErr = abserr;
        }
        // Failure test
        if ( abserr > TOL_FAST_PREDICATE )
        {
            Log_Error( "\tFailed" );
            return -1;
        }

        // Relative error - measure distance to initial planes (the exact distance is 0)
        for (int k = 0 ; k < 3 ; k++)
        {
            const hkVector4d& plane         = planes[k].getApproxEquation();
            const hkSimdDouble64& dotRes    = fvd.dot<3>(plane) + plane.getComponent<3>();
            hkSimdDouble64 absDotRes;       absDotRes.setAbs(dotRes);
            const double err = absDotRes.getReal();
            if ( err > maxErrDist )
            {
                maxErrDist = err;
            }
        }
    }
    Log_Info( "\tOk (Max relative error is {} - max absolute error is {} - max Manhattan error is {})", maxErrDist, maxAbsErr, maxManErr );

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    numiterations = 10 * numIterationFactor;
    Log_Info( "Checking approximateIntersection ({} iterations)", numiterations );
    for (int t = 0 ; t < numiterations ; t++)
    {
        hkcdPlanarGeometryPredicates::Plane planes[3];
        int refPtId;
        get3PlaneCrossingAtRandomPoint(rng, points, refPtId, planes);

        // Compute approx crossing point value
        hkcdPlanarGeometryPredicates::approximateIntersection(planes, fiv);

        // Failure
        if ( !fiv.equalS32(points[refPtId]).allAreSet<hkVector4ComparisonMask::MASK_XYZ>() )
        {
            Log_Error( "\tFailed" );
            return -1;
        }
    }
    Log_Info( "\tOk" );

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    numiterations = 10 * numIterationFactor;
    Log_Info( "Checking fast orientation predicate ({} iterations)", numiterations );
    for (int t = 0 ; t < numiterations ; t++)
    {
        hkcdPlanarGeometryPredicates::Plane planes[3];
        int refPtId;
        get3PlaneCrossingAtRandomPoint(rng, points, refPtId, planes);

        // On plane test
        hkcdPlanarGeometryPredicates::approximateIntersectionFast(planes, fvd);
        const int iPlaneId = int(rng.getRandRange(0, 3));
        if ( hkcdPlanarGeometryPredicates::orientation<1>(fvd, planes[0], planes[1], planes[2], planes[iPlaneId], planeIds, HK_NULL) != hkcdPlanarGeometryPredicates::ON_PLANE)
        {
            Log_Error( "\tFailed (on plane test)" );
            return -1;
        }

        // Random plane test
        hkcdPlanarGeometryPredicates::Plane planeD;
        if (getRandomPlane(rng, points, planeD).isFailure())
        {
            Log_Error("\tFailed (getRandomPlane)");
            return -1;
        }
        const hkcdPlanarGeometryPredicates::Orientation oriFast = hkcdPlanarGeometryPredicates::orientation<1>(fvd, planes[0], planes[1], planes[2], planes[iPlaneId], planeIds, HK_NULL);
        const hkcdPlanarGeometryPredicates::Orientation oriRef  = hkcdPlanarGeometryPredicates::orientation<1>(planes[0], planes[1], planes[2], planes[iPlaneId], planeIds, HK_NULL);
        if ( oriFast != oriRef )
        {
            Log_Error( "\tFailed (regular query)" );
            return -1;
        }
    }
    Log_Info( "\tOk" );

    // VS2012 Durango w/AVX enabled crashes with 'illegal instruction'
#if !((defined HK_PLATFORM_DURANGO) && (_MSC_VER == 1700))
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    numiterations = 10 * numIterationFactor;
    Log_Info( "Checking approximateEdgeDirectionFast precision ({} iterations)", numiterations );
    for (int t = 0 ; t < numiterations ; t++)
    {
        hkcdPlanarGeometryPredicates::Plane planes1[2];
        hkcdPlanarGeometryPredicates::Plane planes2[2];
        get2PairsOfPlaneWithSameIntersection(rng, points, planes1, planes2);

        // On plane test
        hkIntVector edgeDir1, edgeDir2;
        hkcdPlanarGeometryPredicates::approximateEdgeDirectionFast(planes1[0], planes1[1], edgeDir1);
        hkcdPlanarGeometryPredicates::approximateEdgeDirectionFast(planes2[0], planes2[1], edgeDir2);

        if ( !edgeDir1.equalS32(edgeDir2).allAreSet<hkVector4ComparisonMask::MASK_XYZ>() )
        {
            Log_Error( "\tFailed (Iteration {} - {:,3} vs. {:,3})", t, edgeDir1, edgeDir2 );
            return -1;
        }
    }
    Log_Info( "\tOk" );
#endif
    return 0;
}

int predicateSpeedTest()
{
    // To represent a more real-life case, generate points in a reduced aabb
    hkLocalBuffer<hkIntVector> points(NB_POINTS);
    hkPseudoRandomGenerator rng(13);
    {
        const int nbBits = hkcdPlanarGeometryPrimitives::NumBitsVertex::NUM_BITS;
        const int shiftValue = (32 + 12 - nbBits);

        for ( int p = 0 ; p < NB_POINTS ; p++)
        {
            // Get randomly placed vertices
            for (int k = 0 ; k < 3 ; k++)
            {
                points[p].setComponent(k, int(rng.getRand32() >> shiftValue));
            }
            points[p].setComponent<3>(0);
        }
    }

    // Get a pool of planes
    hkLocalBuffer<hkcdPlanarGeometryPredicates::Plane> planes(3*NB_POINTS);
    hkLocalBuffer<hkVector4d> viz(NB_POINTS);
    for (int p = 0, pi = 0 ; p < NB_POINTS ; p++, pi += 3)
    {
        int refPtId;
        get3PlaneCrossingAtRandomPoint(rng, points.begin(), refPtId, &planes[pi]);
        hkcdPlanarGeometryPredicates::Plane planesSrc[3];
        planesSrc[0] = planes[pi + 0];
        planesSrc[1] = planes[pi + 1];
        planesSrc[2] = planes[pi + 2];
        hkcdPlanarGeometryPredicates::approximateIntersectionFast(planesSrc, viz[p]);
    }

    // Check fast predicate timings
    int numiterations = 10000;
    hkIntVector planeIds;
    int nbOnP = 0;
    hkStopwatch timer;
    timer.start();
    for (int t = 0 ; t < numiterations ; t++)
    {
        const int pi = int(rng.getRandRange(0, NB_POINTS - 1));
        const int pis = pi*3;
        const int iPlaneId = int(rng.getRandRange(0, 3*NB_POINTS - 1));
        planeIds.set(pis, pis + 1, pis + 2, iPlaneId);
        const hkcdPlanarGeometryPredicates::Orientation ori = hkcdPlanarGeometryPredicates::orientation<1>(viz[pi], planes[pis], planes[pis + 1], planes[pis + 2], planes[iPlaneId], planeIds, HK_NULL);
        if ( ori == hkcdPlanarGeometryPredicates::ON_PLANE )
        {
            nbOnP++;
        }
    }
    timer.stop();
    Log_Info( "\t Fast orientation predicate total time: {}s. ({})", timer.getElapsedSeconds(), nbOnP );
    timer.reset();

    // Check legacy predicate timings
    nbOnP = 0;
    timer.start();
    for (int t = 0 ; t < numiterations ; t++)
    {
        const int pis = int(rng.getRandRange(0, NB_POINTS - 1))*3;
        const int iPlaneId = int(rng.getRandRange(0, 3*NB_POINTS - 1));
        planeIds.set(pis, pis + 1, pis + 2, iPlaneId);
        const hkcdPlanarGeometryPredicates::Orientation ori = hkcdPlanarGeometryPredicates::orientation<1>(planes[pis], planes[pis + 1], planes[pis + 2], planes[iPlaneId], planeIds, HK_NULL);
        if ( ori == hkcdPlanarGeometryPredicates::ON_PLANE )
        {
            nbOnP++;
        }
    }
    timer.stop();
    Log_Info( "\t Legacy orientation predicate total time: {}s. ({})", timer.getElapsedSeconds(), nbOnP );
    timer.reset();

    return 0;
}

int PlanarPredicates_main()
{
    // create points
    hkIntVector points[NB_POINTS];
    generatePointsPool(points);

    // Run tests
    predicateSpeedTest();

    approxPredicatesValidityTest(points);

    return 0;
}

HK_TEST_REGISTER( PlanarPredicates_main, "Fast", "Geometry/Test/UnitTest/Collide/", __FILE__ );

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
