// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

// this: #include <Geometry/Collide/Types/hkcdRay.h>

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/hkHandle.h>

#if defined(HK_REAL_IS_DOUBLE)
#   define HK_ACC_RAYCAST HK_ACC_FULL
#else
#   define HK_ACC_RAYCAST HK_ACC_MID
#endif


/// Represents the origin, direction and length of a ray.
/// Additionally, a fraction (default 1.0f) is stored in the direction W component.
/// Note that m_origin:W is free for user data.
struct HK_EXPORT_COMMON hkcdRay
{
    HK_INLINE hkcdRay( const hkcdRay& other ) { m_origin = other.m_origin; m_direction = other.m_direction; m_invDirection = other.m_invDirection; }
    HK_ALWAYS_INLINE hkcdRay() {}

    /// Set from start and end points.
    HK_INLINE   void setEndPoints(
        hkVector4Parameter start, hkVector4Parameter end,
        hkSimdRealParameter fraction = hkSimdReal::getConstant( HK_QUADREAL_1 ) );

    /// Set from origin and direction.
    HK_INLINE   void setOriginDirection(
        hkVector4Parameter origin, hkVector4Parameter direction,
        hkSimdRealParameter fraction = hkSimdReal::getConstant( HK_QUADREAL_1 ) );

    /// Computes the ray end point.
    HK_INLINE void getEndPoint( hkVector4& vEndPoint ) const;

    /// Get direction.
    HK_ALWAYS_INLINE const hkVector4& getDirection() const { return m_direction; }

    /// Get inverse direction.
    HK_ALWAYS_INLINE const hkVector4& getInverseDirection() const { return m_invDirection; }

    /// Get fraction.
    HK_ALWAYS_INLINE hkSimdReal getFraction() const { return m_direction.getComponent<3>(); }

    /// Set fraction.
    HK_ALWAYS_INLINE void setFraction( hkSimdRealParameter fraction ) { m_direction.setComponent<3>( fraction ); }

    HK_INLINE void setTransformedRay(const hkTransform& transform, const hkcdRay& ray);

    HK_INLINE void setTransformedInverseRay(const hkTransform& transform, const hkcdRay& ray);

    /// Origin of the ray.
    hkVector4 m_origin;

    /// Direction of the ray. Effectively the (endPoint - startPoint) vector (and length as |direction|).
    hkVector4 m_direction;

    /// Precomputed direction reciprocal with +/-Inf == 0.
    hkVector4 m_invDirection;
};


/// A set of flags used to customize ray casting behavior.
namespace hkcdRayQueryFlags
{
    enum Enum
    {
        /// No flags. Implies default behavior:
        ///  - Triangles are treated as two sided.
        ///  - No inside hits are reported.
        NO_FLAGS = 0,

        /// If set: In triangle based shapes (e.g: mesh, height field), rays will not hit triangle back faces.
        /// Back faces are considered to be those with clockwise vertex winding.
        DISABLE_BACK_FACING_TRIANGLE_HITS = 1 << 0,

        /// If set: In triangle based shapes (e.g: mesh, height field), rays will not hit triangle front faces.
        /// Front faces are considered to be those with counter-clockwise vertex winding.
        DISABLE_FRONT_FACING_TRIANGLE_HITS = 1 << 1,

        /// Rays starting inside convex shapes will by default not hit the shape,
        /// this flag can enable these 'inside' hits.
        /// The normal returned for inside hits will be outward facing and
        /// hkcdRayCastResult::BACK_FACE_HIT | hkcdRayCastResult::INSIDE_HIT returned.
        /// Note: For shapes created using hknpConvexShape::createFromVertices(), this flag only works when
        /// hknpConvexShape::BuildConfig.m_buildFaces is set to true.
        ENABLE_INSIDE_HITS = 1 << 2,
    };

    /// Sets out to true if a given flag is set.
    static HK_INLINE void isFlagSet( hkFlags<Enum, hkUint32> flags, Enum flag, hkVector4Comparison& out );
};


/// The result of a raycast (synchronized with hkcdRayQueryFlags).
struct hkcdRayCastResult
{
    enum Enum
    {
        /// No hit.
        NO_HIT = 0,

        /// Back face hit.
        /// Note: In this case the normal points in the opposite direction of the ray (dot product is negative) unless INSIDE_HIT is set.
        BACK_FACE_HIT = 1 << 0,

        /// Front face/outside hit.
        /// Note: In this case the normal points in the opposite direction of the ray (dot product is negative).
        FRONT_FACE_HIT = 1 << 1,

        /// A convex shape is hit from the inside, that is, the origin of the ray lays inside.
        /// Note: In this case the normal points in the same direction of the ray (dot product is positive).
        INSIDE_HIT = 1 << 2,

        /// Any hit mask.
        ANY_HIT_MASK = BACK_FACE_HIT | FRONT_FACE_HIT | INSIDE_HIT
    };

    // Constructors
    HK_INLINE hkcdRayCastResult() {}
    HK_INLINE hkcdRayCastResult( Enum e ) { m_value = (hkUint32)e; }
    explicit HK_INLINE hkcdRayCastResult( int i ) { m_value = (hkUint32)i; }

    // Queries
    HK_INLINE hkBool32 isHit() const { return m_value; }
    HK_INLINE hkBool32 isInsideHit() const { return m_value & INSIDE_HIT; }
    HK_INLINE hkBool32 isOutsideHit() const { return INSIDE_HIT ^ ( m_value & INSIDE_HIT ); }
    HK_INLINE hkBool32 isFrontSideHit() const { return m_value & FRONT_FACE_HIT; }
    HK_INLINE hkBool32 isBackSideHit() const { return m_value & BACK_FACE_HIT; }

    HK_INLINE hkBool32 operator ==( const hkcdRayCastResult other ) const { return m_value == other.m_value; }
    HK_INLINE hkBool32 operator !=( const hkcdRayCastResult other ) const { return m_value != other.m_value; }

    HK_INLINE hkBool32 operator ==( const hkcdRayCastResult::Enum other ) const { return m_value == (hkUint32)other; }
    HK_INLINE hkBool32 operator !=( const hkcdRayCastResult::Enum other ) const { return m_value != (hkUint32)other; }

    hkUint32 m_value;
};

HK_REFLECT_ENUM( HK_EXPORT_COMMON, hkcdRayCastResult::Enum );

HK_INLINE hkBool32 operator ==( const hkcdRayCastResult::Enum lhs, const hkcdRayCastResult rhs ) { return rhs.m_value == (hkUint32)lhs; }
HK_INLINE hkBool32 operator !=( const hkcdRayCastResult::Enum lhs, const hkcdRayCastResult rhs ) { return rhs.m_value != (hkUint32)lhs; }

HK_INLINE hkOstream& operator<<( hkOstream& stream, const hkcdRayCastResult result );

#include <Geometry/Collide/Types/hkcdRay.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
