// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

/// Oriented bounding box.
/// - Half extents along X, Y and Z are stored in the W components of the rotation matrix columns 0, 1 and 2 respectively.
/// - Flags are store as integer in the W component of the translation.
class HK_EXPORT_COMMON hkcdObb
{
    public:

    HK_DECLARE_CLASS( hkcdObb, Reflect, Pod, New );

    /// Distance result.
    /// Notes:
    /// - The normal is in the space of A.
    /// - The normal is toward the surface of A.
    /// - The point is in the space of A.
    /// - The point is on the surface of A.
    struct DistanceResult
    {
        HK_INLINE int               getCode() const { return m_point_code.getInt16W(); }
        HK_INLINE int               getAxisA() const { return getCode() >> 3; }
        HK_INLINE int               getAxisB() const { return ( getCode() >> 1 ) & 3; }
        HK_INLINE int               getSign() const { return getCode() & 1; }
        HK_INLINE hkSimdReal        getDistance() const { return m_normal_distance.getW(); }
        HK_INLINE const hkVector4&  getNormal() const { return m_normal_distance; }
        HK_INLINE const hkVector4&  getPoint() const { return m_point_code; }

        hkVector4   m_normal_distance;
        hkVector4   m_point_code;
    };

    /// Flags.
    enum Flags
    {
        FLAG_X_EXTENT_IS_ZERO           =   0x01,
        FLAG_Y_EXTENT_IS_ZERO           =   0x02,
        FLAG_Z_EXTENT_IS_ZERO           =   0x04,
        FLAG_ORIENTATION_IS_IDENTITY    =   0x08,
        FLAG_POSITION_IS_ORIGIN         =   0x10
    };

    /// Set to empty.
    HK_INLINE void              setEmpty();

    /// Returns <true> if empty, <false> otherwise.
    HK_INLINE bool              isEmpty() const;

    /// Returns the flags, see Flags.
    HK_INLINE int               getFlags() const;

    /// Set from points, return a value representing the match with 1 for perfect match and 0 for no match.
    hkReal                      set( const hkVector4* points, int numPoints );

    /// Set from points, return a value representing the match with 1 for perfect match and 0 for no match.
    hkReal                      set( const struct hkStridedVertices& points );

    /// Set from a convex hull, return a value representing the match with 1 for perfect match and 0 for no match, -1 if the convex hull does not have mass properties.
    hkReal                      set( const class hkgpConvexHull& hull );

    /// Set from a convex hull and a projection plane, return a value representing the match with 1 for perfect match and 0 for no match, -1 if the convex hull does not have mass properties.
    hkReal                      set( const class hkgpConvexHull& hull, const hkVector4& projectionPlane );

    /// Set from AABB.
    void                        set( const hkAabb& aabb );

    /// Set from components.
    void                        set( const hkQuaternion& orientation, const hkVector4& center, const hkVector4& halfExtents );

    /// Set from components.
    void                        set( const hkRotation& orientation, const hkVector4& center, const hkVector4& halfExtents );

    /// Set from components.
    void                        set( const hkTransform& transform, const hkVector4& halfExtents );

    ///
    void                        setCenter( const hkVector4& newCenter );

    /// Update flags.
    HK_INLINE void              updateFlags();

    /// Sets this to be the product of the inverse of t1 by t2. (this = t1^-1 * t2)
    HK_INLINE void              setMulInverseMul( const hkTransform& bTa, const hkcdObb &bTc );

    /// Sets this to be the product of t1 and t2. (this = t1 * t2)
    HK_INLINE void              setMul( const hkTransform &t1, const hkcdObb &t2 );

    /// Get the world AABB.
    void                        getWorldAabb( hkAabb& aabbOut ) const;

    /// Get the world AABB.
    void                        getWorldAabb( const hkTransform& worldTransform, hkAabb& aabbOut ) const;

    /// Compute the world space vertices.
    /// Note: \p verticesOut must be large enough to hold 8 vertices.
    void                        getWorldVertices( hkVector4* verticesOut ) const;

    /// Get the half-extents.
    HK_INLINE void              getHalfExtents( hkVector4& he ) const;

    /// Set the half-extents.
    HK_INLINE void              setHalfExtents( const hkVector4& he );

    /// Get the local AABB.
    HK_INLINE void              getLocalAabb( hkAabb& aabbOut ) const;

    /// Compute the surface area.
    HK_INLINE hkReal            getSurfaceArea() const;

    /// Compute the volume.
    HK_INLINE hkReal            getVolume() const;

    /// Expand (or shrink) the local AABB in each direction by the given amount.
    HK_INLINE void              expandBy( hkReal exp );


    HK_ALWAYS_INLINE const hkRotation& getRotation()    const { return m_transform.getRotation(); }
    HK_ALWAYS_INLINE const hkVector4&  getTranslation() const { return m_transform.getTranslation(); }

    /// Distance between two OBB.
    static void                 computeDistance( const hkcdObb& a, const hkcdObb& b, bool computeContactPoint, DistanceResult* HK_RESTRICT result );

    /// Distance between an OBB and a point. The result normal and point are in AABB space (and not in obb space)
    static void                 computeDistance( const hkcdObb& a, const hkVector4& b, DistanceResult* HK_RESTRICT result );

    /// Intersect, returns <true> if the two OBB intersect.
    static bool                 intersect( const hkcdObb& a, const hkcdObb& b );

    hkTransform m_transform;

    private:

    static const hkVector4Comparison::Mask c_axisMasks[ 3 ];
};

#include <Geometry/Collide/Types/hkcdObb.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
