// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/Types/hkcdObb.h>

#include <Common/Base/Math/Vector/hkVector4Util.h>

#include <Common/Base/Types/Geometry/hkStridedVertices.h>

#include <Common/Internal/GeometryProcessing/hkGeometryProcessing.h>
#include <Common/Internal/GeometryProcessing/ConvexHull/hkgpConvexHull.h>

#include <Geometry/Collide/Algorithms/ClosestPoint/hkcdClosestPointSegmentSegment.h>

namespace
{
    void        computeSupportingSpan( const hkVector4& direction, const hkVector4* points, int numPoints, hkSimdReal& minOut, hkSimdReal& maxOut )
    {
        hkVector4 mins; mins.setAll( hkSimdReal_Max );
        hkVector4 maxs; maxs.setAll( -hkSimdReal_Max );
        int index = 0;
        for ( int n = numPoints - 3; index < n; index += 4 )
        {
            hkVector4 d;
            hkVector4Util::dot3_1vs4( direction, points[ index + 0 ], points[ index + 1 ], points[ index + 2 ], points[ index + 3 ], d );
            mins.setMin( mins, d );
            maxs.setMax( maxs, d );
        }

        minOut = mins.horizontalMin<4>();
        maxOut = maxs.horizontalMax<4>();
        for ( ; index < numPoints; ++index )
        {
            hkSimdReal d = direction.dot<3>( points[ index ] );
            minOut.setMin( minOut, d );
            maxOut.setMax( maxOut, d );
        }
    }

    HK_INLINE void computeWorldAabb( const hkTransform& transform, const hkVector4& he, hkAabb& aabbOut )
    {
        hkVector4 rx; rx.setMul( transform.getRotation().getColumn<0>(), he.getComponent<0>() ); rx.setAbs( rx );
        hkVector4 ry; ry.setMul( transform.getRotation().getColumn<1>(), he.getComponent<1>() ); ry.setAbs( ry );
        hkVector4 rz; rz.setMul( transform.getRotation().getColumn<2>(), he.getComponent<2>() ); rz.setAbs( rz );
        hkVector4 mx; mx.setAdd( rx, ry ); mx.add( rz );
        aabbOut.setFromCenterRadius( transform.getTranslation(), mx );
    }

    hkReal setInternal( hkcdObb& obb, const hkVector4* points, int numPoints, const hkgpConvexHull& hull )
    {
        obb.setEmpty(); if ( numPoints < 1 ) return 1;

        // Try 'exact' method first.
        bool            valid = hull.getDimensions() == 3;
        hkReal          baseSurfaceArea = -1;

        if ( hull.hasMassProperties() )
        {
            switch ( hull.getDimensions() )
            {
            case 2:
            case 3: baseSurfaceArea = hull.getSurfaceArea().getReal(); break;
            }
        }

        if ( valid )
        {
            hkSimdReal  minSa = hkSimdReal_Max;
            for ( int i = 0; i < hull.getNumPlanes(); ++i )
            {
                hkVector4 zAxis = hull.getPlane( i ); zAxis.normalize<3, HK_ACC_FULL, HK_SQRT_SET_ZERO>(); zAxis.setW( hkSimdReal_0 );
                hkSimdReal zMin, zMax; computeSupportingSpan( zAxis, points, numPoints, zMin, zMax );
                hkSimdReal zWidth = zMax - zMin;

                hkgpConvexHull hull2d;
                if ( hull2d.buildPlanar( points, numPoints, zAxis ) == 2 )
                {
                    for ( int j = 0; j < hull2d.getNumPlanes(); ++j )
                    {
                        hkVector4 yAxis = hull2d.getPlane( j ); yAxis.normalize<3, HK_ACC_FULL, HK_SQRT_SET_ZERO>(); yAxis.setW( hkSimdReal_0 );
                        hkSimdReal yMin, yMax; computeSupportingSpan( yAxis, points, numPoints, yMin, yMax );
                        hkSimdReal yWidth = yMax - yMin;

                        hkVector4   xAxis; xAxis.setCross( yAxis, zAxis );
                        hkSimdReal  xMin, xMax; computeSupportingSpan( xAxis, points, numPoints, xMin, xMax );
                        hkSimdReal  xWidth = xMax - xMin;

                        hkSimdReal  sa = xWidth * yWidth + yWidth * zWidth + zWidth * xWidth;
                        if ( sa < minSa )
                        {
                            minSa = sa;
                            obb.m_transform.getRotation().getColumn<0>().setXYZ_W( xAxis, ( xMax - xMin ) * hkSimdReal_Inv2 );
                            obb.m_transform.getRotation().getColumn<1>().setXYZ_W( yAxis, ( yMax - yMin ) * hkSimdReal_Inv2 );
                            obb.m_transform.getRotation().getColumn<2>().setXYZ_W( zAxis, ( zMax - zMin ) * hkSimdReal_Inv2 );
                            obb.m_transform.getTranslation().setMul( xAxis, ( xMin + xMax ) * hkSimdReal_Inv2 );
                            obb.m_transform.getTranslation().addMul( yAxis, ( yMin + yMax ) * hkSimdReal_Inv2 );
                            obb.m_transform.getTranslation().addMul( zAxis, ( zMin + zMax ) * hkSimdReal_Inv2 );
                        }
                    }
                }
                else
                {
                    valid = false;
                    break;
                }
            }
        }

        // Use gradient descent search if exact method fails.
        if ( !valid )
        {
            const int masterSearchRes = 8;
            const hkReal invRes = 1 / hkReal( masterSearchRes );

            hkVector4   zAxis; zAxis.setZero();
            hkSimdReal  zMin = hkSimdReal_0, zMax = hkSimdReal_0;
            // Find Z axis.
            {
                hkReal      searchRange = 1;
                hkVector4   searchOrigin; searchOrigin.setZero();
                hkSimdReal  minWidth = hkSimdReal_Max;

                for ( int iteration = 0; iteration < 32; ++iteration )
                {
                    int bestI = -1, bestJ = 0;
                    const hkReal searchRangeFactor = invRes * searchRange;
                    const hkReal half = searchRangeFactor / 2;
                    for ( int i = 0; i < masterSearchRes; ++i )
                    {
                        const hkReal u = searchOrigin( 0 ) + i * searchRangeFactor + half;
                        for ( int j = 0; j < masterSearchRes; ++j )
                        {
                            const hkReal v = searchOrigin( 1 ) + j * searchRangeFactor + half;
                            hkVector4 uv; uv.set( u, v, 0, 0 );
                            hkVector4 pn; hkGeometryProcessing::octahedronToNormal( uv, pn );
                            hkSimdReal dmin, dmax; computeSupportingSpan( pn, points, numPoints, dmin, dmax );
                            hkSimdReal width = dmax - dmin;
                            if ( width < minWidth )
                            {
                                zAxis = pn;
                                zMin = dmin;
                                zMax = dmax;
                                minWidth = width;
                                bestI = i;
                                bestJ = j;
                            }
                        }
                    }
                    if ( bestI >= 0 )
                    {
                        const hkReal newU = searchOrigin( 0 ) + bestI * searchRangeFactor;
                        const hkReal newV = searchOrigin( 1 ) + bestJ * searchRangeFactor;
                        searchRange *= invRes;
                        searchOrigin.set( newU, newV, 0, 0 );
                    }
                    else break;
                }
            }

            // Find Y axis.
            hkVector4   yAxis; yAxis.setZero();
            hkSimdReal  yMin = hkSimdReal_0, yMax = hkSimdReal_0;
            {
                hkVector4   refAxis;  hkVector4Util::calculatePerpendicularVector( zAxis, refAxis ); refAxis.normalize<3>();
                hkReal      searchRange = HK_REAL_PI;
                hkReal      searchOrigin = 0;
                hkSimdReal  minWidth = hkSimdReal_Max;

                for ( int iteration = 0; iteration < 16; ++iteration )
                {
                    const hkReal searchRangeFactor = invRes * searchRange;
                    const hkReal half = searchRangeFactor / 2;
                    int bestI = -1;
                    for ( int i = 0; i < masterSearchRes; ++i )
                    {
                        const hkReal a = searchOrigin + i * searchRangeFactor + half;

                        hkRotation  r; r.setAxisAngle( zAxis, hkSimdReal::fromFloat( a ) );
                        hkVector4   pn; pn._setRotatedDir( r, refAxis );
                        hkSimdReal  dmin, dmax; computeSupportingSpan( pn, points, numPoints, dmin, dmax );
                        hkSimdReal  width = dmax - dmin;
                        if ( width < minWidth )
                        {
                            bestI = i;
                            minWidth = width;
                            yAxis = pn;
                            yMin = dmin;
                            yMax = dmax;
                        }
                    }
                    if ( bestI >= 0 )
                    {
                        searchOrigin += bestI * searchRangeFactor;
                        searchRange *= invRes;
                    }
                    else break;
                }
            }

            // Compute X axis
            hkVector4   xAxis; xAxis.setCross( yAxis, zAxis );
            hkSimdReal  xMin, xMax; computeSupportingSpan( xAxis, points, numPoints, xMin, xMax );

            // Build OBB.
            obb.m_transform.getRotation().getColumn<0>().setXYZ_W( xAxis, ( xMax - xMin ) * hkSimdReal_Inv2 );
            obb.m_transform.getRotation().getColumn<1>().setXYZ_W( yAxis, ( yMax - yMin ) * hkSimdReal_Inv2 );
            obb.m_transform.getRotation().getColumn<2>().setXYZ_W( zAxis, ( zMax - zMin ) * hkSimdReal_Inv2 );
            obb.m_transform.getTranslation().setMul( xAxis, ( xMin + xMax ) * hkSimdReal_Inv2 );
            obb.m_transform.getTranslation().addMul( yAxis, ( yMin + yMax ) * hkSimdReal_Inv2 );
            obb.m_transform.getTranslation().addMul( zAxis, ( zMin + zMax ) * hkSimdReal_Inv2 );
        }

        obb.updateFlags();

        hkReal sa = obb.getSurfaceArea();

        if ( baseSurfaceArea < 0 ) return -1;

        if ( sa > HK_REAL_EPSILON )
        {
            if ( sa < baseSurfaceArea ) hkMath::swap( sa, baseSurfaceArea );
            return hkMath::clamp( baseSurfaceArea / sa, hkReal( 0 ), hkReal( 1 ) );
        }
        else
        {
            return 0;
        }
    }
}

//
hkReal  hkcdObb::set( const hkVector4* points, int numPoints )
{
    setEmpty(); if ( numPoints < 1 ) return 1;

    hkgpConvexHull::BuildConfig hullCfg; hullCfg.m_buildMassProperties = true;
    hkgpConvexHull  hull; hull.build( points, numPoints, hullCfg );
    return setInternal( *this, points, numPoints, hull );
}

//
hkReal  hkcdObb::set( const class hkgpConvexHull& hull )
{
    hkArray<hkVector4> vertices; hull.fetchPositions( hkgpConvexHull::SOURCE_VERTICES, vertices );
    return setInternal( *this, vertices.begin(), vertices.getSize(), hull );
}

//
hkReal  hkcdObb::set( const class hkgpConvexHull& hull, const hkVector4& projectionPlane )
{
    hkArray<hkVector4> vertices; hull.fetchPositions( hkgpConvexHull::SOURCE_VERTICES, vertices );

    for ( int i = 0; i < vertices.getSize(); ++i )
    {
        vertices[i].subMul( projectionPlane, projectionPlane.dot4xyz1( vertices[i] ) );
    }

    return setInternal( *this, vertices.begin(), vertices.getSize(), hull );
}

//
hkReal  hkcdObb::set( const hkStridedVertices& points )
{
    if ( points.m_striding != sizeof( hkVector4 ) || (hkUlong(points.m_vertices) & (sizeof( hkVector4 )-1) ) )
    {
        hkInplaceArray<hkVector4, 128>  linearPoints( points.getSize() );
        for ( int i = 0; i < points.getSize(); ++i ) points.getVertex( i, linearPoints[ i ] );
        return set( linearPoints.begin(), linearPoints.getSize() );
    }
    else
    {
        return set( (const hkVector4*) points.m_vertices, points.getSize() );
    }
}

//
void    hkcdObb::set( const hkAabb& aabb )
{
    hkVector4 he; aabb.getHalfExtents( he );
    hkVector4 ctr; aabb.getCenter( ctr );
    set( hkRotation::getIdentity(), ctr, he );
}

//
void    hkcdObb::set( const hkQuaternion& orientation, const hkVector4& center, const hkVector4& halfExtents )
{
    hkRotation o; o.set( orientation );
    set( o, center, halfExtents );
}

void hkcdObb::setCenter( const hkVector4& newCenter )
{
    m_transform.getTranslation().setXYZ_W( newCenter, m_transform.getTranslation().getW() );
}

//
void    hkcdObb::set( const hkRotation& orientation, const hkVector4& center, const hkVector4& halfExtents )
{
    m_transform.getRotation() = orientation;
    m_transform.getTranslation().setXYZ_0( center );
    setHalfExtents( halfExtents );
    updateFlags();
}

//
void    hkcdObb::set( const hkTransform& transform, const hkVector4& halfExtents )
{
    set( transform.getRotation(), transform.getTranslation(), halfExtents );
}

//
void    hkcdObb::getWorldAabb( hkAabb& aabbOut ) const
{
    hkVector4 he; getHalfExtents( he );
    computeWorldAabb( m_transform, he, aabbOut );
}

//
void    hkcdObb::getWorldAabb( const hkTransform& worldTransform, hkAabb& aabbOut ) const
{
    hkTransform t; t.setMul( worldTransform, m_transform );
    hkVector4 he; getHalfExtents( he );
    computeWorldAabb( t, he, aabbOut );
}

//
void    hkcdObb::getWorldVertices( hkVector4* verticesOut ) const
{
    hkAabb localAabb; getLocalAabb( localAabb );
    for ( int i = 0; i < 8; ++i )
    {
        hkVector4 v; localAabb.getVertex( i, v );
        verticesOut[ i ]._setTransformedPos( m_transform, v );
    }
}

//
// Collision methods.
//

#define MAKE_CODE(_a_,_b_)  ( ( (_a_) << 3) | ( (_b_) <<1 ) )
#define GET_AXIS_A(_code_)      ((_code_) >> 3)
#define GET_AXIS_B(_code_)      (((_code_) >> 1) & 3)
#define GET_SIGN(_code_)        ((_code_) & 1)

#define MIN_DISTANCE_EPS        hkReal( 0.00001 )
#define MIN_LENGTH_EPS          hkReal( 0.000001 )

const hkVector4Comparison::Mask hkcdObb::c_axisMasks[ 3 ] = { hkVector4ComparisonMask::MASK_YZ,hkVector4ComparisonMask::MASK_XZ,hkVector4ComparisonMask::MASK_XY };

//
void    hkcdObb::computeDistance( const hkcdObb& a, const hkcdObb& b, bool computeContactPoint, DistanceResult* HK_RESTRICT result )
{
    hkVector4   heA; a.getHalfExtents( heA );
    hkVector4   heB; b.getHalfExtents( heB );

    hkTransform BinA; BinA._setMulInverseMul( a.m_transform, b.m_transform );
    hkTransform AinB; AinB._setMulInverseMul( b.m_transform, a.m_transform );

    hkVector4           distances; distances.setAll( -hkSimdReal_Max );
    hkIntVector         codes; codes.setZero();
    hkVector4Comparison signs; signs.set<hkVector4ComparisonMask::MASK_NONE>();


    // A faces vs B.
    {
        const hkVector4& offsets = BinA.getTranslation();
        hkVector4   nxB; nxB.setAbs( AinB.getRotation().getColumn<0>() );
        hkVector4   nyB; nyB.setAbs( AinB.getRotation().getColumn<1>() );
        hkVector4   nzB; nzB.setAbs( AinB.getRotation().getColumn<2>() );
        hkVector4   dB; hkVector4Util::dot3_1vs3( heB, nxB, nyB, nzB, dB );

        codes.set( MAKE_CODE( 0, 3 ), MAKE_CODE( 1, 3 ), MAKE_CODE( 2, 3 ), 0 );
        signs = offsets.greaterZero();
        hkVector4 s; s.setAbs( offsets );
        distances.setSub( s, heA ); distances.sub( dB );
    }

    // B faces vs A.
    {
        const hkVector4& offsets = AinB.getTranslation();
        hkVector4   nxA; nxA.setAbs( BinA.getRotation().getColumn<0>() );
        hkVector4   nyA; nyA.setAbs( BinA.getRotation().getColumn<1>() );
        hkVector4   nzA; nzA.setAbs( BinA.getRotation().getColumn<2>() );
        hkVector4   dA; hkVector4Util::dot3_1vs3( heA, nxA, nyA, nzA, dA );

        hkVector4 s; s.setAbs( offsets );
        hkVector4 newDistances; newDistances.setSub( s, dA ); newDistances.sub( heB );

        hkVector4Comparison c = newDistances.greater( distances );
        if ( c.anyIsSet<hkVector4ComparisonMask::MASK_XYZ>() )
        {
            hkIntVector newCodes; newCodes.set( MAKE_CODE( 3, 0 ), MAKE_CODE( 3, 1 ), MAKE_CODE( 3, 2 ), 0 );
            hkVector4Comparison newSigns = offsets.lessZero();
            distances.setSelect( c, newDistances, distances );
            codes.setSelect( c, newCodes, codes );
            signs.setSelect( c, newSigns, signs );
        }
    }

    // A edges vs B edges.
    {
        hkVector4   c_minLengthSquared; c_minLengthSquared.setAll( MIN_LENGTH_EPS );
        hkVector4   c_maxLengthSquared; c_maxLengthSquared.setSub( hkVector4::getConstant<HK_QUADREAL_1>(), c_minLengthSquared );
        for ( int i = 0; i < 3; ++i )
        {
            hkMatrix3 normalsInA; normalsInA.setCrossSkewSymmetric( BinA.getColumn( i ) );

            hkMatrix3   tn; tn._setTranspose( normalsInA );
            tn.getColumn<0>().mul( tn.getColumn<0>() );
            tn.getColumn<1>().mul( tn.getColumn<1>() );
            tn.getColumn<2>().mul( tn.getColumn<2>() );
            tn.getColumn<0>().add( tn.getColumn<1>() );
            tn.getColumn<0>().add( tn.getColumn<2>() );

            hkVector4Comparison valids; valids.setAnd( tn.getColumn<0>().greater( c_minLengthSquared ), tn.getColumn<0>().less( c_maxLengthSquared ) );
            hkVector4           invLengths; invLengths.setSqrtInverse<HK_ACC_23_BIT, HK_SQRT_IGNORE>( tn.getColumn<0>() );

            normalsInA.getColumn<0>().mul( invLengths.getComponent<0>() );
            normalsInA.getColumn<1>().mul( invLengths.getComponent<1>() );
            normalsInA.getColumn<2>().mul( invLengths.getComponent<2>() );

            hkMatrix3 normalsInB; normalsInB.setMul( AinB.getRotation(), normalsInA );

            hkVector4 offsets; offsets._setRotatedInverseDir( normalsInA, BinA.getTranslation() );

            hkVector4 dA;
            {
                hkVector4 x; x.setAbs( normalsInA.getColumn<0>() );
                hkVector4 y; y.setAbs( normalsInA.getColumn<1>() );
                hkVector4 z; z.setAbs( normalsInA.getColumn<2>() );
                hkVector4Util::dot3_1vs3( heA, x, y, z, dA );
            }

            hkVector4 dB;
            {
                hkVector4 x; x.setAbs( normalsInB.getColumn<0>() );
                hkVector4 y; y.setAbs( normalsInB.getColumn<1>() );
                hkVector4 z; z.setAbs( normalsInB.getColumn<2>() );
                hkVector4Util::dot3_1vs3( heB, x, y, z, dB );
            }

            hkVector4 s; s.setAbs( offsets );
            hkVector4 newDistances; newDistances.setSub( s, dA ); newDistances.sub( dB );

            hkVector4Comparison c; c.setAnd( newDistances.greater( distances ), valids );
            if ( c.anyIsSet<hkVector4ComparisonMask::MASK_XYZ>() )
            {
                hkIntVector newCodes; newCodes.set( MAKE_CODE( 0, i ), MAKE_CODE( 1, i ), MAKE_CODE( 2, i ), 0 );
                hkVector4Comparison newSigns = offsets.greaterEqualZero();
                distances.setSelect( c, newDistances, distances );
                codes.setSelect( c, newCodes, codes );
                signs.setSelect( c, newSigns, signs );
            }
        }
    }

    // Recover code and distance.
    hkVector4   normalInA;
    hkSimdReal  distance;
    int         code;
    {
        const int i = distances.getIndexOfMaxComponent<3>();
        code = (int)codes.getU32( i ) | ( ( signs.getMask() >> i ) & 1 );
        distance = distances.getComponent( i );
        normalInA.setZero();
    }

    // Build m_pointAinA output.
    int         sign = GET_SIGN( code );
    const int   axisAindex = GET_AXIS_A( code );
    const int   axisBindex = GET_AXIS_B( code );
    hkVector4   point; point.setZero();

    if ( computeContactPoint )
    {
        const bool  reProject = distance > hkSimdReal::fromFloat( MIN_DISTANCE_EPS );

        if ( axisBindex == 3 )
        {
            // Build normal.
            normalInA = hkVector4::getConstant( (hkVectorConstant)( HK_QUADREAL_1000 + axisAindex ) );

            // A faces vs B.
            const hkVector4& axis = AinB.getRotation().getColumn( axisAindex );
            if ( sign )
            {
                point.setFlipSign( heB, axis.greaterEqualZero() );
                point.subMul( axis, distance );
            }
            else
            {
                point.setFlipSign( heB, axis.lessZero() );
                point.addMul( axis, distance );
            }
            point._setTransformedPos( BinA, point );
            if ( reProject )
            {
                hkVector4 nheA; nheA.setNeg<3>( heA );
                hkVector4 prjOnA;
                prjOnA.setMin( heA, point );
                prjOnA.setMax( nheA, prjOnA );
                if ( prjOnA.notEqual( point ).anyIsSet( c_axisMasks[ axisAindex ] ) )
                {
                    hkVector4 nheB; nheB.setNeg<3>( heB );
                    hkVector4 prjOnB; prjOnB._setTransformedPos( AinB, prjOnA );
                    prjOnB.setMin( heB, prjOnB );
                    prjOnB.setMax( nheB, prjOnB );
                    prjOnB._setTransformedPos( BinA, prjOnB );

                    normalInA.setSub( prjOnA, prjOnB );
                    distance = normalInA.normalizeWithLength<3, HK_ACC_23_BIT, HK_SQRT_IGNORE>();
                    point = prjOnA;
                    sign = 0;
                }
            }
        }
        else if ( axisAindex == 3 )
        {
            // Build normal.
            normalInA = BinA.getColumn( axisBindex );

            // B faces vs A.
            const hkVector4& axis = BinA.getColumn( axisBindex );
            point.setFlipSign( heA, sign ? axis.lessZero() : axis.greaterEqualZero() );

            if ( reProject )
            {
                hkVector4 nheB; nheB.setNeg<3>( heB );
                hkVector4 prjBase; prjBase._setTransformedPos( AinB, point );
                hkVector4 prjOnB;
                prjOnB.setMin( heB, prjBase );
                prjOnB.setMax( nheB, prjOnB );
                if ( prjOnB.notEqual( prjBase ).anyIsSet( c_axisMasks[ axisBindex ] ) )
                {
                    prjOnB._setTransformedPos( BinA, prjOnB );

                    hkVector4 nheA; nheA.setNeg<3>( heA );
                    hkVector4 prjOnA;
                    prjOnA.setMin( heA, prjOnB );
                    prjOnA.setMax( nheA, prjOnA );

                    normalInA.setSub( prjOnA, prjOnB );
                    distance = normalInA.normalizeWithLength<3, HK_ACC_23_BIT, HK_SQRT_IGNORE>();
                    point = prjOnA;
                    sign = 0;
                }
            }
        }
        else
        {
            // Build normal.
            normalInA.setCross( BinA.getColumn( axisBindex ), hkVector4::getConstant( (hkVectorConstant)( HK_QUADREAL_1000 + axisAindex ) ) );
            normalInA.normalize<3, HK_ACC_23_BIT, HK_SQRT_IGNORE>();

            // A edges vs B edges.
            hkVector4Comparison cmpA; cmpA.set( ( hkVector4ComparisonMask::Mask )( 1 << axisAindex ) );
            hkVector4   svA0; svA0.setFlipSign( heA, sign ? normalInA.lessZero() : normalInA.greaterEqualZero() );
            hkVector4   svA1; svA1.setFlipSign( svA0, cmpA );
            hkVector4   dA; dA.setSub( svA1, svA0 );

            hkVector4Comparison cmpB; cmpB.set( ( hkVector4ComparisonMask::Mask )( 1 << axisBindex ) );
            hkVector4   normalInB; normalInB._setRotatedDir( AinB.getRotation(), normalInA );
            hkVector4   svB0; svB0.setFlipSign( heB, sign ? normalInB.greaterEqualZero() : normalInB.lessZero() );
            hkVector4   svB1; svB1.setFlipSign( svB0, cmpB );
            hkVector4   dB; dB.setSub( svB1, svB0 );
            svB0._setTransformedPos( BinA, svB0 );
            dB._setRotatedDir( BinA.getRotation(), dB );

            hkSimdReal  u, v;
            hkcdClosestPointSegmentSegment( svA0, dA, svB0, dB, u, v );
            if ( reProject )
            {
                hkVector4 pa; pa.setAddMul( svA0, dA, u );
                hkVector4 pb; pb.setAddMul( svB0, dB, v );
                normalInA.setSub( pa, pb );
                distance = normalInA.normalizeWithLength<3, HK_ACC_23_BIT, HK_SQRT_IGNORE>();
                point = pa;
                sign = 0;
            }
            else
            {
                point.setAddMul( svA0, dA, u );
            }
        }
    }

    // Build m_normalAndDistanceInA output.
    if ( sign ) normalInA.setNeg<3>( normalInA );
    result->m_normal_distance.setXYZ_W( normalInA, distance );
    result->m_point_code = point;
    result->m_point_code.setInt24W( code );
}

//
void    hkcdObb::computeDistance( const hkcdObb& a, const hkVector4& b, DistanceResult* HK_RESTRICT result )
{
    hkAabb      aabb; a.getLocalAabb( aabb );
    hkVector4   point; point._setTransformedInversePos( a.m_transform, b );
    hkVector4   prj; aabb.projectPoint( point, prj );
    hkVector4   normal; normal.setSub( prj, point );
    hkSimdReal  distance = normal.normalizeWithLength<3, HK_ACC_23_BIT, HK_SQRT_SET_ZERO>();
    if ( distance.isLess( hkSimdReal::fromFloat( MIN_DISTANCE_EPS ) ) )
    {
        hkVector4 absPrj; absPrj.setAbs( prj );
        hkVector4 del; del.setSub( absPrj, aabb.m_max );
        const int axis = del.getIndexOfMinAbsComponent<3>();
        normal.setFlipSign( hkVector4::getConstant( (hkVectorConstant)( HK_QUADREAL_1000 + axis ) ), point.greaterZero() );
        distance = del.getComponent( axis );
        prj.setAddMul( point, normal, distance );
    }
    result->m_normal_distance.setXYZ_W( normal, distance );
    result->m_point_code = prj;
    result->m_point_code.setInt24W( MAKE_CODE( 3, 3 ) );
}

//
bool    hkcdObb::intersect( const hkcdObb& a, const hkcdObb& b )
{
    hkVector4   heA; a.getHalfExtents( heA );
    hkVector4   heB; b.getHalfExtents( heB );

    hkTransform BinA; BinA._setMulInverseMul( a.m_transform, b.m_transform );
    hkTransform AinB; AinB._setMulInverseMul( b.m_transform, a.m_transform );

    // A faces vs B.
    {
        hkVector4   nxB; nxB.setAbs( AinB.getRotation().getColumn<0>() );
        hkVector4   nyB; nyB.setAbs( AinB.getRotation().getColumn<1>() );
        hkVector4   nzB; nzB.setAbs( AinB.getRotation().getColumn<2>() );
        hkVector4   dB; hkVector4Util::dot3_1vs3( heB, nxB, nyB, nzB, dB );

        hkVector4   s; s.setAbs( BinA.getTranslation() );
        hkVector4   distances; distances.setSub( s, heA ); distances.sub( dB );
        if ( distances.greaterZero().anyIsSet<hkVector4ComparisonMask::MASK_XYZ>() ) return false;
    }

    // B faces vs A.
    {
        hkVector4   nxA; nxA.setAbs( BinA.getRotation().getColumn<0>() );
        hkVector4   nyA; nyA.setAbs( BinA.getRotation().getColumn<1>() );
        hkVector4   nzA; nzA.setAbs( BinA.getRotation().getColumn<2>() );
        hkVector4   dA; hkVector4Util::dot3_1vs3( heA, nxA, nyA, nzA, dA );

        hkVector4   s; s.setAbs( AinB.getTranslation() );
        hkVector4   distances; distances.setSub( s, dA ); distances.sub( heB );
        if ( distances.greaterZero().anyIsSet<hkVector4ComparisonMask::MASK_XYZ>() ) return false;
    }

    // A edges vs B edges.
    {
        hkVector4   c_minLengthSquared; c_minLengthSquared.setAll( MIN_LENGTH_EPS );
        hkVector4   c_maxLengthSquared; c_maxLengthSquared.setSub( hkVector4::getConstant<HK_QUADREAL_1>(), c_minLengthSquared );
        for ( int i = 0; i < 3; ++i )
        {
            hkMatrix3 normalsInA; normalsInA.setCrossSkewSymmetric( BinA.getColumn( i ) );

            hkMatrix3   tn; tn._setTranspose( normalsInA );
            tn.getColumn<0>().mul( tn.getColumn<0>() );
            tn.getColumn<1>().mul( tn.getColumn<1>() );
            tn.getColumn<2>().mul( tn.getColumn<2>() );
            tn.getColumn<0>().add( tn.getColumn<1>() );
            tn.getColumn<0>().add( tn.getColumn<2>() );

            hkVector4Comparison valids; valids.setAnd( tn.getColumn<0>().greater( c_minLengthSquared ), tn.getColumn<0>().less( c_maxLengthSquared ) );

            hkVector4           invLengths; invLengths.setSqrtInverse<HK_ACC_23_BIT, HK_SQRT_IGNORE>( tn.getColumn<0>() );

            normalsInA.getColumn<0>().mul( invLengths.getComponent<0>() );
            normalsInA.getColumn<1>().mul( invLengths.getComponent<1>() );
            normalsInA.getColumn<2>().mul( invLengths.getComponent<2>() );

            hkMatrix3 normalsInB; normalsInB.setMul( AinB.getRotation(), normalsInA );

            hkVector4 offsets; offsets._setRotatedInverseDir( normalsInA, BinA.getTranslation() );

            hkVector4 dA;
            {
                hkVector4 x; x.setAbs( normalsInA.getColumn<0>() );
                hkVector4 y; y.setAbs( normalsInA.getColumn<1>() );
                hkVector4 z; z.setAbs( normalsInA.getColumn<2>() );
                hkVector4Util::dot3_1vs3( heA, x, y, z, dA );
            }

            hkVector4 dB;
            {
                hkVector4 x; x.setAbs( normalsInB.getColumn<0>() );
                hkVector4 y; y.setAbs( normalsInB.getColumn<1>() );
                hkVector4 z; z.setAbs( normalsInB.getColumn<2>() );
                hkVector4Util::dot3_1vs3( heB, x, y, z, dB );
            }

            hkVector4 s; s.setAbs( offsets );
            hkVector4 distances; distances.setSub( s, dA ); distances.sub( dB );

            hkVector4Comparison c; c.setAnd( distances.greaterZero(), valids );
            if ( c.anyIsSet<hkVector4ComparisonMask::MASK_XYZ>() ) return false;
        }
    }

    return true;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
