// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/DebugUtil/DeterminismUtil/hkCheckDeterminismUtil.h>

/// A contact manifold between two objects, holding up to 4 points.
/// Note that if the number of contact points > 0, all positions/distances must be initialized,
/// even if the number of contact points < 4 (in this case the first point is replicated in the last points)
struct HK_EXPORT_COMMON hkcdManifold4
{
    enum
    {
        MAX_NUM_CONTACT_POINTS = 4
    };

    HK_DECLARE_CLASS(hkcdManifold4, New, Reflect);

    HK_ALWAYS_INLINE hkcdManifold4()
    {
#ifdef HK_WANT_DETERMINISM_CHECKS
        hkString::memSet(this, 0xcd, sizeof(hkcdManifold4)); // deterministically set unused memory
#endif
    }

    HK_INLINE hkcdManifold4(const hkcdManifold4& other)
    {
        m_numPoints = other.m_numPoints;
        m_minimumDistance = other.m_minimumDistance;

        m_normal.set(other.m_normal);
        m_weldNormal.set(other.m_weldNormal);

        for (int i = 0; i < MAX_NUM_CONTACT_POINTS; i++)
        {
            m_distances[i] = other.m_distances[i];
            m_positions[i] = other.m_positions[i];
        }
    }

    /// Get the distance of a given contact.
    HK_INLINE hkSimdReal getDistance( int i ) const
    {
        HK_ASSERT_NO_MSG(0xf05f23de, i < MAX_NUM_CONTACT_POINTS);
        hkSimdReal res; res.load<1>(&m_distances[i]);
        return res;
    }

    /// Get the minimum of all distances.
    HK_INLINE hkSimdReal getMinimumDistance() const
    {
        return hkSimdReal::fromFloat(m_minimumDistance);
    }

    /// Invert the manifold, so it becomes from A to B instead of B to A.
    HK_INLINE void flip()
    {
        hkVector4 normal = m_normal; // cache it to avoid loads
        for( int i = 0; i < MAX_NUM_CONTACT_POINTS; i += 4 )
        {
            hkVector4 distances; distances.load<4>( &m_distances[i] );
            m_positions[i+0].addMul( normal, distances.getComponent<0>() );
            m_positions[i+1].addMul( normal, distances.getComponent<1>() );
            m_positions[i+2].addMul( normal, distances.getComponent<2>() );
            m_positions[i+3].addMul( normal, distances.getComponent<3>() );
        }
        m_normal.setNeg<4>( normal );
    }

#ifdef HK_DEBUG
    HK_INLINE void checkConsistency() const
    {
        if( m_numPoints )
        {
            HK_ASSERT_NO_MSG( 0x0fcdfd34, m_normal.isOk<4>() && m_normal.isNormalized<3>( 1e-3f ) );

            // check all used points
            const hkSimdReal minDist = getMinimumDistance();
            for( int i = 0; i < m_numPoints; i++ )
            {
                hkVector4 vec = m_positions[i];
                HK_ASSERT_NO_MSG( 0x0fcdfd35, vec.isOk<4>() );
                HK_ASSERT_NO_MSG( 0x0fcdfd36, getDistance(i).isOk() && getDistance(i) >= minDist );
            }

            // check all unused points - they should have distance set to ~infinity or that of a used point
            for( int i = m_numPoints; i < MAX_NUM_CONTACT_POINTS; i++ )
            {
                bool found = false;
                for( int j = 0; j < m_numPoints; ++j )
                {
                    if( m_distances[i] == m_distances[j] )
                    {
                        found = true;
                        break;
                    }
                }
                HK_ASSERT_NO_MSG( 0x0fcdfd37, found || m_distances[i] >= HK_REAL_HIGH );    
            }
        }
    }
#endif

    HK_ALIGN_REAL(int m_numPoints); ///< The number of valid contact points in m_positions.

    hkReal m_minimumDistance;       ///< The minimum distance between the 2 objects.

    hkVector4 m_normal;             ///< The manifold normal, from B to A in world space.
    hkVector4 m_weldNormal;         ///< If object B is a triangle or quad, this is plane 0 (needed for welding).

    hkReal    m_distances[MAX_NUM_CONTACT_POINTS];  ///< The distances of each contact point.
    hkVector4 m_positions[MAX_NUM_CONTACT_POINTS];  ///< The positions of each contact point, on B in world space.
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
