// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/SimplexSolver/hkSimplexSolver.h>


struct  hkSimplexSolverActivePlanes
{
    int m_index;
    const hkSurfaceConstraintInfo* m_constraint;
    hkSurfaceConstraintInteraction* m_interaction;
};

struct  hkSimplexSolverInfo
{
    hkSimplexSolverActivePlanes m_supportPlanes[4];
    int                         m_numSupportPlanes;

    hkReal                      m_currentTime;

    const hkSimplexSolverInput* m_input;
    hkSimplexSolverOutput*      m_output;

    hkSurfaceConstraintInteraction &getOutput( const hkSurfaceConstraintInfo* info )
    {
        return m_output->m_planeInteractions[ info - m_input->m_constraints ];
    }
};

static void HK_CALL hkSimplexSolverSortInfo( hkSimplexSolverInfo& info )
{
    // simple bubble sort by (priority,velocity)
    for ( int i = 0; i < info.m_numSupportPlanes-1; i++ )
    {
        for (int j = i+1; j < info.m_numSupportPlanes; j++ )
        {
            hkSimplexSolverActivePlanes& p0 = info.m_supportPlanes[i];
            hkSimplexSolverActivePlanes& p1 = info.m_supportPlanes[j];
            if ( p0.m_constraint->m_priority < p1.m_constraint->m_priority )
            {
                continue;
            }
            if ( p0.m_constraint->m_priority == p1.m_constraint->m_priority )
            {
                hkSimdReal vel0 = p0.m_constraint->m_velocity.lengthSquared<3>();
                hkSimdReal vel1 = p1.m_constraint->m_velocity.lengthSquared<3>();
                if ( vel0.isLess(vel1) )
                {
                    continue;
                }
            }
            hkSimplexSolverActivePlanes h = p0; p0 = p1; p1 = h;
        }
    }
}

static hkBool32 HK_CALL hkSimplexSolverSolveTest1d( const hkSurfaceConstraintInfo& sci, hkVector4Parameter velocityIn )
{
    hkVector4 relativeVelocity; relativeVelocity.setSub( velocityIn, sci.m_velocity );
    hkSimdReal planeVel = relativeVelocity.dot<3>( sci.m_plane );

    return planeVel.isLess( hkSimdReal::fromFloat(-0.001f) ); // replace by maxPenetrationVelocity

    //if ( planeVel >= -0.001f ) // replace by maxPenetrationVelocity
    //{
    //  return false;
    //}

    //return true;
}

static void HK_CALL hkSimplexSolverSolve1d( const hkSimplexSolverInfo& info, const hkSurfaceConstraintInfo& sci, const hkVector4& velocityIn, hkVector4& velocityOut )
{
    const hkVector4& groundVelocity = sci.m_velocity;
    hkSimdReal staticFriction; staticFriction.load<1>(&sci.m_staticFriction);
    hkSimdReal dynamicFriction; dynamicFriction.load<1>(&sci.m_dynamicFriction);

    hkVector4 relativeVelocity; relativeVelocity.setSub( velocityIn, groundVelocity );
    hkSimdReal planeVel = relativeVelocity.dot<3>( sci.m_plane );


    const hkSimdReal origVelocity2 = relativeVelocity.lengthSquared<3>();
    relativeVelocity.subMul( planeVel, sci.m_plane );
    {
        const hkSimdReal vp2 = (planeVel * planeVel);
            // static friction is active if
            //  velProjPlane * friction > |(velParallel)|
            //      vplane   *     f    >         vpar
            //      vp       *     f    >         vpar
            //      vp2      *     f2   >         vpar2
        hkSimdReal extraStaticFriction; extraStaticFriction.load<1>( relativeVelocity.dot<3>( info.m_input->m_upVector ).isGreaterZero() ? &sci.m_extraUpStaticFriction : &sci.m_extraDownStaticFriction );
        if ( extraStaticFriction.isGreaterZero() )
        {
            //
            //  split the friction
            //
            hkVector4 horizontal; horizontal.setCross( info.m_input->m_upVector, sci.m_plane );
            hkSimdReal hor2 = horizontal.lengthSquared<3>();
            hkSimdReal horVel; horVel.setZero();
            if ( hor2.isGreater(hkSimdReal_Eps) )
            {
                horizontal.mul( hor2.sqrtInverse<HK_ACC_23_BIT,HK_SQRT_IGNORE>() );

                horVel = relativeVelocity.dot<3>( horizontal );

                // horizontal component
                {
                    const hkSimdReal horVel2 = horVel * horVel;
                    const hkSimdReal f2 = staticFriction * staticFriction;
                    if ( (vp2 * f2).isGreaterEqual(horVel2) )
                    {
                        relativeVelocity.subMul( horVel, horizontal );
                        horVel.setZero();
                    }
                }
            }

            // vert component
            {
                hkSimdReal vertVel2 = origVelocity2 - horVel * horVel - vp2;
                const hkSimdReal f2 = (staticFriction + extraStaticFriction) * (staticFriction + extraStaticFriction);
                if ( (vp2 * f2).isGreaterEqual(vertVel2) )
                {
                    if ( horVel.isEqualZero() )
                    {
                        velocityOut = groundVelocity;
                        return;
                    }
                    relativeVelocity.setMul( horVel, horizontal );
                }
            }
        }
        else
        {
            // static friction is active if
            //  velProjPlane * friction > |(vel-velProjPlane)|
            //      vp       *     f    >         rvProj
            //
            //  -> vp * f >= rvProj
            //  -> vp * f >= sqrt( vel^2 - vp^2 )
            //  -> vp^2 ( f^2 + 1 ) >= vel^2
            //          const hkReal f2 = staticFriction * staticFriction;
            const hkSimdReal f2 = (staticFriction * staticFriction);
            if ( (vp2 * ( hkSimdReal_1 + f2 )).isGreaterEqual(origVelocity2) )
            {
                // static friction kicks in
                velocityOut = groundVelocity;
                return;
            }
        }
    }


    if ( dynamicFriction.isLess(hkSimdReal_1) )
    {
        //
        //  apply dynamic friction 0 = conserve input velocity 1 = clip against normal
        //
        hkSimdReal velOut2 = relativeVelocity.lengthSquared<3>();
        if ( velOut2.isGreaterEqual(hkSimdReal_Eps) )
        {
            if ( velOut2.isGreater( hkSimdReal::fromFloat(0.001f) * origVelocity2) )
            {
                hkSimdReal f = (origVelocity2/velOut2).sqrt();
                f = dynamicFriction + (hkSimdReal_1 - dynamicFriction) * f;
                relativeVelocity.mul( f );
                hkSimdReal p = sci.m_plane.dot<3>( relativeVelocity );
                relativeVelocity.subMul( p, sci.m_plane );
            }
        }
    }
    velocityOut.setAdd( relativeVelocity, groundVelocity );

}

static void HK_CALL hkSimplexSolverSolve2d( hkSimplexSolverInfo& info, const hkSurfaceConstraintInfo& sci0,const hkSurfaceConstraintInfo& sci1, const hkVector4& velocityIn, hkVector4& velocityOut )
{

    //
    // Calculate the free axis
    //
    hkVector4 axis; axis.setCross( sci0.m_plane, sci1.m_plane );
    hkSimdReal axisLen2 = axis.lengthSquared<3>();

    //
    // Check for parallel planes
    //
    if ( axisLen2.isLessEqual(hkSimdReal_Eps) )
    {
        //
        //  Do the planes sequentially
        //
hkSimplexSolverSolve2dSolveSequentually:
        info.getOutput( &sci0).m_status = hkSurfaceConstraintInteraction::HK_STATUS_2D_FAILURE;
        info.getOutput( &sci1).m_status = hkSurfaceConstraintInteraction::HK_STATUS_2D_FAILURE;

        // HVK-888
        if ( sci0.m_priority > sci1.m_priority )
        {
            hkSimplexSolverSolve1d( info, sci1, velocityIn, velocityOut );
            hkSimplexSolverSolve1d( info, sci0, velocityOut, velocityOut );
        }
        else
        {
            hkSimplexSolverSolve1d( info, sci0, velocityIn, velocityOut );
            hkSimplexSolverSolve1d( info, sci1, velocityOut, velocityOut );
        }
        return;
    }

    hkSimdReal invAxisLen = axisLen2.sqrtInverse<HK_ACC_23_BIT,HK_SQRT_IGNORE>();
    axis.mul( invAxisLen );

    //
    //  Calculate the velocity of the free axis
    //
    hkVector4 axisVel;
    {
        hkMatrix3 m;
        hkVector4& r0 = m.getColumn<0>();
        hkVector4& r1 = m.getColumn<1>();
        hkVector4& r2 = m.getColumn<2>();

        r0.setCross( sci0.m_plane, sci1.m_plane );
        r1.setCross( sci1.m_plane, axis );
        r2.setCross( axis, sci0.m_plane );

        hkVector4 sVel; sVel.setAdd( sci0.m_velocity, sci1.m_velocity );

        hkVector4 t;
        t.set( hkSimdReal_Half * axis.dot<3>( sVel ),
               sci0.m_plane.dot<3>( sci0.m_velocity ),
               sci1.m_plane.dot<3>( sci1.m_velocity ),
               hkSimdReal_0 );

        axisVel._setRotatedDir( m, t );
        axisVel.mul( invAxisLen );

        hkVector4 absVel; absVel.setAbs( axisVel );
        if( absVel.greater(info.m_input->m_maxSurfaceVelocity).anyIsSet<hkVector4ComparisonMask::MASK_XYZ>() )
        {
            goto hkSimplexSolverSolve2dSolveSequentually;
        }
    }

    HK_ASSERT_NO_MSG(0x7e58dad9,  info.m_input->m_upVector.isNormalized<3>());

    const hkVector4& groundVelocity = axisVel;
    hkVector4 relativeVelocity; relativeVelocity.setSub( velocityIn, groundVelocity );

    const hkSimdReal vel2 = relativeVelocity.lengthSquared<3>();
    const hkSimdReal axisVert = info.m_input->m_upVector.dot<3>( axis );
    hkSimdReal axisProjVelocity = relativeVelocity.dot<3>( axis );

    hkSimdReal staticFriction; staticFriction.setFromFloat(sci0.m_staticFriction + sci1.m_staticFriction);
    if ( (axisVert * axisProjVelocity).isGreaterZero() )
    {
        staticFriction.add(hkSimdReal::fromFloat(sci0.m_extraUpStaticFriction + sci1.m_extraUpStaticFriction) * axisVert);
    }
    else
    {
        staticFriction.add(hkSimdReal::fromFloat(sci0.m_extraDownStaticFriction + sci1.m_extraDownStaticFriction) * axisVert);
    }
    staticFriction.mul(hkSimdReal_Half);

    hkSimdReal dynamicFriction; dynamicFriction.setFromFloat(sci0.m_dynamicFriction + sci1.m_dynamicFriction);
    dynamicFriction.mul(hkSimdReal_Half);


    // static friction is active if
    //  |vel-axisProjVelocity|(rv) * friction(f) > axisProjVelocity(av)
    //  -> sqrt( vel2 - av2 ) * f > av
    //  -> (vel2 - av2) * f2  > av2
    {
        const hkSimdReal f2 = staticFriction * staticFriction;
        const hkSimdReal av2 = axisProjVelocity * axisProjVelocity;
        if ( ((vel2 - av2) * f2).isGreaterEqual(av2) )
        {
            // static friction kicks in
            velocityOut = groundVelocity;
            return;
        }
    }

    if ( dynamicFriction.isLess(hkSimdReal_1) )
    {
        //
        //  apply dynamic friction
        //
        if ( (axisProjVelocity * axisProjVelocity).isGreater(hkSimdReal::fromFloat(0.001f) * vel2) )
        {
            hkSimdReal tmp; tmp.setReciprocal(axisProjVelocity);
            hkSimdReal f; f.setAbs(tmp);
            f.mul(vel2.sqrt());
            f.mul(hkSimdReal_1 - dynamicFriction);
            f.add(dynamicFriction);
            axisProjVelocity.mul(f);
            //hkprintf("%f ", axisProjVelocity);
        }
    }
    velocityOut.setAddMul( groundVelocity, axis, axisProjVelocity );
}

static void HK_CALL hkSimplexSolverSolve3d( hkSimplexSolverInfo& info,
                                            const hkSurfaceConstraintInfo* sci0, const hkSurfaceConstraintInfo* sci1,
                                            const hkSurfaceConstraintInfo* sci2, int allowResort, const hkVector4& velocityIn, hkVector4& velocityOut )
{

    //
    //  Calculate the velocity of the point axis
    //
    hkVector4 pointVel;
    {
        hkMatrix3 m;
        hkVector4& r0 = m.getColumn<0>();
        hkVector4& r1 = m.getColumn<1>();
        hkVector4& r2 = m.getColumn<2>();

        r0.setCross( sci1->m_plane, sci2->m_plane );
        r1.setCross( sci2->m_plane, sci0->m_plane );
        r2.setCross( sci0->m_plane, sci1->m_plane );

        hkSimdReal det = r0.dot<3>( sci0->m_plane );
        hkSimdReal tst; tst.setAbs(det);
        if ( tst.isLess(hkSimdReal_Eps) )
        {
hkSimplexSolverSolve3dSolveSequentually:
            if ( allowResort )
            {
                HK_ASSERT_NO_MSG( 0xf0235465, sci0 == info.m_supportPlanes[0].m_constraint );
                HK_ASSERT_NO_MSG( 0xf0235465, sci1 == info.m_supportPlanes[1].m_constraint );
                HK_ASSERT_NO_MSG( 0xf0235465, sci2 == info.m_supportPlanes[2].m_constraint );
                hkSimplexSolverSortInfo( info );
                sci0 = info.m_supportPlanes[0].m_constraint;
                sci1 = info.m_supportPlanes[1].m_constraint;
                sci2 = info.m_supportPlanes[2].m_constraint;
            }
            info.getOutput( sci0).m_status = hkSurfaceConstraintInteraction::HK_STATUS_3D_FAILURE;
            info.getOutput( sci1).m_status = hkSurfaceConstraintInteraction::HK_STATUS_3D_FAILURE;
            info.getOutput( sci2).m_status = hkSurfaceConstraintInteraction::HK_STATUS_3D_FAILURE;

            int oldNum = info.m_numSupportPlanes;
            {
                hkSimplexSolverSolve2d( info, *sci0, *sci1, velocityIn, velocityOut );
            }
            if ( oldNum == info.m_numSupportPlanes )
            {
                hkSimplexSolverSolve2d( info, *sci0, *sci2, velocityOut, velocityOut );
            }
            if ( oldNum == info.m_numSupportPlanes )
            {
                hkSimplexSolverSolve2d( info, *sci1, *sci2, velocityOut, velocityOut );
            }
            return;
        }

        hkVector4 sVel; sVel.setAdd( sci0->m_velocity, sci1->m_velocity );

        hkVector4 t;
        t.set( sci0->m_plane.dot<3>( sci0->m_velocity ),
               sci1->m_plane.dot<3>( sci1->m_velocity ),
               sci2->m_plane.dot<3>( sci2->m_velocity ),
               hkSimdReal_0 );

        pointVel._setRotatedDir( m, t );
        pointVel.mul( det.reciprocal() );

        hkVector4 absVel; absVel.setAbs( pointVel );
        if ( absVel.greater(info.m_input->m_maxSurfaceVelocity).anyIsSet<hkVector4ComparisonMask::MASK_XYZ>() )
        {
            goto hkSimplexSolverSolve3dSolveSequentually;
        }

    }
    velocityOut = pointVel;
}


inline static void HK_CALL hkSimplexSolverExamineActivePlanes( hkSimplexSolverInfo& info )
{
    while(1)
    {
        switch( info.m_numSupportPlanes )
        {
            case 1:
                {
                    const hkSurfaceConstraintInfo& sci = *info.m_supportPlanes[0].m_constraint;
                    hkSimplexSolverSolve1d( info, sci, info.m_input->m_velocity, info.m_output->m_velocity );
                    return;
                }
            case 2:
                {
                    //
                    //  Test whether we need plane 0
                    //
                    hkVector4 velocity;
                    hkSimplexSolverSolve1d( info, *info.m_supportPlanes[1].m_constraint, info.m_input->m_velocity, velocity );

                    hkBool32 plane0Used = hkSimplexSolverSolveTest1d( *info.m_supportPlanes[0].m_constraint, velocity );
                    if ( !plane0Used )
                    {
                        //
                        //  Now we only need plane 1
                        //  so remove plane 0
                        //
                        info.m_output->m_velocity = velocity;
                        info.m_supportPlanes[0] = info.m_supportPlanes[1];
                        info.m_numSupportPlanes = 1;
                    }
                    else
                    {
                        hkSimplexSolverSolve2d( info, *info.m_supportPlanes[0].m_constraint, *info.m_supportPlanes[1].m_constraint, info.m_input->m_velocity, info.m_output->m_velocity );
                    }
                    return;
                }
            case 3:
                {

                    //
                    // Try to drop both planes
                    //
                    {
                        hkVector4 velocity;
                        hkSimplexSolverSolve1d( info, *info.m_supportPlanes[2].m_constraint, info.m_input->m_velocity, velocity );

                        hkBool32 plane0Used = hkSimplexSolverSolveTest1d( *info.m_supportPlanes[0].m_constraint, velocity );
                        if ( !plane0Used )
                        {
                            hkBool32 plane1Used = hkSimplexSolverSolveTest1d( *info.m_supportPlanes[1].m_constraint, velocity );
                            if ( !plane1Used )
                            {
                                info.m_output->m_velocity = velocity;
                                info.m_supportPlanes[0] = info.m_supportPlanes[2];
                                info.m_numSupportPlanes = 1;
                                continue;
                            }
                        }
                    }


                    //
                    //  Try to drop plane 0 or 1
                    //
                    {
                        // We are trying to a drop a plane. This flag will be set to
                        // true if we are successful.
                        bool dropped_a_plane = false;

                        for ( int testPlane = 0; testPlane < 2; testPlane++)
                        {
                            hkVector4 velocity;
                            hkSimplexSolverSolve2d( info, *info.m_supportPlanes[testPlane].m_constraint, *info.m_supportPlanes[2].m_constraint, info.m_input->m_velocity, velocity );
                            hkBool32 planeUsed = hkSimplexSolverSolveTest1d( *info.m_supportPlanes[1-testPlane].m_constraint, velocity );

                            if ( !planeUsed )
                            {
                                info.m_supportPlanes[0] = info.m_supportPlanes[ testPlane  ];
                                info.m_supportPlanes[1] = info.m_supportPlanes[2];
                                info.m_numSupportPlanes--;
                                dropped_a_plane = true;
                                break;
                            }
                        }

                        // If a plane has been dropped, restart the solver.
                        if (dropped_a_plane)
                        {
                            continue;
                        }
                    }

                    // Otherwise, try and solve all three planes:
                    hkSimplexSolverSolve3d( info, info.m_supportPlanes[0].m_constraint,
                                                  info.m_supportPlanes[1].m_constraint,
                                                  info.m_supportPlanes[2].m_constraint,
                                                  true, info.m_input->m_velocity, info.m_output->m_velocity );
                    return;

                }
            case 4:
                hkSimplexSolverSortInfo( info );
                {
                    bool dropped_a_plane = false;

                    for (int i = 0; i < 3; i++)
                    {
                        hkVector4 velocity;
                        hkSimplexSolverSolve3d( info,
                            info.m_supportPlanes[ (i+1) % 3].m_constraint,
                            info.m_supportPlanes[ (i+2) % 3].m_constraint,
                            info.m_supportPlanes[3].m_constraint, false,
                            info.m_input->m_velocity, velocity );
                        hkBool32 planeUsed = hkSimplexSolverSolveTest1d( *info.m_supportPlanes[i].m_constraint, velocity );
                        if ( !planeUsed )
                        {
                            info.m_supportPlanes[i] = info.m_supportPlanes[2];
                            info.m_supportPlanes[2] = info.m_supportPlanes[3];
                            info.m_numSupportPlanes = 3;
                            dropped_a_plane = true;
                            break;
                        }
                    }

                    if (dropped_a_plane)
                    {
                        continue;
                    }

                    // Nothing can be dropped so we've failed to solve
                    // Now we try all 3d combinations
                    {

                        hkVector4 velocity = info.m_input->m_velocity;
                        const hkSurfaceConstraintInfo* sci0 = info.m_supportPlanes[0].m_constraint;
                        const hkSurfaceConstraintInfo* sci1 = info.m_supportPlanes[1].m_constraint;
                        const hkSurfaceConstraintInfo* sci2 = info.m_supportPlanes[2].m_constraint;
                        const hkSurfaceConstraintInfo* sci3 = info.m_supportPlanes[3].m_constraint;
                        int oldNum = info.m_numSupportPlanes;
                        if ( oldNum == info.m_numSupportPlanes ){   hkSimplexSolverSolve3d( info, sci0, sci1, sci2, false, velocity, velocity ); }
                        if ( oldNum == info.m_numSupportPlanes ){   hkSimplexSolverSolve3d( info, sci0, sci1, sci3, false, velocity, velocity ); }
                        if ( oldNum == info.m_numSupportPlanes ){   hkSimplexSolverSolve3d( info, sci0, sci2, sci3, false, velocity, velocity ); }
                        if ( oldNum == info.m_numSupportPlanes ){   hkSimplexSolverSolve3d( info, sci1, sci2, sci3, false, velocity, velocity ); }

                        info.m_output->m_velocity = velocity;
                    }

                    //
                    //  Search a plane to drop
                    //

                    {
                        //
                        //  Search the highest penalty value
                        //
                        int maxStatus = hkSurfaceConstraintInteraction::HK_STATUS_OK;
                        {
                            for (int j = 0; j < 4; j++)
                            {
                                maxStatus = hkMath::max2( maxStatus, (int)info.m_supportPlanes[j].m_interaction->m_status);
                            }
                        }

                        //
                        // Remove the place with the lowest priority and the highest penalty
                        //
                        {
                            int k;
                            for (k = 0; k < 4; k++)
                            {
                                if ( maxStatus == info.m_supportPlanes[k].m_interaction->m_status)
                                {
                                    info.m_supportPlanes[k] = info.m_supportPlanes[3];
                                    break;
                                }
                            }
                            HK_ASSERT(0x65175dda,  k < 4, "Internal simplex error 2376");
                            info.m_numSupportPlanes--;
                        }

                        //
                        //  Clear penalty flags for the other planes
                        //
                        {
                            for (int l = 0;  l < 3; l++)
                            {
                                info.m_supportPlanes[l].m_interaction->m_status = hkSurfaceConstraintInteraction::HK_STATUS_OK;
                            }
                        }

                        continue;
                    }

                }
            default:
                HK_ASSERT(0x28e265d5,  0, "number of supporting planes is not 1,2,3 or 4" );
                break;
        }
        return;
    }

}

void HK_CALL hkSimplexSolverSolve( const hkSimplexSolverInput& input, hkSimplexSolverOutput& output )
{


    hkSimplexSolverActivePlanes planes = { /*HK_INVALID_OBJECT_INDEX*/ int(0xfafafafa), HK_NULL, HK_NULL };

    hkSimplexSolverInfo info =
    {
        { planes, planes, planes, planes },
        0,              // info.m_numSupportPlanes
        0,              // info.m_currentTime
        &input,         // info.m_input
        &output         // info.m_output
    };

    hkSimdReal remainingTime; remainingTime.load<1>(&input.m_deltaTime);

    hkVector4& velocity = output.m_velocity;
    velocity = input.m_velocity;
    HK_ASSERT_NO_MSG(0x49cfe4a7,  velocity.isOk<3>() );
    hkVector4& position = output.m_position;
    position = input.m_position;

    //
    //  Initialize output
    //
    {
        for (int i = 0; i < input.m_numConstraints; i++ )
        {
            struct hkSurfaceConstraintInteraction& planeInteraction = output.m_planeInteractions[i];

            planeInteraction.m_stopped = false;
            planeInteraction.m_touched = false;
            planeInteraction.m_surfaceTime = 0;
            planeInteraction.m_penaltyDistance = 0;
            planeInteraction.m_status = hkSurfaceConstraintInteraction::HK_STATUS_OK;
        }
    }


    while ( remainingTime.isGreaterEqualZero() )
    {
        //
        // search for a plane which collides our current direction
        //
        int hitIndex = -1;
        hkSimdReal minCollisionTime = remainingTime;

        for ( int i = 0; i < input.m_numConstraints; i++ )
        {
            //  Do not search existing active planes
            if ( info.m_numSupportPlanes >=1 && info.m_supportPlanes[0].m_index == i ){ continue; }
            if ( info.m_numSupportPlanes >=2 && info.m_supportPlanes[1].m_index == i ){ continue; }
            if ( info.m_numSupportPlanes >=3 && info.m_supportPlanes[2].m_index == i ){ continue; }
            if ( output.m_planeInteractions[i].m_status != hkSurfaceConstraintInteraction::HK_STATUS_OK){ continue; }


            // Try to find the plane with the shortest time to move
            const hkSurfaceConstraintInfo& sci = input.m_constraints[i];

            hkVector4 relativeVel; relativeVel.setSub( velocity, sci.m_velocity );
            hkSimdReal relativeProjectedVel = -relativeVel.dot<3>( sci.m_plane );

            //
            // if projected velocity is pointing away skip it
            //
            if ( relativeProjectedVel.isLessEqualZero() ) { continue;   }

            //
            //  Calculate the time of impact
            //
            hkVector4 relativePos; relativePos.setSubMul( position, sci.m_velocity, hkSimdReal::fromFloat(info.m_currentTime) );
            hkSimdReal projectedPos = sci.m_plane.dot4xyz1( relativePos );

            // treat penetrations
            hkSimdReal penaltyDist; penaltyDist.load<1>(&info.m_output->m_planeInteractions[i].m_penaltyDistance);
            projectedPos.zeroIfTrue( projectedPos.lessEqual(hkSimdReal_Eps) );
            projectedPos.add(penaltyDist);

            // check for new hit
            if ( projectedPos.isLess(minCollisionTime * relativeProjectedVel) )
            {
                minCollisionTime = (projectedPos / relativeProjectedVel);
                hitIndex = i;
            }
        }

        //
        //  integrate: Walk to our hitPosition we must walk more than 10 microseconds into the future to consider it valid.
        //
        {
            if ( minCollisionTime.isGreater( hkSimdReal::fromFloat(1e-4f) ) )
            {
                info.m_currentTime += minCollisionTime.getReal();
                remainingTime.sub(minCollisionTime);
                position.addMul( minCollisionTime, velocity );
                for (int j = 0; j < info.m_numSupportPlanes; j++ )
                {
                    hkSimdReal surfaceTime;
                    surfaceTime.load<1>(&info.m_supportPlanes[j].m_interaction->m_surfaceTime);
                    surfaceTime.add(minCollisionTime);
                    surfaceTime.store<1>(&info.m_supportPlanes[j].m_interaction->m_surfaceTime);
                    info.m_supportPlanes[j].m_interaction->m_touched = true;
                }

                output.m_deltaTime = info.m_currentTime;

                hkSimdReal inputMinDeltaTime; inputMinDeltaTime.load<1>(&info.m_input->m_minDeltaTime);
                if (info.m_currentTime > inputMinDeltaTime.getReal() )
                {
                    return;
                }
            }
        }

        //
        //  If we have no hit than we are done
        //
        if ( hitIndex < 0)
        {
            output.m_deltaTime = input.m_deltaTime;
            break;
        }

        //
        //  Add our hit to our current list of active planes
        //
        {
            hkSimplexSolverActivePlanes& sp = info.m_supportPlanes[ info.m_numSupportPlanes++ ];
            sp.m_constraint = &input.m_constraints[hitIndex];
            sp.m_interaction = &output.m_planeInteractions[hitIndex];
            sp.m_interaction->m_penaltyDistance = (sp.m_interaction->m_penaltyDistance + HK_REAL_EPSILON) * hkReal(2);
            sp.m_index = hitIndex;
        }

        //
        // Move our character along the current set of active planes
        //
        {
            hkSimplexSolverExamineActivePlanes( info );
        }
    }

    HK_ASSERT_NO_MSG(0x5624faf6,  output.m_velocity.isOk<3>() );
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
