// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !REFLECT
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Geometry/Collide/GeometryProcessing/Triangulator/hkgpTriangulator.h>
#include <Common/Base/System/Io/IArchive/hkIArchive.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>

#if !defined(HKGP_TRIANGULATOR_ENABLE_READ_WRITE)
    #error HKGP_TRIANGULATOR_ENABLE_READ_WRITE is not defined
#endif

struct hkgpTriangulatorDebugger : public hkgpTriangulator
{
    typedef hkgpTriangulator Parent;

    struct Record
    {
        virtual             ~Record() {}
        virtual hkUint32    getType() const=0;
        virtual void        write(hkOArchive& file) const=0;
        virtual void        read(hkIArchive& file)=0;
        virtual void        execute(hkgpTriangulator& triangulator) const=0;
    };

    struct InsertCrossingVertexRecord : public Record
    {
        HK_DECLARE_CLASS(InsertCrossingVertexRecord, NewOpaque);
        static hkUint32     type()          { return 0x6ACA37DD; }
        hkUint32            getType() const { return type(); }
        inline void         write(hkOArchive& file) const;
        inline void         read(hkIArchive& file);
        inline void         execute(hkgpTriangulator& triangulator) const;
        Vertex              m_vertex;
        bool                m_conform;
    };

    struct InsertCrossingEdgeRecord : public Record
    {
        HK_DECLARE_CLASS(InsertCrossingEdgeRecord, NewOpaque);
        static hkUint32     type()          { return 0x43D1C80F; }
        hkUint32            getType() const { return type(); }
        inline void         write(hkOArchive& file) const;
        inline void         read(hkIArchive& file);
        inline void         execute(hkgpTriangulator& triangulator) const;
        Vertex              m_vertices[2];
        EdgeDataValue       m_data;
        bool                m_conform;
    };

    inline              hkgpTriangulatorDebugger();
    inline              ~hkgpTriangulatorDebugger();

    inline void         setRecordEnable(bool enable);
    inline bool         writeRecords(const char* filename) const;
    inline bool         readRecords(const char* filename);
    inline void         clearRecords();
    inline void         replayRecords();
    inline void         fetchLoops(hkArray<hkArray<Vertex>*>& loops) const;

    /* Hooks    */
    template <typename EDGECROSSING>
    inline Insertion    insertVertex(const Vertex& v, bool conform, const EDGECROSSING& icrossing);
    template <typename EDGECROSSING>
    inline Insertion    insertVertex(hkInt32 x, hkInt32 y, bool conform, const EDGECROSSING& icrossing);
    inline Insertion    insertVertex(const Vertex& v, bool conform);
    template <typename EDGECROSSING>
    inline Insertion    insertCrossingEdge(const Vertex& v0,const Vertex& v1, bool conform,const EDGECROSSING& icrossing, const EdgeData& data);
    template <typename EDGECROSSING>
    inline Insertion    insertCrossingEdge(hkInt32 x0, hkInt32 y0, hkInt32 x1, hkInt32 y1, bool conform,const EDGECROSSING& icrossing, const EdgeData& data);
    template <typename EDGECROSSING>
    inline Insertion    insertCrossingEdge(const Vertex& v0,const Vertex& v1, bool conform,const EDGECROSSING& icrossing, const EdgeData* data=HK_NULL);
    template <typename EDGECROSSING>
    inline Insertion    insertCrossingEdge(hkInt32 x0, hkInt32 y0, hkInt32 x1, hkInt32 y1, bool conform,const EDGECROSSING& icrossing, const EdgeData* data=HK_NULL);

    hkArray<Record*>                m_records;
    hkArray<hkArray<hkVector4> >    m_userLoops;
    bool                            m_doRecord;
};

//
// Inline's
//

//
inline                  hkgpTriangulatorDebugger::hkgpTriangulatorDebugger() : Parent()
{
    m_doRecord = true;
}

//
inline                  hkgpTriangulatorDebugger::~hkgpTriangulatorDebugger()
{
    clearRecords();
}

//
inline void             hkgpTriangulatorDebugger::InsertCrossingVertexRecord::write(hkOArchive& file) const
{
    file.write32(m_vertex.m_x);
    file.write32(m_vertex.m_y);
    file.write8u(m_conform?1:0);
}

//
inline void             hkgpTriangulatorDebugger::InsertCrossingVertexRecord::read(hkIArchive& file)
{
    m_vertex.m_x=file.read32();
    m_vertex.m_y=file.read32();
    m_conform=file.read8u()==1?true:false;
}

//
inline void             hkgpTriangulatorDebugger::InsertCrossingVertexRecord::execute(hkgpTriangulator& triangulator) const
{
    triangulator.insertVertex(m_vertex,m_conform,hkgpTriangulator::EdgeCrossing::dummy());
}

//
inline void             hkgpTriangulatorDebugger::InsertCrossingEdgeRecord::write(hkOArchive& file) const
{
    file.write32(m_vertices[0].m_x);
    file.write32(m_vertices[0].m_y);
    file.write32(m_vertices[1].m_x);
    file.write32(m_vertices[1].m_y);
    file.write8u(m_conform?1:0);
    file.write8u(m_data.m_hasData?1:0);
    if(m_data.m_hasData)
    {
        m_data.m_data.write(file);
    }
}

//
inline void             hkgpTriangulatorDebugger::InsertCrossingEdgeRecord::read(hkIArchive& file)
{
    m_vertices[0].m_x=file.read32();
    m_vertices[0].m_y=file.read32();
    m_vertices[1].m_x=file.read32();
    m_vertices[1].m_y=file.read32();
    m_conform=file.read8u()==1;
    m_data.m_hasData=file.read8u()==1;
    if(m_data.m_hasData)
    {
        m_data.m_data.read(file);
    }
}

//
inline void             hkgpTriangulatorDebugger::InsertCrossingEdgeRecord::execute(hkgpTriangulator& triangulator) const
{
    triangulator.insertCrossingEdge(m_vertices[0],m_vertices[1],m_conform,hkgpTriangulator::EdgeCrossing::dummy(),m_data.m_hasData?&m_data.m_data:HK_NULL);
}

//
inline void             hkgpTriangulatorDebugger::setRecordEnable(bool enable)
{
    m_doRecord = enable;
}

//
inline bool             hkgpTriangulatorDebugger::writeRecords(const char* filename) const
{
    hkOArchive  file(filename);
    if(file.isOk())
    {
        file.write32(m_records.getSize());
        for(int i=0;i<m_records.getSize();++i)
        {
            file.write32u(m_records[i]->getType());
            m_records[i]->write(file);
        }
        file.write32(m_userLoops.getSize());
        for(int i=0;i<m_userLoops.getSize();++i)
        {
            const hkArray<hkVector4>&   loop=m_userLoops[i];
            file.write32(loop.getSize());
            for(int j=0;j<loop.getSize();++j)
            {
                file.writeFloat32(hkFloat32(loop[j](0)));
                file.writeFloat32(hkFloat32(loop[j](1)));
                file.writeFloat32(hkFloat32(loop[j](2)));
                file.writeFloat32(hkFloat32(loop[j](3)));
            }
        }
        return true;
    }
    return false;
}

//
inline bool             hkgpTriangulatorDebugger::readRecords(const char* filename)
{
    hkIArchive  file(filename);
    clearRecords();
    m_userLoops.clear();
    if(file.isOk())
    {
        m_records.setSize(file.read32(),HK_NULL);
        for(int i=0;i<m_records.getSize();++i)
        {
            const hkUint32  type=file.read32u();
            Record*         record=HK_NULL;
            if(type==InsertCrossingVertexRecord::type())    record=new InsertCrossingVertexRecord();
            else if(type==InsertCrossingEdgeRecord::type()) record=new InsertCrossingEdgeRecord();
            if(record)
            {
                record->read(file);
                m_records[i]=record;
            }
        }
        m_userLoops.setSize(file.read32());
        for(int i=0;i<m_userLoops.getSize();++i)
        {
            hkArray<hkVector4>& loop=m_userLoops[i];
            loop.setSize(file.read32());
            for(int j=0;j<loop.getSize();++j)
            {
                loop[j](0)=file.readFloat32();
                loop[j](1)=file.readFloat32();
                loop[j](2)=file.readFloat32();
                loop[j](3)=file.readFloat32();
            }
        }
        return true;
    }
    return false;
}

//
inline void             hkgpTriangulatorDebugger::clearRecords()
{
    for(int i=0;i<m_records.getSize();++i) delete m_records[i];
    m_records.clear();
}

//
inline void             hkgpTriangulatorDebugger::replayRecords()
{
    for(int i=0;i<m_records.getSize();++i)
    {
        m_records[i]->execute(*this);
    }
}

//
inline void             hkgpTriangulatorDebugger::fetchLoops(hkArray<hkArray<Vertex>*>& loops) const
{
    hkArray<Vertex>*    loop=HK_NULL;
    Vertex              lastVertex;
    for(int i=0;i<m_records.getSize();++i)
    {
        if(m_records[i]->getType()==InsertCrossingVertexRecord::type() && loop!=HK_NULL)
        {
            loop=HK_NULL;
        }
        if(m_records[i]->getType()==InsertCrossingEdgeRecord::type())
        {
            const InsertCrossingEdgeRecord& e=*(const InsertCrossingEdgeRecord*)m_records[i];
            if(!loop || !(lastVertex==e.m_vertices[0]))
            {
                loop=new hkArray<Vertex>();
                loops.pushBack(loop);
                loop->pushBack(e.m_vertices[0]);
            }
            loop->pushBack(e.m_vertices[1]);
            lastVertex=e.m_vertices[1];
        }
    }
}

//
template <typename EDGECROSSING>
inline hkgpTriangulator::Insertion  hkgpTriangulatorDebugger::insertVertex(const Vertex& v, bool conform, const EDGECROSSING& icrossing)
{
    if(m_doRecord)
    {
        InsertCrossingVertexRecord* rec=new InsertCrossingVertexRecord();
        rec->m_vertex           =   v;
        rec->m_conform          =   conform;
        m_records.pushBack(rec);
    }
    return Parent::insertVertex(v,conform,icrossing);
}

//
template <typename EDGECROSSING>
inline hkgpTriangulator::Insertion  hkgpTriangulatorDebugger::insertVertex(hkInt32 x, hkInt32 y, bool conform, const EDGECROSSING& icrossing)
{
    Vertex  v;
    v.m_x=x;v.m_y=y;
    return insertVertex(v,conform,icrossing);
}

//
inline hkgpTriangulator::Insertion  hkgpTriangulatorDebugger::insertVertex(const Vertex& v, bool conform)
{
    return Parent::insertVertex(v,conform);
}

//
template <typename EDGECROSSING>
inline hkgpTriangulator::Insertion  hkgpTriangulatorDebugger::insertCrossingEdge(const Vertex& v0,const Vertex& v1, bool conform,const EDGECROSSING& icrossing, const EdgeData& data)
{
    if(m_doRecord)
    {
        InsertCrossingEdgeRecord*   rec=new InsertCrossingEdgeRecord();
        rec->m_vertices[0]      =   v0;
        rec->m_vertices[1]      =   v1;
        rec->m_data.m_hasData   =   true;
        rec->m_data.m_data      =   data;
        rec->m_conform          =   conform;
        m_records.pushBack(rec);
    }
    return Parent::insertCrossingEdge(v0,v1,conform,icrossing,data);
}

//
template <typename EDGECROSSING>
inline hkgpTriangulator::Insertion  hkgpTriangulatorDebugger::insertCrossingEdge(hkInt32 x0, hkInt32 y0, hkInt32 x1, hkInt32 y1, bool conform,const EDGECROSSING& icrossing, const EdgeData& data)
{
    Vertex  v0,v1;
    v0.m_x=x0;v0.m_y=y0;
    v1.m_x=x1;v1.m_y=y1;
    return insertCrossingEdge(v0,v1,conform,icrossing,data);
}

//
template <typename EDGECROSSING>
inline hkgpTriangulator::Insertion  hkgpTriangulatorDebugger::insertCrossingEdge(const Vertex& v0,const Vertex& v1, bool conform,const EDGECROSSING& icrossing, const EdgeData* data)
{
    if(m_doRecord)
    {
        InsertCrossingEdgeRecord*   rec=new InsertCrossingEdgeRecord();
        rec->m_vertices[0]      =   v0;
        rec->m_vertices[1]      =   v1;
        rec->m_data.m_hasData   =   data!=HK_NULL;
        if(data) rec->m_data.m_data     =   *data;
        rec->m_conform          =   conform;
        m_records.pushBack(rec);
    }
    return Parent::insertCrossingEdge(v0,v1,conform,icrossing,data);
}

//
template <typename EDGECROSSING>
inline hkgpTriangulator::Insertion  hkgpTriangulatorDebugger::insertCrossingEdge(hkInt32 x0, hkInt32 y0, hkInt32 x1, hkInt32 y1, bool conform,const EDGECROSSING& icrossing, const EdgeData* data)
{
    Vertex  v0,v1;
    v0.m_x=x0;v0.m_y=y0;
    v1.m_x=x1;v1.m_y=y1;
    return insertCrossingEdge(v0,v1,conform,icrossing,data);
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
