// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Geometry/Collide/GeometryProcessing/AbstractMesh/hkgpAbstractMesh.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Base/Math/Quaternion/hkQuaternionUtil.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

#if (HK_PLATFORM_IS_CONSOLE == 1)
#   define HKGP_INLINE
#else
#   define HKGP_INLINE inline
#endif

// To enable draw method, define:
//#define HKGP_TRIANGULATOR_ENABLE_DRAW

// To enable read/write methods, define:
//#define HKGP_TRIANGULATOR_ENABLE_READ_WRITE

// To enable benchmarking method, define:
//#define HKGP_TRIANGULATOR_ENABLE_BENCHMARK

class hkStringBuf;

///
/// hkgpTriangulatorBase
///
class  hkgpTriangulatorBase : public hkReferencedObject
{
public:
    HK_DECLARE_CLASS(hkgpTriangulatorBase, NewOpaque);
    struct TriangleBase
    {
        //+hk.MemoryTracker(ignore=True)
    };
    struct VertexBase
    {
        //+hk.MemoryTracker(ignore=True)
    };

    /// Default edge data
    /// store hkULong tags using dynamic allocations.
    template <typename ALLOCATOR>
    struct DefaultEdgeData
    {
        HK_DECLARE_CLASS(DefaultEdgeData, NewOpaque);
        HKGP_FORCE_INLINE                       DefaultEdgeData()                               {}
        HKGP_FORCE_INLINE                       DefaultEdgeData(hkUlong tag)                    { m_tags.pushBack(tag); }
        HKGP_FORCE_INLINE                       DefaultEdgeData(const void* tag)                { m_tags.pushBack(*(const hkUlong*)&tag); }
        HKGP_FORCE_INLINE void              appendOrReplace(const DefaultEdgeData& newData) { for(int i=0;i<newData.m_tags.getSize();++i) { if(m_tags.indexOf(newData.m_tags[i])==-1) m_tags.pushBack(newData.m_tags[i]); } }
#if defined(HKGP_TRIANGULATOR_ENABLE_READ_WRITE)
        inline void                         read(class hkIArchive&);
        inline void                         write(class hkOArchive&) const;
#endif
        hkInplaceArray<hkUlong,2,ALLOCATOR> m_tags;
    };

    ///
    /// Edge data abstract policy
    ///
    /// The following policies are derived from it:
    ///     - SparseEdgeDataPolicy : By-ref storage.
    ///     - DenseEdgeDataPolicy : By-value storage.
    ///     - NoEdgeDataPoliciy : No edge data allowed nor stored.
    ///
    template <typename EDGEDATA>
    struct EdgeDataPolicy
    {
        struct EdgeDataValue
        {
            HKGP_FORCE_INLINE           EdgeDataValue() : m_hasData(false) {}
            bool        m_hasData;
            EDGEDATA    m_data;
        };

        struct EdgeDataState
        {
            EdgeDataValue   m_values[2];
        };

        /// Retrieve data on both side of an edge and clear them.
        template <typename TRIANGULATOR, typename EDGE>
        inline void                         backupBothEdgeData(TRIANGULATOR& triangulator, EDGE edge, EdgeDataState& state);

        /// Restore data on both side of an edge.
        template <typename TRIANGULATOR, typename EDGE>
        inline void                         restoreBothEdgeData(TRIANGULATOR& triangulator, EDGE edge, const EdgeDataState& state);
    };

    ///
    /// Sparse edge data policy, stored in hash-table
    ///
    template <typename EDGEDATA,typename ALLOCATOR>
    struct SparseEdgeDataPolicy : public EdgeDataPolicy<EDGEDATA>
    {
        HK_DECLARE_CLASS(SparseEdgeDataPolicy, NewOpaque);

        /// Triangle extras data.
        struct TriangleExtras
        {};

        /// EdgeDataBox
        struct EdgeDataBox
        {
            HK_DECLARE_CLASS(EdgeDataBox, NewOpaque);
            HKGP_FORCE_INLINE               EdgeDataBox() : m_data(HK_NULL)             { m_vertices[0]=m_vertices[1]=HK_NULL; }
            template <typename EDGE>
            HKGP_FORCE_INLINE               EdgeDataBox(EDGE edge) : m_data(HK_NULL)    { m_vertices[0]=edge.start(); m_vertices[1]=edge.end(); m_hash=hkGeometryProcessing::makeHash(edge.start()->pack(),edge.end()->pack()); }
            HKGP_FORCE_INLINE hkUlong       getHash() const                             { return m_hash; }
            HKGP_FORCE_INLINE hkBool32      operator==(const EdgeDataBox& other) const  { return m_vertices[0]==other.m_vertices[0] && m_vertices[1]==other.m_vertices[1]; }

            void*       m_vertices[2];
            hkUlong     m_hash;
            EDGEDATA*   m_data;
        };

        /// ctor.
        SparseEdgeDataPolicy() : m_edgeData(HK_NULL) {}

        /// dtor.
        ~SparseEdgeDataPolicy();

        /// Create edge data hash table.
        inline void                             createEdgeDataHashTable();

        /// Get edge data
        template <typename EDGE>
        inline const EDGEDATA*                  getEdgeData(const EDGE& edge) const;

        /// Set edge data
        template <typename EDGE>
        inline void                             setEdgeData(const EDGE& edge, const EDGEDATA& data);

        /// Clear edge data
        template <typename EDGE>
        inline void                             clearEdgeData(const EDGE& edge);

        /// Flip edge
        template <typename EDGE>
        HKGP_FORCE_INLINE EDGE                  flipEdge(const EDGE& edge) const { return edge.flip(); }

        /// Propagate data on split triangle.
        template <typename EDGE,typename TRIANGLE>
        inline void                             propagateDataOnSplitTriangle(const EDGE& edge,TRIANGLE* t0,TRIANGLE* t1) const {}

        /// Propagate data on split bound edge.
        template <typename EDGE,typename TRIANGLE>
        inline void                             propagateDataOnSplitBoundEdge(const EDGE& edge,const EDGE& link,TRIANGLE* t0,TRIANGLE* t1) const {}

        /// Propagate data on split naked edge.
        template <typename EDGE,typename TRIANGLE>
        inline void                             propagateDataOnSplitNakedEdge(const EDGE& edge,TRIANGLE* t0) const {}

        /// Reset
        inline void                             reset() { m_edgeDataAllocator.clear(); if(m_edgeData) m_edgeData->removeAll(); }

        typedef hkGeometryProcessing::HashTable<EdgeDataBox,ALLOCATOR>  EdgeDataHashTable;

        EdgeDataHashTable*                                          m_edgeData;
        hkGeometryProcessing::PoolAllocator<EDGEDATA,32,ALLOCATOR>  m_edgeDataAllocator;

    };

    ///
    /// Dense edge data policy, stored in triangles.
    ///
    template <typename EDGEDATA,typename ALLOCATOR>
    struct DenseEdgeDataPolicy : public EdgeDataPolicy<EDGEDATA>
    {
        /// EdgeDataValue
        typedef typename EdgeDataPolicy<EDGEDATA>::EdgeDataValue    EdgeDataValue;

        /// Extras data to attach to triangles.
        struct TriangleExtras
        {
            EdgeDataValue   m_edgeData[3];
        };

        /// Get edge data
        template <typename EDGE>
        inline const EDGEDATA*                  getEdgeData(const EDGE& edge) const;

        /// Set edge data
        template <typename EDGE>
        inline void                             setEdgeData(const EDGE& edge, const EDGEDATA& data);

        /// Clear edge data
        template <typename EDGE>
        inline void                             clearEdgeData(const EDGE& edge);

        /// Get edge data pointer reference.
        template <typename EDGE>
        inline EdgeDataValue&                   getEdgeDataRef(const EDGE& edge) { return edge.triangle()->m_edgeData[edge.index()]; }

        /// Flip edge
        template <typename EDGE>
        inline EDGE                             flipEdge(const EDGE& edge);

        /// Propagate data on split triangle.
        template <typename EDGE,typename TRIANGLE>
        inline void                             propagateDataOnSplitTriangle(const EDGE& edge,TRIANGLE* t0,TRIANGLE* t1);

        /// Propagate data on split bound edge.
        template <typename EDGE,typename TRIANGLE>
        inline void                             propagateDataOnSplitBoundEdge(const EDGE& edge,const EDGE& link,TRIANGLE* t0,TRIANGLE* t1);

        /// Propagate data on split naked edge.
        template <typename EDGE,typename TRIANGLE>
        inline void                             propagateDataOnSplitNakedEdge(const EDGE& edge,TRIANGLE* t0);

        /// Reset
        HKGP_FORCE_INLINE void                  reset() const {}
    };

    ///
    /// No edge data policy.
    ///
    template <typename EDGEDATA,typename ALLOCATOR>
    struct NoEdgeDataPolicy : public EdgeDataPolicy<EDGEDATA>
    {
        /// Extras data to attach to triangles.
        struct TriangleExtras {};

        /// EdgeData state
        struct EdgeDataState {};

        /// Retrieve data on both side of an edge and clear them.
        template <typename TRIANGULATOR, typename EDGE>
        HKGP_FORCE_INLINE void                  backupBothEdgeData(TRIANGULATOR&, EDGE, EdgeDataState&) const {}

        /// Restore data on both side of an edge.
        template <typename TRIANGULATOR, typename EDGE>
        HKGP_FORCE_INLINE void                  restoreBothEdgeData(TRIANGULATOR&, EDGE, const EdgeDataState&) const {}

        /// Get edge data
        template <typename EDGE>
        HKGP_FORCE_INLINE const EDGEDATA*       getEdgeData(const EDGE&) const { return HK_NULL; }

        /// Set edge data
        template <typename EDGE>
        HKGP_FORCE_INLINE void                  setEdgeData(const EDGE&, const EDGEDATA&) const { HK_ERROR(0x1F851459,"Edge data not allowed by the policy"); }

        /// Clear edge data
        template <typename EDGE>
        HKGP_FORCE_INLINE void                  clearEdgeData(const EDGE&) const {}

        /// Flip edge
        template <typename EDGE>
        HKGP_FORCE_INLINE EDGE                  flipEdge(const EDGE& edge) const { return edge.flip(); }

        /// Propagate data on split triangle.
        template <typename EDGE,typename TRIANGLE>
        HKGP_FORCE_INLINE void                  propagateDataOnSplitTriangle(const EDGE&,TRIANGLE*,TRIANGLE*) const {}

        /// Propagate data on split bound edge.
        template <typename EDGE,typename TRIANGLE>
        HKGP_FORCE_INLINE void                  propagateDataOnSplitBoundEdge(const EDGE&,const EDGE&,TRIANGLE*,TRIANGLE*) const {}

        /// Propagate data on split naked edge.
        template <typename EDGE,typename TRIANGLE>
        HKGP_FORCE_INLINE void                  propagateDataOnSplitNakedEdge(const EDGE&,TRIANGLE*) const {}

        /// Reset
        HKGP_FORCE_INLINE void                  reset() const {}
    };
};

///
/// Extended integer type from BITS
///
template <bool USE_EXTENTED_PRECISION>  struct hkgpTriangulatorTypeFromBits         {};
template <>                             struct hkgpTriangulatorTypeFromBits<true>   { typedef hkInt64 Type; };
template <>                             struct hkgpTriangulatorTypeFromBits<false>  { typedef hkInt32 Type; };

///
/// Conformity predicates.
///
template <bool USE_DELAUNAY_PREDICATE>  struct hkgpTriangulatorConformPredicate     {};

/* Delaunay             */
template <>                             struct hkgpTriangulatorConformPredicate<true>
{
    template <typename TRIANGULATOR,typename VERTEX>
    static HKGP_FORCE_INLINE hkBool32   isConform(TRIANGULATOR*,const VERTEX* a,const VERTEX* b,const VERTEX* c,const VERTEX* p)
    {
        return TRIANGULATOR::notInCircle(a->m_x,a->m_y,b->m_x,b->m_y,c->m_x,c->m_y,p->m_x,p->m_y);
    }
};

/* Approximate Delaunay */
#if 1
template <>                             struct hkgpTriangulatorConformPredicate<false>
{
    template <typename TRIANGULATOR,typename VERTEX>
    static HKGP_FORCE_INLINE hkBool32   isConform(TRIANGULATOR*,const VERTEX* a,const VERTEX* b,const VERTEX* c,const VERTEX* p)
    {
        return TRIANGULATOR::notInCircleFloat(a->m_x,a->m_y,b->m_x,b->m_y,c->m_x,c->m_y,p->m_x,p->m_y);
    }
};
#else
/* Min-edge             */
template <>                             struct hkgpTriangulatorConformPredicate<false>
{
    template <typename TRIANGULATOR,typename VERTEX>
    static HKGP_FORCE_INLINE hkBool32   isConform(TRIANGULATOR*,const VERTEX* a,const VERTEX* b,const VERTEX* c,const VERTEX* p)
    {
        return TRIANGULATOR::distanceSqrd(b,c) <= TRIANGULATOR::distanceSqrd(a,p);
    }
};
#endif

///
/// Planar constrained Delaunay triangulation.
/// Notes: if BITS is set to an higher value than 15, the in-circle predicate is replaced by an approximate predicate to ensure that no overflow occurs.
///
template <  typename ALLOCATOR = hkContainerHeapAllocator,                                              // Allocator.
            typename VERTEX = hkgpTriangulatorBase::VertexBase,                                         // Vertex type.
            typename TRIANGLE = hkgpTriangulatorBase::TriangleBase,                                     // Triangle type.
            typename EDGEDATA = hkgpTriangulatorBase::DefaultEdgeData<ALLOCATOR>,                       // Edge data type.
            typename EDGEDATA_POLICY = hkgpTriangulatorBase::SparseEdgeDataPolicy<EDGEDATA,ALLOCATOR>,  // Edge data policy.
            const int ORIENTATION=-1,                                                                   // Orientation CW = -1 and CCW = +1.
            const int ROOTS=4,                                                                          // Number of roots.
            const int BITS=15,                                                                          // Number bits used in integer space (up to 29).
            const bool USE_FAST_CONFORM=false>                                                          // Force the use of the fast predicate.
class hkgpTriangulatorType : public hkgpTriangulatorBase
{
public:
    HK_DECLARE_CLASS(hkgpTriangulatorType, NewOpaque);
    //
    // Configuration
    //
    enum
    {
        cfgDomainRange      =   BITS,   ///< Domain range as 2^cfgDomainRange, same for 32 and 64 bits platforms for deterministic results.
        cfgRoots            =   ROOTS,  ///< Number of root used to speed up vertex location as 2^cfgRoots.
        cfgStacksize        =   64,     ///< Stack size.
        cfgSearchRadius     =   32,     ///< Maximum search radius when resolving edge crossing.
        cfgSearchSteps      =   128,    ///< Maximum search steps when resolving edge crossing.
        cfgBoundary         =   (1<<cfgDomainRange)-1,
        cfgNumRoots         =   1<<cfgRoots,
        cfgRootShift        =   cfgDomainRange-cfgRoots,
        cfgTotalRoots       =   cfgNumRoots*cfgNumRoots,
        cfgOrientation      =   ORIENTATION,
        cfgIsCW             =   ORIENTATION==-1,
        cfgIsCCW            =   ORIENTATION==+1,
        cfgExtentedInt      =   BITS>15,
        cfgUseDelaunay      =   USE_FAST_CONFORM?false:BITS<=15,
        cfgEND
    };

    enum Status
    {
        STATUS_OK = 0,
        STATUS_OUT_OF_MEMORY,
    };

    typedef typename hkgpTriangulatorTypeFromBits<cfgExtentedInt?true:false>::Type
                                                                        tExtInt;            ///< Upper precision integer.
    typedef hkInt64                                                     tBigInt;            ///< Type to used for computations that required more than 31 bits.
    typedef hkUint16                                                    tSets;              ///< Type used to store sets.
    typedef ALLOCATOR                                                   tAllocator;         ///< Type of the allocator.
    typedef EDGEDATA                                                    EdgeData;           ///< Edge data type.
    typedef EDGEDATA_POLICY                                             EdgeDataPolicy;     ///< Edge data policy.
    typedef typename EdgeDataPolicy::EdgeDataValue                      EdgeDataValue;      ///< Edge data value.
    typedef typename EdgeDataPolicy::EdgeDataState                      EdgeDataState;      ///< Edge data state.
    typedef hkgpTriangulatorConformPredicate<cfgUseDelaunay>            ConformPredicate;   ///< Conformity predicate.

    //
    // Types
    //

    /// Vertex
    struct Vertex : public hkgpAbstractMeshDefinitions::Vertex<Vertex,VERTEX>
    {
        enum { VERTEX_DATA_SIZE_IN_BITS = sizeof(hkUlong)*8-2 };
        HK_DECLARE_CLASS(Vertex, NewOpaque);
        Vertex() : hkgpAbstractMeshDefinitions::Vertex<Vertex,VERTEX>() {}
        HKGP_FORCE_INLINE hkBool32  isBoundary() const                  { return((m_x==0)||(m_x==cfgBoundary)||(m_y==0)||(m_y==cfgBoundary)); }
        HKGP_FORCE_INLINE hkBool32  isCorner() const                    { return(   ((m_x==0)&&(m_y==0))||((m_x==cfgBoundary)&&(m_y==cfgBoundary))||
                                                                                    ((m_x==cfgBoundary)&&(m_y==0))||((m_x==0)&&(m_y==cfgBoundary))); }
        HKGP_FORCE_INLINE hkBool32  isConstrained() const               { return(m_constraint!=0); }
        HKGP_FORCE_INLINE hkBool32  isEqual(int x,int y) const          { return((m_x==x)&&(m_y==y)); }
        HKGP_FORCE_INLINE void      setConstrained()                    { m_constraint=1; }
        HKGP_FORCE_INLINE void      clrConstrained()                    { m_constraint=0; }
        HKGP_FORCE_INLINE hkBool32  operator==(const Vertex& v) const   { return((m_x==v.m_x)&&(m_y==v.m_y)); }
        HKGP_FORCE_INLINE hkBool32  operator!=(const Vertex& v) const   { return((m_x!=v.m_x)||(m_y!=v.m_y)); }
        HKGP_FORCE_INLINE hkUlong       pack() const                    { return((hkUlong) ((m_x<<cfgDomainRange)+m_y)); }
        HKGP_FORCE_INLINE int           compare(const Vertex& v) const
        {
            if(m_x<v.m_x)       return(-1);
            else if(m_x>v.m_x)  return(+1);
            else if(m_y<v.m_y)  return(-1);
            else if(m_y>v.m_y)  return(+1);
            else                return(0);
        }
        hkInt32     m_x;
        hkInt32     m_y;
        hkUlong     m_constraint:1;
        hkUlong     m_marker:1;
        hkUlong     m_data:VERTEX_DATA_SIZE_IN_BITS;
    };

    /// Triangle
    struct Triangle : public hkgpAbstractMeshDefinitions::Triangle<Triangle,TRIANGLE,Vertex*> , public EdgeDataPolicy::TriangleExtras
    {
        HK_DECLARE_CLASS(Triangle, NewOpaque);
        Triangle() : hkgpAbstractMeshDefinitions::Triangle<Triangle,TRIANGLE,Vertex*>() {}
        hkUint16    m_constraints:3;
        hkUint16    m_root:1;
        hkUint16    m_marker:1;
        hkUint16    m_region:11;
        tSets       m_sets;
    };

    /// Edge
    struct Edge : public hkgpTopology::Edge<Edge,Vertex,Triangle>
    {
        HK_DECLARE_CLASS(Edge, NewOpaque);
        typedef hkgpTopology::Edge<Edge,Vertex,Triangle> tBase;
        HKGP_FORCE_INLINE               Edge() : tBase()                            {}
        HKGP_FORCE_INLINE               Edge(Triangle* t,unsigned i) : tBase(t,i)   {}
        HKGP_FORCE_INLINE               Edge(hkgpTopology::Uid id) : tBase(id)      {}
        HKGP_FORCE_INLINE hkBool32  isConstrained() const           { const Edge e=tBase::master();return((e.triangle()->m_constraints&(1<<e.index()))); }
        HKGP_FORCE_INLINE void      setConstrained() const          { const Edge e=tBase::master();e.triangle()->m_constraints|=1<<e.index(); }
        HKGP_FORCE_INLINE void      clrConstrained() const          { const Edge e=tBase::master();e.triangle()->m_constraints&=~(1<<e.index()); }
        HKGP_FORCE_INLINE hkBool32  getAndClearConstraint() const;
        HKGP_FORCE_INLINE void      turnCcwInPlace()                { tBase::linkInPlace();tBase::nextInPlace(); }
        HKGP_FORCE_INLINE void      turnCwInPlace()                 { tBase::prevInPlace();tBase::linkInPlace(); }
        HKGP_FORCE_INLINE hkBool32  canFlip() const;
        HKGP_FORCE_INLINE Edge      flip() const;
        HKGP_FORCE_INLINE int           rank() const;
    };

    /// EdgeSet
    struct EdgeSet
    {
        EdgeSet()                                               {}
        EdgeSet(const Edge& e,hkUint16 s) : m_edge(e),m_set(s)  {}
        Edge        m_edge;
        tSets       m_set;
    };

    /// Boundary
    struct Boundary
    {
        Edge        m_link;
        Vertex*     m_vertex;
        Triangle*   m_triangle;
        Boundary*   m_next;
        hkBool      m_constrained;
    };

    /// Mesh
    typedef hkgpAbstractMesh<Edge,Vertex,Triangle,ALLOCATOR>    Mesh;

    /// Edge UID
    typedef hkgpTopology::Uid                                   Uid;

    /// SortByArea_Predicate
    struct SortByArea_Predicate
    {
        HKGP_FORCE_INLINE hkBool32  operator()(const Triangle* a,const Triangle* b) const;
    };

    /// Location result
    struct Location
    {
        enum eType
        {
            IN_TRIANGLE,
            ON_EDGE,
            ON_VERTEX,
            NOT_FOUND,
            CYCLE,
        };
        hkEnum<eType, int> m_type;
        Edge        m_edge;
        Location()                                              {}
        Location(eType t,const Edge& e) : m_type(t),m_edge(e)   {}
    };

    /// Insertion result
    struct Insertion
    {
        enum eType
        {
            SUCCESS,
            ALREADY_PRESENT,
            ON_CONSTRAINED_EDGE,
            CROSSING_CONSTRAINED_EDGE,
            CROSSING_CONSTRAINED_VERTEX,
            CANNOT_RESOLVE_CROSSING,
            CANNOT_LOCATE_ORIGIN,
            CANNOT_LOCATE_DESTINATION,
            INVALID_ORIENTATION,
            EDGE_IS_VERTEX,
            OUT_OF_MEMORY,              // This is hard failure, triangulator will not be able to continue operations after this.
            FAILED
        };
        hkEnum<eType, int> m_type;
        Edge        m_edge;
        Insertion()                                             {}
        Insertion(eType t,const Edge& e) : m_type(t),m_edge(e)  {}
        HKGP_FORCE_INLINE hkBool32  isValid() const                 { return(m_type==SUCCESS); }
    };

    /// Removal result
    struct Removal
    {
        enum eType
        {
            SUCCESS,
            BELONG_TO_CONSTRAINT,
            NOT_FOUND,
            FAILED,
        };
        hkEnum<eType, int> m_type;
        Removal()                                               {}
        Removal(eType t) : m_type(t)                            {}
        HKGP_FORCE_INLINE hkBool32  isValid() const                 { return(m_type==SUCCESS); }
    };

    /// EdgeCrossing
    /// interface template used to notify of events during edge insertion with crossing enable
    struct EdgeCrossing
    {
        /// The edge or vertex insertion is crossing the edge 'edge'.
        HKGP_FORCE_INLINE void  notifyEdgeCrossing(Edge edge) const {}

        /// The edge or vertex insertion is crossing the edge 'edgeStart' -> 'edgeEnd' at 'fraction' .
        /// The new vertex created along the edge to be inserted is 'newVertex'.
        /// \note fraction and fractionOther might not be accurate, if accurate results are required, re-compute newVertex projection.
        HKGP_FORCE_INLINE void  notifyNewVertex(Vertex* edgeStart,Vertex* edgeEnd,hkReal fraction,hkReal fractionOther,Vertex* newVertex) const {}

        /// The edge insertion is crossing the vertex 'vertex'.
        HKGP_FORCE_INLINE void  notifyVertexCrossing(const Vertex* vertex) const {}

        /// Prevent from been derived from, and define a dummy implementation.
    private:
        EdgeCrossing() {}
    public:
        static HKGP_FORCE_INLINE const EdgeCrossing& dummy() { static EdgeCrossing _;return(_); }
    };

    /// FloodPolicy
    struct FloodPolicy
    {
        /// Retrieve sets value from edge data.
        HKGP_FORCE_INLINE tSets getSetsFromEdgeData(const EdgeData* edgeData) const;
    };

    /// Drawing interface
    struct IDrawing
    {
        IDrawing() {}
        virtual         ~IDrawing() {}
        virtual void    getTriangleString(const hkgpTriangulatorType& triangulator,const Triangle* triangle, hkStringBuf& stringOut) const {}
        virtual void    getVertexString(const hkgpTriangulatorType& triangulator,const Vertex* vertex, hkStringBuf& stringOut) const {}
        virtual void    getEdgeString(const hkgpTriangulatorType& triangulator,Edge edge, hkStringBuf& stringOut) const {}
        virtual void    getHalfEdgeString(const hkgpTriangulatorType& triangulator,Edge edge, hkStringBuf& stringOut) const {}
    };

    //
    // ctor & dtor
    //

    HKGP_FORCE_INLINE                           hkgpTriangulatorType();
    HKGP_FORCE_INLINE                           ~hkgpTriangulatorType();

    //
    // Methods
    //

    /// Reset triangulation
    HKGP_INLINE void                        reset(bool createDomainTriangulation=true);

    /// Clone triangulation
    inline hkResult                         clone(hkgpTriangulatorType* other) const;

    /// Set triangulation domain and reset, assuming XY plane.
    inline void                             setDomain(const hkVector4& min, const hkVector4& max, hkReal angle, bool margin);

    /// Set triangulation domain and reset.
    inline void                             setDomainFromPoints(const hkStridedVertices& points, const hkMatrix4& worldToXYPlane, bool margin, bool keepRatio=true);

    /// Set optimal domain given a points set, assuming XY plane.
    inline void                             setDomainFromPoints(const hkStridedVertices& points, bool margin, bool keepRatio=true);

    /// Set optimal domain given a set of planar points, assuming 'planeNormal' as best projection.
    /// If useMainAxis is true, only the major axis of planeNormal is used to improve accuracy during projection.
    inline void                             setDomainFromPlanarPoints(const hkStridedVertices& points, hkVector4Parameter planeNormal, bool margin, bool keepRatio=true, bool useMainAxis=false);

    /// Get effective accuracy of discretization (minimum representable deltas) along both axis.
    /// Notes: Accuracy is calculated using floating point arithmetic, thus is approximate.
    inline void                             getEffectiveAccuracy(hkVector4& acc) const;

    /// Transform from triangulation domain to world
    HKGP_FORCE_INLINE void                  transform(const Vertex* v, hkVector4& result) const;

    /// Transform the triangles vertices to the world domain
    HKGP_FORCE_INLINE void                  transform(const Triangle* t, hkVector4 vertices[3]) const;

    /// Transform from world to triangulation domain without quantization
    HKGP_FORCE_INLINE void                  transformUnquantized(hkVector4Parameter v, hkVector4& result) const;

    /// Quantize a point of the domain
    HKGP_FORCE_INLINE void                  quantize(hkVector4Parameter v, Vertex& result) const;

    /// Transform from world to triangulation domain
    HKGP_FORCE_INLINE void                  transform(hkVector4Parameter v, Vertex& result) const;

    /// Insert a vertex in the triangulation at a given location
    HKGP_INLINE Insertion                   insertVertex(int x, int y, bool conform, const Location& location, bool edgeSplit);

    /// Insert a vertex in the triangulation
    inline Insertion                        insertVertex(int x, int y, bool conform);

    /// Insert a vertex in the triangulation
    inline Insertion                        insertVertex(const Vertex& v, bool conform);

    /// Insert a vertex in the triangulation
    template <typename EDGECROSSING>
    inline Insertion                        insertVertex(int x, int y, bool conform, const EDGECROSSING& icrossing);

    /// Insert a vertex in the triangulation
    template <typename EDGECROSSING>
    inline Insertion                        insertVertex(const Vertex& v, bool conform, const EDGECROSSING& icrossing);

    /// Remove a vertex, failed if the vertex belong to a constrained edge
    inline Removal                          removeVertex(int x,int y);

    /// Remove a vertex, failed if the vertex belong to a constrained edge
    inline Removal                          removeVertex(const Vertex& v);

    /// Insert an edge in the triangulation
    inline Insertion                        insertEdge(int x0, int y0, int x1, int y1, bool conform, const EdgeData* data=HK_NULL);

    /// Insert an edge in the triangulation
    inline Insertion                        insertEdge(const Vertex& v0,const Vertex& v1, bool conform, const EdgeData* data=HK_NULL);

    /// Insert an edge in the triangulation
    inline Insertion                        insertEdge(int x0, int y0, int x1, int y1, bool conform, const EdgeData& data);

    /// Insert an edge in the triangulation
    inline Insertion                        insertEdge(const Vertex& v0,const Vertex& v1, bool conform, const EdgeData& data);

    /// Insert an edge in the triangulation
    template <typename EDGECROSSING>
    inline Insertion                        insertCrossingEdge(int x0, int y0, int x1, int y1, bool conform,const EDGECROSSING& icrossing, const EdgeData* data=HK_NULL);

    /// Insert an edge in the triangulation
    template <typename EDGECROSSING>
    inline Insertion                        insertCrossingEdge(const Vertex& v0,const Vertex& v1, bool conform,const EDGECROSSING& icrossing, const EdgeData* data=HK_NULL);

    /// Insert an edge in the triangulation
    template <typename EDGECROSSING>
    inline Insertion                        insertCrossingEdge(int x0, int y0, int x1, int y1, bool conform,const EDGECROSSING& icrossing, const EdgeData& data);

    /// Insert an edge in the triangulation
    template <typename EDGECROSSING>
    inline Insertion                        insertCrossingEdge(const Vertex& v0,const Vertex& v1, bool conform,const EDGECROSSING& icrossing, const EdgeData& data);

    /// Locate a position in the triangulation
    HKGP_INLINE Location                    locateVertex(int x,int y) const;

    /// Locate an edge in the triangulation
    inline Edge                             locateExistingEdge(int x0,int y0,int x1,int y1) const;

    /// Force all edges to be Delaunay
    inline void                             conformEdges();

    /// Minimize edge lengths by edge flipping
    inline void                             minimizeEdgeLengths();

    /// Check if triangulation is Delaunay
    inline hkBool32                         isConform() const;

    /// Remove vertices not referenced by constrained edges
    inline void                             removeOrphanVertices();

    //
    // Topology
    //

    /// Return the number of open boundary edges.
    inline int                              getOpenBoundaries(hkArray<Edge>* edges=HK_NULL) const;

    /// Return true if edge data are on the inside (wrt. regions) else false.
    inline hkBool32                         getEdgeDataOrientation() const;

    //
    // Regions methods
    //

    /// Partition into regions by increasing topological embedding.
    inline int                              partition();

    /// Remove constraints and datas on edges for which edge.triangle.m_region == edge.link.triangle.m_region hold.
    inline void                             removeNonBoundaryConstrainedEdgeAndData();

    /// Decompose triangulation into convex pieces, use regions as boundaries, and return the number of pieces
    inline int                              decompose(hkArray<hkArray<Edge,ALLOCATOR>,ALLOCATOR>& pieces);

    /// Faster than decompose, but not 'stable', local triangulation changes might generate completely different decompositions.
    inline int                              decomposeFast(hkArray<hkArray<Edge,ALLOCATOR>,ALLOCATOR>& pieces);

    /// Same as decomposeFast, but uses a fixed block of data for the arrays
    inline int                              decomposeFastFixedSize(hkArrayBase<hkUint32>& pieceSizes, void* storage, int capacity, int& numEdgesOut, int& maxEdgesPerPieceOut);

    //
    // Triangle sets methods
    //

    /// Clear all triangles sets
    inline void                             clearTrianglesSets(tSets value=0);

    /// Flood edge set id to triangles. floodEdgeSets() returns false if out of memory, otherwise true
    template <typename FLOODPOLICY>
    inline hkBool32                         floodEdgeSets(const FLOODPOLICY& policy);


    /// Flood across edges using edge data (given by policy). partitionWithPolicy() returns HK_FAILURE if out of memory, otherwise HK_SUCCESS
    template <typename FLOODPOLICY>
    inline hkResult                         partitionWithPolicy(const FLOODPOLICY& policy);

    //
    // Custom data methods
    //

    /// Get edge data pointer if any, else return HK_NULL.
    const EdgeData*                         getEdgeData(const Edge& edge) const;

    /// Get any edge data pointer, first on the edge side, then on the link side, else return HK_NULL.
    const EdgeData*                         getAnyEdgeData(const Edge& edge) const;

    /// Set edge data.
    void                                    setEdgeData(const Edge& edge,const EdgeData& data);

    /// Set edge data on both side.
    void                                    setDualEdgeData(const Edge& edge,const EdgeData& data);

    /// Clear edge data.
    void                                    clearEdgeData(const Edge& edge);

    //
    // Internals
    //

    /// Set a root using triangle 't'
    HKGP_FORCE_INLINE void                  setRoot(Triangle* t);

    /// Create a new vertex
    HKGP_FORCE_INLINE Vertex*               newVertex(hkInt32 x,hkInt32 y,hkBool32 constraint);

    /// Release a vertex
    HKGP_FORCE_INLINE void                  releaseVertex(Vertex* v);

    /// Create a new triangle without updating roots
    HKGP_FORCE_INLINE Triangle*             newTriangleNoRoot(Vertex* a,Vertex* b,Vertex* c);

    /// Create a new triangle
    HKGP_FORCE_INLINE Triangle*             newTriangle(Vertex* a,Vertex* b,Vertex* c);

    /// Release a triangle
    HKGP_FORCE_INLINE void                  releaseTriangle(Triangle* t);

    /// Conform edge
    HKGP_FORCE_INLINE Edge                  conformEdge(const Edge& edge);

    /// Get root triangle from position
    HKGP_FORCE_INLINE Triangle*             getRoot(int x,int y) const;

    /// Locate a position in the triangulation from a given root
    HKGP_INLINE Location                    locateVertex(const Edge& root,int x,int y) const;

    /// Locate an existing vertex in the triangulation from a given root
    inline Edge                             locateExistingVertex(const Edge& root,const Vertex* v) const;

    /// Locate an existing edge in the triangulation from a given root
    inline Edge                             locateExistingEdge(const Edge& root,const Vertex* v0,const Vertex* v1) const;

    /// Collapse edge ( start() to end() )
    HKGP_FORCE_INLINE Edge                  collapse(const Edge& edge);

    /// Clear vertices marker flag.
    inline void                             clearVerticesMarker();

    /// Clear triangles marker flag.
    inline void                             clearTrianglesMarker();

    /// Fetch fan boundary
    template <typename T>
    static inline hkBool32                  fetchFanBoundary(Edge edge,T& boundaries);

    /// Triangulate boundary
    inline void                             triangulateBoundary(Boundary* boundary);

    /// Flush the conforming stack
    HKGP_INLINE void                        flushConform();

    /// Calculate orientation given 3 points
    static HKGP_FORCE_INLINE tExtInt        orientation(hkInt32 ax,hkInt32 ay,hkInt32 bx,hkInt32 by,hkInt32 cx,hkInt32 cy);

    /// Calculate orientation given 1 vertex and 2 point
    static HKGP_FORCE_INLINE tExtInt        orientation(const Vertex* a,hkInt32 bx,hkInt32 by,hkInt32 cx,hkInt32 cy);

    /// Calculate orientation given 2 vertices and 1 point
    static HKGP_FORCE_INLINE tExtInt        orientation(const Vertex* a,const Vertex* b,hkInt32 x,hkInt32 y);

    /// Calculate orientation given 3 vertices
    static HKGP_FORCE_INLINE tExtInt        orientation(const Vertex* a,const Vertex* b,const Vertex* c);

    /// Calculate orientation given 1 edge and 1 point
    static HKGP_FORCE_INLINE tExtInt        orientation(const Edge& e,hkInt32 x,hkInt32 y);

    /// Calculate orientation given 1 edge and 1 vertex
    static HKGP_FORCE_INLINE tExtInt        orientation(const Edge& e,const Vertex* c);

    /// Check orientation validity of an edge
    static HKGP_FORCE_INLINE hkBool32       checkOrientation(hkInt32 ax,hkInt32 ay,hkInt32 bx,hkInt32 by);

    /// Check if point 'd' in the circum-circle of {a,b,c}, exact computation.
    static HKGP_FORCE_INLINE hkBool32       notInCircle(hkInt32 ax,hkInt32 ay,hkInt32 bx,hkInt32 by,hkInt32 cx,hkInt32 cy,hkInt32 dx,hkInt32 dy);

    /// Check if point 'd' in the circum-circle of {a,b,c}, approximate computation.
    static HKGP_FORCE_INLINE hkBool32       notInCircleFloat(hkInt32 axi,hkInt32 ayi,hkInt32 bxi,hkInt32 byi,hkInt32 cxi,hkInt32 cyi,hkInt32 dxi,hkInt32 dyi);

    /// Clamp an integer such as min <= x <= max
    static HKGP_FORCE_INLINE hkInt32        clamp(hkInt32 x,hkInt32 min,hkInt32 max);

    /// Round a real to the nearest integer
    static HKGP_FORCE_INLINE hkInt32        round(hkReal x);

    /// Find the root index given a point
    static HKGP_FORCE_INLINE int            rootIndex(int x,int y);

    /// Check if two vertex share the same boundary
    static HKGP_FORCE_INLINE hkBool32       sameBoundary(const Vertex* v0,const Vertex* v1);

    /// Check if 'p' is Delaunay relative to {a,b,c}
    static HKGP_FORCE_INLINE hkBool32       isConform(const Vertex* a,const Vertex* b,const Vertex* c,const Vertex* p);

    /// Check if edge.link().apex() if Delaunay relative to {edge.start(),edge.end(),edge.apex()}
    static HKGP_FORCE_INLINE hkBool32       isConform(const Edge& edge);

    /// Calculate the squared of |a-b|
    static HKGP_FORCE_INLINE tExtInt        distanceSqrd(const Vertex* a,const Vertex* b);

    /// Check if an edge can by split at ix,iy
    static inline hkBool32                  canSplitEdge(int ix,int iy,const Edge& edge);

    /// Flip an edge.
    HKGP_FORCE_INLINE Edge                  flipEdge(const Edge& edge);

    /// Copy data set
    static HKGP_FORCE_INLINE void           copyDataset(const Triangle* from,Triangle* to);

    //
    // Helpers
    //

    /// Check triangulation
    inline bool                             check() const;

    /// Draw triangulation
    inline void                             draw(const IDrawing* idrawing=HK_NULL) const;

    /// Read from file
    inline bool                             read(const char* filename);

    /// Write to file
    inline bool                             write(const char* filename);

    /// Execute self benchmarking code
    static inline hkgpTriangulatorType*     benchmark();

    //
    // Fields
    //
    Mesh                                                        m_mesh;
    hkInplaceArray<Edge,cfgStacksize,ALLOCATOR>                 m_stack;
    EdgeDataPolicy                                              m_edgeDataPolicy;
    Triangle*                                                   m_roots[cfgTotalRoots];
    int                                                         m_margin;
    hkEnum<Status, hkUint8>                                     m_status;
    hkMatrix4                                                   m_localToWorld;
    hkMatrix4                                                   m_worldToLocal;
};

#define TRIANGULATOR_HDR    template <typename ALLOCATOR,typename VERTEX,typename TRIANGLE,typename EDGEDATA,typename EDGEDATA_POLICY,const int ORIENTATION,const int ROOTS,const int BITS,const bool USE_FAST_CONFORM>
#define TRIANGULATOR_TYP    hkgpTriangulatorType<ALLOCATOR,VERTEX,TRIANGLE,EDGEDATA,EDGEDATA_POLICY,ORIENTATION,ROOTS,BITS,USE_FAST_CONFORM>

#include <Geometry/Collide/GeometryProcessing/Triangulator/hkgpTriangulator.inl>
#include <Geometry/Collide/GeometryProcessing/Triangulator/hkgpTriangulatorHelpers.inl>

#undef  TRIANGULATOR_HDR
#undef  TRIANGULATOR_TYP

/// Default triangulator, see hkgpTriangulatorType declaration for default specialization.
typedef hkgpTriangulatorType<>  hkgpTriangulator;

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
