// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

    /// Static bounding volume tree.
    /// A compact representation of a dynamic tree optimized for memory and query speed (in that order).
namespace hkcdStaticTree
{
    ///
    /// Filter type.
    ///
    typedef hkArray<hkUint32>   Filter;

    ///
    /// Tree.
    ///
    template <typename STORAGE>
    class Tree : public STORAGE
    {
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE,Tree);
        HK_DECLARE_REFLECTION();
        /// Config.
        enum
        {
            TRAMPOLINE_TYPE = 0,    ///< Default trampoline type: terminal.
            INVALID_INDEX = -1      ///< Invalid node index.
        };

        /// Node type.
        typedef typename STORAGE::Node  Node;

        /// Node index.
        typedef int                     Index;

        /// Constants.
        enum { MAX_LEAVES = 1<<Node::DATA_BITS, DATA_RANGE = MAX_LEAVES };

        /// Contructor
        HK_INLINE Tree() { m_domain.setEmpty(); }
        HK_INLINE Tree(const Tree& other)
            : STORAGE(other), m_domain(other.m_domain)
        {}
        HK_INLINE ~Tree() {}

        //
        // Building.
        //

        /// Convert from a dynamic tree.
        template <typename DTREE>
        inline void                 convertFromDynamicTree(const DTREE& dTree);

        /// Convert to dynamic tree.
        template <typename DTREE>
        inline void                 convertToDynamicTree(DTREE& dTree) const;

        //
        // Serialization
        //

        #include <Geometry/Internal/DataStructures/StaticTree/hkcdStaticTreeSerialization.inl>

        //
        // Methods.
        //

        /// Return true is the tree is empty.
        HK_INLINE bool      isEmpty() const { return STORAGE::getNumNodes() == 0; }

        /// Return the root node index.
        HK_INLINE Index     getRoot() const { return 0; }

        /// Clear the tree.
        HK_INLINE void      clear() { STORAGE::clear(); m_domain.setEmpty(); }

        /// Return the memory foot-print of the tree in bytes.
        HK_INLINE int           getMemoryFootPrint() const  { return STORAGE::getMemoryFootPrint(sizeof(*this)); }

        /// Returns the AABB of the tree domain.
        HK_INLINE void      getDomain(hkAabb& aabbOut) const { aabbOut = m_domain; }

        /// Find an leaf node by its value, O(2n).
        inline Index                findLeafByValue(int value) const;

        /// Get the AABB of a node, O(log n).
        inline void                 getNodeAabb(Index node, hkAabb& aabb) const;

        /// Creates a shallow copy of the tree.
        inline void                 shallowCopy(const Tree<STORAGE>& src);

        /// Save to a structured binary stream.
        template <typename WRITER>
        inline void                 save(WRITER& writer) const;

        /// Load from a structured binary stream.
        template <typename READER>
        inline void                 load(READER& reader);

        /// Collector for leaf indexes.
        struct HK_EXPORT_COMMON LeafIndexCollector
        {
            HK_INLINE           LeafIndexCollector(hkArray<Index>& indicesArray) : m_indicesArray(indicesArray) {}
            HK_INLINE void  append(const hkcdStaticTree::Tree<STORAGE>* tree, Index leaf) { m_indicesArray.pushBack(leaf); }
            hkArray<Index>& m_indicesArray;
        };

        /// Collector for leaf data.
        template <typename DATA_TYPE>
        struct LeafDataCollector
        {
            HK_INLINE           LeafDataCollector(hkArray<DATA_TYPE>& dataArray) : m_dataArray(dataArray) {}
            HK_INLINE void  append(const hkcdStaticTree::Tree<STORAGE>* tree, Index leaf) { m_dataArray.pushBack( (DATA_TYPE) tree->getNode(leaf)->getData()); }
            hkArray<DATA_TYPE>& m_dataArray;
        };

        /// Collect all the leaves starting at this node using collector.
        /// This method DO NOT tunnel in sub-trees if any.
        template <typename COLLECTOR>
        HK_INLINE void      collectLeavesWithCollector(Index node, COLLECTOR& collector) const;

        /// Collect all the leaves data starting at this node.
        /// This method DO NOT tunnel in sub-trees if any.
        template <typename DATA_TYPE>
        HK_INLINE void      collectLeavesData(Index node, hkArray<DATA_TYPE>& data) const;

        /// Collect all the leaves index starting at this node.
        /// This method DO NOT tunnel in sub-trees if any.
        HK_INLINE void      collectLeaves(Index node, hkArray<Index>& indices) const;

        /// Count the number of leaves using traversal.
        HK_INLINE int           countLeaves(Index node) const;

        /// Get the level(X) ancestor of a node. (1->parent, 2->grand parent, and so on).
        HK_INLINE Index     getAnscenstor(Index node, int level) const;

        /// Returns the parent of the node or -1 if node is the root.
        HK_INLINE Index     getParent(Index node) const;

        /// Get the level of a node.
        HK_INLINE Index     getLevel(Index node) const;

        //
        // Query support.
        //

        /// Capabilities
        enum { SUPPORT_STACKLESS_TRAVERSAL = 0, USE_SOFTWARE_CACHE = 1 };

        /// Stack slot
        struct HK_EXPORT_COMMON Slot
        {
            HK_DECLARE_POD_TYPE() { return hkBool::CompileTimeTrueType(); }
            HK_INLINE Slot(){}
            HK_INLINE const Slot& operator=(const Slot& s) { m_aabb = s.m_aabb; m_index = s.m_index; return *this;}

            hkAabb          m_aabb;     ///< AABB.
            int m_index;    ///< Index.
        };

        /// Type given to queries processLeaf / processLeaves method.
        /// Gives access to all information regarding a node being processed:
        struct HK_EXPORT_COMMON NodeContext : public Slot
        {
            HK_DECLARE_POD_TYPE() { return hkBool::CompileTimeTrueType(); }
            HK_INLINE NodeContext(){}
            HK_INLINE const NodeContext& operator=(const NodeContext& s) {Slot::m_aabb = s.m_aabb; Slot::m_index = s.m_index; m_node = s.m_node; return *this;}

            HK_INLINE const Node*   operator->() const { return m_node; }
            const Node* m_node;     ///< The node pointer.
        };

        /// Initialize root context.
        HK_INLINE void      initializeRootContext(NodeContext& context, bool useCache) const;

        /// Fetch the children of a given context.
        HK_INLINE void      fetchChildren(const NodeContext& context, NodeContext* children, bool useCache) const;

        /// Set context from slot.
        HK_INLINE void      setContextFromSlot(const Slot& slot, NodeContext& context, bool useCache) const;

        /// Set context from a node index.
        HK_INLINE void      setContextFromNode(Index node, NodeContext& contextOut, bool useCache) const;

        /// Set slot from context.
        HK_INLINE void      setSlotFromContext(const NodeContext& context, Slot& slot) const;

        /// Process leaf node.
        template <typename QUERY>
        HK_INLINE void      processLeaf(QUERY& q, const NodeContext& leaf) const;

        hkAabb      m_domain;       ///< Root AABB.
    };

    ///
    /// Filtering.
    ///
    template <int BITS_PER_NODE>
    struct HK_EXPORT_COMMON Filtering
    {
        /// Constant.
        enum
        {
            BITS_PER_CLUSTER    =   sizeof(hkUint32) * 8,
            NODES_PER_CLUSTER   =   BITS_PER_CLUSTER / BITS_PER_NODE,
            MASK                =   (1 << BITS_PER_NODE) - 1,
        };

        /// Get the filter value for a given node index.
        static HK_INLINE int            get(const hkUint32* clusters, int node);

        /// Set the filter value for a given node index.
        static HK_INLINE void           set(hkUint32* clusters, int node, int value);

        /// Compute the number of clusters required for a given number of nodes.
        static HK_INLINE int            computeSizeInClusters(int numNodes) { return (numNodes + NODES_PER_CLUSTER - 1) / NODES_PER_CLUSTER; }

        /// Compute a bitfield representing valid tree branches.
        template <typename TREE>
        static HK_NEVER_INLINE bool         computeFilter(const TREE& tree, hkUint32* clusters);

        /// Compute a bitfield representing valid tree branches.
        template <typename TREE>
        static HK_NEVER_INLINE bool         computeFilter(const TREE& tree, const typename TREE::NodeContext& node, hkUint32* clusters);
    };

}

#include <Geometry/Internal/DataStructures/StaticTree/hkcdStaticTreeCodecs.h>
#include <Geometry/Internal/DataStructures/StaticTree/hkcdStaticTreeStorages.h>


// Tree specializations
//
namespace hkcdStaticTree
{
    /// 4 bytes per node, up to 2^7 leaves and 7 bits data range.
    typedef Tree< DynamicStorage4 > DefaultTree4;

    /// 5 bytes per node, up to 2^15 leaves and 15 bits data range.
    typedef Tree< DynamicStorage5 > DefaultTree5;

    /// 6 bytes per node, up to 2^23 leaves and 23 bits data range.
    typedef Tree< DynamicStorage6 > DefaultTree6;

    /// 32 bytes per node, full precision, up to 2^23 leaves and 23 bits data range.
    typedef Tree< DynamicStorage32 > DefaultTree32;

#ifdef HK_DYNAMIC_DLL  // then as we are exporting the templates, we want to export the specializations
    HK_DETAIL_DIAG_PUSH()
    HK_DETAIL_DIAG_CLANG_OFF(ignored-attributes)
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON Tree< DynamicStorage4 >;
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON Tree< DynamicStorage5 >;
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON Tree< DynamicStorage6 >;
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON Tree< DynamicStorage32 >;

    // Fixes error on PlayStation(R)4 [see RSYS-3807]: explicit instantiation of non-templated type 'hkcdStaticTree::Tree<DynamicStorage4>::Slot'
#   ifndef HK_PLATFORM_PS4
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON Tree< DynamicStorage4 >::Slot;
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON Tree< DynamicStorage5 >::Slot;
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON Tree< DynamicStorage6 >::Slot;
    HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON Tree< DynamicStorage32 >::Slot;
#   endif

    HK_DETAIL_DIAG_POP()
#endif
}

HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DefaultTree4, hkcdStaticTree_DefaultTree4);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DefaultTree5, hkcdStaticTree_DefaultTree5);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DefaultTree6, hkcdStaticTree_DefaultTree6);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DefaultTree32, hkcdStaticTree_DefaultTree32);

/// Process leaf node.
template <typename STORAGE>
template <typename QUERY>
HK_INLINE void      hkcdStaticTree::Tree<STORAGE>::processLeaf(QUERY& q, const NodeContext& leaf) const
{
    q.processLeaf(leaf);
}

#include <Geometry/Collide/_Auto/TemplateTypes/hkcdStaticTree_Types.inl>
#include <Geometry/Collide/DataStructures/StaticTree/hkcdStaticTree.inl>

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
