// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Geometry/Collide/DataStructures/StaticTree/hkcdStaticTree.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Container/BitField/hkBitField.h>
#include <Common/Base/Container/Array/hkFixedInplaceArray.h>

#include <Common/Internal/GeometryProcessing/hkGeometryProcessing.h>
#include <Common/Internal/GeometryProcessing/ConvexHull/hkgpConvexHull.h>

#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>

#include <Geometry/Collide/DataStructures/DynamicTree/hkcdDynamicTree.h>
#include <Geometry/Collide/Algorithms/Triangle/hkcdTriangleUtil.h>

#include <Geometry/Internal/Algorithms/TreeQueries/hkcdTreeQueries.h>
#include <Geometry/Internal/Algorithms/RayCast/hkcdRayCastTriangle.h>
#include <Geometry/Collide/Algorithms/Distance/hkcdDistancePointTriangle.h>

#include <Common/Base/System/Log/hkLog.h>
extern HK_EXPORT_COMMON hkLog::RegisteredOrigin s_hkcdStaticMeshLogOrigin;

/// Invalid primitive key ( equal to HK_INVALID_SHAPE_KEY on purpose ).
#define HKCD_INVALID_KEY    hkUint32(hkcdStaticMeshTreeBase::INVALID_PRIMITIVE_KEY)

struct hkcdSimdTree;

    /// Static mesh tree base.
    /// Non-template common definitions.
struct HK_EXPORT_COMMON hkcdStaticMeshTreeBase : public hkcdStaticTree::DefaultTree5
{
    HK_DECLARE_CLASS(hkcdStaticMeshTreeBase, New, Reflect, BypassCtor);
    HK_RECORD_ATTR(hk::Version(3));

    /// Type of the parent tree.
    typedef hkcdStaticTree::DefaultTree5 ParentTree;

    /// Type of the section tree.
    typedef hkcdStaticTree::DefaultTree4 SectionTree;

    /// Cache policy.
    enum { USE_SOFTWARE_CACHE = 1 };

    /// Compile time configuration.
    enum
    {
        /// Attribute used by queries to allow recursive tree traversals.
        HAS_GET_LEAF_TREE,

        /// Invalid primitive key ( equal to HK_INVALID_SHAPE_KEY on purpose ).
        /// Note: the macro HKCD_INVALID_KEY is a short-hand for hkUint32(hkcdStaticMeshTreeBase::INVALID_PRIMITIVE_KEY)
        INVALID_PRIMITIVE_KEY       =   0xffffffff,

        /// Shared vertices page size.
        SHARED_VERTICES_PAGE_SIZE   =   65536
    };

    /// Custom primitive vertex compression mode.
    enum CompressionMode
    {
        /// Global quantization, 8 bytes per vertex, very high accuracy.
        CM_GLOBAL,

        /// Local quantization (Default), 4 bytes per vertex, medium to high accuracy.
        CM_LOCAL_4,

        /// Local quantization (Default), 2 bytes per vertex, low to medium accuracy.
        CM_LOCAL_2,

        /// Only valid at build time, try CM_LOCAL_2, CM_LOCAL_4, CM_GLOBAL in this order until the quantization error is less than m_maxError.
        CM_AUTO
    };

    /// Codec parameters.
    struct HK_EXPORT_COMMON CodecParameters
    {
        HK_INLINE CodecParameters() { m_origin.setZero(); m_scale.setZero(); }
        HK_INLINE void load(const hkReal data[6]) { m_origin.load<4,HK_IO_NATIVE_ALIGNED>(data); m_scale.load<3,HK_IO_NATIVE_ALIGNED>(data + 3); m_scale.zeroComponent<3>(); }
        HK_INLINE void store(hkReal data[6]) const { m_origin.store<4,HK_IO_NATIVE_ALIGNED>(data); m_scale.store<3,HK_IO_NATIVE_ALIGNED>(data + 3); }

        hkVector4   m_origin;   ///< Quantization domain origin.
        hkVector4   m_scale;    ///< Quantization domain scale.
    };

    /// PrimitiveDataRunBase.
    /// Note: This also provide a 'template' of how to implement a custom primitive data run / triangle data.
    template <typename TRIANGLE_DATA>
    struct PrimitiveDataRunBase
    {
        typedef TRIANGLE_DATA   TriangleData;

        HK_DECLARE_CLASS(PrimitiveDataRunBase, New, Pod, Reflect);

        HK_INLINE const TriangleData&   getValue() const { return m_value; }
        HK_INLINE void                  setValue(const TriangleData& value) { m_value = value; }

        HK_INLINE   int                 getIndex() const { return m_index; }
        HK_INLINE   void                setIndex(int index) { m_index = (hkUint8) index; HK_ASSERT(0x5D3B88CE, getIndex() == index, "Index out-of-range"); }

        HK_INLINE   int                 getCount() const { return m_count; }
        HK_INLINE   void                setCount(int count) { m_count = (hkUint8) count; HK_ASSERT(0x84FECDC3, getCount() == count, "Count out-of-range"); }

        private:

        TriangleData    m_value;    ///< Triangle data value.
        hkUint8         m_index;    ///< Primitive index.
        hkUint8         m_count;    ///< Primitive count.
    };

    /// Section.
    struct HK_EXPORT_COMMON Section : public SectionTree
    {
            HK_RECORD_ATTR(hk::Version(4));

        HK_DECLARE_REFLECTION();
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE, Section);

        #define HK_DECL_24_8_STRUCT( _type_ , _name_ , _field24_ , _field8_ ) \
        struct _name_ \
        { \
            HK_DECLARE_CLASS(_name_, New, Pod, Reflect); \
            _type_  get##_field24_() const          { return m_data >> 8; } \
            void    set##_field24_(_type_ value)    { m_data = (value << 8) | (m_data & 0xff); } \
            _type_  get##_field8_() const           { return m_data & 0xff; } \
            void    set##_field8_(_type_ value)     { m_data = (m_data & 0xffffff00) | value; } \
            private: \
            _type_ m_data; \
        };

        HK_INLINE   Section() { }
        HK_INLINE Section(const Section& other)
            : SectionTree(other), m_firstPackedVertex(other.m_firstPackedVertex), m_sharedVertices(other.m_sharedVertices),
            m_primitives(other.m_primitives), m_dataRuns(other.m_dataRuns), m_numPackedVertices(other.m_numPackedVertices),
            m_numSharedIndices(other.m_numSharedIndices), m_leafIndex(other.m_leafIndex), m_flags(other.m_flags)
        {
            hkString::memCpy(&m_codecParms[0], &other.m_codecParms[0], sizeof(m_codecParms));
            hkString::memCpy(&m_unusedData, &other.m_unusedData, sizeof(m_unusedData));
        }

        /// SharedVertices.
        HK_DECL_24_8_STRUCT(hkUint32, SharedVertices, FirstShared, FirstLocal);

        /// Primitives.
        HK_DECL_24_8_STRUCT(hkUint32, Primitives, First, Count);

        /// DataRuns.
        HK_DECL_24_8_STRUCT(hkUint32, DataRuns, First, Count);

        #undef HK_DECL_24_8_STRUCT

        /// Cache policy.
        enum { USE_SOFTWARE_CACHE = 0 };

        /// Flags.
        enum Flags
        {
            SF_REQUIRE_TREE =   0x01,   ///< The section contains data that require the section tree to be available.
        };

        hkReal      m_codecParms[6];            ///< Packed codec parameters (see CodecParameters).
        hkUint32    m_firstPackedVertex;        ///< First packed vertex in the packed vertices array.
        SharedVertices  m_sharedVertices;       ///< Shared vertices infos.
        Primitives      m_primitives;           ///< Primitives infos.
        DataRuns        m_dataRuns;             ///< Data runs infos.
        hkUint8     m_numPackedVertices;        ///< Number of packed vertices.
        hkUint8     m_numSharedIndices;         ///< Number of shared indices.
        hkUint16    m_leafIndex;                ///< Index of this section in the master tree.
        hkUint8         m_page;                 ///< Shared vertices page.
        hkUint8         m_flags;                ///< Flags.
        hkUint8         m_layerData;            ///< Layer data.
        hkUint8         m_unusedData;           ///< Unused padding, reserved for future extensions (set to zero).
    };

    /// Primitive.
    struct HK_EXPORT_COMMON Primitive
    {
        HK_DECLARE_CLASS(Primitive, New, Pod, Reflect);

        enum Type
        {
            INVALID,
            TRIANGLE,   // Triangle shape
            QUAD,       // Planar quad
            CUSTOM,     // Vertices, which should be decoded externally
            EXTERNAL,   // Not stored in the mesh directly - handle to an external structure
            NUM_TYPES
        };

        HK_INLINE   Type    getType() const { return getType(m_indices[0], m_indices[1], m_indices[2], m_indices[3]); }

        HK_INLINE int       getNumTriangles() const
        {
            if(isTriangleOrQuad()) return 1 + (m_indices[2] != m_indices[3]);
            return 0;
        }

        HK_INLINE hkBool32  isTriangleOrQuad() const { return m_indices[1] ^ m_indices[3]; }
        HK_INLINE bool      isFlatConvexQuad( hkUint8 isFeatureEnabled = 0xff ) const
        {
            HK_ASSERT_NO_MSG( 0xf04ffdde, isFeatureEnabled == 0 || isFeatureEnabled == 0xff );
            return ( isFeatureEnabled & m_indices[1] ) > m_indices[3];
        }

        static HK_INLINE Type   getType(int a, int b, int c, int d);

        hkUint8 m_indices[4];   ///< Primitive vertices index, triangles: [0,1,2] and [0,2,3].
    };

    /// Edge.
    struct HK_EXPORT_COMMON Edge
    {
        HK_DECLARE_CLASS(Edge, New, Reflect);

        HK_INLINE           Edge() : m_keyAndIndex(HKCD_INVALID_KEY) {}
        HK_INLINE           Edge(hkUint32 ikey, int idx) : m_keyAndIndex( (ikey << 2) | hkUint32(idx) ) {}
        HK_INLINE bool      isValid() const { return m_keyAndIndex != HKCD_INVALID_KEY; }
        HK_INLINE hkUint32  key() const { return isValid()? m_keyAndIndex >> 2 : HKCD_INVALID_KEY; }
        HK_INLINE int       index() const { return isValid()? int( m_keyAndIndex & 3 ) : 3 ; }
        HK_INLINE bool      operator==(const Edge& other) const { if(isValid() == other.isValid()) return !isValid() || (m_keyAndIndex == other.m_keyAndIndex); else return false; }
        HK_INLINE bool      operator!=(const Edge& other) const { return !operator==(other); }

        hkUint32    m_keyAndIndex;  ///< Primitive key and edge index.
    };

    /// Links, used to store connectivity.
    struct HK_EXPORT_COMMON Links
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE, Links);
        HK_DECLARE_REFLECTION();

        Edge        m_links[3]; ///< Neighbor primitives.
    };

    /// Compressed connectivity.
    /// Stores triangles connectivity using 4 bytes per triangle on average.
    struct HK_EXPORT_COMMON Connectivity
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE, Connectivity);
        HK_DECLARE_REFLECTION();

        /// Section header.
        struct SectionHeader
        {
            HK_DECLARE_CLASS(SectionHeader, NewPlacement, Reflect, Pod);

            hkUint32    m_baseLocal;    ///< First local.
            hkUint32    m_baseGlobal;   ///< First global. Note: the first bit is used to differentiate between compressed(0) and raw(1) sections.
        };

        /// Constructor.
        HK_INLINE       Connectivity() {}

        /// Reset connectivity.
        void            reset();

        /// Return true if the connectivity is empty.
        HK_INLINE bool  isEmpty() const { return m_headers.getSize() == 0; }

        /// Compute the memory foot-print in bytes.
        int             getMemoryFootPrint() const;

        /// Get next edge.
        HK_INLINE Edge  next(const Edge& e) const { return e.isValid() ? Edge(e.key(), (e.index() + 1) % 3) : Edge(); }

        /// Get previous edge.
        HK_INLINE Edge  prev(const Edge& e) const { return e.isValid() ? Edge(e.key(), (e.index() + 2) % 3) : Edge(); }

        /// Get linked edge.
        Edge            link(const Edge& e) const;

        /// Get master edge.
        Edge            master(const Edge& e) const { const Edge l = link(e); return (l.isValid() && l.key() < e.key()) ? l : e; }

        /// Returns true if a given edge is a master edge.
        bool            isMaster(const Edge& e) const { return (e.isValid() && e == master(e)) ? true : false; }

        /// Return true if a given edge is bound.
        bool            isBound(const Edge& e) const { return link(e).isValid(); }

        /// Build connectivity from a given mesh.
        template <typename MESH>
        inline void     build(const MESH& mesh);

        /// Get primitive type.
        HK_INLINE Primitive::Type getPrimitiveType(hkUint32 key) const;

        hkArray<SectionHeader>  m_headers;      ///< Section header.
        hkArray<hkUint8>        m_localLinks;   ///< Per-primitive blocks of 5 bytes (4 indices to neighboring primitives + 1 edge data).
        hkArray<hkUint32>       m_globalLinks;  ///< Global key used if indices & 7 != 0.
    };

    /// Expanded primitive key.
    struct HK_EXPORT_COMMON ExpandedPrimitiveKey
    {
        HK_INLINE           ExpandedPrimitiveKey() : m_section(-1) {}
        HK_INLINE           ExpandedPrimitiveKey(hkUint32 primitiveKey) { setPrimitiveKey(primitiveKey); }
        HK_INLINE void      setPrimitiveKey(hkUint32 primitiveKey);
        HK_INLINE hkUint32  getPrimitiveKey() const;
        HK_INLINE bool      isValid() const { return m_section != -1; }

        int m_section;          ///< Section index.
        int m_primitive;        ///< Primitive index in the section.
        int m_triangle;         ///< Triangle index in the primitive.
    };

    /// Filter
    typedef hkcdStaticTree::Filter  Filter;

    /// Memory statistics.
    struct HK_EXPORT_COMMON MemoryStatistics
    {
        int m_collisionSize;    ///< Memory used to store collision data.
        int m_geometrySize;     ///< Memory used to store geometry data.
        int m_dataSize;         ///< Memory used to store triangle data.
        int m_totalSize;        ///< Total memory foot-print of the static mesh tree.
    };

    /// Build mapping.
    struct HK_EXPORT_COMMON BuildMapping
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE, BuildMapping);

        /// Map from triangle index to key.
        hkArray<hkUint32>   m_triangleIndexToKey;

        /// Map from custom primitive index to key.
        hkArray<hkUint32>   m_customIndexToKey;

        /// Map from triangle index to vertex rotation (can be 0, 1 or 2).
        /// So given a compressed triangle with vertices V and rotation R, it maps to vertices { V[R], V[(R+1)%3], V[(R+2)%3] } in the original triangle.
        hkArray<hkUint8>    m_triangleVertexRotations;
    };

    /// Build tag list.
    typedef hkFixedInplaceArray<hkUint16,3> BuildTagList;

    /// Build custom primitive infos.
    struct HK_EXPORT_COMMON BuildCustomInfos
    {
        HK_INLINE BuildCustomInfos() : m_numVertices(0), m_customType(0), m_layerData(0), m_maxError(HK_REAL_MAX), m_compression(CM_LOCAL_4) { m_aabb.setEmpty(); }

        hkAabb          m_aabb;         ///< Primitive AABB, is not set (empty), this is automatically computed from vertices.
        int             m_numVertices;  ///< Number of vertices (0-255).
        int             m_customType;   ///< Type of the custom primitive for client code usage (0-15).
        hkUint8         m_layerData;    ///< Layer data.
        BuildTagList    m_tags;         ///< Tags.
        hkReal          m_maxError;     ///< Maximum quantization error (only used if m_compression is AUTO).
        CompressionMode m_compression;  ///< Mode of compression.
    };

    /// Build config.
    struct HK_EXPORT_COMMON BuildConfig
    {
        /// Vertex representation policy.
        /// Note that these policies only affects mesh triangles not custom primitives.
        enum Vrp
        {
            /// Vertices are compressed using an adaptive scheme.
            /// This is the default method as it provides the best overall compression.
            VRP_EMBED_ADAPTIVE,

            /// Vertices are compressed using an global scheme.
            /// This provide slightly less efficient compression ratio than VRP_EMBED_ADAPTIVE but is faster to decompress during queries.
            VRP_EMBED_GLOBAL,

            /// Vertices are not embedded but simply referenced by their indices.
            VRP_EXTERN
        };

        /// Post-build degeneracy policy.
        enum Pbd
        {
            PBD_IGNORE,     ///< Keep post-build degenerated triangles.
            PBD_MARK,       ///< Mark post-build degenerated triangles as invalid (see: Primitive::getType and Decoder::isPrimitiveValid).
            PBD_REMOVE,     ///< Remove post-build degenerated triangles. Note: This can be a slow process.
        };

        /// Build verbosity flags.
        enum Verbosity
        {
            VRB_NONE        =   0,      ///< No output.
            VRB_MESSAGES    =   1,      ///< Output messages during build.
            VRB_STATS       =   2,      ///< Output statistics after build.
            VRB_ALL         =   VRB_MESSAGES | VRB_STATS
        };

        /// Vertex flags.
        enum VertexFlags
    {
            VTF_FORCE_SHARED    =   1   ///< Force vertex to be shared (that is, encoded relative to the domain AABB, not sections AABBs).
        };

        BuildConfig()
            : m_globalOptimizations(false)      // Just local optimization by default.
            , m_compactData(true)           // Compact data by default.
            , m_forceConvexQuads(false)         // Allow any orientation by default.
            , m_preserveVertexOrder(false)      // Allow changes in vertex ordering.
            , m_vertexHasFlags(false)           // Do not use vertices W component as flags by default.
            , m_balanceThreshold(0)         // Do not balance the tree by default.
            , m_vrp(VRP_EMBED_ADAPTIVE)         // Best compression by default.
            , m_pbd(PBD_IGNORE)                 // Keep post-build degeneracy by default.
            , m_verbosity(VRB_NONE)             // No verbosity by default.
            , m_weldVertices(true)          // Weld vertices by default.
            , m_weldTolerance(0.0f)         // Weld only vertices sharing the exact same location by default.
            , m_convexRadius(0.0f)          // Zero convex radius by default.
            , m_degenerateTolerance(0.0f)       // Remove zero area triangles only by default.
            , m_maxCompressionError(HK_REAL_MAX)// Do not bound the error by default (as we do not know the model scale in advance).
            , m_flatConvexTolerance(0.001f) // check for convex quads
        {}

        hkBool  m_globalOptimizations;  ///< Use global optimization, very slow, provide a query speedup of about 10%.
        hkBool  m_compactData;          ///< Attempt to compact triangle data.
        hkBool  m_forceConvexQuads;     ///< Allow only convex or flat quads.
        hkBool  m_preserveVertexOrder;  ///< Preserve vertex ordering in triangles. (Note: This make the method 'Primitive::isFlatQuad()' unusable, also increases memory consumption significantly).
        hkBool  m_vertexHasFlags;       ///< Set to true if vertices W component (as in getInt24W()) represent vertex flags (see VertexFlags).
        int     m_balanceThreshold;     ///< Control threshold above which the tree is rebalanced.
        Vrp     m_vrp;                  ///< Vertex representation policy, see Vrp enumeration.
        Pbd     m_pbd;                  ///< Post-build degeneracy handling, see Pbd enumeration.

        int     m_verbosity;            ///< Verbosity flags, see Verbosity enum.
        hkBool  m_weldVertices;         ///< Weld vertices.
        hkReal  m_weldTolerance;        ///< Welding tolerance.
        hkReal  m_convexRadius;         ///< Added margin around every triangle.
        hkReal  m_degenerateTolerance;  ///< Tolerance used to ignore degenerated triangles during build (see hkcdTriangleUtil::isDegenerate).
        hkReal  m_maxCompressionError;  ///< Maximum allowed compression error (distance unit).
        hkReal  m_flatConvexTolerance;  ///< if set > 0, quads will be checked for being flat and convex
    };

    // CODEC decl.
    /// Vertex CODEC.
    template <typename T, int BITS_X, int BITS_Y = BITS_X>
    struct VertexCODEC
    {
        /// Type used to represent a quantized vertex.
        typedef T   Type;

        /// Compile time configuration.
        enum
        {
            TOTAL_BITS  =   sizeof(T) << 3,
            BITS_PER_X  =   BITS_X,
            BITS_PER_Y  =   BITS_Y,
            BITS_PER_Z  =   TOTAL_BITS - (BITS_X + BITS_Y),
            MASK_X      =   (1<<BITS_PER_X)-1,
            MASK_Y      =   (1<<BITS_PER_Y)-1,
            MASK_Z      =   (1<<BITS_PER_Z)-1
        };

        /// Setup config.
        static HK_INLINE void   setupParameters(const hkAabb& aabb, CodecParameters& config);
        /// Encode a vertex.
        static HK_INLINE void   encode(const hkAabb& aabb, const hkVector4& vertex, T& output);

        // Decode a vertex, reference implementation.
        static HK_INLINE void   decode(const hkAabb& aabb, const T& vertex, hkVector4& output);

        /// Decode a vertex.
        static HK_INLINE void   decode(const CodecParameters& config, const T& vertex, hkVector4& output);

        /// Encode and decode a vertex.
        static HK_INLINE void roundTrip(const hkAabb& aabb, const hkVector4& vertexIn, hkVector4& vertexOut);
    };

    //
    // Queries support.
    //

    /// Get the section tree associated with a leaf.
    HK_INLINE const Section&    getLeafTree(const NodeContext& context) const { return m_sections[context->getData()]; }

    /// Initialize a filter.
    void                        initializeFilter(Filter& filter) const;

    /// Enable or disable a key in a filter.
    void                        setKeyFilter(Filter& filter, hkUint32 key, bool enable) const;

    /// Get the status of a key in a filter.
    bool                        getKeyFilter(const Filter& filter, hkUint32 key) const;

    /// Enable or disable a section in a filter.
    void                        setSectionFilter(Filter& filter, int section, bool enable) const;

    /// Get the status of a section in a filter.
    bool                        getSectionFilter(const Filter& filter, int section) const;

    /// Commit filter changes, return true if ANY of the keys is enable.
    bool                        commitFilterChanges(Filter& filter) const;

    /// Get the status of a section in a filter.
    HK_INLINE hkBool32  getNodeFilter(const Filter& filter, const NodeContext& node) const;

    /// Get the status of a section in a filter.
    HK_INLINE hkUint32  getPrimitiveFilter(const Filter& filter, int section, int primitive) const;

    //
    // Fields.
    //

    int         m_numPrimitiveKeys; ///< Number of primitive keys.
    int         m_bitsPerKey;       ///< Minimum number of bits to store a primitive key.
    hkUint32    m_maxKeyValue;      ///< Maximum value of a primitive key.
    hkUint8     m_primitiveStoresIsFlatConvex; ///< if this is 0xff, primitive will contain the isFlatConvex information

    hkArray<Section>            m_sections;             ///< Sections.
    hkArray<Primitive>          m_primitives;           ///< Primitives.
    hkArray<hkUint16>           m_sharedVerticesIndex;  ///< Shared vertices index.

    //
    // Tables.
    //
    static const hkUint8    s_aabbShaftsTable[32][1 + 2 * 8]; ///< Table used to build AABBs convex hull shaft planes.
    static const hkSimdReal s_zeroDistance; ///< Epsilon used to compare distances to zero in signed/unsigned distance queries.

    //
    // Utilities.
    //

    /// Compute the size of an array in bytes.
    template <typename T>
    static HK_INLINE int    arraySizeInBytes(const hkArrayBase<T>& array) { return (int) (sizeof(T) * array.getSize()); }

    /// Compute the plane equation of a triangle using fast and safe arithmetic.
    static HK_INLINE void   planeEquation(hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkVector4& planeOut);

    // Reverse the order of values in blocks.
    #if HK_ENDIAN_LITTLE
    template <typename VALUE, typename BLOCK>
    static HK_INLINE void swapBlocks(BLOCK* blocks, int numBlocks)
    {
        const int valuesPerBlock = sizeof(BLOCK) / sizeof(VALUE);
        for(int i=0; i<numBlocks; ++i)
        {
            const BLOCK     bi = blocks[i];
            const VALUE*    vi = (const VALUE*) &bi;
            BLOCK           bo = 0;
            VALUE*          vo = (VALUE*) &bo;
            for(int j=0; j<valuesPerBlock; ++j)
            {
                vo[valuesPerBlock-j-1] = vi[j];
            }
            blocks[i] = bo;
        }
    }
    #endif

    // Get the index of a value in the block.
    #if HK_ENDIAN_LITTLE
        // Reverse order for little-endian platforms.
    template <typename S,typename T>
    static HK_INLINE int fetchIndex(int i)
    {
        const int m = sizeof(T) / sizeof(S) - 1;
        return ( i & (~m) ) + (m - (i & m));
    }
    #else
        // Identity for big-endian platforms.
    template <typename S,typename T>
    static HK_INLINE int fetchIndex(int i)
    {
        return i;
    }
    #endif

protected:
    HK_INLINE hkcdStaticMeshTreeBase() : m_numPrimitiveKeys(0), m_bitsPerKey(0), m_primitiveStoresIsFlatConvex(0) {}
};

    /// Static mesh tree template.
    /// Stores geometry and collision information in a format optimized for memory and query speed.
    /// Compression rate between 8 and 12 bytes per triangle (from low-res to hi-res).
    /// Can build about 200000 triangles per seconds on a 2.66Ghz PC.
    /// Support all tree queries but stack-less ones.
    ///
    /// In addition of triangle meshes, it can store custom primitives, each custom primitive can store the following:
    /// - A type on 4 bits (getCustomPritmiveType).
    /// - A set of up to 3 tags as hkUint16 (getCustomPrimitiveTag).
    /// - Up to '255 - number of tag' compressed vertices (getCustomPrimitiveVertices).
    /// Implementation notes:
    /// Custom primitive indices are as follow:
    /// m_indices[0] = First index of 2 + num tags in the index array.
    /// m_indices[1] = Leaf node index in the section tree.
    /// First index : Num vertices : 8 | Num tags : 2 | Compression : 2 | Type : 4
    /// Second index : First vertex in the shared vertex array.
    /// Then tags.
    ///
    /// General notes:
    ///     - Support efficient sparse ( number of unique data less than the number of triangles) data storage for each triangle.
    ///     - Support conservative, from-region occlusion queries.
    ///     - Primitive key format: SECTION INDEX(m_bitsPerKey-8 bits) | PRIMITIVE INDEX(7 bits) | TRIANGLE INDEX(1 bit).
    /// - Maximum key capacity: 2^23.
template <typename A, typename B> struct hkcdStaticMeshTreeReflection;

template <typename CT_CONFIG, typename PRIMITIVE_DATA_RUN>
struct hkcdStaticMeshTree : public hkcdStaticMeshTreeBase
{
    HK_DECLARE_CLASS(hkcdStaticMeshTree, New, Reflect);

    //
    // Types
    //

    friend struct hkcdStaticMeshTreeReflection<CT_CONFIG, PRIMITIVE_DATA_RUN>;

    /// Compile time config.
    typedef CT_CONFIG                               CtConfig;

    /// Packed type.
    typedef typename CtConfig::PACKED_TYPE          PackedType;

    /// Shared type.
    typedef typename CtConfig::SHARED_TYPE          SharedType;

    /// Primitive data run (from user defined 'PRIMITIVE_DATA_RUN').
    typedef PRIMITIVE_DATA_RUN                          PrimitiveDataRun;

    /// Triangle data type (from user defined 'PRIMITIVE_DATA_RUN::TriangleData').
    typedef typename PRIMITIVE_DATA_RUN::TriangleData   TriangleDataType;

    /// Shared vertices CODEC.
    typedef VertexCODEC<SharedType, CtConfig::BITS_PER_SHARED>  SharedCodec;

    /// Packed vertices CODEC.
    typedef VertexCODEC<PackedType, CtConfig::BITS_PER_PACKED>  PackedCodec;

    /// Custom vertices CM_LOCAL_4 CODEC.
    typedef VertexCODEC<hkUint32, 11>                           Local4Codec;

    /// Custom vertices CM_LOCAL_2 CODEC.
    typedef VertexCODEC<hkUint16, 5>                            Local2Codec;

    /// Section decoder.
    struct Decoder
    {
        /// Initialize.
        HK_INLINE void                  initialize(const hkcdStaticMeshTree* mesh, hkBool32 forceSingleTriangles );

        /// Set section from index.
        /// Note that even if requireSectionTree is false, the tree will be loaded if the section is flagged with SF_REQUIRE_TREE.
        HK_INLINE void                  setSection(int section, bool requireSectionTree = true);

        /// Set section from reference (adjust reference if called on SPU).
        HK_INLINE void                  setSection(const Section*& section);

        /// Get section.
        HK_INLINE int                   getSection() const;

        /// Decode a vertex.
        HK_INLINE void                  decodeVertex(int localIndex, hkVector4& vertexOut) const;

        /// Decode a vertex given a primitive key and a vertex index.
        HK_INLINE void                  decodeVertex(hkUint32 primitiveKey, int index, hkVector4& vertexOut) const;

        /// Decode a primitive vertices, returns the number of triangles (1 or 2).
        HK_INLINE int                   decodePrimitive(int localIndex, hkVector4* HK_RESTRICT verticesOut) const;

        /// Decode a primitive from a key, returns the number of triangles (1 or 2).
        HK_INLINE int                   decodePrimitiveFromKey(hkUint32 key, hkVector4* verticesOut);

        /// [Only valid when built with VRB_EXTERN] Decode a vertex.
        HK_INLINE void                  decodeVertex(int localIndex, PackedType& indexOut) const;

        /// [Only valid when built with VRB_EXTERN] Decode a vertex given a primitive key and a vertex index.
        HK_INLINE void                  decodeVertex(hkUint32 primitiveKey, int index, PackedType& indexOut) const;

        /// [Only valid when built with VRB_EXTERN] Decode a primitive vertex indices, returns the number of triangles (1 or 2).
        HK_INLINE int                   decodePrimitive(int localIndex, PackedType* indicesOut) const;

        /// [Only valid when built with VRB_EXTERN] Decode a primitive from a key, returns the number of triangles (1 or 2).
        HK_INLINE int                   decodePrimitiveFromKey(hkUint32 key, PackedType* indicesOut);

        /// Check whether the primitive is a triangle or a quad.
        HK_INLINE hkBool32              isPrimitiveTriangleOrQuad(int localIndex) const;

        /// Check if the primitive is a flat convex, assuming its a quad
        HK_INLINE hkBool32              isPrimitiveFlatConvex(int localIndex) const;

        HK_INLINE hkBool32              isPrimitiveFlatConvexQuadFromKey(hkUint32 key);

        /// Get a read-only pointer to a primitive.
        HK_INLINE const Primitive*      getPrimitive(int localIndex) const;

        /// Get the primitive type.
        HK_INLINE Primitive::Type       getPrimitiveType(int localIndex) const;

        /// Get a primitive key.
        HK_INLINE hkUint32              getPrimitiveKey(int localIndex, int triangle) const;

        /// Get a primitive key.
        HK_INLINE int                   getPrimitiveTriangleCount(int localIndex) const;

        /// Get a primitive vertices ids.
        HK_INLINE void                  getPrimitiveVerticesIds(int localIndex, hkUint32* ids) const;

        /// Get a custom primitive key.
        HK_INLINE hkUint32              getCustomPrimitiveKey(int localIndex) const;

        /// Get a custom primitive type.
        HK_INLINE int                   getCustomPrimitiveType(int localIndex) const;

        /// Get a custom primitive number of tags.
        HK_INLINE int                   getCustomPrimitiveNumTags(int localIndex) const;

        /// Get a custom primitive tag.
        template <typename T>
        HK_INLINE const T&              getCustomPrimitiveTag(int localIndex, int tagIndex) const;

        /// Get a custom primitive compression mode.
        HK_INLINE CompressionMode       getCustomPrimitiveCompressionMode(int localIndex) const;

        /// Get a custom primitive number of vertices.
        HK_INLINE int                   getCustomPrimitiveNumVertices(int localIndex) const;

        /// Get custom primitive vertices. Returns the number of vertices actually loaded.
        HK_INLINE int                   getCustomPrimitiveVertices(int localIndex, const hkAabb& aabb, hkVector4* HK_RESTRICT verticesOut, int maxVertices) const;

        /// Internal implementation of getCustomPrimitiveVertices
        template <typename CODEC, typename CPV>
        HK_INLINE void                  decompressCustomPrimitiveVertices(const CodecParameters& parms, int numVertices, const CPV* inputs, hkVector4* HK_RESTRICT verticesOut) const;

        /// Get a primitive data.
        #if defined(HK_PLATFORM_WIN32) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_DURANGO) || defined(HK_PLATFORM_NX)
        HK_INLINE
        #else
        HK_NEVER_INLINE
        #endif
        TriangleDataType                getPrimitiveData(int localIndex) const;

        /// Get a primitive AABB. Note: this should NOT be used in time-critical situations.
        HK_INLINE void                  getPrimitiveAabb(int localIndex, hkAabb& aabbOut) const;

        /// Is primitive valid? (its type is not INVALID).
        HK_INLINE hkBool32              isPrimitiveValid(int localIndex) const;

        HK_INLINE void                  waitForCompletion() const {}

        CodecParameters             m_sharedParms;          ///< Config for shared decoder.
        CodecParameters             m_packedParms;          ///< Config for packed decoder.
        const hkcdStaticMeshTree*   m_mesh;                 ///< Mesh tree reference.
        hkUint8                     m_primitiveStoresIsFlatConvex;
        const Section*              m_section;              ///< Section reference.
        const Primitive*            m_primitives;           ///< Primitives buffer pointer.
        const PackedType*           m_packedVertices;       ///< Packed vertices buffer pointer.
        const SharedType*           m_sharedVertices;       ///< Shared vertices buffer pointer.
        const hkUint16*             m_sharedVerticesIndex;  ///< Shared vertices index buffer pointer.
        const PrimitiveDataRun*     m_primitiveDataRuns;    ///< Primitive data runs.
        int                         m_firstSharedIndex;     ///< First shared index.
        int                         m_sectionIndex;         ///< Index of the current section.
        int                         m_nodeSectionIndex;     ///< Section to which the node data belong too.
    };

    /// Iterator
    /// Note that the iterator will enumerate all primitives, including invalid ones.
    struct Iterator : Decoder
    {
        /// Action taken by the nextTriangle method.
        enum Action { ACTION_INVALID, ACTION_NEXT_TRIANGLE, ACTION_NEXT_PRIMITIVE, ACTION_NEXT_SECTION };

        /// Initialize the iterator.
        HK_INLINE               Iterator(const hkcdStaticMeshTree* mesh);

        /// Is the iterator valid?
        HK_INLINE hkBool        isValid() const;

        /// Get return current key.
        HK_INLINE hkUint32  getKey() const;

        /// Move to the next triangle.
        HK_INLINE Action        nextTriangle();

        /// Move to the next primitive.
        HK_INLINE hkBool        nextPrimitive();

        int     m_currentSection;   ///< Current active section.
        int     m_currentPrimitive; ///< Current primitive in the active section.
        int     m_currentTriangle;  ///< Current triangle in the active primitive.
        hkBool  m_valid;            ///< True if the iterator is valid.
    };

    // Build types.
    #include <Geometry/Internal/DataStructures/StaticMeshTree/hkcdStaticMeshTreeBuildTypes.inl>

    //
    // Methods.
    //

    /// Remove all the data from the static mesh tree.
    inline void                     clear();

    /// Compute the memory foot-print.
    inline int                      getMemoryFootPrint(MemoryStatistics& stats) const;

    /// Compute the memory foot-print.
    inline int                      getMemoryFootPrint() const;

    /// Returns the first primitive key of the mesh or INVALID_PRIMITIVE_KEY if no key is available.
    inline hkUint32                 getFirstKey() const;

    /// Returns the next primitive key of the mesh or INVALID_PRIMITIVE_KEY if no key is available.
    /// Note that this method will enumerate all primitives, including invalid ones.
    inline hkUint32                 getNextKey(hkUint32 primitiveKey) const;

    /// Get the two vertices associated with a connectivity edge.
    inline void                     getEdgeVertices(Edge edge, hkVector4* verticesOut) const { getEdgeVertices(edge.key(), edge.index(), verticesOut); }

    /// Get edge start vertex.
    inline void                     getStart(Edge edge, hkVector4* vertexOut) const { hkVector4 t[3]; getTriangleVertices(edge.key(), t); *vertexOut = t[edge.index()]; }

    /// Get edge end vertex.
    inline void                     getEnd(Edge edge, hkVector4* vertexOut) const { hkVector4 t[3]; getTriangleVertices(edge.key(), t); *vertexOut = t[(edge.index() + 1)%3]; }

    /// Get edge apex vertex.
    inline void                     getApex(Edge edge, hkVector4* vertexOut) const { hkVector4 t[3]; getTriangleVertices(edge.key(), t); *vertexOut = t[(edge.index() + 2)%3]; }

    /// Get the vertices associated with a primitive key and an edge index.
    inline void                     getEdgeVertices(hkUint32 primitiveKey, int edge, hkVector4* verticesOut) const;

    /// Get the triangle vertices associated with a primitive key.
    inline void                     getTriangleVertices(hkUint32 primitiveKey, hkVector4* verticesOut) const;

    /// Get the quad vertices associated with a primitive key, returns the number of valid triangles.
    inline int                      getQuadVertices(hkUint32 primitiveKey, hkVector4* verticesOut) const;

    /// Get the data associated with a primitive key.
    inline TriangleDataType         getPrimitiveData(hkUint32 primitiveKey) const;

    /// Get a reference to a primitive.
    inline Primitive::Type          getPrimitiveType(hkUint32 primitiveKey) const;

    /// Build a static mesh tree using the provided geometry provider.
    /// Note: GEOMETRY_PROVIDER must implement the required set of methods, see DefaultGeometryProvider.
    template <typename GEOMETRY_PROVIDER>
    bool                            build(const BuildConfig& config, const GEOMETRY_PROVIDER& geometry, BuildMapping& mapping);

    /// Build a static mesh tree from geometry using the default triangle data policy (triangle materials as data).
    inline bool                     build(const BuildConfig& config, const hkGeometry& geometry, BuildMapping& mapping);

    /// Build a static mesh tree from geometry using the default triangle data policy and layer data (triangle materials as data).
    inline bool                     build(const BuildConfig& config, const hkGeometry& geometry, const hkUint8* layers, BuildMapping& mapping);

    /// Compute connectivity.
    inline void                     computeConnectivity(hkArray<Links>& connectivity) const;

    /// Base class for conversion of custom primitives and filtering.
    /// By default, convert vertices to convex hull and then to geometry.
    /// You need to derive from this class to implement a custom converter if any of your custom primitives have different requirements.
    struct CustomGeometryConverter
    {
        virtual             ~CustomGeometryConverter() {}
        virtual inline bool isKeyConvertionAllowed(hkUint32 key) const { return true; }
        virtual inline void appendCustomPrimitiveToGeometry(const Decoder& decoder, int primitiveIndex, hkGeometry& geometry) const;
        virtual inline void appendExternalPrimitiveToGeometry(const Decoder& decoder, int primitiveIndex, hkGeometry& geometry) const;
        virtual inline const hkVector4& getScale() const { return hkVector4::getConstant(HK_QUADREAL_1); }

        static inline void  appendVerticesAsConvexHull(const hkVector4* vertices, int numVertices, int material, hkGeometry& geometry);
    };

    /// Define the way triangles material are set during conversion.
    enum TriangleMaterial
    {
        TM_SET_FROM_TRIANGLE_DATA_TYPE, ///< Materials are cast from TriangleDataType.
        TM_SET_FROM_PRIMITIVE_KEY       ///< Materials are set to the primitive key.
    };

    /// Convert a static mesh tree to geometry.
    inline void                     convertToGeometry(hkGeometry& geometry, const CustomGeometryConverter& gc = CustomGeometryConverter(), bool append = false, bool weld = true, TriangleMaterial tm = TM_SET_FROM_TRIANGLE_DATA_TYPE) const;

    /// Convert a specific section of the mesh tree to geometry.
    inline void                     convertToGeometry(int section, hkGeometry& geometry, const CustomGeometryConverter& gc = CustomGeometryConverter(), bool append = false, bool weld = true, TriangleMaterial tm = TM_SET_FROM_TRIANGLE_DATA_TYPE) const;

    /// Build an hkcdSimdTree from this mesh.
    inline void                     buildSimdTree(hkcdSimdTree& treeOut) const;

    //
    // Queries support.
    //

    /// Base query with decoder
    struct BaseUnaryQuery
    {
        HK_INLINE               BaseUnaryQuery(const hkcdStaticMeshTree* mesh = HK_NULL, hkBool32 forceSingleTriangles = false ) { if(mesh) m_decoder.initialize(mesh, forceSingleTriangles); }
        HK_INLINE   hkBool32    enterSubTree(const hkcdStaticMeshTree& mesh, const Section*& section, const NodeContext&) { m_decoder.setSection(section); return 1; }
        HK_INLINE   void        leaveSubTree(const Section& section, const hkcdStaticMeshTree& mesh) const {}

        Decoder m_decoder; ///< Decoder.
    };

    /// Base binary query.
    struct BaseBinaryQuery
    {
        HK_INLINE               BaseBinaryQuery(
            const hkcdStaticMeshTree* meshA = HK_NULL,  const hkcdStaticMeshTree* meshB = HK_NULL, hkBool32 forceSingleTriangles = false)
        {
            if(meshA) m_decoders[0].initialize(meshA, forceSingleTriangles); if(meshB) m_decoders[1].initialize(meshB, forceSingleTriangles); }
        HK_INLINE   hkBool32    enterSubTrees(  const hkcdStaticMeshTree& meshA, const Section*& sectionA, const NodeContext&,
                                                    const hkcdStaticMeshTree& meshB, const Section*& sectionB, const NodeContext&) { m_decoders[0].setSection(sectionA); m_decoders[1].setSection(sectionB); return 1; }
        HK_INLINE   void        leaveSubTrees(  const Section& sectionA, const hkcdStaticMeshTree& meshA,
                                                    const Section& sectionB, const hkcdStaticMeshTree& meshB) const {}

        Decoder m_decoders[2]; ///< Decoders.
    };

#if defined(__HAVOK_PARSER__)
    struct OcclusionQuery {};
#else
    /// Occlusion query.
    #include <Geometry/Internal/DataStructures/StaticMeshTree/hkcdStaticMeshTreeOcclusionQuery.inl>
#endif

    /// Basic queries definitions.
    #include <Geometry/Internal/DataStructures/StaticMeshTree/hkcdStaticMeshTreeQueries.inl>

    /// Ray cast query.
    /// Return the key of the hit triangle (or INVALID_PRIMITIVE_KEY is no hit was found).
    inline hkUint32     rayCast(hkcdRay& ray, hkBool twoSided, hkVector4* normalOut = HK_NULL) const;

    /// Ray cast query using an hkcdSimdTree acceleration data-structure.
    inline hkUint32     rayCast(hkcdRay& ray, const hkcdSimdTree& simdTree, hkBool twoSided, hkBool reportFirstHitOnly, hkVector4* normalOut = HK_NULL) const;

    /// Unsigned distance from a point (note, the W component of projectionAndDistanceInOut must be initialized).
    /// Return the key of the closest triangle the projection belong too (or INVALID_PRIMITIVE_KEY is no triangle was found).
    inline hkUint32     unsignedDistanceWithNmp(hkVector4Parameter point, hkVector4& projectionAndDistanceInOut, hkSimdReal& nmpOut, hkUint32 initialKey = HKCD_INVALID_KEY, hkVector4* normalOut = HK_NULL, const hkcdSimdTree* simdTree = HK_NULL) const;

    /// Unsigned distance from a point to a primitive key.
    /// Return the key of the closest triangle the projection belong too.
    inline hkUint32     unsignedDistanceFromPrimitiveKey(hkUint32 key, hkVector4Parameter point, hkVector4& projectionAndDistanceOut, hkVector4* normalOut = HK_NULL) const;

    /// Unsigned distance from a point (note, the W component of projectionAndDistanceInOut must be initialized).
    /// Return the key of the closest triangle the projection belong too (or INVALID_PRIMITIVE_KEY is no triangle was found).
    inline hkUint32     unsignedDistance(hkVector4Parameter point, hkVector4& projectionAndDistanceInOut, hkVector4* normalOut = HK_NULL, const hkcdSimdTree* simdTree = HK_NULL) const;

    /// Unsigned distance from a point (note, the W component of projectionAndDistanceInOut must be initialized).
    /// Return the key of the closest triangle the projection belong too (or INVALID_PRIMITIVE_KEY is no triangle was found).
    inline hkUint32     unsignedDistance(hkVector4Parameter point, int numKeys, const hkUint32* keys, hkVector4& projectionAndDistanceInOut, hkVector4* normalOut = HK_NULL) const;

    /// Signed distance from a point (note, the W component of projectionAndDistanceInOut must be initialized).
    /// Return the key of the closest triangle the projection belong too (or INVALID_PRIMITIVE_KEY is no triangle was found).
    inline hkUint32     signedDistanceWithNmp(const Connectivity& connectivity, hkVector4Parameter point, hkVector4& projectionAndDistanceInOut, hkSimdReal& nmpOut, hkUint32 initialKey = HKCD_INVALID_KEY, hkVector4* normalOut = HK_NULL, const hkcdSimdTree* simdTree = HK_NULL) const;

    /// Signed distance from a point.
    /// Return the key of the closest triangle the projection belong too.
    inline hkUint32     signedDistanceFromPrimitiveKey(hkUint32 key, const Connectivity& connectivity, hkVector4Parameter point, hkVector4& projectionAndDistanceInOut, hkVector4* normalOut = HK_NULL) const;

    /// Signed distance from a point (note, the W component of projectionAndDistanceInOut must be initialized).
    /// Return the key of the closest triangle the projection belong too (or INVALID_PRIMITIVE_KEY is no triangle was found).
    inline hkUint32     signedDistance(const Connectivity& connectivity, hkVector4Parameter point, hkVector4& projectionAndDistanceInOut, hkVector4* normalOut = HK_NULL, const hkcdSimdTree* simdTree = HK_NULL) const;

    /// Signed distance from a point to a set of keys (note, the W component of projectionAndDistanceInOut must be initialized).
    /// Return the key of the closest triangle the projection belong too (or INVALID_PRIMITIVE_KEY is no triangle was found).
    inline hkUint32     signedDistance(const Connectivity& connectivity, hkVector4Parameter point, int numKeys, const hkUint32* keys, hkVector4& projectionAndDistanceInOut, hkVector4* normalOut = HK_NULL) const;

    /// Compute the sign (-1 = inside, +1 outside or 0 if the key is invalid) of a point with respect to a given triangle (key).
    inline int          computeSign(hkVector4Parameter point, hkUint32 key, const Connectivity& connectivity) const;

    /// Collect all triangles that lay in the provided range.
    /// - pointAndRange must be initialized with the range in its W component.
    /// - if more than maxKeys are found, return maxKeys + 1.
    inline int          collectInRange(hkVector4Parameter pointAndRange, hkUint32* keysOut, int maxKeys) const;

    /// Supporting method to compute normal from key, point, projection and distance.
    /// Note: 'normal' is allowed to be null and key is allowed to be HKCD_INVALID_KEY.
    HK_INLINE void      computeNormal(hkUint32 key, const hkVector4& point, const hkVector4& projectionAndDistance, _Inout_opt_ hkVector4* normal) const;

    /// Supporting method to compute projection and unsigned distance squared to a primitive key.
    /// Returns the nearest key.
    HK_INLINE hkUint32  computeUnsignedProjectionDistanceToPrimitiveKey(hkUint32 key, const hkVector4& point, hkVector4& projectionAndDistanceOut) const;

    //
    // Serialization support.
    //

    HK_INLINE   hkcdStaticMeshTree() { }

    //
    // Fields.
    //

    hkArray<PackedType>         m_packedVertices;       ///< Packed vertices.
    hkArray<SharedType>         m_sharedVertices;       ///< Shared vertices.
    hkArray<PrimitiveDataRun>   m_primitiveDataRuns;    ///< Primitive data runs.
};

//
// Specializations
//

/// Configuration commons.
template <typename PACKED, typename SHARED, int PACKED_BITS, int SHARED_BITS>
struct HK_EXPORT_COMMON hkcdStaticMeshTreeCommonConfig
{
    HK_DECLARE_CLASS(hkcdStaticMeshTreeCommonConfig, NewPlacement, Reflect);

    enum
    {
        BITS_PER_PACKED     =   PACKED_BITS,    ///< Bits per fraction for X and Y coordinates (Z = sizeof(PACKED_TYPE) * 8 - 2 * BITS_PER_PACKED).
        BITS_PER_SHARED     =   SHARED_BITS,    ///< Bits per fraction for X and Y coordinates (Z = sizeof(SHARED_TYPE) * 8 - 2 * BITS_PER_SHARED).
    };

    typedef PACKED  PACKED_TYPE;        ///< Type holding a packed vertex.
    typedef SHARED  SHARED_TYPE;        ///< Type holding a shared vertex.
};


#include <Geometry/Collide/DataStructures/StaticMeshTree/hkcdStaticMeshTree.inl>
#include <Geometry/Collide/_Auto/TemplateTypes/hkcdStaticMeshTree_Types.inl>

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
