// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

#include <Common/Base/Types/hkHandle.h>
#include <Common/Base/Container/FreeListArray/hkFreeListArray.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Algorithm/Collide/1AxisSweep/hk1AxisSweep.h>

#include <Geometry/Collide/DataStructures/Planar/Geometry/hkcdPlanarGeometry.h>

/// A solid geometry, represented as a Bsp tree. The actual geometry is managed by a hkcdPlanarGeometry
class HK_EXPORT_COMMON hkcdPlanarSolid : public hkcdPlanarEntity
{
    public:

        HK_DECLARE_CLASS( hkcdPlanarSolid, New, Reflect, BypassCtor);
        HK_RECORD_ATTR(hk::Version(1));
        HK_RECORD_ATTR(hk::IncludeInMgd(false));

    public:

        enum
        {
            DEFAULT_MAX_PLANES_PER_BVH_CELL = 40
        };

        enum BuildResult
        {
            BUILD_SUCCESS               = 0,            ///< Build successful, no particularity
            BUILD_SUCCESS_CYCLIC_MAT    = 1,            ///< Build successful but had to solve cycling materials on polygons
            BUILD_WITH_DOUBLED_POLYS    = 2,            ///< Attempted to build the tree but some polygons were doubled/flipped. The solid regions might be incorrect
            BUILD_FAILURE               = 3,            ///< Failed to build the tree
            BUILD_ABORTED               = 4,            ///< The build has been interrupted.
        };

        // Types
        struct Node;
        HK_DECLARE_HANDLE(NodeId, hkUint32, 0xFFFFFFFFU);
        HK_DECLARE_HANDLE(AabbId, hkUint32, 0xFFFFFFFFU);
        struct HK_EXPORT_COMMON NodeStorage : public hkReferencedObject
        {
            HK_DECLARE_CLASS(NodeStorage, Reflect, New, Version(1));
            typedef hkAabb          Aabb;

            /// Constructor
            NodeStorage()
            :   hkReferencedObject()
            ,   m_firstFreeNodeId(NodeId::invalid())
            {}

            /// Copy constructor
            NodeStorage(const NodeStorage& other)
            :   hkReferencedObject()
            ,   m_firstFreeNodeId(other.m_firstFreeNodeId)
            {
                m_storage.append(other.m_storage);
                m_aabbs.append(other.m_aabbs);
            }

            virtual ~NodeStorage()
            {}

        public:

            /// Accessors
            HK_INLINE const Node& getNode(NodeId nodeId) const;
            HK_INLINE Node& accessNode(NodeId nodeId);
            HK_INLINE const Aabb& getNodeAabb(NodeId nodeId) const;
            HK_INLINE Aabb& accessNodeAabb(NodeId nodeId);
            HK_INLINE const Aabb& getNodeAabb(AabbId aabbId) const;
            HK_INLINE Aabb& accessNodeAabb(AabbId aabbId);

            /// Allocation
            HK_INLINE NodeId allocate();
            HK_INLINE void clear();
            HK_INLINE int getCapacity() const;
            HK_INLINE void release(NodeId nodeId);

            HK_INLINE AabbId allocateAabb();

            /// Other
            HK_INLINE void swapStorage(hkArray<Node>& nodes, hkArray<Aabb>& aabbs);
            HK_INLINE void compact();

        protected:

            hkArray<Node> m_storage;
            hkArray<Aabb> m_aabbs;
            NodeId m_firstFreeNodeId;           //+overridetype(hkUint32)
        };

    public:

        typedef hkUint16                NodeTypes;

        // Node type
        enum NodeTypesEnum
        {
            NODE_TYPE_INTERNAL  = 0x0,  ///< Internal BSP node
            NODE_TYPE_IN        = 0x1,  ///< Leaf SOLID node, shared by all internal nodes
            NODE_TYPE_OUT       = 0x2,  ///< Leaf EMPTY / OUTSIDE node, shared by all internal nodes
            NODE_TYPE_UNKNOWN   = 0x3,  ///< Leaf unlabeled node, shared by all internal nodes
            NODE_TYPE_INVALID   = 0x4,  ///< Value used to mark pre-allocated nodes that will no longer be used when these nodes will be removed from the code.
            NODE_TYPE_FREE      = 0xF,  ///< The node is not allocated
        };

        // Node in the BSP tree
        struct Node
        {
            //+version(2)

            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdPlanarSolid::Node);
            HK_DECLARE_REFLECTION();

            /// Constructor
            HK_INLINE Node()
            :   m_parent(NodeId::invalid())
            ,   m_type(NODE_TYPE_FREE)
            ,   m_flags(0)
            {}


            /// Returns true if the triangle is allocated
            HK_INLINE bool isAllocated() const;

            /// Marks this node as not-allocated
            HK_INLINE void clear();

            /// Called by the free-list to initialize an unallocated entry
            HK_INLINE static void setEmpty(Node& element, hkUint32 next);

            /// Returns the index of the next free element. Will be called only with empty elements
            HK_INLINE static hkUint32 getNext(const Node& element);

            /// Sets the given child id
            HK_INLINE void setChildId(int childIdx, NodeId childNodeId);

            /// Parent node
            NodeId  m_parent;           //+overridetype(hkUint32)

            /// Left child, contains the space behind the splitting plane (inside)
            NodeId  m_left;             //+overridetype(hkUint32)

            /// Right child, contains the space in front of the splitting plane (outside)
            NodeId  m_right;            //+overridetype(hkUint32)

            /// Address of the next free node
            NodeId  m_nextFreeNodeId;   //+overridetype(hkUint32)

            /// Splitting plane
            PlaneId m_planeId;          //+overridetype(hkUint32)

            /// Aabb
            AabbId m_aabbId;            //+overridetype(hkUint32)

            /// Material information
            Material m_material;

            /// User data, typically used to store a spatial CellId
            hkUint32 m_data;

            /// Type in NodeTypesEnum
            hkUint16 m_type;

            /// Flags
            hkUint16 m_flags;
        };

        /// Array slot. Basically an offset + length into a common, shared memory buffer
        struct ArraySlot
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdPlanarSolid::ArraySlot);
            HK_DECLARE_POD_TYPE();

            /// Constructor
            HK_INLINE ArraySlot() {}

            hkUint32 m_offset;  ///< Array start offset in the shared buffer
            hkUint32 m_size;    ///< Number of elements in the array
        };

        /// A unique Id for an ArraySlot
        HK_DECLARE_HANDLE(ArrayId, hkUint32, 0xFFFFFFFFU);

        /// Manages the arrays allocated during classifyPolygons
        struct ArrayMgr : public hkcdPlanarGeometryPrimitives::Collection<hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_BIT>
        {
        public:

            HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY);

        public:

            /// Constructor
            HK_INLINE ArrayMgr();

        public:

            /// Allocates an array slot. The storage for the array is not yet allocated
            HK_INLINE ArrayId allocArraySlot();

            /// Returns the array slot at the given Id
            HK_INLINE ArraySlot& getArraySlot(ArrayId aid);

            /// Returns the array storage for the given Id
            HK_INLINE PolygonId getPolygonId(ArrayId aid, int k);

            /// Allocates storage for numPolygons in the array at the given slot
            HK_INLINE hkResult allocArrayStorage(ArrayId arraySlot, _In_reads_(numPolygons) const PolygonId* HK_RESTRICT srcPolyIds, int numPolygons);
            HK_INLINE hkResult allocArrayStorage(ArrayId arraySlot, const hkArray<PolygonId>& srcPolyIds);
            HK_INLINE void freeArray(ArrayId arraySlot);

            /// Resets the manager for a new use
            HK_INLINE void reset();

        protected:

            hkFreeListArray<ArraySlot, ArrayId, 128> m_arrays;
        };


    public:

        /// Constructor
        hkcdPlanarSolid(_In_opt_ const PlanesCollection* planesCollection = HK_NULL, int initialNodeCapacity = 0, _In_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL);

        /// Constructor from collection pointers
        hkcdPlanarSolid(_In_ NodeStorage* nodeStorage, NodeId rootNodeId, _In_ const PlanesCollection* planesCollection, _In_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL);

        /// Copy constructor
        hkcdPlanarSolid(const hkcdPlanarSolid& other);

        /// Destructor
        virtual ~hkcdPlanarSolid();

    public:

        /// Builds a convex solid from an array of planes. The planes are assumed to bound a closed convex region, there are no checks to validate the assumption!
        void buildConvex(_In_reads_(numPlanes) const PlaneId* HK_RESTRICT planesIn, int numPlanes);

        /// Builds the tree for the given set of planes
        void buildTree(hkcdPlanarGeometry& polySoup, hkPseudoRandomGenerator& rng, hkArray<PlaneId>& planes, const hkArray<PolygonId>& polygons, bool useBoundaryPlanes, _Inout_opt_ hkcdPlanarEntityDebugger* debugger);

        /// Builds the tree storing a Bounding Volume Hierarchy on nodes (currently AABBs)
        BuildResult buildBVHTree(_In_opt_ const ExecutionContext* executionCtx, hkcdPlanarGeometry& polySoup, const hkArray<PolygonId>& srcPolygonIds, int numMaxPlanesPerCell = DEFAULT_MAX_PLANES_PER_BVH_CELL,
            _Inout_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL, _Inout_opt_ hkArray<PolygonId>* doubledPolyIdsCollector = HK_NULL);

        /// Builds the tree for the given set of planes, considering the plane as delimiting a flat surface
        void buildTree2D(hkcdPlanarGeometry& polySoup, hkPseudoRandomGenerator& rng, hkArray<PlaneId>& planes, const hkArray<PolygonId>& polygons, _Inout_opt_ hkcdPlanarEntityDebugger* debugger);

        /// Sets a new planes collection. If the plane remapping table is non-null, the plane Ids on all nodes will be re-set as well (i.e. to match the plane Ids in the new collection)
        void setPlanesCollection(_In_opt_ const PlanesCollection* newPlanes, _Inout_updates_opt_(_Inexpressible_()) int* HK_RESTRICT planeRemapTable, bool addMissingPlanes = false);

        /// Shift all plane ids on the nodes of the tree
        void shiftPlaneIds(int offsetValue);

        /// Remaps the materials triangle source Ids
        void remapTriangleProviderIds(const hkArray<TriangleProviderId>& triSrcIdRemap);

        /// Simplifies the tree by collapsing nodes with identical labels. Returns true if something was collapsed
        hkBool32 collapseIdenticalLabels();

        /// Optimizes the storage, by moving all unallocated nodes at the end and releasing unused memory. This will
        /// modify the Ids of the nodes!
        void optimizeStorage();

        /// Collapses any nodes still marked as unknown
        hkBool32 collapseUnknownLabels();

        /// Classifies the polygons into inside, boundary, and outside parts
        HK_INLINE hkResult classifyPolygons(    hkcdPlanarGeometry& polySoup, const hkArray<PolygonId>& polygonsIn,
                                                hkArray<PolygonId>& insidePieces, hkArray<PolygonId>& boundaryPieces, hkArray<PolygonId>& outsidePieces, _Inout_opt_ ArrayMgr* arrayMgr = HK_NULL) const;

        /// Special case of the classifies, where only the inside OR the boundary polys are needed
        void classifyInsideOrBoundaryPolygons(hkcdPlanarGeometry& polySoup, const hkArray<PolygonId>& polygonsIn, hkArray<PolygonId>& insideOrBoundPolyIds, _Inout_opt_ ArrayMgr* arrayMgr = HK_NULL);

        /// Collects all the plane Ids used by the Bsp tree
        void collectUsedPlaneIds(hkBitField& usedPlaneIdsOut) const;

        /// Returns the maximum depth of the tree
        int computeMaxDepth() const;

        // Clears the tree
        void clear();

        /// Computes the number of nodes with the specified label
        int computeNumNodesWithLabel(hkUint32 label) const;

        /// Computes the number of leaf nodes
        int computeNumLeafNodes() const;

        /// Invert the node labels from the specified node
        void invertNodeLabels(NodeId rootNodeId);

        /// Return an array of plane ids that hold the valid material of the tree
        void getPlaneIdsWithValidMaterial(hkArray<PlaneId>& planeIdsOut) const;

        /// Debug. Prints all the tree data
        void dbgPrint() const;

    public:

        /// Gets the node having the given Id.
        HK_INLINE const Node& getNode(NodeId nodeId) const;
        HK_INLINE Node& accessNode(NodeId nodeId);

        /// Returns the nodes
        HK_INLINE _Ret_maybenull_ const NodeStorage* getNodes() const;
        HK_INLINE _Ret_maybenull_ NodeStorage* accessNodes();

        /// Returns the collection of planes used by this solid
        HK_INLINE _Ret_maybenull_ PlanesCollection* accessPlanesCollection();
        HK_INLINE _Ret_maybenull_ const PlanesCollection* getPlanesCollection() const;

        /// Returns true if the tree is valid
        HK_INLINE bool isValid() const;

        /// Returns true if the tree represent the empty set
        HK_INLINE bool isEmpty() const;

        /// Returns true if the given node is internal
        HK_INLINE bool isInternal(NodeId nodeId) const;

        /// Default nodes accessors
        HK_INLINE NodeId createInsideNode(NodeId parentId);
        HK_INLINE NodeId createOutsideNode(NodeId parentId);
        HK_INLINE NodeId createUnknownNode(NodeId parentId);

        /// Returns the Id of the root node
        HK_INLINE const NodeId getRootNodeId() const;

        /// Returns the label of a given leaf node
        HK_INLINE NodeTypes getNodeLabel(NodeId nodeId) const;

        /// Creates a new internal node
        HK_INLINE NodeId createNode(PlaneId splitPlaneId, NodeId leftChild, NodeId rightChild);

        /// Sets the root node
        HK_INLINE void setRootNode(NodeId rootNodeId);

        /// Returns true if the given node can be collapsed to its parent
        HK_INLINE bool canCollapse(NodeId nodeId) const;

    protected:

        /// Classifies the polygons into inside, boundary, and outside parts
        hkResult classifyPolygons(  hkcdPlanarGeometry& polySoup, NodeId nodeId,
                                const hkArray<PolygonId>& polygonsIn,
                                hkArray<PolygonId>& insidePieces, hkArray<PolygonId>& boundaryPieces, hkArray<PolygonId>& outsidePieces, ArrayMgr& arrayMgr) const;

    protected:

        /// The nodes in the Bsp tree
        hkRefPtr<NodeStorage> m_nodes;

        /// The planes collection
        hkRefPtr<const PlanesCollection> m_planes;

        /// The root node
        NodeId m_rootNodeId;            //+overridetype(hkUint32)
};

#include <Geometry/Collide/DataStructures/Planar/Solid/hkcdPlanarSolid.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
