// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/DataStructures/Planar/ConvexCellsTree/hkcdConvexCellsTree3D.h>
#include <Common/Base/Container/String/hkStringBuf.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>

#define DEBUG_LOG_IDENTIFIER "geo.collide.planarsolid"
#include <Common/Base/System/Log/hkLog.hxx>

//
//  Constructor

hkcdPlanarSolid::hkcdPlanarSolid(_In_opt_ const PlanesCollection* planesCollection, int initialNodeCapacity, _In_opt_ hkcdPlanarEntityDebugger* debugger)
:   hkcdPlanarEntity(debugger)
,   m_planes(planesCollection)
,   m_rootNodeId(NodeId::invalid())
{
    m_nodes.setAndDontIncrementRefCount(new NodeStorage());

//  // Pre-allocate the nodes array if required
//  if ( initialNodeCapacity )
//  {
//      m_nodes->m_array.grow(initialNodeCapacity);
//  }
}

/// Constructor from collection pointers
hkcdPlanarSolid::hkcdPlanarSolid(_In_ NodeStorage* nodeStorage, NodeId rootNodeId, _In_ const PlanesCollection* planesCollection, _In_opt_ hkcdPlanarEntityDebugger* debugger)
:   hkcdPlanarEntity(debugger)
,   m_nodes(nodeStorage)
,   m_planes(planesCollection)
,   m_rootNodeId(rootNodeId)
{}

//
//  Copy constructor

hkcdPlanarSolid::hkcdPlanarSolid(const hkcdPlanarSolid& other)
:   hkcdPlanarEntity(other)
,   m_planes(other.m_planes)
,   m_rootNodeId(other.m_rootNodeId)
{
    m_nodes.setAndDontIncrementRefCount(new NodeStorage(*other.m_nodes));
}

//
//  Destructor

hkcdPlanarSolid::~hkcdPlanarSolid()
{
    m_planes = HK_NULL;
}

//
//  Clears the tree

void hkcdPlanarSolid::clear()
{
    m_nodes->clear();
    m_rootNodeId = NodeId::invalid();
}

//
//  Builds a convex solid from an array of planes. The planes are assumed to bound a closed convex region, there are no checks to validate the assumption!

void hkcdPlanarSolid::buildConvex(_In_reads_(numPlanes) const PlaneId* HK_RESTRICT planesIn, int numPlanes)
{
    if ( numPlanes == 0 )
    {
        return;
    }

    NodeId nodeId                   = m_nodes->allocate();
    accessNode(nodeId).m_parent     = NodeId::invalid();
    accessNode(nodeId).m_material   = Material::invalid();
    m_rootNodeId                    = nodeId;

    for (int k = 0; k < numPlanes; k++)
    {
        Node& node          = accessNode(nodeId);

        // Set it up
        node.m_data         = 0;
        node.m_type = NODE_TYPE_INTERNAL;
        node.m_planeId      = planesIn[k];
        const NodeId inNode = createInsideNode(nodeId);
        const NodeId ouNode = createOutsideNode(nodeId);
        accessNode(nodeId).m_left       = inNode;
        accessNode(nodeId).m_right      = ouNode;
        nodeId              = getNode(nodeId).m_left;
    }
}

//
//  Sets a new planes collection. If the plane remapping table is non-null, the plane Ids on all nodes will be re-set as well (i.e. to match the plane Ids in the new collection)

void hkcdPlanarSolid::setPlanesCollection(_In_opt_ const PlanesCollection* newPlanes, _Inout_updates_opt_(_Inexpressible_()) int* HK_RESTRICT planeRemapTable, bool addMissingPlanes)
{
    // Remap all plane Ids if necessary
    if ( planeRemapTable && m_planes && newPlanes && (m_planes != newPlanes) )
    {
        for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
        {
            const NodeId nodeId     (ni);
            Node& node              = accessNode(nodeId);
            const PlaneId oldPlaneId = node.m_planeId;

            if ( node.isAllocated() && oldPlaneId.isValid() )
            {
                const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                int newPlaneIdx             = planeRemapTable[oldPlaneIdx] & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                Plane oldPlane;             m_planes->getPlane(oldPlaneId, oldPlane);
                // We may want to add the plane now
                if ( addMissingPlanes && !PlaneId(newPlaneIdx).isValid() )
                {
                    newPlaneIdx                     = ((PlanesCollection*)newPlanes)->addPlane(oldPlane).value();
                    planeRemapTable[oldPlaneIdx]    = newPlaneIdx;
                }
                Plane newPlane;             newPlanes->getPlane(PlaneId(newPlaneIdx), newPlane);
                const PlaneId newPlaneId    (newPlaneIdx | (hkcdPlanarGeometryPredicates::sameOrientation(oldPlane, newPlane) ? 0 : hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG));

                node.m_planeId = newPlaneId;
            }
        }
    }

    m_planes = newPlanes;
}

//
//  Shift all plane ids

void hkcdPlanarSolid::shiftPlaneIds(int offsetValue)
{
    for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
    {
        const NodeId nodeId         (ni);
        Node& node                  = accessNode(nodeId);
        const PlaneId oldPlaneId    = node.m_planeId;
        if ( node.isAllocated() && oldPlaneId.isValid() )
        {
            const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
            if ( oldPlaneIdx >= hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS )
            {
                const PlaneId newPlaneId    ( oldPlaneId.value() + offsetValue );
                node.m_planeId = newPlaneId;
            }
        }
    }
}

//
//  Remaps the materials triangle source Ids

void hkcdPlanarSolid::remapTriangleProviderIds(const hkArray<TriangleProviderId>& triSrcIdRemap)
{
    for (int ni = m_nodes->getCapacity() - 1 ; ni >= 0 ; ni--)
    {
        const NodeId nodeId         (ni);
        Node& bspNode               = accessNode(nodeId);
        if ( bspNode.isAllocated() && (bspNode.m_type == hkcdPlanarSolid::NODE_TYPE_INTERNAL) && bspNode.m_material.isValid() )
        {
            // Remap the tri source locally
            TriangleProviderId tsId = bspNode.m_material.getTriangleProviderId();
            bspNode.m_material.setTriangleProviderId(triSrcIdRemap[tsId.value()]);
        }
    }
}

//
//  Collects all the plane Ids used by the Bsp tree

void hkcdPlanarSolid::collectUsedPlaneIds(hkBitField& usedPlaneIdsOut) const
{
    for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
    {
        const NodeId nodeId     (ni);
        const Node& node        = getNode(nodeId);
        const PlaneId oldPlaneId = node.m_planeId;

        if ( node.isAllocated() && oldPlaneId.isValid() )
        {
            const int planeIdx  = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
            usedPlaneIdsOut.set(planeIdx);
        }
    }

    // Always add the world boundary planes
    for (int k = hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS - 1; k >= 0; k--)
    {
        usedPlaneIdsOut.set(k);
    }
}

//
//  Simplifies the tree by collapsing nodes with identical labels

hkBool32 hkcdPlanarSolid::collapseIdenticalLabels()
{
    bool hasCollapsed;
    int numCollapses = -1;
    do
    {
        hasCollapsed = false;
        numCollapses++;

        for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
        {
            const NodeId nodeId (ni);
            Node& startNode     = accessNode(nodeId);

            if ( startNode.isAllocated() && (startNode.m_type == NODE_TYPE_INTERNAL) )
            {
                // If the node has both leaves equal, we can collapse it
                NodeId crtNodeId = nodeId;
                while ( canCollapse(crtNodeId) )
                {
                    // We can collapse this node
                    Node& crtNode   = accessNode(crtNodeId);

                    // Take the label if its children
                    crtNode.m_type  = getNodeLabel(crtNode.m_left);

                    // Release children
                    m_nodes->release(getNode(crtNodeId).m_left);
                    m_nodes->release(getNode(crtNodeId).m_right);

                    // Invalidate children
                    accessNode(crtNodeId).m_left  = NodeId::invalid();
                    accessNode(crtNodeId).m_right = NodeId::invalid();

                    // Switch to parent to propagate collapse
                    crtNodeId = getNode(crtNodeId).m_parent;
                    hasCollapsed = true;
                }
            }
        }
    } while ( hasCollapsed );

    return numCollapses;
}

//
//  Optimizes the storage, by moving all unallocated nodes at the end and releasing unused memory. This will
//  modify the Ids of the nodes!

void hkcdPlanarSolid::optimizeStorage()
{
    const int maxNumNodes = m_nodes->getCapacity();

    hkArray<Node>   newNodes;       newNodes.reserve(maxNumNodes);
    hkArray<NodeStorage::Aabb> aabbs;           aabbs.reserve(maxNumNodes);
    hkArray<NodeId> newFromOldId;   newFromOldId.setSize(maxNumNodes, NodeId::invalid());

    int newNodeId = 0;
    for (int k = 0; k < maxNumNodes; k++)
    {
        const NodeId oldNodeId(k);
        const Node& node = getNode(oldNodeId);

        if ( node.isAllocated() )
        {
            newNodes.pushBack(node);
            newFromOldId[k] = NodeId(newNodeId);
            newNodeId++;
        }
    }

    // Update ids
    for (int k = 0; k < newNodes.getSize(); k++)
    {
        Node& node = newNodes[k];

        if ( node.m_left.isValid() )    {   node.m_left     = newFromOldId[node.m_left.value()];    }
        if ( node.m_right.isValid() )   {   node.m_right    = newFromOldId[node.m_right.value()];   }
        if ( node.m_parent.isValid() )  {   node.m_parent   = newFromOldId[node.m_parent.value()];  }
    }

    // Update root and make sure the special nodes have not changed!
    if ( m_rootNodeId.isValid() )   {   m_rootNodeId = newFromOldId[m_rootNodeId.value()];  }

    // Finally, swap the new array with the old
    m_nodes->swapStorage(newNodes, aabbs);
    m_nodes->compact();
}

//
//  Utility functions

namespace hkcdBspImpl
{
    // Type shortcuts
    typedef hkcdPlanarGeometry::PlaneId         PlaneId;
    typedef hkcdPlanarGeometry::Plane           Plane;
    typedef hkcdPlanarGeometry::PolygonId       PolygonId;
    typedef hkcdPlanarGeometry::Material        Material;
    typedef hkcdPlanarSolid::NodeId             NodeId;
    typedef hkcdPlanarSolid::AabbId             AabbId;

    /// Stack entry
    struct StackEntry
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdBspImpl::StackEntry);

        PlaneId* m_planeIds;            ///< The array of plane Ids that can be used to classify the polygons
        PolygonId* m_polygonIds;        ///< The array of polygon Ids that still need to be classified
        AabbId m_aabbId;                ///< Id of the aabb corresponding to this entry
        int m_numPlaneIds;              ///< The number of plane Ids
        int m_numPolygonIds;            ///< The number of polygon Ids
        int m_isLeftChild;              ///< True if this child is the left child of the parent node
        NodeId m_parentNodeId;          ///< The parent node that pushed this entry
    };

    /// The stack
    class Stack
    {
        public:

            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdBspImpl::Stack);
            enum { LEFT_CHILD_FLAG  = 0x80000000, NUM_ENTRY_SIG_MEMBERS = 4, };

        public:

            HK_INLINE void pop(StackEntry& entryOut)
            {
                const int n = m_storage.getSize();
                entryOut.m_isLeftChild      = m_storage[n - 2] & LEFT_CHILD_FLAG;
                entryOut.m_parentNodeId     = NodeId(m_storage[n - 1]);
                entryOut.m_numPolygonIds    = m_storage[n - 2] & (~LEFT_CHILD_FLAG);
                entryOut.m_numPlaneIds      = m_storage[n - 3];
                entryOut.m_aabbId           = AabbId(m_storage[n - 4]);
                entryOut.m_polygonIds       = reinterpret_cast<PolygonId*>(&m_storage[n - NUM_ENTRY_SIG_MEMBERS - entryOut.m_numPolygonIds]);
                entryOut.m_planeIds         = reinterpret_cast<PlaneId*>(&m_storage[n - NUM_ENTRY_SIG_MEMBERS - entryOut.m_numPolygonIds - entryOut.m_numPlaneIds]);
                m_storage.setSize(n - NUM_ENTRY_SIG_MEMBERS - entryOut.m_numPolygonIds - entryOut.m_numPlaneIds);
            }

            HK_INLINE void push(const StackEntry& entryIn)
            {
                const int n = m_storage.getSize() + entryIn.m_numPlaneIds + entryIn.m_numPolygonIds + NUM_ENTRY_SIG_MEMBERS;
                m_storage.setSize(n);
                hkString::memCpy4(&m_storage[n - NUM_ENTRY_SIG_MEMBERS - entryIn.m_numPolygonIds - entryIn.m_numPlaneIds], entryIn.m_planeIds, entryIn.m_numPlaneIds);
                hkString::memCpy4(&m_storage[n - NUM_ENTRY_SIG_MEMBERS - entryIn.m_numPolygonIds], entryIn.m_polygonIds, entryIn.m_numPolygonIds);
                m_storage[n - 4]    = entryIn.m_aabbId.valueUnchecked();
                m_storage[n - 3]    = entryIn.m_numPlaneIds;
                m_storage[n - 2]    = entryIn.m_numPolygonIds | (entryIn.m_isLeftChild ? (int)LEFT_CHILD_FLAG : 0);
                m_storage[n - 1]    = entryIn.m_parentNodeId.valueUnchecked();
            }

            HK_INLINE bool isEmpty() const
            {
                return !m_storage.getSize();
            }

        protected:

            hkArray<int> m_storage; ///< The stack storage
    };

    //
    // Selects a splitting plane from the given list

    static int HK_CALL pickSplittingPlane(const hkcdPlanarGeometry& geom, hkPseudoRandomGenerator& rng,
        _In_reads_(numPlaneIds) const PlaneId* HK_RESTRICT planeIds, int numPlaneIds,
        _In_reads_(numPolygonIds) const PolygonId* HK_RESTRICT polygonIds, int numPolygonIds)
    {
        HK_ASSERT_NO_MSG(0xde15ab23, numPlaneIds > 0);

        // Initialize our best estimate for the splitting plane
        int bestCost        = -0x7FFFFFFF;
        int bestPlaneIdx    = -1;

        // Try a fixed number of random planes
        for (int crtTry = 0; crtTry < 5; crtTry++)
        {
            const int pi                = rng.getRand32() % numPlaneIds;
            const PlaneId splitPlaneId  = planeIds[pi];

            // Clear statistics
            int numInFront  = 0;
            int numBehind   = 0;
            int numSplit    = 0;

            // Test all other polygons against the current splitting plane
            for (int pj = 0; pj < numPolygonIds; pj++)
            {
                if ( pj != pi )
                {
                    const hkcdPlanarGeometryPredicates::Orientation ori = geom.approxClassify(polygonIds[pj], splitPlaneId);
                    switch ( ori )
                    {
                    case hkcdPlanarGeometryPredicates::BEHIND:      numBehind++;    break;
                    case hkcdPlanarGeometryPredicates::IN_FRONT_OF: numInFront++;   break;
                    case hkcdPlanarGeometryPredicates::INTERSECT:   numSplit++;     break;
                    default:    break;
                    }
                }
            }

            // Compute heuristic: h(splitPlane) = front - wSplit * split, wSplit = 8
            const int heuristic = -numSplit;
            if ( heuristic > bestCost )
            {
                bestCost = heuristic;
                bestPlaneIdx = pi;
            }
        }

        // Return our best estimate
        HK_ASSERT_NO_MSG(0x31536947, bestPlaneIdx >= 0);
        return bestPlaneIdx;
    }

    //
    // Selects a splitting plane from the given list (2d material version)

    static int HK_CALL pickSplittingPlaneMat(const hkcdPlanarGeometry& geom, hkPseudoRandomGenerator& rng,
        _In_reads_(numPlaneIds) const PlaneId* HK_RESTRICT planeIds, int numPlaneIds,
        _In_reads_(numPolygonIds) const PolygonId* HK_RESTRICT polygonIds, int numPolygonIds)
    {
        HK_ASSERT_NO_MSG(0xde17ab56, numPlaneIds > 0);

        // Initialize our best estimate for the splitting plane
        const int targetSplitNum    = numPolygonIds/2;
        int bestCost                = numPolygonIds*2;
        int bestPlaneIdx            = -1;
        bool hasValidPlane          = false;

        // Try a fixed number of random planes
        for (int crtTry = 0 ; crtTry < 5 || !hasValidPlane ; crtTry++)
        {
            const int pi = ( crtTry < 5 ) ? rng.getRand32() % numPlaneIds : crtTry - 5;
            if ( pi == numPlaneIds )
            {
                return -1;      // Impossible to solve case: e.g. two identical polygons
            }

            const PlaneId splitPlaneId = planeIds[pi];

            // Clear statistics
            int numInFront = 0;
            int numBehind = 0;
            int numSplit = 0;

            // Test all polygons against the current splitting plane
            for (int pj = 0; pj < numPolygonIds; pj++)
            {
                const hkcdPlanarGeometryPredicates::Orientation ori = geom.classify(polygonIds[pj], splitPlaneId);
                switch (ori)
                {
                case hkcdPlanarGeometryPredicates::BEHIND:      numBehind++;    break;
                case hkcdPlanarGeometryPredicates::IN_FRONT_OF: numInFront++;   break;
                case hkcdPlanarGeometryPredicates::INTERSECT:   numSplit++;     break;
                default:    break;
                }
            }

            bool isPlaneValid = (numBehind > 0 && numInFront > 0) || ((numBehind > 0 || numInFront > 0) && numSplit > 0);

            if ( isPlaneValid )
            {
                hasValidPlane = true;
                const int heuristic = hkMath::abs(numInFront - targetSplitNum) + hkMath::abs(numBehind - targetSplitNum);

                if (heuristic < bestCost)
                {
                    bestCost = heuristic;
                    bestPlaneIdx = pi;
                    if (heuristic == 0)
                    {
                        break;  // We have the perfect plane, no need to search further
                    }
                }
            }
        }

        // Return our best estimate
        HK_ASSERT_NO_MSG(0x31536958, bestPlaneIdx >= 0);
        return bestPlaneIdx;
    }

    //
    // Test a splitting plane against the ploys

    static void HK_INLINE testSplittingPlane(
        const hkcdPlanarGeometry& geom, const Plane& splitPlane, _In_reads_(numPolys) const PolygonId* polygonIds, int numPolys,
        int& numInFront, int& numBehind)
    {
        hkArray<PolygonId> polyInFrontIds, polyBehindIds;

        // Test all polygons against the current splitting plane
        for (int pj = 0 ; pj < numPolys ; pj++)
        {
            const PolygonId polyId = polygonIds[pj];
            const hkcdPlanarGeometryPredicates::Orientation ori = geom.approxClassify(polyId, splitPlane);
            switch (ori)
            {
                case hkcdPlanarGeometryPredicates::BEHIND:      polyBehindIds.pushBack(polyId);     break;
                case hkcdPlanarGeometryPredicates::IN_FRONT_OF: polyInFrontIds.pushBack(polyId);    break;
                default:    break;
            }
        }

        // count the number of planes for each set
        hkArray<PlaneId> planeIds;
        geom.getAllPolygonsPlanes(polyBehindIds, planeIds, false);
        numBehind = planeIds.getSize();
        planeIds.setSize(0);
        geom.getAllPolygonsPlanes(polyInFrontIds, planeIds, false);
        numInFront = planeIds.getSize();
    }

    //
    // Struct that holds aligned planes to avoid duplicates

    struct AlignedPlaneFinder
    {
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdBspImpl::AlignedPlaneFinder);

        /// Constructor
        AlignedPlaneFinder(_In_ hkcdPlanarGeometryPlanesCollection* planeCollection)
        :   m_planeCollection(planeCollection)
        {
            m_planeIds.setSize(3);

            // Loop over all the planes and add the already aligned ones
            for (int p = m_planeCollection->getNumPlanes() - 1 ; p >= hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS ; p--)
            {
                const PlaneId planeId(p);
                Plane plane;
                m_planeCollection->getPlane(planeId, plane);
                if ( getPlaneOrientation(plane) != -1 )
                {
                    insertPlane(planeId);
                }
            }
        }

        /// Inserts a new plane in the collection
        void insertPlane(PlaneId planeId)
        {
            // Get the plane orientation
            Plane plane;
            m_planeCollection->getPlane(planeId, plane);
            int ori = getPlaneOrientation(plane);
            HK_ASSERT_NO_MSG(0x8a6ee000, ori >= 0);

            // Store the plane into the collection
            m_planeIds[ori].pushBack(planeId);
        }

        /// try to find an already stored plane
        PlaneId findPlane(const Plane& plane)
        {
            // Get the plane orientation
            int ori = getPlaneOrientation(plane);
            HK_ASSERT_NO_MSG(0x8a6ee000, ori >= 0);

            // Get the offset
            hkInt128 exactOffset;
            plane.getExactOffset(exactOffset);
            const hkInt32 planeOffset = hkInt32(exactOffset.getDoubleWord(0));

            // Try to find it in the list
            hkArray<PlaneId>& planeList = m_planeIds[ori];
            for (int p = planeList.getSize() - 1 ; p >= 0 ; p--)
            {
                // Get the plane
                Plane plane2;
                m_planeCollection->getPlane(planeList[p], plane2);

                // Compare offsets
                hkInt128 exactOffsetp;
                plane2.getExactOffset(exactOffsetp);
                const hkInt32 planeOffsetp = hkInt32(exactOffsetp.getDoubleWord(0));

                if ( planeOffsetp == planeOffset )
                {
                    return planeList[p];
                }
                else if ( -planeOffsetp == planeOffset )
                {
                    return hkcdPlanarGeometryPrimitives::getOppositePlaneId(planeList[p]);
                }
            }

            return PlaneId::invalid();
        }
    protected:

        /// Returns the plane orientation
        int getPlaneOrientation(const Plane& plane)
        {
            // Get the plane and normal
            hkInt64Vector4 planeNormal;
            plane.getExactNormal(planeNormal);

            // Determine orientation
            int ori = -1;
            int numNull = 0;
            for (int o = 0; o < 3; o++)
            {
                hkInt64 c = planeNormal.getComponent(o);
                if (c != 0)
                {
                    ori = o;
                }
                else
                {
                    numNull++;
                }
            }

            return ( numNull == 2 ) ? ori : -1;
        }

        hkcdPlanarGeometryPlanesCollection* m_planeCollection;

        hkArray< hkArray<PlaneId> > m_planeIds;
    };

    //
    // Generate a splitting plane for the current BVH node

    static void HK_CALL generateSplittingPlane(
        const hkcdPlanarGeometry& geom, _In_reads_(numPolys) const PolygonId* polygonIds, int numPolys, int numPlanes, _Inout_ hkcdPlanarGeometryPlanesCollection* planeCollection,
        AlignedPlaneFinder& alignedPlaneFinder, const hkcdPlanarSolid::AabbId& nodeAabbId,
        hkcdPlanarSolid::AabbId& leftNodeAabbId, hkcdPlanarSolid::AabbId& rightNodeAabbId,
        PlaneId& splitPlaneIdOut, _Inout_ hkcdPlanarSolid::NodeStorage* nodeStorage, int numMaxIteration = 2)
    {
        splitPlaneIdOut = PlaneId::invalid();
        const hkcdPlanarSolid::NodeStorage::Aabb* nodeAabb = &nodeStorage->getNodeAabb(nodeAabbId);

        //
        const hkInt64 xes[] = {-1, 0, 0};
        const hkInt64 yes[] = {0, -1, 0};
        const hkInt64 zes[] = {0, 0, -1};

        // New plane variables
        Plane splitPlaneOut;
        hkInt64Vector4 planeNormal;
        hkInt128 planeOffset;
        int numInFront, numBehind;
        const int targetSplitNum = numPlanes/2;

        // Get the aabb of the polys
        hkAabb polysAabb;
        hkArray<PolygonId> polyIdsArray;
        polyIdsArray.setDataUserFree((PolygonId*)polygonIds, numPolys, numPolys);
        geom.computeAabb(polyIdsArray, polysAabb);
        HK_ASSERT_NO_MSG(0x654320ae, nodeAabb->contains(polysAabb));

        // Determine the best orientation
        int bestOri = -1;
        int bestValue = int(0.6f*hkReal(numPlanes));        // we want to separate at least 10 % of the planes, otherwise we consider this split as not worth
        for (int it = 0 ; it < 4 ; it++)
        {
            int ori = ( it < 3 ) ? it : bestOri;

            if ( ori == -1 )
            {
                continue;
            }

            // Get normal
            planeNormal.set(xes[ori], yes[ori], zes[ori]);

            // Get offset
            int midP = int((polysAabb.m_max.getComponent(ori) - polysAabb.m_min.getComponent(ori)).getReal());
            if ( midP < 2 )
            {
                continue;
            }
            midP /= 2;
            midP += int(polysAabb.m_min.getComponent(ori).getReal());
            planeOffset.setFromInt32(midP);

            // Set plane
            splitPlaneOut.setExactEquation(planeNormal, planeOffset, false);

            if ( it < 3 )
            {
                // Test orientation
                testSplittingPlane(geom, splitPlaneOut, polygonIds, numPolys, numInFront, numBehind);

                const int heuristic = hkMath::abs(numInFront - targetSplitNum) + hkMath::abs(numBehind - targetSplitNum);
                if ( heuristic < bestValue )
                {
                    bestOri = ori;
                    bestValue = heuristic;
                }
            }
        }

//      // Iterate to refine
//      numInFront = bestInFront;
//      numBehind = bestBehind;
//      for (int it = 0 ; it < numMaxIteration ; it++)
//      {
//          // See where there is more planes: on the front or on the back
//
//      }

        if (bestOri == -1)
        {
            return;     // No acceptable plane found
        }

        // See if the plane already exists
        splitPlaneIdOut = alignedPlaneFinder.findPlane(splitPlaneOut);
        if ( !splitPlaneIdOut.isValid() )
        {
            // Allocate a new plane
            splitPlaneIdOut = planeCollection->addPlane(splitPlaneOut);
            alignedPlaneFinder.insertPlane(splitPlaneIdOut);
        }

        // outputs the two aabbs
        leftNodeAabbId  = nodeStorage->allocateAabb();
        rightNodeAabbId = nodeStorage->allocateAabb();
        hkcdPlanarSolid::NodeStorage::Aabb& leftNodeAabb = nodeStorage->accessNodeAabb(leftNodeAabbId);
        hkcdPlanarSolid::NodeStorage::Aabb& rightNodeAabb = nodeStorage->accessNodeAabb(rightNodeAabbId);
        nodeAabb = &nodeStorage->getNodeAabb(nodeAabbId);
        leftNodeAabb    = *nodeAabb;
        rightNodeAabb   = *nodeAabb;
        const int valOffset = int(planeOffset.getDoubleWord(0));
        leftNodeAabb. m_min.setComponent(bestOri, hkSimdReal::fromInt32(valOffset - 1));
        rightNodeAabb.m_max.setComponent(bestOri, hkSimdReal::fromInt32(valOffset + 1));

    }

    //
    // Compute the aabb of the given geometry

    static void HK_CALL computeGeomAabb(const hkcdPlanarGeometry& geom, const hkArray<PolygonId>& polygonIds, hkcdPlanarSolid::NodeStorage::Aabb& aabbOut)
    {
        geom.computeAabb(polygonIds, aabbOut);

        // Expand the aabb by 2 (1 for vertex imprecision, 1 to secure overlap)
        aabbOut.expandBy(hkSimdReal_2);
    }
}

//
//  Recursively builds the tree for the given set of planes

void hkcdPlanarSolid::buildTree(hkcdPlanarGeometry& polySoup, hkPseudoRandomGenerator& rng,
    hkArray<PlaneId>& srcPlaneIds, const hkArray<PolygonId>& srcPolygonIds, bool useBoundaryPlanes,
    _Inout_opt_ hkcdPlanarEntityDebugger* debugger)
{
    // Type shortcuts
    typedef hkcdBspImpl::Stack      Stack;
    typedef hkcdBspImpl::StackEntry StackEntry;

    // Preallocate the working memory
    const int numSrcPlanes  = srcPlaneIds.getSize();
    const int numSrcPolys   = srcPolygonIds.getSize();
    hkArray<PolygonId> frontPolyIds;    frontPolyIds.reserve(numSrcPolys);
    hkArray<PolygonId> backPolyIds;     backPolyIds.reserve(numSrcPolys);
    hkArray<PlaneId> frontPlaneIds;     frontPlaneIds.reserve(numSrcPlanes);
    hkArray<PlaneId> backPlaneIds;      backPlaneIds.reserve(numSrcPlanes);
    hkArray<PlaneId> tempPlaneIds;      tempPlaneIds.reserve(numSrcPlanes);

    // Push all original data on the stack
    Stack stack;
    {
        StackEntry entry;
        entry.m_planeIds        = const_cast<PlaneId*>(srcPlaneIds.begin());
        entry.m_polygonIds      = const_cast<PolygonId*>(srcPolygonIds.begin());
        entry.m_numPlaneIds     = numSrcPlanes;
        entry.m_numPolygonIds   = numSrcPolys;
        entry.m_parentNodeId    = NodeId::invalid();
        entry.m_isLeftChild     = false;
        stack.push(entry);
    }

    // While the stack is not empty, pop and process each entry
    while ( !stack.isEmpty() )
    {
        // Pop entry
        StackEntry entry;
        stack.pop(entry);
        HK_ASSERT_NO_MSG(0x75217846, entry.m_numPolygonIds && entry.m_numPlaneIds);

        // Pick a splitting plane
        const int splitPlaneIdx     = hkcdBspImpl::pickSplittingPlane(polySoup, rng, entry.m_planeIds, entry.m_numPlaneIds, entry.m_polygonIds, entry.m_numPolygonIds);
        const PlaneId splitPlaneId  = entry.m_planeIds[splitPlaneIdx];
        for (int k = splitPlaneIdx + 1; k < entry.m_numPlaneIds; k++)
        {
            entry.m_planeIds[k - 1] = entry.m_planeIds[k];
        }
        entry.m_numPlaneIds--;
        HK_ASSERT_NO_MSG(0x57c5b1ea, splitPlaneId.isValid());

        // Classify polygons w.r.t. the splitting plane
        frontPolyIds.setSize(0);    backPolyIds.setSize(0);
        int numSameCoplanar = 0, numOppositeCoplanar = 0;
        for (int k = 0; k < entry.m_numPolygonIds; k++)
        {
            const PolygonId polyId = entry.m_polygonIds[k];
            const hkcdPlanarGeometryPredicates::Orientation ori = polySoup.classify(polyId, splitPlaneId);

            switch ( ori )
            {
            case hkcdPlanarGeometryPredicates::IN_FRONT_OF: frontPolyIds.pushBack(polyId);  break;
            case hkcdPlanarGeometryPredicates::BEHIND:      backPolyIds.pushBack(polyId);   break;

            case hkcdPlanarGeometryPredicates::INTERSECT:
                {
                    // We need to split the polygon with the plane
                    PolygonId splitInside, splitOutside;
                    polySoup.split(polyId, splitPlaneId, splitInside, splitOutside);

                    if ( splitInside.isValid() && splitOutside.isValid() )
                    {
                        backPolyIds.pushBack(splitInside);
                        frontPolyIds.pushBack(splitOutside);
                    }
                }
                break;

            default:    // On plane
                {
                    const PlaneId polySupportId = polySoup.getPolygon(polyId).getSupportPlaneId();
                    if ( hkcdPlanarGeometryPrimitives::sameOrientationPlaneIds(polySupportId, splitPlaneId) )   {   numSameCoplanar++;      }
                    else                                                                                        {   numOppositeCoplanar++;  }
                }
                break;
            }
        }

        // Allocate a new node
        NodeId nodeId = m_nodes->allocate();
        {
            Node& node          = accessNode(nodeId);
            node.m_left         = NodeId::invalid();
            node.m_right        = NodeId::invalid();
            node.m_type = NODE_TYPE_INTERNAL;
            node.m_planeId      = splitPlaneId;
            node.m_parent       = entry.m_parentNodeId;

            if ( entry.m_parentNodeId.isValid() )
            {
                Node& parent = accessNode(entry.m_parentNodeId);
                if ( entry.m_isLeftChild )  {   parent.m_left   = nodeId;   }
                else                        {   parent.m_right  = nodeId;   }
            }
            else
            {
                m_rootNodeId = nodeId;
            }
        }

        // Gather the left & right planes
        if ( backPolyIds.getSize() )
        {
            tempPlaneIds.setSize(0);                polySoup.getAllPolygonsPlanes(backPolyIds, tempPlaneIds, useBoundaryPlanes);
            backPlaneIds.setSize(numSrcPlanes);     backPlaneIds.setSize(hkAlgorithm::intersectionOfSortedLists(entry.m_planeIds, entry.m_numPlaneIds, tempPlaneIds.begin(), tempPlaneIds.getSize(), backPlaneIds.begin()));
            HK_ASSERT_NO_MSG(0xfde56432, backPlaneIds.getSize());
        }
        if ( frontPolyIds.getSize() )
        {
            tempPlaneIds.setSize(0);                polySoup.getAllPolygonsPlanes(frontPolyIds, tempPlaneIds, useBoundaryPlanes);
            frontPlaneIds.setSize(numSrcPlanes);    frontPlaneIds.setSize(hkAlgorithm::intersectionOfSortedLists(entry.m_planeIds, entry.m_numPlaneIds, tempPlaneIds.begin(), tempPlaneIds.getSize(), frontPlaneIds.begin()));
            HK_ASSERT_NO_MSG(0xfde56432, frontPlaneIds.getSize());
        }

        // Recurse on left
        if ( backPolyIds.getSize() )
        {
            StackEntry leftEntry;
            leftEntry.m_isLeftChild     = true;
            leftEntry.m_polygonIds      = backPolyIds.begin();
            leftEntry.m_numPolygonIds   = backPolyIds.getSize();
            leftEntry.m_planeIds        = backPlaneIds.begin();
            leftEntry.m_numPlaneIds     = backPlaneIds.getSize();
            leftEntry.m_parentNodeId    = nodeId;
            stack.push(leftEntry);
        }
        else
        {
            const NodeId leafNode           = ( numSameCoplanar ) ? createInsideNode(nodeId) : createOutsideNode(nodeId);
            accessNode(nodeId).m_left       = leafNode;
        }

        // Recurse on right
        if ( frontPolyIds.getSize() )
        {
            StackEntry rightEntry;
            rightEntry.m_isLeftChild    = false;
            rightEntry.m_polygonIds     = frontPolyIds.begin();
            rightEntry.m_numPolygonIds  = frontPolyIds.getSize();
            rightEntry.m_planeIds       = frontPlaneIds.begin();
            rightEntry.m_numPlaneIds    = frontPlaneIds.getSize();
            rightEntry.m_parentNodeId   = nodeId;
            stack.push(rightEntry);
        }
        else
        {
            const NodeId leafNode           = ( numSameCoplanar ) ? createOutsideNode(nodeId) : createInsideNode(nodeId);
            accessNode(nodeId).m_right      = leafNode;
        }
    }
}

//
//  Builds the tree storing a Bounding Volume Hierarchy on nodes (currently AABBs)

hkcdPlanarSolid::BuildResult hkcdPlanarSolid::buildBVHTree(_In_opt_ const ExecutionContext* executionCtx, hkcdPlanarGeometry& polySoup, const hkArray<PolygonId>& srcPolygonIds, int numMaxPlanesPerCell,
    _Inout_opt_ hkcdPlanarEntityDebugger* debugger, _Inout_opt_ hkArray<PolygonId>* doubledPolyIdsCollector)
{
    // Type shortcuts
    typedef hkcdBspImpl::Stack          Stack;
    typedef hkcdBspImpl::StackEntry     StackEntry;
    BuildResult buildResult = BUILD_SUCCESS;

    // Init aligned plane finder
    hkcdPlanarGeometryPlanesCollection* planesCollection = polySoup.accessPlanesCollection();
    hkcdBspImpl::AlignedPlaneFinder alignedPlaneFinder(planesCollection);
    hkPseudoRandomGenerator rng(13);

    // Get the planes of the polygon
    hkArray<PlaneId> srcPlaneIds;
    polySoup.getAllPolygonsPlanes(srcPolygonIds, srcPlaneIds, false);
    if (srcPlaneIds.getSize() == 0)
    {
        return buildResult;
    }

    // Preallocate the working memory
    const int numSrcPlanes = srcPlaneIds.getSize();
    const int numSrcPolys = srcPolygonIds.getSize();
    hkArray<PolygonId> frontPolyIds;    frontPolyIds.reserve(numSrcPolys);
    hkArray<PolygonId> backPolyIds;     backPolyIds.reserve(numSrcPolys);
    hkArray<PolygonId> interPolyIds;    interPolyIds.reserve(numSrcPolys);
    hkArray<PolygonId> onPolyIds;       onPolyIds.reserve(numSrcPolys);
    hkArray<PlaneId> frontPlaneIds;     frontPlaneIds.reserve(numSrcPlanes);
    hkArray<PlaneId> backPlaneIds;      backPlaneIds.reserve(numSrcPlanes);
    hkArray<PlaneId> splitMatPlaneIds;

    // Initial aabb
    AabbId rootAabbId           = m_nodes->allocateAabb();
    hkcdBspImpl::computeGeomAabb(polySoup, srcPolygonIds, m_nodes->accessNodeAabb(rootAabbId));

    // Init the stack
    Stack stack;
    {
        StackEntry entry;
        entry.m_planeIds        = const_cast<PlaneId*>(srcPlaneIds.begin());
        entry.m_polygonIds      = const_cast<PolygonId*>(srcPolygonIds.begin());
        entry.m_numPlaneIds     = srcPlaneIds.getSize();
        entry.m_numPolygonIds   = numSrcPolys;
        entry.m_parentNodeId    = NodeId::invalid();
        entry.m_aabbId          = rootAabbId;
        entry.m_isLeftChild     = false;
        stack.push(entry);
    }

    // Variables to allow disabling bvh split if too slow convergence
    bool noMoreBVHSplit = false;
    int numBVHSplit = 0;
    const int numMaxBVHSplits = srcPlaneIds.getSize();

    // While the stack is not empty, pop and process each entry
    while ( !stack.isEmpty() && !hkTask::receivedAbortRequest(executionCtx) )
    {
        // Pop entry
        StackEntry entry;
        stack.pop(entry);
        HK_ASSERT_NO_MSG(0x75217846, entry.m_numPolygonIds && entry.m_numPlaneIds);

        // Depending on the number of planes, pick either a plane from the plane pool, or generate one
        PlaneId splitPlaneId= PlaneId::invalid();
        AabbId leftAabbId   = AabbId::invalid();
        AabbId rightAabbId  = AabbId::invalid();
        bool splitBVH = !noMoreBVHSplit && (entry.m_numPlaneIds > numMaxPlanesPerCell) && entry.m_aabbId.isValid();
        if ( splitBVH )
        {
            // Too many planes, divide!
            hkcdBspImpl::generateSplittingPlane(
                polySoup, entry.m_polygonIds, entry.m_numPolygonIds, entry.m_numPlaneIds, planesCollection,
                alignedPlaneFinder, entry.m_aabbId,
                leftAabbId, rightAabbId, splitPlaneId, m_nodes);

            numBVHSplit++;
            noMoreBVHSplit = ( numBVHSplit > numMaxBVHSplits );
        }

        // Choose a suitable splitting plane
        int numSameCoplanar, numOppositeCoplanar;
        Material onPlaneMaterial;
        bool onPlaneMatValid;
        bool pickingSplitMatPlane   = false;
        bool keepOnPlanePolys       = false;
        int numTrialsToSplitMat     = 0;        // To avoid infinite cycling while truing to solve on plane material
        bool emergencyExit = false;             // Triggred when impossible to solve cases arise
        do
        {
            onPlaneMaterial = Material::invalid();
            onPlaneMatValid = false;

            if ( pickingSplitMatPlane )
            {
                // Try to pick a plane to split material ids
                int planeIdx = hkcdBspImpl::pickSplittingPlaneMat(polySoup, rng, splitMatPlaneIds.begin(), splitMatPlaneIds.getSize(),
                    onPolyIds.begin(), onPolyIds.getSize());
                if ( planeIdx >= 0 )
                {
                    splitPlaneId = splitMatPlaneIds[planeIdx];
                }
                else
                {
                    // Impossible to solve case e.g. two identical (or flipped) polys with different material.
                    // Set the material to one of the material we will found on this face
                    emergencyExit   = true;
                    splitPlaneId    = polySoup.getPolygon(onPolyIds[0]).getSupportPlaneId();
                    if ( doubledPolyIdsCollector )
                    {
                        doubledPolyIdsCollector->append(onPolyIds);
                    }
                    buildResult     = BUILD_WITH_DOUBLED_POLYS;
                }
            }
            else if ( !splitPlaneId.isValid() )
            {
                splitBVH = false;

                // Pick a splitting plane
                const int splitPlaneIdx = hkcdBspImpl::pickSplittingPlane(polySoup, rng, entry.m_planeIds, entry.m_numPlaneIds, entry.m_polygonIds, entry.m_numPolygonIds);
                splitPlaneId            = entry.m_planeIds[splitPlaneIdx];
            }

            // Classify polygons w.r.t. the splitting plane
            numSameCoplanar = numOppositeCoplanar = 0;
            frontPolyIds.setSize(0);    backPolyIds.setSize(0);     interPolyIds.setSize(0);    onPolyIds.setSize(0);
            for (int k = 0; k < entry.m_numPolygonIds; k++)
            {
                const PolygonId polyId = entry.m_polygonIds[k];
                const hkcdPlanarGeometryPredicates::Orientation ori = polySoup.classify(polyId, splitPlaneId);

                switch (ori)
                {
                case hkcdPlanarGeometryPredicates::IN_FRONT_OF: frontPolyIds.pushBack(polyId);  break;
                case hkcdPlanarGeometryPredicates::BEHIND:      backPolyIds.pushBack(polyId);   break;
                case hkcdPlanarGeometryPredicates::INTERSECT:   interPolyIds.pushBack(polyId);  break;

                default:
                {
                    // On plane
                    const PlaneId polySupportId = polySoup.getPolygon(polyId).getSupportPlaneId();
                    if (hkcdPlanarGeometryPrimitives::sameOrientationPlaneIds(polySupportId, splitPlaneId)) { numSameCoplanar++; }
                    else                                                                                    { numOppositeCoplanar++; }

                    onPolyIds.pushBack(polyId);

                    // Test for on plane material
                    const Material polyMat = polySoup.getPolygon(polyId).getMaterial();
                    if ( !onPlaneMaterial.isValid() )
                    {
                        onPlaneMaterial = polyMat;
                        onPlaneMatValid = true;
                    }
                    else if ( polyMat != onPlaneMaterial )
                    {
                        onPlaneMatValid = false;
                    }
                }
                    break;
                }
            }

            // If we have several mat on this split plane, divide them
            if ( !emergencyExit )
            {
                if ( onPlaneMaterial.isValid() && !onPlaneMatValid )
                {
                    if ( pickingSplitMatPlane && numTrialsToSplitMat >= 4 )
                    {
                        // Potential cycle detected, keep the plane anyway, keep the on plane polys as well
                        keepOnPlanePolys    = true;
                        buildResult         = ( buildResult == BUILD_SUCCESS ) ? BUILD_SUCCESS_CYCLIC_MAT : buildResult;
                    }
                    else
                    {
                        splitPlaneId = PlaneId::invalid();
                        // Gather and save boundary planes
                        splitMatPlaneIds.setSize(0);
                        polySoup.getAllPolygonsPlanes(onPolyIds, splitMatPlaneIds, true, false);
                        numTrialsToSplitMat++;
                        pickingSplitMatPlane = true;
                        leftAabbId = AabbId::invalid();
                        rightAabbId = AabbId::invalid();
                    }
                }
                else
                    // if we fail to split the poly in BVH build, retry with an poly plane
                if ( splitBVH && !interPolyIds.getSize() && (!backPolyIds.getSize() || !frontPolyIds.getSize()) )
                {
                    splitPlaneId = PlaneId::invalid();
                    leftAabbId = AabbId::invalid();
                    rightAabbId = AabbId::invalid();
                }
            }
        } while ( !splitPlaneId.isValid() );

        // Split all the polys to be split with the splitting plane
        for (int pIdx = interPolyIds.getSize() - 1 ; pIdx >= 0 ; pIdx--)
        {
            // We need to split the polygon with the plane
            PolygonId splitInside, splitOutside;
            polySoup.split(interPolyIds[pIdx], splitPlaneId, splitInside, splitOutside);

            if ( splitInside.isValid() && splitOutside.isValid() )
            {
                backPolyIds.pushBack(splitInside);
                frontPolyIds.pushBack(splitOutside);
            }
        }

        // Allocate a new node
        NodeId nodeId = m_nodes->allocate();
        {
            Node& node          = accessNode(nodeId);
            node.m_left         = NodeId::invalid();
            node.m_right        = NodeId::invalid();
            node.m_type         = NODE_TYPE_INTERNAL;
            node.m_planeId      = splitPlaneId;
            node.m_parent       = entry.m_parentNodeId;
            node.m_aabbId       = entry.m_aabbId;
            node.m_material     = ( onPlaneMatValid || emergencyExit ) ? onPlaneMaterial : Material::invalid();

            if (entry.m_parentNodeId.isValid())
            {
                Node& parent = accessNode(entry.m_parentNodeId);
                if (entry.m_isLeftChild)    { parent.m_left = nodeId; }
                else                        { parent.m_right = nodeId; }
            }
            else
            {
                m_rootNodeId    = nodeId;
            }
        }

        if ( keepOnPlanePolys )
        {
            // In the very rare case where we couldn't solve the materials, keep all the polys for this plane
            // So that the plane will be picked later again, this time with a better chance of finding a spliting plane for materials
            backPolyIds.append(onPolyIds);
            frontPolyIds.append(onPolyIds);
        }

        // Gather the left & right planes
        backPlaneIds.setSize(0);    frontPlaneIds.setSize(0);
        if ( backPolyIds.getSize() )
        {
            polySoup.getAllPolygonsPlanes(backPolyIds, backPlaneIds, false);
            HK_ASSERT_NO_MSG(0xfde56432, backPlaneIds.getSize());
        }
        if ( frontPolyIds.getSize() )
        {
            polySoup.getAllPolygonsPlanes(frontPolyIds, frontPlaneIds, false);
            HK_ASSERT_NO_MSG(0xfde56432, frontPlaneIds.getSize());
        }

        // Recurse on left
        if ( backPolyIds.getSize() )
        {
            StackEntry leftEntry;
            leftEntry.m_isLeftChild     = true;
            leftEntry.m_polygonIds      = backPolyIds.begin();
            leftEntry.m_numPolygonIds   = backPolyIds.getSize();
            leftEntry.m_planeIds        = backPlaneIds.begin();
            leftEntry.m_numPlaneIds     = backPlaneIds.getSize();
            leftEntry.m_parentNodeId    = nodeId;
            leftEntry.m_aabbId          = leftAabbId;
            stack.push(leftEntry);
        }
        else
        {
            const NodeId leafNode = ( numSameCoplanar && !leftAabbId.isValid() ) ? createInsideNode(nodeId) : createOutsideNode(nodeId);
            accessNode(nodeId).m_left = leafNode;
        }

        // Recurse on right
        if ( frontPolyIds.getSize() )
        {
            StackEntry rightEntry;
            rightEntry.m_isLeftChild    = false;
            rightEntry.m_polygonIds     = frontPolyIds.begin();
            rightEntry.m_numPolygonIds  = frontPolyIds.getSize();
            rightEntry.m_planeIds       = frontPlaneIds.begin();
            rightEntry.m_numPlaneIds    = frontPlaneIds.getSize();
            rightEntry.m_parentNodeId   = nodeId;
            rightEntry.m_aabbId         = rightAabbId;
            stack.push(rightEntry);
        }
        else
        {
            const NodeId leafNode = ( numSameCoplanar || rightAabbId.isValid() ) ? createOutsideNode(nodeId) : createInsideNode(nodeId);
            accessNode(nodeId).m_right = leafNode;
        }
    }

    return hkTask::receivedAbortRequest(executionCtx) ? BUILD_ABORTED : buildResult;
}

//
//  Builds the tree for the given set of planes, considering the plane as delimiting a flat surface

void hkcdPlanarSolid::buildTree2D(hkcdPlanarGeometry& polySoup, hkPseudoRandomGenerator& rng, hkArray<PlaneId>& srcPlaneIds, const hkArray<PolygonId>& srcPolygonIds, _Inout_opt_ hkcdPlanarEntityDebugger* debugger)
{
    // Type shortcuts
    typedef hkcdBspImpl::Stack      Stack;
    typedef hkcdBspImpl::StackEntry StackEntry;

    // Preallocate the working memory
    const int numSrcPlanes  = srcPlaneIds.getSize();
    const int numSrcPolys   = srcPolygonIds.getSize();
    hkArray<PolygonId> frontPolyIds;    frontPolyIds.reserve(numSrcPolys);
    hkArray<PolygonId> backPolyIds;     backPolyIds.reserve(numSrcPolys);
    hkArray<PlaneId> frontPlaneIds;     frontPlaneIds.reserve(numSrcPlanes);
    hkArray<PlaneId> backPlaneIds;      backPlaneIds.reserve(numSrcPlanes);

    // Push all original data on the stack
    Stack stack;
    {
        StackEntry entry;
        entry.m_planeIds        = const_cast<PlaneId*>(srcPlaneIds.begin());
        entry.m_polygonIds      = const_cast<PolygonId*>(srcPolygonIds.begin());
        entry.m_numPlaneIds     = numSrcPlanes;
        entry.m_numPolygonIds   = numSrcPolys;
        entry.m_parentNodeId    = NodeId::invalid();
        entry.m_isLeftChild     = false;
        stack.push(entry);
    }

    // While the stack is not empty, pop and process each entry
    while ( !stack.isEmpty() )
    {
        // Pop entry
        StackEntry entry;
        stack.pop(entry);
        HK_ASSERT_NO_MSG(0x75217846, entry.m_numPolygonIds && entry.m_numPlaneIds);

        // Pick a splitting plane
        const int splitPlaneIdx     = hkcdBspImpl::pickSplittingPlane(polySoup, rng, entry.m_planeIds, entry.m_numPlaneIds, entry.m_polygonIds, entry.m_numPolygonIds);
        const PlaneId splitPlaneId  = entry.m_planeIds[splitPlaneIdx];
        for (int k = splitPlaneIdx + 1; k < entry.m_numPlaneIds; k++)
        {
            entry.m_planeIds[k - 1] = entry.m_planeIds[k];
        }
        entry.m_numPlaneIds--;
        HK_ASSERT_NO_MSG(0x57c5b1ea, splitPlaneId.isValid());

        // Classify polygons w.r.t. the splitting plane
        frontPolyIds.setSize(0);    backPolyIds.setSize(0);
        int numSameCoplanar = 0, numOppositeCoplanar = 0;
        for (int k = 0; k < entry.m_numPolygonIds; k++)
        {
            const PolygonId polyId = entry.m_polygonIds[k];
            const hkcdPlanarGeometryPredicates::Orientation ori = polySoup.classify(polyId, splitPlaneId);

            switch ( ori )
            {
            case hkcdPlanarGeometryPredicates::IN_FRONT_OF: frontPolyIds.pushBack(polyId);  break;
            case hkcdPlanarGeometryPredicates::BEHIND:      backPolyIds.pushBack(polyId);   break;

            case hkcdPlanarGeometryPredicates::INTERSECT:
                {
                    // We need to split the polygon with the plane
                    PolygonId splitInside, splitOutside;
                    polySoup.split(polyId, splitPlaneId, splitInside, splitOutside);

                    if ( splitInside.isValid() && splitOutside.isValid() )
                    {
                        backPolyIds.pushBack(splitInside);
                        frontPolyIds.pushBack(splitOutside);
                    }
                }
                break;

            default:    // On plane
                {
                    const PlaneId polySupportId = polySoup.getPolygon(polyId).getSupportPlaneId();
                    if ( hkcdPlanarGeometryPrimitives::sameOrientationPlaneIds(polySupportId, splitPlaneId) )   {   numSameCoplanar++;      }
                    else                                                                                        {   numOppositeCoplanar++;  }
                }
                break;
            }
        }

        // Allocate a new node
        NodeId nodeId = m_nodes->allocate();
        {
            Node& node          = accessNode(nodeId);
            node.m_left         = NodeId::invalid();
            node.m_right        = NodeId::invalid();
            node.m_type = NODE_TYPE_INTERNAL;
            node.m_planeId      = splitPlaneId;
            node.m_parent       = entry.m_parentNodeId;

            if ( entry.m_parentNodeId.isValid() )
            {
                Node& parent = accessNode(entry.m_parentNodeId);
                if ( entry.m_isLeftChild )  {   parent.m_left   = nodeId;   }
                else                        {   parent.m_right  = nodeId;   }
            }
            else
            {
                m_rootNodeId = nodeId;
            }
        }

        // Gather the left & right planes
        backPlaneIds.setSize(0);
        if ( backPolyIds.getSize() )
        {
            // Polygon planes might not match the BSP planes, just pass all!
            backPlaneIds.append(entry.m_planeIds, entry.m_numPlaneIds);
        }
        frontPlaneIds.setSize(0);
        if ( frontPolyIds.getSize() )
        {
            // Polygon planes might not match the BSP planes, just pass all!
            frontPlaneIds.append(entry.m_planeIds, entry.m_numPlaneIds);
        }

        // Recurse on left
        if ( backPlaneIds.getSize() )
        {
            StackEntry leftEntry;
            leftEntry.m_isLeftChild     = true;
            leftEntry.m_polygonIds      = backPolyIds.begin();
            leftEntry.m_numPolygonIds   = backPolyIds.getSize();
            leftEntry.m_planeIds        = backPlaneIds.begin();
            leftEntry.m_numPlaneIds     = backPlaneIds.getSize();
            leftEntry.m_parentNodeId    = nodeId;
            stack.push(leftEntry);
        }
        else
        {
            const NodeId leafNode           = ( backPolyIds.getSize() ) ? createInsideNode(nodeId) : createOutsideNode(nodeId);
            accessNode(nodeId).m_left       = leafNode;
        }

        // Recurse on right
        if ( frontPlaneIds.getSize() )
        {
            StackEntry rightEntry;
            rightEntry.m_isLeftChild    = false;
            rightEntry.m_polygonIds     = frontPolyIds.begin();
            rightEntry.m_numPolygonIds  = frontPolyIds.getSize();
            rightEntry.m_planeIds       = frontPlaneIds.begin();
            rightEntry.m_numPlaneIds    = frontPlaneIds.getSize();
            rightEntry.m_parentNodeId   = nodeId;
            stack.push(rightEntry);
        }
        else
        {
            const NodeId leafNode           = ( frontPolyIds.getSize() ) ? createInsideNode(nodeId) : createOutsideNode(nodeId);
            accessNode(nodeId).m_right      = leafNode;
        }
    }
}

//
//  Collapses any nodes still marked as unknown

hkBool32 hkcdPlanarSolid::collapseUnknownLabels()
{
    int numCollapses = 0;

    bool found;
    do
    {
        found = false;

        for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
        {
            const NodeId nodeId(ni);
            Node& bspNode = accessNode(nodeId);

            if ( bspNode.isAllocated() && (bspNode.m_type == NODE_TYPE_INTERNAL) )
            {
                const Node& nodeLeftChild = getNode(bspNode.m_left);
                const Node& nodeRightChild = getNode(bspNode.m_right);

                if ( (nodeLeftChild.m_type == NODE_TYPE_UNKNOWN) || (nodeRightChild.m_type == NODE_TYPE_UNKNOWN) )
                {
                    // We can remove this node and link its parent directly to the right child
                    const NodeId knownChildNodeId = (nodeLeftChild.m_type == NODE_TYPE_UNKNOWN) ? bspNode.m_right : bspNode.m_left;
                    Node& knownChild = accessNode(knownChildNodeId);

                    if ( bspNode.m_parent.isValid() )
                    {
                        Node& parentNode = accessNode(bspNode.m_parent);
                        if ( parentNode.m_left == nodeId )  {                                                           parentNode.m_left   = knownChildNodeId; }
                        else                                {   HK_ASSERT_NO_MSG(0xd0536f1, parentNode.m_right == nodeId);      parentNode.m_right  = knownChildNodeId; }
                    }
                    else
                    {
                        // This is the root node, we can set the root directly to the right child
                        m_rootNodeId = knownChildNodeId;
                    }

                    // Set the parent
                    if ( knownChild.isAllocated() )
                    {
                        HK_ASSERT_NO_MSG(0x606c5428, knownChild.m_parent == nodeId);
                        knownChild.m_parent = bspNode.m_parent;
                    }

                    // We may want to remove the unknown child
                    const NodeId unknownChildNodeId = (nodeLeftChild.m_type == NODE_TYPE_UNKNOWN) ? bspNode.m_left : bspNode.m_right;
                    m_nodes->release(unknownChildNodeId);

                    m_nodes->release(nodeId);
                    numCollapses++;
                    found = true;
                }
            }
        }
    } while ( found );

    return numCollapses;
}

//
//  Utility functions for the iterative implementation of classifyPolygons. The original recursive algorithm is taken from
//  Thibault's phd thesis.

namespace hkcdBspImpl
{
    /// Polygon category
    enum PolyId
    {
        POLYS_IN    = 0,
        POLYS_ON    = 1,
        POLYS_OUT   = 2,
    };

    /// Job, i.e. classifyPolygons stack entry
    struct Job
    {
        public:

            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdBspImpl::Job);

        public:

            /// Sets the job as a classify job. It takes one array as input and produces 3 arrays as output, one for each inside / outside / on boundary polygons.
            HK_INLINE void setClassify(NodeId nodeId, hkcdPlanarSolid::ArrayId inPolys, const hkcdPlanarSolid::ArrayId (&outPolys)[3])
            {
                HK_ASSERT_NO_MSG(0x2df7ea52, nodeId.isValid());
                m_nodeId                = nodeId;
                input()                 = inPolys;
                outputs<POLYS_IN>()     = outPolys[POLYS_IN];
                outputs<POLYS_ON>()     = outPolys[POLYS_ON];
                outputs<POLYS_OUT>()    = outPolys[POLYS_OUT];
            }

            /// Sets the job as a merge job. It takes 3 arrays as input and produces one merged array as output.
            HK_INLINE void setMerge(hkcdPlanarSolid::ArrayId inPolysA, hkcdPlanarSolid::ArrayId inPolysB, hkcdPlanarSolid::ArrayId inPolysC, hkcdPlanarSolid::ArrayId outPolys)
            {
                m_nodeId            = NodeId::invalid();
                output()            = outPolys;
                inputs<POLYS_IN>()  = inPolysA;
                inputs<POLYS_ON>()  = inPolysB;
                inputs<POLYS_OUT>() = inPolysC;
            }

        public:

            // Accessors for a classify job
            HK_INLINE hkcdPlanarSolid::ArrayId& input() {   return m_arrays[0];     }
            template <PolyId T>
            HK_INLINE hkcdPlanarSolid::ArrayId& outputs()   {   return m_arrays[1 + T]; }

            // Accessors for a merge job
            template <PolyId T>
            HK_INLINE hkcdPlanarSolid::ArrayId& inputs()    {   return m_arrays[T];     }
            HK_INLINE hkcdPlanarSolid::ArrayId& output()    {   return m_arrays[3];     }

        public:

            NodeId m_nodeId;                        ///< The Bsp tree node Id.
            hkcdPlanarSolid::ArrayId m_arrays[4];   ///< The inputs / outputs.
    };

    struct MatPoly
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdBspImpl::MatPoly);

        HK_INLINE static hkBool32 HK_CALL less(const MatPoly& a, const MatPoly& b) { return a.m_matId.value() < b.m_matId.value(); }

        Material m_matId;
        PolygonId m_polyId;
    };
}

//
//  Subtracts the solid from the given polygon and returns a list of polygons outside / on the boundary of the solid

hkResult hkcdPlanarSolid::classifyPolygons( hkcdPlanarGeometry& polySoup, NodeId rootNodeId, const hkArray<PolygonId>& origPolygonsIn,
                                        hkArray<PolygonId>& insidePolygonsOut, hkArray<PolygonId>& boundaryPolygonsOut, hkArray<PolygonId>& outsidePolygonsOut, ArrayMgr& polyStorage) const
{
    typedef hkcdBspImpl::Job            Job;

#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
    hkMonitorStream* timerStream = hkMonitorStream::getInstancePtr();
#endif

    // Early-out if no polygons provided
    if ( !origPolygonsIn.getSize() )
    {
        insidePolygonsOut.setSize(0);
        outsidePolygonsOut.setSize(0);
        boundaryPolygonsOut.setSize(0);
        return HK_SUCCESS;
    }

    HK_TIMER_BEGIN2(timerStream, "Inits", HK_NULL);
    hkArray<Job> jobStack;
    jobStack.reserve(64);

    // Allocate outputs and push root node on the stack
    const ArrayId aidPolys[3]   = { polyStorage.allocArraySlot(), polyStorage.allocArraySlot(), polyStorage.allocArraySlot() };
    const ArrayId nullPolys     = polyStorage.allocArraySlot();
    {
        Job& job = jobStack.expandOne();
        job.setClassify(rootNodeId, polyStorage.allocArraySlot(), aidPolys);
        if ( polyStorage.allocArrayStorage(job.input(), origPolygonsIn).isFailure() )
        {
            HK_TIMER_END(); // "Inits"
            return HK_FAILURE;
        }
    }
    HK_TIMER_END(); // "Inits"

    hkArray<PolygonId> tempPolys[3];
    while ( jobStack.getSize() )
    {
        // Pop last job on the stack
        Job job;
        {
            const int jobIdx = jobStack.getSize() - 1;
            job = jobStack[jobIdx];
            jobStack.removeAt(jobIdx);
        }

        // Process it
        if ( job.m_nodeId.isValid() )
        {
            // CLASSIFY job. Get current node
            const NodeId nodeId         = job.m_nodeId;
            const Node& bspNode         = getNode(nodeId);
            const ArrayId jobInput      = job.input();
            const ArraySlot inputPolys  = polyStorage.getArraySlot(jobInput);

            switch ( bspNode.m_type )
            {
            case NODE_TYPE_OUT: {   polyStorage.getArraySlot(job.outputs<hkcdBspImpl::POLYS_OUT>()) = inputPolys;   }   continue;
            case NODE_TYPE_IN:  {   polyStorage.getArraySlot(job.outputs<hkcdBspImpl::POLYS_IN>())  = inputPolys;   }   continue;
            default:    break;  // Internal node, recurse!
            }

            // Classify our polygons w.r.t. the node splitting plane
            const PlaneId splitPlaneId  = bspNode.m_planeId;
            tempPolys[hkcdBspImpl::POLYS_IN].setSize(0);
            tempPolys[hkcdBspImpl::POLYS_ON].setSize(0);
            tempPolys[hkcdBspImpl::POLYS_OUT].setSize(0);

            HK_TIMER_BEGIN2(timerStream, "Predicates", HK_NULL);
            for (int k = (int)inputPolys.m_size - 1; k >= 0; k--)
            {
                const PolygonId polyId  = polyStorage.getPolygonId(jobInput, k);
                const Orientation ori   = polySoup.classify(polyId, splitPlaneId);

                switch ( ori )
                {
                case hkcdPlanarGeometryPredicates::IN_FRONT_OF: tempPolys[hkcdBspImpl::POLYS_OUT].pushBack(polyId); break;
                case hkcdPlanarGeometryPredicates::BEHIND:      tempPolys[hkcdBspImpl::POLYS_IN].pushBack(polyId);  break;
                case hkcdPlanarGeometryPredicates::INTERSECT:
                    {
                        // We need to split the polygon into 2 parts
                        PolygonId splitInside, splitOutside;
                        polySoup.split(polyId, splitPlaneId, splitInside, splitOutside);

                        if ( splitInside.isValid() && splitOutside.isValid() )
                        {
                            tempPolys[hkcdBspImpl::POLYS_IN].pushBack(splitInside);
                            tempPolys[hkcdBspImpl::POLYS_OUT].pushBack(splitOutside);
                        }
                    }
                    break;
                default:
                    {
                        HK_ASSERT_NO_MSG(0x53da54ac, ori == hkcdPlanarGeometryPredicates::ON_PLANE);
                        tempPolys[hkcdBspImpl::POLYS_ON].pushBack(polyId);
                    }
                    break;
                }
            }
            HK_TIMER_END();

            // Free input
            polyStorage.freeArray(jobInput);

            // Allocate output arrays
            const ArrayId polys_L[3]        = { polyStorage.allocArraySlot(), polyStorage.allocArraySlot(), polyStorage.allocArraySlot() };
            const ArrayId polys_R[3]        = { polyStorage.allocArraySlot(), polyStorage.allocArraySlot(), polyStorage.allocArraySlot() };
            const ArrayId coplanar_L[3]     = { polyStorage.allocArraySlot(), polyStorage.allocArraySlot(), polyStorage.allocArraySlot() };
            const ArrayId coplanar_InL[3]   = { polyStorage.allocArraySlot(), polyStorage.allocArraySlot(), polyStorage.allocArraySlot() };
            const ArrayId coplanar_OutL[3]  = { polyStorage.allocArraySlot(), polyStorage.allocArraySlot(), polyStorage.allocArraySlot() };

            // Merge all into the job outputs
            if ( inputPolys.m_size )
            {
                // Merge coplanar polygons
                const ArrayId tempMergedArrayA = polyStorage.allocArraySlot();
                const ArrayId tempMergedArrayB = polyStorage.allocArraySlot();
                Job* jobs = jobStack.expandBy(5);

                jobs[0].setMerge(polys_L[hkcdBspImpl::POLYS_IN], polys_R[hkcdBspImpl::POLYS_IN], coplanar_InL[hkcdBspImpl::POLYS_IN], job.outputs<hkcdBspImpl::POLYS_IN>());
                jobs[1].setMerge(polys_L[hkcdBspImpl::POLYS_OUT], polys_R[hkcdBspImpl::POLYS_OUT], coplanar_OutL[hkcdBspImpl::POLYS_OUT], job.outputs<hkcdBspImpl::POLYS_OUT>());
                jobs[2].setMerge(tempMergedArrayA, tempMergedArrayB, nullPolys, job.outputs<hkcdBspImpl::POLYS_ON>());
                jobs[3].setMerge(polys_L[hkcdBspImpl::POLYS_ON], polys_R[hkcdBspImpl::POLYS_ON], coplanar_InL[hkcdBspImpl::POLYS_OUT], tempMergedArrayA);
                jobs[4].setMerge(coplanar_InL[hkcdBspImpl::POLYS_ON], coplanar_OutL[hkcdBspImpl::POLYS_ON], coplanar_OutL[hkcdBspImpl::POLYS_IN], tempMergedArrayB);
            }

            // Recurse on left
            if ( tempPolys[hkcdBspImpl::POLYS_IN].getSize() )
            {
                Job& jLeft = jobStack.expandOne();
                jLeft.setClassify(bspNode.m_left, polyStorage.allocArraySlot(), polys_L);
                if ( polyStorage.allocArrayStorage(jLeft.input(), tempPolys[hkcdBspImpl::POLYS_IN]).isFailure() )
                {
                    return HK_FAILURE;
                }
            }

            // Recurse on right
            if ( tempPolys[hkcdBspImpl::POLYS_OUT].getSize() )
            {
                Job& jRight = jobStack.expandOne();
                jRight.setClassify(bspNode.m_right, polyStorage.allocArraySlot(), polys_R);
                if ( polyStorage.allocArrayStorage(jRight.input(), tempPolys[hkcdBspImpl::POLYS_OUT]).isFailure() )
                {
                    return HK_FAILURE;
                }
            }

            // Recurse on coplanar
            if ( tempPolys[hkcdBspImpl::POLYS_ON].getSize() )
            {
                Job* jobs = jobStack.expandBy(3);
                jobs[0].setClassify(bspNode.m_right, coplanar_L[hkcdBspImpl::POLYS_IN], coplanar_InL);
                jobs[1].setClassify(bspNode.m_right, coplanar_L[hkcdBspImpl::POLYS_OUT], coplanar_OutL);
                jobs[2].setClassify(bspNode.m_left, polyStorage.allocArraySlot(), coplanar_L);
                if ( polyStorage.allocArrayStorage(jobs[2].input(), tempPolys[hkcdBspImpl::POLYS_ON]).isFailure() )
                {
                    return HK_FAILURE;
                }
            }
        }
        else
        {
            // Merge
            const ArrayId srcIdA    = job.inputs<hkcdBspImpl::POLYS_IN>();
            const ArrayId srcIdB    = job.inputs<hkcdBspImpl::POLYS_ON>();
            const ArrayId srcIdC    = job.inputs<hkcdBspImpl::POLYS_OUT>();

            const int numPolysA     = polyStorage.getArraySlot(srcIdA).m_size;
            const int numPolysB     = polyStorage.getArraySlot(srcIdB).m_size;
            const int numPolysC     = polyStorage.getArraySlot(srcIdC).m_size;
            const int numMerged     = numPolysA + numPolysB + numPolysC;

            hkArray<PolygonId>& mergedPolys = tempPolys[0];
            mergedPolys.setSize(numMerged);
            if ( numMerged )
            {
                PolygonId* HK_RESTRICT ptr = &mergedPolys[0];
                for (int si = numPolysA - 1; si >= 0; si--) {   ptr[si] = polyStorage.getPolygonId(srcIdA, si); }   ptr += numPolysA;
                for (int si = numPolysB - 1; si >= 0; si--) {   ptr[si] = polyStorage.getPolygonId(srcIdB, si); }   ptr += numPolysB;
                for (int si = numPolysC - 1; si >= 0; si--) {   ptr[si] = polyStorage.getPolygonId(srcIdC, si);}
            }

            // Free inputs
            if ( srcIdA != nullPolys )  {   polyStorage.freeArray(srcIdA);  }
            if ( srcIdB != nullPolys )  {   polyStorage.freeArray(srcIdB);  }
            if ( srcIdC != nullPolys )  {   polyStorage.freeArray(srcIdC);  }

            // Write output
            if ( numMerged )
            {
                if ( polyStorage.allocArrayStorage(job.output(), mergedPolys).isFailure() )
                {
                    return HK_FAILURE;
                }
            }
        }
    }

    // Copy results to the caller arrays
    {
        const int numPolysIn    = polyStorage.getArraySlot(aidPolys[hkcdBspImpl::POLYS_IN]).m_size;
        const int numPolysOut   = polyStorage.getArraySlot(aidPolys[hkcdBspImpl::POLYS_OUT]).m_size;
        const int numPolysOn    = polyStorage.getArraySlot(aidPolys[hkcdBspImpl::POLYS_ON]).m_size;

        insidePolygonsOut.setSize(numPolysIn);
        outsidePolygonsOut.setSize(numPolysOut);
        boundaryPolygonsOut.setSize(numPolysOn);

        for (int si = numPolysIn - 1; si >= 0; si--)    {   insidePolygonsOut[si]   = polyStorage.getPolygonId(aidPolys[hkcdBspImpl::POLYS_IN], si);    }
        for (int si = numPolysOut - 1; si >= 0; si--)   {   outsidePolygonsOut[si]  = polyStorage.getPolygonId(aidPolys[hkcdBspImpl::POLYS_OUT], si);   }
        for (int si = numPolysOn - 1; si >= 0; si--)    {   boundaryPolygonsOut[si] = polyStorage.getPolygonId(aidPolys[hkcdBspImpl::POLYS_ON], si);    }
    }

    // Try simplifying the results by replacing cut polygons with the original ones when possible
    HK_TIMER_BEGIN2(timerStream, "PolysRestore", HK_NULL);
    {
        typedef hkcdBspImpl::MatPoly    MatPoly;

        // Collect all material ids from the original geometry
        const int numOrigPolys = origPolygonsIn.getSize();
        hkArray<MatPoly> origMatToPoly;
        const int NB_TABLES = 3;
        PolygonId* inTables[NB_TABLES];
        MatPoly* outTables[NB_TABLES];
        hkArray<PolygonId>* outArrays[NB_TABLES];

        int inTableSizes[NB_TABLES];
        origMatToPoly.setSize(numOrigPolys);
        hkArray<MatPoly> inMatToPoly;                       hkArray<MatPoly> outMatToPoly;                  hkArray<MatPoly> boundMatToPoly;
        inTables[0] = boundaryPolygonsOut.begin();          inTables[1] = insidePolygonsOut.begin();        inTables[2] = outsidePolygonsOut.begin();
        inTableSizes[0] = boundaryPolygonsOut.getSize();    inTableSizes[1] = insidePolygonsOut.getSize();  inTableSizes[2] = outsidePolygonsOut.getSize();
        boundMatToPoly.setSize(inTableSizes[0]);            inMatToPoly.setSize(inTableSizes[1]);           outMatToPoly.setSize(inTableSizes[2]);
        outTables[0] = boundMatToPoly.begin();              outTables[1] = inMatToPoly.begin();             outTables[2] = outMatToPoly.begin();
        outArrays[0] = &boundaryPolygonsOut;                outArrays[1] = &insidePolygonsOut;              outArrays[2] = &outsidePolygonsOut;

        // Fill all the MatPoly structures
        for (int k = origPolygonsIn.getSize() - 1 ; k >= 0 ; k--)
        {
            PolygonId pId = origPolygonsIn[k];
            const Polygon& poly = polySoup.getPolygon(pId);
            origMatToPoly[k].m_polyId = pId;
            origMatToPoly[k].m_matId = poly.getMaterial();
        }

        for (int i = 0 ; i < NB_TABLES ; i++)
        {
            for (int k = 0 ; k < inTableSizes[i] ; k++)
            {
                outTables[i][k].m_polyId    = (*outArrays[i])[k];
                const Polygon& poly     = polySoup.getPolygon(outTables[i][k].m_polyId);
                outTables[i][k].m_matId = poly.getMaterial();
            }
        }

        boundaryPolygonsOut.clear();
        insidePolygonsOut.clear();
        outsidePolygonsOut.clear();

        // Sort all the results by material, and initialize variables for sweeping
        hkSort(origMatToPoly.begin(), origMatToPoly.getSize(), MatPoly::less);
        int cursorPos[NB_TABLES];
        int cursorOriginal = 0;
        bool tableFinished[NB_TABLES];
        bool sweepingFinished = true;
        int maxTableSize = 0;
        for (int i = 0 ; i < NB_TABLES ; i++)
        {
            hkSort(outTables[i], inTableSizes[i], MatPoly::less);
            cursorPos[i]        = 0;
            tableFinished[i]    = cursorPos[i] >= inTableSizes[i];
            sweepingFinished    = sweepingFinished && tableFinished[i];
            maxTableSize        = hkMath::max2(maxTableSize, inTableSizes[i]);
        }

        // Sweep the three arrays and try to replace group of polygons of the same material with original ones
        while ( !sweepingFinished )
        {
            // Get the smallest value at cursor index
            int ixMin = -1;
            for (int t = 0 ; t < NB_TABLES ; t++)
            {
                if ( !tableFinished[t] )
                {
                    ixMin = t;
                    break;
                }
            }

            for (int t = 0 ; t < NB_TABLES ; t++)
            {
                ixMin = ( !tableFinished[t] && (outTables[t][cursorPos[t]].m_matId.value() < outTables[ixMin][cursorPos[ixMin]].m_matId.value()) ) ? t : ixMin;
            }
            const Material cMat = outTables[ixMin][cursorPos[ixMin]].m_matId;

            // Move the cursor of the original array to reach the material
            while ( origMatToPoly[cursorOriginal].m_matId != cMat )
            {
                cursorOriginal++;
            }

            // See if the corresponding material is material isolated in only one part
            bool isolated       = true;
            for (int t = 0 ; t < NB_TABLES ; t++)
            {
                if ( tableFinished[t] )
                {
                    continue;
                }

                if ( (t != ixMin) && (cMat == outTables[t][cursorPos[t]].m_matId) )
                {
                    // found another array containing same material: material is not isolated
                    isolated = false;
                    break;
                }
            }

            if ( isolated )
            {
                // isolated material, all the cut polygons can be replaced by the original ones!
                while ( (cursorOriginal < numOrigPolys) && (origMatToPoly[cursorOriginal].m_matId == cMat) )
                {
                    outArrays[ixMin]->pushBack(origMatToPoly[cursorOriginal].m_polyId);
                    cursorOriginal++;
                }
            }

            // Update all cursors and finish state
            sweepingFinished = true;
            for (int t = 0 ; t < NB_TABLES ; t++)
            {
                if ( tableFinished[t] )
                {
                    continue;
                }

                while ( (cursorPos[t] < inTableSizes[t]) && (outTables[t][cursorPos[t]].m_matId == cMat) )
                {
                    if ( !isolated )
                    {
                        outArrays[t]->pushBack(outTables[t][cursorPos[t]].m_polyId);
                    }
                    cursorPos[t]++;
                }
                tableFinished[t]    = cursorPos[t] >= inTableSizes[t];
                sweepingFinished    = sweepingFinished && tableFinished[t];
            }
        }
    }
    HK_TIMER_END(); // "PolysRestore"
    return HK_SUCCESS;
}

//
//  Special case of the classifies, where only the inside OR boundary polys are needed

void hkcdPlanarSolid::classifyInsideOrBoundaryPolygons(hkcdPlanarGeometry& polySoup, const hkArray<PolygonId>& polygonsIn, hkArray<PolygonId>& insideOrBoundPolyIds, _Inout_opt_ ArrayMgr* arrayMgr)
{
    hkArray< hkArray<PlaneId> > solidCellsPlaneIds;

    // First, get all the solid nodes
    int biggestArrayId = -1;
    int numPlanesinBiggestArray = -1;
    for (int k = m_nodes->getCapacity() - 1; k >= 0; k--)
    {
        NodeId nodeId(k);
        Node* node = &accessNode(nodeId);
        if ( node->isAllocated() )
        {
            if ( node->m_type == NODE_TYPE_IN )
            {
                hkArray<PlaneId>& solidCellPlaneIds = solidCellsPlaneIds.expandOne();
                // We got one, collect all the planes that form the convex solid cell
                int nbPlanes = 0;
                while ( node->m_parent.isValid() )
                {
                    node = &accessNode(node->m_parent);
                    nbPlanes++;
                }
                if ( nbPlanes > numPlanesinBiggestArray )
                {
                    numPlanesinBiggestArray = nbPlanes;
                    biggestArrayId          = solidCellsPlaneIds.getSize() - 1;
                }
                solidCellPlaneIds.reserve(nbPlanes);
                node = &accessNode(nodeId);
                NodeId prevNodeId = nodeId;
                while ( node->m_parent.isValid() )
                {
                    const NodeId parentId   = node->m_parent;
                    node                    = &accessNode(parentId);
                    if ( node->m_left == prevNodeId )
                    {
                        solidCellPlaneIds.pushBack(node->m_planeId);
                    }
                    else
                    {
                        solidCellPlaneIds.pushBack(hkcdPlanarGeometryPrimitives::getOppositePlaneId(node->m_planeId));
                    }
                    prevNodeId = parentId;
                }
            }
        }
    }

    const int numSolidCells = solidCellsPlaneIds.getSize();
    if ( numSolidCells == 0 )
    {
        return;
    }

    // Compute a common plane path
    if ( numSolidCells > 1 )
    {
        hkArray<PlaneId> commonPlanePath;
        commonPlanePath.append(solidCellsPlaneIds[biggestArrayId]);
        for (int i = 0 ; i < numSolidCells ; i++)
        {
            if ( i == biggestArrayId )      continue;
            // Find the common planes (starting from the end)
            hkArray<PlaneId>& currPlaneArray = solidCellsPlaneIds[i];
            int numCommonPlanes = 0;
            for (int p = currPlaneArray.getSize() - 1, pRef = commonPlanePath.getSize() - 1 ; p >= 0 && pRef >= 0 ; p--, pRef--)
            {
                if ( currPlaneArray[p].value() == commonPlanePath[p].value() )
                {
                    numCommonPlanes++;
                }
                else
                {
                    break;
                }
            }
            commonPlanePath.setSize(numCommonPlanes);
        }
        solidCellsPlaneIds.pushBack(commonPlanePath);
    }

    // Classify all the polys against the common plane path, then against all solid cells
    hkArray<PolygonId> polysOnCommonPath, currInsidePolyIds, nextInsidePolyIds;
    insideOrBoundPolyIds.setSize(0);
    insideOrBoundPolyIds.reserve(polygonsIn.getSize()*2);
    nextInsidePolyIds.reserve(polygonsIn.getSize()*2);
    currInsidePolyIds.append(polygonsIn);

    for (int sc = solidCellsPlaneIds.getSize() - 1 ; sc >= 0 ; sc--)
    {
        hkArray<PlaneId>& currPlaneArray = solidCellsPlaneIds[sc];
        if ( sc != solidCellsPlaneIds.getSize() - 1 )
        {
            currInsidePolyIds.setSize(0);
            currInsidePolyIds.append(polysOnCommonPath);
        }
        for (int p = currPlaneArray.getSize() - 1 ; p >= 0 ; p--)
        {
            for (int polyIdx = currInsidePolyIds.getSize() - 1 ; polyIdx >= 0 ; polyIdx--)
            {
                const PolygonId polyId  = currInsidePolyIds[polyIdx];
                const Orientation ori   = polySoup.classify(polyId, currPlaneArray[p]);

                switch ( ori )
                {
                case hkcdPlanarGeometryPredicates::IN_FRONT_OF: break;
                case hkcdPlanarGeometryPredicates::BEHIND:      nextInsidePolyIds.pushBack(polyId); break;
                case hkcdPlanarGeometryPredicates::INTERSECT:
                    {
                        // We need to split the polygon into 2 parts
                        PolygonId splitInside, splitOutside;
                        polySoup.split(polyId, currPlaneArray[p], splitInside, splitOutside);

                        if ( splitInside.isValid() && splitOutside.isValid() )
                        {
                            nextInsidePolyIds.pushBack(splitInside);
                        }
                    }
                    break;
                default:
                    {
                        HK_ASSERT_NO_MSG(0x53da54bd, ori == hkcdPlanarGeometryPredicates::ON_PLANE);
                        nextInsidePolyIds.pushBack(polyId);     // Add on boundary as well
                    }
                    break;
                }
            }

            // Swap poly ids
            currInsidePolyIds.swap(nextInsidePolyIds);
            nextInsidePolyIds.setSize(0);
        }

        // Add the polys to the results
        if ( sc != solidCellsPlaneIds.getSize() - 1 || solidCellsPlaneIds.getSize() == 1 )
        {
            insideOrBoundPolyIds.append(currInsidePolyIds);
        }
        else
        {
            polysOnCommonPath.append(currInsidePolyIds);
        }
    }

}

//
//  Returns the maximum depth of the tree

int hkcdPlanarSolid::computeMaxDepth() const
{
    int maxDepth = 0;
    for (int k = m_nodes->getCapacity() - 1; k >= 0; k--)
    {
        NodeId leafId           (k);
        const Node& leafNode    = getNode(leafId);

        if ( leafNode.isAllocated() )
        {
            if ( (leafNode.m_type != NODE_TYPE_INTERNAL) ||
                !leafNode.m_left.isValid() || (getNode(leafNode.m_left).m_type == NODE_TYPE_INTERNAL) ||
                !leafNode.m_right.isValid() || (getNode(leafNode.m_right).m_type == NODE_TYPE_INTERNAL) )
            {
                continue;
            }

            // This is a leaf, compute depth
            int depth = 0;
            while ( leafId.isValid() )
            {
                leafId = getNode(leafId).m_parent;
                depth++;
            }
            maxDepth = hkMath::max2(depth, maxDepth);
        }
    }

    return maxDepth;
}

//
//  Computes the number of leaf nodes

int hkcdPlanarSolid::computeNumLeafNodes() const
{
    int sum = 0;
    for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
    {
        const NodeId nodeId(ni);
        const Node& bspNode = getNode(nodeId);
        if ( bspNode.isAllocated() && (bspNode.m_type != NODE_TYPE_INTERNAL) )
        {
            sum++;
        }
    }

    return sum;
}

//
//  Computes the number of nodes with the specified label

int hkcdPlanarSolid::computeNumNodesWithLabel(hkUint32 label) const
{
    int sum = 0;
    for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
    {
        const NodeId nodeId(ni);
        const Node& bspNode = getNode(nodeId);
        if ( bspNode.isAllocated() && (bspNode.m_type == label) )
        {
            sum++;
        }
    }

    return sum;
}

//
//  Invert the node labels from the specified node

void hkcdPlanarSolid::invertNodeLabels(NodeId rootNodeId)
{
    const NodeTypes flipLabel[] = { hkcdPlanarSolid::NODE_TYPE_INTERNAL, hkcdPlanarSolid::NODE_TYPE_OUT,
                                    hkcdPlanarSolid::NODE_TYPE_IN, hkcdPlanarSolid::NODE_TYPE_UNKNOWN };

    hkArray<NodeId> stack;
    stack.pushBack(rootNodeId);
    while ( !stack.isEmpty() )
    {
        // Get current node
        NodeId cNodeId = stack[0];
        stack.removeAt(0);
        Node& cNode = accessNode(cNodeId);

        if ( cNode.m_type == hkcdPlanarSolid::NODE_TYPE_IN || cNode.m_type == hkcdPlanarSolid::NODE_TYPE_OUT )
        {
            cNode.m_type = flipLabel[cNode.m_type];
        }
        else if ( cNode.m_type != hkcdPlanarSolid::NODE_TYPE_UNKNOWN )
        {
            // Flip material
            if ( cNode.m_material.isValid() )
            {
                cNode.m_material.setFlags(cNode.m_material.getFlags() ^ Material::IS_FLIPPED);
            }

            // Add children to the stack
            stack.pushBack(cNode.m_left);
            stack.pushBack(cNode.m_right);
        }
    }
}

//
//  Return an array of plane ids that hold the valid material of the tree

void hkcdPlanarSolid::getPlaneIdsWithValidMaterial(hkArray<PlaneId>& planeIdsOut) const
{
    for (int ni = m_nodes->getCapacity() - 1; ni >= 0; ni--)
    {
        const NodeId nodeId(ni);
        const Node& bspNode = getNode(nodeId);
        if ( bspNode.isAllocated() && bspNode.m_material.isValid() )
        {
            planeIdsOut.pushBack(PlaneId(bspNode.m_planeId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG)));
        }
    }

    // Sort and remove duplicates
    hkSort(planeIdsOut.begin(), planeIdsOut.getSize());
    planeIdsOut.setSize(hkAlgorithm::removeDuplicatesFromSortedList(planeIdsOut.begin(), planeIdsOut.getSize()));
}

//
//  Debug. Prints all the tree data

void hkcdPlanarSolid::dbgPrint() const
{
    for (int ni = 0; ni < m_nodes->getCapacity(); ni++)
    {
        const NodeId nodeId (ni);
        const Node& node    = getNode(nodeId);
        const int planeIdx  = node.m_planeId.isValid() ? ((node.m_planeId.valueUnchecked() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG)) - 6) : node.m_planeId.valueUnchecked();
        const PlaneId pid   = node.m_planeId.isValid() ? PlaneId((node.m_planeId.valueUnchecked() & hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG) | planeIdx) : node.m_planeId;

        Log_Info( "Node {}. Parent {}, Left {}, Right {}, PlaneId = {} ({}), Data = {}, Type = 0x{:X8}",
            ni, node.m_parent.valueUnchecked(), node.m_left.valueUnchecked(), node.m_right.valueUnchecked(),
            pid.valueUnchecked(), planeIdx,
            node.m_data, node.m_type);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
