// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

//
//  Gets / Sets the fixed-precision normal

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::getExactNormal(hkInt64Vector4& iNormalOut) const
{
    const int nShr = 64 - NumBitsPlaneNormal::NUM_BITS;
    iNormalOut.setShiftRight<nShr, HK_VALUE_SIGNED>(m_iEqn);
}

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::setExactNormal(hkInt64Vector4Parameter iNormalIn)
{
    // Make sure the .w component is zero
    hkInt64Vector4 vN = iNormalIn;
    vN.zeroComponent<3>();
    const int nShl = 64 - NumBitsPlaneNormal::NUM_BITS;
    m_iEqn.setShiftLeft<nShl>(vN);
}

//
//  Gets / Sets the fixed-precision offset

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::getExactOffset(hkInt128& iOffsetOut) const
{
    const int nSh = 128 - NumBitsPlaneOffset::NUM_BITS;
    hkInt128 iO;    iO.setDoubleWord(0, m_iEqn.getComponent<3>()); iO.setDoubleWord(1, m_iEqn.getComponent<2>());
    iO.setShiftLeft(iO, nSh);   // At this point, we have the sign bit in position 127
    iOffsetOut.setShiftRightSigned(iO, nSh);
}

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::setExactOffset(const hkInt128& iOffsetIn)
{
    // Zero the top bits so we can store the offset in the same vector
    const int nSh = 128 - NumBitsPlaneOffset::NUM_BITS;
    hkInt128 iO;    iO.setShiftLeft(iOffsetIn, nSh);
                        iO.setShiftRight(iO, nSh);
    hkInt128 zw;    zw.setDoubleWord(0, m_iEqn.getComponent<3>()); zw.setDoubleWord(1, m_iEqn.getComponent<2>());
                        iO.setOr(zw, iO);
    m_iEqn.setComponent<2>(iO.getDoubleWord(1));
    m_iEqn.setComponent<3>(iO.getDoubleWord(0));
}

//
//  Sets the exact plane equation

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::setExactEquation(hkInt64Vector4Parameter iNormalIn, const hkInt128& iOffsetIn, bool simplifyEquation)
{
    setExactNormal(iNormalIn);
    setExactOffset(iOffsetIn);

#if ( ENABLE_HKCD_PLANAR_GEOMETRY_DEBUG_CHECKS )
    {
        hkInt64Vector4 iN;  getExactNormal(iN);
        hkInt128 iO;    getExactOffset(iO);
        HK_ASSERT_NO_MSG(0x6a30edf2, iN.equal(iNormalIn).allAreSet<hkVector4ComparisonMask::MASK_XYZ>() && iO.equal(iOffsetIn));
    }
#endif

    if ( simplifyEquation )
    {
        simplify();
    }
    computeApproxEquation();
}

//
//  Computes a plane in the opposite direction of the given plane

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::setOpposite(const Plane& srcPlane)
{
    hkInt64Vector4 iNrm;        srcPlane.getExactNormal(iNrm);
                                iNrm.setNeg<4>(iNrm);
    hkInt128 iOffset;       srcPlane.getExactOffset(iOffset);
                                iOffset.setNeg(iOffset);

    setExactNormal(iNrm);
    setExactOffset(iOffset);
    accessApproxEquation().setNeg<4>(srcPlane.getApproxEquation());
}

//
//  Compares two planes for equality

HK_INLINE hkBool32 hkcdPlanarGeometryPrimitives::Plane::isEqual(const Plane& other) const
{
    const hkVector4fComparison cmp = m_iEqn.equal(other.m_iEqn).horizontalAnd<4>();

#if ( ENABLE_HKCD_PLANAR_GEOMETRY_DEBUG_CHECKS )
    {
        hkInt64Vector4 nA;  getExactNormal(nA);
        hkInt64Vector4 nB;  other.getExactNormal(nB);
        hkInt128 oA;    getExactOffset(oA);
        hkInt128 oB;    other.getExactOffset(oB);

        hkVector4Comparison cmpN    = nA.equal(nB).horizontalAnd<3>();
        hkVector4Comparison cmpO;   cmpO.set(oA.equal(oB) ? hkVector4ComparisonMask::MASK_XYZW : hkVector4ComparisonMask::MASK_NONE);
                                    cmpO.setAnd(cmpN, cmpO);
                                    cmpO.setXor(cmpO, cmp);
        HK_ASSERT_NO_MSG(0x3c4d65d8, !cmpO.anyIsSet());
    }
#endif

    return cmp.anyIsSet();
}

//
//  Gets / Sets the floating-point equation

HK_INLINE const hkVector4d& hkcdPlanarGeometryPrimitives::Plane::getApproxEquation() const
{
    return m_dEqn;
}

HK_INLINE hkVector4d& hkcdPlanarGeometryPrimitives::Plane::accessApproxEquation()
{
    return m_dEqn;
}

//
//  Returns the Id of the opposite plane

HK_INLINE hkcdPlanarGeometryPrimitives::PlaneId HK_CALL hkcdPlanarGeometryPrimitives::getOppositePlaneId(PlaneId planeId)
{
    const int pid   = planeId.value() & (~FLIPPED_PLANE_FLAG);
    const int flip  = (planeId.value() ^ FLIPPED_PLANE_FLAG) & FLIPPED_PLANE_FLAG;
    return PlaneId(pid | flip);
}

//
//  Computes the plane passing through 3 vertices

HK_INLINE hkResult HK_CALL hkcdPlanarGeometryPrimitives::computePlaneEquation(hkIntVectorParameter vA, hkIntVectorParameter vB, hkIntVectorParameter vC, hkInt64Vector4& normalOut, hkInt128& offsetOut)
{
    // Compute edges
    hkIntVector eAB;    eAB.setSubU32(vB, vA);
    hkIntVector eAC;    eAC.setSubS32(vC, vA);

    // Compute normal & plane offset
    normalOut.setCross(eAB, eAC);
    if ( normalOut.equalZero().allAreSet<hkVector4ComparisonMask::MASK_XYZ>() )
    {
        return HK_FAILURE;
    }

    offsetOut.setNeg(normalOut.dot<3>(vA));
    return HK_SUCCESS;
}

//
//  Returns true if the plane Ids are coplanar

HK_INLINE bool HK_CALL hkcdPlanarGeometryPrimitives::coplanarPlaneIds(PlaneId planeIdA, PlaneId planeIdB)
{
    const int pidA  = planeIdA.value() & (~FLIPPED_PLANE_FLAG);
    const int pidB  = planeIdB.value() & (~FLIPPED_PLANE_FLAG);
    return pidA == pidB;
}

//
//  Returns true if the plane Ids are coplanar. Note that this only compares the Ids and does not evaluate any geometric predicate.
//  The mesh must be built such that all planes are distinct!

HK_INLINE bool HK_CALL hkcdPlanarGeometryPrimitives::sameOrientationPlaneIds(PlaneId planeIdA, PlaneId planeIdB)
{
    return (planeIdA == planeIdB);
}

//
//  Simplifies the fixed-precision equation so that gcd(nx, ny, nz, offset) = 1

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::simplify()
{
    // Get normal and offset
    hkInt64Vector4 iNrm;        getExactNormal(iNrm);
    hkInt128 iOff;      getExactOffset(iOff);

    // Compute gcd
    hkInt64Vector4 iAbsNrm;     iAbsNrm.setAbs(iNrm);
    const hkUint64 gcdXY        = hkMath::greatestCommonDivisor<hkUint64>(iAbsNrm.getComponent<0>(), iAbsNrm.getComponent<1>());
    hkUint64 gcdXYZ             = hkMath::greatestCommonDivisor<hkUint64>(gcdXY, iAbsNrm.getComponent<2>());
    hkInt128 iAbsOff;       iAbsOff.setAbs(iOff);
    hkInt128 gcdXYZO;       gcdXYZO.setFromUint64(gcdXYZ);
                                gcdXYZO.setGreatestCommonDivisor(iAbsOff, gcdXYZO);

    // Divide everything by gcd
    iAbsOff.setUnsignedDiv(iAbsOff, gcdXYZO);
    hkInt64Vector4 vGcd;    vGcd.setComponent<3>(gcdXYZO.getDoubleWord(0));
                            vGcd.setPermutation<hkVectorPermutation::WWWW>(vGcd);
    iAbsNrm.setUnsignedDiv(iAbsNrm, vGcd);

    // Set signs and store
    hkInt64Vector4 iScaledNrm;  iScaledNrm.setFlipSign(iAbsNrm, iNrm.lessZero());
    hkInt128 iScaledOff;    iScaledOff.setFlipSign(iAbsOff, iOff.lessZero());
    setExactNormal(iScaledNrm);
    setExactOffset(iScaledOff);
}

//
//  Recomputes the floating-point equation to match the fixed-precision one within a known eps

HK_INLINE void hkcdPlanarGeometryPrimitives::Plane::computeApproxEquation()
{
    // We want to shift everything so we have at least a bit in the highest allowed position
    hkInt64Vector4 iNrm;        getExactNormal(iNrm);
    hkInt128 iOff;      getExactOffset(iOff);
    hkInt64Vector4 iAbsNrm;     iAbsNrm.setAbs(iNrm);
    hkInt128 iAbsOff;       iAbsOff.setAbs(iOff);

    // At least one of these is non-zero
    hkIntVector msb;    iAbsNrm.countLeadingZeros<3>(msb);
    const int msbXYZ    = 63 - msb.horizontalMinS32<3>();
    const int msbW      = 127 - iAbsOff.countLeadingZeros();
    const int shlXYZ    = (hkcdPlanarGeometryPrimitives::NumBitsPlaneNormal::NUM_BITS - 1 - 1) - msbXYZ;
    const int shlW      = (hkcdPlanarGeometryPrimitives::NumBitsPlaneOffset::NUM_BITS - 1 - 1) - msbW;
    const int shlXYZW   = hkMath::min2(shlXYZ, shlW);

    // Shift normal and offset so we have at least one MSB != 0.
    iAbsNrm.setShiftLeft(iAbsNrm, shlXYZW);
    iAbsOff.setShiftLeft(iAbsOff, shlXYZW);

    // Convert to double
    {
        const int shl   = 4;
        const int shrN  = shl + NumBitsPlaneNormal::NUM_BITS - 1 - NumBitsMantissaD::NUM_BITS;
        const int shrO  = shl + NumBitsPlaneOffset::NUM_BITS - 1 - NumBitsMantissaD::NUM_BITS;

        hkInt64Vector4 nxyzw;       nxyzw.setShiftLeft<shl>(iAbsNrm);
                                    nxyzw.setShiftRight<shrN>(nxyzw);
        hkInt128 nw;            nw.setShiftLeft(iAbsOff, shl);
                                    nw.setShiftRight(nw, shrO);
                                    nxyzw.setComponent<3>(nw.getDoubleWord(0));

        hkVector4d dEqn;            nxyzw.convertS64ToF64(dEqn);
        const double scaleN         = 1.0 / (double)(hkInt64)(1LL << (hkInt64)NumBitsMantissaD::NUM_BITS);  // 1 / 2^53 = 1 / 9007199254740992
        const double scaleO         = 1.0 / (double)(hkInt64)(1LL << (hkInt64)(NumBitsMantissaD::NUM_BITS + NumBitsPlaneNormal::NUM_BITS - NumBitsPlaneOffset::NUM_BITS));  // 1 / 2^29 = 1 / 2^(25 - (53 + 1)) =  1 / 536870912
        hkVector4d dScale;          dScale.set(scaleN, scaleN, scaleN, scaleO);
                                    dEqn.mul(dScale);

        // Set proper signs
        hkVector4dComparison iOffLessZero;      iOffLessZero.set(iOff.lessZero() ? hkVector4ComparisonMask::MASK_XYZW : hkVector4ComparisonMask::MASK_NONE);
        hkVector4d dEqnW;       dEqnW.setFlipSign(dEqn, iOffLessZero);
        hkVector4d dEqnN;       dEqnN.setFlipSign(dEqn, iNrm.lessZero<hkVector4dComparison>());
        accessApproxEquation().setXYZ_W(dEqnN, dEqnW);
    }
}

//
//  Sets the position of this vertex

HK_INLINE void hkcdPlanarGeometryPrimitives::Vertex::set(hkIntVectorParameter iPos)
{
    m_iPos = iPos;
    accessApproxPosition().set((hkDouble64)iPos.getComponent<0>(), (hkDouble64)iPos.getComponent<1>(), (hkDouble64)iPos.getComponent<2>(), 0.0);
}

//
//  Returns the fixed precision position

HK_INLINE const hkIntVector& hkcdPlanarGeometryPrimitives::Vertex::getExactPosition() const         {   return m_iPos;  }

//
//  Returns the approximate position

HK_INLINE const hkVector4d& hkcdPlanarGeometryPrimitives::Vertex::getApproxPosition() const
{
    return m_dPos;
}

HK_INLINE hkVector4d& hkcdPlanarGeometryPrimitives::Vertex::accessApproxPosition()
{
    return m_dPos;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
