// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Geometry/Collide/DataStructures/Planar/Primitives/hkcdPlanarGeometryPrimitives.h>
#include <Geometry/Collide/DataStructures/Planar/Utils/hkcdMathErrorBoundsCalculator.h>

/// Exact geometric predicates
class HK_EXPORT_COMMON hkcdPlanarGeometryPredicates
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdPlanarGeometryPredicates);

        // Types
        typedef hkcdPlanarGeometryPrimitives::Plane     Plane;
        typedef const Plane&                            PlaneParam;
        typedef hkcdMathErrorBoundsCalculator::Vector   ApproxVertex;

    public:

        /// Orientation types
        enum Orientation
        {
            ON_PLANE            = 0,
            IN_FRONT_OF         = 1,
            BEHIND              = 2,
            INTERSECT           = 3,
        };

        /// Coplanarity types
        enum Coplanarity
        {
            NOT_COPLANAR    = 0,
            COINCIDENT      = 1,
            OPPOSITE        = 2,
        };

        /// Winding types
        enum Winding
        {
            WINDING_CW          = -1,   ///< The winding is clockwise, Dot[planeNormal, Cross[AB, AC]] is negative
            WINDING_COLLINEAR   = 0,    ///< No winding, the points are collinear!
            WINDING_CCW         = 1,    ///< The winding is counter-clockwise, Dot[planeNormal, Cross[AB, AC]] is positive
            WINDING_UNKNOWN     = 3,    ///< Special code returned by the floating-point function to trigger the execution of the fixed-precision predicate
        };

        /// Orientation cache
        template <int N>
        struct OrientationCacheBase
        {
            public:

                HK_DECLARE_CLASS(OrientationCacheBase, New);

            public:

                /// Constructor. Initializes the multiplier with 4 large prime numbers
                HK_INLINE OrientationCacheBase();

#ifdef HK_DEBUG

                /// Gets the ratio of cache hit
                HK_INLINE hkReal getCacheHitRatio() const;

                /// Resets cache hit counters
                HK_INLINE void resetCacheHitCounters();
#endif

            public:

                hkArray<hkIntVector> m_data;
                hkIntVector m_mul;

                HK_DEBUG_ONLY_MEMBER(int, m_numCacheHits);
                HK_DEBUG_ONLY_MEMBER(int, m_numCacheAccesses);
        };

    public:

        /// Returns true if the given planes are coplanar (i.e. coincident)
        static inline Coplanarity HK_CALL coplanar(PlaneParam planeA, PlaneParam planeB);

        /// Returns true if the vertex defined by three planes (ptPlaneA, ptPlaneB, ptPlaneC) is coplanar to all the given test planes
        static hkBool32 HK_CALL coplanar(PlaneParam ptPlaneA, PlaneParam ptPlaneB, PlaneParam ptPlaneC, _In_reads_(numTestPlanes) const Plane* HK_RESTRICT testPlanes, int numTestPlanes);
        static hkBool32 HK_CALL coplanar(hkVector4dParameter p, PlaneParam ptPlaneA, PlaneParam ptPlaneB, PlaneParam ptPlaneC, _In_reads_(numTestPlanes) const Plane* HK_RESTRICT testPlanes, int numTestPlanes);

        /// Tests for same orientation. Should be preceded by a coplanarity test
        static HK_INLINE int HK_CALL sameOrientation(PlaneParam planeA, PlaneParam planeB);

        /// Computes the orientation of the point (ptPlaneA, ptPlaneB, ptPlaneC) with respect to the planeD
        static HK_INLINE Orientation HK_CALL approximateOrientation(PlaneParam ptPlaneA, PlaneParam ptPlaneB, PlaneParam ptPlaneC, PlaneParam planeD);
        static HK_INLINE Orientation HK_CALL approximateOrientation(hkVector4dParameter p, PlaneParam plane);

        /// Computes the orientation of the point v with respect to the plane p
        static inline Orientation HK_CALL orientation(hkIntVectorParameter v, PlaneParam p);

        /// Returns true if the edge determined by the intersection of planes A and B is contained on plane C
        static hkBool32 HK_CALL edgeOnPlane(PlaneParam edgePlaneA, PlaneParam edgePlaneB, PlaneParam planeC);

        /// Computes the vector of determinants [PX, PY, PZ, PW], where:
        ///             | ax ay az |            | aw ay az |            | ax aw az |            | ax ay aw |
        ///     PW =    | bx by bz |,   PX = -  | bw by bz |,   PY = -  | bx bw bz |,   PZ = -  | bx by bw |
        ///             | cx cy cz |            | cw cy cz |            | cx cw cz |            | cx cy cw |
        static inline void HK_CALL computeIntersectionDeterminants(hkVector4dParameter planeEqnA, hkVector4dParameter planeEqnB, hkVector4dParameter planeEqnC, ApproxVertex& determinantsOut);
        static void HK_CALL computeIntersectionDeterminants(const Plane (&planes)[3], _Inout_opt_ hkInt256* detX, _Inout_opt_ hkInt256* detY, _Inout_opt_ hkInt256* detZ, _Inout_opt_ hkInt256* detW);

        // Approximates the intersection (i.e. in fixed coordinates) of the 3 given planes
        static inline void HK_CALL approximateIntersection(const Plane (&planes)[3], hkIntVector& intersectionOut);

        // Approximates the intersection (in floating point precision) of the 3 given planes. Faster but less accurate then the floating point version
        static inline void HK_CALL approximateIntersectionFast(const Plane (&planes)[3], hkVector4d& intersectionOut);

        /// Computes an approximate direction for the edge resulting from the intersection of the two given planes
        static void HK_CALL approximateEdgeDirection(PlaneParam planeA, PlaneParam planeB, hkIntVector& edgeDirectionOut);
        static void HK_CALL approximateEdgeDirectionFast(PlaneParam planeA, PlaneParam planeB, hkIntVector& edgeDirectionOut);

        /// Computes the winding of the 3 given vertices w.r.t. the given triangle normal.
        static HK_INLINE Winding HK_CALL triangleWinding(const Plane (&planesA)[3], const Plane (&planesB)[3], const Plane (&planesC)[3], PlaneParam supportPlane);

    protected:

        // Computes the Det3. Returns true if the result is trustworthy
        static HK_INLINE hkBool32 computeApproxDet3(const Plane& planeA, const Plane& planeB, const Plane& planeC, hkVector4dComparison& det3LessZero, hkVector4dComparison& det3EqualZero);
        static HK_INLINE void computeExactDet3(const Plane& planeA, const Plane& planeB, const Plane& planeC, hkVector4dComparison& det3LessZero, hkVector4dComparison& det3EqualZero);

        /// Computes the Det4. Returns true if the result is trustworthy
        static HK_INLINE hkBool32 computeApproxDet4(const Plane& planeA, const Plane& planeB, const Plane& planeC, const Plane& planeD, hkVector4dComparison& det4LessZero, hkVector4dComparison& det4EqualZero);
        static HK_INLINE void computeExactDet4(const Plane& planeA, const Plane& planeB, const Plane& planeC, const Plane& planeD, hkVector4dComparison& det4LessZero, hkVector4dComparison& det4EqualZero);

        static HK_INLINE int HK_CALL mulSigns(hkVector4dComparisonParameter negA, hkVector4dComparisonParameter zeroA, hkVector4dComparisonParameter negB, hkVector4dComparisonParameter zeroB);

        /// Estimates the winding of the 3 given vertices w.r.t. the given triangle normal (floating-point).
        static Winding HK_CALL estimateTriangleWinding(const Plane (&planesA)[3], const Plane (&planesB)[3], const Plane (&planesC)[3], const Plane& supportPlane, _In_range_(0, 2) int axisX, _In_range_(0, 2) int axisY, _In_range_(0, 2) int axisZ);

        /// Computes the triangle winding using fixed precision
        static Winding HK_CALL computeExactTriangleWinding(const Plane (&planesA)[3], const Plane (&planesB)[3], const Plane (&planesC)[3], const Plane& supportPlane, _In_range_(0, 2) int axisX, _In_range_(0, 2) int axisY, _In_range_(0, 2) int axisZ);

    public:

        /// Computes the orientation of the point p defined by (ptPlaneA, ptPlaneB, ptPlaneC) with respect to the planeD, using an orientation cache
        template <int N>
        static inline Orientation HK_CALL orientation(hkVector4dParameter p, PlaneParam ptPlaneA, PlaneParam ptPlaneB, PlaneParam ptPlaneC, PlaneParam planeD, hkIntVectorParameter planeIds, _Inout_opt_ OrientationCacheBase<N>* orientationCache);

        /// Computes the orientation of the point (ptPlaneA, ptPlaneB, ptPlaneC) with respect to the planeD, using an orientation cache
        template <int N>
        static inline Orientation HK_CALL orientation(PlaneParam ptPlaneA, PlaneParam ptPlaneB, PlaneParam ptPlaneC, PlaneParam planeD, hkIntVectorParameter planeIds, _Inout_opt_ OrientationCacheBase<N>* orientationCache);

    public:

        // Statistics
#ifdef HK_DEBUG
        static hkUint32 s_numFastCalls;
        static hkUint32 s_numFastCallsFailed;

        static hkUint32 s_numApproxCalls;   ///< Number of times when a call was resolved using floating point arithmetic
        static hkUint32 s_numExactCalls;    ///< Number of times when a call was resolved using exact arithmetic

        static hkUint32 s_numApproxDet3;
        static hkUint32 s_numExactDet3;

        static hkUint32 s_numApproxDet4;
        static hkUint32 s_numExactDet4;
#endif
};

#include <Geometry/Collide/DataStructures/Planar/Predicates/hkcdPlanarGeometryPredicates.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
