// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/DataStructures/Planar/Memory/hkcdPlanarGeometryPlanesCollection.h>
#include <Geometry/Collide/DataStructures/Planar/Utils/hkcdPlanarGeometryWeldUtil.h>

//
//  Empty constructor

hkcdPlanarGeometryPlanesCollection::hkcdPlanarGeometryPlanesCollection()
:   hkReferencedObject()
,   m_cache(HK_NULL)
,   m_criticalAccess(new hkCriticalSection())
{
    m_offsetAndScale.setZero();

    // Add boundary planes
    createBoundaryPlanes();
}

//
//  Default constructor

hkcdPlanarGeometryPlanesCollection::hkcdPlanarGeometryPlanesCollection(const hkAabb& origMaxAabb, int initialPlaneCapacity)
:   hkReferencedObject()
,   m_cache(HK_NULL)
,   m_criticalAccess(new hkCriticalSection())
{
    // Compute factors for float to fixed precision conversion
    hkAabb maxAabb              = origMaxAabb;
    maxAabb.expandBy(hkSimdReal_1);                                 // We want to be able to add / subtract 1 from the bounds and still get a valid fixed precision coordinate
    hkVector4 vExtents;         maxAabb.getExtents(vExtents);
    hkSimdReal maxExtent;       maxExtent.setAdd(vExtents.horizontalMax<3>(), hkSimdReal_Inv2); // Round to nearest

    // We want the smallest power of 2 larger than the max extent
    hkInt32 intValue;           maxExtent.storeSaturateInt32(&intValue);
    intValue++;
    if ( intValue < 1 )         intValue = 1;
    const int logValue          = 32 - hkMath::countLeadingZeros(intValue);

    HK_ASSERT_NO_MSG(0x7803bf08, logValue <= (hkcdPlanarGeometryPrimitives::NumBitsVertex::NUM_BITS - 1));
    hkSimdReal intFromFloatScale;   intFromFloatScale.setFromInt32(1 << (hkcdPlanarGeometryPrimitives::NumBitsVertex::NUM_BITS - 1 - logValue));

    // Store conversion offset and scale
    m_offsetAndScale.setXYZ_W(maxAabb.m_min, intFromFloatScale);

    // Pre-allocate planes
    if ( initialPlaneCapacity )
    {
        m_planes.reserve(initialPlaneCapacity + 6);
    }

    // Add boundary planes
    createBoundaryPlanes();
}

//
//  Copy constructor

hkcdPlanarGeometryPlanesCollection::hkcdPlanarGeometryPlanesCollection(const hkcdPlanarGeometryPlanesCollection& other)
:   hkReferencedObject()
,   m_offsetAndScale(other.m_offsetAndScale)
,   m_cache(HK_NULL)
,   m_criticalAccess(new hkCriticalSection())
{
    m_planes.append(other.m_planes);
}

void hkcdPlanarGeometryPlanesCollection::afterReflectNew()
{
    // Compute the approx. plane equations
    for (int p = m_planes.getSize() - 1 ; p >= 0 ; p--)
    {
        m_planes[p].computeApproxEquation();
    }
    m_criticalAccess = new hkCriticalSection();
}

//
//  Destructor

hkcdPlanarGeometryPlanesCollection::~hkcdPlanarGeometryPlanesCollection()
{
    if ( m_criticalAccess )
    {
        delete m_criticalAccess;
        m_criticalAccess = HK_NULL;
    }

    clearCaches();
}

//
//  Appends the world boundary planes to the collection. These 6 planes are used by various other
//  algorithms and will always be present in the first 6 slots of any planes collection.

void hkcdPlanarGeometryPlanesCollection::createBoundaryPlanes()
{
    HK_ASSERT_NO_MSG(0x64886180, m_planes.isEmpty());

    // Build the min and max vertices
    const int iMin = 0;
    const int iMax = (1 << (hkcdPlanarGeometryPrimitives::NumBitsVertex::NUM_BITS - 1)) - 1;

    hkInt64Vector4 iNrm;
    hkInt128 iOffset;

    // Create 6 boundary planes, one for each axis
    m_planes.setSize(6);

    // Initialize them
    iNrm.set(+1, 0, 0);         iOffset.setFromInt32(-iMax);    m_planes[0].setExactEquation(iNrm, iOffset);
    iNrm.set(-1, 0, 0);         iOffset.setFromInt32(+iMin);    m_planes[1].setExactEquation(iNrm, iOffset);
    iNrm.set(0, +1, 0);         iOffset.setFromInt32(-iMax);    m_planes[2].setExactEquation(iNrm, iOffset);
    iNrm.set(0, -1, 0);         iOffset.setFromInt32(+iMin);    m_planes[3].setExactEquation(iNrm, iOffset);
    iNrm.set(0, 0, +1);         iOffset.setFromInt32(-iMax);    m_planes[4].setExactEquation(iNrm, iOffset);
    iNrm.set(0, 0, -1);         iOffset.setFromInt32(+iMin);    m_planes[5].setExactEquation(iNrm, iOffset);
}

//
//  Creates a new collection containing the welded planes of both source collections

_Ret_notnull_ hkcdPlanarGeometryPlanesCollection* HK_CALL hkcdPlanarGeometryPlanesCollection::createMergedCollection(_In_ const hkcdPlanarGeometryPlanesCollection* planesColA,
    _In_ const hkcdPlanarGeometryPlanesCollection* planesColB,
    _Inout_opt_ hkArray<int>* planesRemapTableOut)
{
    // Create the result
    hkcdPlanarGeometryPlanesCollection* planesColAB = new hkcdPlanarGeometryPlanesCollection();
    {
        const hkVector4 vOffset = planesColA->getPositionOffset();
        const hkSimdReal vScale = planesColA->getPositionScale();
        HK_ASSERT_NO_MSG(0x20e13b42,    vOffset.allEqual<3>(planesColB->getPositionOffset(), hkSimdReal_Eps) && \
                                vScale.approxEqual(planesColB->getPositionScale(), hkSimdReal_Eps));

        planesColAB->setPositionOffset(vOffset);
        planesColAB->setPositionScale(vScale);
    }

    // Merge planes
    hkArray<int> planesRemapTable;
    if ( !planesRemapTableOut )
    {
        planesRemapTableOut = &planesRemapTable;
    }

    const int numPlanesA = planesColA->getNumPlanes();              HK_ASSERT_NO_MSG(0x2ebbb536, numPlanesA >= NUM_BOUNDS);
    const int numPlanesB = planesColB->getNumPlanes();              HK_ASSERT_NO_MSG(0x5152b9d, numPlanesB >= NUM_BOUNDS);
    const hkArray<Plane>& planesA   = planesColA->m_planes;
    const hkArray<Plane>& planesB   = planesColB->m_planes;
    hkArray<Plane>& planesAB        = planesColAB->m_planes;

    // Keep only one set of boundary planes
    planesAB.reserve(numPlanesA + numPlanesB - NUM_BOUNDS);
    planesAB.setSize(0);
    planesAB.append(planesA);
    if ( numPlanesB - NUM_BOUNDS )
    {
        planesAB.append(&planesB[NUM_BOUNDS], numPlanesB - NUM_BOUNDS);
    }

    // Weld planes from both geometries
    hkArray<Plane> weldedPlanesAB;
    if ( planesRemapTableOut )
    {
        planesRemapTableOut->reserve(numPlanesA + numPlanesB);
    }
    hkcdPlanarGeometryWeldUtil::weldPlanes(planesAB, weldedPlanesAB, *planesRemapTableOut);

    // Fix-up the remap table
    if ( planesRemapTableOut )
    {
        HK_ASSERT_NO_MSG(0x3da4f873, planesRemapTableOut->getSize() == numPlanesA + numPlanesB - NUM_BOUNDS);
        planesRemapTableOut->expandAt(numPlanesA, NUM_BOUNDS);

        // Remapped boundary planes of B
        for (int k = 0; k < NUM_BOUNDS; k++)
        {
            (*planesRemapTableOut)[numPlanesA + k] = k;
        }
    }

    // Build a new geometry
    planesAB.swap(weldedPlanesAB);

//  Log_Info( "That's a new plane collection with {} planes", planesAB.getSize() );

    return planesColAB;
}

//
//  Computes a mapping between two plane collection. The first plane set provided is assumed to be included in the second plane collection.

void HK_CALL hkcdPlanarGeometryPlanesCollection::computeMappingBetweenPlaneSets(_In_ const hkcdPlanarGeometryPlanesCollection* planedIncluded, _In_ const hkcdPlanarGeometryPlanesCollection* planesGlobal, hkArray<int>& remapTableOut)
{
    hkcdPlanarGeometryWeldUtil::computeMappingBetweenPlaneSets(planedIncluded->m_planes, planesGlobal->m_planes, remapTableOut, (int)(PlaneId::invalid().valueUnchecked()));
}

//
//  Computes the maximum number of bits used by the plane equations

void hkcdPlanarGeometryPlanesCollection::computeMaxNumUsedBits(_Out_ int& numBitsNormal, _Out_ int& numBitsOffset) const
{
    const int numPlanes = m_planes.getSize();
    hkIntVector msbMax; msbMax.setZero();
    hkIntVector cv;     cv.set(63, 63, 63, 127);

    for (int pi = numPlanes - 1; pi >= NUM_BOUNDS; pi--)
    {
        const Plane& p      = m_planes[pi];
        hkInt64Vector4 iN;  p.getExactNormal(iN);
        iN.setAbs(iN);
        hkInt128 iO;    p.getExactOffset(iO);
        iO.setAbs(iO);

        hkIntVector msb;    iN.countLeadingZeros<3>(msb);
        msb.setComponent<3>(iO.countLeadingZeros());
        msb.setSubS32(cv, msb);
        msbMax.setMaxS32(msbMax, msb);
    }

    // Write output
    numBitsNormal = msbMax.horizontalMaxS32<3>();
    numBitsOffset = msbMax.getComponent<3>();
}

//
//  Helper functions

namespace hkcdPlanarGeometryImpl
{
    //
    //  Computes a discretized quaternion

    static HK_INLINE void HK_CALL discretizeRot(hkQuaternionParameter qIn, int numBitsRot, hkIntVector& qOut)
    {
        const int N     = 1 << numBitsRot;
        hkVector4 x;    x.setAll((hkReal)N / 1.1f);
                        x.mul(qIn.m_vec);
                        x.add(hkVector4::getConstant<HK_QUADREAL_INV_2>());

        qOut.setConvertF32toS32(x);

        // Test
#ifdef HK_DEBUG
        hkIntVector vv;     vv.setAbsS32(qOut);
        hkIntVector vvv;    vvv.setAll(N);
        HK_ASSERT_NO_MSG(0x3fd39b5, vv.compareLessThanS32(vvv).allAreSet());
#endif
    }

    //
    //  Computes a rotation matrix from the given quaternion

    static HK_INLINE int HK_CALL computeRotationMatrix( hkIntVectorParameter qIn,
                                                                /*hkIntVector& col0Out, hkIntVector& col1Out, hkIntVector& col2Out,*/
                                                                hkIntVector& row0Out, hkIntVector& row1Out, hkIntVector& row2Out)
    {
        const int x = qIn.getComponent<0>();
        const int y = qIn.getComponent<1>();
        const int z = qIn.getComponent<2>();
        const int w = qIn.getComponent<3>();

        const int xx = x * x,   yy = y * y,     zz = z * z,     ww = w * w;
        const int xy = x * y,   xz = x * z,     xw = x * w;
        const int yz = y * z,   yw = y * w;
        const int zw = z * w;

        row0Out.set((xx + ww) - (yy + zz),  (xy - zw) << 1,         (xz + yw) << 1,         0);
        row1Out.set((xy + zw) << 1,         (yy + ww) - (xx + zz),  (yz - xw) << 1,         0);
        row2Out.set((xz - yw) << 1,         (yz + xw) << 1,         (zz + ww) - (xx + yy),  0);

        // Compute Transpose[R].R. Should be a scaled unit matrix.
        const int det = (xx + yy + zz + ww);
#ifdef HK_DEBUG
        {
            hkIntVector c0;     c0.set((xx + ww) - (yy + zz),   (xy + zw) << 1,         (xz - yw) << 1,         0);
            hkIntVector c1;     c1.set((xy - zw) << 1,          (yy + ww) - (xx + zz),  (yz + xw) << 1,         0);
            hkIntVector c2;     c2.set((xz + yw) << 1,          (yz - xw) << 1,         (zz + ww) - (xx + yy),  0);

            const hkInt64 m00 = c0.dot<3>(c0);
            const hkInt64 m11 = c1.dot<3>(c1);
            const hkInt64 m22 = c2.dot<3>(c2);
            HK_ASSERT_NO_MSG(0x3e9bbc18, (m00 == m11) && (m11 == m22) && (m00 == (hkInt64)det * (hkInt64)det));

            const hkInt64 m01 = c0.dot<3>(c1);
            const hkInt64 m02 = c0.dot<3>(c2);
            const hkInt64 m12 = c1.dot<3>(c2);
            HK_ASSERT_NO_MSG(0x1514c0fb, (m01 == 0) && (m02 == 0) && (m12 == 0));
        }
#endif

        return det;
    }
}

//
//  Applies the given transform on the planes. Note that this will lose precision!

void hkcdPlanarGeometryPlanesCollection::applyTransform(const hkQTransform& transform, int numBitsTransform, bool simplifyEquations, int startPlaneIdx, int endPlaneIdx)
{
    // Discretize rotation quaternion
    hkIntVector iRot;
    hkcdPlanarGeometryImpl::discretizeRot(transform.m_rotation, numBitsTransform, iRot);

    // Compute rotation matrix
    hkIntVector rX, rY, rZ;
    const hkInt64 rDet = hkcdPlanarGeometryImpl::computeRotationMatrix(iRot, rX, rY, rZ);

    // Compute translation delta. Rotation takes place around the real world's origin, so we need to rotate around that as well
    hkIntVector t;  convertWorldDirection(transform.m_translation, t);
    hkIntVector c;  convertWorldPosition(hkVector4::getConstant<HK_QUADREAL_0>(), c);

    // Transform planes
    startPlaneIdx = hkMath::max2(startPlaneIdx, NUM_BOUNDS);
    const int numPlanes = ( endPlaneIdx < NUM_BOUNDS ) ? m_planes.getSize() : hkMath::min2(m_planes.getSize(), endPlaneIdx + 1);
    for (int pi = startPlaneIdx ; pi < numPlanes; pi++)
    {
        Plane& p            = m_planes[pi];
        hkInt64Vector4 iN;  p.getExactNormal(iN);
        hkInt128 iO;    p.getExactOffset(iO);
                            iO.setAdd(iO, iN.dot<3>(c));
#ifdef HK_DEBUG
        // Check for Normal overflow
        {
            const hkInt128 nx = iN.dot<3>(rX);
            const hkInt128 ny = iN.dot<3>(rY);
            const hkInt128 nz = iN.dot<3>(rZ);
            hkInt128 absX;  absX.setAbs(nx);
            hkInt128 absY;  absY.setAbs(ny);
            hkInt128 absZ;  absZ.setAbs(nz);
            const int msbX      = 127 - absX.countLeadingZeros();
            const int msbY      = 127 - absY.countLeadingZeros();
            const int msbZ      = 127 - absZ.countLeadingZeros();
            const int msbXYZ    = hkMath::max2(hkMath::max2(msbX, msbY), msbZ);
            HK_ASSERT_NO_MSG(0x1f181269, msbXYZ <= hkcdPlanarGeometryPrimitives::NumBitsPlaneNormal::NUM_BITS - 2);
        }
#endif
        // Rotate normal & offset
        iN.set(iN.dot_64<3>(rX), iN.dot_64<3>(rY), iN.dot_64<3>(rZ), 0);

        // Add translation
        hkIntVector ct;     ct.setAddS32(c, t);
        hkInt128 dO = iN.dot<3>(ct);
        iO.setMul(iO, rDet);
        iO.setSub(iO, dO);


#ifdef HK_DEBUG
        // Check for Offset overflow
        {
            hkInt128 absO;  absO.setAbs(iO);
            const int msbW          = 127 - absO.countLeadingZeros();
            HK_ASSERT_NO_MSG(0x117dc11, msbW <= hkcdPlanarGeometryPrimitives::NumBitsPlaneOffset::NUM_BITS - 2);
        }
#endif

        // Set new plane equation
        p.setExactEquation(iN, iO, simplifyEquations);

    }
}

//
//  Appends the planes of the given collection to this collection. Optionally returns the array of plane Ids for the merged planes.
//  Appending planes can preserve the cache, since the cached planes do not change.

void hkcdPlanarGeometryPlanesCollection::append(const hkcdPlanarGeometryPlanesCollection& src, _Inout_opt_ hkArray<PlaneId>* appendedPlaneIdsOut)
{
    // Only planes with same conversion aabb should be appended
    HK_ASSERT_NO_MSG(0x20e13b5d, src.getPositionOffset().allEqual<3>(getPositionOffset(), hkSimdReal_Eps) && \
        src.getPositionScale().approxEqual(getPositionScale(), hkSimdReal_Eps));

    const hkArray<Plane>& srcPlanes = src.m_planes;
    const int numSrcPlanes          = srcPlanes.getSize();

    // Do not append the boundary planes!
    const int numOldplanes  = m_planes.getSize();
    const int numNewPlanes  = numSrcPlanes - NUM_BOUNDS;
    Plane* newPlanes        = m_planes.expandBy(numNewPlanes);

    PlaneId* idsOut = HK_NULL;
    if ( appendedPlaneIdsOut )
    {
        idsOut = appendedPlaneIdsOut->expandBy(numNewPlanes);
    }

    for (int k = 0; k < numNewPlanes; k++)
    {
        const Plane p   = srcPlanes[k + NUM_BOUNDS];
        newPlanes[k]    = p;

        if ( idsOut )
        {
            idsOut[k] = PlaneId(numOldplanes + k);
        }
    }
}

//
//  Welds the planes

void hkcdPlanarGeometryPlanesCollection::weldPlanes(_Inout_opt_ hkArray<int>* planesRemapTableOut)
{
    hkArray<Plane> weldedPlanes;
    hkArray<int> remapTable;
    if ( !planesRemapTableOut )
    {
        planesRemapTableOut = &remapTable;
    }

    hkcdPlanarGeometryWeldUtil::weldPlanes(m_planes, weldedPlanes, *planesRemapTableOut);

    m_planes.swap(weldedPlanes);
    clearCaches();
}

//
//  Removes all planes set in the bit-field.

void hkcdPlanarGeometryPlanesCollection::removePlanes(const hkBitField& planesToRemove, _Inout_opt_ hkArray<int>* planeRemapOut)
{
    hkArray<int> localPlaneRemap;
    if ( !planeRemapOut )
    {
        planeRemapOut = &localPlaneRemap;
    }

    const int maxNumPlanes      = m_planes.getSize();
    hkArray<Plane> newPlanes;   newPlanes.reserve(maxNumPlanes);
    planeRemapOut->setSize(maxNumPlanes, -1);

    // Make sure the boundary planes are preserved
    for (int k = hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS - 1; k >= 0; k--)
    {
        HK_ASSERT_NO_MSG(0x2f020323, !planesToRemove.get(k));
    }

    // Add all planes that were requested
    for (int k = 0; k < planesToRemove.getSize(); k++)
    {
        if ( !planesToRemove.get(k) )
        {
            HK_ASSERT_NO_MSG(0x7a7bae2a, (*planeRemapOut)[k] < 0);
            (*planeRemapOut)[k] = newPlanes.getSize();
            newPlanes.pushBack(m_planes[k]);
        }
    }

    // Replace with new planes
    newPlanes.optimizeCapacity(0, true);
    m_planes.swap(newPlanes);
    clearCaches();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
