// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

#include <Geometry/Collide/DataStructures/Planar/hkcdPlanarEntity.h>
#include <Geometry/Collide/DataStructures/Planar/Predicates/hkcdPlanarGeometryPredicates.h>
#include <Geometry/Collide/DataStructures/Planar/Memory/hkcdPlanarGeometryPlanesCollection.h>

/// Geometry representation where vertices are stored implicitly as intersections of 3 planes.
class HK_EXPORT_COMMON hkcdPlanarGeometry : public hkcdPlanarEntity
{
    public:
        HK_DECLARE_CLASS(hkcdPlanarGeometry, New, Reflect);
        HK_RECORD_ATTR(hk::Version(1));
        HK_RECORD_ATTR(hk::IncludeInMgd(false));

        typedef hkcdPlanarGeometryPolygonCollection::VertexId       VertexId;

        struct VertexStorage : public hkReferencedObject, public hkArray<hkVector4d>
        {
            HK_DECLARE_CLASS(VertexStorage, New, NoReflect2(hkReferencedObject));

            virtual ~VertexStorage()
            {}
        };

    public:

        /// Constructor
        hkcdPlanarGeometry(_In_opt_ PlanesCollection* planesCollection = HK_NULL, int initialPolyCapacity = 0, _In_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL);

        /// Constructor with polygon collection
        hkcdPlanarGeometry(_In_opt_ PlanesCollection* planesCollection, _In_opt_ hkcdPlanarGeometryPolygonCollection* polygonCollection, _In_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL);

        /// Copy constructor
        hkcdPlanarGeometry(const hkcdPlanarGeometry& other);

        void afterReflectNew();

        /// Destructor
        virtual ~hkcdPlanarGeometry();

    public:

        /// Retrieves all valid polygon Ids
        void getAllPolygons(hkArray<PolygonId>& polygonsOut) const;

        /// Collects all planes used by the given polygons
        void getAllPolygonsPlanes(const hkArray<PolygonId>& polygonsIn, hkArray<PlaneId>& planesOut, bool collectBoundaryPlanes, bool collectSupportPlanes = true) const;

        /// Classifies a triangle w.r.t. a plane. The result is approximative, as it uses floating-point operations
        hkcdPlanarGeometryPredicates::Orientation approxClassify(PolygonId polygonId, PlaneId planeId) const;
        hkcdPlanarGeometryPredicates::Orientation approxClassify(PolygonId polygonId, const Plane& splitPlane) const;

        /// Classifies a triangle w.r.t. a plane
        hkcdPlanarGeometryPredicates::Orientation classify(PolygonId polygonId, PlaneId planeId) const;

        /// Splits a polygon with the given splitting plane. Returns the part of the polygon inside the given plane
        void split(PolygonId polygonId, PlaneId splitPlaneId, PolygonId& insidePolygonIdOut, PolygonId& outsidePolygonIdOut);

        /// Returns true if two given polygon on the same support plane potentially overlap. False means that the surfaces of the two polygon do not overlap.
        bool check2dIntersection(const PolygonId& polygonA, const PolygonId& polygonB) const;
        static bool HK_CALL check2dIntersection(const hkcdPlanarGeometry& geomA, const PolygonId& polygonIdA,
                                                const hkcdPlanarGeometry& geomB, const PolygonId& polygonIdB);

        /// Clips a polygon with the given planes. Returns the number of boundary planes.
        hkUint32 clipPolygon(PlaneId supportPlaneId, _Inout_updates_(numBounds) PlaneId*& boundsIn, _Inout_updates_(numBounds) PlaneId*& boundsOut, int numBounds, _In_reads_(numClippingPlanesIn) const PlaneId* clippingPlanesIn, int numClippingPlanesIn);

        /// Removes the given polygons from the mesh
        void removePolygons(const hkArray<PolygonId>& polygonIds);

        /// Adds the given polygons to this geometry and returns the newly added polygon Ids
        hkResult appendGeometryPolygons(const hkcdPlanarGeometry& srcGeom, const hkArray<PolygonId>& srcPolygonIds, hkArray<PolygonId>& dstPolygonIds);

        /// Removes all polygons that are not present in the given list. The given list of polygon Ids must be sorted ascending!
        void keepPolygons(const hkArray<PolygonId>& polygonIds);

        /// Builds a vertex-based geometry representation from this entity.
        virtual void extractGeometry(hkGeometry& geomOut) const HK_OVERRIDE;
        void extractGeometry(hkGeometry& geomOut, hkArray<Material>& trisMat, _Inout_opt_ hkArray<PlaneId>* triSupportPlaneIds = HK_NULL, _Inout_opt_ hkArray<PlaneId>* triBoundaryPlaneIds = HK_NULL) const;

        /// Builds a vertex-based geometry representation for the given polygon
        void extractPolygonGeometry(PolygonId polyId, hkGeometry& geomOut) const;

        /// Builds a vertex-based geometry representation for the given polygon
        void extractPolygonsGeometry(const hkArray<PolygonId>& polyIds, hkGeometry& geomOut) const;

        /// Welds the planes so that all planes are unique
        void weldPlanes(_Inout_opt_ hkArray<int>* planeRemapTable = HK_NULL);

        /// Weld polygon vertices to save memory
        void weldPolygonVertices();

        /// Collects a bit-field of plane Ids used by the polygons
        void collectUsedPlaneIds(hkBitField& usedPlaneIdsOut) const;

        /// Returns the aabb of the given polygon set
        void computeAabb(const hkArray<PolygonId>& polyIds, hkAabb& aabbOut) const;

    public:

        /// Sets a new planes collection. If the plane remapping table is non-null, the plane Ids on all nodes will be re-set as well (i.e. to match the plane Ids in the new collection)
        void setPlanesCollection(_In_opt_ PlanesCollection* newPlanes, _In_reads_opt_(_Inexpressible_()) const int* HK_RESTRICT planeRemapTable);

        /// Shift all plane ids in the geom polygons
        void shiftPlaneIds(int offsetValue);

        /// Remaps the materials triangle source Ids
        void remapTriangleProviderIds(const hkArray<TriangleProviderId>& triSrcIdRemap);

        /// Checks planes consistency within the geometry
        bool checkPlanesConsistency() const;

        /// Deduce the approximate vertex position for the given polygon using plane intersections
        void computePolygonApproxVertices(const PolygonId& polyId);

        /// (Debug only) check if the cached value at polygon vertices are valid
        void checkPolygonCachedValues(const PolygonId& polyId);

        /// (Debug only) check if the given polygon is convex
        void checkPolygonConvexity(const PolygonId& polyId);

        /// Recompute all the vertices pos given the current plane state. This assumes that all polygons have valid vertex ids
        void recomputeVerticesCache(int nbEstimatedVertices = 0);

        /// Set the vertex cache from the given array
        void setVerticesCacheFromArray(const hkArray<hkVector4d>& verticesPos);

        /// Returns the plane collection associated with this geometry
        HK_INLINE _Ret_maybenull_ const PlanesCollection* getPlanesCollection() const;
        HK_INLINE _Ret_maybenull_ PlanesCollection* accessPlanesCollection();

        /// Adds a new polygon
        HK_INLINE PolygonId addPolygon(PlaneId supportPlane, const Material& materialIn, int numBounds);

        /// Returns the polygon at the given Id
        HK_INLINE const Polygon& getPolygon(PolygonId polyId) const;
        HK_INLINE Polygon& accessPolygon(PolygonId polyId);

        /// Computes the number of boundary planes
        HK_INLINE int getNumBoundaryPlanes(PolygonId polyId) const;

        /// Returns the collection of polygons
        HK_INLINE const hkcdPlanarGeometryPolygonCollection& getPolygons() const;
        HK_INLINE hkcdPlanarGeometryPolygonCollection& accessPolygons();
        HK_INLINE void setPolygons(_In_ hkcdPlanarGeometryPolygonCollection* polys);

    protected:

        /// The planes
        hkRefPtr<PlanesCollection> m_planes;

        /// The polygons
        hkRefPtr<hkcdPlanarGeometryPolygonCollection> m_polys;

        /// The vertices
        hkRefPtr<VertexStorage> m_vertices;                     //+nosave
};

#include <Geometry/Collide/DataStructures/Planar/Geometry/hkcdPlanarGeometry.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
