// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/TaskQueue/hkTask.h>

#include <Geometry/Collide/DataStructures/Planar/ConvexCellsTree/hkcdConvexCellsTree.h>
#include <Geometry/Collide/DataStructures/Planar/Memory/hkcdNewCellsCollection.h>

/// A binary tree of convex cells. The root node is generally
/// a box (typically initialized to an AABB larger than the object). Child cells are generated by cutting parent cells with a plane.
class HK_EXPORT_COMMON hkcdConvexCellsTree3D : public hkcdConvexCellsTree<hkcdNewCellsCollection::Cell, hkcdNewCellsCollection::CellId, hkcdNewCellsCollection>
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY);

        // Types
        typedef hkcdNewCellsCollection::Labels  CellLabel;
        typedef hkcdPlanarSolid::PlaneId        PlaneId;
        typedef hkUint32                        VertexId;
        typedef hkUint32                        EdgeId;
        typedef hkUint32                        FaceId;

    public:

        /// Types of surfaces used to infer cell labels
        enum FaceSurfaceType
        {
            FACE_SURFACE_OPEN           = 0,
            FACE_SURFACE_DIRECT         = 0x01,
            FACE_SURFACE_INDIRECT       = 0x02,
            FACE_SURFACE_WATERTIGHT     = FACE_SURFACE_DIRECT | FACE_SURFACE_INDIRECT,
            FACE_SURFACE_INVALID        = 0x04,

            FACE_SURFACE_TYPE_MASK      = 0x0f,

            FACE_VISITED_FLAG           = 0x10,
        };

    public:

        /// Constructor for compatibility with previous approach using geom
        hkcdConvexCellsTree3D(_In_ const hkcdPlanarGeometryPlanesCollection* planes, bool withConnectivity = false, bool allowCellDealloc = false);

        /// Copy constructor
        hkcdConvexCellsTree3D(const hkcdConvexCellsTree3D& other);

        /// Destructor
        virtual ~hkcdConvexCellsTree3D();

    public:

        /// Build a convex cell tree out of a solid bsp  tree
        void buildFromSolid(_In_opt_ const ExecutionContext* executionCtx, _Inout_ hkcdPlanarSolid* planes);

        /// Splits the given cell by the given plane. Updates all adjacency information
        /// If the cell is split, returns the new face created
        FaceId splitCell(CellId cellId, PlaneId splitPlaneId, const Material& splitMaterial, CellId& insideCellIdOut, CellId& outsideCellIdOut, bool repaintOnlyInvalidMat = true);

        /// Creates a box cell that encloses the entire "known" space
        CellId createBoundaryCell();

        /// Collects all the leaf cells
        void collectLeafCells(hkArray<CellId>& cellIdsOut) const;

        /// Collects all the cells marked as solid
        void collectSolidCells(hkArray<CellId>& cellIdsOut) const;

        /// Converts the given cell to geometry
        void extractCellGeometry(CellId cellId, hkGeometry& cellGeomOut) const;

        /// Computes the boundary geometry
        void computeBoundaryGeometry(const hkArray<CellId>& cellIdsIn, hkGeometry& cellGeomOut,
            hkArray<Material>& triMaterials, _Inout_opt_ hkArray<PlaneId>* triSupportPlaneIds = HK_NULL,
            _Inout_opt_ hkArray<PlaneId>* triBoundaryPlaneIds = HK_NULL, _Inout_opt_ hkArray<hkIntVector>* verticesIntPos = HK_NULL) const;

        /// Compute and creates boundary polygons for a given set of cells. Requires connectivity
        hkResult extractBoundaryPolygonsFromCellIds(const hkArray<CellId>& cellIdsIn, hkcdPlanarGeometry& geomOut, hkArray<PolygonId>& boundaryPolygonIdsOut) const;

        /// Re-assigns the labels of a BSP tree using the label store in the convex cell tree
        void reassignSolidGeomLabels(_In_opt_ const ExecutionContext* executionCtx, const hkcdPlanarGeometry& originalGeom, _Inout_ hkcdPlanarSolid* solid, const hkArray<PolygonId>& originalBoundaryPolygonIds, _Inout_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL);

        /// Computes the approximate position of the vertices of the given convex cell (floating point precision)
        void collectCellVerticesWorld(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<hkVector4>& verticesPos) const;
        void collectCellVerticesD(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<hkVector4d>& verticesPos) const;
        void collectCellVerticesInt(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<hkIntVector>& verticesPos) const;

        /// Extracts a solid planar geometry from a subset of selected cells
        _Ret_maybenull_ hkcdPlanarSolid* buildSolidFromSubsetOfCells(_In_opt_ const ExecutionContext* executionCtx, const hkArray<CellId>& cellIdsIn) const;

        /// Returns a set of cell ids representing disconnected islands of solid regions
        void computeSolidRegionIslands(hkArray< hkArray<CellId> >& islands) const;

        /// Collect all plane ids of a given cell
        void collectCellPlaneIds(CellId cellId, hkArray<PlaneId>& planeIdsOut) const;

        /// Enables / disables the collection of connectivity information during the building of the tree.
        HK_INLINE void enableConnectivityCollection(bool doEnable);

        /// Returns whether the tree has been built with connectivity information.
        HK_INLINE bool hasManifoldCells() const;

        /// Returns the cell with the given Id
        HK_INLINE const Cell& getCell(CellId cellId) const;
        HK_INLINE Cell& accessCell(CellId cellId);

        /// Accessor to the plane collection
        HK_INLINE _Ret_notnull_ const hkcdPlanarGeometryPlanesCollection* getPlanesCollection() const;

        /// Returns whether repaint face material events occured during the splitting of the cells
        HK_INLINE bool hasRepaintedFaces() const;

        /// Sets a new planes collection, remapping plane ids optionally.
        void setPlanesCollection(_In_opt_ const PlanesCollection* newPlanes, _In_reads_opt_(_Inexpressible_()) const int* HK_RESTRICT planeRemapTable);

        /// Shift all plane ids
        void shiftPlaneIds(int offsetValue);

        /// Remaps the materials triangle source Ids
        void remapTriangleProviderIds(const hkArray<TriangleProviderId>& triSrcIdRemap);

        /// Collects all the plane Ids used by the Bsp tree
        void collectUsedPlaneIds(hkBitField& usedPlaneIdsOut) const;

    protected:

        struct Vertex
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdConvexCellsTree3D::Vertex);
            Vertex() {}
            PlaneId m_planeIds[3];
            hkVector4d m_pos;               ///< Approximate position
        };

        struct Edge
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdConvexCellsTree3D::Edge);
            VertexId m_vertexIds[2];
            FaceId m_faceIds[2];

            /// For connectivity build only:
            PlaneId m_otherPlaneId;         ///< Additional plane id used when the two faces referenced by the edge are coplanar
            EdgeId m_nextCoincidentEdgeId;
            EdgeId m_firstEdgeOfChainId;
            CellId m_owningCellId;
            int m_indexInCell;
        };

        struct Face
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdConvexCellsTree3D::Face);

            HK_INLINE void setVisited()             { m_flags |= FACE_VISITED_FLAG;             }
            HK_INLINE void setNotVisited()          { m_flags &= ~(FACE_VISITED_FLAG);          }
            HK_INLINE bool isVisited() const        { return ( m_flags & FACE_VISITED_FLAG );   }

            HK_INLINE int getSurfaceType()  const   { return m_flags & FACE_SURFACE_TYPE_MASK;  }

            PlaneId m_planeId;              ///< Support plane for the face
            CellId m_negCellId;             ///< In case of building connectivity, cell Id on the negative side (behind the plane)
            CellId m_posCellId;             ///< In case of building connectivity, cell Id on the positive side (in front of the plane)
            Material m_materialId;          ///< material of the face
            hkUint16 m_userData;
            hkUint16 m_flags;
        };

        // all data
        struct Data : public hkReferencedObject
        {
            HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY);

            /// Constructor
            Data() {}

            /// Copy constructor
            Data(const Data& other);

        public:

            HK_INLINE _Ret_notnull_ Vertex* accessVertices();
            HK_INLINE _Ret_notnull_ Edge* accessEdges();
            HK_INLINE _Ret_notnull_ Face* accessFaces();

            HK_INLINE Vertex& allocateNewVertex(VertexId& newVertexId);
            HK_INLINE Edge& allocateNewEdge(EdgeId& newEdgeId);
            HK_INLINE Face& allocateNewFace(FaceId& newFaceId);

            HK_INLINE void freeVertex(VertexId vertexId);
            HK_INLINE void freeEdge(EdgeId edgeId);
            HK_INLINE void freeFace(FaceId faceId);

            HK_INLINE int getNumAllocatedVertices() const;
            HK_INLINE int getNumAllocatedEdges() const;
            HK_INLINE int getNumAllocatedFaces() const;

        protected:
            hkArray<Vertex> m_vertices;
            hkArray<Edge> m_edges;
            hkArray<Face> m_faces;

            // Mem handling
            hkArray<int> m_freeVerticesId;
            hkArray<int> m_freeEdgesId;
            hkArray<int> m_freeFacesId;
        };

    protected:

        /// Check if the direction cell to polygon is valid
        HK_INLINE bool canGoFromCellThroughFace(CellId fromCellId, FaceId throughFaceId, bool fromOutside);

        /// Infers in/out labels by flood filling thanks to the boundary properties computed on each polygon of each cell
        void inferCellsLabels(_Inout_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL);

        /// From a set of polygons of the original geometry, mark all the boundary of all the cells with an intersection test
        void markBoundaryCells(_In_opt_ const ExecutionContext* executionCtx, const hkcdPlanarGeometry& originalGeom, const hkArray<PolygonId>& originalBoundaryPolygonIds, _Inout_opt_ hkcdPlanarEntityDebugger* debugger = HK_NULL);

        /// Finds an empty cell
        CellId findOutputCell();

        /// Returns a list of unique face ids from a set of cells
        void getUniqueFaceIdsFromCellIds(const hkArray<CellId>& cellIdsIn, hkArray<FaceId>& faceIdsOut);

        /// Splits an edge
        void splitEdge(EdgeId edgeId, bool flipped, PlaneId outsidePtId, EdgeId& edgeIdIn, EdgeId& edgeIdOut, VertexId& newVertexId);

        /// Updates the connectivity on a neighboring cell
        void updateConnectivity(CellId cellIdToUpdate,
            const FaceId faceSplitId, const FaceId faceSplitInId, const FaceId faceSplitOutId,
            const EdgeId newEdgeInId, const int faceNumEdgesSplit);

        /// Internal helper for splitCell: given a staring point for a new edge, and an adjacent inside edge, decide the winding of the edge. Returns true if edge should be flipped
        HK_INLINE bool shouldFlipNewEdgeForInFace(const Edge& adjInEdge, const VertexId newEdgeFirstVertexId, const FaceId inFaceId) const;

        /// Returns plane id with correct orientation given a face and cellId
        HK_INLINE PlaneId getCellFacePlaneId(const Face& face, CellId cellId) const;

        /// Helper to link to given edges during connectivity build
        HK_INLINE void linkEdges(EdgeId edgeToLinkId, EdgeId edgeRefId);

        /// Helper to update the linked list f chained edges with a new edgeId
        HK_INLINE void replaceEdgeInLinkedChain(EdgeId edgeToReplaceId, EdgeId replacingEdgeId);

        /// Creates a polygon from a cellId and faceId
        PolygonId createPolygonFromFace(CellId cellId, FaceId faceId, hkcdPlanarGeometry& workingGeom) const;
        void createPolygonFromFace(CellId cellId, FaceId faceId, hkArray<VertexId>& polyVerticeIds, _Inout_opt_ hkArray<PlaneId>* bPlaneIds = HK_NULL) const;

        /// Gather unique vertex ids for the given cells
        void collectUniqueVertexIds(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<VertexId>& vertexIdsOut) const;

        /// Clones a cell, cloning as well all its edges
        CellId cloneCell(CellId cellId);

        /// Debug only: checks a cell integrity
        void checkCellIntegrity(CellId cellId) const;

    protected:

        /// Data
        hkRefPtr<Data> m_data;
        hkRefPtr<const hkcdPlanarGeometryPlanesCollection> m_planes;

        /// Whether the tree is kept manifold and conectivity computed between cells
        bool m_buildCellConnectivity;

        /// Whether non leaf cells are de-allocted while the tree is built to save memory
        bool m_preserveNonLeafCells;

        /// Whether some faces have been repainted
        bool m_hasRepaintedFaces;
};

#include <Geometry/Collide/DataStructures/Planar/ConvexCellsTree/hkcdConvexCellsTree3D.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
