// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/DataStructures/Planar/ConvexCellsTree/hkcdConvexCellsTree3D.h>
#include <Geometry/Collide/DataStructures/Planar/Solid/hkcdPlanarSolid.h>

#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Algorithm/Collide/1AxisSweep/hk1AxisSweep.h>
#include <Common/GeometryUtilities/Mesh/Utils/FindUniquePositionsUtil/hkFindUniquePositionsUtil.h>

#define DEBUG_LOG_IDENTIFIER "geo.collide.convexcelltree3d"
#include <Common/Base/System/Log/hkLog.hxx>

#define HK_DEBUG_CELLS_INTEGRITY                        (0)
#define HK_DEBUG_CELLS_INTEGRITY_VERTEX_CACHE           (0)
#define HK_DEBUG_CELLS_INTEGRITY_EDGE_WINDING           (0)
#define HK_DEBUG_CELLS_INTEGRITY_EDGE_SPLIT             (0)         // SLOW!!
#define HK_DEBUG_CELLS_INTEGRITY_CELL_CONVEXITY         (0)
#define HK_DEBUG_INTEGRITY_POLY_CONVEXITY               (0)

//
//  Constructor

hkcdConvexCellsTree3D::hkcdConvexCellsTree3D(_In_ const hkcdPlanarGeometryPlanesCollection* planes, bool withConnectivity, bool allowCellDealloc)
:   hkcdConvexCellsTree<hkcdNewCellsCollection::Cell, hkcdNewCellsCollection::CellId, hkcdNewCellsCollection>()
,   m_planes(planes)
,   m_buildCellConnectivity(withConnectivity)
,   m_preserveNonLeafCells(allowCellDealloc)
,   m_hasRepaintedFaces(false)
{
    m_data.setAndDontIncrementRefCount(new Data());
}

//
//  Copy constructor

hkcdConvexCellsTree3D::hkcdConvexCellsTree3D(const hkcdConvexCellsTree3D& other)
:   hkcdConvexCellsTree<hkcdNewCellsCollection::Cell, hkcdNewCellsCollection::CellId, hkcdNewCellsCollection>(other)
,   m_planes(other.m_planes)
,   m_buildCellConnectivity(other.m_buildCellConnectivity)
,   m_preserveNonLeafCells(other.m_preserveNonLeafCells)
,   m_hasRepaintedFaces(other.m_hasRepaintedFaces)
{
    // Clone the data
    m_data.setAndDontIncrementRefCount(new Data(*other.m_data));
}

//
//  Data copy constructor

hkcdConvexCellsTree3D::Data::Data(const Data& other)
:   hkReferencedObject()
{
    m_vertices.append(other.m_vertices);
    m_edges.append(other.m_edges);
    m_faces.append(other.m_faces);
    m_freeVerticesId.append(other.m_freeVerticesId);
    m_freeEdgesId.append(other.m_freeEdgesId);
    m_freeFacesId.append(other.m_freeFacesId);
}

//
//  Destructor

hkcdConvexCellsTree3D::~hkcdConvexCellsTree3D()
{}

//
//  Build a convex cell tree out of a solid bsp tree

void hkcdConvexCellsTree3D::buildFromSolid(_In_opt_ const ExecutionContext* executionCtx, _Inout_ hkcdPlanarSolid* solid)
{
    // Set-up boundary cell
    {
        CellId boundaryCellId = createBoundaryCell();
        solid->accessNode(solid->getRootNodeId()).m_data = boundaryCellId.value();
        accessCell(boundaryCellId).setUserData(solid->getRootNodeId().value());
    }
    m_hasRepaintedFaces = false;

    // Initialize nodes stack
    hkArray<NodeId> nodeStack;
    nodeStack.pushBack(solid->getRootNodeId());

    // Split boundary cell with each plane in the BSP tree
    while ( nodeStack.getSize() && !hkTask::receivedAbortRequest(executionCtx) )
    {
        // Pop node from the stack
        const NodeId nodeId = nodeStack[0];
        const Node& node = solid->getNode(nodeId);
        nodeStack.removeAt(0);

        // Get the cell associated with this node
        const CellId nodeCellId(node.m_data);

        // Check if the node is internal
        if ( node.m_type == hkcdPlanarSolid::NODE_TYPE_INTERNAL )
        {
            HK_ASSERT_NO_MSG(0x73df24d1, node.m_left.isValid());
            HK_ASSERT_NO_MSG(0x43141527, node.m_right.isValid());

            CellId inCell   = CellId::invalid();
            CellId outCell  = CellId::invalid();

            if ( nodeCellId.isValid() )
            {
                // Split it with the plane in the BSP node
                splitCell(nodeCellId, node.m_planeId, node.m_material, inCell, outCell);
            }

            // Recurse on left
            {
                Node& leftChild     = solid->accessNode(node.m_left);
                leftChild.m_data    = inCell.valueUnchecked();
                nodeStack.pushBack(node.m_left);
            }

            // Recurse on right
            {
                Node& rightChild    = solid->accessNode(node.m_right);
                rightChild.m_data   = outCell.valueUnchecked();
                nodeStack.pushBack(node.m_right);
            }
        }
        else if ( nodeCellId.isValid() )
        {
            Cell& cell =  accessCell(nodeCellId);
            cell.setLeaf(true);
            cell.setLabel( ( node.m_type == hkcdPlanarSolid::NODE_TYPE_IN ) ? hkcdNewCellsCollection::CELL_SOLID : hkcdNewCellsCollection::CELL_EMPTY);
        }
    }
}

//
//  Collects all the leaf cells

void hkcdConvexCellsTree3D::collectLeafCells(hkArray<CellId>& cellIdsOut) const
{
    for (CellId cellId = m_cells->getFirstCellId(); cellId.isValid(); cellId = m_cells->getNextCellId(cellId))
    {
        // Get the cell and its children
        const Cell& cell = getCell(cellId);
        if ( !cell.isLeaf() )
        {
            continue;
        }

        cellIdsOut.pushBack(cellId);
    }
}

//
//  Collects all the cells marked as solid

void hkcdConvexCellsTree3D::collectSolidCells(hkArray<CellId>& cellIdsOut) const
{
    for (CellId cellId = m_cells->getFirstCellId(); cellId.isValid(); cellId = m_cells->getNextCellId(cellId))
    {
        // Get the cell and its children
        const Cell& cell = getCell(cellId);
        if ( !cell.isSolid() || !cell.isLeaf() )
        {
            continue;
        }

        cellIdsOut.pushBack(cellId);
    }
}

//
//  Converts a single cell to geometry

void hkcdConvexCellsTree3D::extractCellGeometry(CellId cellId, hkGeometry& cellGeomOut) const
{
    hkArray<int> polyIb;
    const Cell& cell    = getCell(cellId);
    const int numFaces      = cell.getNumFaces();
    const FaceId* faceIds   = cell.getFaceIdsPointer();

    hkFindUniquePositionsUtil vtxWelder;

    // Get offset and scale to be applied to vertices
    hkSimdDouble64 scaleD;
    hkVector4d offsetD;
    {
        hkDouble64 doubleBuff[4];
        hkVector4 offset = m_planes->getPositionOffset();
        offset.store<4, HK_IO_NATIVE_ALIGNED>(doubleBuff);  offsetD.load<4, HK_IO_NATIVE_ALIGNED>(doubleBuff);
        hkSimdReal scale = m_planes->getPositionScale();
        scaleD.setFromFloat(scale.getReal());
        scaleD.setReciprocal(scaleD);
    }

    for (int f = 0 ; f < numFaces ; f++)
    {
        // Each polygon is convex
        hkArray<VertexId> polyVertexIds;
        createPolygonFromFace(cellId, faceIds[f], polyVertexIds);

        // Compute all vertices
        const int numPolyVerts = polyVertexIds.getSize();
        polyIb.setSize(numPolyVerts);

        for (int crt = 0 ; crt < numPolyVerts ; crt++)
        {
            hkVector4d fvd = m_data->accessVertices()[polyVertexIds[crt]].m_pos;
            hkFloat32 realBuff[4];
            fvd.mul(scaleD);    fvd.add(offsetD);   fvd.store<4, HK_IO_NATIVE_ALIGNED>(realBuff);
            hkVector4 fv;       fv.load<4, HK_IO_NATIVE_ALIGNED>(realBuff);
            polyIb[crt]         = vtxWelder.addPosition(fv);
        }

        // Triangulate. The polygon is convex
        int ixMin = 0;
        for (int i = 1 ; i < numPolyVerts ; i++)
        {
            if ( polyIb[i] < polyIb[ixMin] ) ixMin = i;
        }
        int ixP1 = ixMin + 1;
        if ( ixP1 == numPolyVerts ) ixP1 = 0;
        int ixP2 = ixP1 + 1;
        if ( ixP2 == numPolyVerts ) ixP2 = 0;
        for (int k = 2; k < numPolyVerts; k++)
        {
            cellGeomOut.m_triangles.expandOne().set(polyIb[ixMin], polyIb[ixP1], polyIb[ixP2], cellId.value());
            ixP1++;
            if ( ixP1 == numPolyVerts ) ixP1 = 0;
            ixP2 = ixP1 + 1;
            if ( ixP2 == numPolyVerts ) ixP2 = 0;
        }
    }

    cellGeomOut.m_vertices.swap(vtxWelder.m_positions);
}

//
//  Computes the boundary geometry

void hkcdConvexCellsTree3D::computeBoundaryGeometry(
    const hkArray<CellId>& cellIdsIn, hkGeometry& cellGeomOut, hkArray<Material>& triMaterials, _Inout_opt_ hkArray<PlaneId>* triSupportPlaneIds,
    _Inout_opt_ hkArray<PlaneId>* triBoundaryPlaneIds, _Inout_opt_ hkArray<hkIntVector>* verticesIntPos) const
{
    hkArray<VertexId> polysVertexIds;
    hkArray<int> polysNumVertices;
    hkArray<Material> polysMat;
    hkArray<PlaneId> polysPlaneIds;
    hkArray<PlaneId> polysBPlaneIds;

    // Loop over all the faces
    for (int cIdx = cellIdsIn.getSize() - 1 ; cIdx >= 0 ; cIdx--)
    {
        const CellId cellId = cellIdsIn[cIdx];
        const Cell& cell    = getCell(cellIdsIn[cIdx]);

        // Loop over the faces, and get the boundary ones
        for (int fIdx = cell.getNumFaces() - 1 ; fIdx >= 0 ; fIdx--)
        {
            const FaceId faceId     = cell.getFaceIdsPointer()[fIdx];
            const Face& cellFace    = m_data->accessFaces()[faceId];

            // Check that valid neighbors
            if ( !m_buildCellConnectivity || (cellFace.m_negCellId.isValid() && cellFace.m_posCellId.isValid()) )
            {
                if ( !m_buildCellConnectivity || getCell(cellFace.m_negCellId).isEmpty() != getCell(cellFace.m_posCellId).isEmpty() )
                {
                    // This is a boundary face !! Get the poly out of it
                    const int numVbefore = polysVertexIds.getSize();
                    createPolygonFromFace(cellId, faceId, polysVertexIds, ( triBoundaryPlaneIds ) ? &polysBPlaneIds : HK_NULL);

                    polysMat.pushBack(cellFace.m_materialId);

                    if ( triSupportPlaneIds )
                    {
                        polysPlaneIds.pushBack(getCellFacePlaneId(cellFace, cellId));
                    }
                    polysNumVertices.pushBack(polysVertexIds.getSize() - numVbefore);
                }
            }
        }
    }

    if ( polysNumVertices.getSize() == 0 )
    {
        return;
    }

    // Get offset and scale to be applied to vertices
    hkSimdDouble64 scaleD;
    hkVector4d offsetD;
    {
        HK_ALIGN16(hkDouble64) doubleBuff[4];
        hkVector4 offset = m_planes->getPositionOffset();
        offset.store<4, HK_IO_NATIVE_ALIGNED>(doubleBuff);  offsetD.load<4, HK_IO_NATIVE_ALIGNED>(doubleBuff);
        hkSimdReal scale = m_planes->getPositionScale();
        scaleD.setFromFloat(scale.getReal());
        scaleD.setReciprocal(scaleD);
    }

    // We've gathered all the polys... Get the unique vertices array
    hkArray<VertexId> vertexRampTable;
    {
        hkArray<VertexId> uniqueVertices;
        uniqueVertices.append(polysVertexIds);
        hkSort(uniqueVertices.begin(), uniqueVertices.getSize());
        vertexRampTable.setSize(uniqueVertices[uniqueVertices.getSize() - 1] + 1, VertexId(-1));
        uniqueVertices.setSize(hkAlgorithm::removeDuplicatesFromSortedList(uniqueVertices.begin(), uniqueVertices.getSize()));
        const int nv = uniqueVertices.getSize();
        cellGeomOut.m_vertices.setSize(nv);
        if ( verticesIntPos )
        {
            verticesIntPos->setSize(nv);
        }
        for (int v = 0 ; v < nv ; v++)
        {
            vertexRampTable[uniqueVertices[v]] = v;

            // Create the vertex in world space
            hkVector4d fvd = m_data->accessVertices()[uniqueVertices[v]].m_pos;
            if ( verticesIntPos )
            {
                (*verticesIntPos)[v].setConvertF32toU32(fvd);
            }
            hkFloat32 realBuff[4];
            fvd.mul(scaleD);    fvd.add(offsetD);   fvd.store<4, HK_IO_NATIVE_ALIGNED>(realBuff);
            cellGeomOut.m_vertices[v].load<4, HK_IO_NATIVE_ALIGNED>(realBuff);
        }
    }

    // Triangulate each polygon
    int vIdxStart = 0;
    for (int p = 0, nPolys = polysNumVertices.getSize() ; p < nPolys ; p++)
    {
        const VertexId* polyVertexIds   = &polysVertexIds[vIdxStart];
        const int polyNumVertices       = polysNumVertices[p];

        Material* triMaterialsPtr       = triMaterials.expandBy(polyNumVertices - 2);
        PlaneId* triSupportPlaneIdsPtr  = triSupportPlaneIds ? triSupportPlaneIds->expandBy(polyNumVertices - 2) : HK_NULL;
        PlaneId* triBoundaryPlaneIdsPtr = triBoundaryPlaneIds ? triBoundaryPlaneIds->expandBy(3 * (polyNumVertices - 2)) : HK_NULL;

        for (int v = 1 ; v < polyNumVertices - 1 ; v++)
        {
            hkGeometry::Triangle& newTri =  cellGeomOut.m_triangles.expandOne();
            newTri.m_a = vertexRampTable[polyVertexIds[0]];
            newTri.m_b = vertexRampTable[polyVertexIds[v]];
            newTri.m_c = vertexRampTable[polyVertexIds[v + 1]];

            triMaterialsPtr[v - 1] = polysMat[p];

            if ( triSupportPlaneIdsPtr )
            {
                triSupportPlaneIdsPtr[v - 1] = polysPlaneIds[p];
            }
            if ( triBoundaryPlaneIdsPtr )
            {
                triBoundaryPlaneIdsPtr[3 * (v - 1) + 0] = ( v == 1 ) ? polysBPlaneIds[vIdxStart] : PlaneId::invalid();                                          // boundary plane for edge AB
                triBoundaryPlaneIdsPtr[3 * (v - 1) + 1] = polysBPlaneIds[vIdxStart + v];                                                                        // boundary plane for edge BC
                triBoundaryPlaneIdsPtr[3 * (v - 1) + 2] = ( v == polyNumVertices - 2 ) ? polysBPlaneIds[vIdxStart + polyNumVertices - 1] : PlaneId::invalid();  // boundary plane for edge CA
            }
        }

        vIdxStart += polyNumVertices;
    }
}

//
//  Gather unique vertex ids for the given cells

void hkcdConvexCellsTree3D::collectUniqueVertexIds(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<VertexId>& vertexIdsOut) const
{
    // Collect ids
    for (int c = 0 ; c < numCellIds ; c++)
    {
        const Cell& cell = getCell(cellIds[c]);
        vertexIdsOut.append(cell.getVertexIdsPointer(), cell.getNumVertices());
    }

    // Keep only unique ones
    if ( numCellIds > 1 )
    {
        hkSort(vertexIdsOut.begin(), vertexIdsOut.getSize());
        vertexIdsOut.setSize(hkAlgorithm::removeDuplicatesFromSortedList(vertexIdsOut.begin(), vertexIdsOut.getSize()));
    }
}

//
//  Compute and creates boundary polygons for a given set of cells. Requires connectivity

hkResult hkcdConvexCellsTree3D::extractBoundaryPolygonsFromCellIds(const hkArray<CellId>& cellIdsIn, hkcdPlanarGeometry& geomOut, hkArray<PolygonId>& boundaryPolygonIdsOut) const
{
    HK_ASSERT_NO_MSG(0xc568a3e0, geomOut.getPolygons().getNumPolygons() == 0);

    // Loop over all the faces
    for (int cIdx = cellIdsIn.getSize() - 1; cIdx >= 0; cIdx--)
    {
        const CellId cellId = cellIdsIn[cIdx];
        const Cell& cell = getCell(cellIdsIn[cIdx]);

        // Loop over the faces, and get the boundary ones
        for (int fIdx = cell.getNumFaces() - 1; fIdx >= 0; fIdx--)
        {
            const FaceId faceId = cell.getFaceIdsPointer()[fIdx];
            const Face& cellFace = m_data->accessFaces()[faceId];

            // Check that valid neighbors
            if (cellFace.m_negCellId.isValid() && cellFace.m_posCellId.isValid())
            {
                // Check that one is in and the other out
                if (getCell(cellFace.m_negCellId).isEmpty() != getCell(cellFace.m_posCellId).isEmpty())
                {
                    // This is a boundary face !! Get the poly out of it
                    hkcdConvexCellsTree3D::PolygonId polyId = createPolygonFromFace(cellId, faceId, geomOut);
                    if( polyId.isValid() )
                    {
                        boundaryPolygonIdsOut.pushBack(polyId);
                    }
                    else
                    {
                        return HK_FAILURE;
                    }
                }
            }
        }
    }

    // Reset the vertex cache of the geometry
    hkArray<hkVector4d> verticesPos(m_data->getNumAllocatedVertices());
    for (int vIdx = 0 ; vIdx < m_data->getNumAllocatedVertices() ; vIdx++)
    {
        verticesPos[vIdx] = m_data->accessVertices()[vIdx].m_pos;
    }
    geomOut.setVerticesCacheFromArray(verticesPos);

    return HK_SUCCESS;
}

//
//  Creates a box cell that encloses the entire "known" space

hkcdConvexCellsTree3D::CellId hkcdConvexCellsTree3D::createBoundaryCell()
{
    // allocate the cell
    CellId cellId = m_cells->allocCell();
    accessCell(cellId).setSizes(6, 8, 12);

    // Add the 6 faces
    for (int f = 0 ; f < 6 ; f++)
    {
        FaceId newFaceId;
        Face& newFace = m_data->allocateNewFace(newFaceId);
        newFace.m_planeId       = PlaneId(f);
        newFace.m_negCellId     = cellId;
        newFace.m_posCellId     = CellId::invalid();
        newFace.m_materialId    = Material::invalid();
        newFace.m_flags         |= FACE_SURFACE_INVALID;
    }

    // Add the 8 extremity points
    const VertexId PT_ID_NX_NY_NZ = 0;
    const VertexId PT_ID_NX_NY_PZ = 1;
    const VertexId PT_ID_NX_PY_NZ = 2;
    const VertexId PT_ID_NX_PY_PZ = 3;
    const VertexId PT_ID_PX_NY_NZ = 4;
    const VertexId PT_ID_PX_NY_PZ = 5;
    const VertexId PT_ID_PX_PY_NZ = 6;
    const VertexId PT_ID_PX_PY_PZ = 7;
    const Bounds vPlaneIds[8][3] =
    {
        {PlanesCollection::BOUND_NEG_X, PlanesCollection::BOUND_NEG_Y, PlanesCollection::BOUND_NEG_Z},
        {PlanesCollection::BOUND_NEG_X, PlanesCollection::BOUND_NEG_Y, PlanesCollection::BOUND_POS_Z},
        {PlanesCollection::BOUND_NEG_X, PlanesCollection::BOUND_POS_Y, PlanesCollection::BOUND_NEG_Z},
        {PlanesCollection::BOUND_NEG_X, PlanesCollection::BOUND_POS_Y, PlanesCollection::BOUND_POS_Z},
        {PlanesCollection::BOUND_POS_X, PlanesCollection::BOUND_NEG_Y, PlanesCollection::BOUND_NEG_Z},
        {PlanesCollection::BOUND_POS_X, PlanesCollection::BOUND_NEG_Y, PlanesCollection::BOUND_POS_Z},
        {PlanesCollection::BOUND_POS_X, PlanesCollection::BOUND_POS_Y, PlanesCollection::BOUND_NEG_Z},
        {PlanesCollection::BOUND_POS_X, PlanesCollection::BOUND_POS_Y, PlanesCollection::BOUND_POS_Z},
    };
    for (int v = 0 ; v < 8 ; v++)
    {
        VertexId newVertexId;
        Vertex& newVertex = m_data->allocateNewVertex(newVertexId);
        Plane planes[3];
        for (int k = 0 ; k < 3 ; k++)
        {
            newVertex.m_planeIds[k] = PlaneId(vPlaneIds[v][k]);
            m_planes->getPlane(newVertex.m_planeIds[k], planes[k]);
        }
        hkcdPlanarGeometryPredicates::approximateIntersectionFast(planes, newVertex.m_pos);
    }

    // Add the 12 edges
    const hkUint32 vEdgePtAndFaceIds[12][4]=
    {
        {PT_ID_NX_NY_NZ, PT_ID_PX_NY_NZ, hkUint32(PlanesCollection::BOUND_NEG_Z), hkUint32(PlanesCollection::BOUND_NEG_Y)},
        {PT_ID_NX_NY_NZ, PT_ID_NX_NY_PZ, hkUint32(PlanesCollection::BOUND_NEG_Y), hkUint32(PlanesCollection::BOUND_NEG_X)},
        {PT_ID_NX_NY_PZ, PT_ID_PX_NY_PZ, hkUint32(PlanesCollection::BOUND_NEG_Y), hkUint32(PlanesCollection::BOUND_POS_Z)},
        {PT_ID_PX_NY_PZ, PT_ID_PX_NY_NZ, hkUint32(PlanesCollection::BOUND_NEG_Y), hkUint32(PlanesCollection::BOUND_POS_X)},

        {PT_ID_PX_NY_NZ, PT_ID_PX_PY_NZ, hkUint32(PlanesCollection::BOUND_NEG_Z), hkUint32(PlanesCollection::BOUND_POS_X)},
        {PT_ID_NX_NY_NZ, PT_ID_NX_PY_NZ, hkUint32(PlanesCollection::BOUND_NEG_X), hkUint32(PlanesCollection::BOUND_NEG_Z)},
        {PT_ID_NX_NY_PZ, PT_ID_NX_PY_PZ, hkUint32(PlanesCollection::BOUND_POS_Z), hkUint32(PlanesCollection::BOUND_NEG_X)},
        {PT_ID_PX_NY_PZ, PT_ID_PX_PY_PZ, hkUint32(PlanesCollection::BOUND_POS_X), hkUint32(PlanesCollection::BOUND_POS_Z)},

        {PT_ID_NX_PY_NZ, PT_ID_NX_PY_PZ, hkUint32(PlanesCollection::BOUND_NEG_X), hkUint32(PlanesCollection::BOUND_POS_Y)},
        {PT_ID_NX_PY_PZ, PT_ID_PX_PY_PZ, hkUint32(PlanesCollection::BOUND_POS_Z), hkUint32(PlanesCollection::BOUND_POS_Y)},
        {PT_ID_PX_PY_PZ, PT_ID_PX_PY_NZ, hkUint32(PlanesCollection::BOUND_POS_X), hkUint32(PlanesCollection::BOUND_POS_Y)},
        {PT_ID_PX_PY_NZ, PT_ID_NX_PY_NZ, hkUint32(PlanesCollection::BOUND_NEG_Z), hkUint32(PlanesCollection::BOUND_POS_Y)},
    };
    for (int e = 0 ; e < 12 ; e++)
    {
        EdgeId newEdgeId;
        Edge& newEdge = m_data->allocateNewEdge(newEdgeId);
        for (int k = 0 ; k < 2 ; k++)
        {
            newEdge.m_vertexIds[k]  = VertexId(vEdgePtAndFaceIds[e][k]);
            newEdge.m_faceIds[k]    = FaceId(vEdgePtAndFaceIds[e][k + 2]);

            // Connectivity info
            newEdge.m_indexInCell   = e;
            newEdge.m_owningCellId  = cellId;
        }
    }

    // Copy the data in the cell
    Cell& newCell       = accessCell(cellId);
    FaceId* faceIds = newCell.accessFaceIdsPointer();
    for (int f = 0 ; f < 6 ; f++)
    {
        faceIds[f] = FaceId(f);
    }
    VertexId* vertIds = newCell.accessVertexIdsPointer();
    for (int v = 0 ; v < 8 ; v++)
    {
        vertIds[v] = VertexId(v);
    }
    EdgeId* edgeIds = newCell.accessEdgeIdsPointer();
    for (int e = 0 ; e < 12 ; e++)
    {
        edgeIds[e] = EdgeId(e);
    }

#if defined HK_DEBUG
    checkCellIntegrity(cellId);
#endif
    // Return the Id of the boundary cell
    return cellId;
}

namespace hkcdConvexCellsTree3DImpl
{
    //
    // Internal helper for splitCell: updates the face split vertex buffer (Each split face can be have at most 2 split points)

    static HK_INLINE void HK_CALL addVertexSplitOnFace(
                    const int faceIdx, hkLocalBuffer<char>& facesNumVertSplit, hkLocalBuffer<int>& facesFirstPointInEdgeId, hkLocalBuffer<int>& facesSplitVertsIds,
                    const hkcdConvexCellsTree3D::VertexId newVertexId, const hkcdConvexCellsTree3D::EdgeId edgeIdIn)
    {
        if ( facesNumVertSplit[faceIdx] == 0 )
        {
            HK_ASSERT_NO_MSG(0xc12a3ae6, facesFirstPointInEdgeId[faceIdx] == -1);
            facesFirstPointInEdgeId[faceIdx] = edgeIdIn;
        }
        facesSplitVertsIds[(faceIdx << 1) + facesNumVertSplit[faceIdx]] = newVertexId;
        facesNumVertSplit[faceIdx]++;
        HK_ASSERT_NO_MSG(0xc12a35e6, facesNumVertSplit[faceIdx] <= 2);
    }


}

//
//  Internal helper for splitCell: given a staring point for a new edge, and an adjacent inside edge, decide the winding of the edge.
//  Returns true if edge should be flipped ( flipped = in plane on left side of new edge )

HK_INLINE bool hkcdConvexCellsTree3D::shouldFlipNewEdgeForInFace(const Edge& adjInEdge, const VertexId newEdgeFirstVertexId, const FaceId inFaceId) const
{
    HK_ASSERT_NO_MSG(0xc12a35f0, adjInEdge.m_vertexIds[0] == newEdgeFirstVertexId || adjInEdge.m_vertexIds[1] == newEdgeFirstVertexId);
    HK_ASSERT_NO_MSG(0xc12a35a2, adjInEdge.m_faceIds[0] == inFaceId || adjInEdge.m_faceIds[1] == inFaceId);
    const bool inEdgeFlipped = ( adjInEdge.m_vertexIds[0] == newEdgeFirstVertexId );
    const bool inPlaneOnRightSide = ( adjInEdge.m_faceIds[1] == inFaceId );

    // If inPlaneOnRightSide, the adjacent edge is oriented properly w.r.t the inside face,
    // we need to flip only if first point of th new edge doesn't match last point of adjacent edge
    return (inPlaneOnRightSide && inEdgeFlipped) || (!inPlaneOnRightSide && !inEdgeFlipped);
}

//
//  Splits the given cell by the given plane. Updates all adjacency information

hkcdConvexCellsTree3D::FaceId hkcdConvexCellsTree3D::splitCell(CellId cellId, PlaneId splitPlaneId, const Material& splitMaterial, CellId& insideCellIdOut, CellId& outsideCellIdOut, bool repaintOnlyInvalidMat)
{
    insideCellIdOut         = CellId::invalid();
    outsideCellIdOut        = CellId::invalid();
    FaceId newFaceId        = FaceId(-1);

    // Get the cell
    Cell* cell              = &accessCell(cellId);

    // First of all, classify all the vertices of the cell
    const int numVertices   = cell->getNumVertices();
    const VertexId* vertIds = cell->getVertexIdsPointer();

    // Compute orientation values
    int numVertsIn = 0, numVertsOut = 0, numVertsInFront = 0, numVertsBehind = 0, numVertsOnPlane = 0;
    hkLocalBuffer<VertexId> verticesInOut(numVertices << 2);
    VertexId* verticesIn    = &verticesInOut[0];
    VertexId* verticesOut   = &verticesInOut[numVertices << 1];
    {
        Plane planes[3];
        Plane splittingPlane;
        m_planes->getPlane(splitPlaneId, splittingPlane);
        hkIntVector vPlaneIds;
        for (int v = 0 ; v < numVertices ; v++)
        {
            Vertex& vertex = m_data->accessVertices()[vertIds[v]];
            vPlaneIds.set(vertex.m_planeIds[0].value(), vertex.m_planeIds[1].value(), vertex.m_planeIds[2].value(), splitPlaneId.value());
            // Get the involved planes
            m_planes->getPlane(vertex.m_planeIds[0], planes[0]);
            m_planes->getPlane(vertex.m_planeIds[1], planes[1]);
            m_planes->getPlane(vertex.m_planeIds[2], planes[2]);

            // compute orientation
            Orientation ori = hkcdPlanarGeometryPredicates::orientation(vertex.m_pos, planes[0], planes[1], planes[2], splittingPlane, vPlaneIds, m_planes->getOrientationCache());

            // store the result in the w component of the vertex
            vertex.m_pos.setInt24W(ori);

            // add vertices to result array
            switch (ori)
            {
            case hkcdPlanarGeometryPredicates::BEHIND:          { verticesIn[numVertsIn++] = vertIds[v];    numVertsBehind++;   break;  }
            case hkcdPlanarGeometryPredicates::IN_FRONT_OF:     { verticesOut[numVertsOut++] = vertIds[v];  numVertsInFront++;  break;  }
            case hkcdPlanarGeometryPredicates::ON_PLANE:
                {
                    verticesIn[numVertsIn++] = vertIds[v];
                    verticesOut[numVertsOut++] = vertIds[v];
                    numVertsOnPlane++;
                }
                break;

            default:
                break;
            }
        }
    }

    // Check if the cell is split by the plane
    if ( numVertsInFront > 0 && numVertsBehind > 0 )
    {
        if ( m_preserveNonLeafCells )
        {
            // If we want to preserve the non-leaf cells, clone the cell before splitting it so that
            // it doesn't get altered
            cellId = cloneCell(cellId);
        }

        insideCellIdOut         = m_cells->allocCell();
        outsideCellIdOut        = m_cells->allocCell();

        // Reset every face
        cell                    = &accessCell(cellId);      // The address may have changed due to allocation
        cell->setLeaf(false);

        const int numFaces      = cell->getNumFaces();
        const FaceId* faceIds   = cell->getFaceIdsPointer();
        // Face temporary additionnal info
        hkLocalBuffer<int> facesSplitInfoIn(numFaces);              // For each split face the id of the in face part (-1 if face is not split)
        hkLocalBuffer<int> facesSplitInfoOut(numFaces);             // For each split face the id of the ou face part (-1 if face is not split)
        hkLocalBuffer<char> facesNumVertSplit(numFaces);            // For each face the number of vertices forming the splitting edge (0 for non split faces)
        hkLocalBuffer<int> facesNumEdgesInFront(numFaces);          // >0 if the face has edge (or edges parts) in front of the splitting plane
        hkLocalBuffer<int> facesNumEdgesBehind(numFaces);           // >0 if the face has edge (or edges parts) in front of the splitting plane
        hkLocalBuffer<int> facesFirstPointInEdgeId(numFaces);       // Id of the in edge sharing a vertex with the first vertex of the spit segment of the face
        hkLocalBuffer<Orientation> facesOrientation(numFaces);      // For each face, corresponding orientation (undefined for split faces)
        hkLocalBuffer<int> facesSplitVertsIds(numFaces*2);          // Id of the potentially two vertices forming the splitting edge for this faces (undefined if plane is not split)

        hkString::memSet4(facesSplitInfoIn.begin(), -1, numFaces);
        hkString::memSet4(facesSplitInfoOut.begin(), -1, numFaces);
        hkString::memSet(facesNumVertSplit.begin(), 0, numFaces);
        hkString::memSet4(facesNumEdgesInFront.begin(), 0, numFaces);
        hkString::memSet4(facesNumEdgesBehind.begin(), 0, numFaces);
        hkString::memSet4(facesFirstPointInEdgeId.begin(), -1, numFaces);
#ifdef HK_DEBUG
        hkString::memSet4(facesSplitVertsIds.begin(), -1, numFaces * 2);
        hkString::memSet(facesOrientation.begin(), -1, numFaces*sizeof(Orientation));
#endif

        for (int f = 0 ; f < numFaces ; f++)
        {
            m_data->accessFaces()[faceIds[f]].m_userData = hkUint16(f);
        }
        hkLocalBuffer<FaceId> facesInOut((numFaces + 1) * 2);
        FaceId* facesIn         = &facesInOut[0];
        FaceId* facesOut        = &facesInOut[numFaces + 1];

        // Inits
        const int numEdges      = cell->getNumEdges();
        const EdgeId* edgeIds   = cell->getEdgeIdsPointer();
        hkLocalBuffer<EdgeId> edgesInOut(numEdges << 2);
        EdgeId* edgesIn         = &edgesInOut[0];
        EdgeId* edgesOut        = &edgesInOut[numEdges << 1];
        int numFacesIn = 0, numFacesOut = 0, numEdgesIn = 0, numEdgesOut = 0;

        // Trace of all edges split for deallocation
        hkLocalArray<int> edgeSplitIds(numEdges);

        // Edge info in case of connectivity
        const int numEdgeInfo   = ( m_buildCellConnectivity ) ? numEdges : 0;
        const int numFaceInfo   = ( m_buildCellConnectivity ) ? numFaces : 0;
        hkLocalBuffer<int> edgesSplitInfoIn(numEdgeInfo);           // For each split edge the id of the in edge part (-1 if edge at index is not split)
        hkLocalBuffer<int> edgesSplitInfoOut(numEdgeInfo);          // For each split edge the id of the out edge part (-1 if edge at index is not split)
        hkLocalBuffer<int> facesNumEdgesSplit(numFaceInfo);         // For each face the number of edges cut.
        hkLocalBuffer<int> facesEdgeSplitIdx(numFaceInfo*2);        // For each face the index (NOT id!) in local cell of the potentially two edges cut
        if ( m_buildCellConnectivity )
        {
            hkString::memSet4(edgesSplitInfoIn.begin(), -1, numEdges);
            hkString::memSet4(edgesSplitInfoOut.begin(), -1, numEdges);
            hkString::memSet4(facesNumEdgesSplit.begin(), 0, numFaceInfo);
#ifdef HK_DEBUG
            hkString::memSet4(facesEdgeSplitIdx.begin(), -1, numFaceInfo*2);
#endif
        }

        // Create the new face created by the cut. Note: we only create one face for the two new cells, this face will be shared (no need to create two)
        {
            Face& newFace           = m_data->allocateNewFace(newFaceId);
            newFace.m_planeId       = splitPlaneId;
            newFace.m_negCellId     = insideCellIdOut;
            newFace.m_posCellId     = outsideCellIdOut;
            newFace.m_materialId    = splitMaterial;
        }
        facesIn[numFacesIn++]   = newFaceId;
        facesOut[numFacesOut++] = newFaceId;

        // Cut edges
        bool hasOnPlaneEdges = false;
        for (int e = 0 ; e < numEdges ; e++)
        {
            const EdgeId edgeId     = edgeIds[e];
            const Edge* cEdge       = &(m_data->accessEdges()[edgeId]);
            const Vertex& vtxStart  = m_data->accessVertices()[cEdge->m_vertexIds[0]];
            const Vertex& vtxEnd    = m_data->accessVertices()[cEdge->m_vertexIds[1]];
            Orientation oriPt[2];
            oriPt[0]    = Orientation(vtxStart.m_pos.getInt16W());
            oriPt[1]    = Orientation(vtxEnd.m_pos.getInt16W());

            // update face split info
            for (int k = 0 ; k < 2 ; k++)
            {
                const Face& cFace   = m_data->accessFaces()[cEdge->m_faceIds[k]];
                const int faceIdx   = cFace.m_userData;
                //for (int t = 0 ; t < 2 ; t++)
                {
                    if ( oriPt[0] == hkcdPlanarGeometryPredicates::IN_FRONT_OF )    facesNumEdgesInFront[faceIdx]++;
                    if ( oriPt[0] == hkcdPlanarGeometryPredicates::BEHIND )         facesNumEdgesBehind[faceIdx]++;
                }
                {
                    if ( oriPt[1] == hkcdPlanarGeometryPredicates::IN_FRONT_OF )    facesNumEdgesInFront[faceIdx]++;
                    if ( oriPt[1] == hkcdPlanarGeometryPredicates::BEHIND )         facesNumEdgesBehind[faceIdx]++;
                }
            }

            // Classify and split the edge
            if (   (oriPt[0] == hkcdPlanarGeometryPredicates::IN_FRONT_OF && oriPt[1] == hkcdPlanarGeometryPredicates::BEHIND)
                || (oriPt[1] == hkcdPlanarGeometryPredicates::IN_FRONT_OF && oriPt[0] == hkcdPlanarGeometryPredicates::BEHIND) )
            {
                // The edge is crossed by the plane, split it in two edges
                const bool flipped = (oriPt[0] == hkcdPlanarGeometryPredicates::IN_FRONT_OF);
                EdgeId edgeIdIn;
                EdgeId edgeIdOut;
                VertexId newVertexId;
                splitEdge(edgeId, flipped, splitPlaneId, edgeIdIn, edgeIdOut, newVertexId);
                edgeSplitIds.pushBack(edgeId);
                cEdge                       = &(m_data->accessEdges()[edgeId]);
                verticesIn[numVertsIn++]    = newVertexId;
                verticesOut[numVertsOut++]  = newVertexId;

                // In connectivity mode, keep track of how edges were split
                if ( m_buildCellConnectivity )
                {
                    edgesSplitInfoIn[e]     = edgeIdIn;
                    edgesSplitInfoOut[e]    = edgeIdOut;
                    for (int k = 0; k < 2; k++)
                    {
                        const Face& cFace = m_data->accessFaces()[cEdge->m_faceIds[k]];
                        const int faceIdx = cFace.m_userData;
                        HK_ASSERT_NO_MSG(0x1a3d5e88, facesNumEdgesSplit[faceIdx] <= 2);
                        facesEdgeSplitIdx[(faceIdx << 1) + facesNumEdgesSplit[faceIdx]++] = e;
                    }
                }

                // update vertex ids for the adjacent faces of the edge
                for (int k = 0 ; k < 2 ; k++)
                {
                    const Face& cFace   = m_data->accessFaces()[cEdge->m_faceIds[k]];
                    const int faceIdx   = cFace.m_userData;
                    hkcdConvexCellsTree3DImpl::addVertexSplitOnFace(faceIdx, facesNumVertSplit, facesFirstPointInEdgeId, facesSplitVertsIds, newVertexId, edgeIdIn);
                }

                // Add the inside and outside edges
                edgesIn[numEdgesIn++]   = edgeIdIn;
                edgesOut[numEdgesOut++] = edgeIdOut;
            }
            else if ( oriPt[0] == hkcdPlanarGeometryPredicates::BEHIND || oriPt[1] == hkcdPlanarGeometryPredicates::BEHIND )
            {
                edgesIn[numEdgesIn++]   = edgeId;
                for (int k = 0 ; k < 2 ; k++)
                {
                    const Face& cFace = m_data->accessFaces()[cEdge->m_faceIds[k]];
                    const int faceIdx   = cFace.m_userData;
                    facesOrientation[faceIdx] = hkcdPlanarGeometryPredicates::BEHIND;

                    // only behind edges touching the planes can add point split to faces (by convention... could have been the in front ones)
                    // There should be at most 2 by face (ignoring the on plane edges)
                    if ( oriPt[0] == hkcdPlanarGeometryPredicates::ON_PLANE ||  oriPt[1] == hkcdPlanarGeometryPredicates::ON_PLANE )
                    {
                        const VertexId newVertexId = ( oriPt[0] == hkcdPlanarGeometryPredicates::ON_PLANE ) ? cEdge->m_vertexIds[0] : cEdge->m_vertexIds[1];
                        hkcdConvexCellsTree3DImpl::addVertexSplitOnFace(faceIdx, facesNumVertSplit, facesFirstPointInEdgeId, facesSplitVertsIds, newVertexId, edgeId);
                    }
                }
            }
            else if ( oriPt[0] == hkcdPlanarGeometryPredicates::IN_FRONT_OF || oriPt[1] == hkcdPlanarGeometryPredicates::IN_FRONT_OF )
            {
                edgesOut[numEdgesOut++] = edgeId;
                for (int k = 0 ; k < 2 ; k++)
                {
                    const Face& cFace = m_data->accessFaces()[cEdge->m_faceIds[k]];
                    const int faceIdx   = cFace.m_userData;
                    facesOrientation[faceIdx] = hkcdPlanarGeometryPredicates::IN_FRONT_OF;
                }
            }
            else
            {
                hasOnPlaneEdges = true;
                //We will take care of it later as wee need to know the faces orientation to create the new edges
            }
        }

        // Treat on plane edges (after the cut edge loop, the faces orientation are well determined)
        if ( hasOnPlaneEdges )
        {
            for (int e = 0 ; e < numEdges ; e++)
            {
                const Edge* cEdge       = &(m_data->accessEdges()[edgeIds[e]]);
                if ( cEdge->m_vertexIds[0] == VertexId(-1) )
                {
                    continue;   // Edge has been split, and is therefore not on plane...
                }

                const Vertex& vtxStart  = m_data->accessVertices()[cEdge->m_vertexIds[0]];
                const Vertex& vtxEnd    = m_data->accessVertices()[cEdge->m_vertexIds[1]];
                Orientation oriPt[2];
                oriPt[0]    = Orientation(vtxStart.m_pos.getInt24W());
                oriPt[1]    = Orientation(vtxEnd.m_pos.getInt24W());
                if ( oriPt[0] == hkcdPlanarGeometryPredicates::ON_PLANE && oriPt[1] == hkcdPlanarGeometryPredicates::ON_PLANE )
                {
                    // Get the inside and the outside face of the edge
                    FaceId inFaceId     = FaceId(-1);
                    FaceId outFaceId    = FaceId(-1);
                    for (int t = 0 ; t < 2 ; t++)
                    {
                        const int faceIdx = m_data->accessFaces()[cEdge->m_faceIds[t]].m_userData;
                        HK_ASSERT_NO_MSG(0xa36e8f5b, facesSplitInfoIn[faceIdx] == -1);      // Face shouldn't be marked as split
                        if ( facesOrientation[faceIdx] == hkcdPlanarGeometryPredicates::BEHIND )
                        {
                            inFaceId = cEdge->m_faceIds[t];
                        }
                        else
                        {
                            outFaceId = cEdge->m_faceIds[t];
                        }
                    }
                    HK_ASSERT_NO_MSG(0xc3abb1e2, inFaceId != FaceId(-1) && outFaceId != FaceId(-1));
                    HK_ASSERT_NO_MSG(0xc3abb2e2, facesNumVertSplit[int(m_data->accessFaces()[inFaceId].m_userData)] == 2);

                    // On plane Create two new edges, one for each cell
                    EdgeId newInEdgeId;
                    {
                        Edge& newEdge           = m_data->allocateNewEdge(newInEdgeId);
                        cEdge                   = &(m_data->accessEdges()[edgeIds[e]]);
                        const bool flip         = ( cEdge->m_faceIds[0] == outFaceId ); // Form the first edge so that it always has inFace on left
                        newEdge.m_vertexIds[0]  = cEdge->m_vertexIds[( flip ) ? 1 : 0];
                        newEdge.m_vertexIds[1]  = cEdge->m_vertexIds[( flip ) ? 0 : 1];
                        newEdge.m_faceIds[0]    = inFaceId;
                        newEdge.m_faceIds[1]    = newFaceId;
                        edgesIn[numEdgesIn++]   = newInEdgeId;
                        // We need to unlink the edge of this cell as it is not split
                        replaceEdgeInLinkedChain(edgeIds[e], newInEdgeId);
                    }
                    {
                        EdgeId newEdgeId;
                        Edge& newEdge           = m_data->allocateNewEdge(newEdgeId);
                        const Edge& inEdge      = m_data->accessEdges()[newInEdgeId];
                        newEdge.m_vertexIds[0]  = inEdge.m_vertexIds[1];
                        newEdge.m_vertexIds[1]  = inEdge.m_vertexIds[0];
                        newEdge.m_faceIds[0]    = outFaceId;
                        newEdge.m_faceIds[1]    = newFaceId;
                        edgesOut[numEdgesOut++] = newEdgeId;
                        linkEdges(newEdgeId, newInEdgeId);
                    }
                }
            }
        }

        // Determine split faces (faces that have edges both behind and in front)
        // For each split face, allocate two new faces: the in and out one
        for (int f = 0 ; f < numFaces ; f++)
        {
            Face* cFace         = &(m_data->accessFaces()[faceIds[f]]);
            const int faceIdx   = cFace->m_userData;
            if ( facesNumEdgesBehind[faceIdx] > 0 && facesNumEdgesInFront[faceIdx] > 0 )
            {
                const bool isCellOnNegSide = ( cFace->m_negCellId == cellId );

                Face tmpFace;
                if ( m_buildCellConnectivity )
                {
                    tmpFace.m_negCellId = cFace->m_negCellId;
                    tmpFace.m_posCellId = cFace->m_posCellId;
                }
                else
                {
                    tmpFace.m_negCellId = CellId::invalid();
                    tmpFace.m_posCellId = CellId::invalid();
                }

                // Face is split, create the two sub-faces
                Face& inFace                = m_data->allocateNewFace(((FaceId*)(facesSplitInfoIn.begin()))[faceIdx]);
                facesIn[numFacesIn++]       = facesSplitInfoIn[faceIdx];
                cFace                       = &(m_data->accessFaces()[faceIds[f]]);
                inFace.m_planeId            = cFace->m_planeId;
                inFace.m_materialId         = cFace->m_materialId;
                inFace.m_negCellId          = (  isCellOnNegSide ) ? insideCellIdOut : tmpFace.m_negCellId;
                inFace.m_posCellId          = ( !isCellOnNegSide ) ? insideCellIdOut : tmpFace.m_posCellId;

                Face& outFace               = m_data->allocateNewFace(((FaceId*)(facesSplitInfoOut.begin()))[faceIdx]);
                facesOut[numFacesOut++]     = facesSplitInfoOut[faceIdx];
                cFace                       = &(m_data->accessFaces()[faceIds[f]]);
                outFace.m_planeId           = cFace->m_planeId;
                outFace.m_materialId        = cFace->m_materialId;
                outFace.m_negCellId         = (  isCellOnNegSide ) ? outsideCellIdOut : tmpFace.m_negCellId;
                outFace.m_posCellId         = ( !isCellOnNegSide ) ? outsideCellIdOut : tmpFace.m_posCellId;

                // invalidate split face
                cFace->m_negCellId          = (  isCellOnNegSide ) ? CellId::invalid() : cFace->m_negCellId;
                cFace->m_posCellId          = ( !isCellOnNegSide ) ? CellId::invalid() : cFace->m_posCellId;
            }
        }

        // Remap the face ids on all the edges by replacing references to split faces with the in and out ones
        // After this step only uncut faces should be still referenced by the new edges
        // Note that we don't create new edges, but rather overwrite the existing one... Only leaf cells are valid!!!
        for (int e = 0 ; e < numEdgesIn ; e++)
        {
            Edge& cEdge = m_data->accessEdges()[edgesIn[e]];
            for (int k = 0 ; k < 2 ; k++)
            {
                if ( cEdge.m_faceIds[k] == newFaceId )  continue;
                const Face& cFace   = m_data->accessFaces()[cEdge.m_faceIds[k]];
                const int faceIdx   = cFace.m_userData;
                if ( facesSplitInfoIn[faceIdx] >= 0 )
                {
                    cEdge.m_faceIds[k] = facesSplitInfoIn[faceIdx];
                }
            }
        }
        for (int e = 0 ; e < numEdgesOut ; e++)
        {
            Edge& cEdge = m_data->accessEdges()[edgesOut[e]];
            for (int k = 0 ; k < 2 ; k++)
            {
                if ( cEdge.m_faceIds[k] == newFaceId )  continue;
                const Face& cFace   = m_data->accessFaces()[cEdge.m_faceIds[k]];
                const int faceIdx   = cFace.m_userData;
                if ( facesSplitInfoOut[faceIdx] >= 0 )
                {
                    cEdge.m_faceIds[k] = facesSplitInfoOut[faceIdx];
                }
            }
        }

        // Allocate newly created edges, and add uncut faces to the in/out faces array
        for (int f = 0 ; f < numFaces ; f++)
        {
            // Check if the face is split
            Face& face = m_data->accessFaces()[faceIds[f]];
            const int faceIdx = face.m_userData;

            if ( facesSplitInfoIn[faceIdx] >= 0 )
            {
                HK_ASSERT_NO_MSG(0x1f3e2b55, facesNumVertSplit[faceIdx] == 2);

                // Create two new edges with the two new vertices (intersection between the two edges and the split plane)
                EdgeId newInEdgeId;
                {
                    Edge& newEdge           = m_data->allocateNewEdge(newInEdgeId);
                    newEdge.m_vertexIds[0]  = VertexId(facesSplitVertsIds[(faceIdx << 1) + 0]);
                    newEdge.m_vertexIds[1]  = VertexId(facesSplitVertsIds[(faceIdx << 1) + 1]);
                    newEdge.m_faceIds[0]    = newFaceId;
                    newEdge.m_faceIds[1]    = facesSplitInfoIn[faceIdx];
                    edgesIn[numEdgesIn++]   = newInEdgeId;
                    const Edge& adjInEdge   = m_data->accessEdges()[facesFirstPointInEdgeId[faceIdx]];
                    if ( shouldFlipNewEdgeForInFace(adjInEdge, newEdge.m_vertexIds[0], facesSplitInfoIn[faceIdx]) )
                    {
                        hkAlgorithm::swap(newEdge.m_vertexIds[0], newEdge.m_vertexIds[1]);
                    }
                }
                EdgeId newEdgeOutId;
                {
                    Edge& newEdge           = m_data->allocateNewEdge(newEdgeOutId);
                    const Edge& inEdge      = m_data->accessEdges()[newInEdgeId];
                    newEdge.m_vertexIds[0]  = inEdge.m_vertexIds[1];
                    newEdge.m_vertexIds[1]  = inEdge.m_vertexIds[0];
                    newEdge.m_faceIds[0]    = ( inEdge.m_faceIds[0] == newFaceId ) ? newFaceId : facesSplitInfoOut[faceIdx];
                    newEdge.m_faceIds[1]    = ( inEdge.m_faceIds[1] == newFaceId ) ? newFaceId : facesSplitInfoOut[faceIdx];
                    newEdge.m_firstEdgeOfChainId = newEdgeOutId;
                    linkEdges(newInEdgeId, newEdgeOutId);   // Link the two edges for connectivity
                    edgesOut[numEdgesOut++] = newEdgeOutId;
                }

                // Update face connectivity
                if ( m_buildCellConnectivity )
                {
                    HK_ASSERT_NO_MSG(0x1d2a65ee, !face.m_negCellId.isValid() || !face.m_posCellId.isValid());

                    updateConnectivity(( face.m_negCellId.isValid() ) ? face.m_negCellId : face.m_posCellId,
                        faceIds[f], facesSplitInfoIn[faceIdx], facesSplitInfoOut[faceIdx],
                        newInEdgeId, facesNumEdgesSplit[faceIdx]);
                    //, edgeIds, &facesEdgeSplitIdx[(faceIdx << 1)], (EdgeId*)edgesSplitInfoIn.begin(), (EdgeId*)edgesSplitInfoOut.begin());

                    // Free the face
                    m_data->freeFace(faceIds[f]);
                }
                else if ( !face.m_negCellId.isValid() && !face.m_posCellId.isValid() )
                {
                    // Face is no longer referenced, free it
                    m_data->freeFace(faceIds[f]);
                }
            }
            else
            {
                // Untouched face, the orientation buffer must have been initialized with the correct value
                HK_ASSERT_NO_MSG(0x9a53e22d, int(facesOrientation[faceIdx]) != -1);
                if ( facesOrientation[faceIdx] == hkcdPlanarGeometryPredicates::BEHIND )
                {
                    facesIn[numFacesIn++]   = faceIds[f];
                    face.m_negCellId        = ( face.m_negCellId == cellId ) ? insideCellIdOut : face.m_negCellId;
                    face.m_posCellId        = ( face.m_posCellId == cellId ) ? insideCellIdOut : face.m_posCellId;
                }
                else
                {
                    facesOut[numFacesOut++] = faceIds[f];
                    face.m_negCellId        = ( face.m_negCellId == cellId ) ? outsideCellIdOut : face.m_negCellId;
                    face.m_posCellId        = ( face.m_posCellId == cellId ) ? outsideCellIdOut : face.m_posCellId;
                }
            }
        }

        // Checks
        HK_ASSERT_NO_MSG(0x2ad3ff6e, numEdgesIn <= numEdges*2 && numEdgesIn >= 4);
        HK_ASSERT_NO_MSG(0x2ad3ff7f, numEdgesOut <= numEdges*2 && numEdgesOut >= 4);

        HK_ASSERT_NO_MSG(0x2bc3ff6e, numFacesIn <= numFaces + 1 && numFacesIn >= 4);
        HK_ASSERT_NO_MSG(0x2fa3ff7f, numFacesOut <= numFaces + 1 && numFacesOut >= 4);

        HK_ASSERT_NO_MSG(0x2ad3aa6e, numVertsIn <= numVertices*2 && numVertsIn >= 4);
        HK_ASSERT_NO_MSG(0x2ad3bb6e, numVertsOut <= numVertices*2 && numVertsOut >= 4);

        // Patch edges in case of connectivity
        if ( m_buildCellConnectivity )
        {
            for (int e = 0 ; e < numEdgesIn ; e++)
            {
                Edge& cEdge = m_data->accessEdges()[edgesIn[e]];
                cEdge.m_owningCellId    = insideCellIdOut;
                cEdge.m_indexInCell     = e;
            }
            for (int e = 0 ; e < numEdgesOut ; e++)
            {
                Edge& cEdge = m_data->accessEdges()[edgesOut[e]];
                cEdge.m_owningCellId    = outsideCellIdOut;
                cEdge.m_indexInCell     = e;
            }
        }

        // Deallocate split edges
        for (int eIdx = edgeSplitIds.getSize() - 1; eIdx >= 0; eIdx--)
        {
            m_data->freeEdge(edgeSplitIds[eIdx]);
        }

        // Allocate the inside and outside cells
        {
            Cell& inCell = accessCell(insideCellIdOut);
            inCell.setSizes(numFacesIn, numVertsIn, numEdgesIn);

            FaceId* fIds = inCell.accessFaceIdsPointer();
            hkString::memCpy(fIds, facesIn, numFacesIn*sizeof(FaceId));
            VertexId* vIds = inCell.accessVertexIdsPointer();
            hkString::memCpy(vIds, verticesIn, numVertsIn*sizeof(VertexId));
            EdgeId* eIds = inCell.accessEdgeIdsPointer();
            hkString::memCpy(eIds, edgesIn, numEdgesIn*sizeof(EdgeId));

        }
        {
            Cell& outCell = accessCell(outsideCellIdOut);
            outCell.setSizes(numFacesOut, numVertsOut, numEdgesOut);

            FaceId* fIds = outCell.accessFaceIdsPointer();
            hkString::memCpy(fIds, facesOut, numFacesOut*sizeof(FaceId));
            VertexId* vIds = outCell.accessVertexIdsPointer();
            hkString::memCpy(vIds, verticesOut, numVertsOut*sizeof(VertexId));
            EdgeId* eIds = outCell.accessEdgeIdsPointer();
            hkString::memCpy(eIds, edgesOut, numEdgesOut*sizeof(EdgeId));
        }

        // Free the split cell
        m_cells->freeCell(cellId);

#if (HK_DEBUG_CELLS_INTEGRITY)
        checkCellIntegrity(insideCellIdOut);
        checkCellIntegrity(outsideCellIdOut);
#endif

    }
    else
    {
        if ( (numVertsOnPlane > 2) && splitMaterial.isValid() )
        {
            m_hasRepaintedFaces = true;

            if ( m_buildCellConnectivity )
            {
                // this split plane is coincident with one or more of the cell faces
                // Figure out which one to know whether we need to repaint the face with the new material
                const int numFaces      = cell->getNumFaces();
                const FaceId* faceIds   = cell->getFaceIdsPointer();
                hkLocalBuffer<int> facesNumNotOnPlane(numFaces);
                hkString::memSet4(facesNumNotOnPlane.begin(), 0, numFaces);
                for (int f = 0; f < numFaces; f++)
                {
                    m_data->accessFaces()[faceIds[f]].m_userData = hkUint16(f);
                }

                // Loop on edges to tag planes
                const int numEdges = cell->getNumEdges();
                const EdgeId* edgeIds = cell->getEdgeIdsPointer();
                for (int e = 0; e < numEdges; e++)
                {
                    const EdgeId edgeId     = edgeIds[e];
                    const Edge* cEdge       = &(m_data->accessEdges()[edgeId]);
                    const Vertex& vtxStart  = m_data->accessVertices()[cEdge->m_vertexIds[0]];
                    const Vertex& vtxEnd    = m_data->accessVertices()[cEdge->m_vertexIds[1]];
                    Orientation oriPt[2];
                    oriPt[0] = Orientation(vtxStart.m_pos.getInt16W());
                    oriPt[1] = Orientation(vtxEnd.m_pos.getInt16W());

                    // update face split info
                    for (int k = 0; k < 2; k++)
                    {
                        if ( oriPt[k] != hkcdPlanarGeometryPredicates::ON_PLANE )
                        {
                            for (int j = 0 ; j < 2 ; j++)
                            {
                                const Face& cFace = m_data->accessFaces()[cEdge->m_faceIds[j]];
                                const int faceIdx = cFace.m_userData;
                                facesNumNotOnPlane[faceIdx]++;
                            }
                        }
                    }
                }

                // Loop on faces and repaint the free ones
                for (int f = 0; f < numFaces; f++)
                {
                    if ( facesNumNotOnPlane[f] == 0 )
                    {
                        // On plane face
                        Face& face = m_data->accessFaces()[faceIds[f]];
                        if ( !face.m_materialId.isValid() || (!repaintOnlyInvalidMat && !getCell(face.m_negCellId).isSolid() && !getCell(face.m_posCellId).isSolid()) )
                        {
                            // Face between two non solid cells, we can repaint it!
                            face.m_materialId = splitMaterial;
                        }
                    }
                }
            }
        }

        if (numVertsBehind > 0)
        {
            insideCellIdOut = cellId;
        }
        else
        {
            HK_ASSERT_NO_MSG(0x2ad8c56e, numVertsInFront > 0);
            outsideCellIdOut = cellId;
        }
    }

    return newFaceId;
}

//
//  Splits an edge

void hkcdConvexCellsTree3D::splitEdge(EdgeId edgeId, bool flipped, PlaneId splitPlaneId, EdgeId& edgeIdIn, EdgeId& edgeIdOut, VertexId& newVertexId)
{
    // Add the new vertex
    {
        const Edge& cEdge = m_data->accessEdges()[edgeId];
        Vertex& newVertex = m_data->allocateNewVertex(newVertexId);
        newVertex.m_planeIds[0] = m_data->accessFaces()[cEdge.m_faceIds[0]].m_planeId;
        newVertex.m_planeIds[1] = ( cEdge.m_otherPlaneId.isValid() ) ? cEdge.m_otherPlaneId : m_data->accessFaces()[cEdge.m_faceIds[1]].m_planeId;
        newVertex.m_planeIds[2] = splitPlaneId;

        // Compute approx position to accelerate further predicates
        Plane planes[3];
        for (int k = 0 ; k < 3 ; k++)
        {
            m_planes->getPlane(newVertex.m_planeIds[k], planes[k]);
        }
        hkcdPlanarGeometryPredicates::approximateIntersectionFast(planes, newVertex.m_pos);
        newVertex.m_pos.setInt24W(hkcdPlanarGeometryPredicates::ON_PLANE);
    }

    // flipped means vertex 0 is out and vertex 1 is in
    // Create the inside and outside edges
    {
        Edge& inEdge        = m_data->allocateNewEdge(edgeIdIn);
        const Edge& cEdge   = m_data->accessEdges()[edgeId];
        inEdge.m_vertexIds[0]   = ( flipped ) ? newVertexId : cEdge.m_vertexIds[0];
        inEdge.m_vertexIds[1]   = ( flipped ) ? cEdge.m_vertexIds[1] : newVertexId;
        inEdge.m_faceIds[0]     = cEdge.m_faceIds[0];       // Face ids will be remapped later
        inEdge.m_faceIds[1]     = cEdge.m_faceIds[1];
        inEdge.m_otherPlaneId   = cEdge.m_otherPlaneId;
        inEdge.m_firstEdgeOfChainId = edgeIdIn;
    }

    {
        Edge& outEdge       = m_data->allocateNewEdge(edgeIdOut);
        const Edge& cEdge   = m_data->accessEdges()[edgeId];
        outEdge.m_vertexIds[0]  = ( flipped ) ? cEdge.m_vertexIds[0] : newVertexId;
        outEdge.m_vertexIds[1]  = ( flipped ) ? newVertexId : cEdge.m_vertexIds[1];
        outEdge.m_faceIds[0]    = cEdge.m_faceIds[0];
        outEdge.m_faceIds[1]    = cEdge.m_faceIds[1];
        outEdge.m_otherPlaneId  = cEdge.m_otherPlaneId;
        outEdge.m_firstEdgeOfChainId = edgeIdOut;
    }

    if ( m_buildCellConnectivity )
    {
        // In connectivity mode, cut all the linked edges
        EdgeId curEdgeId = m_data->accessEdges()[edgeId].m_firstEdgeOfChainId;
        EdgeId lastAddedInId = edgeIdIn;
        EdgeId lastAddedOutId = edgeIdOut;
        Edge* curEdge;
        while ( int(curEdgeId) != -1)
        {
            curEdge = &m_data->accessEdges()[curEdgeId];

            if ( curEdgeId != edgeId )
            {
                // Cut the edge "curEdge"
                EdgeId newEdgeInId, newEdgeOutId;
                m_data->allocateNewEdge(newEdgeInId);
                m_data->allocateNewEdge(newEdgeOutId);
                curEdge     = &m_data->accessEdges()[curEdgeId];
                Cell& nCell = accessCell(curEdge->m_owningCellId);

                const bool locFlipped = ( m_data->accessEdges()[edgeId].m_vertexIds[0] == curEdge->m_vertexIds[0] ) ? flipped : !flipped;

                // Cut them
                {
                    Edge& inEdge = m_data->accessEdges()[newEdgeInId];
                    inEdge.m_vertexIds[0]   = ( locFlipped ) ? newVertexId : curEdge->m_vertexIds[0];
                    inEdge.m_vertexIds[1]   = ( locFlipped ) ? curEdge->m_vertexIds[1] : newVertexId;
                    inEdge.m_faceIds[0]     = curEdge->m_faceIds[0];
                    inEdge.m_faceIds[1]     = curEdge->m_faceIds[1];
                    inEdge.m_otherPlaneId   = curEdge->m_otherPlaneId;
                    inEdge.m_indexInCell    = curEdge->m_indexInCell;
                    inEdge.m_owningCellId   = curEdge->m_owningCellId;
                    linkEdges(newEdgeInId, lastAddedInId);
                    lastAddedInId = newEdgeInId;
                }

                {
                    Edge& outEdge = m_data->accessEdges()[newEdgeOutId];
                    outEdge.m_vertexIds[0]  = ( locFlipped ) ? curEdge->m_vertexIds[0] : newVertexId;
                    outEdge.m_vertexIds[1]  = ( locFlipped ) ? newVertexId : curEdge->m_vertexIds[1];
                    outEdge.m_faceIds[0]    = curEdge->m_faceIds[0];
                    outEdge.m_faceIds[1]    = curEdge->m_faceIds[1];
                    outEdge.m_otherPlaneId  = curEdge->m_otherPlaneId;
                    outEdge.m_indexInCell   = nCell.getNumEdges();
                    outEdge.m_owningCellId  = curEdge->m_owningCellId;
                    linkEdges(newEdgeOutId, lastAddedOutId);
                    lastAddedOutId = newEdgeOutId;
                }

                // Update the cell data
                nCell.setNumEdges(nCell.getNumEdges() + 1);
                EdgeId* nEdgeIds = nCell.accessEdgeIdsPointer();
                nEdgeIds[curEdge->m_indexInCell]        = newEdgeInId;
                nEdgeIds[nCell.getNumEdges() - 1]       = newEdgeOutId;
                nCell.setNumVertices(nCell.getNumVertices() + 1);
                nCell.accessVertexIdsPointer()[nCell.getNumVertices() - 1] = newVertexId;

#if (HK_DEBUG_CELLS_INTEGRITY)
#if (HK_DEBUG_CELLS_INTEGRITY_EDGE_SPLIT)
                checkCellIntegrity(curEdge->m_owningCellId);
#endif
#endif
            }

            const EdgeId edgIdToRemove = curEdgeId;
            curEdgeId = curEdge->m_nextCoincidentEdgeId;

            if ( edgIdToRemove != edgeId )
            {
                m_data->freeEdge(edgIdToRemove);
            }
        }
    }
}

//
//  Updates the connectivity information for a given polygon after a split

void hkcdConvexCellsTree3D::updateConnectivity(CellId cellIdToUpdate,
    const FaceId faceSplitId, const FaceId faceSplitInId, const FaceId faceSplitOutId,
    const EdgeId newEdgeInId, const int faceNumEdgesSplit)
{
    if ( !cellIdToUpdate.isValid() )
    {
        return;
    }

    Cell& cellToUpdate = accessCell(cellIdToUpdate);

    // Update the split face
    CellId otherCellId;
    {
        // Find the split face
        int faceIdx;
        {
            const FaceId* faceIds = cellToUpdate.getFaceIdsPointer();
            const int numFaces = cellToUpdate.getNumFaces();
            for (faceIdx = numFaces - 1; faceIdx >= 0; faceIdx--)
            {
                if (faceIds[faceIdx] == faceSplitId)
                {
                    break;
                }
            }
            HK_ASSERT_NO_MSG(0xe5a8d99e, faceIdx >= 0);
        }

        const Face& face = m_data->accessFaces()[faceSplitInId];
        otherCellId = ( face.m_negCellId == cellIdToUpdate ) ? face.m_posCellId : face.m_negCellId;

        // Replace it with in and out split face
        cellToUpdate.setNumFaces(cellToUpdate.getNumFaces() + 1);
        FaceId* faceIds = cellToUpdate.accessFaceIdsPointer();
        faceIds[faceIdx]                        = faceSplitInId;
        faceIds[cellToUpdate.getNumFaces() - 1] = faceSplitOutId;
    }

    // Update the edges of the split face
    const int numEdges = cellToUpdate.getNumEdges();
    cellToUpdate.setNumEdges(numEdges + 1);
    EdgeId* edgeIds = cellToUpdate.accessEdgeIdsPointer();
    {
        // Gather all the edges of the split face
        hkLocalBuffer<int> splitFaceEdgeIdx(cellToUpdate.getNumEdges());
        int numSplitFaceEdges = 0;
        {
            for (int eIdx = 0; eIdx < numEdges; eIdx++)
            {
                const EdgeId eId    = edgeIds[eIdx];
                const Edge& edge    = m_data->accessEdges()[eId];
                if (edge.m_faceIds[0] == faceSplitId || edge.m_faceIds[1] == faceSplitId)
                {
                    splitFaceEdgeIdx[numSplitFaceEdges++] = eIdx;
                }
            }
        }

        // Replace caut edge by new edges and update face ids on all other edges
        for (int eIdxx = 0 ; eIdxx < numSplitFaceEdges ; eIdxx++)
        {
            const int eIdx      = splitFaceEdgeIdx[eIdxx];
            const EdgeId eId    = edgeIds[eIdx];
            Edge& edge          = m_data->accessEdges()[eId];

            {
                // This is not a split edge, update face ids depending on edge orientation
                if ( Orientation(m_data->accessVertices()[edge.m_vertexIds[0]].m_pos.getInt16W()) == hkcdPlanarGeometryPredicates::IN_FRONT_OF
                    || Orientation(m_data->accessVertices()[edge.m_vertexIds[1]].m_pos.getInt16W()) == hkcdPlanarGeometryPredicates::IN_FRONT_OF )
                {
                    // Out edge, link with Out split face
                    edge.m_faceIds[0] = ( edge.m_faceIds[0] == faceSplitId ) ? faceSplitOutId : edge.m_faceIds[0];
                    edge.m_faceIds[1] = ( edge.m_faceIds[1] == faceSplitId ) ? faceSplitOutId : edge.m_faceIds[1];
                }
                else
                {
                    // In edge, link with In split face
                    edge.m_faceIds[0] = (edge.m_faceIds[0] == faceSplitId) ? faceSplitInId : edge.m_faceIds[0];
                    edge.m_faceIds[1] = (edge.m_faceIds[1] == faceSplitId) ? faceSplitInId : edge.m_faceIds[1];
                }
            }
        }

        // Add the splitting edge
        {
            EdgeId splittingEdgeId;
            m_data->allocateNewEdge(splittingEdgeId);
            edgeIds[cellToUpdate.getNumEdges() - 1] = splittingEdgeId;

            Edge& srcEdge = m_data->accessEdges()[newEdgeInId];
            Edge& splittingEdge = m_data->accessEdges()[splittingEdgeId];
            splittingEdge.m_vertexIds[0] = srcEdge.m_vertexIds[0];
            splittingEdge.m_vertexIds[1] = srcEdge.m_vertexIds[1];
            const bool filpEdge = ( srcEdge.m_faceIds[0] == faceSplitInId );
            splittingEdge.m_faceIds[0] = (filpEdge) ? faceSplitOutId : faceSplitInId;
            splittingEdge.m_faceIds[1] = (filpEdge) ? faceSplitInId : faceSplitOutId;

            // Link srcEdge with this one
            linkEdges(splittingEdgeId, newEdgeInId);
            splittingEdge.m_owningCellId    = cellIdToUpdate;
            splittingEdge.m_indexInCell     = cellToUpdate.getNumEdges() - 1;

            // Because this newly created edge connects two coplanar faces, cutting it would generate vertices with
            // wrong plane ids (two of the plane being coplanar). Thus, we store on the edge an extra plane id
            // that will be used during the edge split
            const FaceId otherFaceId = ( srcEdge.m_faceIds[0] == faceSplitInId ) ? srcEdge.m_faceIds[1] : srcEdge.m_faceIds[0];
            splittingEdge.m_otherPlaneId = getCellFacePlaneId(m_data->accessFaces()[otherFaceId], otherCellId);
        }

#if HK_DEBUG_CELLS_INTEGRITY
        checkCellIntegrity(cellIdToUpdate);
#endif
    }
}

//
//  Creates a polygon from a cellId and faceId

hkcdConvexCellsTree3D::PolygonId hkcdConvexCellsTree3D::createPolygonFromFace(CellId cellId, FaceId faceId, hkcdPlanarGeometry& workingGeom) const
{
    // Get the cell
    const Cell& cell    = m_cells->getCell(cellId);

    const int numEdges      = cell.getNumEdges();
    const EdgeId* edgeIds   = cell.getEdgeIdsPointer();
    hkLocalBuffer<int> faceEdgeIds(numEdges);
    int numEdgeInFace = 0;

    // Mark all vertices (this could be done outside and pass as a parameter of the function)
    const int numVertices   = cell.getNumVertices();
    const VertexId* vertIds = cell.getVertexIdsPointer();
    hkLocalBuffer<int> verticesEdgeId(numVertices << 1);
    hkLocalBuffer<int> verticesNumEdgeId(numVertices);
    {
        int vIdx = 0;
        for (int v = 0 ; v < numVertices ; v++)
        {
            Vertex& vertex = m_data->accessVertices()[vertIds[v]];
            vertex.m_pos.setInt24W(vIdx++);
        }
    }

    // Get all the edges forming the FACE
    for (int e = 0 ; e < numEdges ; e++)
    {
        const Edge& edge    = m_data->accessEdges()[edgeIds[e]];
        if ( edge.m_faceIds[0] == faceId || edge.m_faceIds[1] == faceId )
        {
            faceEdgeIds[numEdgeInFace++] = edgeIds[e];
        }
        // Reset the edge ref counter for the vertex
        for (int k = 0 ; k < 2 ; k++)
        {
            const Vertex& vertex    = m_data->accessVertices()[edge.m_vertexIds[k]];
            const int vIdx          = vertex.m_pos.getInt24W();
            verticesNumEdgeId[vIdx] = 0;
        }
    }
    HK_ASSERT_NO_MSG(0xd1a2e6dd, numEdgeInFace > 0);

    // Build a vertex to edges array
    for (int e = 0 ; e < numEdgeInFace ; e++)
    {
        const Edge& edge    = m_data->accessEdges()[faceEdgeIds[e]];
        for (int k = 0 ; k < 2 ; k++)
        {
            const Vertex& vertex    = m_data->accessVertices()[edge.m_vertexIds[k]];
            const int vIdx          = vertex.m_pos.getInt24W();
            verticesEdgeId[(vIdx << 1) + verticesNumEdgeId[vIdx]] = faceEdgeIds[e];
            verticesNumEdgeId[vIdx]++;
        }
    }

    // Add all the boundary planes (1 per edge)
    int numAddedEdges   = 0;
    int currEdgeId      = faceEdgeIds[0];
    hkLocalArray<PlaneId> bPlaneIds(numEdgeInFace);
    hkLocalArray<VertexId> bVertexIds(numEdgeInFace);
    while ( numAddedEdges < numEdgeInFace )
    {
        // Get boundary plane id
        const Edge& edge    = m_data->accessEdges()[currEdgeId];
        const bool flipEdge = ( edge.m_faceIds[0] == faceId );      // The face extracted should always be on the right of the edge, by convention
        const Face& bFace   = ( flipEdge ) ? m_data->accessFaces()[edge.m_faceIds[1]] : m_data->accessFaces()[edge.m_faceIds[0]];
        PlaneId bPlaneId    =
            edge.m_otherPlaneId.isValid() ?
                flipEdge ? edge.m_otherPlaneId : hkcdPlanarGeometryPrimitives::getOppositePlaneId(edge.m_otherPlaneId)
                : getCellFacePlaneId(bFace, cellId);

        bPlaneIds.pushBack(bPlaneId);

        // Go to the next edge
        {
            const VertexId nextVId  = ( flipEdge ) ? edge.m_vertexIds[0] : edge.m_vertexIds[1];

            bVertexIds.pushBack(nextVId);

            const Vertex& vertex    = m_data->accessVertices()[nextVId];
            const int nextvIdx      = vertex.m_pos.getInt24W();
            const int* nEdgeIds     = &(verticesEdgeId[(nextvIdx << 1)]);
            currEdgeId              = ( nEdgeIds[0] == currEdgeId ) ? nEdgeIds[1] : nEdgeIds[0];
        }

        numAddedEdges++;
    }

    // If connectivity build, remove the double planes
    int numFinalBounds = bPlaneIds.getSize();
    if ( m_buildCellConnectivity )
    {
        hkLocalArray<PlaneId> bPlaneIdsTmp(numEdgeInFace);
        hkLocalArray<VertexId> bVertexIdsTmp(numEdgeInFace);
        for (int p = 0, nb = bPlaneIds.getSize() ; p < nb ; p++)
        {
            const int np = ( p < nb - 1 ) ? p + 1 : 0;
            if ( bPlaneIds[np] != bPlaneIds[p] )
            {
                // Valid plane, add it
                bPlaneIdsTmp.pushBack(bPlaneIds[np]);
                bVertexIdsTmp.pushBack(bVertexIds[p]);
            }
        }

        numFinalBounds = bPlaneIdsTmp.getSize();
        for (int i = 0 ; i < numFinalBounds ; i++)
        {
            bPlaneIds[i]    = bPlaneIdsTmp[i];
            bVertexIds[i]   = bVertexIdsTmp[i];
        }
    }
    else
    {
        // shift vertices by -1
        VertexId vTmpId = bVertexIds[0];
        for (int v = 0; v < numEdgeInFace - 1; v++)
        {
            bVertexIds[v] = bVertexIds[v + 1];
        }
        bVertexIds[numEdgeInFace - 1] = vTmpId;
    }

    // Add the polygon to the geometry
    const Face& face        = m_data->accessFaces()[faceId];
    PlaneId supportPlaneId  = getCellFacePlaneId(face, cellId);
    PolygonId pIdRes        = workingGeom.addPolygon(supportPlaneId, face.m_materialId, numFinalBounds);
    if ( !pIdRes.isValid() )
    {
        return PolygonId::invalid();
    }

    Polygon& poly           = workingGeom.accessPolygon(pIdRes);

    // Write the boundaries of the poly
    for (int b = 0 ; b < numFinalBounds ; b++)
    {
        // Set it on the poly
        poly.setBoundaryPlaneId(b, bPlaneIds[b]);
        poly.setVertexId(b, hkcdPlanarGeometryPolygonCollection::VertexId(bVertexIds[b]));
    }

#if (HK_DEBUG_INTEGRITY_POLY_CONVEXITY)
    for (int b = 0; b < numFinalBounds; b++)
    {
        // Reset vertex ids values
        poly.setVertexId(b, hkcdPlanarGeometryPolygonCollection::VertexId::invalid());
    }
    workingGeom.computePolygonApproxVertices(pIdRes);
    workingGeom.checkPolygonConvexity(pIdRes);
    for (int b = 0; b < numFinalBounds; b++)
    {
        // Reset vertex ids values
        poly.setVertexId(b, hkcdPlanarGeometryPolygonCollection::VertexId(bVertexIds[b]));
    }
#endif

    return pIdRes;
}

//
//  Creates a polygon from a cellId and faceId

void hkcdConvexCellsTree3D::createPolygonFromFace(CellId cellId, FaceId faceId, hkArray<VertexId>& polyVerticeIds, _Inout_opt_ hkArray<PlaneId>* bPlaneIds) const
{
    // Get the cell
    const Cell& cell = m_cells->getCell(cellId);

    const int numEdges      = cell.getNumEdges();
    const EdgeId* edgeIds   = cell.getEdgeIdsPointer();
    hkLocalBuffer<int> faceEdgeIds(numEdges);
    int numEdgeInFace = 0;

    // Mark all vertices (this could be done outside and pass as a parameter of the function)
    const int numVertices   = cell.getNumVertices();
    const VertexId* vertIds = cell.getVertexIdsPointer();
    hkLocalBuffer<int> verticesEdgeId(numVertices << 1);
    hkLocalBuffer<int> verticesNumEdgeId(numVertices);
    {
        int vIdx = 0;
        for (int v = 0; v < numVertices; v++)
        {
            Vertex& vertex = m_data->accessVertices()[vertIds[v]];
            vertex.m_pos.setInt24W(vIdx++);
        }
    }

    // Get all the edges forming the FACE
    for (int e = 0; e < numEdges; e++)
    {
        const Edge& edge = m_data->accessEdges()[edgeIds[e]];
        if (edge.m_faceIds[0] == faceId || edge.m_faceIds[1] == faceId)
        {
            faceEdgeIds[numEdgeInFace++] = edgeIds[e];
        }
        // Reset the edge ref counter for the vertex
        for (int k = 0; k < 2; k++)
        {
            const Vertex& vertex    = m_data->accessVertices()[edge.m_vertexIds[k]];
            const int vIdx          = vertex.m_pos.getInt24W();
            verticesNumEdgeId[vIdx] = 0;
        }
    }
    HK_ASSERT_NO_MSG(0xd1a2e6dd, numEdgeInFace > 0);

    // Build a vertex to edges array
    for (int e = 0; e < numEdgeInFace; e++)
    {
        const Edge& edge = m_data->accessEdges()[faceEdgeIds[e]];
        for (int k = 0; k < 2; k++)
        {
            const Vertex& vertex = m_data->accessVertices()[edge.m_vertexIds[k]];
            const int vIdx = vertex.m_pos.getInt24W();
            verticesEdgeId[(vIdx << 1) + verticesNumEdgeId[vIdx]] = faceEdgeIds[e];
            verticesNumEdgeId[vIdx]++;
        }
    }

    // Output the vertices
    int numAddedEdges = 0;
    int currEdgeId = faceEdgeIds[0];
    while ( numAddedEdges < numEdgeInFace )
    {
        // Get boundary plane id
        const Edge& edge = m_data->accessEdges()[currEdgeId];
        const bool flipEdge = (edge.m_faceIds[0] == faceId);        // The face extracted should always be on the right of the edge, by convention

        // Get the boundary plane if wanted
        if ( bPlaneIds )
        {
            const Face& bFace   = ( flipEdge ) ? m_data->accessFaces()[edge.m_faceIds[1]] : m_data->accessFaces()[edge.m_faceIds[0]];
            PlaneId bPlaneId    =
                edge.m_otherPlaneId.isValid() ?
                flipEdge ? edge.m_otherPlaneId : hkcdPlanarGeometryPrimitives::getOppositePlaneId(edge.m_otherPlaneId)
                : getCellFacePlaneId(bFace, cellId);

            bPlaneIds->pushBack(bPlaneId);
        }

        // Go to the next edge
        {
            const VertexId nextVId = ( flipEdge ) ? edge.m_vertexIds[0] : edge.m_vertexIds[1];

            // Add the vertex
            polyVerticeIds.pushBack(nextVId);

            const Vertex& vertex = m_data->accessVertices()[nextVId];
            const int nextvIdx = vertex.m_pos.getInt24W();
            const int* nEdgeIds = &(verticesEdgeId[(nextvIdx << 1)]);
            currEdgeId = ( nEdgeIds[0] == currEdgeId ) ? nEdgeIds[1] : nEdgeIds[0];
        }

        numAddedEdges++;
    }

    if ( bPlaneIds )
    {
        const int bStart = bPlaneIds->getSize() - numEdgeInFace;
        // shift planes
        PlaneId vTmpId = (*bPlaneIds)[bStart];
        for (int v = 0 ; v < numEdgeInFace - 1 ; v++)
        {
            (*bPlaneIds)[bStart + v] = (*bPlaneIds)[bStart + v + 1];
        }
        (*bPlaneIds)[bPlaneIds->getSize() - 1] = vTmpId;
    }
}

//
//  Clones a cell, cloning as well all its edges

hkcdConvexCellsTree3D::CellId hkcdConvexCellsTree3D::cloneCell(CellId cellId)
{
    CellId newCellId    = m_cells->allocCell();
    Cell& newCell       = accessCell(newCellId);
    Cell& srcCell       = accessCell(cellId);

    newCell.setSizes(srcCell.getNumFaces(), srcCell.getNumVertices(), srcCell.getNumEdges());

    // Copy vertices
    const int numVertices = newCell.getNumVertices();
    for (int v = 0 ; v < numVertices ; v++)
    {
        newCell.accessVertexIdsPointer()[v] = srcCell.accessVertexIdsPointer()[v];
    }

    // Duplicate edges
    const int numEdges = newCell.getNumEdges();
    for (int e = 0 ; e < numEdges ; e++)
    {
        Edge& newEdge = m_data->allocateNewEdge(newCell.accessEdgeIdsPointer()[e]);
        newEdge = m_data->accessEdges()[srcCell.accessEdgeIdsPointer()[e]];
    }

    // Duplicate faces
    const int numFaces = newCell.getNumFaces();
    hkLocalBuffer<EdgeId> faceRemap(numFaces);
    for (int f = 0; f < numFaces; f++)
    {
        const FaceId srcFaceId  = srcCell.accessFaceIdsPointer()[f];
        Face& dstFace           = m_data->allocateNewFace(newCell.accessFaceIdsPointer()[f]);
        dstFace                 = m_data->accessFaces()[srcFaceId];
        faceRemap[f]            = newCell.accessFaceIdsPointer()[f];

        // Remap cell ids link
        dstFace.m_negCellId     = ( dstFace.m_negCellId == cellId ) ? newCellId : CellId::invalid();
        dstFace.m_posCellId     = ( dstFace.m_posCellId == cellId ) ? newCellId : CellId::invalid();

        // Store face idx
        m_data->accessFaces()[srcFaceId].m_userData = hkUint16(f);
    }

    // Remap edges
    for (int e = 0; e < numEdges; e++)
    {
        Edge& dstEdge = m_data->accessEdges()[newCell.accessEdgeIdsPointer()[e]];
        dstEdge.m_faceIds[0] = faceRemap[m_data->accessFaces()[dstEdge.m_faceIds[0]].m_userData];
        dstEdge.m_faceIds[1] = faceRemap[m_data->accessFaces()[dstEdge.m_faceIds[1]].m_userData];
    }

    return newCellId;
}

//
//  Debug only: checks a cell integrity

void hkcdConvexCellsTree3D::checkCellIntegrity(CellId cellId) const
{
    const Cell& cell    = getCell(cellId);

    const int numVertices   = cell.getNumVertices();
    const int numFaces      = cell.getNumFaces();
    const int numEdges      = cell.getNumEdges();

    const FaceId* faceIds   = cell.getFaceIdsPointer();
    const VertexId* vertIds = cell.getVertexIdsPointer();
    const EdgeId* edgeIds   = cell.getEdgeIdsPointer();
//  const int mask          = ~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG;

    // Get referenced points by edges
    hkArray<VertexId> refPtIds, actPtsIds;
    hkArray<FaceId> refFacesIds, actFaceIds;
    for (int e = 0 ; e < numEdges ; e++)
    {
        const Edge& edge    = m_data->accessEdges()[edgeIds[e]];
        refPtIds.pushBack(edge.m_vertexIds[0]);
        refPtIds.pushBack(edge.m_vertexIds[1]);
        refFacesIds.pushBack(edge.m_faceIds[0]);
        refFacesIds.pushBack(edge.m_faceIds[1]);
    }
    for (int f = 0 ; f < numFaces ; f++)
    {
        actFaceIds.pushBack(faceIds[f]);
    }
    hkSort(actFaceIds.begin(), actFaceIds.getSize());
    for (int v = 0 ; v < numVertices ; v++)
    {
        actPtsIds.pushBack(vertIds[v]);
    }
    hkSort(actPtsIds.begin(), actPtsIds.getSize());

    // Do diff with actual points ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    hkSort(refPtIds.begin(), refPtIds.getSize());
    // check at least three edges must reference each point
    for (int p = 0 ; p < refPtIds.getSize() ;)
    {
        hkUint32 curVal = refPtIds[p];
        int num = 0;
        while ( p < refPtIds.getSize() && curVal == refPtIds[p] ) { p++; num++; }
        if ( m_buildCellConnectivity && num == 2 )
        {
            // Check that the two edges sharing the point with ref num 2 are collinear
            int nFoundEdges = 0;
            int edgePointIds[2];

            for (int e = 0; e < numEdges; e++)
            {
                const Edge& edge = m_data->accessEdges()[edgeIds[e]];
                if ( edge.m_vertexIds[0] == curVal || edge.m_vertexIds[1] == curVal )
                {
                    edgePointIds[nFoundEdges++] = edgeIds[e];
                }
            }

            int planeIds[4];
            int np = 0;
            for (int e = 0 ; e < 2 ; e++)
            {
                const Edge& edgePoint = m_data->accessEdges()[edgePointIds[e]];
                const Face& face1 = m_data->accessFaces()[edgePoint.m_faceIds[0]];
                const Face& face2 = m_data->accessFaces()[edgePoint.m_faceIds[1]];
                planeIds[np++] = face1.m_planeId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                planeIds[np++] = face2.m_planeId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
            }
            if ( !(
                (planeIds[0] == planeIds[2] || planeIds[0] == planeIds[3]) &&
                (planeIds[1] == planeIds[2] || planeIds[1] == planeIds[3])
                ) )
            {
                    Log_Error( "Two supposedly collinear edges are not!" );
                    HK_BREAKPOINT(0);
                    return;
            }
        }
        else if ( !m_buildCellConnectivity && num < 3 )
        {
            Log_Error( "Point referenced by less than 3 edges!" );
            HK_BREAKPOINT(0);
            return;
        }
    }

    refPtIds.setSize(hkAlgorithm::removeDuplicatesFromSortedList(refPtIds.begin(), refPtIds.getSize()));
    {
        hkArray<VertexId> diff(refPtIds.getSize());
        diff.setSize(hkAlgorithm::differenceOfSortedLists(refPtIds.begin(), refPtIds.getSize(), actPtsIds.begin(), actPtsIds.getSize(), diff.begin()));
        if ( diff.getSize() > 0 )
        {
            Log_Error( "Inconsitency detected in points!" );
            HK_BREAKPOINT(0);
            return;
        }
    }

    // Do the same for faces ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    hkSort(refFacesIds.begin(), refFacesIds.getSize());
    // check at least three edges must reference each face
    for (int f = 0 ; f < refFacesIds.getSize() ;)
    {
        hkUint32 curVal = refFacesIds[f];
        int num = 0;
        while ( f < refFacesIds.getSize() && curVal == refFacesIds[f] ) { f++; num++; }
        if ( num < 3 )
        {
            Log_Error( "Face referenced by less than 3 edges!" );
            HK_BREAKPOINT(0);
            return;
        }
    }

    refFacesIds.setSize(hkAlgorithm::removeDuplicatesFromSortedList(refFacesIds.begin(), refFacesIds.getSize()));
    {
        hkArray<VertexId> diff(refFacesIds.getSize());
        diff.setSize(hkAlgorithm::differenceOfSortedLists(refFacesIds.begin(), refFacesIds.getSize(), actFaceIds.begin(), actFaceIds.getSize(), diff.begin()));
        if ( diff.getSize() > 0 )
        {
            Log_Error( "Inconsitency detected in faces!" );
            HK_BREAKPOINT(0);
            return;
        }
    }

    // Face consistency ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    for (int fIdx = 0 ; fIdx < numFaces ; fIdx++)
    {
        // Get all the edges and vertices on this faces
        hkArray<EdgeId> faceEdgeIds;
        hkArray<VertexId> faceVertices;
        for (int eIdx = 0 ; eIdx < numEdges ; eIdx++)
        {
            const Edge& edge = m_data->accessEdges()[edgeIds[eIdx]];
            if ( edge.m_faceIds[0] == faceIds[fIdx] || edge.m_faceIds[1] == faceIds[fIdx] )
            {
                faceEdgeIds.pushBack(edgeIds[eIdx]);
                faceVertices.pushBack(edge.m_vertexIds[0]);
                faceVertices.pushBack(edge.m_vertexIds[1]);
            }
        }

        // check neg and pos cell are different
        {
            const Face& face = m_data->accessFaces()[faceIds[fIdx]];
            if ( face.m_negCellId.isValid() && face.m_negCellId == face.m_posCellId )
            {
                Log_Error( "Inconsitency detected: negative and positive cells are the same!" );
                HK_BREAKPOINT(0);
                return;
            }
        }

        // check that every vertex is present exactly twice
        hkSort(faceVertices.begin(), faceVertices.getSize());
        for (int i = 0 ; i < faceVertices.getSize() ; i += 2)
        {
            if ( faceVertices[i] != faceVertices[i + 1] )
            {
                Log_Error( "Inconsitency detected: vertex not present exactly twice!" );
                HK_BREAKPOINT(0);
                return;
            }
        }
    }

    // Edge plane ids consistency ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    for (int e = 0 ; e < numEdges ; e++)
    {
        const Edge& edge    = m_data->accessEdges()[edgeIds[e]];
        const Face& face1   = m_data->accessFaces()[edge.m_faceIds[0]];
        const Face& face2   = m_data->accessFaces()[edge.m_faceIds[1]];
        if ( !edge.m_otherPlaneId.isValid() &&
            (face1.m_planeId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG)) ==
            (face2.m_planeId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG)) )
        {
            Log_Error( "Inconsitency detected in edge plane ID!" );
            HK_BREAKPOINT(0);
            return;
        }
    }

#if HK_DEBUG_CELLS_INTEGRITY_VERTEX_CACHE
    // check vertices cached pos and planes ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    for (int v = 0 ; v < numVertices ; v++)
    {
        const Vertex& vertex = m_data->accessVertices()[v];
        hkcdPlanarGeometryPlanesCollection::Plane planes[3];
        m_planes->getPlane(vertex.m_planeIds[0], planes[0]);
        m_planes->getPlane(vertex.m_planeIds[1], planes[1]);
        m_planes->getPlane(vertex.m_planeIds[2], planes[2]);
        hkVector4d vCached      = vertex.m_pos;
        hkVector4d vCheck;
        hkcdPlanarGeometryPredicates::approximateIntersectionFast(planes, vCheck);

        if ( vCheck.distanceTo(vCached).isGreater(hkSimdDouble64::fromFloat(2)) )
        {
            Log_Error( "Invalid vertexcaches value!!" );
            HK_BREAKPOINT(0);
            return;
        }
    }
#endif

#if HK_DEBUG_CELLS_INTEGRITY_EDGE_WINDING
    // check edge convention: the face 0 should always be on the "left" side of the edge
    // this is approximate test as it will use vertices positions ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    for (int e = 0 ; e < numEdges ; e++)
    {
        const Edge& edge    = m_data->accessEdges()[edgeIds[e]];
        const Vertex& vA    = m_data->accessVertices()[edge.m_vertexIds[0]];
        const Vertex& vB    = m_data->accessVertices()[edge.m_vertexIds[1]];
        Plane pA, pB;
        m_planes->getPlane(getCellFacePlaneId(m_data->accessFaces()[edge.m_faceIds[0]], cellId), pA);
        m_planes->getPlane(getCellFacePlaneId(m_data->accessFaces()[edge.m_faceIds[1]], cellId), pB);
        const hkVector4d& nA=pA.getApproxEquation();
        const hkVector4d& nB=pB.getApproxEquation();
        hkVector4d dir;     dir.setSub(vB.m_pos, vA.m_pos);
        if ( dir.length<3>().isGreater(hkSimdDouble64_4) )
        {
            hkVector4d cross;   cross.setCross(nB, nA);
            if ( dir.dot<3>(cross).isLessZero() )
            {
                Log_Error( "Edge convention not respected" );
                HK_BREAKPOINT(0);
                return;
            }
        }
    }
#endif

#if HK_DEBUG_CELLS_INTEGRITY_CELL_CONVEXITY
    // check cell convexity ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    hkIntVector vPlaneIds;
    Plane planes[3];
    Plane sPlane;
    for (int f = 0 ; f < numFaces ; f++)
    {
        // Get all lthe points on the face
        hkArray<VertexId> ptIdsOnFace;
        for (int e = 0 ; e < numEdges ; e++)
        {
            const Edge& edge    = m_data->accessEdges()[edgeIds[e]];
            if ( edge.m_faceIds[0] == faceIds[f] || edge.m_faceIds[1] == faceIds[f] )
            {
                ptIdsOnFace.pushBack(edge.m_vertexIds[0]);
                ptIdsOnFace.pushBack(edge.m_vertexIds[1]);
            }
        }
        hkSort(ptIdsOnFace.begin(), ptIdsOnFace.getSize());
        ptIdsOnFace.setSize(hkAlgorithm::removeDuplicatesFromSortedList(ptIdsOnFace.begin(), ptIdsOnFace.getSize()));

        // Classify all the points vs the face plane. Must be behind or on plane
        PlaneId splaneId = getCellFacePlaneId(m_data->accessFaces()[faceIds[f]], cellId);
        m_planes->getPlane(splaneId, sPlane);

        for (int v = 0 ; v < numVertices ; v++)
        {
            Vertex& vertex = m_data->accessVertices()[vertIds[v]];
            vPlaneIds.set(vertex.m_planeIds[0].value(), vertex.m_planeIds[1].value(), vertex.m_planeIds[2].value(), splaneId.value());
            // Get the involved planes
            m_planes->getPlane(vertex.m_planeIds[0], planes[0]);
            m_planes->getPlane(vertex.m_planeIds[1], planes[1]);
            m_planes->getPlane(vertex.m_planeIds[2], planes[2]);

            // compute orientation
            Orientation ori = hkcdPlanarGeometryPredicates::orientation(vertex.m_pos, planes[0], planes[1], planes[2], sPlane, vPlaneIds, m_planes->getOrientationCache());
            if ( ori == hkcdPlanarGeometryPredicates::IN_FRONT_OF )
            {
                Log_Error( "Cell not convex!" );
                HK_BREAKPOINT(0);
                return;
            }
            int ind = hkAlgorithm::binarySearch(vertIds[v], ptIdsOnFace.begin(), ptIdsOnFace.getSize());
            // face point must be on plane
            if ( ind >= 0 && ori != hkcdPlanarGeometryPredicates::ON_PLANE )
            {
                Log_Error( "Face point not on face plane!" );
                HK_BREAKPOINT(0);
                return;
            }
        }
    }
#endif

/*
    // Check edges versus points     ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    for (int e = 0 ; e < numEdges ; e++)
    {
        const Edge& edge    = m_data->m_edges[edgeIds[e]];

        // check if planes of edge are included in planes of the vertex
        for (int v = 0 ; v < 2 ; v++)
        {
            const Vertex& vert  = m_data->m_vertices[edge.m_vertexIds[v]];
            for (int pl = 0 ; pl < 2 ; pl++)
            {
                PlaneId pIdRef = PlaneId(m_data->m_faces[edge.m_faceIds[pl]].m_planeId.value() & mask);
                bool found = false;
                for (int plv = 0 ; plv < 3 ; plv++)
                {
                    PlaneId pIdcomp = PlaneId(vert.m_planeIds[plv].value() & mask);
                    if ( pIdRef == pIdcomp )
                    {
                        found = true;
                        break;
                    }
                }
                if ( !found )
                {
                    Log_Error( "Inconsitent edge vs vertices" );
                    HK_BREAKPOINT(0);
                    return;
                }
            }
        }
    }
*/
}

//
//  Returns a list of unique face ids from a set of cells

void hkcdConvexCellsTree3D::getUniqueFaceIdsFromCellIds(const hkArray<CellId>& cellIdsIn, hkArray<FaceId>& faceIdsOut)
{
    // First, count the number of polygon to allocate
    int nbMaxFaces = 0;
    for (int i = cellIdsIn.getSize() - 1; i >= 0; i--)
    {
        nbMaxFaces  += getCell(cellIdsIn[i]).getNumFaces();
    }

    // Allocate and retrieve the polygons
    faceIdsOut.reserve(nbMaxFaces);
    for (int i = cellIdsIn.getSize() - 1 ; i >=0 ; i--)
    {
        const Cell& cell        = getCell(cellIdsIn[i]);
        const int numFaces      = cell.getNumFaces();
        const FaceId* faceIds   = cell.getFaceIdsPointer();

        for (int pi = 0; pi < numFaces; pi++)
        {
            const FaceId faceId = faceIds[pi];
            Face& face          = m_data->accessFaces()[faceId];

            // When the convex cell tree is built with connectivity, polygon are shared between cells
            if ( !face.isVisited() )
            {
                faceIdsOut.pushBack(faceId);
                face.setVisited();
            }
        }
    }

    // Remove visited flag
    for (int f = faceIdsOut.getSize() - 1 ; f >= 0 ; f--)
    {
        m_data->accessFaces()[faceIdsOut[f]].setNotVisited();
    }
}

//
//  From a set of polygons of the original geometry, mark all the boundary of all the cells with an intersection test

void hkcdConvexCellsTree3D::markBoundaryCells(_In_opt_ const ExecutionContext* executionCtx, const hkcdPlanarGeometry& originalGeom, const hkArray<PolygonId>& originalBoundaryPolygonIds, _Inout_opt_  hkcdPlanarEntityDebugger* debugger)
{
    // First of all, build a necessary acceleration structure: a map from Plane Id => Polygon Ids
    hkArray< hkArray<PolygonId> > planeIdToPolyIds;
    planeIdToPolyIds.setSize(m_planes->getNumPlanes());

    // Add all the original polygons into the array
    for (int i = originalBoundaryPolygonIds.getSize() - 1 ; i >=0 ; i--)
    {
        int planeIdx = originalGeom.getPolygon(originalBoundaryPolygonIds[i]).getSupportPlaneId().value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
        planeIdToPolyIds[planeIdx].pushBack(originalBoundaryPolygonIds[i]);
    }

    // Temp geom for polygon intersections
    hkcdPlanarGeometry tempGeom(const_cast<hkcdPlanarGeometryPlanesCollection*>(m_planes.val()));

    // Collect all the leaf cells
    hkArray<CellId> leafIds;
    collectLeafCells(leafIds);

    // Get all the polygon ids from the cells
    hkArray<FaceId> faceToCheckIds;
    getUniqueFaceIdsFromCellIds(leafIds, faceToCheckIds);

    // For each poly, check if belongs to the boundary or not
    int nbIntersectionTests = 0;
    for (int i = faceToCheckIds.getSize() - 1; i >= 0; i--)
    {
        // Get the plane Id corresponding to this cell boundary
        const FaceId faceId         = faceToCheckIds[i];
        Face& face                  = m_data->accessFaces()[faceId];
        if ( face.getSurfaceType() == FACE_SURFACE_INVALID ) continue;  // Cannot be a boundary, invalid already (e.g. bounding box planes)
        PlaneId planeId             = face.m_planeId;
        const int planeIdx          = planeId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);

        // Test for interruption every once in a while
        if ( !(i & 0xF) && hkTask::receivedAbortRequest(executionCtx) )
        {
            break;
        }

        // Get the polygons to check for this plane
        hkArray<PolygonId>& candidateOriginalpolyIds = planeIdToPolyIds[planeIdx];
        for (int j = candidateOriginalpolyIds.getSize() - 1 ; j >= 0 ; j--)
        {
            // Check for intersection between the polygons
            const PolygonId candPolyId      = candidateOriginalpolyIds[j];
            const PlaneId candPlaneId       = originalGeom.getPolygon(candPolyId).getSupportPlaneId();
            const bool sameOrientation      = !hkcdPlanarGeometryPrimitives::sameOrientationPlaneIds(planeId, candPlaneId);
            if (  sameOrientation && (face.getSurfaceType() & FACE_SURFACE_DIRECT) )    continue;
            if ( !sameOrientation && (face.getSurfaceType() & FACE_SURFACE_INDIRECT) )  continue;

            nbIntersectionTests++;

            // Generate the face poly
            PolygonId polyId = createPolygonFromFace(face.m_negCellId, faceId, tempGeom);
            if( !polyId.isValid() )
            {
                return;
            }
            if ( hkcdPlanarGeometry::check2dIntersection(originalGeom, candPolyId, tempGeom, polyId) )
            {
                // Update face surface info
                face.m_flags |= ( sameOrientation ) ? FACE_SURFACE_DIRECT : FACE_SURFACE_INDIRECT;
            }
        }
    }
}

//
//  Finds an empty cell

hkcdConvexCellsTree3D::CellId hkcdConvexCellsTree3D::findOutputCell()
{
    for (CellId cellId = m_cells->getFirstCellId(); cellId.isValid(); cellId = m_cells->getNextCellId(cellId))
    {
        // Check if leaf
        const Cell& cell = getCell(cellId);
        if ( cell.isLeaf() )
        {
            // Check if it has a boundary belonging to the bounding box
            const int numFaces  = cell.getNumFaces();
            for (int pi = 0; pi < numFaces; pi++)
            {
                const FaceId faceId     = cell.getFaceIdsPointer()[pi];
                const Face& face        = m_data->accessFaces()[faceId];
                if ( !face.m_negCellId.isValid() || !face.m_posCellId.isValid() )
                {
                    return cellId;
                }
            }
        }
    }

    return CellId::invalid();
}

//
//  Infers in/out labels by flood filling thanks to the boundary properties computed on each polygon of each cell

void hkcdConvexCellsTree3D::inferCellsLabels(_Inout_opt_ hkcdPlanarEntityDebugger* debugger)
{
    // Get all cells
    hkArray<CellId> leafIds;
    collectLeafCells(leafIds);
    for (int i = leafIds.getSize() - 1 ; i >= 0 ; i--) { accessCell(leafIds[i]).setVisited(false); accessCell(leafIds[i]).setUserData(0); }
    hkArray<CellId> cellQueue(leafIds.getSize());

    hkUint32 currentLevel   = 1;            // even number is outside, odd number inside (0 is invalid level)
    int nbLabeledRegions    = 0;
    bool outLevel;
    int queueHead, queueTail;

    // Start flood/peeling alg
    CellId nextStartingCellId = findOutputCell();
    HK_ASSERT_NO_MSG(0xfd12a35e, nextStartingCellId.isValid());
    accessCell(nextStartingCellId).setUserData(currentLevel);

    // Loop over the "peeling" levels, usually small loop number (2 for standard filled geometry)
    while ( nextStartingCellId.isValid() )
    {
        currentLevel                            = getCell(nextStartingCellId).getUserData();
        outLevel                                = ( currentLevel % 2 == 1 );
        queueHead = queueTail                   = 0;
        cellQueue[queueTail++]                  = nextStartingCellId;
        accessCell(nextStartingCellId).setVisited(true);

        // flood the in/out information
        while ( queueHead < queueTail )
        {
            // Get the current cell
            const CellId cellId = cellQueue[queueHead++];
            Cell& cell          = accessCell(cellId);
            cell.setLabel(outLevel ? hkcdNewCellsCollection::CELL_EMPTY : hkcdNewCellsCollection::CELL_SOLID);
            cell.setUserData(currentLevel);
            nbLabeledRegions++;

            // Look at all its neighbors
            const int numFaces  = cell.getNumFaces();
            for (int pi = 0; pi < numFaces; pi++)
            {
                const FaceId faceId = cell.getFaceIdsPointer()[pi];
                const Face& face    = m_data->accessFaces()[faceId];

                // Get the neigbhor cell id
                const CellId nCellId    = ( face.m_negCellId == cellId ) ? face.m_posCellId : face.m_negCellId;
                if ( nCellId.isValid() && !getCell(nCellId).isVisited() )
                {
                    Cell& nCell = accessCell(nCellId);
                    // If the poly represent a boundary (stored in material of poly), we cannot go through!
                    if ( canGoFromCellThroughFace(cellId, faceId, outLevel) )
                    {
                        // Add the neighbor to list
                        cellQueue[queueTail++] = nCellId;
                        nCell.setVisited(true);
                    }
                    else
                    {
                        // We may have found a inside cell candidate for the next peeling stage
                        if ( nCell.getUserData() == 0 )  nCell.setUserData(currentLevel + 1);
                    }
                }
            }
        }

        // If we don't have a starting cell, but not all cells are labeled, find one
        nextStartingCellId                      = CellId::invalid();
        if ( nbLabeledRegions < leafIds.getSize() )
        {
            hkUint32 minLevelFound  = 0xffffffff;
            for (int i = leafIds.getSize() - 1 ; i >= 0 ; i--)
            {
                const CellId cellId = leafIds[i];
                Cell& cell          = accessCell(cellId);
                if ( !cell.isVisited() && cell.getUserData() > 0 && cell.getUserData() < minLevelFound )
                {
                    nextStartingCellId  = cellId;
                    minLevelFound       = cell.getUserData();
                }
            }
        }
    }

    HK_ASSERT_NO_MSG(0xe987a65c, nbLabeledRegions == leafIds.getSize());
}

//
//  Re-assign the labels of a bsp tree using the label store in the convex cell tree.
//  EXPECTS connectivity!

void hkcdConvexCellsTree3D::reassignSolidGeomLabels(_In_opt_ const ExecutionContext* executionCtx, const hkcdPlanarGeometry& originalGeom, _Inout_ hkcdPlanarSolid* solid, const hkArray<PolygonId>& originalBoundaryPolygonIds, _Inout_opt_ hkcdPlanarEntityDebugger* debugger)
{
    HK_ASSERT_NO_MSG(0xb37a654c, solid);

    // infer all the labels
    markBoundaryCells(executionCtx, originalGeom, originalBoundaryPolygonIds, debugger);
    if ( hkTask::receivedAbortRequest(executionCtx) )
    {
        return;
    }

    inferCellsLabels(debugger);
    if ( hkTask::receivedAbortRequest(executionCtx) )
    {
        return;
    }

    // Copy cell labels into the tree
    const hkcdPlanarSolid::NodeStorage* nodes = solid->getNodes();
    for (int ni = nodes->getCapacity() - 1; ni >= 0; ni--)
    {
        const NodeId nodeId(ni);
        Node& bspNode = solid->accessNode(nodeId);

        if ( bspNode.isAllocated() && ((bspNode.m_type == hkcdPlanarSolid::NODE_TYPE_IN) || (bspNode.m_type == hkcdPlanarSolid::NODE_TYPE_OUT)) )
        {
            // Get its associated cell
            const CellId cellId(bspNode.m_data);
            if ( !cellId.isValid() )
            {
                bspNode.m_type = hkcdPlanarSolid::NODE_TYPE_UNKNOWN;
                continue;
            }
            Cell& cell = accessCell(cellId);
            HK_ASSERT_NO_MSG(0x3d76c3d9, !cell.isUnknown());
            bspNode.m_type = hkUint16(cell.isSolid() ? hkcdPlanarSolid::NODE_TYPE_IN : hkcdPlanarSolid::NODE_TYPE_OUT);
        }
    }
}

//
//  Computes the approximate position of the vertices of the given convex cell

void hkcdConvexCellsTree3D::collectCellVerticesWorld(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<hkVector4>& verticesPos) const
{
    // Collect vertices ids
    hkArray<VertexId> vertexIds;
    collectUniqueVertexIds(cellIds, numCellIds, vertexIds);

    // Get offset and scale to be applied to vertices
    hkSimdDouble64 scaleD;
    hkVector4d offsetD;
    {
        HK_ALIGN16(hkDouble64) doubleBuff[4];
        hkVector4 offset = m_planes->getPositionOffset();
        offset.store<4, HK_IO_NATIVE_ALIGNED>(doubleBuff);  offsetD.load<4, HK_IO_NATIVE_ALIGNED>(doubleBuff);
        hkSimdReal scale = m_planes->getPositionScale();
        scaleD.setFromFloat(scale.getReal());
        scaleD.setReciprocal(scaleD);
    }

    // Add all of them
    verticesPos.reserve(verticesPos.getSize() + vertexIds.getSize());
    for (int v = 0, nv = vertexIds.getSize() ; v < nv ; v++)
    {
        // Create the vertex in world space
        hkVector4d fvd = m_data->accessVertices()[vertexIds[v]].m_pos;
        hkFloat32 realBuff[4];
        fvd.mul(scaleD);    fvd.add(offsetD);   fvd.store<4, HK_IO_NATIVE_ALIGNED>(realBuff);
        verticesPos.expandOne().load<4, HK_IO_NATIVE_ALIGNED>(realBuff);
    }
}

void hkcdConvexCellsTree3D::collectCellVerticesD(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<hkVector4d>& verticesPos) const
{
    // Collect vertices ids
    hkArray<VertexId> vertexIds;
    collectUniqueVertexIds(cellIds, numCellIds, vertexIds);

    verticesPos.reserve(verticesPos.getSize() + vertexIds.getSize());
    for (int v = 0, nv = vertexIds.getSize(); v < nv; v++)
    {
        verticesPos.pushBack(m_data->accessVertices()[vertexIds[v]].m_pos);
    }
}

void hkcdConvexCellsTree3D::collectCellVerticesInt(_In_reads_(numCellIds) const CellId* cellIds, int numCellIds, hkArray<hkIntVector>& verticesPlanes) const
{
    // Collect vertices ids
    hkArray<VertexId> vertexIds;
    collectUniqueVertexIds(cellIds, numCellIds, vertexIds);

    verticesPlanes.reserve(verticesPlanes.getSize() + vertexIds.getSize());
    for (int v = 0, nv = vertexIds.getSize(); v < nv; v++)
    {
        const Vertex& vertex = m_data->accessVertices()[vertexIds[v]];
        hkIntVector planes;
        for (int i = 0 ; i < 3 ; i++)
        {
            planes.setComponent(i, vertex.m_planeIds[i].value());
        }
        verticesPlanes.pushBack(planes);
    }
}
//
//  Extracts a solid planar geometry from a subset of selected cells

_Ret_maybenull_
hkcdPlanarSolid* hkcdConvexCellsTree3D::buildSolidFromSubsetOfCells(_In_opt_ const ExecutionContext* executionCtx, const hkArray<CellId>& cellIdsIn) const
{
    HK_ASSERT_NO_MSG(0x56a82e11, m_buildCellConnectivity);


    // Get all the boundary polygons from the cells
    hkArray<PolygonId> boundaryPolygonIds;
    hkcdPlanarGeometry geom(const_cast<hkcdPlanarGeometryPlanesCollection*>(m_planes.val()), 0, HK_NULL);
    if( extractBoundaryPolygonsFromCellIds(cellIdsIn, geom, boundaryPolygonIds).isFailure() )
    {
        return HK_NULL;
    }

    // Compute the corresponding bsp tree
    // If no polygon, no boundary is present, don't change the tree
    if ( boundaryPolygonIds.getSize() )
    {
        hkcdPlanarSolid* newSolidGeom = new hkcdPlanarSolid(geom.getPlanesCollection(), 0, geom.accessDebugger());

        // Build the tree
        newSolidGeom->buildBVHTree(executionCtx, geom, boundaryPolygonIds);

        return newSolidGeom;
    }

    return HK_NULL;
}

//
//  Collect all plane ids of a given cell

void hkcdConvexCellsTree3D::collectCellPlaneIds(CellId cellId, hkArray<PlaneId>& planeIdsOut) const
{
    const Cell& cell = getCell(cellId);
    for (int f = 0, nf = cell.getNumFaces() ; f < nf ; f++)
    {
        const Face& face = m_data->accessFaces()[cell.getFaceIdsPointer()[f]];
        planeIdsOut.pushBack(getCellFacePlaneId(face, cellId));
    }
}

//
//  Return a set of cell ids representing disconnected islands of solid regions

void hkcdConvexCellsTree3D::computeSolidRegionIslands(hkArray< hkArray<CellId> >& islands) const
{
    // Get all leaf cells and init them
    hkArray<CellId> leafIds;    collectLeafCells(leafIds);
    const int numLeaves         = leafIds.getSize();
    hkLocalBitField visited     (numLeaves, hkBitFieldValue::ZERO);
    hkArray<CellId> cellQueue   (numLeaves);

    // Sort the Ids so we can use binary search
    hkSort(leafIds.begin(), numLeaves);

    CellId startCellId;
    islands.clear();
    islands.reserve(8);

    do
    {
        startCellId = CellId::invalid();

        // Find a starting point to flood a solid island
        for (int i = numLeaves - 1 ; i >= 0 ; i--)
        {
            const CellId cellId = leafIds[i];
            const Cell& cell    = getCell(cellId);
            if ( !visited.get(i) && cell.isSolid() )
            {
                startCellId = cellId;
                break;
            }
        }

        // Flood the island
        if ( startCellId.isValid() )
        {
            int queueHead = 0;
            int queueTail = 0;
            visited.set(hkAlgorithm::binarySearch(startCellId, leafIds.begin(), numLeaves));

            cellQueue[queueTail++] = startCellId;

            hkArray<CellId>& islandCellIds = islands.expandOne();
            while ( queueHead < queueTail )
            {
                // Get the current cell
                const CellId cellId = cellQueue[queueHead++];
                const Cell& cell    = getCell(cellId);
                islandCellIds.pushBack(cellId);

                // Look at all its neighbors
                const int numFaces  = cell.getNumFaces();
                for (int pi = 0; pi < numFaces; pi++)
                {
                    const FaceId faceId = cell.getFaceIdsPointer()[pi];
                    const Face& face    = m_data->accessFaces()[faceId];

                    // Get the neighbor cell id
                    const CellId nCellId = ( face.m_negCellId == cellId ) ? face.m_posCellId : face.m_negCellId;
                    if ( nCellId.isValid() )
                    {
                        const Cell& nCell   = getCell(nCellId);
                        const int nCellIdx  = hkAlgorithm::binarySearch(nCellId, leafIds.begin(), numLeaves);

                        // Add the neighbor to queue
                        if ( nCell.isSolid() && !visited.get(nCellIdx) )
                        {
                            cellQueue[queueTail++] = nCellId;
                            visited.set(nCellIdx);
                        }
                    }
                }
            }
        }
    } while ( startCellId.isValid() );
}

//
//  Sets a new planes collection, remapping plane ids optionally.

void hkcdConvexCellsTree3D::setPlanesCollection(_In_opt_ const PlanesCollection* newPlanes, _In_reads_opt_(_Inexpressible_()) const int* HK_RESTRICT planeRemapTable)
{
    // Remap all plane Ids if necessary
    if ( planeRemapTable && m_planes && newPlanes )
    {
        // Remap vertices
        for (int v = 0, nv = m_data->getNumAllocatedVertices() ; v < nv ; v++)
        {
            Vertex& vertex = m_data->accessVertices()[v];
            if ( vertex.m_planeIds[0].isValid() )
            {
                for (int i = 0 ; i < 3 ; i++)
                {
                    const PlaneId oldPlaneId    = vertex.m_planeIds[i];

                    const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                    int newPlaneIdx             = planeRemapTable[oldPlaneIdx] & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                    Plane oldPlane;             m_planes->getPlane(oldPlaneId, oldPlane);
                    Plane newPlane;             newPlanes->getPlane(PlaneId(newPlaneIdx), newPlane);
                    const PlaneId newPlaneId(newPlaneIdx | (hkcdPlanarGeometryPredicates::sameOrientation(oldPlane, newPlane) ? 0 : hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG));

                    vertex.m_planeIds[i] = newPlaneId;
                }
            }
        }

        // Remap edges
        for (int e = 0, ne = m_data->getNumAllocatedEdges(); e < ne; e++)
        {
            Edge& edge = m_data->accessEdges()[e];
            if ( edge.m_vertexIds[0] != VertexId(-1) && edge.m_otherPlaneId.isValid() )
            {
                const PlaneId oldPlaneId    = edge.m_otherPlaneId;

                const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                int newPlaneIdx             = planeRemapTable[oldPlaneIdx] & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                Plane oldPlane;             m_planes->getPlane(oldPlaneId, oldPlane);
                Plane newPlane;             newPlanes->getPlane(PlaneId(newPlaneIdx), newPlane);
                const PlaneId newPlaneId(newPlaneIdx | (hkcdPlanarGeometryPredicates::sameOrientation(oldPlane, newPlane) ? 0 : hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG));

                 edge.m_otherPlaneId = newPlaneId;
            }
        }

        // Remap faces
        for (int f = 0, nf = m_data->getNumAllocatedFaces(); f < nf; f++)
        {
            Face& face = m_data->accessFaces()[f];
            if ( face.m_planeId.isValid() )
            {
                const PlaneId oldPlaneId    = face.m_planeId;

                const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                int newPlaneIdx             = planeRemapTable[oldPlaneIdx] & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                Plane oldPlane;             m_planes->getPlane(oldPlaneId, oldPlane);
                Plane newPlane;             newPlanes->getPlane(PlaneId(newPlaneIdx), newPlane);
                const PlaneId newPlaneId(newPlaneIdx | (hkcdPlanarGeometryPredicates::sameOrientation(oldPlane, newPlane) ? 0 : hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG));

                face.m_planeId = newPlaneId;
            }
        }
    }

    m_planes = newPlanes;
}

//
//  Shift all plane ids

void hkcdConvexCellsTree3D::shiftPlaneIds(int offsetValue)
{
    // Remap vertices
    for (int v = 0, nv = m_data->getNumAllocatedVertices() ; v < nv ; v++)
    {
        Vertex& vertex = m_data->accessVertices()[v];
        if ( vertex.m_planeIds[0].isValid() )
        {
            for (int i = 0 ; i < 3 ; i++)
            {
                const PlaneId oldPlaneId    = vertex.m_planeIds[i];

                const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                if ( oldPlaneIdx >= hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS )
                {
                    const PlaneId newPlaneId(oldPlaneId.value() + offsetValue);
                    vertex.m_planeIds[i] = newPlaneId;
                }
            }
        }
    }

    // Remap edges
    for (int e = 0, ne = m_data->getNumAllocatedEdges(); e < ne; e++)
    {
        Edge& edge = m_data->accessEdges()[e];
        if ( edge.m_vertexIds[0] != VertexId(-1) && edge.m_otherPlaneId.isValid() )
        {
            const PlaneId oldPlaneId    = edge.m_otherPlaneId;

            const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
            if ( oldPlaneIdx >= hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS )
            {
                const PlaneId newPlaneId(oldPlaneId.value() + offsetValue);
                edge.m_otherPlaneId = newPlaneId;
            }
        }
    }

    // Remap faces
    for (int f = 0, nf = m_data->getNumAllocatedFaces(); f < nf; f++)
    {
        Face& face = m_data->accessFaces()[f];
        if ( face.m_planeId.isValid() )
        {
            const PlaneId oldPlaneId    = face.m_planeId;

            const int oldPlaneIdx       = oldPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
            if ( oldPlaneIdx >= hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS )
            {
                const PlaneId newPlaneId(oldPlaneId.value() + offsetValue);
                face.m_planeId  = newPlaneId;
            }
        }
    }
}

//
//  Remaps the materials triangle source Ids

void hkcdConvexCellsTree3D::remapTriangleProviderIds(const hkArray<TriangleProviderId>& triSrcIdRemap)
{
    // Remap faces
    for (int f = 0, nf = m_data->getNumAllocatedFaces(); f < nf; f++)
    {
        Face& face = m_data->accessFaces()[f];
        if ( face.m_planeId.isValid() )
        {
            Material& dstMat = face.m_materialId;

            if ( dstMat.isValid() )
            {
                // Remap the tri source locally
                TriangleProviderId tsId = dstMat.getTriangleProviderId();
                dstMat.setTriangleProviderId(triSrcIdRemap[tsId.value()]);
            }
        }
    }
}

//
//  Collects all the plane Ids used by the Bsp tree

void hkcdConvexCellsTree3D::collectUsedPlaneIds(hkBitField& usedPlaneIdsOut) const
{

    // Remap vertices
    for (int v = 0, nv = m_data->getNumAllocatedVertices() ; v < nv ; v++)
    {
        Vertex& vertex = m_data->accessVertices()[v];
        if ( vertex.m_planeIds[0].isValid() )
        {
            for (int i = 0 ; i < 3 ; i++)
            {
                const int planeIdx = vertex.m_planeIds[i].value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
                usedPlaneIdsOut.set(planeIdx);
            }
        }
    }

    // Remap edges
    for (int e = 0, ne = m_data->getNumAllocatedEdges(); e < ne; e++)
    {
        Edge& edge = m_data->accessEdges()[e];
        if ( edge.m_vertexIds[0] != VertexId(-1) && edge.m_otherPlaneId.isValid() )
        {
            const int planeIdx = edge.m_otherPlaneId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
            usedPlaneIdsOut.set(planeIdx);
        }
    }

    // Remap faces
    for (int f = 0, nf = m_data->getNumAllocatedFaces(); f < nf; f++)
    {
        Face& face = m_data->accessFaces()[f];
        if ( face.m_planeId.isValid() )
        {
            const int planeIdx = face.m_planeId.value() & (~hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_FLAG);
            usedPlaneIdsOut.set(planeIdx);
        }
    }

    // Always add the world boundary planes
    for (int k = hkcdPlanarGeometryPlanesCollection::NUM_BOUNDS - 1 ; k >= 0 ; k--)
    {
        usedPlaneIdsOut.set(k);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
