// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/DataStructures/Planar/CSG/hkcdPlanarGeometryBooleanUtil.h>
#include <Geometry/Collide/DataStructures/Planar/Utils/hkcdPlanarGeometryWeldUtil.h>
#include <Geometry/Collide/DataStructures/Planar/ConvexCellsTree/hkcdConvexCellsTree3D.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/String/hkStringBuf.h>

//
//  Debug flags

#define ENABLE_TEXT_DEBUG   (0)

//
//  Types

typedef hkcdPlanarGeometryPlanesCollection  PlanesCollection;

//
//  Implementation

struct hkcdBspBooleanImpl : hkcdPlanarGeometryBooleanUtil::BooleanState
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdBspBooleanImpl);

        // Types
        typedef hkcdPlanarGeometry::Plane                   Plane;
        typedef hkcdPlanarGeometry::PlaneId                 PlaneId;
        typedef hkcdPlanarSolid::Node                       Node;
        typedef hkcdPlanarSolid::NodeId                     NodeId;
        typedef hkcdPlanarSolid::NodeTypes                  NodeType;
        typedef hkcdConvexCellsTree3D::Cell                 Cell;
        typedef hkcdConvexCellsTree3D::CellId               CellId;
        typedef hkcdPlanarGeometryBooleanUtil::Operation    Operation;
        typedef hkcdPlanarGeometryPredicates::Orientation   Orientation;
        typedef hkcdPlanarCsgOperand                        CsgOperand;
        typedef hkcdPlanarEntity::Material                  Material;

    public:

        /// Stack entry, for the iterative implementation of the BSP tree merge
        struct StackEntry
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY, hkcdBspBooleanImpl::StackEntry);

            /// Flags
            enum
            {
                IDX_TREE_A_BIT      = hkcdPlanarGeometryPrimitives::FLIPPED_PLANE_BIT,  ///< The index in m_trees of the tree acting as treeA.
                IS_RIGHT_CHILD_BIT  = IDX_TREE_A_BIT + 1,                               ///< The bit is set if the stack entry is the right child of m_parentNodeId A / B.
                COMPUTE_RESULT_A    = IS_RIGHT_CHILD_BIT + 1,                           ///< The bit is set if m_result[0] should be computed
                COMPUTE_RESULT_B    = COMPUTE_RESULT_A + 1,                             ///< The bit is set if m_result[1] should be computed
            };

            /// Sets the stack entry
            HK_INLINE void set( NodeId nodeId, NodeId parentNodeId)
            {
                m_nodeId        = nodeId.valueUnchecked();
                m_parentNodeId  = parentNodeId.valueUnchecked();
            }

            HK_INLINE NodeId getNodeId()            const   {   return NodeId(m_nodeId);                    }
            HK_INLINE NodeId getParentNodeId()      const   {   return NodeId(m_parentNodeId);              }

        protected:

            hkUint32 m_nodeId;
            hkUint32 m_parentNodeId;
        };

    public:

        // Constructor
        HK_INLINE hkcdBspBooleanImpl()
        :   m_operandToMerge(HK_NULL)
        ,   m_regions(HK_NULL)
        {
            m_result[0] = HK_NULL;
            m_result[1] = HK_NULL;
        }

        // Dummy assignment operator, to silence the compiler
        HK_INLINE void operator=(const hkcdBspBooleanImpl&) {}

        /// Initializes the boolean
        void setOperands(_In_opt_ const hkTask::ExecutionContext* executionCtx, _In_ const CsgOperand* treeA, _In_ const CsgOperand* treeB, bool computeOp, bool computeDualOp, Operation op, bool dynamicSplit);

        // Starts merging the trees
        void merge(_In_opt_ const hkTask::ExecutionContext* executionCtx, Operation op, bool computeOp, bool computeDualOp, bool simplifyResults = true);

    protected:

        // Evaluates an operation between leaf Bsp nodes
        HK_INLINE NodeType eval(NodeType labelA, Operation op, NodeType labelB)
        {
            // Build code. Each operand has 4 combinations, both operands have 16 combinations. We need to return the result as 2 bits.
            const int labelCode = ((labelA << 2) | labelB) << 1;
            HK_ASSERT_NO_MSG(0x7afe2e58, (labelA < 3) && (labelB < 3) && (labelCode < 32) && (op != hkcdPlanarGeometryBooleanUtil::OP_DIFFERENCE));

            const hkUint32 opCode   = (op == hkcdPlanarGeometryBooleanUtil::OP_UNION) ? 0xFFE4D5C7: 0xFFEAE4E3;
            const NodeType ret      = (NodeType)((opCode >> labelCode) & 3);
            return ret;
        }

        /// Recursively merges the given trees
        void doMerge(_In_opt_ const hkTask::ExecutionContext* executionCtx, _In_ const CsgOperand* treeToMergeWith, Operation op, bool computeOp, bool computeDualOp);

        /// Initialize a stack with the leaves of the given tree
        void initializeStackWithLeavesForDualOp(_Inout_ CsgOperand* leftOperand, NodeId treeBRootNodeId, hkArray<StackEntry>& stack, _Inout_opt_ hkArray<NodeId>* startNodeIds);

        /// flips a material
        static HK_INLINE Material HK_CALL flipMaterial(const Material& matIn);

    public:

        hkRefPtr<CsgOperand> m_leftOperand;             ///< Left operand
        hkRefPtr<const CsgOperand> m_operandToMerge;    ///< right operand
        hkcdConvexCellsTree3D* m_regions;           ///< The cell graph, used to track the feasible regions
        bool m_swapInputTrees;

        /// The resulting Bsp trees. In the case of UNION, only m_result[0] is evaluated, while for INTERSECTION, both m_result[0] and m_result[1] are evaluated.
        /// In the latter case, m_result[0] = INTERSECTION and m_result[1] = DIFFERENCE, as the resulting trees are generally identical albeit with flipped labels.
        hkRefPtr<CsgOperand> m_result[2];

};

//
//  flips a material

HK_INLINE hkcdPlanarEntity::Material hkcdBspBooleanImpl::flipMaterial(const Material& matIn)
{
    if ( matIn.isValid() )
    {
        Material ret = matIn;
        ret.setFlags(ret.getFlags() ^ hkcdPlanarEntity::Material::IS_FLIPPED);
        return ret;
    }

    return Material::invalid();
}

//
//  Merge the input tree in the results.

void hkcdBspBooleanImpl::doMerge(_In_opt_ const hkTask::ExecutionContext* executionCtx, _In_ const CsgOperand* operandToMergeWith, Operation op, bool computeOp, bool computeDualOp)
{
    hkArray<StackEntry> stack;
    const NodeType flipLabel[]      = { hkcdPlanarSolid::NODE_TYPE_INTERNAL, hkcdPlanarSolid::NODE_TYPE_OUT,
                                        hkcdPlanarSolid::NODE_TYPE_IN, hkcdPlanarSolid::NODE_TYPE_UNKNOWN };

    const bool needBothResults      = ( computeOp && computeDualOp );
    const bool invertRightOperand   = ( !needBothResults && computeDualOp && !m_swapInputTrees );

    // Copy tree A into results
    hkcdPlanarSolid* solid                  = m_leftOperand->accessSolid();
    hkcdConvexCellsTree3D* regions          = m_leftOperand->accessRegions();
    const hkcdPlanarSolid* treeToMergeWith  = operandToMergeWith->getSolid();

    // Push the tree leaves on the stack
    hkArray<NodeId> startNodeIds;
    initializeStackWithLeavesForDualOp(m_leftOperand, treeToMergeWith->getRootNodeId(), stack, ( needBothResults ) ? &startNodeIds : HK_NULL);

    // Iterate
    while ( stack.getSize() && !hkTask::receivedAbortRequest(executionCtx) )
    {
        // Pop the first element on the stack
        const StackEntry entry = stack[0];
        stack.removeAt(0);
        NodeId nodeRId          = entry.getNodeId();
        NodeId nodeToMergeId    = entry.getParentNodeId();
        const Node& nodeToMerge = treeToMergeWith->getNode(nodeToMergeId);

        // If the region is empty, return the special NULL node (we'll reuse the unknown node for that!)
        Node& nodeRes               = solid->accessNode(nodeRId);
        const CellId regionId       = CellId(nodeRes.m_data);
        if ( !regionId.isValid() )
        {
            nodeRes.m_type = hkcdPlanarSolid::NODE_TYPE_UNKNOWN;
            continue;
        }

        // Get node labels
        NodeType labelLeftOp, labelRightOp;
        labelLeftOp     = solid->getNodeLabel(nodeRId);
        labelRightOp    = treeToMergeWith->getNodeLabel(nodeToMergeId);
        labelRightOp    = ( invertRightOperand ) ? flipLabel[labelRightOp] : labelRightOp;

        // Check if it's worth to split the region represented by the leaf cell of A
        if ( labelLeftOp == hkcdPlanarSolid::NODE_TYPE_IN || labelLeftOp == hkcdPlanarSolid::NODE_TYPE_OUT )
        {
            const NodeType In_op_B  = eval(hkcdPlanarSolid::NODE_TYPE_IN , op, labelLeftOp);
            const NodeType Out_op_B = eval(hkcdPlanarSolid::NODE_TYPE_OUT, op, labelLeftOp);
            if ( In_op_B == Out_op_B )
            {
                continue;
            }
        }

        // If the node of the tree B is a leaf, we can terminate!
        if ( labelRightOp == hkcdPlanarSolid::NODE_TYPE_IN || labelRightOp == hkcdPlanarSolid::NODE_TYPE_OUT )
        {
            nodeRes.m_type = labelRightOp;
            regions->accessCell(regionId).setLeaf(true);
            regions->accessCell(regionId).setLabel(( labelRightOp == hkcdPlanarSolid::NODE_TYPE_IN ) ? hkcdNewCellsCollection::CELL_SOLID : hkcdNewCellsCollection::CELL_EMPTY);
            continue;
        }

        // Main recursion. Get splitting plane from the current node of B
        CellId insideRegionId, outsideRegionId;
        const PlaneId splitPlaneId  = nodeToMerge.m_planeId;

        // Split current region with the splitting plane
        regions->splitCell(regionId, splitPlaneId, nodeToMerge.m_material, insideRegionId, outsideRegionId, op != hkcdPlanarGeometryBooleanUtil::OP_UNION);

        // Replace the current result node of tree the node of tree B and allocate its children
        nodeRes.m_planeId       = splitPlaneId;
        nodeRes.m_material      = invertRightOperand ? flipMaterial(nodeToMerge.m_material) : nodeToMerge.m_material;
        nodeRes.m_type          = hkcdPlanarSolid::NODE_TYPE_INTERNAL;

        // Recurse on both subtrees of A
        {
            NodeId newChildId0  = NodeId::invalid();
            newChildId0         = solid->createNode(PlaneId::invalid(), NodeId::invalid(), NodeId::invalid());
            Node& node0         = solid->accessNode(newChildId0);
            node0.m_parent      = nodeRId;
            node0.m_data        = insideRegionId.valueUnchecked();
            if ( insideRegionId.isValid() ) regions->accessCell(insideRegionId).setUserData(newChildId0.value());
            node0.m_type        = hkcdPlanarSolid::NODE_TYPE_UNKNOWN;

            NodeId newChildId1  = NodeId::invalid();
            newChildId1         = solid->createNode(PlaneId::invalid(), NodeId::invalid(), NodeId::invalid());
            Node& node1         = solid->accessNode(newChildId1);
            node1.m_parent      = nodeRId;
            node1.m_data        = outsideRegionId.valueUnchecked();
            if ( outsideRegionId.isValid() ) regions->accessCell(outsideRegionId).setUserData(newChildId1.value());
            node1.m_type        = hkcdPlanarSolid::NODE_TYPE_UNKNOWN;

            Node& node          = solid->accessNode(nodeRId);       // Pointers may have changed due to realocaation of nodes
            node.setChildId(0, newChildId0);
            node.setChildId(1, newChildId1);
        }


        // Set up the stack for the next recursion step
        StackEntry* se = stack.expandBy(2);
        se[0].set(solid->getNode(nodeRId).m_left, nodeToMerge.m_left);
        se[1].set(solid->getNode(nodeRId).m_right, nodeToMerge.m_right);
    }

    // Write results
    if ( !hkTask::receivedAbortRequest(executionCtx) )
    {
        if ( needBothResults )
        {
            m_result[0] = m_leftOperand;

            // Compute dual solid
            m_result[1].setAndDontIncrementRefCount(new hkcdPlanarCsgOperand());
            m_result[1]->copyData(*m_result[0]);
            hkcdPlanarSolid* dualSolid = m_result[1]->accessSolid();
            for ( int nIdx = startNodeIds.getSize() - 1; nIdx >=0; nIdx-- )
            {
                dualSolid->invertNodeLabels(startNodeIds[nIdx]);
            }
        }
        else
        {
            if ( computeOp )
            {
                m_result[0] = m_leftOperand;
            }
            if ( computeDualOp )
            {
                m_result[1] = m_leftOperand;
            }
        }
    }

    // Done
    m_leftOperand       = HK_NULL;
    m_operandToMerge    = HK_NULL;
}

void hkcdBspBooleanImpl::initializeStackWithLeavesForDualOp(_Inout_ CsgOperand* leftOperand, NodeId treeBRootNodeId, hkArray<StackEntry>& stack, _Inout_opt_ hkArray<NodeId>* startNodeIds)
{
    hkcdPlanarSolid* solid  = leftOperand->accessSolid();
    NodeId nodeId;
    nodeId = solid->getRootNodeId();

    // Put the root on the stack
    {
        StackEntry& entry   = stack.expandOne();
        entry.set(nodeId, treeBRootNodeId);
    }

    // Recursively adds all the leaves
    int head = 1;
    int queue = 0;
    while ( queue < head )
    {
        const StackEntry& entry = stack[queue];

        nodeId              = entry.getNodeId();
        const Node& node    = solid->getNode(nodeId);

        if ( node.m_type == hkcdPlanarSolid::NODE_TYPE_INTERNAL )
        {
            // Remove the current node from the stack: it's not a leaf
            stack.removeAt(queue);

            // Add the two children as potential leaves
            StackEntry& leftEntry = stack.expandOne();
            leftEntry.set(solid->getNode(nodeId).m_left, treeBRootNodeId);

            //const Node& rightNode = treeA->getNode(node.m_right);
            StackEntry& rightEntry = stack.expandOne();
            rightEntry.set(solid->getNode(nodeId).m_right, treeBRootNodeId);

            head++;
        }
        else
        {
            if ( startNodeIds )
            {
                Node& nodeToFlip = solid->accessNode(nodeId);
                if ( nodeToFlip.m_type == hkcdPlanarSolid::NODE_TYPE_IN )   // Check only in node cause this happens only when building both normal and dual results, trees not swapped
                {
                    startNodeIds->pushBack(nodeId);
                }
            }

            queue++;
        }
    }
}

//
//  Starts merging the trees

void hkcdBspBooleanImpl::merge(_In_opt_ const hkTask::ExecutionContext* executionCtx, Operation op, bool computeOp, bool computeDualOp, bool simplifyResults)
{
    // Performs the merge
    HK_TIMER_BEGIN("Merge", HK_NULL);
    doMerge(executionCtx, m_operandToMerge, op, computeOp, computeDualOp);
    HK_TIMER_END();

    const hkcdConvexCellsTree3D* regions    = ( m_result[0] && m_result[0]->getRegions() ) ? m_result[0]->getRegions() : \
                                             (( m_result[1] && m_result[1]->getRegions() ) ? m_result[1]->getRegions() : HK_NULL);
    const bool canCollapse                  = regions && !regions->hasRepaintedFaces() && !regions->hasManifoldCells();

    // Treat results
    HK_TIMER_BEGIN("SimplifyOperands", HK_NULL);
    for (int k = 0; k < 2; k++)
    {
        CsgOperand* result = m_result[k];

        // Optimize tree
        if ( result && result->getSolid() )
        {
            hkcdPlanarSolid* solid = result->accessSolid();

            if ( solid->isValid() )
            {
                if ( simplifyResults )
                {
                    if ( canCollapse )
                    {
                        solid->collapseUnknownLabels();
                        solid->collapseIdenticalLabels();
                        solid->optimizeStorage();
                    }

                   if ( result->simplifyFromBoundaries(executionCtx).isFailure() )
                    {
                        result = HK_NULL;
                        return;
                    }
                }
                else
                {
                    solid->collapseUnknownLabels();
                    solid->collapseIdenticalLabels();
                    solid->optimizeStorage();
                }
            }
        }
    }
    HK_TIMER_END();
}

//
//  Initializes the boolean

void hkcdBspBooleanImpl::setOperands(_In_opt_ const hkTask::ExecutionContext* executionCtx, _In_ const CsgOperand* operandA, _In_ const CsgOperand* operandB, bool computeOp, bool computeDualOp, Operation op, bool dynamicSplit)
{
#if ( ENABLE_TEXT_DEBUG )
    {
        Log_Info( "hkcdBspBooleanImpl::setOperands" );
        Log_Info( "Operand A" );
        operandA->getSolid()->dbgPrint();
        Log_Info( "Operand B" );
        operandB->getSolid()->dbgPrint();
    }
#endif

    HK_TIMER_BEGIN("SetOperands", HK_NULL);

    HK_ASSERT_NO_MSG(0x728e434, operandA && operandA->getSolid() && operandA->getSolid()->isValid());
    HK_ASSERT_NO_MSG(0x728e434, operandB && operandB->getSolid() && operandB->getSolid()->isValid());

    const bool needBothResults = ( computeOp && computeDualOp );
    if ( needBothResults || computeDualOp )
    {
        m_swapInputTrees = false;       // If both result are needed, there is no point swapping for a best solution
                                        // Also force operand A to be the left on when computing dual result to avoid potentially conflicting material on B to have priority
    }
    else
    {
        if ( dynamicSplit )
        {
            m_swapInputTrees = true;
        }
        else
        {
            const hkUint32 label    = (op == hkcdPlanarGeometryBooleanUtil::OP_UNION) ? hkcdPlanarSolid::NODE_TYPE_OUT : hkcdPlanarSolid::NODE_TYPE_IN;
            const hkUint32 labelOpp = (op == hkcdPlanarGeometryBooleanUtil::OP_UNION) ? hkcdPlanarSolid::NODE_TYPE_IN : hkcdPlanarSolid::NODE_TYPE_OUT;
            const int nbMinOpsIfNotSwapped = operandA->getSolid()->computeNumNodesWithLabel(label);
            if ( computeDualOp )
            {
                m_swapInputTrees = operandB->getSolid()->computeNumNodesWithLabel(labelOpp) < nbMinOpsIfNotSwapped;
            }
            else
            {
                const int nbMinOpsBIfNotSwapped = operandB->getSolid()->computeNumNodesWithLabel(label);
                if (nbMinOpsIfNotSwapped != nbMinOpsBIfNotSwapped)
                {
                    m_swapInputTrees = nbMinOpsBIfNotSwapped < nbMinOpsIfNotSwapped;
                }
                else
                {
                    m_swapInputTrees = operandB->getSolid()->computeNumLeafNodes() < operandA->getSolid()->computeNumLeafNodes();
                }
            }
        }
    }

    // Choose an operand to initialize the results, and the "operand to be merged with"
    hkRefPtr<CsgOperand> newOperandA = HK_NULL, newOperandB = HK_NULL;
    //m_leftOperand     = m_swapInputTrees ? operandB : operandA;
    m_operandToMerge    = m_swapInputTrees ? operandA : operandB;

    // If the trees have different geometries, merge them now into the new one
    const PlanesCollection* planesColA = operandA->getSolid()->getPlanesCollection();
    const PlanesCollection* planesColB = operandB->getSolid()->getPlanesCollection();
    if ( planesColA != planesColB )
    {
        newOperandA.setAndDontIncrementRefCount(new CsgOperand());
        {
            newOperandA->copyData(*operandA);
        }

        newOperandB.setAndDontIncrementRefCount(new CsgOperand());
        {
            newOperandB->copyData(*operandB);
        }

        // Set left and right operands
        m_leftOperand       = m_swapInputTrees ? newOperandB : newOperandA;
        m_operandToMerge    = m_swapInputTrees ? newOperandA : newOperandB;

        // Must build a new geometry, shared between the two trees
        // Create the merged collection
        hkArray<int> remapTable;
        const int numPlanesA = planesColA->getNumPlanes();
        PlanesCollection* planesColAB = PlanesCollection::createMergedCollection(planesColA, planesColB, &remapTable);

        // Re-index the planes inside the new trees
        newOperandA->setPlanesCollection(planesColAB, &remapTable[0]);
        newOperandB->setPlanesCollection(planesColAB, &remapTable[numPlanesA]);
        planesColAB->removeReference();
    }
    else
    {
        // Copy only the left operand
        m_leftOperand.setAndDontIncrementRefCount(new CsgOperand());
        {
            m_leftOperand->copyData(m_swapInputTrees ? *operandB : *operandA, dynamicSplit);
        }
    }

    // Determine whether there is a plane material conflict
    bool hasPlaneMaterialConflict = false;
    if ( op == hkcdPlanarGeometryBooleanUtil::OP_UNION )
    {
        // Get planes of left and right operands with valid material
        hkArray<PlaneId> planeIdsLeft, planeIdsRight, tmpArray;
        m_leftOperand->getSolid()->getPlaneIdsWithValidMaterial(planeIdsLeft);
        m_operandToMerge->getSolid()->getPlaneIdsWithValidMaterial(planeIdsRight);
        tmpArray.reserve(hkMath::max2(planeIdsLeft.getSize(), planeIdsRight.getSize()));

        // see if there is a conflict
        hasPlaneMaterialConflict =
            ( hkAlgorithm::intersectionOfSortedLists(planeIdsLeft.begin(), planeIdsLeft.getSize(), planeIdsRight.begin(), planeIdsRight.getSize(), tmpArray.begin()) > 0 );
    }

    // Invert the tree if necessary
    const bool invertLeftOperand    = ( !needBothResults && computeDualOp && m_swapInputTrees );
    if ( invertLeftOperand )
    {
        m_leftOperand->accessSolid()->invertNodeLabels(m_leftOperand->accessSolid()->getRootNodeId());
    }

    // Compute the convex cell trees of the results if it is still not there...
    m_leftOperand->getOrCreateConvexCellTree(executionCtx, hasPlaneMaterialConflict, true);
    m_result[0] = HK_NULL;
    m_result[1] = HK_NULL;

    HK_TIMER_END();
}

//
//  Computes A op B

void HK_CALL hkcdPlanarGeometryBooleanUtil::compute(_In_opt_ const hkTask::ExecutionContext* executionCtx,
    _In_opt_ const hkcdPlanarCsgOperand* operandA, Operation op, _In_opt_ const hkcdPlanarCsgOperand* operandB,
    _Inout_opt_ hkRefPtr<const hkcdPlanarCsgOperand>* resultOut, _Inout_opt_ hkRefPtr<const hkcdPlanarCsgOperand>* dualResultOut, bool dynamicSplit)
{
    // If we are required to perform a difference, mark tree B as flipped and do an intersection
    bool computeOp      = resultOut != HK_NULL;
    bool computeDualOp  = dualResultOut != HK_NULL;

    int resultIdx = 0;
    if ( op == OP_DIFFERENCE )
    {
        resultIdx = 1;
        op = OP_INTERSECTION;
        hkAlgorithm::swap(computeOp, computeDualOp);
    }

    // Handle null inputs
    const bool validTreeA = operandA && operandA->getSolid() && operandA->getSolid()->isValid();
    const bool validTreeB = operandB && operandB->getSolid() && operandB->getSolid()->isValid();
    if ( !validTreeA || !validTreeB )
    {
        // Either one of:
        //      Intersection with at least one null operand, empty intersection!
        //      Union, with solidA == NULL, return solidB
        //      Union, with solidA != NULL, return solidA
        if ( resultOut )
        {
            *resultOut = (op == OP_INTERSECTION) ? HK_NULL : (operandA ? operandA : operandB);
        }
        return;
    }

    // Start the algorithm
    hkcdBspBooleanImpl impl;
    impl.setOperands(executionCtx, operandA, operandB, computeOp, computeDualOp, op, dynamicSplit);
    if ( !hkTask::receivedAbortRequest(executionCtx) )
    {
        impl.merge(executionCtx, op, computeOp, computeDualOp, !dynamicSplit);
    }

    // Return the result of the Boolean operation
    if ( resultOut )
    {
        *resultOut = impl.m_result[resultIdx];
    }
    if ( dualResultOut )
    {
        *dualResultOut = impl.m_result[1 - resultIdx];
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
