// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/DataStructures/Planar/CSG/hkcdPlanarCsgOperand.h>
#include <Geometry/Collide/DataStructures/Planar/CSG/hkcdPlanarGeometryBooleanUtil.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

//
//  Types

typedef hkcdPlanarGeometry::Plane                   Plane;
typedef hkcdPlanarGeometry::PlaneId                 PlaneId;
typedef hkcdPlanarGeometry::Polygon                 Polygon;
typedef hkcdPlanarGeometry::PolygonId               PolygonId;

//
//  Constructor

hkcdPlanarCsgOperand::hkcdPlanarCsgOperand()
:   hkReferencedObject()
,   m_solid(HK_NULL)
,   m_danglingGeometry(HK_NULL)
,   m_regions(HK_NULL)
{}


void hkcdPlanarCsgOperand::afterReflectNew()
{
    m_regions = HK_NULL;
}

//
//  Destructor

hkcdPlanarCsgOperand::~hkcdPlanarCsgOperand()
{
    m_regions = HK_NULL;
    m_solid = HK_NULL;
    m_danglingGeometry = HK_NULL;
}

//
//  Returns the convex cell tree corresponding to this solid planar geom. Build it if necessary.

_Ret_notnull_ hkcdConvexCellsTree3D* hkcdPlanarCsgOperand::getOrCreateConvexCellTree(_In_opt_ const ExecutionContext* executionCtx, bool withConnectivity, bool rebuildIfConnectivityDoesntMatch)
{
    if ( !m_regions || ( rebuildIfConnectivityDoesntMatch && (m_regions->hasManifoldCells() != withConnectivity)) )
    {
        // Build the tree now
        m_regions.setAndDontIncrementRefCount(new hkcdConvexCellsTree3D(m_solid->getPlanesCollection(), withConnectivity));
        m_regions->buildFromSolid(executionCtx, m_solid);
    }

    return m_regions;
}

//
//  Sets a new planes collection. If the plane remapping table is non-null, the plane Ids on all nodes will be re-set as well (i.e. to match the plane Ids in the new collection)

void hkcdPlanarCsgOperand::setPlanesCollection(_In_opt_ const PlanesCollection* newPlanes, _In_reads_(_Inexpressible_()) const int* HK_RESTRICT planeRemapTable)
{
    // Make sure our geometry and solid share the same planes collection
    HK_ASSERT_NO_MSG(0x54070aa4, !m_solid || !m_regions || (m_solid->getPlanesCollection() == m_regions->getPlanesCollection()));

    PlanesCollection* planesCol = const_cast<PlanesCollection*>(newPlanes);
    if ( m_danglingGeometry )
    {
        m_danglingGeometry->setPlanesCollection(planesCol, planeRemapTable);
    }

    if ( m_solid )
    {
        m_solid->setPlanesCollection(newPlanes, (int*)planeRemapTable);
    }

    if ( m_regions )
    {
        m_regions->setPlanesCollection(const_cast<PlanesCollection*>(newPlanes), (int*)planeRemapTable);
    }
}

//
//  Shift all plane ids of the operand elements

void hkcdPlanarCsgOperand::shiftPlaneIds(int offsetValue)
{
    // Make sure our geometry and solid share the same planes collection
    HK_ASSERT_NO_MSG(0x54070aa4, !m_solid || !m_regions || (m_solid->getPlanesCollection() == m_regions->getPlanesCollection()));

    if ( m_danglingGeometry )
    {
        m_danglingGeometry->shiftPlaneIds(offsetValue);
    }

    if ( m_solid )
    {
        m_solid->shiftPlaneIds(offsetValue);
    }

    if ( m_regions )
    {
        m_regions->shiftPlaneIds(offsetValue);
    }
}

//
//  Remaps the materials triangle source Ids

void hkcdPlanarCsgOperand::remapTriangleProviderIds(const hkArray<TriangleProviderId>& triSrcIdRemap)
{
    if ( m_danglingGeometry )
    {
        m_danglingGeometry->remapTriangleProviderIds(triSrcIdRemap);
    }

    if ( m_solid )
    {
        m_solid->remapTriangleProviderIds(triSrcIdRemap);
    }

    if ( m_regions )
    {
        m_regions->remapTriangleProviderIds(triSrcIdRemap);
    }
}

//
//  Retrieves the planes collection

_Ret_maybenull_
const hkcdPlanarGeometryPlanesCollection* hkcdPlanarCsgOperand::getPlanesCollection() const
{
    HK_ASSERT_NO_MSG(0x45126e8b, !m_solid || !m_regions || (m_solid->getPlanesCollection() == m_regions->getPlanesCollection()));

    if ( m_solid )      {   return m_solid->getPlanesCollection();      }
    if ( m_regions )    {   return m_regions->getPlanesCollection();    }

    return HK_NULL;
}

//
//  Simplifies this by rebuilding its tree from its boundaries

hkResult hkcdPlanarCsgOperand::simplifyFromBoundaries(_In_opt_ const ExecutionContext* executionCtx)
{
    // Compute the boundaries
    getOrCreateConvexCellTree(executionCtx, true, true);
    hkcdPlanarGeometry boundaries(accessPlanesCollection());
    hkArray<PolygonId> boundaryPolygonsIds;

    if ( !hkTask::receivedAbortRequest(executionCtx) )
    {
        // Extract the boundary
        hkArray<hkcdConvexCellsTree3D::CellId> solidCellIds;
        m_regions->collectSolidCells(solidCellIds);
       if ( m_regions->extractBoundaryPolygonsFromCellIds(solidCellIds, boundaries, boundaryPolygonsIds).isFailure() )
        {
            return HK_FAILURE;
        }
    }

    // If no polygon, no boundary is present, don't change the tree
    if ( boundaryPolygonsIds.getSize() )
    {
        m_solid->clear();

        // Build the tree
        m_solid->buildBVHTree(executionCtx, boundaries, boundaryPolygonsIds);
    }

    // Invalidate the convex cell tree (no longer valid)
    m_regions = HK_NULL;
    return HK_SUCCESS;
}

//
//  Builds dangling geometry form input operands and bool operation

void hkcdPlanarCsgOperand::buildDanglingGeometry(bool split, bool negPart, _In_ const hkcdPlanarSolid* cutoutSolid)
{
    if ( !m_danglingGeometry )
    {
        return;
    }

    if ( split )
    {
        // Classify dangling polys against the cutout solid
        hkArray<PolygonId> insidePolyIds, boundaryPolyIds, outsidePolyIds, originalPolyIds, tmpPolyIds;
        m_danglingGeometry->getAllPolygons(originalPolyIds);
        if ( cutoutSolid->classifyPolygons(*m_danglingGeometry, originalPolyIds, insidePolyIds, boundaryPolyIds, outsidePolyIds).isFailure() )
        {
            return;
        }

        // Create a new geom for dangling polys
        if ( negPart )
        {
            hkSort((PolygonId::Type*)outsidePolyIds.begin(), outsidePolyIds.getSize());
            m_danglingGeometry->keepPolygons(outsidePolyIds);
        }
        else
        {
            hkSort((PolygonId::Type*)insidePolyIds.begin(), insidePolyIds.getSize());
            m_danglingGeometry->keepPolygons(insidePolyIds);
        }
    }
}

//
//  Copy the desired data from another operand

void hkcdPlanarCsgOperand::copyData(const hkcdPlanarCsgOperand& operandSrc, bool copyRegions)
{
    HK_TIME_CODE_BLOCK_LIST( "CloneOperand", "CloneSolid" );
    if ( operandSrc.getSolid() )
    {
        hkcdPlanarSolid* clonedTree = new hkcdPlanarSolid(*operandSrc.getSolid());
        setSolid(clonedTree);
        clonedTree->removeReference();
    }

    HK_TIMER_SPLIT_LIST( "CloneRegions" );
    if ( copyRegions && operandSrc.getRegions() )
    {
        hkcdConvexCellsTree3D* clonedRegions = new hkcdConvexCellsTree3D(*operandSrc.getRegions());
        setRegions(clonedRegions);
        clonedRegions->removeReference();
    }

    HK_TIMER_SPLIT_LIST( "CloneDanglingPolys" );
    if ( operandSrc.getDanglingGeometry() )
    {
        hkcdPlanarGeometry* clonedGeom = new hkcdPlanarGeometry(*operandSrc.getDanglingGeometry());
        setDanglingGeometry(clonedGeom);
        clonedGeom->removeReference();
    }
}

/// Copy the desired data from another operand, using provided collection managers
void hkcdPlanarCsgOperand::shallowCopyData(const hkcdPlanarCsgOperand& operandSrc, _In_ const hkcdPlanarGeometryPlanesCollection* dstPlaneCollection)
{
    HK_TIME_CODE_BLOCK_LIST( "CopyOperand", "CopySolid" );
    if ( operandSrc.getSolid() )
    {
        hkcdPlanarSolid* srcSolid = (hkcdPlanarSolid*)operandSrc.getSolid();
        hkcdPlanarSolid* solid = new hkcdPlanarSolid(srcSolid->accessNodes(), srcSolid->getRootNodeId(), dstPlaneCollection);
        setSolid(solid);
        solid->removeReference();
    }
}

//  Create an operand with the same data, but a duplicated plane collection
void HK_CALL hkcdPlanarCsgOperand::createOperandWithSharedDataAndClonedPlanes(_In_ const hkcdPlanarCsgOperand* operandSrc, hkRefPtr<hkcdPlanarCsgOperand>& operandDst)
{
    // Create a new plane collection
    const PlanesCollection* srcPlanes   = operandSrc->getPlanesCollection();
    PlanesCollection* dstPlanes         = new PlanesCollection(*srcPlanes);
    hkcdPlanarCsgOperand* dstOperand    = new hkcdPlanarCsgOperand();
    dstOperand->setPlanesCollection(dstPlanes, HK_NULL);

    // Copy the solid (should not clone the nodes)
    hkcdPlanarSolid* srcSolid           = (hkcdPlanarSolid*)operandSrc->getSolid();
    if ( srcSolid )
    {
        hkRefPtr<hkcdPlanarSolid> dstSolid  = HK_NULL;
        dstSolid.setAndDontIncrementRefCount(new hkcdPlanarSolid(*srcSolid));
        dstSolid->setPlanesCollection(dstPlanes, HK_NULL);
        dstOperand->setSolid(dstSolid);
    }

    // Write output
    operandDst.setAndDontIncrementRefCount(dstOperand);
    dstPlanes->removeReference();
}

//
//  Collects a bit-field of plane Ids used by the operand

void hkcdPlanarCsgOperand::collectUsedPlaneIds(hkBitField& usedPlaneIdsOut) const
{
    if ( m_danglingGeometry )
    {
        m_danglingGeometry->collectUsedPlaneIds(usedPlaneIdsOut);
    }

    if ( m_solid )
    {
        m_solid->collectUsedPlaneIds(usedPlaneIdsOut);
    }

    if ( m_regions )
    {
        m_regions->collectUsedPlaneIds(usedPlaneIdsOut);
    }
}

//
//  Removes all planes not used by the entities

void hkcdPlanarCsgOperand::removeUnusedPlanes()
{
    // Get the plane collection and allocate the bit-field
    PlanesCollection* newPlanes     = accessPlanesCollection();
    const int numPlanes             = newPlanes->getNumPlanes();
    hkBitField usedPlaneIds;    usedPlaneIds.setSizeAndFill(0, numPlanes, 0);

    // Gather all the planes in use
    collectUsedPlaneIds(usedPlaneIds);

    // Clone the planes collection so we can perform the remap
    PlanesCollection oldPlanes(*newPlanes);
    newPlanes->addReference();
    setPlanesCollection(&oldPlanes, HK_NULL);

    // Remove the unused planes from the collection
    hkArray<int> remappedPlanes;
    usedPlaneIds.setNot(usedPlaneIds);
    newPlanes->removePlanes(usedPlaneIds, &remappedPlanes);

    // Set the new planes collection
    setPlanesCollection(newPlanes, remappedPlanes.begin());
    newPlanes->removeReference();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
