// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

//
// Predefined tree specializations.
//

namespace hkcdDynamicTree
{
        /// RebuildSAH Flags.
    enum RebuildSAH_Flags
    {
        RSAH_FLAG_MINIMIZE  =   0x01,   ///< Apply local rotations the improve tree quality.
        RSAH_FLAG_COMPACT   =   0x02,   ///< Compact indices after rebuild.
        RSAH_FLAG_DEFAULT   =   RSAH_FLAG_MINIMIZE | RSAH_FLAG_COMPACT
    };


    ///
    /// Tree.
    ///
    template <typename STORAGE>
    struct Tree : public STORAGE
    {
        HK_DECLARE_CLASS(Tree, New, Reflect);
        typedef typename STORAGE::Metric    Metric;     ///< Metric.
        typedef typename STORAGE::Node      Node;       ///< Node.
        typedef typename STORAGE::Index     Index;      ///< Index.
        typedef typename STORAGE::PackAabb  PackAabb;   ///< Storage AABB type.
        typedef typename STORAGE::RawAabb   RawAabb;    ///< Processing AABB type.

        enum { INVALID_INDEX = 0 };

        /// Tree insertion context
        struct Tic
        {
            Index       m_nodeA;
            Index       m_nodeB;
            hkSimdReal  m_saA;
            hkSimdReal  m_saB;
        };

        /// Constructor.
        HK_INLINE   Tree()                              { clear(); }

        /// Constructor.
        template <typename P0>
        HK_INLINE   Tree(P0 p0) : STORAGE(p0)           { clear(); }

        /// Constructor.
        template <typename P0,typename P1>
        HK_INLINE   Tree(P0 p0,P1 p1) : STORAGE(p0,p1)  { clear(); }

        //
        //  Add, remove and update operations.
        //

        /// Remove all nodes from the tree.
        HK_INLINE void      clear() { STORAGE::releaseAll(); m_root=0; m_path=m_numLeaves=0; }

        /// Insert a new leaf in the tree and return its index.
        template <typename DATA_TYPE>
        HK_INLINE Index     insert(const RawAabb& aabb, DATA_TYPE data);

        /// Insert a new leaf in the tree as a point and return its index.
        template <typename DATA_TYPE>
        HK_INLINE Index     insert(const hkVector4& point, DATA_TYPE data);

        /// Insert an array of hkVector4 or RawAabb and set data to item index.
        template <typename TYPE>
        HK_INLINE void      insert(const TYPE* items, int numItems, int firstIndex=0);

        /// Insert a sub-tree.
        template <typename SETLEAFINDEX>
        HK_NEVER_INLINE void        insert(const Tree& subTree, SETLEAFINDEX& sli);

        /// Insert a new leaf in the tree and return its index.
        /// Insertion fails if STORAGE::tryAllocateNode() reports failure. In this case res is to set to HK_FAILURE. Otherwise it is set to HK_SUCCESS.
        template <typename DATA_TYPE>
        HK_INLINE Index     tryInsert(const RawAabb& aabb, DATA_TYPE data, hkResult& res);

        /// Set the data associated with the given leaf.
        template <typename DATA_TYPE>
        HK_INLINE void      setNodeData(Index leaf, DATA_TYPE data);

        /// Update a leaf.
        HK_INLINE void      update(Index leaf, const RawAabb& aabb);

        /// Update a leaf using local rebalancing, faster than update but degrade tree quality slightly.
        HK_INLINE void      kineticUpdate(Index leaf, const RawAabb& aabb);

        /// Update a leaf using local refit, faster than kineticUpdate but degrade tree quality significantly.
        /// Note this method only expands AABBs.
        HK_INLINE void      refitUpdate(Index leaf, const RawAabb& aabb);

        /// Update a leaf using local refit, faster than kineticUpdate but degrade tree quality significantly.
        /// Note this method expands AND shrinks AABBs. Returns true iff the root AABB was changed.
        HK_INLINE bool      shrinkRefitUpdate(Index leaf, const RawAabb& aabb);

        /// Remove a leaf from the tree.
        HK_INLINE void      remove(Index leaf);

        /// Locate insertion point of an AABB.
        HK_INLINE Index     locateInsertionPoint(const RawAabb& aabb, Index insertionPoint = 0) const;

        //
        // Serialization.
        //

        #include <Geometry/Internal/DataStructures/DynamicTree/hkcdDynamicTreeSerialization.inl>

        /// Save to a structured binary stream.
        template <typename WRITER>
        inline void                 save(WRITER& writer) const;

        /// Load from a structured binary stream.
        template <typename READER>
        inline void                 load(READER& reader);

        //
        // Optimizations and other functions.
        //

        /// Return the memory foot-print of the tree in bytes.
        HK_INLINE int           getMemoryFootPrint() const { return sizeof(*this) + (m_numLeaves*2-1)*sizeof(Node); }

        /// Returns the AABB of the tree domain.
        HK_INLINE void      getDomain(RawAabb& aabbOut) const { if(m_root) getNode(m_root)->getAabb(aabbOut); else aabbOut.setEmpty(); }

        /// Copy the this tree to another. NOTE: indices are compacted in the output tree.
        template <typename TREE>
        inline void                 copyTo(TREE& output) const;

        /// Compute the traversal cost a the tree.
        inline hkSimdReal           computeCost() const;

        /// Incremental optimization of the tree.
        inline void                 optimizeIncremental(int passes, int lookahead = 2);

        /// Complete rebuild of the tree using discretized SAH.
        /// Note: Leaf indices are changed after rebuild.
        /// Recommended method for complete rebuild.
        /// For flags definition, refer to the RebuildSAH_Flags enum.
        HK_INLINE hkResult  rebuildSAH(int flags = RSAH_FLAG_DEFAULT, int numBins = 32, int bottomUpThreshold = 16);

        /// Complete rebuild of a tree branch using discretized SAH.
        inline hkResult             rebuildBranchSAH(Index branch, hkBool minimize, int numBins, int bottomUpThreshold);

        /// Complete rebuild of the tree using discretized SAH.
        /// This method attemp to find a global minima and is very slow.
        inline void                 rebuildGlobalSAH(int minBins, int maxBins, int step);

        /// Complete rebuild of the tree.
        /// This is using optimizeIncremental.
        /// Quality: Average, Speed: Very fast.
        inline void                 rebuildFast(int passes) { optimizeIncremental(m_numLeaves * passes); }

        /// Call rebuildFast(1) until no improvement greater than \a minImprovement is possible, return the number of iteration used.
        /// Note that each iteration is slightly slower than individual call to ::rebuildFast because ::computeCost needs to called as well.
        inline int                  rebuildFastAuto(hkReal minImprovement = 0.1f/*10%*/, int maxIterations = 32, hkBool report = false);

        /// Compact node indices to improved cache performances.
        inline void                 compactIndices();

        /// Detach a branch from the tree.
        inline void                 detachBranch(Index node, Index leafData, Tree& branchOut);

        /// Attach a branch to the tree.
        inline void                 attachBranch(const Tree& branch, Index leaf);

        /// Apply a left rotation to a node, returns the new root node.
        inline Index                rotateLeft(Index node);

        /// Apply a right rotation to a node, returns the new root node.
        inline Index                rotateRight(Index node);

        /// Rebalance the tree such as the maximum height delta is less or equal to maxDelta.
        inline void                 balance(int maxDelta, Index node = 0);

        /// Merge leaves.
        template <typename MERGER>
        inline void                 mergeLeaves(int numLeavesToMerge, MERGER& merger);

        /// Rebuild tree AABBs from leaves.
        inline void                 rebuildAabbsFromLeaves();

        /// Sort nodes in a depth-first fashion along a branch.
        template <typename MAPPING>
        inline void                 sortBranch(hkUint32 branch, MAPPING& mapping, int maxDepth = 65535);

        /// Get the level(X) ancestor of a node. (1->parent, 2->grand parent, and so on).
        HK_INLINE Index     getAnscenstor(Index node, int level) const;

        /// Get the surface area of a node AABB.
        HK_INLINE hkSimdReal    surfaceArea(Index node) const;

        /// Get the next leaf node if any, else return 0.
        HK_INLINE Index     nextLeafNode(Index node, Index root=0) const;

        /// Get a node pointer given its index.
        HK_INLINE Node*     getNode(Index node)                 { HK_ASSERT(0xC53DB6F6,node,"Null node index"); return STORAGE::getNodesBuffer()+node; }

        /// Get a node index given its pointer.
        HK_INLINE Index     getIndex(const Node* node) const    { HK_ASSERT(0xC53DB6F6,node,"Null node pointer"); return (Index)(node-STORAGE::getNodesBuffer()); }

        /// Get the level of a node.
        HK_INLINE int           getLevel(Index node) const          { int level=0; while(getNode(node)->getParent()) { ++level;node=getNode(node)->getParent(); } return level; }

        /// Get the height of a node.
        HK_INLINE int           getHeight(Index node) const;

        /// Get the number of leaves starting at this node.
        HK_INLINE int           countLeaves(Index node) const;

        /// Get the underlying storage current capacity.
        HK_INLINE int           getNodeCapacity() const             { return STORAGE::getNodeCapacity(); }

        /// Collect all the nodes starting at this node.
        HK_INLINE void      collectNodes(Index node, hkArray<Index>& indices) const;

        /// Collect all the leaves starting at this node.
        HK_INLINE void      collectLeaves(Index node, hkArray<Index>& indices) const;

        /// Collect all the leaves starting at this node.
        template <typename DATA_TYPE>
        HK_INLINE void      collectLeavesData(Index node, hkArray<DATA_TYPE>& data) const;

        /// Collect all the leaves starting at this node using collector.
        template <typename COLLECTOR>
        HK_INLINE void      collectLeavesWithCollector(Index node, COLLECTOR& collector) const;

        /// Convert from another dynamic tree.
        template <typename TREE_TYPE>
        HK_INLINE void      convertFromDynamicTree(const TREE_TYPE& other) { other.copyTo(*this); }

        /// Check the integrity of the tree.
        inline hkBool               checkIntegrity(Index root=0) const;

    protected:

        HK_INLINE   hkResult    internalInsert(Index leafIndex, Index insertionPoint);
        HK_INLINE   hkResult    internalInsert(Index leafIndex, Index insertionPoint, const RawAabb& leafAabb);
        HK_INLINE   Index       internalRemove(Index leafIndex, const RawAabb& aabb);
        HK_INLINE   Index       internalRemove(Index leafIndex) { RawAabb aabb; getNode(leafIndex)->getAabb(aabb); return internalRemove(leafIndex, aabb); }

        HK_INLINE Index     nextOptimizationNode();
        HK_INLINE void      replaceChild(Index node, Index child, Index newChild);
        HK_INLINE void      updateAabbFromChildren(Index node);
        HK_INLINE hkBool32  updateAabbFromChildrenAndCheckContainement(Index node, const RawAabb& aabb);
        HK_INLINE hkBool32  updateAabbFromChildrenAndCheckChange(Index node);

    public:

        //
        // Available on SPU
        //

        /// Return true is the tree is empty.
        HK_INLINE bool      isEmpty() const { return m_root == 0; }

        /// Return the root node index.
        HK_INLINE Index     getRoot() const { return m_root; }

        /// Get a constant node pointer given its index.
        HK_INLINE const Node*   getNode(Index node, bool useCache = false) const;

        /// Get the next left node if any, else return the next right node.
        HK_INLINE Index     nextLeftNode(Index node, Index root=0) const;

        /// Get the next right node if any, else return 0.
        HK_INLINE Index     nextRightNode(Index node, Index root=0) const;

        /// Get the index of the child node wrt. \a node.
        HK_INLINE int           internalIndex(Index node, Index child) const;

        //
        // Query support.
        //

        /// Capabilities
        enum { SUPPORT_STACKLESS_TRAVERSAL = 1, USE_SOFTWARE_CACHE = 1 };

        /// Query stack slot type.
        typedef typename STORAGE::Index     Slot;

        /// Type given to queries processLeaf / processLeaves method.
        /// Gives access to all information regarding a node being processed:
        struct NodeContext
        {
            HK_DECLARE_POD_TYPE();

            HK_INLINE void  updateFromIndex(const Tree& tree) { m_node = tree.getNode(m_index); m_node->getAabb(m_aabb); }

            HK_INLINE const Node*   operator->() const { return m_node; }

            RawAabb         m_aabb;     ///< The AABB of the node.
            const Node*     m_node;     ///< The node pointer (transient on SPU).
            Index           m_index;    ///< The node index.
        };

        /// Initialize root context.
        HK_INLINE void      initializeRootContext(NodeContext& context, bool useCache) const;

        /// Fetch the children of a given context.
        HK_INLINE void      fetchChildren(const NodeContext& context, NodeContext* children, bool useCache) const;

        /// Set context from slot.
        HK_INLINE void      setContextFromSlot(const Slot& slot, NodeContext& context, bool useCache) const;

        /// Set context from a node index.
        HK_INLINE void      setContextFromNode(Index node, NodeContext& contextOut, bool useCache) const;

        /// Set slot from context.
        HK_INLINE void      setSlotFromContext(const NodeContext& context, Slot& slot) const;

        /// Process leaf node.
        template <typename QUERY>
        HK_INLINE void      processLeaf(QUERY& q, const NodeContext& leaf) const;

    private:

        /// Node pair, used for merging.
        struct Pair
        {
            template <typename MERGER>
            static HK_INLINE hkSimdReal computeMetric(const Tree& tree, Index node, MERGER& mrg);
                   HK_INLINE hkBool32       operator<(const Pair& other) const { return m_metric < other.m_metric; }

            hkSimdReal  m_metric;
            Index       m_node;
        };

        /// SAH Rebuild task.
        struct SAHTask
        {
            HK_DECLARE_CLASS(SAHTask, NewOpaque);

            HK_INLINE SAHTask(Index parent=0, int side=-1) : m_parent(parent), m_side(side) {}
            HK_INLINE void init(Index parent, int side)
            {
                m_parent = parent;
                m_side = side;
            }

            hkArray<Index>  m_nodes;
            Index           m_parent;
            int             m_side;

        private:
            SAHTask( const SAHTask& other );
        };

        /// SAH Rebuild bin.
        struct SAHBin
        {
            HK_DECLARE_CLASS(SAHBin, NewOpaque);

            struct Sorter
            {
                HK_INLINE               Sorter(int axis, const Tree& tree) : m_axis(axis), m_tree(tree) {}
                HK_INLINE hkBool32  operator()(Index x, Index y) const;

                int         m_axis;
                const Tree& m_tree;
            };

            HK_INLINE   void reset() { m_nodes.clear(); m_aabb.setEmpty(); m_cost = 0.0f; }

            hkArray<Index>  m_nodes;
            RawAabb         m_aabb;
            hkReal          m_cost;
        };

    public:

        hkUint32    m_numLeaves;    ///< Number of leaves.
        hkUint32    m_path;         ///< Current optimization path.
        Index       m_root;         ///< Root of the tree.
    };

}

#include <Geometry/Internal/DataStructures/DynamicTree/hkcdDynamicTreeStorages.h>


namespace hkcdDynamicTree
{
        /// Pointer size indices and data, full precision AABBs.
    typedef Tree<DynamicStoragePtr> DefaultTreePtr;

        /// 32 bits indices and data, full precision AABBs.
    typedef Tree<DynamicStorage32>  DefaultTree48;

        /// 16 bits indices and data, full precision AABBs.
    typedef Tree<DynamicStorage16>  DefaultTree32;

        /// Integer 16 bits.
    typedef Tree<DynamicStorageInt16>   DefaultTreeInt16;
}

HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DefaultTreePtr, hkcdDynamicTree_DefaultTreePtr);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DefaultTree48, hkcdDynamicTree_DefaultTree48);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DefaultTree32, hkcdDynamicTree_DefaultTree32);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DefaultTreeInt16, hkcdDynamicTree_DefaultTreeInt16);


#include <Geometry/Collide/DataStructures/DynamicTree/hkcdDynamicTree.inl>
#include <Geometry/Collide/_Auto/TemplateTypes/hkcdDynamicTree_Types.inl>

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
