// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/TaskQueue/hkTaskGraph.h>

/// Base class for a fracture engine
class HK_EXPORT_COMMON hkcdFractureEngine : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY);
        HK_DECLARE_REFLECTION();

    public:

        /// Splitting result types
        enum SplitResult
        {
            SPLIT_SUCCESS           = 0,    ///< Success, everything went fine
            SPLIT_FAILURE           = 1,    ///< Something went wrong; either a critical general failure or one where the failing child couldn't be determined or it is irrelevant which once failed
            SPLIT_FAILURE_INSIDE    = 2,    ///< Something went wrong while splitting of child 0
            SPLIT_FAILURE_OUTSIDE   = 3,    ///< Something went wrong while splitting of child 1
            SPLIT_ABORTED           = 4,    ///< The operation was aborted
        };

        /// Engine type
        enum Type
        {
            TYPE_EXACT,         ///< The exact fracture engine
            TYPE_USER_0 = 16,   ///< All engine types are unused from here on
        };

    public:

        /// Fracture execution flags
        enum ExecutionFlagValues
        {
            CREATE_PHYSICS_SHAPES           = 0x0001,           ///< The fracture engine should create physics shapes for the fractured pieces
            CREATE_GRAPHICS_SHAPES          = 0x0002,           ///< The fracture engine should create graphics shapes for the fractured pieces
            RECOMPUTE_INSIDE_TRANSFORM      = 0x0004,           ///< If set, the fracture engine will ignore the provided m_hierarchyFromInsidePiece transform and recompute it to roughly match the inside physics shape's com.
            RECOMPUTE_OUTSIDE_TRANSFORM     = 0x0008,           ///< If set, the fracture engine will ignore the provided m_hierarchyFromOutsidePiece transform and recompute it to roughly match the outside physics shape's com.
            COMPUTE_INSIDE_SHAPES           = 0x0010,           ///< True if the inside shapes are to be computed / returned
            COMPUTE_OUTSIDE_SHAPES          = 0x0020,           ///< True if the outside shapes are to be computed / returned
            FRACTURE_PHYSICS                = 0x0040,           ///< True if the physics geometry should be split / merged
            FRACTURE_GRAPHICS               = 0x0080,           ///< True if the graphics geometry should be split / merged
            DYNAMIC_SPLIT                   = 0x0100,           ///< True if runtime fracture is desired (typically leveraging simpler geoms and heavier precomputations to skip costly steps)

            DEFAULT_EXECUTION_FLAGS =   CREATE_PHYSICS_SHAPES | CREATE_GRAPHICS_SHAPES |
            COMPUTE_INSIDE_SHAPES | COMPUTE_OUTSIDE_SHAPES |
            FRACTURE_PHYSICS | FRACTURE_GRAPHICS |
            RECOMPUTE_INSIDE_TRANSFORM | RECOMPUTE_OUTSIDE_TRANSFORM,
        };

        /// Topology types
        enum Topology
        {
            TOPOLOGY_CLOSED_MANIFOLD,           ///< The input mesh is a closed 2-manifold. The solid regions are bounded by the mesh polygons.
            TOPOLOGY_OPEN_SELF_INTERSECTING,    ///< The input mesh can have open edges or is self-intersecting. The solid regions are inferred by a space flooding algorithm.
            TOPOLOGY_UNKNOWN,                   ///< The input mesh has unknown type and its type will be automatically detected.
        };

        // Types
        typedef hkFlags<ExecutionFlagValues, hkUint16>  ExecutionFlags;
        typedef hkTaskGraph::TaskId                     TaskId;
        typedef hkTask::ExecutionContext                ExecutionContext;
        class FractureTask;

        /// Base class for a block of data shared between multiple tasks
        class HK_EXPORT_COMMON TaskDataBlock : public hkReferencedObject
        {
            public:

                HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_GEOMETRY);
                friend class FractureTask;

            public:

                /// Constructor
                TaskDataBlock();

                /// Copy constructor
                TaskDataBlock(const TaskDataBlock& other);

                /// Destructor
                virtual ~TaskDataBlock();

            public:

                /// Called to destroy the data payload once it no longer has dependent tasks.
                virtual void destroyPayload() {}

                /// Increases the numbers of tasks that depend on this data block
                void addDependency();

                /// Decreases the number of tasks that depend on this data block
                void removeDependency();

                /// Returns the Id of the task responsible for writing this data block
                HK_INLINE hkTaskGraph::TaskId getWriterTaskId() const { return m_writerTaskId;  }

            protected:

                /// Id of the task responsible for generating the payload of this data block
                hkTaskGraph::TaskId m_writerTaskId;

            private:

                /// Number of tasks that reference this data block
                hkUint32 m_numDependentTasks;
        };

        /// A non-referenced fracture task
        class HK_EXPORT_COMMON FractureTask : public hkTask
        {
            public:

                HK_DECLARE_CLASS(FractureTask, New);

                /// Constructor
                FractureTask(hkTaskGraph& taskGraphIn);

                /// Destructor
                virtual ~FractureTask() {}

                /// Processes the task
                virtual void process(const hkTask::Input&) HK_OVERRIDE;

                /// Adds the given data block as input of the current task
                void addInput(hkTaskGraph& taskGraphIn, _Inout_ TaskDataBlock* inputDataBlock);

                /// Adds the given data block as output of the current task
                void addOutput(_Inout_ TaskDataBlock* outputDataBlock);

                /// Clears the task memory
                void clear();

                /// Returns the Id of this task
                HK_INLINE hkTaskGraph::TaskId getId() const {   return m_id;    }

            protected:

                /// The array of inputs
                hkArray< hkRefPtr<TaskDataBlock> > m_inputs;

                /// The array of outputs
                hkArray< hkRefPtr<TaskDataBlock> > m_outputs;

                /// The Id of this task
                hkTaskGraph::TaskId m_id;
        };

    public:

        /// Constructor
        hkcdFractureEngine();

        // -------------------------------------------------------------------------------------------------------------
        //  Single-threaded interface
        // -------------------------------------------------------------------------------------------------------------

    public:

        /// Returns the type of the engine
        virtual Type getType() const = 0;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
