// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/Math/Vector/hkVector4Util.h>

//
//  Computes and returns the squared distance from vP to the triangle (vA, vB, vC). The normal and barycentric coordinates of the closest point on the triangle
//  are also provided on output.

inline hkSimdReal HK_CALL hkcdPointTriangleWithBarycentricCoord(
    hkVector4_ point, hkVector4_ vA, hkVector4_ vB, hkVector4_ vC,
    hkVector4* HK_RESTRICT normalOut, hkVector4* HK_RESTRICT baryOut )
{
    // Set point w component to 1 to compute the plane equation easier
    hkVector4 vP;
    {
        hkSimdReal one = hkSimdReal::getConstant<HK_QUADREAL_1>();
        vP.setXYZ_W(point, one);
    }

    // Compute plane equation
    hkVector4 planeEq;
    hkVector4 normal;
    hkcdTriangleUtil::calcUnitPlaneEquation(vA, vB, vC, planeEq);

    // Closest point in triangle computation
    {
        // Compute projection of P (i.e. Q) on the triangle plane
        hkSimdReal distanceToPlane = planeEq.dot<4>(vP);
        hkVector4 vQ;
        vQ.setSubMul(vP, planeEq, distanceToPlane);

        // Set normal to triangle normal. Will be overwritten later if case
        normal.setFlipSign(planeEq, distanceToPlane);

        // Determine whether the point is inside the triangle
        hkSimdReal epsilon = hkSimdReal::getConstant<HK_QUADREAL_EPS>();
        if ( hkcdIntersectPointTriangle(vQ, vA, vB, vC, planeEq, epsilon, baryOut) )
        {
            *normalOut = normal;
            return distanceToPlane * distanceToPlane;
        }
    }

    // Closest point is on one of the edges
    {
        hkVector4 vAB;  vAB.setSub(vB, vA);
        hkVector4 vBC;  vBC.setSub(vC, vB);
        hkVector4 vCA;  vCA.setSub(vA, vC);

        hkVector4 vAP;  vAP.setSub(vP, vA);
        hkVector4 vBP;  vBP.setSub(vP, vB);
        hkVector4 vCP;  vCP.setSub(vP, vC);

        // Compute edge lengths
        hkVector4 vEdgeLen, vSegLen;
        hkVector4Util::dot3_3vs3(vAP, vAB, vBP, vBC, vCP, vCA, vSegLen);
        hkVector4Util::dot3_3vs3(vAB, vAB, vBC, vBC, vCA, vCA, vEdgeLen);

        // Prevent division by zero
        vEdgeLen.setMax(vEdgeLen, hkVector4::getConstant<HK_QUADREAL_EPS>());

        // Compute times of intersection
        hkVector4 vTois;
        {
            vTois.setDiv(vSegLen, vEdgeLen);
            vTois.setClampedZeroOne(vTois);
        }

        // Compute normals & distances
        hkVector4 vDistances;
        HK_ALIGN_REAL(hkVector4) normals[3];
        {
            normals[0].setSubMul(vAP, vAB, vTois.getComponent<0>());
            normals[1].setSubMul(vBP, vBC, vTois.getComponent<1>());
            normals[2].setSubMul(vCP, vCA, vTois.getComponent<2>());

            hkVector4Util::dot3_3vs3(normals[0], normals[0], normals[1], normals[1], normals[2], normals[2], vDistances);
        }

        // Compute the index of the minimum length normal
        int bestIndex   = vDistances.getIndexOfMinComponent<3>();
        hkSimdReal minLen = vDistances.getComponent(bestIndex);
        normal = normals[bestIndex];

        // Compute barycentric coords if required
        if ( baryOut )
        {
            const hkVector4 vOne= hkVector4::getConstant<HK_QUADREAL_1>();

            hkVector4 b0;   b0.setSub(vOne, vTois); // (u0, u1, u2, *)
            hkVector4 b1 = vTois;                   // (v0, v1, v2, *)
            hkVector4 b2;   b2.setZero();           // (0, 0, 0, 0)

            // b0 = (u0, v0, 0, *)
            // b1 = (u1, v1, 0, *)
            // b2 = (u2, v2, 0, *)
            HK_TRANSPOSE3(b0, b1, b2);
            b1.setPermutation<hkVectorPermutation::ZXYZ>(b1);   // b1 = (0, u1, v1, 0)
            b2.setPermutation<hkVectorPermutation::YZXZ>(b2);   // b2 = (v2, 0, u2, 0)

            HK_ALIGN_REAL(hkVector4) bary[3] = { b0, b1, b2 };
            *baryOut = bary[bestIndex];

//          baryOut->setZero();
//          hkReal v = vTois(bestIndex);
//          (*baryOut)( bestIndex ) = 1.0f -v;
//          (*baryOut)( "\1\2\0\0"[bestIndex] ) = v;
        }

        *normalOut = normal;
        return minLen;
    }
}


template <enum hkcdPointTriangleCache::Enum CACHE, int RESULT_FLAGS>
HK_INLINE hkSimdReal HK_CALL hkcdDistancePointTriangleImpl(
    hkVector4_ x, hkVector4_ a, hkVector4_ b, hkVector4_ c, hkcdPointTriangleCache* HK_RESTRICT cache,
    hkcdPointTriangleResult* HK_RESTRICT result )
{
    hkVector4   ab; ab.setSub(b,a);
    hkVector4   bc; bc.setSub(c,b);
    hkVector4   ca; ca.setSub(a,c);

    hkVector4   ax; ax.setSub(x,a);
    hkVector4   bx; bx.setSub(x,b);
    hkVector4   cx; cx.setSub(x,c);

    hkVector4   n; n.setCross(ab,bc);

    hkVector4   cab; cab.setCross(ab, ax);
    hkVector4   cbc; cbc.setCross(bc, bx);
    hkVector4   cca; cca.setCross(ca, cx);

    hkVector4   signs; hkVector4Util::dot3_3vs3(n, cab, n, cbc, n, cca, signs);

    if ( RESULT_FLAGS & hkcdPointTriangleResult::TRIANGLE_NORMAL )
    {
        result->m_triangleNormal = n;
    }

    hkVector4   edgeLenSqrd;
    if ( CACHE == hkcdPointTriangleCache::BUILD)
    {
        hkVector4Util::dot3_4vs4(ab, ab, bc, bc, ca, ca, n, n, edgeLenSqrd);
        hkVector4   invEdgeLenSqrd; invEdgeLenSqrd.setReciprocal<HK_ACC_MID, HK_DIV_SET_ZERO>(edgeLenSqrd);
        cache->m_invEdgeLengthsSqrd = invEdgeLenSqrd;
    }

    if( signs.greaterZero().allAreSet<hkVector4ComparisonMask::MASK_XYZ>() )    // cannot be >= because of degenerated triangles
    {
        hkSimdReal  sol;
        hkSimdReal nLenSqrd;
        if ( CACHE == hkcdPointTriangleCache::USE || CACHE == hkcdPointTriangleCache::BUILD )
        {
            nLenSqrd = edgeLenSqrd.getW();
            sol.setMul( n.dot<3>(ax), cache->m_invEdgeLengthsSqrd.getW() );
        }
        else
        {
            nLenSqrd = n.lengthSquared<3>();
            sol.setDiv<HK_ACC_MID,HK_DIV_SET_ZERO>(n.dot<3>(ax), nLenSqrd);
        }
        result->m_direction = n;
        result->m_directionScale = sol;
        return nLenSqrd * (sol*sol);
    }
    else
    {
        hkVector4   sols;
        {
            hkVector4   nums; hkVector4Util::dot3_3vs3(ab, ax, bc, bx, ca, cx, nums);
            if ( CACHE == hkcdPointTriangleCache::USE || CACHE == hkcdPointTriangleCache::BUILD)
            {
                sols.setMul( cache->m_invEdgeLengthsSqrd, nums );
            }
            else
            {
                hkVector4   dens; hkVector4Util::dot3_3vs3(ab, ab, bc, bc, ca, ca, dens);
                sols.setDiv<HK_ACC_MID,HK_DIV_IGNORE>(nums, dens);  // we can use div ignore since clamp will fix our inf/nan
            }
        }
        sols.setClamped(sols, hkVec4_0, hkVec4_1);

        hkVector4   prj0, prj1, prj2;
        prj0.setSubMul(ax, ab, sols.getComponent<0>());
        prj1.setSubMul(bx, bc, sols.getComponent<1>());
        prj2.setSubMul(cx, ca, sols.getComponent<2>());

        // Find the projection vector with the smallest magnitude
        hkSimdReal len0sqrd = prj0.lengthSquared<3>();
        hkSimdReal len1sqrd = prj1.lengthSquared<3>();
        hkSimdReal len2sqrd = prj2.lengthSquared<3>();

        // make prj0 to min of prj0 and 1
        {
            prj0.setSelect( len0sqrd.less(len1sqrd), prj0, prj1 );
            len0sqrd.setMin( len0sqrd, len1sqrd );
        }
        // make prj0 to min of prj0 and 2
        {
            prj0.setSelect( len0sqrd.less(len2sqrd), prj0, prj2 );
            len0sqrd.setMin( len0sqrd, len2sqrd );
        }
        result->m_direction = prj0;
        result->m_directionScale = hkSimdReal_1;
        return len0sqrd;
    }
}


template <int RESULT_FLAGS>
HK_INLINE hkSimdReal HK_CALL hkcdDistancePointQuadImpl(
    hkVector4_ x, hkVector4_ a, hkVector4_ b, hkVector4_ c, hkVector4_ d,
    hkcdPointTriangleResult* HK_RESTRICT result )
{
    hkVector4   ab; ab.setSub(b,a);
    hkVector4   bc; bc.setSub(c,b);
    hkVector4   cd; cd.setSub(d,c);
    hkVector4   da; da.setSub(a,d);
    HK_ASSERT( 0xf0345465, (0==a.allExactlyEqual<3>(b)) && (0 == b.allExactlyEqual<3>(c)), "Degenerated triangles not allowed in hkcdDistancePointQuadImpl. However c==d is allowed " );
    hkVector4   ax; ax.setSub(x,a);
    hkVector4   bx; bx.setSub(x,b);
    hkVector4   cx; cx.setSub(x,c);
    hkVector4   dx; dx.setSub(x,d);

    hkVector4   n; n.setCross(ab,bc);

    hkVector4   cab; cab.setCross(ab, ax);
    hkVector4   cbc; cbc.setCross(bc, bx);
    hkVector4   ccd; ccd.setCross(cd, cx);
    hkVector4   cda; cda.setCross(da, dx);

    hkVector4   signs; hkVector4Util::dot3_4vs4(n, cab, n, cbc, n, ccd, n, cda, signs);

    if ( RESULT_FLAGS & hkcdPointTriangleResult::TRIANGLE_NORMAL )
    {
        result->m_triangleNormal = n;
    }

    if( signs.greaterEqualZero().allAreSet() )      // can be >= because of degenerated triangles are disallowed
    {
        hkSimdReal  nLenSqrd = n.lengthSquared<3>();
        hkSimdReal  sol; sol.setDiv<HK_ACC_MID,HK_DIV_SET_ZERO>(n.dot<3>(ax), nLenSqrd);
        result->m_direction = n;
        result->m_directionScale = sol;
        return nLenSqrd * (sol*sol);
    }
    else
    {
        hkVector4   sols;
        {
            hkVector4   nums; hkVector4Util::dot3_4vs4(ab, ax, bc, bx, cd, cx, da, dx, nums);
            hkVector4   dens; hkVector4Util::dot3_4vs4(ab, ab, bc, bc, cd, cd, da, da, dens);
            sols.setDiv<HK_ACC_MID,HK_DIV_IGNORE>(nums, dens);  // we can use div ignore since clamp will fix our inf/nan
        }
        sols.setClamped(sols, hkVec4_0, hkVec4_1);

        hkVector4   prj0, prj1, prj2, prj3;
        prj0.setSubMul(ax, ab, sols.getComponent<0>());
        prj1.setSubMul(bx, bc, sols.getComponent<1>());
        prj2.setSubMul(cx, cd, sols.getComponent<2>());
        prj3.setSubMul(dx, da, sols.getComponent<3>());

        // Find the projection vector with the smallest magnitude
        hkSimdReal len0sqrd = prj0.lengthSquared<3>();
        hkSimdReal len1sqrd = prj1.lengthSquared<3>();
        hkSimdReal len2sqrd = prj2.lengthSquared<3>();
        hkSimdReal len3sqrd = prj3.lengthSquared<3>();

        // make prj0 to min of prj0 and 1
        {
            prj0.setSelect( len0sqrd.less(len1sqrd), prj0, prj1 );
            len0sqrd.setMin( len0sqrd, len1sqrd );
            prj2.setSelect( len2sqrd.less(len3sqrd), prj2, prj3 );
            len2sqrd.setMin( len2sqrd, len3sqrd );
        }
        // make prj0 to min of prj0 and 2
        {
            prj0.setSelect( len0sqrd.less(len2sqrd), prj0, prj2 );
            len0sqrd.setMin( len0sqrd, len2sqrd );
        }
        result->m_direction = prj0;
        result->m_directionScale = hkSimdReal_1;
        return len0sqrd;
    }
}

template <int RESULT_FLAGS>
HK_INLINE hkSimdReal HK_CALL hkcdDistancePointTransposedQuadImpl(
    hkVector4_ v, hkVector4_ qx, hkVector4_ qy, hkVector4_ qz,
    hkcdPointTriangleResult* result )
{
#ifdef HK_DEBUG
    {
        hkVector4 a; a.set( qx( 0 ), qy( 0 ), qz( 0 ) );
        hkVector4 b; b.set( qx( 1 ), qy( 1 ), qz( 1 ) );
        hkVector4 c; c.set( qx( 2 ), qy( 2 ), qz( 2 ) );
        HK_ASSERT( 0xf0345565, (0 == a.allExactlyEqual<3>( b )) && (0 == b.allExactlyEqual<3>( c )), "Degenerated triangles not allowed in hkcdDistancePointQuadImpl. However c==d is allowed " );
    }
#endif

    // Calculate edges
    hkVector4 ex; ex.setPermutation<hkVectorPermutation::YZWX>( qx ); ex.sub( qx );
    hkVector4 ey; ey.setPermutation<hkVectorPermutation::YZWX>( qy ); ey.sub( qy );
    hkVector4 ez; ez.setPermutation<hkVectorPermutation::YZWX>( qz ); ez.sub( qz );

    // Transpose the query point
    hkVector4 vx; vx.setAll( v.getComponent<0>() );
    hkVector4 vy; vy.setAll( v.getComponent<1>() );
    hkVector4 vz; vz.setAll( v.getComponent<2>() );

    // Calculate vector from each quad vertex to v
    hkVector4 dx; dx.setSub( vx, qx );
    hkVector4 dy; dy.setSub( vy, qy );
    hkVector4 dz; dz.setSub( vz, qz );

    // Calculate the quad normal direction (not normalized)
    hkVector4 n; 
    {
        hkVector4 ezxy0 = ez;    // ezxy0 = ( ez.getX(), ex.getX(), ey.getX(), 0 );
        hkVector4 ezxy1 = ex;    // ezxy1 = ( ez.getY(), ex.getY(), ey.getY(), 0 );
        hkVector4 ezxy2 = ey;
        hkMath::transpose(ezxy0,ezxy1, ezxy2 );
        hkVector4 eyzx0; eyzx0.setPermutation<hkVectorPermutation::ZXYW>( ezxy0 );
        hkVector4 eyzx1; eyzx1.setPermutation<hkVectorPermutation::ZXYW>( ezxy1 );
        n = ezxy1 * eyzx0 - ezxy0 * eyzx1;
    }

    hkVector4 signs;
    {
        // Cross edges with dx, dy, dz
        hkVector4 cx = dz * ey - ez * dy;
        hkVector4 cy = dx * ez - ex * dz;
        hkVector4 cz = dy * ex - ey * dx;

        // Dot crosses with n
        signs = cx * n.getComponent<0>() + cy * n.getComponent<1>() + cz * n.getComponent<2>();
    }

    if ( RESULT_FLAGS & hkcdPointTriangleResult::TRIANGLE_NORMAL )
    {
        result->m_triangleNormal = n;
    }

    if ( signs.greaterEqualZero().allAreSet() )     // can be >= because of degenerated triangles are disallowed
    {
        hkSimdReal  sol;
        hkSimdReal nLenSqrd = n.lengthSquared<3>();
        hkVector4 diff; diff.set( dx.getComponent<0>(), dy.getComponent<0>(), dz.getComponent<0>(), hkSimdReal_0 );
        sol.setDiv<HK_ACC_MID, HK_DIV_SET_ZERO>( n.dot<4>( diff ), nLenSqrd );
        result->m_direction = n;
        result->m_directionScale = sol;
        return nLenSqrd * (sol*sol);
    }
    else
    {
        hkVector4   sols;
        {
            hkVector4   nums = dx * ex + dy * ey + dz * ez;
            hkVector4   dens = ex * ex + ey * ey + ez * ez;
            sols.setDiv<HK_ACC_MID, HK_DIV_IGNORE>( nums, dens );   // we can use div ignore since clamp will fix our inf/nan
        }
        sols.setClamped( sols, hkVec4_0, hkVec4_1 );

        hkVector4 prjx = dx - ex * sols;
        hkVector4 prjy = dy - ey * sols;
        hkVector4 prjz = dz - ez * sols;

        // Find the projection vector with the smallest magnitude
        hkVector4 lengthSquared = prjx * prjx + prjy * prjy + prjz * prjz;

        hkSimdReal minLengthSquared = lengthSquared.horizontalMinI<4>();
        const hkVector4fComparison isMin = (lengthSquared <= minLengthSquared);
        int minIndex = isMin.getIndexOfFirstComponentSetElseZero();

        result->m_direction.set( prjx( minIndex ), prjy( minIndex ), prjz( minIndex ) );
        result->m_directionScale = hkSimdReal_1;
        return minLengthSquared;
    }
}

HK_INLINE hkSimdReal HK_CALL hkcdPointTriangleProject(
    hkVector4_ x, hkVector4_ a, hkVector4_ b, hkVector4_ c,
    hkVector4* HK_RESTRICT projectionOut )
{
    hkcdPointTriangleCache cache;
    hkcdPointTriangleResult result;
    hkSimdReal distSqrd = hkcdDistancePointTriangleImpl<hkcdPointTriangleCache::BUILD, 0>( x,a,b,c, &cache, &result );
    projectionOut->setSubMul( x, result.m_direction, result.m_directionScale );
    return distSqrd;
}


HK_INLINE hkSimdReal HK_CALL hkcdPointTriangleProjectWithTriNormal(
    hkVector4_ x, hkVector4_ a, hkVector4_ b, hkVector4_ c,
    hkVector4* HK_RESTRICT projectionOut, hkVector4* HK_RESTRICT triangleNormalOut )
{
    hkcdPointTriangleCache cache;
    hkcdPointTriangleResult result;
    hkSimdReal distSqrd = hkcdDistancePointTriangleImpl<hkcdPointTriangleCache::BUILD, hkcdPointTriangleResult::TRIANGLE_NORMAL>( x,a,b,c, &cache, &result );
    projectionOut->setSubMul( x, result.m_direction, result.m_directionScale );
    *triangleNormalOut = result.m_triangleNormal;
    return distSqrd;
}


HK_INLINE hkcdTriangleVoronoi::Region HK_CALL hkcdPointTriangleProjectWithVoronoi(
    hkVector4_ x, hkVector4_ a, hkVector4_ b, hkVector4_ c,
    hkVector4* HK_RESTRICT projectionOut )
{
    hkVector4   loBounds; loBounds.setXYZ_W(hkVector4::getConstant<HK_QUADREAL_0>(), hkSimdReal::getConstant<HK_QUADREAL_MINUS_MAX>());
    hkVector4   hiBounds; hiBounds.setXYZ_W(hkVector4::getConstant<HK_QUADREAL_1>(), hkSimdReal::getConstant<HK_QUADREAL_MAX>());

    hkVector4   ab; ab.setSub(b,a);
    hkVector4   bc; bc.setSub(c,b);
    hkVector4   ca; ca.setSub(a,c);

    hkVector4   ax; ax.setSub(x,a);
    hkVector4   bx; bx.setSub(x,b);
    hkVector4   cx; cx.setSub(x,c);

    hkVector4   n; n.setCross(ab,bc);

    hkVector4   cab; cab.setCross(ab, ax);
    hkVector4   cbc; cbc.setCross(bc, bx);
    hkVector4   cca; cca.setCross(ca, cx);

    hkVector4   signs; hkVector4Util::dot3_3vs3(n, cab, n, cbc, n, cca, signs);

    hkcdTriangleVoronoi::Region ret = hkcdTriangleVoronoi::INSIDE;

    if(signs.greater(loBounds).allAreSet<hkVector4ComparisonMask::MASK_XYZ>())
    {
        hkSimdReal  sol; sol.setDiv<HK_ACC_12_BIT,HK_DIV_SET_ZERO>(n.dot<3>(ax), n.lengthSquared<3>());
        projectionOut->setSubMul(x, n, sol);
    }
    else
    {
        hkVector4   dens; hkVector4Util::dot3_3vs3(ab, ab, bc, bc, ca, ca, dens);
        hkVector4   nums; hkVector4Util::dot3_3vs3(ab, ax, bc, bx, ca, cx, nums);

        hkVector4   sols; sols.setDiv<HK_ACC_12_BIT,HK_DIV_SET_ZERO>(nums, dens);
        sols.setClamped(sols, loBounds, hiBounds);

        hkVector4   prj[3];
        prj[0].setSubMul(ax, ab, sols.getComponent<0>());
        prj[1].setSubMul(bx, bc, sols.getComponent<1>());
        prj[2].setSubMul(cx, ca, sols.getComponent<2>());

        hkVector4   distancesSquared;
        hkVector4Util::dot3_3vs3(prj[0], prj[0], prj[1], prj[1], prj[2], prj[2], distancesSquared);

        static const hkcdTriangleVoronoi::Region regions[3][3] =
        {
            { hkcdTriangleVoronoi::VERTEX_0 , hkcdTriangleVoronoi::EDGE_0 , hkcdTriangleVoronoi::VERTEX_1 } ,
            { hkcdTriangleVoronoi::VERTEX_1 , hkcdTriangleVoronoi::EDGE_1 , hkcdTriangleVoronoi::VERTEX_2 } ,
            { hkcdTriangleVoronoi::VERTEX_2 , hkcdTriangleVoronoi::EDGE_2 , hkcdTriangleVoronoi::VERTEX_0 }
        };

        const int           minDist = distancesSquared.getIndexOfMinComponent<3>();
        const hkSimdReal    sol = sols.getComponent(minDist);
#if 0 && defined(HK_ARCH_INTEL)
        int  solRegion = sol.greaterZero().getMask(hkVector4ComparisonMask::Mask(1)) + sol.equal(hkSimdReal_1).getMask(hkVector4ComparisonMask::::Mask(1));
#else
        int solRegion = 1;
        if(sol.isEqualZero())               --solRegion;
        else if(sol.isEqual(hkSimdReal_1))  ++solRegion;
#endif
        projectionOut->setSub(x, prj[minDist]);

        ret = regions[minDist][solRegion];
    }

    return ret;
}

/// This is a dummy function to test potential performance of capsule vs triangle cast,
/// this function is not correct yet.
HK_INLINE hkBool32 hkCdDisjointTriangleVsSweptCaspule(
    hkVector4_ capsA, hkVector4_ capsB, hkSimdReal_ radius, hkVector4_ vel,
    hkVector4_ a, hkVector4_ b, hkVector4_ c )
{
    hkVector4   ab; ab.setSub(b,a);
    hkVector4   bc; bc.setSub(c,b);
    hkVector4   ca; ca.setSub(a,c);

    hkVector4   n; n.setCross(ab,bc);

    hkVector4   cab; cab.setCross(ab, vel);
    hkVector4   cbc; cbc.setCross(bc, vel);
    hkVector4   cca; cca.setCross(ca, vel);
    hkVector4 len2; hkVector4Util::dot3_4vs4( cab,cab, cbc,cbc, cca,cca, n,n, len2 );

    hkVector4   ax; ax.setSub(capsA,a);
    hkVector4   bx; bx.setSub(capsA,b);
    //hkVector4 cx; cx.setSub(capsA,c); // we can use ax for this
    hkVector4 dotsA; hkVector4Util::dot3_4vs4( ax,cab, bx,cbc, ax,cca, ax, n, dotsA );

    hkVector4 invLen; invLen.setSqrtInverse<HK_ACC_MID, HK_SQRT_SET_ZERO>( len2 );

    ax.setSub(capsB,a);
    bx.setSub(capsB,b);
    //cx.setSub(capsB,c);
    hkVector4 dotsB; hkVector4Util::dot3_4vs4( ax,cab, bx,cbc, ax,cca, ax, n, dotsB );

    hkSimdReal sign = ax.dot3(cbc); // needed to check the winding
    invLen.setFlipSign( invLen, sign );
    hkVector4 dots; dots.setMin( dotsA, dotsB );
    dots *= invLen;
    hkVector4 radV; radV.setAll( radius );
    dots -= radV;

    return dots.greaterZero().anyIsSet<hkVector4ComparisonMask::MASK_XYZ>();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
