// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_REFLECTION_PARSER_EXCLUDE_FILE

#pragma once

#include <Geometry/Collide/Algorithms/Triangle/hkcdTriangleUtil.h>
#include <Geometry/Collide/Algorithms/Intersect/hkcdIntersectPointTriangle.h>
#include <Geometry/Internal/Algorithms/Gsk/hkcdGskData.h>

/// Identify the Voronoi region of a point projected to a triangle.
struct HK_EXPORT_COMMON hkcdTriangleVoronoi
{
    enum Region
    {
        INVALID     =   -1,
        VERTEX_0    =   0,
        EDGE_0      =   1,
        VERTEX_1    =   2,
        EDGE_1      =   3,
        VERTEX_2    =   4,
        EDGE_2      =   5,
        INSIDE      =   6
    };

    enum Feature
    {
        VERTEX      =   0,
        EDGE        =   1,
        FACE        =   2
    };

    static HK_INLINE hkBool32   isEdge(Region r) { return (hkBool32) (r&1); }
    static HK_INLINE hkBool32   isVertex(Region r) { return r < 6 ? (hkBool32)((r+1)&1) : (hkBool32) hkFalse32; }
    static HK_INLINE Feature    getFeature(Region r) { return r == 6 ? FACE : (r&1 ? EDGE : VERTEX); }
};

struct hkcdPointTriangleCache
{
    enum Enum
    {
        CIGNORE,
        BUILD,
        USE
    };
    hkVector4 m_invEdgeLengthsSqrd; // w is normal length
};

struct hkcdPointTriangleResult
{
    enum Enum
    {
        DEFAULT = 0,                ///< just set m_direction and m_directionScale
        TRIANGLE_NORMAL = 1<<0,     ///< if set, the triangle normal will be reported
    };

    /// calc the hit normal, if query point is on an edge of the triangle, return the triangle normal or 1,0,0 if triangle is degenerate
    HK_INLINE hkVector4 calcNormal()
    {
        hkVector4 n; n.setSelect( m_direction.lengthSquared<3>().equalZero(), m_triangleNormal, m_direction );
        n.setNormalizedEnsureUnitLength<3>(n);
        n.setFlipSign( n, m_directionScale );
        return n;
    }

    /// calc the hit normal not normalized
    HK_INLINE hkVector4 calcNormalDirection() { hkVector4 n = m_direction; n.setFlipSign( n, m_directionScale ); return n; }

    /// calc the point projected onto the target triangle
    HK_INLINE hkVector4 calcProjectedOntoTriangle( hkVector4_ queryPoint ) { hkVector4 p; p.setSubMul( queryPoint, m_direction, m_directionScale ); return p; }


    hkVector4 m_direction;          ///< the shortest direction between triangle and query, not normalized and sign is not defined!
    hkSimdReal  m_directionScale;       ///< if you multiply direction with this you get the vector (triangle to query), warning can be negative.
    hkVector4 m_triangleNormal;     ///< the triangle normal, only set if result flags contains hkcdPointTriangleResult::TRIANGLE_NORMAL
};


/// Computes and returns the squared distance from vP to the triangle (vA, vB, vC).
/// The normal and optionally the barycentric coordinates of the closest point on the triangle
/// are also provided on output.
HK_INLINE hkSimdReal HK_CALL hkcdPointTriangleWithBarycentricCoord(
    hkVector4_ vP, hkVector4_ vA, hkVector4_ vB, hkVector4_ vC,
    hkVector4* normalOut, hkVector4* baryOut );


/// Project a point on a triangle, returns the distance squared.
/// Note that the triangle normal is not normalized.
HK_INLINE hkSimdReal HK_CALL hkcdPointTriangleProject(
    hkVector4_ vP, hkVector4_ vA, hkVector4_ vB, hkVector4_ vC,
    hkVector4* HK_RESTRICT projectionOut );

/// Same as hkcdPointTriangleProject(), just the triangle normal is returned, this is useful for 'pseudo' signed distances
HK_INLINE hkSimdReal HK_CALL hkcdPointTriangleProjectWithTriNormal(
    hkVector4_ vP,  hkVector4_ vA, hkVector4_ vB, hkVector4_ vC,
    hkVector4* HK_RESTRICT projectionOut, hkVector4* HK_RESTRICT triangleNormalOut );

/// Project a point on a triangle, returns the voronoi region that the projected point is in.
/// Note that the triangle normal is not normalized.
HK_INLINE hkcdTriangleVoronoi::Region HK_CALL hkcdPointTriangleProjectWithVoronoi(
    hkVector4_ vP, hkVector4_ vA, hkVector4_ vB, hkVector4_ vC,
    hkVector4* HK_RESTRICT projectionOut );


/// implementation of point triangle, you need to pass in the result flags to get the appropriate values in the result structure
template <enum hkcdPointTriangleCache::Enum CACHE, int RESULT_FLAGS>
HK_INLINE hkSimdReal HK_CALL hkcdDistancePointTriangleImpl(
    hkVector4_ x, hkVector4_ a, hkVector4_ b, hkVector4_ c, hkcdPointTriangleCache* cache,
    hkcdPointTriangleResult* result );

/// Distance between a point and a triangle.
/// Degenerate triangles are not allowed except quad[2] == quad[3] is OK, quad must be convex and flat
template <int RESULT_FLAGS>
HK_INLINE hkSimdReal HK_CALL hkcdDistancePointQuadImpl(
    hkVector4_ x, hkVector4_ a, hkVector4_ b, hkVector4_ c, hkVector4_ d,
    hkcdPointTriangleResult* result );

/// Distance between a point v and a transposed quad qx, qy, qz.
/// Degenerate triangles are not allowed except quad[2] == quad[3] is OK, quad must be convex and flat
template <int RESULT_FLAGS>
HK_INLINE hkSimdReal HK_CALL hkcdDistancePointTransposedQuadImpl(
    hkVector4_ v, hkVector4_ qx, hkVector4_ qy, hkVector4_ qz,
    hkcdPointTriangleResult* result );

/// Distance between a segment and a quad. Algorithm is numerical sensitive, so please make sure to use small values, e.g.
/// do everything in space of the segment instead of the space of the quad (a segment is typically a character controller with a centered capsule,
/// the quad typically comes from a large mesh).
/// Degenerate triangles are not allowed except quad[2] == quad[3] is OK
HK_EXPORT_COMMON hkSimdReal HK_CALL hkcdDistanceSegmentQuad(
    hkVector4_ x0, hkVector4_ x1, const hkVector4* quad,
    hkcdGsk::GetClosestPointOutput* HK_RESTRICT result );

#include <Geometry/Collide/Algorithms/Distance/hkcdDistancePointTriangle.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
