// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Geometry/Collide/Algorithms/Distance/hkcdDistancePointConvex.h>
#include <Geometry/Collide/Algorithms/Distance/hkcdDistancePointTriangle.h>
#include <Geometry/Collide/Algorithms/ClosestPoint/hkcdClosestPointSegmentSegment.h>

hkSimdReal HK_CALL hkcdDistanceSegmentQuad( hkVector4_ x0, hkVector4_ x1,   const hkVector4* quad,  hkcdGsk::GetClosestPointOutput* HK_RESTRICT result)
{
    // find the point which is closest to the triangle center (ignore the quad)
    hkVector4 a = quad[0];  hkVector4 b = quad[1];      hkVector4 c = quad[2];      hkVector4 d = quad[3];
    hkVector4 center = (a+b+c) * hkSimdReal_Inv3;
    hkSimdReal dist0 = center.distanceToSquared( x0 );
    hkSimdReal dist1 = center.distanceToSquared( x1 );

    hkVector4Comparison selectD0 = dist0.less(dist1);
    hkVector4 c0; c0.setSelect( selectD0, x0,x1);
    hkVector4 c1; c1.setSelect( selectD0, x1,x0);

    // now c0 should be the closer point, try that first
    hkcdPointTriangleResult ptresult;
    ptresult.m_triangleNormal.setZero();    // silence compiler warning
    for (int i =0; i < 2; i++)
    {
        hkSimdReal distSqrd = hkcdDistancePointQuadImpl<hkcdPointTriangleResult::TRIANGLE_NORMAL>(c0, a,b,c,d, &ptresult );
        hkVector4 direction = ptresult.calcNormalDirection();

        hkVector4 c01; c01.setSub( c1, c0 );
        hkSimdReal dir = c01.dot3( direction );
        if ( dir > -hkSimdReal_Eps )
        {
            // done, we found the separating axis
            direction.normalize<3, HK_ACC_MID, HK_SQRT_SET_ZERO>();
            result->m_distance  = distSqrd.sqrt<HK_ACC_MID, HK_SQRT_SET_ZERO>();
            result->m_pointAinA = c0;
            result->m_normalInA = direction;
            return distSqrd;
        }
        // swap points and try again
        hkMath::swap( c0,c1);
    }

    // now we do have an edge edge collision
    hkVector4 c01; c01.setSub( c1, c0 );
    hkVector4 ab; ab.setSub( b, a );
    hkVector4 bc; bc.setSub( c, b );
    hkVector4 cd; cd.setSub( d, c );
    hkVector4 da; da.setSub( a, d );
    hkVector4 t, u;
    hkcdClosestPointSegment4Segments( c0, c01, a,ab, b,bc, c,cd, d,da, t, u );

    hkVector4 d0 = c0-a + (c01*t.getComponent<0>() - ab*u.getComponent<0>());   // c0-a first to improve accuracy
    hkVector4 d1 = c0-b + (c01*t.getComponent<1>() - bc*u.getComponent<1>());
    hkVector4 d2 = c0-c + (c01*t.getComponent<2>() - cd*u.getComponent<2>());
    hkVector4 d3 = c0-d + (c01*t.getComponent<3>() - da*u.getComponent<3>());

    hkVector4 edgeDistSqrd; hkVector4Util::dot3_4vs4( d0,d0, d1,d1, d2,d2, d3,d3, edgeDistSqrd );
    hkVector4 ds[4] = { d0,d1,d2,d3 };
    int edgeIndex = edgeDistSqrd.getIndexOfMinComponent<4>();

    hkSimdReal te = u.getComponent(edgeIndex);
    hkVector4 pointOnEdge = quad[edgeIndex] * (hkSimdReal_1-te) + quad[ 3&(edgeIndex+1)] * te; // setInterpolate but ensures that getting second point if t == 1

    hkSimdReal distSqrd = edgeDistSqrd.getComponent(edgeIndex);
    hkVector4 direction  = ds[edgeIndex];   // pointing from triangle to capsule

    direction.normalize<3, HK_ACC_MID, HK_SQRT_SET_ZERO>();

    // optimistically set the result assuming edge edge collision
    result->m_pointAinA = c0 + c01 * t.getComponent(edgeIndex); // optimistically set the hitpoint
    result->m_distance  = distSqrd.sqrt<HK_ACC_MID, HK_SQRT_SET_ZERO>();
    result->m_normalInA = direction;

    if ( te.isGreaterZero() && te.isLess(hkSimdReal_1))
    {
        // now we do have a proper edge edge collision, check for penetration case
        // by checking if direction is indeed a separating axis
        hkVector4 edge = quad[ 3 & (edgeIndex+1) ] - quad[edgeIndex];
        direction.subMul( edge, direction.dot3(edge) / edge.lengthSquared<3>() );   // make orthonormal to the edge

        hkVector4 sepCheck0 = quad[ 3 & (edgeIndex-1) ] - pointOnEdge;  // edge is between edgeindex and edgeindex+1
        hkVector4 sepCheck1 = quad[ 3 & (edgeIndex+2) ] - pointOnEdge;

        hkSimdReal sepDist0 = direction.dot3( sepCheck0 );
        hkSimdReal sepDist1 = direction.dot3( sepCheck1 );
        hkSimdReal sepMax; sepMax.setMax( sepDist0, sepDist1 );
        hkSimdReal sepMin; sepMin.setMin( sepDist0, sepDist1 );
        sepMin.setMin( sepMin, hkSimdReal_0 );

        hkSimdReal threshold = result->m_distance * hkSimdReal::fromFloat(0.1f) - sepMin * hkSimdReal::fromFloat(0.3f);
        if ( sepMax < threshold  )  // all
        {
            // now we have no penetration, simply return the shortest edge edge distance
            return distSqrd;
        }
    }
    else
    {
        // now a triangle vertex is the closest, that means we can't be penetrating
        return distSqrd;
    }

    // penetration, we simply reverse the sepDist, but we also have to check
    // if one of the vertices of the capsule is closer to the triangle plane
    // (this can only happen if that vertex is inside the triangle).
    hkVector4 triNormal = ptresult.m_triangleNormal;
    triNormal.normalize<3, HK_ACC_MID, HK_SQRT_IGNORE>();   // triangle normal can not be 0, otherwise sepDist would have been zero as well

    hkSimdReal c0Dist = triNormal.dot3( c0 - quad[0] );
    hkSimdReal c1Dist = triNormal.dot3( c1 - quad[0] );

    hkSimdReal c0Dist2 = c0Dist * c0Dist;
    hkSimdReal c1Dist2 = c1Dist * c1Dist;
    if ( c0Dist2 > c1Dist2 )
    {
        hkMath::swap( c0Dist, c1Dist);
        c0Dist2 = c1Dist2;
        c0 = c1;
    }
    if ( c0Dist2 < distSqrd )
    {
        // now the vertex c0 is closer
        result->m_normalInA.setFlipSign( triNormal, c1Dist );   // use c1Dist, this also solves non penetrating situations
        result->m_pointAinA  = c0;
        hkSimdReal absDist; absDist.setAbs(c0Dist);
        result->m_distance.setFlipSign( absDist, c0Dist*c1Dist);    // use neg distance if the cxDist have different signs
        return c0Dist2;
    }

    // now we have a penetrating edge edge situation. just reverse the sign
    direction.normalize<3, HK_ACC_MID, HK_SQRT_SET_ZERO>();
    result->m_normalInA = -direction;
    result->m_distance = -result->m_distance;
    return distSqrd;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
