// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Geometry/Collide/hkcdCollide.h>
#include <Geometry/Collide/Algorithms/ConvexHullFaceOptimizer/hkcdConvexHullFaceOptimizer.h>

#include <Common/Internal/GeometryProcessing/CollisionGeometryOptimizer/hkgpCgo.h>
#include <Common/Internal/GeometryProcessing/ConvexHull/hkgpConvexHull.h>
#include <Geometry/Collide/Types/hkcdObb.h>

//
hkcdConvexHullFaceOptimizer::hkcdConvexHullFaceOptimizer()
{
    m_hull = HK_NULL;
    clear();
}

//
hkcdConvexHullFaceOptimizer::~hkcdConvexHullFaceOptimizer()
{
    delete m_hull;
}

//
void hkcdConvexHullFaceOptimizer::clear()
{
    m_baseFaces = 0;
    delete m_hull;
    m_hull = HK_NULL;
    m_planes.clear();
}

//
void hkcdConvexHullFaceOptimizer::initialize( const hkgpConvexHull& hull )
{
    clear();

    if ( hull.getDimensions() == 3 )
    {
        hull.fetchPlanes( m_planes );
        m_baseFaces = m_planes.getSize();

        hkArray<hkVector4>  vertices; hull.fetchPositions( hkgpConvexHull::SOURCE_VERTICES, vertices );
        hkcdObb             obb; obb.set( vertices.begin(), vertices.getSize() );
        for ( int axis = 0; axis < 3; ++axis )
        {
            hkVector4       projectionPlane; projectionPlane.setXYZ_0( obb.getRotation().getColumn( axis ) );
            hkgpConvexHull  hull2d; hull2d.buildPlanar( vertices, projectionPlane );
            if ( hull2d.getDimensions() == 2 )
            {
                const int firstPlane = m_planes.getSize();
                hull2d.fetchPlanes( m_planes );
                for ( int i = firstPlane; i < m_planes.getSize(); ++i )
                {
                    hkVector4&  plane = m_planes[ i ];
                    hkVector4   sv; hull.getSupportingVertex( plane, sv );
                    plane.setW( -plane.dot<3>( sv ) );
                }
            }
        }

        hkVector4 v[ 8 ]; obb.getWorldVertices( v );
        m_hull = new hkgpConvexHull();
        if ( m_hull->build( v, 8 ) == 3 )
        {
            m_hull->buildMassProperties();
            m_hull->buildIndices();
        }
        else
        {
            clear();
        }
    }
}

//
bool hkcdConvexHullFaceOptimizer::refine()
{
    if ( m_hull && m_planes.getSize() )
    {
        int         bestPlane = -1;
        hkSimdReal  bestScore = -hkSimdReal_Max;
        for ( int i = 0; i < m_planes.getSize(); ++i )
        {
            hkgpConvexHull *cut = HK_NULL, *leftOver = HK_NULL;
            m_hull->splitByPlane( m_planes[ i ], &cut, &leftOver );
            if ( leftOver && cut && leftOver->getDimensions() == 3 && cut->getDimensions() == 3 )
            {
                leftOver->buildMassProperties();

                hkSimdReal score = leftOver->getSphericity() * cut->getWidth( m_planes[ i ] );

                if ( score > bestScore )
                {
                    bestPlane = i;
                    bestScore = score;
                }
            }
            else
            {
                m_planes.removeAt( i-- );
            }
            delete leftOver;
            delete cut;
        }

        if ( bestPlane > -1 )
        {
            hkgpConvexHull* result = HK_NULL;
            m_hull->splitByPlane( m_planes[ bestPlane ], HK_NULL, &result );
            delete m_hull;
            m_hull = result;
            m_hull->buildMassProperties();
            m_hull->buildIndices();
            m_planes.removeAt( bestPlane );
            return true;
        }
    }
    return false;
}

//
void    hkcdConvexHullFaceOptimizer::optimize( int numFaces, const hkReal maxCgoError )
{
    if ( m_hull && m_planes.getSize() )
    {
        hkgpCgo::Config cgoConfig;
        cgoConfig.m_maxDistance = maxCgoError;

        hkgpConvexHull* bestHull = m_hull->clone();
        const int       maxLookahead = 8;
        int             lookahead = maxLookahead;
        while ( m_planes.getSize() )
        {
            if ( refine() )
            {
                hkgpConvexHull* hull = m_hull->clone();
                hkgpCgo::optimize( cgoConfig, *hull );
                hull->buildIndices();
                if ( hull->getNumPlanes() > numFaces )
                {
                    delete hull;
                    if ( 0 == ( lookahead-- ) ) break;
                }
                else
                {
                    delete bestHull;
                    bestHull = hull;
                    lookahead = maxLookahead;
                }
            }
        }
        delete m_hull;
        m_hull = bestHull;
        m_hull->buildMassProperties();
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
