// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

/*static*/ hctUtilities::ColorCache hctUtilities::s_colorCache;

/*static*/ void hctUtilities::decomposeMatrix( const MMatrix& matrixIn, Decomposition& decompOut )
{
    // Read translation row
    decompOut.m_translation = matrixIn[3];

    // Build rotation only matrix
    MMatrix rotationMatrix;
    {
        rotationMatrix = matrixIn;
        rotationMatrix(0,3) = 0.0;
        rotationMatrix(1,3) = 0.0;
        rotationMatrix(2,3) = 0.0;
        rotationMatrix(3,0) = 0.0;
        rotationMatrix(3,1) = 0.0;
        rotationMatrix(3,2) = 0.0;
        rotationMatrix(3,3) = 1.0;
    }

    // Iterate decomposition
    MMatrix qCur, qNext;
    {
        const int maxNumIterations = 30; // Usually around 7
        const float tolerance = FLT_EPSILON;

        qNext = rotationMatrix;
        int numIterations = 0;
        do
        {
            qCur = qNext;
            qNext = 0.5f * ( qCur + ( qCur.inverse().transpose() ) );
            numIterations++;
        }
        while( (numIterations<maxNumIterations) && (!qNext.isEquivalent(qCur, tolerance )) );
    }

    MVector c1( qNext[0] ); c1.normalize();
    MVector c2( qNext[1] ); c2.normalize();
    MVector c3( qNext[2] ); c3.normalize();
    MVector r0 = c2 ^ c3;
    const double determinant = c1 * r0;

    if( determinant < 0.0 )
    {
        // The rotation is a change of hand-ness
        // Switch one of the columns
        // The scale component should counter-act this
        c1 *= -1.0;
    }

    // Rotation
    {
        MMatrix rot;
        memcpy( rot[0], &c1[0], 3*sizeof(double) );
        memcpy( rot[1], &c2[0], 3*sizeof(double) );
        memcpy( rot[2], &c3[0], 3*sizeof(double) );
        decompOut.m_rotation = rot;
        decompOut.m_rotation.normalizeIt();
    }

    // Scale and Skew
    {
        MTransformationMatrix xform;
        xform.rotateBy( decompOut.m_rotation.inverse(), MSpace::kTransform );
        decompOut.m_scaleAndSkew = xform.asMatrix() * rotationMatrix;
    }

    // Scale
    {
        decompOut.m_scale = MVector(decompOut.m_scaleAndSkew(0,0), decompOut.m_scaleAndSkew(1,1), decompOut.m_scaleAndSkew(2,2));
    }

    // Skew
    {
        MMatrix scaleInv = MMatrix::identity;
        scaleInv(0,0) = 1.0f/decompOut.m_scale[0];
        scaleInv(1,1) = 1.0f/decompOut.m_scale[1];
        scaleInv(2,2) = 1.0f/decompOut.m_scale[2];
        decompOut.m_skew = scaleInv * decompOut.m_scaleAndSkew;
    }
}

/*static*/ void hctUtilities::removeScaleAndSkew( const MMatrix& matrixIn, MMatrix& matrixOut )
{
    // Decompose the input matrix
    Decomposition decomp;
    decomposeMatrix( matrixIn, decomp );

    // Build a new transform using translation and rotation only
    MTransformationMatrix xform;
    xform.setTranslation( decomp.m_translation, MSpace::kTransform );
    xform.rotateTo( decomp.m_rotation );

    // Set output matrix
    matrixOut = xform.asMatrix();
}

/*static*/ void hctUtilities::removeGlScaleAndSkew()
{
    // Read current GL modelview matrix as Maya matrix
    MMatrix modelviewMatrix;
    {
        GLdouble mv[16];
        glGetDoublev( GL_MODELVIEW_MATRIX, mv );
        for( int i=0; i<16; ++i )
        {
            modelviewMatrix(i/4,i%4) = mv[i];
        }
    }

    // Recompose as a scale-free matrix
    removeScaleAndSkew( modelviewMatrix, modelviewMatrix );
    glLoadMatrixd( &modelviewMatrix.matrix[0][0] );
}

/*static*/ MColor hctUtilities::getColor( MString name )
{
    // Access the color preferences through MEL if not in cache
    std::string colorName(name.asChar());
    ColorCache::const_iterator iter = s_colorCache.find(colorName);
    if (iter != s_colorCache.end())
    {
        return MColor(iter->second);
    }

    MCommandResult result;
    MStatus status;

    MString cmd = "exists hkProcPhysics_getColor";
    status = MGlobal::executeCommand( cmd, result, false, false );
    if( status != MStatus::kSuccess )
    {
        MGlobal::displayError("Execution of \"" + cmd + "\" failed.");
        return MColor();
    }

    // The procedure may not exist yet, e.g. if the Havok plugin is unloaded and then a file containing
    // Havok nodes is loaded. Do not bother generating an obtrusive error in this case.
    int cmdExists;
    result.getResult(cmdExists);
    if (cmdExists==0) return MColor();

    cmd = "hkProcPhysics_getColor(\"" + name + "\")";
    status = MGlobal::executeCommand( cmd, result, false, false );
    if( status != MStatus::kSuccess )
    {
        MGlobal::displayError("Execution of \"" + cmd + "\" failed.");
        return MColor();
    }

    MVector vec;
    result.getResult( vec );
    MColor color( (float)vec.x, (float)vec.y, (float)vec.z, 1.0f );

    // add to cache
    s_colorCache[colorName] = color;

    return MColor(color);
}


/*static*/ MStatus hctUtilities::getOptionalPivotOffset( const MDagPath& path, MPoint& out )
{
    out = MPoint::origin;

    // Go through MEL to find the state of the export option
    int usePivot;
    MStatus status = MGlobal::executeCommand( "hkProcPhysics_includePivotInTransforms", usePivot, false, false );
    if( status == MStatus::kSuccess && usePivot )
    {
        // Construct the pivot point
        MFnTransform transformFn( path, &status );
        if( status == MStatus::kSuccess )
        {
            out = transformFn.rotatePivot( MSpace::kPreTransform );
        }
    }

    return status;
}

/*static*/ MStatus hctUtilities::getOptionalPivotWorldTransform( MDagPath path, MMatrix& out )
{
    // Set as the pivot offset matrix
    MPoint pivot;
    MStatus status = getOptionalPivotOffset( path, pivot );
    out.setToIdentity();
    out(3,0) = pivot.x;
    out(3,1) = pivot.y;
    out(3,2) = pivot.z;

    // Post-multiply the node->world transform
    out *= path.inclusiveMatrix();
    return status;
}

/*static*/ MMatrix hctUtilities::getViewportScaleMatrix( const M3dView& view, const MMatrix& objToWorld,
                                            const float& worldUnits, const float& pixelUnits )
{
    // Get viewport coordinates of object
    short x, y;
    MVector pos( objToWorld[3] );
    view.worldToView( pos, x, y );

    // Get local space ray of diagonally shifted viewport coordinates
    x = x + short( pixelUnits / sqrtf(2.0f) );
    y = y + short( pixelUnits / sqrtf(2.0f) );
    MPoint raySrcP;
    MVector rayVec;
    view.viewToObjectSpace( x, y, objToWorld.inverse(), raySrcP, rayVec );

    // Get closest distance to ray
    MVector raySrc = raySrcP;
    MVector closest = raySrc - ( (raySrc * rayVec) / (rayVec * rayVec) )*rayVec;

    // Build the scale matrix
    double scale = closest.length() / worldUnits;
    MMatrix matrix = MMatrix::identity;
    matrix(0,0) = scale;
    matrix(1,1) = scale;
    matrix(2,2) = scale;
    return matrix;
}

/*static*/ void hctUtilities::drawCube( float size )
{
    const GLfloat hs = size*0.5f;
    GLfloat vertices[8][3] =
    {
        {-hs, hs, hs}, { hs, hs, hs}, { hs, hs,-hs}, {-hs, hs,-hs},
        {-hs,-hs, hs}, { hs,-hs, hs}, { hs,-hs,-hs}, {-hs,-hs,-hs}
    };
    GLshort indices[24] =
    {
        4, 5, 6, 7, 1, 2, 6, 5, 0, 1, 5, 4,
        0, 3, 2, 1, 0, 4, 7, 3, 2, 3, 7, 6
    };

    glBegin(GL_QUADS);
    for( int i=0; i<24; i+=2 )
    {
        GLshort i1 = indices[i];
        GLshort i2 = indices[i+1];
        glVertex3f( vertices[i1][0], vertices[i1][1], vertices[i1][2] );
        glVertex3f( vertices[i2][0], vertices[i2][1], vertices[i2][2] );
    }
    glEnd();
}


/*static*/ void hctUtilities::drawCone( float radius, float height, int sides )
{
    const int numPoints = sides + 1;

    const double theta = (radius == 0 )? 0.0: atan( height / radius );
    const double sinTheta = sin(theta);
    const double cosTheta = cos(theta);

    // Generate base points and normals
    MPointArray points( numPoints );
    MPointArray normals( numPoints );
    for( int i=0; i<numPoints; ++i )
    {
        const double phi = i*2.0*PI/(numPoints-1);
        const double sinPhi = sin(phi);
        const double cosPhi = cos(phi);

        points[i].x = radius * cosPhi;
        points[i].y = radius * sinPhi;
        points[i].z = 0.0;

        normals[i].x = sinTheta * cosPhi;
        normals[i].y = sinTheta * sinPhi;
        normals[i].z = cosTheta;
    }

    // Surface
    glBegin( GL_TRIANGLES );
    for( int i=1; i<numPoints; ++i )
    {
        glNormal3dv( &normals[i-1].x );
        glVertex3dv( &points[i-1].x );

        glNormal3dv( &normals[i].x );
        glVertex3dv( &points[i].x );

        MVector avgNormal = normals[i-1] + normals[i];
        avgNormal.normalize();

        glNormal3dv( &avgNormal.x );
        glVertex3d( 0.0, 0.0, height );
    }
    glEnd();

    // Bottom cap
    if( radius != 0.0 )
    {
        glBegin( GL_POLYGON );
        glNormal3f( 0.0f, 0.0f, -1.0f );
        for( int i=numPoints-1; i>=0; --i )
        {
            glVertex3dv( &points[i].x );
        }
        glEnd();
    }
}


/*static*/ void hctUtilities::drawCylinder( float radius, float height, int sides )
{
    const int numPoints = sides + 1;

    // Generate base points and normals
    MPointArray points( numPoints );
    MPointArray normals( numPoints );
    for( int i=0; i<numPoints; ++i )
    {
        const double phi = i*2.0*PI/(numPoints-1);
        const double cosPhi = cos(phi);
        const double sinPhi = sin(phi);

        points[i].x = radius * cosPhi;
        points[i].y = radius * sinPhi;
        points[i].z = 0.0;

        normals[i].x = cosPhi;
        normals[i].y = sinPhi;
        normals[i].z = 0.0;
    }

    // Surface
    glBegin( GL_QUAD_STRIP );
    for( int i=1; i<numPoints; ++i )
    {
        glNormal3dv( &normals[i-1].x );
        glVertex3d( points[i-1].x, points[i-1].y, height );
        glVertex3d( points[i-1].x, points[i-1].y, 0.0 );

        glNormal3dv( &normals[i].x );
        glVertex3d( points[i].x, points[i].y, height );
        glVertex3d( points[i].x, points[i].y, 0.0 );
    }
    glEnd();

    // Bottom cap
    glBegin( GL_POLYGON );
    glNormal3f( 0.0f, 0.0f, -1.0f );
    for( int i=numPoints-1; i>=0; --i )
    {
        glVertex3d( points[i].x, points[i].y, 0.0 );
    }
    glEnd();

    // Top cap
    glBegin( GL_POLYGON );
    glNormal3f( 0.0f, 0.0f, 1.0f );
    for( int i=0; i<numPoints; ++i )
    {
        glVertex3d( points[i].x, points[i].y, height );
    }
    glEnd();
}



static void _nameToDagPath (const MString& name, MDagPath& dagPathOut)
{
    MSelectionList selList;
    selList.add(name);
    selList.getDagPath(0, dagPathOut);
}

MStatus hctUtilities::getMeshFromNurbs( const MDagPath& nurbsDagPath, MDagPath& generatedMeshPath )
{
    MStatus status;
    MDagPath convertedNodePath;

    {
        // EXP-1147 : NurbsToPoly will collapse the transform into the converted polygon. We don't want that, so we start by
        // instancing the nurbs in an "identity" transform node
        MString identityNodeName;
        {
            // Create identity transform node
            MStringArray resultArray;
            MString cmd = "createNode transform";
            status = MGlobal::executeCommand(cmd, resultArray, false, false);

            identityNodeName = resultArray[0];
        }

        // instance the nurbs into that node
        MDagPath newNurbsPath;
        {
            MStringArray names;
            MString nurbsName = nurbsDagPath.fullPathName();
            MString cmd = "parent -add -s "+nurbsName+" "+identityNodeName+";";

            status = MGlobal::executeCommand(cmd, names, false, false);

            MString reparentedName = names[0];
            _nameToDagPath(reparentedName, newNurbsPath);
        }

        // Convert it to a mesh
        {
            MStringArray resultArray;
            MString cmd = "nurbsToPoly -chr 0.9 -uss 1 "+newNurbsPath.fullPathName()+";" ;
            status = MGlobal::executeCommand( cmd, resultArray, false, false );

            MString convertedNodeName = resultArray[0];
            _nameToDagPath(convertedNodeName, convertedNodePath);

            // We  safely assume the first child of the new transform node is the converted mesh
            generatedMeshPath = convertedNodePath;
            generatedMeshPath.push(convertedNodePath.child(0));
        }

        // Remove the identity node
        {
            MString cmd = "delete "+identityNodeName+";";
            MCommandResult res;
            status = MGlobal::executeCommand(cmd, res, false, false);
        }

        // Reparent the mesh to the original node
        // We do this in order to use the same node transform (pivot, etc) during export
        {
            MDagPath nurbsNodePath = nurbsDagPath; nurbsNodePath.pop();
            MString nurbsNodeName = nurbsNodePath.fullPathName();
            MString convertedMeshName = generatedMeshPath.fullPathName();

            MStringArray names;
            MString cmd = "parent -add -s "+convertedMeshName+" "+nurbsNodeName+";";

            status = MGlobal::executeCommand(cmd, names, false, false);

            // The path has changed - update it
            MString reparentedName = names[0];
            _nameToDagPath(reparentedName, generatedMeshPath);
        }

        // Remove the extra node
        {
            MString convertedNodeName = convertedNodePath.fullPathName();
            MString cmd = "delete "+convertedNodeName+";";
            MCommandResult res;
            status = MGlobal::executeCommand(cmd, res, false, false);
        }
    }

    return status;
}

void HK_CALL hctUtilities::getKeyTimes(const MDagPath& dagPath, hkArray<MTime>& keyTimes, MTime startTime, MTime endTime )
{
    MStatus stat;
    MObject node = dagPath.node();
    MItDependencyGraph dgIter( node, MFn::kAnimCurve, MItDependencyGraph::kUpstream, MItDependencyGraph::kBreadthFirst, MItDependencyGraph::kNodeLevel, &stat );
    keyTimes.setSize(0);
    if (stat == MStatus::kSuccess)
    {
        for ( ; !dgIter.isDone(); dgIter.next())
        {
            MObject anim = dgIter.thisNode( &stat );
            MFnAnimCurve animCurve( anim, &stat );
            hkUint32 numKeys = animCurve.numKeyframes(&stat);
            if (stat == MStatus::kSuccess)
            {
                keyTimes.reserve(numKeys);
                for (hkUint32 l=0; l < numKeys; ++l)
                {
                    MTime keyTime = animCurve.time(l);
                    // In our range
                    if ( (keyTime >= startTime) && (keyTime <= endTime))
                    {
                        // As maya will have a curve per sub component (x,y,z) etc, you will get duplicates
                        if (keyTimes.indexOf(keyTime) < 0)
                            keyTimes.pushBack(keyTime);
                    }
                    else // handle case of [EXP-2436], where no keys in the range but range is affected by keys outside, so have to mark at start and end
                    {
                        if ( (keyTime < startTime) && (keyTimes.indexOf(startTime) < 0) )
                            keyTimes.pushBack(startTime);
                        else if ( (keyTime > endTime) && (keyTimes.indexOf(endTime) < 0) )
                            keyTimes.pushBack(endTime);
                    }
                }
            }
        }
    }

    // sort keys
    if( keyTimes.getSize() > 1)
    {
        hkSort( keyTimes.begin(), keyTimes.getSize() );
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
