// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>


/*static*/ void hctMayaUtilities::attributeToClassMember(const MPlug& attributePlug, const hkReflect::Var& field)
{
    struct AttributeToFieldVisitor : public hkReflect::VarVisitor<AttributeToFieldVisitor, void, const MPlug&>
    {
        void visit(const hkReflect::VoidVar&, const MPlug&) { HK_UNREACHABLE(0x20c7dc8f, "Void field found" ); }
        void visit(const hkReflect::ArrayVar&, const MPlug&) { HK_WARN(0x4baedb88, "Field should not be an array"); }
        void visit(const hkReflect::RecordVar&, const MPlug&) { HK_WARN(0x5985f61e, "Field should not be a record"); }

        void visitUuid(hkUuid* uuid, const MPlug& attributePlug)
        {
            MString val;
            if (attributePlug.getValue(val) == MStatus::kSuccess)
            {
                hkStringPtr str = hkString::strDup(val.asChar());
                uuid->setFromFormattedString(str.cString());
            }
            else
            {
                (*uuid) = hkUuid::getNil();
            }
        }

        void visit(const hkReflect::IntVar& var, const MPlug& attributePlug)
        {
            switch(var.getType()->getSizeOf())
            {
                case 1: { char val; attributePlug.getValue(val); var.setValue(val); break; }
                case 2: { short val; attributePlug.getValue(val); var.setValue(val); break; }
                case 4: { int val; attributePlug.getValue(val); var.setValue(val); break; }
                case 8:
                    {
                        MObject obj;
                        MStatus returnStatus;
                        returnStatus = attributePlug.getValue(obj);
                        if ( returnStatus != MStatus::kSuccess )
                        {
                            break;
                        }

                        MFnNumericData numericData(obj, &returnStatus);
                        if ( (returnStatus != MStatus::kSuccess) || (numericData.numericType() == MFnNumericData::k2Int) )
                        {
                            break;  // The numericData can be different than the expected k2Int, as Maya emulates it by creating 2 attributes.
                        }

                        int hi, lo;
                        returnStatus = numericData.getData(hi, lo);
                        if ( returnStatus == MStatus::kSuccess )
                        {
                            const hkUint64 val = (hkUint64(hi) << hkUint64(32)) | hkUint64(lo);
                            var.setValue(val);
                        }
                    }
                    break;
            }
        }

        void visit(const hkReflect::FloatVar& var, const MPlug& attributePlug)
        {
            hkReal val;
            attributePlug.getValue(val);
            var.setValue(val);
        }

        void visit(const hkReflect::BoolVar& var, const MPlug& attributePlug)
        {
            bool val;
            attributePlug.getValue(val);
            var.setValue(val);
        }

        void visit(const hkReflect::StringVar& var, const MPlug& attributePlug)
        {
            MString val;
            attributePlug.getValue(val);
            if (char** cstring = var.dynCast<char*>())
            {
                *cstring = hkString::strDup(val.asChar());
            }
            else
            {
                var.setValue(val.asChar());
            }
        }
        void visitVector4(hkVector4* vec4, const MPlug& attributePlug)
        {
            // Assign first element (keeps compatibility with old behavior).
            hkReal val;
            attributePlug.getValue(val);
            vec4->setComponent<0>(val);
        }

        void visit(const hkReflect::PointerVar& var, const MPlug& attributePlug)
        {
            // We set the member value to HK_NULL because we cannot fix up the pointer now.
            var.setValue(hkReflect::Var());
        }
    } visitor;

    if (hkUuid* uuid = field.dynCast<hkUuid>())
    {
        visitor.visitUuid(uuid, attributePlug);
    }
    else if (hkVector4* vec4 = field.dynCast<hkVector4>())
    {
        visitor.visitVector4(vec4, attributePlug);
    }
    else
    {
        visitor.dispatch(field, attributePlug);
    }
}

/*static*/ void hctMayaUtilities::classMemberToAttribute(const hkReflect::Var& field, MPlug& attributePlug)
{
    struct FieldToAttributeVisitor : public hkReflect::VarVisitor < FieldToAttributeVisitor, void, MPlug& >
    {
        void visit(const hkReflect::VoidVar&, MPlug&) { HK_UNREACHABLE(0x8e8921f, "Void field found" ); }
        void visit(const hkReflect::ContainerVar&, MPlug&) { HK_WARN(0x76c80d4f, "Field should not be a container"); }
        void visit(const hkReflect::RecordVar&, MPlug&) { HK_WARN(0x5af8576e, "Field should not be a record"); }

        void visit(const hkReflect::IntVar& var, MPlug& attributePlug)
        {
            switch (var.getType()->getSizeOf())
            {
            case 1: { attributePlug.setValue(var.getValue().convertTo<char>()); break; }
            case 2: { attributePlug.setValue(var.getValue().convertTo<short>()); break; }
            case 4: { attributePlug.setValue(var.getValue().convertTo<int>()); break; }
            case 8:
                {
                    MStatus returnStatus;
                    MFnNumericData numericData;
                    MObject obj = numericData.create(MFnNumericData::k2Int, &returnStatus );
                    if ( returnStatus != MStatus::kSuccess )
                    {
                        break;
                    }

                    const hkUint64 val  = var.getValue().convertTo<hkUint64>();
                    const int hi        = int((val >> hkUint64(32)) & 0xFFFFFFFF);
                    const int lo        = int(val & 0xFFFFFFFF);

                    returnStatus = numericData.setData(hi, lo);
                    if ( returnStatus == MStatus::kSuccess )
                    {
                        returnStatus = attributePlug.setValue(obj);
                        CHECK_MSTATUS(returnStatus);
                    }
                }
                break;
            }
        }

        void visit(const hkReflect::FloatVar& var, MPlug& attributePlug)
        {
            hkReal val;
            val = static_cast<hkReal>(var.getValue());
            attributePlug.setValue(val);
        }

        void visit(const hkReflect::BoolVar& var, MPlug& attributePlug)
        {
            bool val;
            val = var.getValue();
            attributePlug.setValue(val);
        }

        void visit(const hkReflect::StringVar& var, MPlug& attributePlug)
        {
            MString val;
            val = var.getValue();
            attributePlug.setValue(val);
        }

        void visitVector4(const hkVector4* vec4, MPlug& attributePlug)
        {
            // Assign first element (keeps compatibility with old behavior).
            hkReal val = vec4->getComponent<0>();
            attributePlug.setValue(val);
        }

        void visit(const hkReflect::PointerVar& var, MPlug& attributePlug)
        {
            // NOT support currently
            HK_ASSERT_NO_MSG(0x7d7b5fbc, false);
        }
    } visitor;

    if (const hkVector4* vec4 = field.dynCast<hkVector4>())
    {
        visitor.visitVector4(vec4, attributePlug);
    }
    else
    {
        visitor.dispatch(field, attributePlug);
    }
}


// Returns the name of the object if the correct object and class are found, NULL otherwise
// Also updates obj to the correct value.
const char* getActualObjectAndClass(const char* attributeName, hkReflect::Var& obj)
{
    const char* remainingName = attributeName;
    const char* colonPos;

    while ( (colonPos = hkString::strChr(remainingName, ':')) != 0 )
    {
        int substringLen = int(colonPos - remainingName);
        hkStringOld structName(remainingName, substringLen);
        hkReflect::Var strukt = obj[structName.cString()];
        if ( !strukt )
        {
            return HK_NULL;
        }

        obj = strukt;
        remainingName = colonPos + 1;
    }

    return remainingName;
}


/*static*/ hkReflect::Var hctMayaUtilities::getField(const char* attributeName, const hkReflect::Var& obj)
{
    hkReflect::Var actualObject = obj;
    const char* actualName = getActualObjectAndClass(attributeName, actualObject);

    if (actualName == HK_NULL)
    {
        return hkReflect::Var();
    }

    // Try to retrieve the member description through its name. If this fails, we'll do some hack and
    // for the time being assume that we are actually dealing with some Maya auto-created name and
    // that Maya simply appended a number to our original name (e.g. when dealing with hkVector4).
    // We'll then retrieve this postfix, shorten the name back accordingly and retry in retrieving
    // the member description.
    hkReflect::Var field = actualObject[actualName];
    if ( !field.isValid() )
    {
        int attributeNameLength = hkString::strLen( attributeName );
        int postFixNumber = attributeName[attributeNameLength-1] - '0';

        hkStringOld modifiedRemainingName(actualName);
        (const_cast<char*>(modifiedRemainingName.cString()))[modifiedRemainingName.getLength()-1] = 0;
        field = actualObject[modifiedRemainingName.cString()];
        if (hkReflect::ArrayVar asArray = field)
        {
            if (asArray.getType()->getFixedCount())
            {
                field = asArray[postFixNumber];
            }
        }
    }

    return field;
}

/*static*/ void hctMayaUtilities::mayaObjectToClassObject(const MFnDependencyNode& nodeFn, const hkReflect::Var& object)
{
    int attributeCount = nodeFn.attributeCount();
    int firstAttributeIndex = 0;

    MStatus status;

    for (int i = firstAttributeIndex; i < attributeCount; ++i)
    {
        MObject attrObject = nodeFn.attribute(i);
        MFnAttribute attribute( attrObject, &status );
        const char* attributeName = attribute.name().asChar();

        {
            hkReflect::Var field = getField(attributeName, object);

            if ( !field.isValid() )
            {
                HK_WARN_ALWAYS( 0xf0125446, "Cannot find member: " << attributeName );
                continue;
            }


            MPlug attributePlug( nodeFn.object(), attribute.object() );

            hctMayaUtilities::attributeToClassMember(attributePlug, field);
        }
    }
}

#pragma warning(disable:4702)

#ifdef HK_PLATFORM_WIN64
    extern const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT( "Software\\Havok\\hkFilters_x64" );
#else
    extern const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT( "Software\\Havok\\hkFilters" );
#endif

/*static*/ bool HK_CALL hctMayaUtilities::getFilterManagerPath( const char* currentPluginPath, MString& filterManagerPath )
{

    //CK: Removed hkStringOld so that we can call this function without Havok mem
    bool found = false;

    // See if the environment variable is set up
    {
        char* path = ::getenv(ENVVAR_FILTER_ROOT);

        if (HK_NULL != path)
        {
            filterManagerPath = path;
            FILE* fcheck = ::fopen((filterManagerPath + MString("\\hctFilterManager.dll")).asChar(), "r");
            found = fcheck != HK_NULL;
            if (found)
            {
                fclose(fcheck);
            }
        }
    }

    // Look for a path in the registry
    if( !found )
    {
        HKEY animReg;
        DWORD dispos;
        RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

        BYTE filtersPath[1024];
        DWORD filtersPathSize = 1024;

        // Check the default FilterPath value
        if( RegQueryValueEx( animReg, TEXT("FilterPath"), NULL, NULL, filtersPath, &filtersPathSize ) == ERROR_SUCCESS )
        {
            filterManagerPath = (const char*)filtersPath;

            FILE* fcheck = ::fopen( (filterManagerPath + MString("\\hctFilterManager.dll")).asChar(), "r" );
            found = fcheck != HK_NULL;
            if (found)
            {
                fclose(fcheck);
            }
        }

        RegCloseKey(animReg);
    }

    return found;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
