// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctManipContainer.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

bool hctManipContainer::m_snapRotation = false;
MAngle hctManipContainer::m_rotationSnapAngle( 5.0, MAngle::kDegrees );


MStatus hctManipContainer::createChildren()
{
    // Create manipulator storage
    m_numManipulators = 0;
    while( getManipulatorDesc( m_numManipulators ) != NULL ) m_numManipulators++;
    m_manipulators = new ManipItem[ m_numManipulators ];

    // Create manip nodes - no need to name them
    for( int i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];

        const ManipDesc& manipDesc = *getManipulatorDesc(i);
        MFnAttribute attrFn( manipDesc.m_attribute );
        MString attrName = attrFn.name();

        switch( manipDesc.m_manipType )
        {
        case ManipDesc::MANIP_TYPE_POINT:
            manip.m_path = addFreePointTriadManip("","");
            manip.m_numConnections = 1;
            break;
        case ManipDesc::MANIP_TYPE_SCALE:
            manip.m_path = addScaleManip("","");
            manip.m_numConnections = 1;
            break;
        case ManipDesc::MANIP_TYPE_ROTATE:
            manip.m_path = addRotateManip("","");
            manip.m_numConnections = 2;
            break;
        case ManipDesc::MANIP_TYPE_DIRECTION:
            manip.m_path = addDirectionManip("","");
            manip.m_numConnections = 1;
            break;
        case ManipDesc::MANIP_TYPE_DISTANCE:
            manip.m_path = addDistanceManip("","");
            manip.m_numConnections = 1;
            break;
        default:
            break;
        }
        manip.m_connections = new ManipItem::Connection[manip.m_numConnections];
    }

    return MStatus::kSuccess;
}

MStatus hctManipContainer::connectToDependNode( const MObject& node )
{
    // Store target node
    m_target = node;

    MStatus status;
    status = MStatus::kSuccess;

    // For each manipulator
    for( int i=0; i<m_numManipulators; ++i )
    {
        connectManipToNode(i, node, status);
    }

    // Finish adding manips
    finishAddingManips();
    MPxManipContainer::connectToDependNode( node );

    return status;
}

void hctManipContainer::connectManipToNode( int i, const MObject& node, MStatus& status )
{
    const ManipDesc& manipDesc = *getManipulatorDesc(i);
    ManipItem& manip = m_manipulators[i];
    MFnDependencyNode nodeFn( node );


    // Get a plug to the attribute
    MFnAttribute attrFn( manipDesc.m_attribute );
    MPlug plug = nodeFn.findPlug( attrFn.name() );
    if( plug.isNull() )
    {
        return;
    }

    // Connect the manipulator to the attribute
    switch( manipDesc.m_manipType )
    {
    case ManipDesc::MANIP_TYPE_POINT:
        {
            MFnFreePointTriadManip pointFn( manip.m_path );
            ManipItem::Connection& conn = manip.m_connections[0];
            conn.m_manipIndex = pointFn.pointIndex();
            conn.m_plugIndex = addManipToPlugConversionCallback( plug, (manipToPlugConversionCallback)&hctManipContainer::manipToPlugTranslationCallback );
            addPlugToManipConversionCallback( conn.m_manipIndex, (plugToManipConversionCallback)&hctManipContainer::plugToManipTranslationCallback );
#if MAYA_API_VERSION >= 800
            MTransformationMatrix xform = getManipulatorBaseTransform(i);
            pointFn.setDirection( xform.eulerRotation().asVector() );
#endif
            break;
        }
    case ManipDesc::MANIP_TYPE_SCALE:
        {
            MFnScaleManip scaleFn( manip.m_path );
            ManipItem::Connection& conn = manip.m_connections[0];
            scaleFn.connectToScalePlug( plug );
            conn.m_manipIndex = scaleFn.scaleCenterIndex();
            addPlugToManipConversionCallback( conn.m_manipIndex, (plugToManipConversionCallback)&hctManipContainer::plugToManipCenterCallback );
            break;
        }
    case ManipDesc::MANIP_TYPE_ROTATE:
        {
            MFnRotateManip rotateFn( manip.m_path );
            ManipItem::Connection& convRot = manip.m_connections[0];
            ManipItem::Connection& convPos = manip.m_connections[1];

            convRot.m_manipIndex = rotateFn.rotationIndex();
            convRot.m_plugIndex = addManipToPlugConversionCallback( plug, (manipToPlugConversionCallback)&hctManipContainer::manipToPlugRotationCallback );
            addPlugToManipConversionCallback( convRot.m_manipIndex, (plugToManipConversionCallback)&hctManipContainer::plugToManipRotationCallback );
            rotateFn.setRotateMode( MFnRotateManip::kObjectSpace );
            rotateFn.setSnapMode( m_snapRotation );
            rotateFn.setSnapIncrement( m_rotationSnapAngle.asDegrees() );

            convPos.m_manipIndex = rotateFn.rotationCenterIndex();
            addPlugToManipConversionCallback( convPos.m_manipIndex, (plugToManipConversionCallback)&hctManipContainer::plugToManipCenterCallback );
            break;
        }
    case ManipDesc::MANIP_TYPE_DIRECTION:
        {
            MFnDirectionManip dirFn( manip.m_path );
            ManipItem::Connection& conn = manip.m_connections[0];
            dirFn.connectToDirectionPlug( plug );
            dirFn.setNormalizeDirection(true);
            conn.m_manipIndex = dirFn.directionIndex();
//          conn.m_plugIndex = addManipToPlugConversionCallback( plug, (manipToPlugConversionCallback)&hctManipContainer::manipToPlugDirectionCallback );
//          addPlugToManipConversionCallback( conn.m_manipIndex, (plugToManipConversionCallback)&hctManipContainer::plugToManipDirectionCallback );
            break;
        }
    case ManipDesc::MANIP_TYPE_DISTANCE:
        {
            MFnDistanceManip distFn( manip.m_path );
            ManipItem::Connection& conn = manip.m_connections[0];
            distFn.connectToDistancePlug( plug );
            conn.m_manipIndex = distFn.distanceIndex();
            conn.m_plugIndex = addManipToPlugConversionCallback( plug, (manipToPlugConversionCallback)&hctManipContainer::manipToPlugDistanceCallback );
            addPlugToManipConversionCallback( conn.m_manipIndex, (plugToManipConversionCallback)&hctManipContainer::plugToManipDistanceCallback );
            break;
        }
    default:
        break;
    }
}

void hctManipContainer::draw( M3dView& view, const MDagPath& path,
            M3dView::DisplayStyle displayStyle, M3dView::DisplayStatus displayStatus )
{
    // Set visibilities
    for( int i=0; i<m_numManipulators; ++i )
    {
        MFnManip3D manipFn( m_manipulators[i].m_path );

        // Validate the attribute
        MFnDependencyNode nodeFn( m_target );
        MFnAttribute attrFn( getManipulatorDesc(i)->m_attribute );
        MPlug plug = nodeFn.findPlug( attrFn.name() );
        if( plug.isNull() )
        {
            manipFn.setVisible( false );
        }
        else
        {
            manipFn.setVisible( getManipulatorVisibility(i) );
        }
    }

    // Let Maya do the drawing
    MPxManipContainer::draw( view, path, displayStyle, displayStatus );
}


MManipData hctManipContainer::manipToPlugTranslationCallback( unsigned int index )
{
    // Find the manipulator with this plug index
    int i,j;
    for( i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];
        for( j=0; j<manip.m_numConnections; ++j )
        {
            ManipItem::Connection& conn = manip.m_connections[j];
            if( conn.m_plugIndex == int(index) )
            {
                MFnManip3D manipFn( manip.m_path );
                MStatus status;
                MVector translation;

                // getConverterManipValue() gives an incorrect value the first time
                // this method is called, and whenever the manipulator is hidden :(
                if( conn.m_initialized && manipFn.isVisible(&status) )
                {
                    // Convert manip value to base space
                    getConverterManipValue( conn.m_manipIndex, translation );
                    MTransformationMatrix xform = getManipulatorBaseTransform(i).inverse();
                    xform.addTranslation( translation, MSpace::kPreTransform );
                    translation = xform.translation( MSpace::kPostTransform );
                }
                else
                {
                    // Set current plug value, ie. do nothing
                    getConverterPlugValue( conn.m_plugIndex, translation );
                    conn.m_initialized = true;
                }

                // Return the new value
                MFnNumericData numFn;
                numFn.create( MFnNumericData::k3Double );
                numFn.setData( translation.x, translation.y, translation.z );
                return MManipData( numFn.object() );
            }
        }
    }
    return MManipData();
}

MManipData hctManipContainer::plugToManipTranslationCallback( unsigned int index )
{
    // Find the manipulator with this manip index
    int i,j;
    for( i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];
        for( j=0; j<manip.m_numConnections; ++j )
        {
            ManipItem::Connection& conn = manip.m_connections[j];
            if( conn.m_manipIndex == int(index) )
            {
                // Convert plug value to world space
                MVector translation;
                getConverterPlugValue( conn.m_plugIndex, translation );
                MTransformationMatrix xform = getManipulatorBaseTransform(i);
                xform.addTranslation( translation, MSpace::kPreTransform );
                translation = xform.translation( MSpace::kPostTransform );

                // Return the value
                MFnNumericData numFn;
                numFn.create( MFnNumericData::k3Double );
                numFn.setData( translation.x, translation.y, translation.z );
                return MManipData( numFn.object() );
            }
        }
    }
    return MManipData();
}


MManipData hctManipContainer::manipToPlugRotationCallback( unsigned int index )
{
    // Find the manipulator with this plug index
    int i,j;
    for( i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];
        for( j=0; j<manip.m_numConnections; ++j )
        {
            ManipItem::Connection& conn = manip.m_connections[j];
            if( conn.m_plugIndex == int(index) )
            {
                MFnManip3D manipFn( manip.m_path );
                MStatus status;
                MEulerRotation rotation;

                // getConverterManipValue() gives an incorrect value the first time
                // this method is called, and whenever the manipulator is hidden :(
                if( conn.m_initialized && manipFn.isVisible(&status) )
                {
                    // Convert manip value to base space
                    getConverterManipValue( conn.m_manipIndex, rotation );
                    hctUtilities::Decomposition decomp;
                    hctUtilities::decomposeMatrix( getManipulatorBaseTransform(i).inverse(), decomp );
                    rotation *= decomp.m_rotation.asEulerRotation();
                }
                else
                {
                    // Set current plug value, ie. do nothing
                    getConverterPlugValue( conn.m_plugIndex, rotation );
                    conn.m_initialized = true;
                }

                // Return the (bounded) value
                rotation.boundIt();
                MFnNumericData numFn;
                numFn.create( MFnNumericData::k3Double );
                numFn.setData( rotation.x, rotation.y, rotation.z );
                return MManipData( numFn.object() );
            }
        }
    }
    return MManipData();
}

MManipData hctManipContainer::plugToManipRotationCallback( unsigned int index )
{
    // Find the manipulator with this manip index
    int i,j;
    for( i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];
        for( j=0; j<manip.m_numConnections; ++j )
        {
            ManipItem::Connection& conn = manip.m_connections[j];
            if( conn.m_manipIndex == int(index) )
            {
                // Convert plug value to world space
                MEulerRotation rotation;
                getConverterPlugValue( conn.m_plugIndex, rotation );
                hctUtilities::Decomposition decomp;
                hctUtilities::decomposeMatrix( getManipulatorBaseTransform(i), decomp );
                rotation *= decomp.m_rotation.asEulerRotation();

                // Return the (bounded) value
                rotation.boundIt();
                MFnNumericData numFn;
                numFn.create( MFnNumericData::k3Double );
                numFn.setData( rotation.x, rotation.y, rotation.z );
                return MManipData( numFn.object() );
            }
        }
    }
    return MManipData();
}


MManipData hctManipContainer::plugToManipCenterCallback( unsigned int index )
{
    // Find the manipulator with this manip index
    int i,j;
    for( i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];
        for( j=0; j<manip.m_numConnections; ++j )
        {
            ManipItem::Connection& conn = manip.m_connections[j];
            if( conn.m_manipIndex == int(index) )
            {
                // Return the base world space translation
                MVector translation = MVector( getManipulatorBaseTransform(i)[3] );
                MFnNumericData numFn;
                numFn.create( MFnNumericData::k3Double );
                numFn.setData( translation.x, translation.y, translation.z );
                return MManipData( numFn.object() );
            }
        }
    }
    return MManipData();
}


MManipData hctManipContainer::plugToManipDistanceCallback( unsigned int index )
{
    // Find the manipulator with this manip index
    int i,j;
    for( i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];
        for( j=0; j<manip.m_numConnections; ++j )
        {
            ManipItem::Connection& conn = manip.m_connections[j];
            if( conn.m_manipIndex == int(index) )
            {
                // Return the base world space translation
                MVector translation = MVector( getManipulatorBaseTransform(i)[3] );
                double distance = translation.length();
                return MManipData( distance );
            }
        }
    }
    return MManipData();
}


MManipData hctManipContainer::manipToPlugDistanceCallback( unsigned int index )
{
    // Find the manipulator with this plug index
    int i,j;
    for( i=0; i<m_numManipulators; ++i )
    {
        ManipItem& manip = m_manipulators[i];
        for( j=0; j<manip.m_numConnections; ++j )
        {
            ManipItem::Connection& conn = manip.m_connections[j];
            if( conn.m_plugIndex == int(index) )
            {
                MFnManip3D manipFn( manip.m_path );
                MStatus status;
                double distance;

                // getConverterManipValue() gives an incorrect value the first time
                // this method is called, and whenever the manipulator is hidden :(
                if( conn.m_initialized && manipFn.isVisible(&status) )
                {
                    // Convert manip value to base space
                    getConverterManipValue( conn.m_manipIndex, distance );
                    MTransformationMatrix xform = getManipulatorBaseTransform(i).inverse();
                    MVector translation(distance, 0, 0);
                    xform.addTranslation( translation, MSpace::kPreTransform );
                    translation = xform.translation( MSpace::kPostTransform );
                    distance = translation[0];
                }
                else
                {
                    // Set current plug value, ie. do nothing
                    getConverterPlugValue( conn.m_plugIndex, distance );
                    conn.m_initialized = true;
                }

                // Return the new value
                return MManipData( distance );
            }
        }
    }
    return MManipData();
}

// MManipData hctManipContainer::plugToManipDirectionCallback( unsigned int index )
// {
//  // Find the manipulator with this manip index
//  int i,j;
//  for( i=0; i<m_numManipulators; ++i )
//  {
//      ManipItem& manip = m_manipulators[i];
//      for( j=0; j<manip.m_numConnections; ++j )
//      {
//          ManipItem::Connection& conn = manip.m_connections[j];
//          if( conn.m_manipIndex == int(index) )
//          {
//              // Return the base world space translation
//              MVector translation = MVector( getManipulatorBaseTransform(i)[3] );
//              MFnNumericData numFn;
//              numFn.create( MFnNumericData::k3Double );
//              numFn.setData( translation.x, translation.y, translation.z );
//              return MManipData( numFn.object() );
//          }
//      }
//  }
//  return MManipData();
// }
//
//
// MManipData hctManipContainer::manipToPlugDirectionCallback( unsigned int index )
// {
//  // Find the manipulator with this plug index
//  int i,j;
//  for( i=0; i<m_numManipulators; ++i )
//  {
//      ManipItem& manip = m_manipulators[i];
//      for( j=0; j<manip.m_numConnections; ++j )
//      {
//          ManipItem::Connection& conn = manip.m_connections[j];
//          if( conn.m_plugIndex == int(index) )
//          {
//              MFnManip3D manipFn( manip.m_path );
//              MStatus status;
//              MVector translation;
//
//              // getConverterManipValue() gives an incorrect value the first time
//              // this method is called, and whenever the manipulator is hidden :(
//              if( conn.m_initialized && manipFn.isVisible(&status) )
//              {
//                  // Convert manip value to base space
//                  getConverterManipValue( conn.m_manipIndex, translation );
//                  MTransformationMatrix xform = getManipulatorBaseTransform(i).inverse();
//                  xform.addTranslation( translation, MSpace::kPreTransform );
//                  translation = xform.translation( MSpace::kPostTransform );
//              }
//              else
//              {
//                  // Set current plug value, ie. do nothing
//                  getConverterPlugValue( conn.m_plugIndex, translation );
//                  conn.m_initialized = true;
//              }
//
//              // Return the new value
//              MFnNumericData numFn;
//              numFn.create( MFnNumericData::k3Double );
//              numFn.setData( translation.x, translation.y, translation.z );
//              return MManipData( numFn.object() );
//          }
//      }
//  }
//  return MManipData();
// }

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
