// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0


#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>

// this
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctDestructionUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/ScriptedFracture/hctScriptedFractureNode.h>

// VC 8.0  - STL vector needs to know that it is to link with Release CRT, not use funcs only in the Debug CRT
#if defined(_DEBUG) && defined(_MSC_VER) && (_MSC_VER >= 1400)
#   undef _DEBUG
#   define HK_RESET_DEBUG_FLAG
#endif

// List of registered nodes

// Initialize own static member
std::vector<MTypeId> hctMayaPhysicsDestructionUtilities::m_destructionNodes;
MObject              hctMayaPhysicsDestructionUtilities::m_pluginObject;
hctGenericNodeHelper hctMayaPhysicsDestructionUtilities::m_nodeHelper[ HCT_MAX_NUM_CLASS_ENTRIES ];
MIntArray            hctMayaPhysicsDestructionUtilities::m_callbackIds;

// The generating template for registering all Destruction nodes
template <int N>
inline void registerNodeUnroll(hkAttributeHideCriteria::Types hideCriteria, MFnPlugin& plugin, const hctModelerNodeIdToClassMap* modelerNodeIdToClassMap, std::vector<MTypeId>& typeRegistry, MStatus& status)
{
    const hkReflect::Type* nodeClass = modelerNodeIdToClassMap->m_class;

    if ( nodeClass )
    {
        // Get class UI settings (if any)
        const hkUiAttribute* uiAttrib = nodeClass->findAttribute<hkUiAttribute>();

        // Do nothing if the class is hidden under the current UI scheme !
        const bool bHidden = uiAttrib && (uiAttrib->m_hideCriteria & hideCriteria);
        if ( !bHidden )
        {
            MPxNode::Type nodeType = MPxNode::kDependNode;
            hctMayaPhysicsDestructionUtilities::m_nodeHelper[N].setId(N);
            MCreatorFunction creatorFunc = &hctGenericNode<N>::creator;
            MInitializeFunction initFunc = &hctGenericNode<N>::initialize;

            bool isLocator = false;
            {
                for ( const hkReflect::Type* c = modelerNodeIdToClassMap->m_class; c; c = c->getParent() )
                {
                    const hkModelerNodeTypeAttribute* attrib = c->findAttribute<hkModelerNodeTypeAttribute>();
                    if ( attrib && attrib->m_type == hkModelerNodeTypeAttribute::LOCATOR)
                    {
                        isLocator = true;
                        break;
                    }
                }
            }
            if (isLocator)
            {
                nodeType = MPxNode::kLocatorNode;
                creatorFunc = &hctGenericLocatorNode<N>::creator;
                initFunc = &hctGenericLocatorNode<N>::initialize;
            }
            if ( modelerNodeIdToClassMap->m_mayaNodeId == hkdScriptedFractureID )
            {
                creatorFunc = &hctScriptedFractureNode::creator;
                initFunc = &hctScriptedFractureNode::initialize;
            }

            status = plugin.registerNode(modelerNodeIdToClassMap->m_nodeName, modelerNodeIdToClassMap->m_mayaNodeId, creatorFunc, initFunc, nodeType );

            char buf[256];
            if( status == MStatus::kSuccess )
            {
                typeRegistry.push_back( modelerNodeIdToClassMap->m_mayaNodeId );

                hkString::sprintf(buf,"registered Destruction node '%s' (0x%x)", modelerNodeIdToClassMap->m_displayName, modelerNodeIdToClassMap->m_mayaNodeId);
                MGlobal::displayInfo(buf);
            }
            else
            {
                hkString::sprintf(buf,"registration of Destruction node '%s' (0x%x) failed", modelerNodeIdToClassMap->m_displayName, modelerNodeIdToClassMap->m_mayaNodeId);
                MGlobal::displayError(buf);
                status.perror( buf );
            }
        }
    }

    registerNodeUnroll<N+1>(hideCriteria, plugin, ++modelerNodeIdToClassMap, typeRegistry, status);
}

template <>
inline void registerNodeUnroll<HCT_MAX_NUM_CLASS_ENTRIES>(hkAttributeHideCriteria::Types hideCriteria, MFnPlugin& plugin, const hctModelerNodeIdToClassMap* modelerNodeIdToClassMap, std::vector<MTypeId>& typeRegistry, MStatus& status)
{
}

void hctMayaPhysicsDestructionUtilities::initializeDestructionPlugin(MFnPlugin& plugin, MStatus& status)
{
    hctSdkUtils::loadAllUtils();

    // Query UI scheme and set additional hide criteria
    const hkAttributeHideCriteria::Types hideCriteria = hctSdkUtils::getUiAttributesHideCriteria(hkAttributeHideCriteria::MODELER_IS_MAYA);

    // Sync statics
    const HMODULE dllHandle = hctSdkUtils::m_destructionUtilsDll;
    if ( dllHandle )
    {
        // Get base Dll interface
        typedef class hctBaseDll* (HK_CALL *hctGetBaseDllInterfaceFunc)();
        hctGetBaseDllInterfaceFunc destBaseDllFunc = (hctGetBaseDllInterfaceFunc) GetProcAddress(dllHandle, "getBaseDllInterface");
        if ( !destBaseDllFunc )
        {
            HK_WARN_ALWAYS(0x6fdba42b,"Destruction Util DLL found but could not find baseDLL sync function. Install is invalid.");
            return;
        }

        // Sync statics
        hctBaseDll* dll = destBaseDllFunc();
        hctSdkUtils::m_baseDestructionUtilsDll = dll;
        if ( dll )
        {
            if ( dll->getDllVersion() != HCT_CURRENT_VERSION )
            {
                HK_WARN_ALWAYS(0x7f425fa0, "Could not load Destruction Util DLL interface as it was built off an older version of Havok than your current tools install. Please reinstall." );
                return;
            }

            // Init the shared mem system.
            hkMemoryInitUtil::SyncInfo baseSystemInfo;
            hkSceneExportMemory::getBaseSystemSyncInfo(baseSystemInfo);
            dll->initDll(baseSystemInfo, HK_NULL);
        }
    }

    hctSdkUtils::loadAllClasses();

    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();

    m_destructionNodes.reserve(HCT_MAX_NUM_CLASS_ENTRIES);

    registerNodeUnroll<0>(hkAttributeHideCriteria::Types(hideCriteria), plugin, modelerNodeIdToClassMap.begin(), m_destructionNodes, status);

    // Register an 'after scene loaded" callback where we can perform the node versioning.
    MCallbackId id = MSceneMessage::addCallback(MSceneMessage::kAfterOpen, hctMayaPhysicsDestructionUtilities::afterSceneLoadCallback, NULL, &status);
    m_callbackIds.append((int)id);

    // Register a 'before scene save" callback where we can update the version values on all nodes.
    id = MSceneMessage::addCallback(MSceneMessage::kBeforeSave, hctMayaPhysicsDestructionUtilities::beforeSceneSaveCallback, NULL, &status);
    m_callbackIds.append((int)id);
}

//
//  Attempts to de-register the given command

void hctMayaPhysicsDestructionUtilities::deregisterCommand(MFnPlugin& plugin, const MString& commandName, MStatus& status)
{
    if ( status == MStatus::kSuccess )
    {
        status = plugin.deregisterCommand(commandName);
        if ( status != MStatus::kSuccess )
        {
            char buf[256];
            hkString::sprintf(buf, "deregistration of Destruction command %s failed", commandName.asChar());
            MGlobal::displayError(buf);
            status.perror(buf);
        }
    }
}

void hctMayaPhysicsDestructionUtilities::shutdownDestructionPlugin(MFnPlugin& plugin, MStatus& status)
{
    MMessage::removeCallbacks(m_callbackIds);

    // De-register commands
    {
        MStatus statusDeregister = MStatus::kSuccess;
        deregisterCommand(plugin, "hkdCmdScriptedFractureOpenScript", statusDeregister);
        deregisterCommand(plugin, "hkdCmdScriptedFractureBuildUi", statusDeregister);
    }

    // De-register any nodes that were registered
    char buf[256];
    std::vector<MTypeId>::iterator it = m_destructionNodes.begin();
    while( it != m_destructionNodes.end() )
    {
        status = plugin.deregisterNode( *it );
        if( status != MStatus::kSuccess )
        {
            hkString::sprintf(buf,"deregistration of Destruction node 0x%x failed", it->id());
            MGlobal::displayError(buf);
            status.perror( buf );
        }
        else
        {
            hkString::sprintf(buf,"deregistered Destruction node 0x%x", it->id());
            MGlobal::displayInfo(buf);
        }

        ++it;
    }
    m_destructionNodes.clear();

    

    // Unsync statics
    hctBaseDll* dll = hctSdkUtils::m_baseDestructionUtilsDll;
    if ( dll )
    {
        hctSdkUtils::m_baseDestructionUtilsDll->quitDll();
        hctSdkUtils::m_baseDestructionUtilsDll = HK_NULL;
    }

    hctSdkUtils::unloadAllUtils();
    hctSdkUtils::unloadAllClasses();
}

bool hctMayaPhysicsDestructionUtilities::nodeHasGizmos(MObject node, MStatus& status)
{
    MFnDagNode nodeFn( node, &status );
    if( status != MStatus::kSuccess )
    {
        char buf[256];
        hkString::sprintf(buf,"error accessing node type information");
        MGlobal::displayError(buf);
        status.perror( buf );
        return false;
    }

    unsigned int nodeId = nodeFn.typeId().id();

    if ( nodeId == hkNodeBreakableShapeID )
        return true;

    return false;
}


void hctMayaPhysicsDestructionUtilities::afterSceneLoadCallback(void* clientData)
{
    MStatus status = MStatus::kSuccess;

    MItDependencyNodes nodeIt(MFn::kInvalid, &status);

    for (; !nodeIt.isDone(); nodeIt.next())
    {
        MObject oNode = nodeIt.item();
        if ( (oNode.apiType() != MFn::kPluginLocatorNode) && (oNode.apiType() != MFn::kPluginDependNode) )
        {
            continue;
        }

        MFnDependencyNode fnNode(oNode);
        int mayaNodeId = fnNode.typeId().id();

        const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();

        const hkReflect::Type* nodeClass = HK_NULL;

        {
            for (int i = 0; i < modelerNodeIdToClassMap.getSize(); i++)
            {
                const hctModelerNodeIdToClassMap& mapEntry = modelerNodeIdToClassMap[i];
                if ( mapEntry.m_mayaNodeId == mayaNodeId )
                {
                    nodeClass = mapEntry.m_class;
                }
            }
        }
        if ( !nodeClass )
        {
            // Skip all nodes that cannot be found in the class map.
            continue;
        }

        int classVersion = -1;
        {
            for (int i = 0; i < int(fnNode.attributeCount()); i++)
            {
                MObject attrObject = fnNode.attribute(i);
                MFnAttribute attribute(attrObject, &status);
                const char* attributeName = attribute.name().asChar();
                if ( hkString::strCmp(attributeName, "internalClassVersion") == 0 )
                {
                    MPlug attributePlug(oNode, attrObject);
                    attributePlug.getValue(classVersion);
                }
            }
        }

        {
            for (int i = 0; i < int(fnNode.attributeCount()); i++)
            {
                MObject attrObject = fnNode.attribute(i);
                MFnAttribute attribute(attrObject, &status);
                const char* attributeName = attribute.name().asChar();
                MPlug attributePlug(oNode, attrObject);

                // Special versioning code for HKD-332: only if class is hkdShape and member is m_bodyQualityType and version is older than the one where we added the new ENUM member
                if ( (hkString::strCmp(nodeClass->getName(), "hkdShape") == 0) && (hkString::strCmp(attributeName, "bodyQualityType") == 0) && (classVersion < 2) )
                {
                    int bodyQualityType;
                    attributePlug.getValue(bodyQualityType);

                    // Do not change QUALITY_INHERITED type but shift all subsequent ones by 1.
                    if ( bodyQualityType >= 1 )
                    {
                        bodyQualityType++;
                        attributePlug.setValue(bodyQualityType);
                    }
                }
                if ( (hkString::strCmp(nodeClass->getName(), "hkdBody") == 0) && (hkString::strCmp(attributeName, "uuid") == 0) && (classVersion < 5) )
                {
                    // Auto-generate unique UUIDs for breakable bodies
                    hkUuid uuid;        uuid.setRandom();
                    hkStringBuf strb;   uuid.toString(strb);
                    MString mstr        (strb.cString());
                    attributePlug.setValue(mstr);
                }
                else if ( hkString::strCmp(attributeName, "internalClassVersion") == 0 )
                {
                    // There's currently no proper "generalized" versioning for modifiers available, so we simply
                    // ramp up the modifier's version value to the class's current version, assuming that "all"
                    // that was necessary will have been done once we return from this method.
                    attributePlug.setValue(nodeClass->getVersion());
                }
            }
        }
    }
}


void hctMayaPhysicsDestructionUtilities::beforeSceneSaveCallback(void* clientData)
{
    MStatus status = MStatus::kSuccess;

    MItDependencyNodes nodeIt(MFn::kInvalid, &status);

    for (; !nodeIt.isDone(); nodeIt.next())
    {
        MObject oNode = nodeIt.item();
        if ( (oNode.apiType() != MFn::kPluginLocatorNode) && (oNode.apiType() != MFn::kPluginDependNode) )
        {
            continue;
        }

        MFnDependencyNode fnNode(oNode);
        int mayaNodeId = fnNode.typeId().id();

        const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();

        const hkReflect::Type* nodeClass = HK_NULL;
        {
            for (int i = 0; i < modelerNodeIdToClassMap.getSize(); i++)
            {
                const hctModelerNodeIdToClassMap& mapEntry = modelerNodeIdToClassMap[i];
                if ( mapEntry.m_mayaNodeId == mayaNodeId )
                {
                    nodeClass = mapEntry.m_class;
                }
            }
        }
        if ( !nodeClass )
        {
            // Skip all nodes that cannot be found in the class map.
            continue;
        }

        {
            for (int i = 0; i < int(fnNode.attributeCount()); i++)
            {
                MObject attrObject = fnNode.attribute(i);
                MFnAttribute attribute(attrObject, &status);
                const char* attributeName = attribute.name().asChar();
                MPlug attributePlug(oNode, attrObject);

                if ( hkString::strCmp(attributeName, "internalClassVersion") == 0 )
                {
                    // The version value has been set to -1 by default during node construction. If the node has been loaded
                    // from a scene file, its version value will already have been updated (if necessary). If it has been
                    // newly created then it will still hold the default of -1. We must assert that when saving the scene
                    // all nodes hold their proper (current) version value.
                    attributePlug.setValue(nodeClass->getVersion());    
                    break;
                }
            }
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
