// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctContext.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctDestructionUtilities.h>

#include <ContentTools/Maya/MayaSceneExport/Nodes/RigidBody/hctRigidBodyNode.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Constraints/hctConstraintNode.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/LocalFrame/hctLocalFrameNode.h>

// User event name
#define HK_CONTEXT_EVENT "havokContextModified"

bool hctContext::m_settingsAreGlobal;


hctContext::hctContext()
{
    setTitleString( "Havok Physics Tool" );
    setImage( "havokPhysicsCtx.bmp", MPxContext::kImage1 );
    setHelpString( "Select Havok objects to manipulate their properties" );

    m_settingsAreGlobal = true;
    m_rigidBodyDisplayType = hctRigidBodyNode::m_displayType;
    m_constraintDisplayType = hctConstraintNode::m_displayType;
}


void hctContext::toolOnSetup( MEvent& )
{
    if( !m_settingsAreGlobal )
    {
        // Restore the saved modes
        hctRigidBodyNode::m_displayType = m_rigidBodyDisplayType;
        hctConstraintNode::m_displayType = m_constraintDisplayType;
        MGlobal::executeCommand( "refresh -f" );
    }

    // Add any required manipulators
    update( this );

    // Refresh manipulators if the selection changes or the context modes change
    MStatus status;
    m_callbackIds[0] = MModelMessage::addCallback( MModelMessage::kActiveListModified, update, this, &status );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Couldn't add selection callback" );
    }
    status = MUserEventMessage::registerUserEvent( HK_CONTEXT_EVENT );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Couldn't register user event" );
    }
    m_callbackIds[1] = MUserEventMessage::addUserEventCallback( HK_CONTEXT_EVENT, update, this, &status );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Couldn't add context callback" );
    }

    setCursor( MCursor::editCursor );
}

void hctContext::toolOffCleanup()
{
    // Remove any existing manipulators
    deleteManipulators();

    if( !m_settingsAreGlobal )
    {
        // Save modes before setting both to draw nothing
        m_rigidBodyDisplayType = hctRigidBodyNode::m_displayType;
        m_constraintDisplayType = hctConstraintNode::m_displayType;
        hctRigidBodyNode::m_displayType = hctLocatorNode::DISPLAY_TYPE_DIRECT;
        hctConstraintNode::m_displayType = hctLocatorNode::DISPLAY_TYPE_DIRECT;
        MGlobal::executeCommand( "refresh -f" );
    }

    // Remove our callbacks
    MStatus status;
    status = MModelMessage::removeCallback( m_callbackIds[0] );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Couldn't remove selection callback" );
    }
    status = MUserEventMessage::removeCallback( m_callbackIds[1] );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Couldn't remove context callback" );
    }
    status = MUserEventMessage::deregisterUserEvent( HK_CONTEXT_EVENT );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Couldn't deregister user event" );
    }
}


void hctContext::update( void* data )
{
    // Retrieve the context
    if( !data ) return;
    hctContext& context = *(hctContext*)data;

    // Remove any existing manipulators
    context.deleteManipulators();

    // Get active selection
    MSelectionList activeList;
    MGlobal::getActiveSelectionList( activeList );

    // Merge the selection with its children
    MStatus status = MStatus::kSuccess;
    MSelectionList mergedList( activeList );
    {
        MItSelectionList iter( activeList, MFn::kInvalid );
        for( ; !iter.isDone(); iter.next() )
        {
            MObject node;
            iter.getDependNode( node );
            MFnDagNode nodeFn( node, &status );
            if( status == MStatus::kSuccess )
            {
                unsigned int numChildren = nodeFn.childCount();
                for( unsigned int i=0; i<numChildren; ++i )
                {
                    mergedList.add( nodeFn.child(i) );
                }
            }
        }
    }

    // Create manipulators for each suitable node
    MItSelectionList iter( mergedList, MFn::kInvalid );
    for(; !iter.isDone(); iter.next() )
    {
        MObject node;
        iter.getDependNode( node );
        MFnDagNode nodeFn( node, &status );
        if( status != MStatus::kSuccess ) continue;

        // Check the selection status and display type
        bool manipulate = false;
        bool isActive = activeList.hasItem( node );
        unsigned int nodeId = nodeFn.typeId().id();
        switch( nodeId )
        {
        case hkNodeBallAndSocketConstraintID:
        case hkNodeHingeConstraintID:
        case hkNodeRagDollConstraintID:
        case hkNodeStiffSpringConstraintID:
        case hkNodePrismaticConstraintID:
        case hkNodeWheelConstraintID:
        case hkNodeFixedConstraintID:
            if( isActive || hctConstraintNode::m_displayType != hctConstraintNode::DISPLAY_TYPE_DIRECT )
            {
                manipulate = true;
            }
            break;
        case hkNodeRigidBodyID:
            if( isActive || hctRigidBodyNode::m_displayType != hctRigidBodyNode::DISPLAY_TYPE_DIRECT )
            {
                manipulate = true;
            }
            break;
        default:
            // If not listed then skip this node
            break;
        }

        if( manipulate )
        {
#if 0
            char tmp[128];
            hkString::sprintf(tmp,"create manip container %sManip",nodeFn.typeName());
            MGlobal::displayInfo(tmp);
#endif
            // Create the manip container within the context, and connect it to the node
            MObject manipObject;
            MPxManipContainer* manip = MPxManipContainer::newManipulator( nodeFn.typeName() + "Manip", manipObject );
            context.addManipulator( manipObject) ;
            manip->connectToDependNode( node );
        }
    }

    // Dirty the context so that Maya knows to redraw the property sheet
    MToolsInfo::setDirtyFlag( context );
}



//
// Context command
//

MStatus hctContextCommand::appendSyntax()
{
    MSyntax s = syntax();
    s.addFlag( "lf", "localFrameMode", MSyntax::kUnsigned );
    s.addFlag( "r", "rigidBodyMode", MSyntax::kUnsigned );
    s.addFlag( "c", "constraintMode", MSyntax::kUnsigned );
    s.addFlag( "g", "global", MSyntax::kBoolean );
    s.addFlag( "sr", "snapRotation", MSyntax::kBoolean );
    s.addFlag( "rsa", "rotationSnapAngle", MSyntax::kAngle );
    s.addFlag( "sg", "showGizmos", MSyntax::kBoolean );
    return MStatus::kSuccess;
}

MStatus hctContextCommand::doEditFlags()
{
    MArgParser p = parser();
    if( p.isFlagSet( "lf" ) )
    {
        int arg;
        p.getFlagArgument( "lf", 0, arg );
        hctLocalFrameNode::m_displayType = (hctLocalFrameNode::DisplayType)arg;
        MUserEventMessage::postUserEvent( HK_CONTEXT_EVENT );
    }
    if( p.isFlagSet( "r" ) )
    {
        int arg;
        p.getFlagArgument( "r", 0, arg );
        hctRigidBodyNode::m_displayType = (hctRigidBodyNode::DisplayType)arg;
        MUserEventMessage::postUserEvent( HK_CONTEXT_EVENT );
    }
    if( p.isFlagSet( "c" ) )
    {
        int arg;
        p.getFlagArgument( "c", 0, arg );
        hctConstraintNode::m_displayType = (hctConstraintNode::DisplayType)arg;
        MUserEventMessage::postUserEvent( HK_CONTEXT_EVENT );
    }
    if( p.isFlagSet( "g" ) )
    {
        p.getFlagArgument( "g", 0, hctContext::m_settingsAreGlobal );
    }
    if( p.isFlagSet( "sr" ) )
    {
        p.getFlagArgument( "sr", 0, hctManipContainer::m_snapRotation );
        MUserEventMessage::postUserEvent( HK_CONTEXT_EVENT );
    }
    if( p.isFlagSet( "rsa" ) )
    {
        p.getFlagArgument( "rsa", 0, hctManipContainer::m_rotationSnapAngle );
        MUserEventMessage::postUserEvent( HK_CONTEXT_EVENT );
    }
    if( p.isFlagSet( "sg" ) )
    {
        p.getFlagArgument( "sg", 0, hctGenericNodeHelper::m_showGizmos );
        MUserEventMessage::postUserEvent( HK_CONTEXT_EVENT );
    }
    return MStatus::kSuccess;
}

MStatus hctContextCommand::doQueryFlags()
{
    MArgParser p = parser();
    if( p.isFlagSet( "lf" ) )
    {
        setResult( (int)hctLocalFrameNode::m_displayType );
    }
    else if( p.isFlagSet( "r" ) )
    {
        setResult( (int)hctRigidBodyNode::m_displayType );
    }
    else if( p.isFlagSet( "c" ) )
    {
        setResult( (int)hctConstraintNode::m_displayType );
    }
    else if( p.isFlagSet( "g" ) )
    {
        setResult( hctContext::m_settingsAreGlobal );
    }
    else if( p.isFlagSet( "sr" ) )
    {
        setResult( hctManipContainer::m_snapRotation );
    }
    else if( p.isFlagSet( "rsa" ) )
    {
        setResult( hctManipContainer::m_rotationSnapAngle.asDegrees() );
    }
    else if( p.isFlagSet( "sg" ) )
    {
        setResult( hctGenericNodeHelper::m_showGizmos );
    }
    return MStatus::kSuccess;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
