// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <Common/Base/Container/String/hkStringBuf.h>

#include <sstream>

#include <ContentTools/Maya/MayaSceneExport/Nodes/Generic/hctGenericNode.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctDestructionUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/ScriptedFracture/hctScriptedFractureNode.h>
#include <ContentTools/Common/Filters/Common/Utils/hctLocaleScope.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>

//
//  Command syntax

MSyntax hctScriptedFractureNode::CmdOpenScript::getCommandSyntax()
{
    MSyntax syntax;
    syntax.addFlag( "-f", "-filename", MSyntax::kString );
    syntax.addFlag( "-n", "-node", MSyntax::kString );
    syntax.addFlag( "-l", "-layout", MSyntax::kString );
    syntax.enableQuery( true );
    return syntax;
}

//
//  Command syntax

MSyntax hctScriptedFractureNode::CmdBuildUi::getCommandSyntax()
{
    MSyntax syntax;
    syntax.addFlag( "-f", "-filename", MSyntax::kString );
    syntax.addFlag( "-n", "-node", MSyntax::kString );
    syntax.addFlag( "-l", "-layout", MSyntax::kString );
    syntax.enableQuery( true );
    return syntax;
}

//
//  Loads and parses script, then adds custom attributes to the node.

MStatus hctScriptedFractureNode::CmdOpenScript::doIt( const MArgList& args )
{
    // Get command arguments
    MArgDatabase argData( syntax(), args );

    // Check for filename
    if (!argData.isFlagSet( "-f" ) || !argData.isFlagSet( "-n" ) || !argData.isFlagSet( "-l" ) )
    {
        // No filename, node name or layout name
        return MStatus::kFailure;
    }

    // Get filename
    MString fileNameStr;
    argData.getFlagArgument( "-f", 0, fileNameStr );
    hkStringBuf scriptFileName = fileNameStr.asChar();

    // Get node name
    MString nodeNameStr;
    argData.getFlagArgument( "-n", 0, nodeNameStr );
    const char *nodeName = nodeNameStr.asChar();

    // Get layout name
    MString layoutNameStr;
    argData.getFlagArgument( "-l", 0, layoutNameStr );
    const char *layoutName = layoutNameStr.asChar();

    // Attempt to resolve the path
    hkStringBuf scriptFilePath;
    {
        hkStringBuf mayaPath;
        {
            MString assetFilePath;
            MStatus status = MGlobal::executeCommand( "file -q -sceneName", assetFilePath );
            if( status == MStatus::kSuccess && assetFilePath.length() > 0 )
            {
                mayaPath = assetFilePath.asChar();
                mayaPath.replace( '/', '\\' );
            }
        }

        if ( hctFilterUtils::resolvePath(scriptFileName, mayaPath, scriptFilePath).isFailure() )
        {
            HK_WARN_ALWAYS(0xabba5649, "Failed to open script file " << scriptFilePath << "!");
            return MStatus::kFailure;
        }
    }

    // Resolve params
    hkArray< hkStringMap<const char*>* > params;
    typedef hkResult (HK_CALL *FunPtr)(const char*, hkArray< hkStringMap<const char*>* >&);
    FunPtr getLuaScriptParameters = (FunPtr)hctSdkUtils::getUtilityFunction("getLuaScriptParameters");
    if ( !getLuaScriptParameters )
    {
        return MStatus::kSuccess;   // Destruction not enabled!
    }
    if ( getLuaScriptParameters(scriptFilePath.cString(), params).isFailure() )
    {
        return MStatus::kFailure;   // Script parsing failed!
    }

    // Get num params
    const int numParams = params.getSize();

    // The final mel script
    hkStringBuf execString;

    // Remove old extra attributes
    // Don't delete ones with parent, because the will be deleted with the parent.
    execString.printf       ("string $attributes[] = `listAttr -ud %s`;\n", nodeName);
    execString.appendPrintf ("for ( $attrib in $attributes )\n");
    execString.appendPrintf ("{\n");
    execString.appendPrintf (" \tif ( !`objExists (\"%s.\"+$attrib)` )\n", nodeName);
    execString.appendPrintf (" \t{\n");
    execString.appendPrintf ("  \t\tcontinue;\n");
    execString.appendPrintf (" \t}\n");
    execString.appendPrintf ("\t\tprint( $attrib + \"\\n\" );\n");
    execString.appendPrintf ("\t\tdeleteAttr -at $attrib %s;\n", nodeName);
    execString.appendPrintf ("}\n");

    // Add params
    for (int i = 0; i < numParams; i++)
    {
        const hkStringMap<const char*>* attributes = params[i];

        // Get id, label, value and type
        const char* id;     attributes->get( "id", &id );
        const char *label   = attributes->getWithDefault( "label", id );
        const char* value;  attributes->get( "value", &value );
        const char *type    = attributes->getWithDefault( "type", "spinner" );

        // Check type
        if ( hkString::strCmp( type, "text" ) == 0 )
        {
            // String
            execString.appendPrintf("addAttr -dt \"string\" -nn \"%s\" -ln \"%s\" %s;\n", label, id, nodeName);
        }
        else if ( hkString::strCmp( type, "spinner" ) == 0 )
        {
            // Float value
            execString.appendPrintf("addAttr -nn \"%s\" -ln \"%s\" -dv ", label, id, value);

            // Check for min/max
            if ( attributes->hasKey( "min" ) && attributes->hasKey( "max" ) )
            {
                const char *min; attributes->get( "min", &min );
                const char *max; attributes->get( "max", &max );
                execString.appendPrintf(" -min %s -max %s", min, max);
            }

            // Finish up
            execString.appendPrintf(" %s;\n", nodeName);
        }
        else if ( hkString::strCmp( type, "intspinner" ) == 0 )
        {
            // Integer value
            execString.appendPrintf("addAttr -at long -nn \"%s\" -ln \"%s\" -dv %s", label, id, value);

            // Check for min/max
            if ( attributes->hasKey( "min" ) && attributes->hasKey( "max" ) )
            {
                const char *min; attributes->get( "min", &min );
                const char *max; attributes->get( "max", &max );
                execString.appendPrintf(" -min %s -max %s", min, max);
            }

            // Finish up
            execString.appendPrintf(" %s;\n", nodeName);
        }
        else if ( hkString::strCmp( type, "checkbox" ) == 0 )
        {
            // Boolean value
            execString.appendPrintf("addAttr -at bool -nn \"%s\" -ln \"%s\" -dv %s %s;\n", label, id, value, nodeName);
        }
        else if ( hkString::strCmp( type, "vector" ) == 0 )
        {
            // 3d vector

            // Get default values
            float x,y,z;
            sscanf_s( value, "[%f,%f,%f]", &x, &y, &z );

            // Create float3 master attribute
            execString.appendPrintf("addAttr -at float3 -nn \"%s\" -ln \"%s\" %s;\n", label, id, nodeName);

            // Add x, y, z
            execString.appendPrintf("addAttr -at \"float\" -ln \"%s_x\" -dv %f -p \"%s\" %s;\n", id, x, id, nodeName);
            execString.appendPrintf("addAttr -at \"float\" -ln \"%s_y\" -dv %f -p \"%s\" %s;\n", id, y, id, nodeName);
            execString.appendPrintf("addAttr -at \"float\" -ln \"%s_z\" -dv %f -p \"%s\" %s;\n", id, z, id, nodeName);
        }
        else if ( hkString::strCmp( type, "quaternion" ) == 0 )
        {
            // Quaternion
            execString.appendPrintf("addAttr -at float4 -nn \"%s\" -ln \"%s\" %s;\n", label, id, nodeName);
        }
        else if ( hkString::strCmp( type, "node" ) == 0 )
        {
            // Link to other node
            execString.appendPrintf("addAttr -hidden 1 -at message -nn \"%s\" -ln \"%s\" %s;\n", label, id, nodeName);
        }
        else if ( hkString::strCmp( type, "enum" ) == 0 )
        {
            // Link to other node
            execString.appendPrintf("addAttr -hidden 1 -dt \"string\" -nn \"%s\" -ln \"%s\" %s;\n", label, id, nodeName);
            execString.appendPrintf("setAttr %s.%s -type \"string\" \"%s\";\n", nodeName, id, value);
        }
    }

    // Build additional UI
    execString.appendPrintf("hkdCmdScriptedFractureBuildUi -f \"%s\" -n \"%s\" -l \"%s\";\n", scriptFileName.cString(), nodeName, layoutName);

    // Run script
    const char* execStr = execString.cString();
    const MString mstr(execStr);
    MGlobal::executeCommand( mstr );

    // Done
    return MStatus::kSuccess;
}

//
//  Loads and parses script, then creates additional UI on the attributes rollout.

MStatus hctScriptedFractureNode::CmdBuildUi::doIt( const MArgList& args )
{
    // Get command arguments
    MArgDatabase argData( syntax(), args );

    // Check for filename
    if (!argData.isFlagSet( "-f" ) || !argData.isFlagSet( "-n" ) || !argData.isFlagSet( "-l" ) )
    {
        // No filename, nodename or layout name
        return MStatus::kFailure;
    }

    // Get filename
    MString fileNameStr;
    argData.getFlagArgument( "-f", 0, fileNameStr );
    hkStringBuf scriptFileName = fileNameStr.asChar();

    // Get node name
    MString nodeNameStr;
    argData.getFlagArgument( "-n", 0, nodeNameStr );
    const char *nodeName = nodeNameStr.asChar();

    // Get layout name
    MString layoutNameStr;
    argData.getFlagArgument( "-l", 0, layoutNameStr );
    const char *layoutName = layoutNameStr.asChar();

    // Attempt to resolve the path
    hkStringBuf scriptFilePath;
    {
        hkStringBuf mayaPath;
        {
            MString assetFilePath;
            MStatus status = MGlobal::executeCommand( "file -q -sceneName", assetFilePath );
            if( status == MStatus::kSuccess && assetFilePath.length() > 0 )
            {
                mayaPath = assetFilePath.asChar();
                mayaPath.replace( '/', '\\' );
            }
        }

        if ( hctFilterUtils::resolvePath(scriptFileName, mayaPath, scriptFilePath).isFailure() )
        {
            HK_WARN_ALWAYS(0xabba5649, "Failed to open script file " << scriptFilePath << "!");
            return MStatus::kFailure;
        }
    }

    // Resolve params
    hkArray< hkStringMap<const char*>* > params;
    typedef hkResult (HK_CALL *FunPtr)(const char*, hkArray< hkStringMap<const char*>* >&);
    FunPtr getLuaScriptParameters = (FunPtr)hctSdkUtils::getUtilityFunction("getLuaScriptParameters");
    if ( !getLuaScriptParameters )
    {
        return MStatus::kSuccess;   // Destruction not enabled!
    }
    if ( getLuaScriptParameters(scriptFilePath.cString(), params).isFailure() )
    {
        return MStatus::kFailure;   // Script parsing failed!
    }

    // Get num params
    const int numParams = params.getSize();

    hkStringBuf execString;

    // Remove old layout ( anything what's it starts with 'custom_' )
    execString.printf       ("string $children[] = `layout -q -childArray %s`;\n", layoutName);
    execString.appendPrintf ("for ( $child in $children )\n");
    execString.appendPrintf ("{\n");
    execString.appendPrintf ("\t\tif ( !startsWith( $child, \"custom_\" ) )\n");
    execString.appendPrintf ("\t\t{\n");
    execString.appendPrintf ("\t\t\tcontinue;\n");
    execString.appendPrintf ("\t\t}\n");
    execString.appendPrintf ("\t\tdeleteUI $child;\n");
    execString.appendPrintf ("}\n");

    // Set parent for adding the new UI
    execString.appendPrintf ("setParent %s;\n", layoutName);

    // Add params
    for (int i = 0; i < numParams; i++)
    {
        const hkStringMap<const char*>* attributes = params[i];

        // Get id, label, value and type
        const char* id;     attributes->get( "id", &id );
        const char *label   = attributes->getWithDefault( "label", id );
        const char* value;  attributes->get( "value", &value );
        const char *type    = attributes->getWithDefault( "type", "spinner" );

        // Check type
        if ( hkString::strCmp( type, "node" ) == 0 )
        {
            // Link to other node.
            execString.appendPrintf("string $attribName = \"%s.%s\";\n", nodeName, id);
            execString.appendPrintf("AEhctScriptedFracture_linkCreate $attribName \"%s\";\n", label);
        }
        else if ( hkString::strCmp( type, "enum" ) == 0 )
        {
            // Get options
            const char* itemsText; attributes->get( "items", &itemsText );

            // Get attributes
            hkArray<hkStringBuf*> items;

            const char* start = HK_NULL;
            for ( const char* pos = itemsText; *pos != 0; pos++ )
            {
                // Check for escaped characters
                if ( *pos == '\\' && pos[1] != 0 )
                {
                    pos++;
                    continue;
                }

                // Check for quote marks
                if ( *pos != '\"' )
                {
                    continue;
                }

                // Start
                if ( start == HK_NULL )
                {
                    start = pos + 1;
                    continue;
                }

                // End
                items.pushBack( new hkStringBuf( start, (int)( pos - start ) ) );

                start = HK_NULL;
            }

            const int numItems = items.getSize();

            // Combo box
            execString.appendPrintf("setUITemplate -pst attributeEditorTemplate;\n");
            execString.appendPrintf("columnLayout (\"custom_%s\");\n", id);
            execString.appendPrintf(" \toptionMenuGrp -label \"%s\" -cc (\"string $temp=`optionMenuGrp -q -v \\\"options\\\"`; setAttr %s.%s -type \\\"string\\\" $temp;\") \"options\";\n", label, nodeName, id);
            for (int itemIndex = 0; itemIndex < numItems; itemIndex++)
            {
                execString.appendPrintf("  \t\tmenuItem -label \"%s\";\n", items[itemIndex]->cString());
                delete items[itemIndex];
            }
            execString.appendPrintf("string $value = `getAttr %s.%s`;\n", nodeName, id);
            execString.appendPrintf("optionMenuGrp -e -value $value \"options\";\n");
        }
    }

    // Run script
    const char* execStr = execString.cString();
    const MString mstr(execStr);
    MGlobal::executeCommand(mstr);

    // Done
    return MStatus::kSuccess;
}

//
// Register scripted fracture and additional mel bindings
//

MStatus hctScriptedFractureNode::initialize()
{
    // Register C functions
    {
        MFnPlugin plugin( hctMayaPhysicsDestructionUtilities::m_pluginObject, "Havok - Scene Exporter", HCT_CURRENT_VERSION_STRING );

        MStatus status = MStatus::kSuccess;
        bool error = false;

        // Register functions
        error = error || ( ( status = plugin.registerCommand("hkdCmdScriptedFractureOpenScript",    hctScriptedFractureNode::CmdOpenScript::creator,    hctScriptedFractureNode::CmdOpenScript::getCommandSyntax ) )    != MStatus::kSuccess );
        error = error || ( ( status = plugin.registerCommand("hkdCmdScriptedFractureBuildUi",       hctScriptedFractureNode::CmdBuildUi::creator,       hctScriptedFractureNode::CmdBuildUi::getCommandSyntax ) )       != MStatus::kSuccess );

        // Check for errors
        if( error )
        {
            status.perror( "ScriptedFracture command registration" );
            return status;
        }
    }


    hctLocaleScope scope;

    MStatus status = MStatus::kSuccess;

    // Register fracture
    {
        MFnStringData stringFn;
        MFnTypedAttribute typedFn;

        // Note: the type name has to start with "hkType" as we will use certain (older) utility functions to collect attributes that rely on this.
        MObject m_hkType = typedFn.create( "hkTypeDestruction", "hktd", MFnData::kString, stringFn.create("hkdScriptedFracture"), &status );
        if ( !status )
        {
            MGlobal::displayError("Couldn't create hkDestructionType attribute.");
            return MStatus::kFailure;
        }

        typedFn.setWritable(false);
        typedFn.setHidden(true);
        MPxNode::addAttribute(m_hkType);
    }

    // Add versioning attribute
    {
        MFnNumericAttribute numFn;
        hctGenericNodeHelper::Attribute attrib("internalClassVersion");
        attrib.m_object = numFn.create("internalClassVersion", "internalClassVersion", MFnNumericData::kInt, -1, &status);
        MPxNode::addAttribute(attrib.m_object);
    }

    // Add script path attribute
    {
        hctGenericNodeHelper::Attribute attrib( "scriptPath" );
        MFnTypedAttribute typedFn;
        MFnStringData     stringFn;
        attrib.m_object = typedFn.create("scriptPath", "scriptPath", MFnData::kString, stringFn.create("Select script"), &status);
        MPxNode::addAttribute( attrib.m_object );
    }

    return status;
}

//
// Nothing to draw
//

void hctScriptedFractureNode::draw( M3dView& view, const MDagPath& path, M3dView::DisplayStyle style, M3dView::DisplayStatus displayStatus )
{
    // Do nothing
}

//
// Always save
//

MStatus hctScriptedFractureNode::shouldSave( const MPlug& plug, bool& isSaving )
{
    isSaving = true;

    return MStatus::kSuccess;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
