// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/RigidBody/hctRigidBodyNode.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

// Statics
MObject hctRigidBodyNode::m_hkType;
MObject hctRigidBodyNode::m_mass;
MObject hctRigidBodyNode::m_changeCenterOfMass;
MObject hctRigidBodyNode::m_centerOfMass;
MObject hctRigidBodyNode::m_changeInertiaTensor;
MObject hctRigidBodyNode::m_inertiaTensor;
MObject hctRigidBodyNode::m_scaleInertiaTensor;
MObject hctRigidBodyNode::m_inertiaTensorScale;
MObject hctRigidBodyNode::m_friction;
MObject hctRigidBodyNode::m_restitution;

MObject hctRigidBodyNode::m_changeQualityType;
MObject hctRigidBodyNode::m_qualityType;
MObject hctRigidBodyNode::m_changeSolverDeactivation;
MObject hctRigidBodyNode::m_solverDeactivation;
MObject hctRigidBodyNode::m_changeCollisionFilterInfo;
MObject hctRigidBodyNode::m_collisionFilterInfo;
MObject hctRigidBodyNode::m_changeRigidBodyDeactivatorType;
MObject hctRigidBodyNode::m_rigidBodyDeactivatorType;
MObject hctRigidBodyNode::m_changeMaxAngularVelocity;
MObject hctRigidBodyNode::m_maxAngularVelocity;
MObject hctRigidBodyNode::m_changeMaxLinearVelocity;
MObject hctRigidBodyNode::m_maxLinearVelocity;
MObject hctRigidBodyNode::m_changeLinearDamping;
MObject hctRigidBodyNode::m_linearDamping;
MObject hctRigidBodyNode::m_changeAngularDamping;
MObject hctRigidBodyNode::m_angularDamping;
MObject hctRigidBodyNode::m_changeAllowedPenetrationDepth;
MObject hctRigidBodyNode::m_allowedPenetrationDepth;


hctLocatorNode::DisplayType hctRigidBodyNode::m_displayType = hctLocatorNode::DISPLAY_TYPE_ALL;


MStatus hctRigidBodyNode::initialize()
{
    MStatus status = MStatus::kSuccess;

    MFnTypedAttribute   typedFn;
    MFnMessageAttribute messageFn;
    MFnMatrixAttribute  matrixFn;
    MFnEnumAttribute    enumFn;
    MFnNumericAttribute numFn;
    MFnUnitAttribute    unitFn;

    // hkType
    {
        MFnStringData stringFn;
        m_hkType = typedFn.create( "hkType", "hkt", MFnData::kString, stringFn.create( "hkRigidBody" ), &status );
        if( !status ) MGlobal::displayError( "Couldn't create hkType attr" );
        typedFn.setWritable( false );
        typedFn.setHidden( true );
        addAttribute( m_hkType );
    }

    // Mass
    {
        m_mass = numFn.create( "mass", "ma", MFnNumericData::kDouble, 0.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create mass attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        numFn.setSoftMax( 100.0 );
        addAttribute( m_mass );

        m_changeCenterOfMass = numFn.create( "changeCenterOfMass", "ccom", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create mass change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeCenterOfMass );

        MObject comx = unitFn.create( "centerOfMassX", "comX", MFnUnitAttribute::kDistance, 0.0);
        MObject comy = unitFn.create( "centerOfMassY", "comY", MFnUnitAttribute::kDistance, 0.0);
        MObject comz = unitFn.create( "centerOfMassZ", "comZ", MFnUnitAttribute::kDistance, 0.0);
        m_centerOfMass = numFn.create( "centerOfMass", "com", comx, comy, comz, &status );
        if( !status ) MGlobal::displayError( "Couldn't create center of mass attr" );
        numFn.setKeyable( true );
        addAttribute( m_centerOfMass );

        m_changeInertiaTensor = numFn.create( "changeInertiaTensor", "cit", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create inertia change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeInertiaTensor );

        MObject itx = numFn.create( "inertiaTensorX", "itX", MFnNumericData::kDouble, 1.0);
        MObject ity = numFn.create( "inertiaTensorY", "itY", MFnNumericData::kDouble, 1.0);
        MObject itz = numFn.create( "inertiaTensorZ", "itZ", MFnNumericData::kDouble, 1.0);
        m_inertiaTensor = numFn.create( "inertiaTensor", "it", itx, ity, itz, &status );
        if( !status ) MGlobal::displayError( "Couldn't create inertia tensor attr" );
        numFn.setKeyable( true );
        addAttribute( m_inertiaTensor );

        m_scaleInertiaTensor = numFn.create( "scaleInertiaTensor", "sit", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create inertia scale attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_scaleInertiaTensor );

        m_inertiaTensorScale = numFn.create( "inertiaTensorScale", "its", MFnNumericData::kDouble, 1.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create inertia scale value attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.001 );
        numFn.setSoftMax( 10.0 );
        addAttribute( m_inertiaTensorScale );
    }

    // Other
    {
        m_friction = numFn.create( "friction", "f", MFnNumericData::kDouble, 0.5, &status );
        if( !status ) MGlobal::displayError( "Couldn't create friction attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        numFn.setSoftMax( 1.0 );
        numFn.setMax( 255.0 );
        addAttribute( m_friction );

        m_restitution = numFn.create( "restitution", "r", MFnNumericData::kDouble, 0.4, &status );
        if( !status ) MGlobal::displayError( "Couldn't create restitution attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        numFn.setSoftMax (1.0);
        numFn.setMax( 1.98 );
        addAttribute( m_restitution );
    }

    // Advanced attributes
    {
        m_changeQualityType = numFn.create( "changeQualityType", "cqt", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create qualityType change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeQualityType );

        m_qualityType = enumFn.create( "qualityType", "qt", 0, &status);
        if( !status ) MGlobal::displayError( "Couldn't create qualityType attr" );
        enumFn.addField( "Fixed", 1 );
        enumFn.addField( "Keyframed", 2 );
        enumFn.addField( "Keyframed Reporting", 3 );
        enumFn.addField( "Debris", 4);
        enumFn.addField( "Moving", 5 );
        enumFn.addField( "Critical", 6 );
        enumFn.addField( "Bullet", 7 );
        enumFn.addField( "Debris Simple TOI", 8);
        enumFn.setDefault( 5 );
        enumFn.setKeyable( true );
        addAttribute( m_qualityType );

        m_changeSolverDeactivation = numFn.create( "changeSolverDeactivation", "csd", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create solverDeactivation change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeSolverDeactivation );

        m_solverDeactivation = enumFn.create( "solverDeactivation", "sd", 0, &status);
        if( !status ) MGlobal::displayError( "Couldn't create solverDeactivation attr" );
        enumFn.addField( "Off", 1 );
        enumFn.addField( "Low", 2 );
        enumFn.addField( "Medium", 3 );
        enumFn.addField( "High", 4 );
        enumFn.setDefault( 3 );
        enumFn.setKeyable( true );
        addAttribute( m_solverDeactivation );

        m_changeRigidBodyDeactivatorType = numFn.create( "changeRigidBodyDeactivatorType", "crbd", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create rigidBodyDeactivatorType change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeRigidBodyDeactivatorType );

        m_rigidBodyDeactivatorType = enumFn.create( "rigidBodyDeactivatorType", "rbd", 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create rigidBodyDeactivatorType attr" );
        enumFn.addField( "Spatial", 1 );
        enumFn.addField( "Never", 2 );
        enumFn.setDefault( 1 );
        enumFn.setKeyable( true );
        addAttribute( m_rigidBodyDeactivatorType );

        m_changeCollisionFilterInfo = numFn.create( "changeCollisionFilterInfo", "ccfi", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create collisionFilterInfo change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeCollisionFilterInfo );

        m_collisionFilterInfo = numFn.create( "collisionFilterInfo", "cfi", MFnNumericData::kInt, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create collisionFilterInfo attr" );
        numFn.setKeyable( true );
        addAttribute( m_collisionFilterInfo );

        m_changeMaxAngularVelocity = numFn.create( "changeMaxAngularVelocity", "cmav", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create maxAngularVelocity change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeMaxAngularVelocity );

        m_maxAngularVelocity = numFn.create( "maxAngularVelocity", "mav", MFnNumericData::kDouble, 200.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create maxAngularVelocity attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        addAttribute( m_maxAngularVelocity );

        m_changeMaxLinearVelocity = numFn.create( "changeMaxLinearVelocity", "cmlv", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create maxLinearVelocity change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeMaxLinearVelocity );

        m_maxLinearVelocity = numFn.create( "maxLinearVelocity", "mlv", MFnNumericData::kDouble, 200.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create maxLinearVelocity attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        addAttribute( m_maxLinearVelocity );

        m_changeLinearDamping = numFn.create( "changeLinearDamping", "cld", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create linearDamping change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeLinearDamping );

        m_linearDamping = numFn.create( "linearDamping", "ld", MFnNumericData::kDouble, 0.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create linearDamping attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        addAttribute( m_linearDamping );

        m_changeAngularDamping = numFn.create( "changeAngularDamping", "cad", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create angularDamping change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeAngularDamping );

        m_angularDamping = numFn.create( "angularDamping", "ad", MFnNumericData::kDouble, 0.05, &status );
        if( !status ) MGlobal::displayError( "Couldn't create angularDamping attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        addAttribute( m_angularDamping );

        m_changeAllowedPenetrationDepth = numFn.create( "changeAllowedPenetrationDepth", "capd", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create allowedPenetrationDepth change attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_changeAllowedPenetrationDepth );

        m_allowedPenetrationDepth = numFn.create( "allowedPenetrationDepth", "apd", MFnNumericData::kDouble, 0.05, &status );
        if( !status ) MGlobal::displayError( "Couldn't create allowedPenetrationDepth attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        addAttribute( m_allowedPenetrationDepth );

    }

    // Allow manipulation
    MTypeId id = typeId();
    MPxManipContainer::addToManipConnectTable( id );

    return status;
}


void hctRigidBodyNode::draw( M3dView& view, const MDagPath&, M3dView::DisplayStyle, M3dView::DisplayStatus )
{
    // Check if it should be displayed
    MObject node = thisMObject();
    if( !shouldDisplay( node, m_displayType ) )
    {
        return;
    }

    MDagPath nodePath;
    MDagPath::getAPathTo( node, nodePath );

    MDagPath parentPath( nodePath );
    parentPath.pop();

    MFnDependencyNode nodeFn( node );
    MFnAttribute attrFn;
    MFnNumericData numFn;
    MObject obj;

    // Get basic attribute values
    bool useCom;
    {
        attrFn.setObject( m_changeCenterOfMass );
        nodeFn.findPlug( attrFn.name() ).getValue( useCom );
    }
    bool useIt;
    {
        attrFn.setObject( m_changeInertiaTensor );
        nodeFn.findPlug( attrFn.name() ).getValue( useIt );
    }
    double mass;
    {
        attrFn.setObject( m_mass );
        nodeFn.findPlug( attrFn.name() ).getValue( mass );
    }
    bool scaleIt;
    {
        attrFn.setObject( m_scaleInertiaTensor );
        nodeFn.findPlug( attrFn.name() ).getValue( scaleIt );
    }

    bool changeQuality;
    {
        attrFn.setObject(m_changeQualityType);
        nodeFn.findPlug( attrFn.name()).getValue( changeQuality );
    }

    int qualityType;
    {
        attrFn.setObject(m_qualityType);
        nodeFn.findPlug( attrFn.name()).getValue( qualityType );
    }

    // Begin OpenGL
    view.beginGL();
    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_LINE_BIT | GL_POLYGON_BIT );
    glPushMatrix();

    // Get center of mass, draw it if necessary
    MVector com;
    MPoint pivot;
    hctUtilities::getOptionalPivotOffset( parentPath, pivot );
    if( useCom )
    {
        attrFn.setObject( m_centerOfMass );
        nodeFn.findPlug( attrFn.name() ).getValue( obj );
        numFn.setObject( obj );
        numFn.getData( com.x, com.y, com.z );

        // Add parent transform's pivot offset
        com += pivot;

        // Draw indicator (a box)
        view.setDrawColor( hctUtilities::getColor( "CenterOfMass" ) );
        glPointSize( 6.0 );
        glBegin( GL_POINTS );
        glVertex3dv( &com.x );
        glEnd();
    }

    // Draw inertia tensor if necessary
    if( useIt || scaleIt )
    {
        // Read normalize attribute
        MVector it;

        if ( useIt )
        {
            attrFn.setObject( m_inertiaTensor );
            nodeFn.findPlug( attrFn.name() ).getValue( obj );
            numFn.setObject( obj );
            numFn.getData( it.x, it.y, it.z );

            it.x = fabs( it.x );
            it.y = fabs( it.y );
            it.z = fabs( it.z );
            const double normalizeScaleFactor = pow( 1 / ( it.x * it.y * it.z ), 0.33333 );
            it *= normalizeScaleFactor;
        }
        else
        {
            it.x = it.y = it.z = 1;
        }

        if ( scaleIt )
        {
            double inertiaTensorScale;
            attrFn.setObject( m_inertiaTensorScale );
            nodeFn.findPlug( attrFn.name() ).getValue( inertiaTensorScale );

            it *= inertiaTensorScale;
        }

        // Rescale to match bounding box
        MBoundingBox boundingBox;
        getMeshBoundingBox( nodePath, boundingBox );

        it.x *= boundingBox.width();
        it.y *= boundingBox.height();
        it.z *= boundingBox.depth();

        // Draw scaled wireframe unit cube, without depth testing
        glPushMatrix();
        {
            glTranslated( com.x, com.y, com.z );
            hctUtilities::removeGlScaleAndSkew();
            glScaled( it.x, it.y, it.z );

            view.setDrawColor( hctUtilities::getColor( "InertiaTensor" ) );
            glDisable( GL_DEPTH_TEST );
            glLineWidth( 1.0 );
            glEnable( GL_LINE_STIPPLE );
            glLineStipple( 1, 0xAAAA );
            glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
            hctUtilities::drawCube();
        }
        glPopMatrix();
    }

    bool displayAsFixed;
    {
        // EXP-1021
        if (changeQuality)
        {
            switch (qualityType)
            {
                case 1: // fixed
                case 2: // keyframed
                case 3: // keyframed reporting
                    displayAsFixed = true;
                    break;
                default:
                    displayAsFixed = (mass == 0.0f);
                    break;
            }
        }
        else
        {
            displayAsFixed = (mass==0.0f);
        }
    }

    // Draw label
    if( displayAsFixed )
    {
        view.setDrawColor( hctUtilities::getColor( "FixedRigidBodyLabel" ) );
        view.drawText( "[RB]",  pivot, M3dView::kCenter );
    }
    else
    {
        view.setDrawColor( hctUtilities::getColor( "RigidBodyLabel" ) );
        view.drawText( "RB",  pivot, M3dView::kCenter );
    }

    // End OpenGL
    glPopMatrix();
    glPopAttrib();
    view.endGL();
}

void hctRigidBodyNode::getMeshBoundingBox( const MDagPath& path, MBoundingBox& bb )
{
    MDagPath parentPath = path;
    parentPath.pop();
    for( unsigned int i=0; i<parentPath.childCount(); ++i )
    {
        const MObject& child = parentPath.child(i);
        if( child.hasFn( MFn::kMesh ) )
        {
            MFnDagNode dagFn( child );
            bb = dagFn.boundingBox();
            break;
        }
    }
}




const hctManipContainer::ManipDesc* hctRigidBodyManip::getManipulatorDesc( int id ) const
{
    static ManipDesc comManipDesc( ManipDesc::MANIP_TYPE_POINT, hctRigidBodyNode::m_centerOfMass );
    static ManipDesc itManipDesc( ManipDesc::MANIP_TYPE_SCALE, hctRigidBodyNode::m_inertiaTensor );

    switch( id )
    {
    case MANIP_CENTER_OF_MASS:  return &comManipDesc;
    case MANIP_INERTIA_TENSOR:  return &itManipDesc;
    default:                    return NULL;
    }
}

MMatrix hctRigidBodyManip::getManipulatorBaseTransform( int id ) const
{
    MDagPath targetPath;
    MDagPath::getAPathTo( m_target, targetPath );
    MFnDependencyNode nodeFn( targetPath.node() );

    MMatrix matrix;

    // Read center of mass override state
    bool useCom;
    {
        MFnAttribute attrFn( hctRigidBodyNode::m_changeCenterOfMass );
        nodeFn.findPlug( attrFn.name() ).getValue( useCom );
    }

    switch( id )
    {
    case MANIP_CENTER_OF_MASS:
        {
            // Center of mass may work from the pivot point or the node transform
            if( useCom )
            {
                MDagPath parentPath( targetPath ); parentPath.pop();
                hctUtilities::getOptionalPivotWorldTransform( parentPath, matrix );
            }
            else
            {
                matrix = targetPath.inclusiveMatrix();
            }
        }
        break;

    case MANIP_INERTIA_TENSOR:
        {
            // Match it to the COM manip
            matrix = getManipulatorBaseTransform( MANIP_CENTER_OF_MASS );

            // Add the COM offset
            if( useCom )
            {
                MVector com;
                {
                    MObject obj;
                    MFnAttribute attrFn( hctRigidBodyNode::m_centerOfMass );
                    nodeFn.findPlug( attrFn.name() ).getValue( obj );
                    MFnNumericData numFn( obj );
                    numFn.getData( com.x, com.y, com.z );
                }
                MTransformationMatrix xform;
                xform.setTranslation( com, MSpace::kTransform );
                matrix = xform.asMatrix() * matrix;
            }
        }
        break;

    default:
        break;
    }

    return matrix;
}

bool hctRigidBodyManip::getManipulatorVisibility( int id ) const
{
    MFnDependencyNode nodeFn( m_target );
    bool visible = true;

    switch( id )
    {
    case MANIP_CENTER_OF_MASS:
    {
        MFnAttribute attrFn( hctRigidBodyNode::m_changeCenterOfMass );
        nodeFn.findPlug( attrFn.name() ).getValue( visible );
        break;
    }

    case MANIP_INERTIA_TENSOR:
    {
        MFnAttribute attrFn( hctRigidBodyNode::m_changeInertiaTensor );
        nodeFn.findPlug( attrFn.name() ).getValue( visible );
        break;
    }

    default:
        break;
    }

    return visible;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
