// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <sstream>

#include <ContentTools/Maya/MayaSceneExport/Gizmos/hctSimpleGizmos.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Generic/hctGenericNode.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctDestructionUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>
#include <ContentTools/Common/Filters/Common/Utils/hctLocaleScope.h>

#include <Common/Base/Math/Quaternion/hkQuaternionUtil.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>
#include <Common/Base/Types/Uuid/hkUuid.h>


bool hctGenericNodeHelper::m_showGizmos = false;


MStatus hctGenericNodeHelper::initialize()
{
    hctLocaleScope scope;

    MStatus status = MStatus::kSuccess;

    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();
    const hkReflect::Type* klass = modelerNodeIdToClassMap[m_classNumber].m_class;

    //
    // Find out if this node supports documentation.
    //
    bool docsAvailable = false;
    {
        const hkDocumentationAttribute* documentationAttribute = klass->findAttribute<hkDocumentationAttribute>();
        if ( documentationAttribute )
        {
            const hkAttributeHideCriteria::Types hideCriteria = hctSdkUtils::getUiAttributesHideCriteria(hkAttributeHideCriteria::MODELER_IS_MAYA);
            const char* docsSectionTag = documentationAttribute->m_getDocsSectionTag(hideCriteria);
            if ( docsSectionTag && docsSectionTag[0] != 0 )
            {
                docsAvailable = true;
            }
        }
    }

    // Query UI scheme and set additional hide criteria
    const hkAttributeHideCriteria::Types hideCriteria = hctSdkUtils::getUiAttributesHideCriteria(hkAttributeHideCriteria::MODELER_IS_MAYA);

    //
    // Create a hierarchy from all the class's members (taking GROUP tagging and structs into account).
    // This will for now include INVISIBLE members as well. Those will get removed later on below right
    // before we build the actual GUI creation code.
    //
    std::string groupPathDelimiter(":"); // sync id: <0xaf1ee231>
    hkReal unitConversionModifier = 1.0; // no conversion
    std::vector<hctClassHierarchyUtil::ParamGroup*> groupsArray;
    hctClassHierarchyUtil::createDisplayHierarchyFromClass(klass, groupPathDelimiter, unitConversionModifier, groupsArray);
    hctClassHierarchyUtil::markVisuallyEmptyGroup(hideCriteria, groupsArray);

    {
        MFnStringData stringFn;
        MFnTypedAttribute typedFn;

        // Note: the type name has to start with "hkType" as we will use certain (older) utility functions to collect attributes that rely on this.
        MObject m_hkType = typedFn.create( "hkTypeDestruction", "hktd", MFnData::kString, stringFn.create(klass->getName()), &status );
        if ( !status )
        {
            MGlobal::displayError("Couldn't create hkDestructionType attribute.");
            return MStatus::kFailure;
        }

        typedFn.setWritable(false);
        typedFn.setHidden(true);
        MPxNode::addAttribute(m_hkType);
    }

    //
    // Create AETemplate output file.
    //

    hkArray<char> filenameBuffer;
    {
        MFnPlugin pluginFn(hctMayaPhysicsDestructionUtilities::m_pluginObject);
        MString path = pluginFn.loadPath(&status);
        const char* pluginPath = path.asChar();
        hkOstream filenameCreator(filenameBuffer);
        filenameCreator.printf("%s/../scripts/AETemplates/AE%sTemplate.mel", pluginPath, modelerNodeIdToClassMap[m_classNumber].m_nodeName);
    }
    hkOstream streamWriter(filenameBuffer.begin());

    //
    // Output AETemplate header.
    //
    {
        streamWriter.printf("//\n");
        streamWriter.printf("// Havok Node editor template\n");
        streamWriter.printf("// This file is auto-generated!\n");
        streamWriter.printf("//\n");
        streamWriter.printf("\n");
        streamWriter.printf("global proc AE%sTemplate( string $node )\n", modelerNodeIdToClassMap[m_classNumber].m_nodeName);
        streamWriter.printf("{\n");
        streamWriter.printf("\thEhkNodeCommon_suppressLocatorAttributes();\n");
        streamWriter.printf("\thEhkGenericNode_suppressUnexposed();\n");
        streamWriter.printf("\n");
        streamWriter.printf("\teditorTemplate -beginScrollLayout;\n");
        streamWriter.printf("\t\teditorTemplate -beginNoOptimize;\n");
    }

    //
    // Add a disclaimer if a feature is currently under review.
    //
    if (modelerNodeIdToClassMap[m_classNumber].m_internalClassification == hctModelerNodeIdToClassMap::UNDERGOING_REVIEW)
    {
        createUnderReviewNotificationGroup(klass, streamWriter);
    }

    //
    // Create a dedicated HELP group.
    //
    if ( docsAvailable )
    {
        createHelpGroup(klass, streamWriter);
    }

    //
    // Output class internal version number. Default is set to -1. This is necessary so that old scene files that don't
    // have the version value stored will get properly versioned. When loading the scene this class version value will get
    // overwritten by the actual version of the node in the file. For newly created nodes this version value will be properly
    // set just prior to saving the scene via a beforeSceneSave callback.
    //
    {
            MFnNumericAttribute numFn;
            Attribute attrib("internalClassVersion");
            attrib.m_object = numFn.create("internalClassVersion", "internalClassVersion", MFnNumericData::kInt, -1, &status);
            MPxNode::addAttribute(attrib.m_object);
            streamWriter.printf("\t\teditorTemplate -suppress \"internalClassVersion\";\n");
    }

    //
    // Output the class parameters & groups.
    //
    {
        for (int i = 0; i < int(groupsArray.size()); i++)
        {
            hctClassHierarchyUtil::ParamGroup* group = groupsArray[i];
            createAttributesFromParamGroup(hkAttributeHideCriteria::Types(hideCriteria), group, streamWriter);
        }
    }

    //
    // Output AETemplate footer.
    //
    {
        streamWriter.printf("\t\teditorTemplate -endNoOptimize;\n");
        streamWriter.printf("\teditorTemplate -addExtraControls;\n");
        streamWriter.printf("\teditorTemplate -endScrollLayout;\n");
        streamWriter.printf("}\n\n");
        if (modelerNodeIdToClassMap[m_classNumber].m_internalClassification == hctModelerNodeIdToClassMap::UNDERGOING_REVIEW)
        {
            createUnderReviewNotificationText(streamWriter);
        }
        if ( docsAvailable )
        {
            createHelpButton(klass, streamWriter);
        }
        streamWriter.flush();
    }

    //
    // Allow manipulation on hkdShape.
    //
    {
        MTypeId id = modelerNodeIdToClassMap[m_classNumber].m_mayaNodeId;
        if ( id == hkNodeBreakableShapeID )
        {
            MPxManipContainer::addToManipConnectTable(id);
        }
    }

    return status;
}


void hctGenericNodeHelper::createAttributesFromParamGroup(hkAttributeHideCriteria::Types hideCriteria, const hctClassHierarchyUtil::ParamGroup* group, hkOstream& streamWriter)
{
    std::stringstream indentStream; indentStream << "\t\t";
    std::string indentString = indentStream.str();
    const char* indent = indentString.c_str();

    //
    // Automatically collapse "Advanced" group.
    //
    int collapseGroup = 0;
    if ( hkString::strCasecmp(group->m_groupLabel, "Advanced") == 0 )
    {
        collapseGroup = 1;
    }

    if ( group->m_visuallyEmpty )
    {
        streamWriter.printf("%s// Invisible Group: %s\n", indent, group->m_groupLabel);
    }
    else
    {
        streamWriter.printf("%seditorTemplate -beginLayout \"%s\" -collapse %d;\n", indent, group->m_groupLabel, collapseGroup);
    }

    createAttributesFromParamGroupRecursively(hideCriteria, group, indentStream, streamWriter);

    if ( group->m_visuallyEmpty )
    {
        streamWriter.printf("%s//\n", indent);
    }
    else
    {
        streamWriter.printf("%seditorTemplate -endLayout;\n", indent);
    }
}


hkReal hctGenericNodeHelper::getDistanceUnitConversionFactor()
{
    MDistance internalUnitToMeterConverter;
    internalUnitToMeterConverter.setUnit(MDistance::internalUnit());
    internalUnitToMeterConverter.setValue(1.0f);
    hkReal unitConversionFactor = hkReal(internalUnitToMeterConverter.asMeters());
    return 1.0f / unitConversionFactor;
}


hkReal hctGenericNodeHelper::getAngleUnitConversionFactor()
{
    MAngle internalUnitToDegreesConverter;
    internalUnitToDegreesConverter.setUnit(MAngle::internalUnit());
    internalUnitToDegreesConverter.setValue(1.0f);
    hkReal unitConversionFactor = hkReal(internalUnitToDegreesConverter.asDegrees());
    return 1.0f / unitConversionFactor;
}


void hctGenericNodeHelper::createAttributesFromParamGroupRecursively(hkAttributeHideCriteria::Types hideCriteria, const hctClassHierarchyUtil::ParamGroup* group, const std::stringstream& indentIn, hkOstream& streamWriter)
{
    // Advance indent.
    std::stringstream indentStream;
    indentStream << indentIn.str() << "\t";
    std::string indentString = indentStream.str();
    const char* indent = indentString.c_str();

    {
        for (int i = 0; i < int(group->m_members.size()); i++)
        {
            const hctClassHierarchyUtil::ParamGroup::ClassAndMember& memberData = group->m_members[i];

            hkReflect::FieldDecl member = memberData.m_member;

            const char* internalName = memberData.m_memberInternalName.c_str();
            const char* label        = memberData.m_memberLabel.c_str();
            const char* defaultValue = memberData.m_defaultValue.c_str();

            const hkGizmoAttribute*     gizmoAttributes     = member.getType()->findAttribute<hkGizmoAttribute>();
            const hkLinkAttribute*      linkAttribute       = member.getType()->findAttribute<hkLinkAttribute>();
            const hkUiAttribute*        uiAttributes        = member.getType()->findAttribute<hkUiAttribute>();

            Attribute attrib(internalName);

            MStatus status = MStatus::kFailure;

            struct Visitor : public hkReflect::TypeVisitor<Visitor, void, const char*, const char*, MStatus&>
            {
                Visitor(Attribute& attrib) : attrib(attrib) {}

                Attribute& attrib;

                void visit(const hkReflect::VoidType*, const char*, const char*, MStatus&) { HK_UNREACHABLE(0x641f25c4, "Void field found" ); }
                void visit(const hkReflect::RecordType*, const char*, const char*, MStatus&) { }

                void visitUuid(const char* internalName, MStatus& status)
                {
                    MFnTypedAttribute typedFn;
                    MFnStringData     stringFn;
                    hkStringBuf strb;   hkUuid::getNil().toString(strb);

                    attrib.m_object = typedFn.create(internalName, internalName, MFnData::kString, stringFn.create(strb.cString()), &status);
                }

                void visit(const hkReflect::IntType* member, const char* internalName, const char* defaultValue, MStatus& status)
                {
                    MFnNumericData::Type valueType;
                    if (member->equals<char>())
                    {
                        valueType = MFnNumericData::kChar;
                    }
                    else
                    {
                        switch(member->getFormat().get())
                        {
                            case hkReflect::Format::OfInt<hkInt8>::Value:
                            case hkReflect::Format::OfInt<hkUint8>::Value:
                            {
                                valueType = MFnNumericData::kByte;
                                break;
                            }
                            case hkReflect::Format::OfInt<hkInt16>::Value:
                            case hkReflect::Format::OfInt<hkUint16>::Value:
                            {
                                valueType = MFnNumericData::kShort;
                                break;
                            }
                            case hkReflect::Format::OfInt<hkInt32>::Value:
                            case hkReflect::Format::OfInt<hkUint32>::Value:
                            {
                                valueType = MFnNumericData::kInt;
                                break;
                            }
                            case hkReflect::Format::OfInt<hkInt64>::Value:
                            case hkReflect::Format::OfInt<hkUint64>::Value:
                            {
                                valueType = MFnNumericData::k2Int;
                                break;
                            }
                            default:
                            {
                                valueType = MFnNumericData::kLong;
                                break;
                            }
                        }
                    }
                    MFnNumericAttribute numFn;
                    attrib.m_object = numFn.create(internalName, internalName, valueType, hkString::atoi(defaultValue), &status);
                    setInt32MinMaxValues(member, numFn);
                }

                void visit(const hkReflect::BoolType*, const char* internalName, const char* defaultValue, MStatus& status)
                {
                    MFnNumericAttribute numFn;
                    attrib.m_object = numFn.create(internalName, internalName, MFnNumericData::kBoolean, hkString::atoi(defaultValue), &status);
                }

                void visit(const hkReflect::FloatType* member, const char* internalName, const char* defaultValue, MStatus& status)
                {
                    const hkSemanticsAttribute* semanticsAttributes = member->findAttribute<hkSemanticsAttribute>();
                    hkReal distanceUnitConversionFactor = getDistanceUnitConversionFactor();
                    hkReal angleUnitConversionFactor = getAngleUnitConversionFactor();

                    if ( semanticsAttributes && semanticsAttributes->m_type == hkSemanticsAttribute::DISTANCE )
                    {
                        MFnUnitAttribute unitFn;
                        attrib.m_object = unitFn.create(internalName, internalName, MFnUnitAttribute::kDistance, hkString::atof(defaultValue)*distanceUnitConversionFactor, &status);
                        setFloatMinMaxValues(member, unitFn, distanceUnitConversionFactor);
                    }
                    else if ( semanticsAttributes && semanticsAttributes->m_type == hkSemanticsAttribute::ANGLE )
                    {
                        MFnUnitAttribute unitFn;
                        attrib.m_object = unitFn.create(internalName, internalName, MFnUnitAttribute::kAngle, hkString::atof(defaultValue)*angleUnitConversionFactor, &status);
                        setFloatMinMaxValues(member, unitFn, angleUnitConversionFactor);
                    }
                    else
                    {
                        MFnNumericAttribute numFn;
                        attrib.m_object = numFn.create(internalName, internalName, MFnNumericData::kFloat, hkString::atof(defaultValue), &status);
                        setFloatMinMaxValues(member, numFn);
                    }
                }

                void visit(const hkReflect::StringType*, const char* internalName, const char* defaultValue, MStatus& status)
                {
                    MFnTypedAttribute typedFn;
                    MFnStringData     stringFn;
                    attrib.m_object = typedFn.create(internalName, internalName, MFnData::kString, stringFn.create(defaultValue), &status);
                }

                void visitVector4(const hkReflect::Type* member, const char* internalName, const char* defaultValue, MStatus& status)
                {
                    hkReal distanceUnitConversionFactor = getDistanceUnitConversionFactor();
                    const hkSemanticsAttribute* semanticsAttributes = member->findAttribute<hkSemanticsAttribute>();

                    const char* posVal1 = hkString::strChr(defaultValue, ',') + 1;
                    const char* posVal2 = hkString::strChr(posVal1, ',') + 1;

                    hkReal defaultValue0 = hkString::atof(defaultValue);
                    hkReal defaultValue1 = hkString::atof(posVal1);
                    hkReal defaultValue2 = hkString::atof(posVal2);

                    MString internalSubname0(internalName); internalSubname0 += "0";
                    MString internalSubname1(internalName); internalSubname1 += "1";
                    MString internalSubname2(internalName); internalSubname2 += "2";

                    MFnNumericAttribute numFn;

                    if ( semanticsAttributes && semanticsAttributes->m_type == hkSemanticsAttribute::POSITION )
                    {
                        MFnUnitAttribute unitFn0;
                        MFnUnitAttribute unitFn1;
                        MFnUnitAttribute unitFn2;
                        MObject x = unitFn0.create(internalSubname0, internalSubname0, MFnUnitAttribute::kDistance);
                        MObject y = unitFn1.create(internalSubname1, internalSubname1, MFnUnitAttribute::kDistance);
                        MObject z = unitFn2.create(internalSubname2, internalSubname2, MFnUnitAttribute::kDistance);
                        attrib.m_object = numFn.create(internalName, internalName, x, y, z, &status);
                        unitFn0.setDefault(defaultValue0*distanceUnitConversionFactor);
                        unitFn1.setDefault(defaultValue1*distanceUnitConversionFactor);
                        unitFn2.setDefault(defaultValue2*distanceUnitConversionFactor);
                        setVectorMinMaxValues(member, numFn, distanceUnitConversionFactor);
                    }
                    else
                    {
                        MFnNumericAttribute subNumFn0;
                        MFnNumericAttribute subNumFn1;
                        MFnNumericAttribute subNumFn2;
                        MObject x = subNumFn0.create(internalSubname0, internalSubname0, MFnNumericData::kFloat);
                        MObject y = subNumFn1.create(internalSubname1, internalSubname1, MFnNumericData::kFloat);
                        MObject z = subNumFn2.create(internalSubname2, internalSubname2, MFnNumericData::kFloat);
                        attrib.m_object = numFn.create(internalName, internalName, x, y, z, &status);
                        subNumFn0.setDefault(defaultValue0);
                        subNumFn1.setDefault(defaultValue1);
                        subNumFn2.setDefault(defaultValue2);
                        setVectorMinMaxValues(member, numFn, 1.0f);
                    }
                }

                void visit(const hkReflect::PointerType* member, const char* internalName, const char* defaultValue, MStatus& status)
                {
                    const hkLinkAttribute*      linkAttribute = member->findAttribute<hkLinkAttribute>();
                    if (!linkAttribute || linkAttribute->m_type == hkLinkAttribute::NONE)
                    {
                        return;
                    }

                    MFnMessageAttribute messageFn;
                    attrib.m_object = messageFn.create(internalName, internalName, &status);
                }

                void visitEnum(const hk::Presets* presets, const char* internalName, const char* defaultValue, MStatus& status)
                {
                    MFnEnumAttribute enumFn;
                    attrib.m_object = enumFn.create(internalName, internalName, 0, &status);

                    int indexOfDefaultValue = hkString::atoi(defaultValue); // 'defaultValue' holds the index(!) into the enum array and NOT the value of the default!

                    {
                        for (int i = 0; i < presets->getNumPresets(); i++)
                        {
                            short value = hkReflect::IntVar(presets->getPreset(i)).getValue().convertTo<short>();
                            enumFn.addField(presets->getPresetName(i), value);
                            if ( i == indexOfDefaultValue )
                            {
                                enumFn.setDefault(value);
                            }
                        }
                    }
                    enumFn.setKeyable(true);
                }

                void visit(const hkReflect::ArrayType* type, const char* internalName, const char* defaultValue, MStatus& status) {}
            } visitor(attrib);

            if (member.getType()->equals<hkUuid>())
            {
                visitor.visitUuid(internalName, status);
            }
            else if (member.getType()->equals<hkVector4>())
            {
                visitor.visitVector4(member.getType(), internalName, defaultValue, status);
            }
            else if (const hk::Presets* presets = member.getType()->findAttribute<hk::Presets>())
            {
                if (member.getType()->asInteger())
                {
                    visitor.visitEnum(presets, internalName, defaultValue, status);
                }
            }
            else
            {
                visitor.dispatch(member.getType(), internalName, defaultValue, status);
            }

            if ( !status )
            {
                MString errorText = "Couldn't create Attribute for ";
                errorText += memberData.m_class->getName();
                errorText += "::m_";
                errorText += internalName;
                errorText += ".";
                MGlobal::displayError(errorText);
                continue;
            }

            //
            // Add attribute to Maya node.
            //
            m_attributes.push_back(attrib);
            MPxNode::addAttribute( attrib.m_object );

            //
            // Create gizmo if specified.
            //
            if ( gizmoAttributes )
            {
                // add gizmo to draw
                Gizmo gizmo(*gizmoAttributes, attrib);
                m_gizmos.push_back(gizmo);
            }

            //
            // Display or suppress display of attribute.
            //
            const bool bHidden  = uiAttributes && (uiAttributes->m_hideCriteria & hideCriteria);
            const bool bVisible = memberData.m_visible && !bHidden;
            if ( bVisible )
            {
                const hkReflect::Type* memberType = member.getType();
                bool bIsVector = memberType->equals<hkVector4>();

                if ( (memberType->asPointer() || memberType->asArray()) && !bIsVector )
                {
                    streamWriter.printf("%seditorTemplate -callCustom \"AEhctGenericNode_linkCreate\" \"AEhctGenericNode_linkReplace\" \"%s\" \"%s\";\n", indent, internalName, label);
                    streamWriter.printf("%seditorTemplate -suppress \"%s\";\n", indent, internalName);
                }
                else if ( memberType->asString() && linkAttribute && linkAttribute->m_type == hkLinkAttribute::IMGSELECT )
                {
                    streamWriter.printf("%seditorTemplate -callCustom \"AEhctGenericNode_imgPickerCreate\" \"AEhctGenericNode_imgPickerReplace\" \"%s\" \"%s\";\n", indent, internalName, label);
                    streamWriter.printf("%seditorTemplate -suppress \"%s\";\n", indent, internalName);
                }
                else
                {
                    streamWriter.printf("%seditorTemplate -label \"%s\" -addControl \"%s\";\n", indent, label, internalName);
                }
            }
            else
            {
                streamWriter.printf("%seditorTemplate -suppress \"%s\";\n", indent, internalName);
            }

        }
    }

    //
    // Recursively add members from whole subtree.
    //
    {
        for (int i = 0; i < int(group->m_subGroups.size()); i++)
        {
            hctClassHierarchyUtil::ParamGroup* subGroup = group->m_subGroups[i];

            if ( subGroup->m_visuallyEmpty )
            {
                streamWriter.printf("%s// Invisible Group: %s\n", indent, subGroup->m_groupLabel);
            }
            else
            {
                streamWriter.printf("%seditorTemplate -beginLayout \"%s\" -collapse 0;\n", indent, subGroup->m_groupLabel);
            }

            createAttributesFromParamGroupRecursively(hideCriteria, subGroup, indentStream, streamWriter);

            if ( group->m_visuallyEmpty )
            {
                streamWriter.printf("%s//\n", indent);
            }
            else
            {
                streamWriter.printf("%seditorTemplate -endLayout;\n", indent);
            }
        }
    }
}


void hctGenericNodeHelper::setInt32MinMaxValues(const hkReflect::Type* member, MFnNumericAttribute& numFn)
{
    if (const hkRangeInt32Attribute* ranges = member->findAttribute<hkRangeInt32Attribute>())
    {
        numFn.setMin    (ranges->m_absmin);
        numFn.setMax    (ranges->m_absmax);
        numFn.setSoftMin(ranges->m_softmin);
        numFn.setSoftMax(ranges->m_softmax);
    }
}


void hctGenericNodeHelper::setFloatMinMaxValues(const hkReflect::Type* member, MFnNumericAttribute& numFn)
{
    if (const hkRangeRealAttribute* ranges = member->findAttribute<hkRangeRealAttribute>())
    {
        numFn.setMin(ranges->m_absmin);
        numFn.setMax(ranges->m_absmax);
        numFn.setSoftMin(ranges->m_softmin);
        numFn.setSoftMax(ranges->m_softmax);
    }
}


void hctGenericNodeHelper::setFloatMinMaxValues(const hkReflect::Type* member, MFnUnitAttribute& unitFn, hkReal unitConversionFactor)
{
    if (const hkRangeRealAttribute* ranges = member->findAttribute<hkRangeRealAttribute>())
    {
        unitFn.setMin(ranges->m_absmin*unitConversionFactor);
        unitFn.setMax(ranges->m_absmax*unitConversionFactor);
        unitFn.setSoftMin(ranges->m_softmin*unitConversionFactor);
        unitFn.setSoftMax(ranges->m_softmax*unitConversionFactor);
    }
}


void hctGenericNodeHelper::setVectorMinMaxValues(const hkReflect::Type* member, MFnNumericAttribute& numFn, hkReal unitConversionFactor)
{
    if (const hkRangeRealAttribute* ranges = member->findAttribute<hkRangeRealAttribute>())
    {
        numFn.setMin(ranges->m_absmin*unitConversionFactor, ranges->m_absmin*unitConversionFactor, ranges->m_absmin*unitConversionFactor);
        numFn.setMax(ranges->m_absmax*unitConversionFactor, ranges->m_absmax*unitConversionFactor, ranges->m_absmax*unitConversionFactor);
        numFn.setSoftMin(ranges->m_softmin*unitConversionFactor);
        numFn.setSoftMax(ranges->m_softmax*unitConversionFactor);
    }
}


void hctGenericNodeHelper::createUnderReviewNotificationGroup(const hkReflect::Type* klass, hkOstream& streamWriter)
{
    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();

    streamWriter.printf("\t\teditorTemplate -beginLayout \"Attention\" -collapse 0;\n");
    streamWriter.printf("\t\t\teditorTemplate -callCustom \"AE%sTemplate_createUnderReviewNote\" \"AE%sTemplate_replaceUnderReviewNote\";\n", modelerNodeIdToClassMap[m_classNumber].m_nodeName, modelerNodeIdToClassMap[m_classNumber].m_nodeName);
    streamWriter.printf("\t\teditorTemplate -endLayout;\n");
}


void hctGenericNodeHelper::createUnderReviewNotificationText(hkOstream& streamWriter)
{
    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();

    streamWriter.printf("global proc AE%sTemplate_createUnderReviewNote()\n", modelerNodeIdToClassMap[m_classNumber].m_nodeName);
    streamWriter.printf("{\n");

    streamWriter.printf("\tsetUITemplate -pst NONE;\n");
    streamWriter.printf("\trowLayout -nc 1 -adj 1 -cl1 center;\n");

    streamWriter.printf("\ttext\n");
    streamWriter.printf("\t\t-l \"\\nUNDER REVIEW\\n\\nPlease contact Havok Support\\n(http://support.havok.com)\\nif you are actively using this feature.\\n\";\n");

    streamWriter.printf("\tsetUITemplate -ppt;\n");

    streamWriter.printf("}\n");

    streamWriter.printf("\n\n");

    streamWriter.printf("global proc AE%sTemplate_replaceUnderReviewNote()\n", modelerNodeIdToClassMap[m_classNumber].m_nodeName);
    streamWriter.printf("{\n");
    streamWriter.printf("}\n");
}


void hctGenericNodeHelper::createHelpGroup(const hkReflect::Type* klass, hkOstream& streamWriter)
{
    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();

    streamWriter.printf("\t\teditorTemplate -beginLayout \"Help\" -collapse 0;\n");
    streamWriter.printf("\t\t\teditorTemplate -callCustom \"AE%sTemplate_createHelpButton\" \"AE%sTemplate_replaceHelpButton\";\n", modelerNodeIdToClassMap[m_classNumber].m_nodeName, modelerNodeIdToClassMap[m_classNumber].m_nodeName);
    streamWriter.printf("\t\teditorTemplate -endLayout;\n");
}


void hctGenericNodeHelper::createHelpButton(const hkReflect::Type* klass, hkOstream& streamWriter)
{
    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();
    const char* className = modelerNodeIdToClassMap[m_classNumber].m_class->getName();

    streamWriter.printf("global proc AE%sTemplate_createHelpButton()\n", modelerNodeIdToClassMap[m_classNumber].m_nodeName);
    streamWriter.printf("{\n");

        streamWriter.printf("\tsetUITemplate -pst attributeEditorTemplate;\n");
        streamWriter.printf("\trowLayout -nc 1 -cl1 center;\n");
        streamWriter.printf("\tbutton\n");

            streamWriter.printf("\t\t-l \"Open Documentation\"\n");
            streamWriter.printf("\t\t-ann \"Open the Documentation for %s.\"\n", className);
            streamWriter.printf("\t\t-h 21 -w 150\n");
            streamWriter.printf("\t\t-c \"hkCmdOpenChmDocs \\\"%s\\\"\"\n", className);
            streamWriter.printf("\t\t\"helpButton\";\n");

        streamWriter.printf("\tsetUITemplate -ppt;\n");

    streamWriter.printf("}\n");

    streamWriter.printf("\n\n");

    streamWriter.printf("global proc AE%sTemplate_replaceHelpButton()\n", modelerNodeIdToClassMap[m_classNumber].m_nodeName);
    streamWriter.printf("{\n");
    streamWriter.printf("}\n");
}


MStatus hctGenericNodeHelper::shouldSave(const MPlug& plug, bool& forceSaving)
{
#if 0 // Disabled in order to fix HKD-576.

    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();
    const hkReflect::Type* klass = modelerNodeIdToClassMap[m_classNumber].m_class;

    MString mName = plug.partialName();
    const char* attributeName = mName.asChar();

    // Only force saving of attribute
    // o if it represents a member of our Destruction class OR
    // o if it is the class's version number OR
    // o if it is the node's internal 'Visibility' flag.
    if ( hctMayaUtilities::isValidMember(klass, attributeName) ||
        (hkString::strCmp(attributeName, "internalClassVersion") == 0)  ||
        (hkString::strCmp(attributeName, "v") == 0) )
    {
        forceSaving = true;
    }

#else

    // Seems like there's no straightforward way to figure out if an attribute is either a custom attribute
    // (see HKD-576) or if it has changed from its default value. Hence we will force *all* attributes for
    // a generic node to be saved.
    forceSaving = true;

#endif

    return MStatus::kSuccess;
}


void hctGenericNodeHelper::recursivelyDrawLinkedNodes( MObject node, M3dView& view, const MDagPath& dagPath, M3dView::DisplayStyle style, int numRecursionLeft )
{
    for (int a = 0; a < (int)m_attributes.size(); ++a)
    {
        MObject attrib = m_attributes[a].m_object;
        MPlug attributePlug( node, attrib );

        MStatus status;
        MFnMessageAttribute messageFn(attrib, &status);
        if (MStatus::kSuccess != status)
        {
#if 0
            char tmp[128];
            hkString::sprintf(tmp,"not a message attrib %s", m_attributes[a]->m_name);
            MGlobal::displayInfo(tmp);
#endif
            continue;
        }

        MPlugArray plugArray;
        attributePlug.connectedTo(plugArray, true, false, &status);
        if (MStatus::kSuccess != status)
        {
#if defined(HK_DEBUG)
            char tmp[128];
            hkString::sprintf(tmp,"error link is not destination?");
            MGlobal::displayInfo(tmp);
#endif
            continue;
        }

        if( plugArray.length() == 1)
        {
            MFnDependencyNode nodeFn( plugArray[0].node() );
            MString typeStr = nodeFn.typeName();
#if 0
            char tmp[128];
            hkString::sprintf(tmp,"recursivelyDraw recursing into %s", typeStr.asChar());
            MGlobal::displayInfo(tmp);
#endif

            const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();

            int classIndex = -1;
            for (int i = 0; i < modelerNodeIdToClassMap.getSize(); ++i)
            {
                const char* name  = modelerNodeIdToClassMap[i].m_nodeName;
                if ( !name )
                {
                    break;
                }
                if (hkString::strCmp(name, typeStr.asChar()) == 0)
                {
                    classIndex = i;
                    break;
                }
            }

            if (classIndex >= 0)
            {
                // note that this passes the path to the top-level object to the draw call to allow using its properties
                hctMayaPhysicsDestructionUtilities::m_nodeHelper[classIndex].draw(plugArray[0].node(), view, dagPath, style, M3dView::kLead, numRecursionLeft);
            }
        }
    }

}

void hctGenericNodeHelper::draw( MObject node, M3dView& view, const MDagPath& dagPath, M3dView::DisplayStyle style, M3dView::DisplayStatus displayStatus, int numRecursionLeft )
{
    // Begin OpenGL
    view.beginGL();
    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_POLYGON_BIT | GL_LIGHTING_BIT | GL_DEPTH_BUFFER_BIT );

    // Set the main OpenGL light position to the world camera position,
    // in case we want to use lighting in derived constraints
    glPushMatrix();
    glLoadIdentity();
    MDagPath cameraPath;
    view.getCamera( cameraPath );
    MFnTransform transformFn( cameraPath.node() );

    MStatus status;
    MVector camPosition = transformFn.getTranslation( MSpace::kWorld, &status );
    if (MStatus::kSuccess != status)
    {
        return;
    }

    GLfloat lightPosition[] = { (float)camPosition.x, (float)camPosition.y, (float)camPosition.z, 1.0f };
    glLightfv( GL_LIGHT0, GL_POSITION, lightPosition );
    glPopMatrix();

    // Set up the material properties
    glEnable( GL_COLOR_MATERIAL );
    glColorMaterial( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE );
    GLfloat mat_specular[] = { 0.5f, 0.5f, 0.5f, 1.0f };
    GLfloat mat_emission[] = { 0.0f, 0.0f, 0.0f, 0.0f };
    GLfloat mat_shininess[] = { 5.0f };
    glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular );
    glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, mat_emission );
    glMaterialfv( GL_FRONT_AND_BACK, GL_SHININESS, mat_shininess );

    // Make sure we have the correct depth test function
    // Maya seems to use different ones depending on selection
    glDepthFunc( GL_LESS );

    // Draw the solid version
    if (m_showGizmos && (displayStatus == M3dView::kLead)) drawStoredGizmos( node, view, dagPath, style == M3dView::kWireFrame );

    glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
    glEnable( GL_DEPTH_TEST );

    // Draw annotation
    const hkArray<hctModelerNodeIdToClassMap>& modelerNodeIdToClassMap = hctSdkUtils::getNodeIdToClassMap();
    MTypeId id = modelerNodeIdToClassMap[m_classNumber].m_mayaNodeId;
    if (id == hkNodeBreakableBodyID)
    {
        view.setDrawColor( hctUtilities::getColor( "BreakableBodyLabel" ) );
        view.drawText( "     BB", MPoint::origin, M3dView::kLeft );
    }

    // End OpenGL
    glPopAttrib();
    view.endGL();

    if (displayStatus == M3dView::kLead)
    {
        // check dependent nodes
        if ( numRecursionLeft > 0 )
        {
            recursivelyDrawLinkedNodes( node, view, dagPath, style, numRecursionLeft-1 );
        }
        else
        {
            char tmp[128];
            hkString::sprintf(tmp, "Recursion error. Please check your Destruction links for any cyclic linkage.");
            MGlobal::displayError(tmp);
        }
    }

}

void hctGenericNodeHelper::drawStoredGizmos( MObject node, M3dView& view, const MDagPath& dagPath, bool wireframe ) const
{
    MTransformationMatrix nodeToWorld = MTransformationMatrix::identity;

    for (int g = 0; g < (int)m_gizmos.size(); ++g)
    {
        const hkGizmoAttribute& gizmo = m_gizmos[g].m_gizmo;

        if (! gizmo.m_visible)
        {
            continue;
        }

#if 0
        char tmp[128];
        hkString::sprintf(tmp,"drawStoredGizmos drawing gizmo %s", gizmo.m_label);
        MGlobal::displayInfo(tmp);
#endif

        const MObject& attrib = m_gizmos[g].m_attribute.m_object;
        MPlug attributePlug( node, attrib );

        MStatus status;
        MFnNumericAttribute numericAttribute(attrib, &status);
        if (MStatus::kSuccess != status)
        {
#if defined(HK_DEBUG)
            char tmp[128];
            hkString::sprintf(tmp,"cannot obtain gizmo value");
            MGlobal::displayInfo(tmp);
#endif
            continue;
        }

        switch (gizmo.m_type)
        {
            case hkGizmoAttribute::PLANE:
                {
                    HK_ASSERT_NO_MSG(0x78746bc2, numericAttribute.unitType() == MFnNumericData::k3Float );
                    HK_ASSERT_NO_MSG(0x7042ce7e, attributePlug.numChildren() == 3 );

                    MPlug xPlug = attributePlug.child(0);
                    MPlug yPlug = attributePlug.child(1);
                    MPlug zPlug = attributePlug.child(2);

                    // rotate to the given normal
                    hkVector4 normal;
                    xPlug.getValue(normal(0));
                    yPlug.getValue(normal(1));
                    zPlug.getValue(normal(2));

                    // normal zero means automatic plane orientation, don't draw it
                    if (normal.lengthSquared3() < HK_REAL_EPSILON)
                    {
                        break;
                    }

                    normal.normalize3();
                    hkVector4 defaultAxis;
                    defaultAxis.set(0,0,1);
                    hkQuaternion q;
                    hkQuaternionUtil::_computeShortestRotation(defaultAxis, normal, q);
                    MQuaternion planeNormalRotation(q(0), q(1), q(2), q(3));
                    nodeToWorld.rotateTo(planeNormalRotation);

                    {
                        MStatus pivotStatus;
                        MFnTransform pivotTransformFn( dagPath, &pivotStatus );
                        if( pivotStatus == MStatus::kSuccess )
                        {
                            MPoint offset = pivotTransformFn.rotatePivot( MSpace::kPreTransform );
                            MVector floatOffset( offset );
                            nodeToWorld.addTranslation(floatOffset, MSpace::kPostTransform);
                        }
                    }


                    // scale to 110% of the bounding box size of the target object
                    MFnDagNode dagFn( dagPath );
                    MBoundingBox bb = dagFn.boundingBox();
                    bb.transformUsing( dagPath.exclusiveMatrix() );

                    double scale[3];
                    float f = 0.55f;
                    scale[0] = bb.width()  * f;
                    scale[1] = bb.height() * f; scale[1] = hkMath::max2(scale[0], scale[1]);
                    scale[2] = bb.depth()  * f; scale[2] = hkMath::max2(scale[1], scale[2]);
                    scale[0] = scale[1] = scale[2];
                    if (scale[0] > 0.0) // non locator nodes do not have a bbox
                    {
                        nodeToWorld.addScale(scale, MSpace::kPostTransform);
                    }

#if 0
                    char tmp[128];
                    MString name = numericAttribute.name();
                    hkString::sprintf(tmp,"plane %s normal (%f, %f, %f)", name.asChar(), axis(0), axis(1), axis(2));
                    MGlobal::displayInfo(tmp);
#endif

                    hctSimpleGizmos::drawPlane(nodeToWorld, wireframe);
                }
                break;

            case hkGizmoAttribute::SPHERE:
                {
                    HK_ASSERT_NO_MSG(0x348d868b, numericAttribute.unitType() == MFnNumericData::kFloat );

                    hkReal radius = 1.0f;

                    MPlug attributePlugSphere( node, attrib );
                    attributePlugSphere.getValue( radius );

#if 0
                    char tmp[128];
                    MString name = numericAttribute.name();
                    hkString::sprintf(tmp,"sphere %s radius %f", name.asChar(), radius);
                    MGlobal::displayInfo(tmp);
#endif

                    hctSimpleGizmos::drawSphere(nodeToWorld, radius, wireframe);
                }
                break;

            case hkGizmoAttribute::ARROW:
                {
                    HK_ASSERT_NO_MSG(0x3eaf66dd, numericAttribute.unitType() == MFnNumericData::k3Float );
                    HK_ASSERT_NO_MSG(0xb559311, attributePlug.numChildren() == 3 );

                    MPlug xPlug = attributePlug.child(0);
                    MPlug yPlug = attributePlug.child(1);
                    MPlug zPlug = attributePlug.child(2);

                    // rotate to the given axis
                    hkVector4 axis;
                    xPlug.getValue(axis(0));
                    yPlug.getValue(axis(1));
                    zPlug.getValue(axis(2));

                    axis.normalize3();
                    hkVector4 defaultAxis;
                    defaultAxis.set(0,0,1);
                    hkQuaternion q;
                    hkQuaternionUtil::_computeShortestRotation(defaultAxis, axis, q);
                    MQuaternion planeNormalRotation(q(0), q(1), q(2), q(3));
                    nodeToWorld.rotateTo(planeNormalRotation);

                    {
                        MStatus pivotStatus;
                        MFnTransform pivotTransformFn( dagPath, &pivotStatus );
                        if( pivotStatus == MStatus::kSuccess )
                        {
                            MPoint offset = pivotTransformFn.rotatePivot( MSpace::kPreTransform );
                            MVector floatOffset( offset );
                            nodeToWorld.addTranslation(floatOffset, MSpace::kPostTransform);
                        }
                    }

                    // scale to 70% of the bounding box size of the target object
                    MFnDagNode dagFn( dagPath );
                    MBoundingBox bb = dagFn.boundingBox();
                    bb.transformUsing( dagPath.exclusiveMatrix() );

                    double scale[3];
                    double s = 0.7;
                    scale[0] = bb.width()  * s;
                    scale[1] = bb.height() * s; scale[1] = hkMath::max2(scale[0], scale[1]);
                    scale[2] = bb.depth()  * s;  scale[2] = hkMath::max2(scale[1], scale[2]);
                    scale[0] = scale[1] = scale[2];
                    if (scale[0] > 0.0) // non locator nodes do not have a bbox
                    {
                        nodeToWorld.addScale(scale, MSpace::kPostTransform);
                    }

#if 0
                    char tmp[128];
                    MString name = numericAttribute.name();
                    hkString::sprintf(tmp,"direction %s axis (%f, %f, %f)", name.asChar(), axis(0), axis(1), axis(2));
                    MGlobal::displayInfo(tmp);
#endif

                    hctSimpleGizmos::drawArrow(nodeToWorld, wireframe);
                }
                break;

            default:
                {
#if defined(HK_DEBUG)
                    char tmp[128];
                    hkString::sprintf(tmp,"gizmo not implemented");
                    MGlobal::displayInfo(tmp);
#endif
                }
                break;
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
