// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Constraints/hctConstraintNode.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

// Statics
MObject hctConstraintNode::m_hkType;
MObject hctConstraintNode::m_parentNode;
MObject hctConstraintNode::m_parentNodeLast;
MObject hctConstraintNode::m_childTranslationLocked;
MObject hctConstraintNode::m_childTranslation;
MObject hctConstraintNode::m_childRotationLocked;
MObject hctConstraintNode::m_childRotation;
MObject hctConstraintNode::m_parentTranslationLocked;
MObject hctConstraintNode::m_parentTranslation;
MObject hctConstraintNode::m_parentRotationLocked;
MObject hctConstraintNode::m_parentRotation;
MObject hctConstraintNode::m_isBreakable;
MObject hctConstraintNode::m_breakThreshold;

hctLocatorNode::DisplayType hctConstraintNode::m_displayType = hctLocatorNode::DISPLAY_TYPE_OBJECT;


MStatus hctConstraintNode::initialize( MString hkType, MTypeId typeId, bool hasRotations )
{
    MStatus status = MStatus::kSuccess;

    MFnTypedAttribute   typedFn;
    MFnMessageAttribute messageFn;
    MFnNumericAttribute numFn;
    MFnMatrixAttribute  matrixFn;
    MFnUnitAttribute    unitFn;

    // hkType
    {
        MFnStringData stringFn;
        m_hkType = typedFn.create( "hkType", "hkt", MFnData::kString, stringFn.create( hkType ), &status );
        if( !status ) MGlobal::displayError( "Couldn't create hkType attr" );
        typedFn.setWritable( false );
        typedFn.setHidden( true );
        addAttribute( m_hkType );
    }

    // Parent node (if any)
    {
        m_parentNode = messageFn.create( "parent", "par", &status );
        if( !status ) MGlobal::displayError( "Couldn't create parent message attr" );
        addAttribute( m_parentNode );

        m_parentNodeLast = messageFn.create( "parentLast", "parl", &status );
        if( !status ) MGlobal::displayError( "Couldn't create parent last message attr" );
        messageFn.setHidden( true );
        addAttribute( m_parentNodeLast );
    }

    // Child space translation
    {
        m_childTranslationLocked = numFn.create( "childSpaceTranslationLocked", "cstl", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create child trans locked attr" );
        numFn.setConnectable( false );
        numFn.setHidden( true );
        numFn.setKeyable( true );
        addAttribute( m_childTranslationLocked );

        MObject x = unitFn.create( "childSpaceTranslationX", "cstX", MFnUnitAttribute::kDistance, 0.0);
        MObject y = unitFn.create( "childSpaceTranslationY", "cstY", MFnUnitAttribute::kDistance, 0.0);
        MObject z = unitFn.create( "childSpaceTranslationZ", "cstZ", MFnUnitAttribute::kDistance, 0.0);
        m_childTranslation = numFn.create( "childSpaceTranslation", "cst", x, y, z, &status );
        if( !status ) MGlobal::displayError( "Couldn't create child trans attr" );
        numFn.setKeyable( true );
        addAttribute( m_childTranslation );
    }

    // Child space rotation
    if( hasRotations )
    {
        m_childRotationLocked = numFn.create( "childSpaceRotationLocked", "csrl", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create child trans locked attr" );
        numFn.setConnectable( false );
        numFn.setHidden( true );
        numFn.setKeyable( true );
        addAttribute( m_childRotationLocked );

        MObject x = unitFn.create( "childSpaceRotationX", "csrX", MFnUnitAttribute::kAngle, 0.0);
        MObject y = unitFn.create( "childSpaceRotationY", "csrY", MFnUnitAttribute::kAngle, 0.0);
        MObject z = unitFn.create( "childSpaceRotationZ", "csrZ", MFnUnitAttribute::kAngle, 0.0);
        m_childRotation = numFn.create( "childSpaceRotation", "csr", x, y, z, &status );
        if( !status ) MGlobal::displayError( "Couldn't create child rot attr" );
        numFn.setKeyable( true );
        addAttribute( m_childRotation );
    }

    // Parent space translation
    {
        m_parentTranslationLocked = numFn.create( "parentSpaceTranslationLocked", "pstl", MFnNumericData::kBoolean, 1, &status );
        if( !status ) MGlobal::displayError( "Couldn't create parent trans locked attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_parentTranslationLocked );

        MObject x = unitFn.create( "parentSpaceTranslationX", "pstX", MFnUnitAttribute::kDistance, 0.0);
        MObject y = unitFn.create( "parentSpaceTranslationY", "pstY", MFnUnitAttribute::kDistance, 0.0);
        MObject z = unitFn.create( "parentSpaceTranslationZ", "pstZ", MFnUnitAttribute::kDistance, 0.0);
        m_parentTranslation = numFn.create( "parentSpaceTranslation", "pst", x, y, z, &status );
        if( !status ) MGlobal::displayError( "Couldn't create parent trans attr" );
        numFn.setKeyable( true );
        addAttribute( m_parentTranslation );
    }

    // Parent space rotation
    if( hasRotations )
    {
        m_parentRotationLocked = numFn.create( "parentSpaceRotationLocked", "psrl", MFnNumericData::kBoolean, 1, &status );
        if( !status ) MGlobal::displayError( "Couldn't create parent rot locked attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_parentRotationLocked );

        MObject x = unitFn.create( "parentSpaceRotationX", "psrX", MFnUnitAttribute::kAngle, 0.0);
        MObject y = unitFn.create( "parentSpaceRotationY", "psrY", MFnUnitAttribute::kAngle, 0.0);
        MObject z = unitFn.create( "parentSpaceRotationZ", "psrZ", MFnUnitAttribute::kAngle, 0.0);
        m_parentRotation = numFn.create( "parentSpaceRotation", "psr", x, y, z, &status );
        if( !status ) MGlobal::displayError( "Couldn't create parent rot attr" );
        numFn.setKeyable( true );
        addAttribute( m_parentRotation );
    }

    // Breakable
    {
        m_isBreakable = numFn.create( "isBreakable", "isbrk", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create is breakable attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        addAttribute( m_isBreakable );

        m_breakThreshold = numFn.create( "breakThreshold", "brkth", MFnNumericData::kDouble, 0.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create break threshold attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        addAttribute( m_breakThreshold );
    }

    // Allow manipulation of derived node
    status = MPxManipContainer::addToManipConnectTable( const_cast<MTypeId&>( typeId ) );

    return status;
}


MStatus hctConstraintNode::legalConnection( const MPlug& plug, const MPlug& otherPlug, bool asSrc, bool& result ) const
{
    // Check incoming connections only
    if( asSrc )
    {
        return MStatus::kUnknownParameter;
    }

    // The name of the attribute that is being connected to
    const MString attrName = plug.partialName( false, false, false, false, false, true );

    // Check against incoming parent node attributes
    MFnAttribute parentAttr( m_parentNode );
    MFnAttribute parentLastAttr( m_parentNodeLast );
    if( ( attrName == parentAttr.name() ) ||
        ( attrName == parentLastAttr.name() ) )
    {
        // Disallowed until proven innocent
        result = false;

        // Source must be a transform node
        MDagPath sourcePath;
        MDagPath::getAPathTo( otherPlug.node(), sourcePath );
        if( sourcePath.apiType() != MFn::kTransform )
        {
            MGlobal::displayWarning( "Requested constraint parent is not a transform node" );
            return MStatus::kSuccess;
        }

        // It must not be the constraint's direct parent
        MDagPath parentPath;
        MDagPath::getAPathTo( plug.node(), parentPath );
        parentPath.pop();
        if( sourcePath == parentPath )
        {
            MGlobal::displayWarning( "My transform node cannot be my constraint parent" );
            return MStatus::kSuccess;
        }

        // Ok, its valid
        result = true;
        return MStatus::kSuccess;
    }

    // Let Maya deal with it instead
    return MStatus::kUnknownParameter;
}


void hctConstraintNode::draw( M3dView& view, const MDagPath&, M3dView::DisplayStyle, M3dView::DisplayStatus )
{
    // Check if it should be displayed
    MObject node = thisMObject();
    if( !shouldDisplay( node, m_displayType ) )
    {
        return;
    }

    MDagPath nodePath;
    MDagPath::getAPathTo( node, nodePath );

    // Build child/parent to node transforms
    MMatrix childToNode, parentToNode;
    {
        // Get child/parent to world transforms
        MMatrix childToWorld;   getSpaceInWorld( node, SPACE_CHILD, childToWorld );
        MMatrix parentToWorld;  getSpaceInWorld( node, SPACE_PARENT, parentToWorld );

        // Get viewport scaling matrices
        MMatrix childScale = hctUtilities::getViewportScaleMatrix( view, childToWorld, 1.0f, 100 * MFnManip3D::globalSize() );
        m_parentScale = hctUtilities::getViewportScaleMatrix( view, parentToWorld, 1.0f, 100 * MFnManip3D::globalSize() );

        // Combine the matrices
        MMatrix worldToNode = nodePath.inclusiveMatrixInverse();
        childToNode = childScale * childToWorld * worldToNode;
        parentToNode = m_parentScale * parentToWorld * worldToNode;
    }


    // Begin OpenGL
    view.beginGL();
    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_POLYGON_BIT | GL_LIGHTING_BIT | GL_DEPTH_BUFFER_BIT );

    // Set the main OpenGL light position to the world camera position,
    // in case we want to use lighting in derived constraints
    glPushMatrix();
    glLoadIdentity();
    MDagPath cameraPath;
    view.getCamera( cameraPath );
    MFnTransform transformFn( cameraPath.node() );

    MStatus status;
    MVector camPosition = transformFn.getTranslation( MSpace::kWorld, &status );
    if (MStatus::kSuccess != status)
    {
        return;
    }

    GLfloat lightPosition[] = { (float)camPosition.x, (float)camPosition.y, (float)camPosition.z, 1.0f };
    glLightfv( GL_LIGHT0, GL_POSITION, lightPosition );
    glPopMatrix();

    // Set up the material properties
    glEnable( GL_COLOR_MATERIAL );
    glColorMaterial( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE );
    GLfloat mat_specular[] = { 0.5f, 0.5f, 0.5f, 1.0f };
    GLfloat mat_emission[] = { 0.0f, 0.0f, 0.0f, 0.0f };
    GLfloat mat_shininess[] = { 5.0f };
    glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular );
    glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, mat_emission );
    glMaterialfv( GL_FRONT_AND_BACK, GL_SHININESS, mat_shininess );

    // Make sure we have the correct depth test function
    // Maya seems to use different ones depending on selection
    glDepthFunc( GL_LESS );

    // Draw the solid version
    draw( childToNode, parentToNode, false );

    // Draw the wireframe version without lighting or depth testing
    glDisable( GL_DEPTH_TEST );
    glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
    glLineWidth( 1.0 );
    draw( childToNode, parentToNode, true );
    glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
    glEnable( GL_DEPTH_TEST );

    // End OpenGL
    glPopAttrib();
    view.endGL();
}

void hctConstraintNode::draw( const MMatrix& childSpaceToWorld, const MMatrix& parentSpaceToWorld, bool ) const
{
    // Draws a basic constraint representation:
    // Two dots joined by a dotted line

    MVector childPos( childSpaceToWorld[3] );
    MVector parentPos( parentSpaceToWorld[3] );
    MColor childColor( hctUtilities::getColor( "ConstraintChildSpace" ) );
    MColor parentColor( hctUtilities::getColor( "ConstraintParentSpace" ) );

    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_LINE_BIT | GL_POINT_BIT );

    glEnable( GL_POINT_SMOOTH );    glHint( GL_POINT_SMOOTH_HINT, GL_NICEST );
    glEnable( GL_LINE_SMOOTH );     glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );
    glEnable( GL_BLEND );           glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glEnable( GL_LINE_STIPPLE );
    glShadeModel( GL_SMOOTH );

    // No depth testing - so the points are always visible
    glDisable( GL_DEPTH_TEST );

    // Draw a line connecting the points
    childColor.a = 0.5;
    parentColor.a = 0.5;
    glLineWidth( 2.0f );
    glLineStipple( 4, 0xAAAA );
    glBegin( GL_LINES );
    glColor4fv( &parentColor.r );
    glVertex3dv( &parentPos.x );
    glColor4fv( &childColor.r );
    glVertex3dv( &childPos.x );
    glEnd();

    // Draw the points
    // The child is drawn twice so it always appears on top
    childColor.a = 1.0;
    parentColor.a = 1.0;
    glPointSize( 8.0f );
    glBegin( GL_POINTS );
    glColor4fv( &childColor.r );
    glVertex3dv( &childPos.x );
    glColor4fv( &parentColor.r );
    glVertex3dv( &parentPos.x );
    glColor4fv( &childColor.r );
    glVertex3dv( &childPos.x );
    glEnd();

    glPopAttrib();
}


void hctConstraintNode::getParentNode( const MObject& node, MDagPath& parentOut )
{
    MFnDependencyNode nodeFn( node );
    MFnAttribute attrFn( m_parentNode );
    MPlug parentPlug = nodeFn.findPlug( attrFn.name() );
    MPlugArray incomingConnections;
    parentPlug.connectedTo( incomingConnections, true, false );
    if( incomingConnections.length() > 0 )
    {
        MObject srcNode = incomingConnections[0].node();
        MDagPath::getAPathTo( srcNode, parentOut );
    }
    else
    {
        parentOut = MDagPath();     // no parent, ie. world
    }
}

void hctConstraintNode::getSpaceInWorld( const MObject& node, SpaceType space, MMatrix& spaceToWorldOut )
{
    MDagPath path;
    MDagPath::getAPathTo( node, path );
    MFnDependencyNode nodeFn( node );
    MFnAttribute attrFn;
    MObject obj;

    // Begin as identity
    spaceToWorldOut = MMatrix::identity;

    switch( space )
    {
    case SPACE_PARENT_NODE:
        {
            // Get pivot->world transform of parent node - it remains as the world transform otherwise
            MDagPath parentPath;
            getParentNode( node, parentPath );
            if( parentPath.isValid() )
            {
                hctUtilities::getOptionalPivotWorldTransform( parentPath, spaceToWorldOut );
            }
        }
        break;

    case SPACE_CHILD:
        {
            MFnNumericData numFn;

            // Build child space transform
            MTransformationMatrix childToPivot;
            {
                attrFn.setObject( m_childTranslation );
                MPlug transPlug = nodeFn.findPlug( attrFn.name() );
                if( !transPlug.isNull() )
                {
                    transPlug.getValue( obj );
                    numFn.setObject( obj );
                    MVector pos;
                    numFn.getData( pos.x, pos.y, pos.z );
                    childToPivot.setTranslation( pos, MSpace::kTransform );
                }

                attrFn.setObject( m_childRotation );
                MPlug rotPlug = nodeFn.findPlug( attrFn.name() );
                if( !rotPlug.isNull() )
                {
                    rotPlug.getValue( obj );
                    numFn.setObject( obj );
                    MEulerRotation rot;
                    numFn.getData( rot.x, rot.y, rot.z );
                    childToPivot.rotateTo( rot );
                }
            }

            // Get pivot->world transform of parent node
            MMatrix pivotToWorld;
            {
                MDagPath parentPath( path );
                parentPath.pop();
                hctUtilities::getOptionalPivotWorldTransform( parentPath, pivotToWorld );
            }

            // Combine the transforms
            spaceToWorldOut = childToPivot.asMatrix() * pivotToWorld;

            // Remove any scale/skew
            hctUtilities::removeScaleAndSkew( spaceToWorldOut, spaceToWorldOut );
        }
        break;

    case SPACE_PARENT:
        {
            MFnNumericData numFn;

            // Get parent node -> world transform
            MMatrix parentNodeToWorld;
            getSpaceInWorld( node, SPACE_PARENT_NODE, parentNodeToWorld );

            // Get child -> world transform
            MMatrix childToWorld;
            getSpaceInWorld( node, SPACE_CHILD, childToWorld );

            // Build translation component, if any
            MVector translation;
            {
                attrFn.setObject( m_parentTranslation );
                MPlug transPlug = nodeFn.findPlug( attrFn.name() );
                attrFn.setObject( m_parentTranslationLocked );
                MPlug lockPlug = nodeFn.findPlug( attrFn.name() );
                if( !transPlug.isNull() && !lockPlug.isNull() )
                {
                    // Read attribute values
                    MVector parentTranslation;
                    transPlug.getValue( obj );
                    numFn.setObject( obj );
                    numFn.getData( parentTranslation.x, parentTranslation.y, parentTranslation.z );

                    // Transform from child/parent node, based on lock state
                    bool locked;
                    lockPlug.getValue( locked );
                    MTransformationMatrix relativeToWorld = locked? childToWorld: parentNodeToWorld;
                    relativeToWorld.addTranslation( parentTranslation, MSpace::kPreTransform );
                    translation = relativeToWorld.translation( MSpace::kPostTransform );
                }
            }

            // Build rotation component, if any
            MQuaternion rotation;
            {
                attrFn.setObject( m_parentRotation );
                MPlug rotPlug =  nodeFn.findPlug( attrFn.name() );
                attrFn.setObject( m_parentRotationLocked );
                MPlug lockPlug = nodeFn.findPlug( attrFn.name() );
                if( !rotPlug.isNull() && !lockPlug.isNull() )
                {
                    // Read attribute values
                    MEulerRotation parentEuler;
                    rotPlug.getValue( obj );
                    numFn.setObject( obj );
                    numFn.getData( parentEuler.x, parentEuler.y, parentEuler.z );

                    // Get parent matrix decomposition, based on lock state
                    bool locked;
                    lockPlug.getValue( locked );
                    const MMatrix& relativeToWorld = locked? childToWorld: parentNodeToWorld;
                    hctUtilities::Decomposition decomp;
                    hctUtilities::decomposeMatrix( relativeToWorld, decomp );

                    // Set product
                    rotation = parentEuler.asQuaternion() * decomp.m_rotation;
                }
            }

            // Build 'out' matrix
            MTransformationMatrix xform;
            xform.setTranslation( translation, MSpace::kTransform );
            xform.rotateTo( rotation );
            spaceToWorldOut = xform.asMatrix();
        }
        break;

    default:
        MGlobal::displayError( "Wrong space in getSpaceInWorld()" );
        break;
    }
}




//
// Constraint manipulators
//

const hctManipContainer::ManipDesc* hctConstraintManip::getManipulatorDesc( int id ) const
{
    static ManipDesc childTransDesc (   ManipDesc::MANIP_TYPE_POINT,    hctConstraintNode::m_childTranslation   );
    static ManipDesc childRotDesc   (   ManipDesc::MANIP_TYPE_ROTATE,   hctConstraintNode::m_childRotation      );
    static ManipDesc parentTransDesc(   ManipDesc::MANIP_TYPE_POINT,    hctConstraintNode::m_parentTranslation  );
    static ManipDesc parentRotDesc  (   ManipDesc::MANIP_TYPE_ROTATE,   hctConstraintNode::m_parentRotation     );

    switch( id )
    {
    case MANIP_CHILD_TRANSLATION:   return &childTransDesc;
    case MANIP_CHILD_ROTATION:      return &childRotDesc;
    case MANIP_PARENT_TRANSLATION:  return &parentTransDesc;
    case MANIP_PARENT_ROTATION:     return &parentRotDesc;
    default:                        return NULL;
    }
}

MMatrix hctConstraintManip::getManipulatorBaseTransform( int id ) const
{
    MMatrix matrix;

    switch( id )
    {
    case MANIP_CHILD_TRANSLATION:
        {
            // Set as transform node's pivot space
            MDagPath path;
            MDagPath::getAPathTo( m_target, path );
            path.pop();
            hctUtilities::getOptionalPivotWorldTransform( path, matrix );
            break;
        }

    case MANIP_CHILD_ROTATION:
        {
            // Set as transform node's pivot space
            matrix = getManipulatorBaseTransform( MANIP_CHILD_TRANSLATION );

            // Premultiply translation offset
            MFnDependencyNode nodeFn( m_target );
            MObject obj;
            MFnAttribute attrFn( hctConstraintNode::m_childTranslation );
            MPlug plug = nodeFn.findPlug( attrFn.name() );
            if( !plug.isNull() )
            {
                plug.getValue( obj );
                MFnNumericData numFn( obj );
                MVector offset;
                numFn.getData( offset.x, offset.y, offset.z );
                MMatrix transMatrix;
                transMatrix(3,0) += offset.x;
                transMatrix(3,1) += offset.y;
                transMatrix(3,2) += offset.z;
                matrix = transMatrix * matrix;
            }
            break;
        }

    case MANIP_PARENT_TRANSLATION:
        {
            MFnDependencyNode nodeFn( m_target );

            // Set as relative's pivot space
            bool locked;
            MFnAttribute attrFn( hctConstraintNode::m_parentTranslationLocked );
            MPlug plug = nodeFn.findPlug( attrFn.name() );
            if( !plug.isNull() )
            {
                plug.getValue( locked );
                if( locked )
                {
                    // Use the virtual child space
                    hctConstraintNode::getSpaceInWorld( m_target, hctConstraintNode::SPACE_CHILD, matrix );
                }
                else
                {
                    // Use the real connected node, if there is one
                    MDagPath parentPath;
                    hctConstraintNode::getParentNode( m_target, parentPath );
                    if( parentPath.isValid() )
                    {
                        hctUtilities::getOptionalPivotWorldTransform( parentPath, matrix );
                    }
                }
            }
            break;
        }

    case MANIP_PARENT_ROTATION:
        {
            MFnDependencyNode nodeFn( m_target );

            // Translation component
            MVector translation;
            {
                // Set as relative's pivot space
                matrix = getManipulatorBaseTransform( MANIP_PARENT_TRANSLATION );

                // Premultiply translation offset
                MVector offset;
                MObject obj;
                MFnAttribute attrFn( hctConstraintNode::m_parentTranslation );
                MPlug plug = nodeFn.findPlug( attrFn.name() );
                if( !plug.isNull() )
                {
                    plug.getValue( obj );
                    MFnNumericData numFn( obj );
                    numFn.getData( offset.x, offset.y, offset.z );

                    MMatrix offsetMatrix;
                    offsetMatrix(3,0) += offset.x;
                    offsetMatrix(3,1) += offset.y;
                    offsetMatrix(3,2) += offset.z;
                    matrix = offsetMatrix * matrix;

                    MTransformationMatrix xform( matrix );
                    translation = xform.translation( MSpace:: kTransform );
                }
            }

            // Rotation component
            MQuaternion rotation;
            {
                // Set as relative's pivot space
                bool locked;
                MFnAttribute attrFn( hctConstraintNode::m_parentRotationLocked );
                MPlug plug = nodeFn.findPlug( attrFn.name() );
                if( !plug.isNull() )
                {
                    plug.getValue( locked );
                    if( locked )
                    {
                        // Use the virtual child space
                        hctConstraintNode::getSpaceInWorld( m_target, hctConstraintNode::SPACE_CHILD, matrix );
                    }
                    else
                    {
                        // Use the real connected node
                        MDagPath parentPath;
                        hctConstraintNode::getParentNode( m_target, parentPath );
                        if( parentPath.isValid() )
                        {
                            hctUtilities::getOptionalPivotWorldTransform( parentPath, matrix );
                        }
                        else
                        {
                            matrix = MMatrix::identity;
                        }
                    }

                    hctUtilities::Decomposition decomp;
                    hctUtilities::decomposeMatrix( matrix, decomp );
                    rotation = decomp.m_rotation;
                }
            }

            // Combine them
            MTransformationMatrix xform;
            xform.setTranslation( translation, MSpace::kTransform );
            xform.rotateTo( rotation );
            matrix = xform.asMatrix();
            break;
        }

    default:
        break;
    }

    return matrix;
}

bool hctConstraintManip::getManipulatorVisibility( int id ) const
{
    // Get lock attribute for this manipulator
    MObject lockAttr;
    switch( id )
    {
    case MANIP_CHILD_TRANSLATION:
        lockAttr = hctConstraintNode::m_childTranslationLocked;
        break;

    case MANIP_CHILD_ROTATION:
        lockAttr = hctConstraintNode::m_childRotationLocked;
        break;

    case MANIP_PARENT_TRANSLATION:
        lockAttr = hctConstraintNode::m_parentTranslationLocked;
        break;

    case MANIP_PARENT_ROTATION:
        lockAttr = hctConstraintNode::m_parentRotationLocked;
        break;

    default:
        return false;
    }

    // Read lock value from the target node
    MFnDependencyNode nodeFn( m_target );
    MFnAttribute attrFn( lockAttr );
    MPlug lockPlug = nodeFn.findPlug( attrFn.name() );
    if( !lockPlug.isNull() )
    {
        bool locked;
        if( lockPlug.getValue( locked ) == MStatus::kSuccess )
        {
            // Must be unlocked to be visible
            return !locked;
        }
    }

    return false;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
