// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Constraints/Wheel/hctWheelConstraintNode.h>

#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>


// Attributes
MObject hctWheelConstraintNode::m_suspensionLimitMin;
MObject hctWheelConstraintNode::m_suspensionLimitMax;
MObject hctWheelConstraintNode::m_suspensionOffset;
MObject hctWheelConstraintNode::m_suspensionStrength;
MObject hctWheelConstraintNode::m_suspensionDamping;

MStatus hctWheelConstraintNode::initialize()
{
    // Initialize the base class
    MStatus status = hctConstraintNode::initialize( "hkWheelConstraint", typeId() );
    if( status != MStatus::kSuccess )
    {
        return status;
    }

    // Add additional attributes
    MFnNumericAttribute numFn;
    MFnUnitAttribute    unitFn;
    MFnEnumAttribute    enumFn;

    m_suspensionLimitMin = unitFn.create( "suspensionLimitMin", "lmin", MFnUnitAttribute::kDistance, 0.0 );
    unitFn.setSoftMin( -10.0f );
    unitFn.setMax( 0.0f );
    unitFn.setKeyable( true );
    addAttribute( m_suspensionLimitMin );

    m_suspensionLimitMax = unitFn.create( "suspensionLimitMax", "lmax", MFnUnitAttribute::kDistance, 0.0 );
    unitFn.setMin( 0.0f );
    unitFn.setSoftMax( 10.0f );
    unitFn.setKeyable( true );
    addAttribute( m_suspensionLimitMax );

    m_suspensionOffset = unitFn.create( "suspensionOffset", "offs", MFnUnitAttribute::kAngle, 0.0 );
    unitFn.setMin( -PI );
    unitFn.setMax( PI );
    unitFn.setKeyable( true );
    addAttribute( m_suspensionOffset );

    m_suspensionStrength = numFn.create( "suspensionStrength", "strg", MFnNumericData::kFloat, 0.0f);
    numFn.setMin(0.0f);
    numFn.setKeyable( true );
    addAttribute(m_suspensionStrength);

    m_suspensionDamping = numFn.create ("suspensionDamping", "damp", MFnNumericData::kFloat, 0.0f);
    numFn.setMin(0.0f);
    numFn.setKeyable( true );
    addAttribute(m_suspensionDamping);

    return status;
}


void hctWheelConstraintNode::draw( const MMatrix& childSpaceToWorld, const MMatrix& parentSpaceToWorld, bool wireframe ) const
{
    // Get attribute values
    float limitMin, limitMax, offset_angle;
    {
        MFnDependencyNode nodeFn( thisMObject() );
        nodeFn.findPlug( m_suspensionLimitMin ).getValue( limitMin );
        nodeFn.findPlug( m_suspensionLimitMax ).getValue( limitMax );
        nodeFn.findPlug( m_suspensionOffset ).getValue( offset_angle );
        offset_angle *= static_cast<float>(180.0/PI);
    }

    // Draw base constraint representation
    hctConstraintNode::draw( childSpaceToWorld, parentSpaceToWorld, wireframe );

    MColor childColor( hctUtilities::getColor( "ConstraintChildSpace" ) );
    MColor parentColor( hctUtilities::getColor( "ConstraintParentSpace" ) );
    MColor volumeColor( hctUtilities::getColor( "ConstraintVolume" ) );

    // Draw additional representation
    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_LINE_BIT | GL_POLYGON_BIT );
    glEnable( GL_LINE_SMOOTH );     glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );
    glEnable( GL_BLEND );           glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glShadeModel( GL_SMOOTH );

    if( !wireframe )
    {
        glEnable( GL_LIGHTING );
        glEnable( GL_CULL_FACE );
        glCullFace( GL_BACK );
        volumeColor.a = 0.5;
    }

    float wheelRadius = 1.0f;
    float wheelHalfWidth = 0.1f;

    // undo overall scale matrix to keep length of suspension volume independent of display size
    MMatrix undoscale = m_parentScale.inverse();
    double scale = m_parentScale.matrix[2][2];
    wheelRadius *= (float)scale;
    wheelHalfWidth *= (float)scale;

    // Draw child space
    glPushMatrix();

    glMultMatrixd( &childSpaceToWorld.matrix[0][0] );
    glMultMatrixd( &undoscale.matrix[0][0] );

    glColor4fv( &childColor.r );
    drawChildSpace(scale);
    drawWheel(wheelHalfWidth, wheelRadius, wireframe);

    glPopMatrix();

    // Draw parent space
    glPushMatrix();

    glMultMatrixd( &parentSpaceToWorld.matrix[0][0] );
    glMultMatrixd( &undoscale.matrix[0][0] );

    glColor4fv( &parentColor.r );
    drawParentSpace(wheelHalfWidth, wheelRadius, scale, offset_angle, wireframe);

    glColor4fv( &volumeColor.r );
    drawSuspensionVolume(limitMin, limitMax, offset_angle, 1.005f*wheelHalfWidth, 1.005f*wheelRadius, wireframe);

    glPopMatrix();

    // Restore original state
    glPopAttrib();
}


void hctWheelConstraintNode::drawChildSpace(double scale)
{
    glLineWidth( 1.0f );
    glPushMatrix();
    drawChildAxis(scale);
    glPopMatrix();
}


void hctWheelConstraintNode::drawChildAxis(double scale)
{
    glPushAttrib( GL_ENABLE_BIT | GL_POLYGON_BIT | GL_LIGHTING );

    glPushMatrix();
    {
        // spinning-axis arrow
        glScaled(scale, scale, scale);
        glRotatef(90.0f, 0.0f, 1.0f, 0.0f);
        hctUtilities::drawCylinder( 0.1f, 0.75f, 16 );
        glTranslatef( 0.0f, 0.0f, 0.75f );
        hctUtilities::drawCone( 0.2f, 0.25f, 16 );
    }
    glPopMatrix();

    glPopAttrib();
}


void hctWheelConstraintNode::drawParentAxes(double scale, double offset_angle, bool wireframe)
{
    glPushAttrib( GL_ENABLE_BIT | GL_POLYGON_BIT | GL_LIGHTING );

    // spinning-axis arrow
    {
        glPushMatrix();
        glScaled(scale, scale, scale);
        glRotatef(90.0f, 0.0f, 1.0f, 0.0f);
        hctUtilities::drawCylinder( 0.1f, 0.75f, 16 );
        glTranslatef( 0.0f, 0.0f, 0.75f );
        hctUtilities::drawCone( 0.2f, 0.25f, 16 );
        glPopMatrix();
    }

    // steering-axis arrow (wireframe only)
    if (wireframe)
    {
        glPushMatrix();

        glScaled(scale, scale, scale);
        hctUtilities::drawCylinder( 0.075f, 0.5f, 16 );
        glTranslatef( 0.0f, 0.0f, 0.5f );
        hctUtilities::drawCone( 0.15f, 0.25f, 16 );

        //draw an annulus around steering axis
        glTranslatef( 0.0f, 0.0f, -0.2f );
        const int numSweepPoints = 16;
        float diskRadius = 0.15f;
        float diskWidth = 0.2f*diskRadius;

        MFloatVectorArray sweepPoints( numSweepPoints+1 );
        MFloatVectorArray sweepNorms( numSweepPoints+1 );

        for( int i=0; i<=numSweepPoints; ++i )
        {
            float rad = (float)i*2.0f*PI/numSweepPoints;
            sweepNorms[i].x = cos(rad);
            sweepNorms[i].y = sin(rad);
            sweepPoints[i].x = diskRadius*cos(rad);
            sweepPoints[i].y = diskRadius*sin(rad);
        }

        // Rim
        {
            glBegin( GL_QUAD_STRIP );
            for( int i=0; i<=numSweepPoints; ++i )
            {
                glNormal3d( sweepNorms[i].x, sweepNorms[i].y, 0.0 );
                glVertex3d( sweepPoints[i].x, sweepPoints[i].y, 0.0 );
                glVertex3d( sweepPoints[i].x, sweepPoints[i].y, -diskWidth );
            }
            glNormal3d( sweepNorms[0].x, sweepNorms[0].y, 0.0 );
            glVertex3d( sweepPoints[0].x, sweepPoints[0].y, 0.0 );
            glVertex3d( sweepPoints[0].x, sweepPoints[0].y, -diskWidth );
            glEnd();
        }

        glPopMatrix();
    }

    // suspension-axis arrow (wireframe only)
    if (wireframe)
    {
        glPushMatrix();
        glRotated(offset_angle, 1.0, 0.0, 0.0);
        glScaled(scale, scale, scale);
        hctUtilities::drawCylinder( 0.075f, 0.5f, 16 );
        glTranslatef( 0.0f, 0.0f, 0.5f );
        hctUtilities::drawCone( 0.15f, 0.25f, 16 );
        glPopMatrix();
    }

    glPopAttrib();
}



void hctWheelConstraintNode::drawParentPlane(float wheelHalfWidth, float wheelRadius, double offset_angle, bool wireframe)
{
    glPushAttrib( GL_ENABLE_BIT | GL_POLYGON_BIT | GL_LIGHTING_BIT | GL_DEPTH_BUFFER_BIT );

    glPushMatrix();

    glRotated(offset_angle, 1.0, 0.0, 0.0);

    // wireframe
    if (wireframe)
    {
        glBegin( GL_LINE_LOOP );
        glVertex3f( -wheelHalfWidth, wheelRadius, 0.0f );
        glVertex3f( -wheelHalfWidth, -wheelRadius, 0.0f );
        glVertex3f( wheelHalfWidth, -wheelRadius, 0.0f );
        glVertex3f( wheelHalfWidth, wheelRadius, 0.0f );
        glEnd();
    }

    // solid
    if (!wireframe)
    {
        glBegin( GL_QUADS );
        glNormal3f( 0.0f, 1.0f, 0.0f );
        glVertex3f( -wheelHalfWidth, wheelRadius, 0.0f );
        glVertex3f( -wheelHalfWidth, -wheelRadius, 0.0f );
        glVertex3f( wheelHalfWidth, -wheelRadius, 0.0f );
        glVertex3f( wheelHalfWidth, wheelRadius, 0.0f );
        glEnd();
    }

    glPopMatrix();

    glPopAttrib();
}

void hctWheelConstraintNode::drawParentSpace(float wheelHalfWidth, float wheelRadius, double scale, double offset_angle, bool wireframe)
{
    glLineWidth( 1.0f );
    drawParentPlane(wheelHalfWidth, wheelRadius, offset_angle, wireframe);
    glPushMatrix();
    glLineWidth( 0.5f );
    drawParentAxes(scale, offset_angle, wireframe);
    glPopMatrix();
}


void hctWheelConstraintNode::drawWheel(float wheelHalfWidth, float wheelRadius, bool wireframe)
{
    // Subdivision settings
    const int numSweepPoints = 32;

    // Generate x & y values of sweep points
    MFloatVectorArray sweepPoints( numSweepPoints+1 );
    MFloatVectorArray sweepNorms( numSweepPoints+1 );
    {
        // Create the points
        for( int i=0; i<=numSweepPoints; ++i )
        {
            float rad = static_cast<float>(i*2.0*PI/numSweepPoints);
            sweepNorms[i].x = cos(rad);
            sweepNorms[i].y = sin(rad);
            sweepPoints[i].x = wheelRadius*cos(rad);
            sweepPoints[i].y = wheelRadius*sin(rad);
        }
    }

    glRotatef(90.0f, 0.0f, 1.0f, 0.0f);

    glPushAttrib( GL_ENABLE_BIT | GL_POLYGON_BIT );

    // Wireframe wheel
    if (wireframe)
    {
        for( int i=0; i<numSweepPoints; ++i )
        {
            glBegin( GL_LINE_LOOP );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, wheelHalfWidth );
            glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y, wheelHalfWidth );
            glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y, -wheelHalfWidth );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, -wheelHalfWidth );
            glEnd();
        }
    }

    // solid wheel
    if (!wireframe)
    {
        // Circles
        {
            glBegin( GL_TRIANGLE_FAN );
            glNormal3d( 0.0, 0.0, 1.0 );
            glVertex3d( 0.0, 0.0, wheelHalfWidth );
            for( int i=0; i<numSweepPoints; ++i )
            {
                glVertex3d( sweepPoints[i].x, sweepPoints[i].y, wheelHalfWidth );
                glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y, wheelHalfWidth );
            }
            glEnd();

            glBegin( GL_TRIANGLE_FAN );
            glNormal3d( 0.0, 0.0, -1.0 );
            glVertex3d( 0.0, 0.0, -wheelHalfWidth );
            for( int i=numSweepPoints-1; i>=0; i-- )
            {
                glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y, -wheelHalfWidth );
                glVertex3d( sweepPoints[i].x, sweepPoints[i].y, -wheelHalfWidth );
            }
            glEnd();

            // Rim
            glBegin( GL_QUAD_STRIP );
            for( int i=0; i<=numSweepPoints; ++i )
            {
                glNormal3d( sweepNorms[i].x, sweepNorms[i].y, 0.0 );
                glVertex3d( sweepPoints[i].x, sweepPoints[i].y, wheelHalfWidth );
                glVertex3d( sweepPoints[i].x, sweepPoints[i].y, -wheelHalfWidth );
            }
            glNormal3d( sweepNorms[0].x, sweepNorms[0].y, 0.0 );
            glVertex3d( sweepPoints[0].x, sweepPoints[0].y, wheelHalfWidth );
            glVertex3d( sweepPoints[0].x, sweepPoints[0].y, -wheelHalfWidth );
            glEnd();
        }

    }

    glPopAttrib();
}

void hctWheelConstraintNode::drawSuspensionVolume( double limitMin, double limitMax, double offset_angle,
                                                  float wheelHalfWidth, float wheelRadius, bool wireframe )
{
    // Subdivision settings
    const int numSweepPoints = 32;

    wheelHalfWidth *= 1.001f;

    // Generate x & y values of sweep points
    MFloatVectorArray sweepPoints( numSweepPoints+1 );
    MFloatVectorArray sweepNorms( numSweepPoints+1 );
    {
        // Create the points
        for( int i=0; i<=numSweepPoints; ++i )
        {
            float rad = static_cast<float>(i*2.0*PI/numSweepPoints);
            sweepNorms[i].x = cos(rad);
            sweepNorms[i].y = sin(rad);
            sweepPoints[i].x = wheelRadius*cos(rad);
            sweepPoints[i].y = wheelRadius*sin(rad);
        }
    }

    glRotatef((float)offset_angle, 1.0f, 0.0f, 0.0f);

    glRotatef(90.0f, 1.0f, 0.0f, 0.0f);
    glRotatef(90.0f, 0.0f, 1.0f, 0.0f);

    glPushAttrib( GL_ENABLE_BIT | GL_POLYGON_BIT );

    // Wireframe unlit non-depth-tested version
    if (wireframe)
    {
        glLineWidth( 1.0f );
        glBegin( GL_LINE_LOOP );
        for( int i=0; i<numSweepPoints/2; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMax, wheelHalfWidth );
            glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y + limitMax, wheelHalfWidth );
        }
        for( int i=numSweepPoints/2; i<numSweepPoints; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMin, wheelHalfWidth );
            glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y + limitMin, wheelHalfWidth );
        }
        glEnd();

        glBegin( GL_LINE_LOOP );
        for( int i=0; i<numSweepPoints/2; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMax, -wheelHalfWidth );
            glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y + limitMax, -wheelHalfWidth );
        }
        for( int i=numSweepPoints/2; i<numSweepPoints; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMin, -wheelHalfWidth );
            glVertex3d( sweepPoints[i+1].x, sweepPoints[i+1].y + limitMin, -wheelHalfWidth );
        }
        glEnd();
    }

    // Solid lit depth-tested version
    if (!wireframe)
    {
        //half-circles
        glBegin( GL_TRIANGLE_FAN );
        glNormal3d( 0.0, 0.0, 1.0 );
        glVertex3d( 0.0, limitMax, wheelHalfWidth );
        for( int i=0; i<=numSweepPoints/2; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMax, wheelHalfWidth );
        }
        glEnd();

        glBegin( GL_TRIANGLE_FAN );
        glNormal3d( 0.0, 0.0, 1.0 );
        glVertex3d( 0.0, limitMin, wheelHalfWidth );
        for( int i=numSweepPoints/2; i<=numSweepPoints; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMin, wheelHalfWidth );
        }
        glEnd();

        glBegin( GL_TRIANGLE_FAN );
        glNormal3d( 0.0, 0.0, -1.0 );
        glVertex3d( 0.0, limitMax, -wheelHalfWidth );
        for( int i=numSweepPoints/2; i>=0; i-- )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMax, -wheelHalfWidth );
        }
        glEnd();

        glBegin( GL_TRIANGLE_FAN );
        glNormal3d( 0.0, 0.0, -1.0 );
        glVertex3d( 0.0, limitMin, -wheelHalfWidth );
        for( int i=numSweepPoints; i>=numSweepPoints/2; i-- )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMin, -wheelHalfWidth );
        }
        glEnd();

        // side panels
        glBegin( GL_QUADS );
        glNormal3d( 0.0, 0.0, 1.0 );
        glVertex3d( sweepPoints[0].x, sweepPoints[0].y + limitMax, wheelHalfWidth );
        glVertex3d( sweepPoints[numSweepPoints/2].x, sweepPoints[numSweepPoints/2].y + limitMax, wheelHalfWidth );
        glVertex3d( sweepPoints[numSweepPoints/2].x, sweepPoints[numSweepPoints/2].y + limitMin, wheelHalfWidth );
        glVertex3d( sweepPoints[numSweepPoints].x, sweepPoints[numSweepPoints].y + limitMin, wheelHalfWidth );
        glEnd();

        glBegin( GL_QUADS );
        glNormal3d( 0.0, 0.0, -1.0 );
        glVertex3d( sweepPoints[numSweepPoints].x, sweepPoints[numSweepPoints].y + limitMin, -wheelHalfWidth );
        glVertex3d( sweepPoints[numSweepPoints/2].x, sweepPoints[numSweepPoints/2].y + limitMin, -wheelHalfWidth );
        glVertex3d( sweepPoints[numSweepPoints/2].x, sweepPoints[numSweepPoints/2].y + limitMax, -wheelHalfWidth );
        glVertex3d( sweepPoints[0].x, sweepPoints[0].y + limitMax, -wheelHalfWidth );
        glEnd();

        // Rim
        glBegin( GL_QUAD_STRIP );
        for( int i=0; i<=numSweepPoints/2; ++i )
        {
            glNormal3d( sweepNorms[i].x, sweepNorms[i].y, 0.0 );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMax, wheelHalfWidth );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMax, -wheelHalfWidth );
        }
        for( int i=numSweepPoints/2; i<=numSweepPoints; ++i )
        {
            glNormal3d( sweepNorms[i].x, sweepNorms[i].y, 0.0 );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMin, wheelHalfWidth );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y + limitMin, -wheelHalfWidth );
        }
        glNormal3d( sweepNorms[0].x, sweepNorms[0].y, 0.0 );
        glVertex3d( sweepPoints[0].x, sweepPoints[0].y + limitMax, wheelHalfWidth );
        glVertex3d( sweepPoints[0].x, sweepPoints[0].y + limitMax, -wheelHalfWidth );
        glEnd();
    }

    glPopAttrib();
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
