// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Constraints/Ragdoll/hctRagDollConstraintNode.h>

#include <ContentTools/Maya/MayaSceneExport/Nodes/Constraints/Hinge/hctHingeConstraintNode.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>


// Attributes
MObject hctRagDollConstraintNode::m_coneAngle;
MObject hctRagDollConstraintNode::m_planeAngleMax;
MObject hctRagDollConstraintNode::m_planeAngleMin;
MObject hctRagDollConstraintNode::m_twistMin;
MObject hctRagDollConstraintNode::m_twistMax;
MObject hctRagDollConstraintNode::m_maxFrictionTorque;
MObject hctRagDollConstraintNode::m_motorType;
MObject hctRagDollConstraintNode::m_displayCones;
MObject hctRagDollConstraintNode::m_displayTwist;


MStatus hctRagDollConstraintNode::initialize()
{
    // Initialize the base class
    MStatus status = hctConstraintNode::initialize( "hkRagdollConstraint", typeId() );
    if( status != MStatus::kSuccess )
    {
        return status;
    }

    // Add additional attributes
    MFnUnitAttribute unitFn;
    MFnNumericAttribute numFn;
    MFnEnumAttribute    enumFn;

    m_coneAngle = unitFn.create( "coneAngle", "cone", MFnUnitAttribute::kAngle, PI*0.25f );
    unitFn.setMin( 0.0f );
    unitFn.setMax( PI );
    unitFn.setKeyable( true );
    addAttribute( m_coneAngle );

    m_planeAngleMin = unitFn.create( "planeAngleMin", "pmin", MFnUnitAttribute::kAngle, -PI/6.0f );
    unitFn.setMin( -PI*0.5 );
    unitFn.setMax( 0.0 );
    unitFn.setKeyable( true );
    addAttribute( m_planeAngleMin );

    m_planeAngleMax = unitFn.create( "planeAngleMax", "pmax", MFnUnitAttribute::kAngle, PI/6.0f );
    unitFn.setMin( 0.0f );
    unitFn.setMax( PI*0.5 );
    unitFn.setKeyable( true );
    addAttribute( m_planeAngleMax );

    m_twistMin = unitFn.create( "twistMin", "tmin", MFnUnitAttribute::kAngle, -PI );
    unitFn.setMin( -PI );
    unitFn.setMax( PI );
    unitFn.setKeyable( true );
    addAttribute( m_twistMin );

    m_twistMax = unitFn.create( "twistMax", "tmax", MFnUnitAttribute::kAngle, PI );
    unitFn.setMin( -PI );
    unitFn.setMax( PI );
    unitFn.setKeyable( true );
    addAttribute( m_twistMax );

    m_maxFrictionTorque = numFn.create("maxFrictionTorque", "mft", MFnNumericData::kFloat, 0.0f);
    numFn.setMin(0.0f);
    numFn.setKeyable( true );
    addAttribute (m_maxFrictionTorque);

    m_motorType = enumFn.create( "motorType", "motor", 0 );
    enumFn.addField( "None", 0 );
    enumFn.addField( "Position", 1 );
    enumFn.addField( "Velocity", 2);
    enumFn.addField( "Spring Damper", 3 );
    enumFn.setKeyable( true );
    addAttribute( m_motorType );


    m_displayCones = numFn.create( "displayCones", "dispc", MFnNumericData::kBoolean, 0 );
    numFn.setHidden( true );
    numFn.setKeyable( true );
    addAttribute( m_displayCones );

    m_displayTwist = numFn.create( "displayTwist", "dispt", MFnNumericData::kBoolean, 1 );
    numFn.setHidden( true );
    numFn.setKeyable( true );
    addAttribute( m_displayTwist );

    return status;
}


void hctRagDollConstraintNode::draw( const MMatrix& childSpaceToWorld, const MMatrix& parentSpaceToWorld, bool wireframe ) const
{
    // Get attribute values
    float coneAngle;
    float planeAngleMin;
    float planeAngleMax;
    float twistMin;
    float twistMax;
    bool displayCones;
    bool displayTwist;
    {
        MObject thisObj = thisMObject();
        MFnDependencyNode nodeFn( thisObj );
        nodeFn.findPlug( m_coneAngle ).getValue( coneAngle );
        nodeFn.findPlug( m_planeAngleMin ).getValue( planeAngleMin );
        nodeFn.findPlug( m_planeAngleMax ).getValue( planeAngleMax );
        nodeFn.findPlug( m_twistMin ).getValue( twistMin );
        nodeFn.findPlug( m_twistMax ).getValue( twistMax );
        nodeFn.findPlug( m_displayCones ).getValue( displayCones );
        nodeFn.findPlug( m_displayTwist ).getValue( displayTwist );
    }


    // Draw base constraint representation
    hctConstraintNode::draw( childSpaceToWorld, parentSpaceToWorld, wireframe );

    MColor childColor( hctUtilities::getColor( "ConstraintChildSpace" ) );
    MColor parentColor( hctUtilities::getColor( "ConstraintParentSpace" ) );
    MColor volumeColor( hctUtilities::getColor( "ConstraintVolume" ) );

    // Draw additional representation
    // We re-use some of the hinge methods here, but that used Z as its main
    // axis so we rotate to set X as our main axis. We also scale down the
    // display of the hinge volume
    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_LINE_BIT | GL_POLYGON_BIT );
    glEnable( GL_LINE_SMOOTH );     glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );
    glEnable( GL_BLEND );           glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glShadeModel( GL_SMOOTH );

    if( !wireframe )
    {
        glEnable( GL_LIGHTING );
        glEnable( GL_CULL_FACE );
        glCullFace( GL_BACK );
        volumeColor.a = 0.5;
    }

    // Draw child space axis
    glColor4fv( &childColor.r );
    glPushMatrix();
    glMultMatrixd( &childSpaceToWorld.matrix[0][0] );
    glRotatef( 90.0f, 0.0f, 1.0f, 0.0f );
    hctHingeConstraintNode::drawAxis();
    glPopMatrix();

    // Draw child space plane - displayed on parent space
    if( displayTwist )
    {
        glPushMatrix();
        glMultMatrixd( &parentSpaceToWorld.matrix[0][0] );
        glRotatef( 90.0f, 0.0f, 1.0f, 0.0f );
        const float childTwist = getTwist( parentSpaceToWorld, childSpaceToWorld );
        glRotatef( childTwist*180.0f/PI, 0.0f, 0.0f, 1.0f );
        glScalef( 0.33f, 0.33f, 0.33f );
        hctHingeConstraintNode::drawPlane();
        glPopMatrix();
    }

    // Draw parent space
    parentColor.a = 0.66f;
    glColor4fv( &parentColor.r );
    glPushMatrix();
    glMultMatrixd( &parentSpaceToWorld.matrix[0][0] );
    glRotatef( 90.0f, 0.0f, 1.0f, 0.0f );
    {
        hctHingeConstraintNode::drawAxis();
        glPushMatrix();

        if( displayTwist )
        {
            glScalef( 0.33f, 0.33f, 0.33f );
            hctHingeConstraintNode::drawPlane();
            glPopMatrix();
        }

        // Draw the plane cones - wireframe only
        if( displayCones && wireframe )
        {
            glColor4fv( &parentColor.r );
            glPushMatrix();
            glRotatef( 90.0f, 1.0f, 0.0f, 0.0f );
            glTranslatef( 0.0f, 0.0f, -0.25f*sinf(planeAngleMax) );
            hctUtilities::drawCone( 0.25f*cosf(planeAngleMax), 0.25f*sinf(planeAngleMax) );
            glPopMatrix();
            glPushMatrix();
            glRotatef( 90.0f, 1.0f, 0.0f, 0.0f );
            glTranslatef( 0.0f, 0.0f, -0.25f*sinf(planeAngleMin) );
            hctUtilities::drawCone( 0.25f*cosf(planeAngleMin), 0.25f*sinf(planeAngleMin) );
            glPopMatrix();
        }

        // Set the volume color
        glColor4fv( &volumeColor.r );

        // Draw the hinge volume
        if( displayTwist )
        {
            glPushMatrix();
            glScalef( 0.33f, 0.33f, 0.33f );
            glDepthMask( GL_FALSE );
            hctHingeConstraintNode::drawVolume( twistMin, twistMax, wireframe );
            glDepthMask( GL_TRUE );
            glPopMatrix();
        }

        // Draw the rag doll volume
        drawVolume( coneAngle, PI*0.5f - planeAngleMax, PI*0.5f + planeAngleMin, wireframe );
    }
    glPopMatrix();

    glPopAttrib();
}


void hctRagDollConstraintNode::drawVolume( float inclusiveAngle, float upperExclusiveAngle, float lowerExclusiveAngle, bool wireframe )
{
    // We will cache the output of this function using display lists
    static GLuint solidDisplayList = glGenLists( 1 );
    static GLuint wireframeDisplayList = glGenLists( 1 );

    // If the inputs haven't changed then just call the appropriate display list
    static float lastInclusiveAngle = 0.0;
    static float lastUpperExclusiveAngle = 0.0;
    static float lastLowerExclusiveAngle = 0.0;
    if( inclusiveAngle == lastInclusiveAngle &&
        upperExclusiveAngle == lastUpperExclusiveAngle &&
        lowerExclusiveAngle == lastLowerExclusiveAngle )
    {
        glCallList( wireframe? wireframeDisplayList: solidDisplayList );
        return;
    }
    else
    {
        lastInclusiveAngle = inclusiveAngle;
        lastUpperExclusiveAngle = upperExclusiveAngle;
        lastLowerExclusiveAngle = lowerExclusiveAngle;
    }


    //
    // Compile (and/or execute) the new display lists
    //

    // Solid version
    {
        glNewList( solidDisplayList, wireframe? GL_COMPILE: GL_COMPILE_AND_EXECUTE );

        // Inner surface
        glDepthMask( GL_FALSE );
        glPushMatrix();
        glFrontFace( GL_CCW );
        drawQuadrantSides( inclusiveAngle, upperExclusiveAngle );
        glScalef( -1.0f, 1.0f, 1.0f );
        glFrontFace( GL_CW );
        drawQuadrantSides( inclusiveAngle, upperExclusiveAngle );
        glScalef( 1.0f, -1.0f, 1.0f );
        glFrontFace( GL_CCW );
        drawQuadrantSides( inclusiveAngle, lowerExclusiveAngle );
        glScalef( -1.0f, 1.0f, 1.0f );
        glFrontFace( GL_CW );
        drawQuadrantSides( inclusiveAngle, lowerExclusiveAngle );
        glPopMatrix();
        glDepthMask( GL_TRUE );

        // Outer surface
        glPushMatrix();
        glFrontFace( GL_CCW );
        drawQuadrantSurface( inclusiveAngle, upperExclusiveAngle );
        glScalef( -1.0f, 1.0f, 1.0f );
        glFrontFace( GL_CW );
        drawQuadrantSurface( inclusiveAngle, upperExclusiveAngle );
        glScalef( 1.0f, -1.0f, 1.0f );
        glFrontFace( GL_CCW );
        drawQuadrantSurface( inclusiveAngle, lowerExclusiveAngle );
        glScalef( -1.0f, 1.0f, 1.0f );
        glFrontFace( GL_CW );
        drawQuadrantSurface( inclusiveAngle, lowerExclusiveAngle );
        glPopMatrix();

        glEndList();
    }

    // Wireframe version
    {
        glNewList( wireframeDisplayList, wireframe? GL_COMPILE_AND_EXECUTE: GL_COMPILE );

        glPushMatrix();
        drawQuadrantEdges( inclusiveAngle, upperExclusiveAngle );
        glScalef( -1.0f, 1.0f, 1.0 );
        drawQuadrantEdges( inclusiveAngle, upperExclusiveAngle );
        glScalef( 1.0f, -1.0f, 1.0f );
        drawQuadrantEdges( inclusiveAngle, lowerExclusiveAngle );
        glScalef( -1.0f, 1.0f, 1.0f );
        drawQuadrantEdges( inclusiveAngle, lowerExclusiveAngle );
        glPopMatrix();

        glEndList();
    }
}


void hctRagDollConstraintNode::drawQuadrantSurface( float coneAngle, float limitAngle )
{
    glPushAttrib( GL_TRANSFORM_BIT );

    // Set the subdivision accuracy
    const int numTheta = 16;
    const int numPhi = 32;

    // Create an OpenGL clipping plane if there is an intersection
    if( limitAngle != 0 && coneAngle + limitAngle > PI*0.5 )
    {
        const float h = cos( limitAngle );
        GLdouble eqn[4] = { 0.0, -1.0, 0.0, h };
        glClipPlane( GL_CLIP_PLANE0, eqn );
        glEnable( GL_CLIP_PLANE0 );
    }

    // Create the points
    MPointArray points( numTheta * numPhi );
    for( int i=0; i<numTheta; ++i )
    {
        const float theta = i*( PI*0.5f )/(numTheta-1);
        for( int j=0; j<numPhi; ++j )
        {
            float phi = j*coneAngle/(numPhi-1);
            MPoint& point = points[ i*numPhi + j ];
            point.x = cos(theta) * sin(phi);
            point.y = sin(theta) * sin(phi);
            point.z = cos(phi);
        }
    }

    // Draw the strips
    for( int i=1; i<numTheta; ++i )
    {
        glBegin( GL_TRIANGLE_STRIP );
        glNormal3d( 0.0, 0.0, 1.0 );
        glVertex3d( 0.0, 0.0, 1.0 );
        for( int j=1; j<numPhi; ++j )
        {
            glNormal3dv( &points[ (i-1)*numPhi + j ].x );
            glVertex3dv( &points[ (i-1)*numPhi + j ].x );
            glNormal3dv( &points[ i*numPhi + j ].x );
            glVertex3dv( &points[ i*numPhi + j ].x );
        }
        glEnd();
    }

    glPopAttrib();
}


void hctRagDollConstraintNode::drawQuadrantSides( float coneAngle, float limitAngle )
{
    // Set the subdivision accuracy
    const int numTheta = 32;

    // The cone sides
    if( coneAngle < 2.0*PI )
    {
        const float thetaMaxZ = (float)asin( hkMath::min2( 1.0, cos( limitAngle ) / sin( coneAngle ) ) );
        const float h = cos( coneAngle );
        const float r = sin( coneAngle );
        MVectorArray points( numTheta );
        MVectorArray normals( numTheta );
        for( int i=0; i<numTheta; ++i )
        {
            const float theta = i*thetaMaxZ/(numTheta-1);
            points[i] = MVector( r*cos(theta), r*sin(theta), h );

            // Approximate normal
            MVector bin = i==0? (points[i+1]-points[i]): i==(numTheta-1)? (points[i]-points[i-1]): (points[i+1]-points[i-1]);
            normals[i] = ( bin ^ points[i] ).normal();
        }
        normals[0] = ( ( points[1] - points[0] ) ^ points[0] ).normal();

        glBegin( GL_TRIANGLES );
        for( int i=1; i<numTheta; ++i )
        {
            glNormal3dv( &normals[i].x );
            glVertex3dv( &points[i].x );
            glNormal3dv( &normals[i-1].x );
            glVertex3dv( &points[i-1].x );
            glVertex3f( 0.0f, 0.0f, 0.0f );
        }
        glEnd();
    }

    // The limit sides
    if( limitAngle != 0 && coneAngle + limitAngle > PI*0.5 )
    {
        const float thetaMaxY = float( PI*0.5 - asin( hkMath::max2( -1.0, cos( coneAngle ) / sin( limitAngle ) ) ) );
        const float h = cos( limitAngle );
        const float r = sin( limitAngle );
        MVectorArray points( numTheta );
        MVectorArray normals( numTheta );
        for( int i=0; i<numTheta; ++i )
        {
            float theta = i*thetaMaxY/(numTheta-1);
            points[i] = MVector( r*sin(theta), h, r*cos(theta) );

            // Approximate normal
            MVector bin = i==0? (points[i]-points[i+1]): i==(numTheta-1)? (points[i-1]-points[i]): (points[i-1]-points[i+1]);
            normals[i] = ( bin ^ points[i] ).normal();
        }
        normals[0] = normals[1];

        glBegin( GL_TRIANGLES );
        for( int i=1; i<numTheta; ++i )
        {
            glNormal3dv( &normals[i-1].x );
            glVertex3dv( &points[i-1].x );
            glNormal3dv( &normals[i].x );
            glVertex3dv( &points[i].x );
            glVertex3f( 0.0f, 0.0f, 0.0f );
        }
        glEnd();
    }
}


void hctRagDollConstraintNode::drawQuadrantEdges( float coneAngle, float limitAngle )
{
    // Set the subdivision accuracy
    const int numTheta = 32;

    // The cone edge
    if( coneAngle < 2.0*PI )
    {
        const float thetaMaxZ = (float)asin( hkMath::min2( 1.0, cos( limitAngle ) / sin( coneAngle ) ) );
        const float h = cos( coneAngle );
        const float r = sin( coneAngle );
        glBegin( GL_LINE_STRIP );
        for( int i=0; i<numTheta; ++i )
        {
            const float theta = i*thetaMaxZ/(numTheta-1);
            glVertex3d( r*cos(theta), r*sin(theta), h );
        }
        glEnd();
    }

    // The limit edge
    if( limitAngle != 0 && coneAngle + limitAngle > PI*0.5 )
    {
        const float thetaMaxY = float( PI*0.5 - asin( hkMath::max2( -1.0, cos( coneAngle ) / sin( limitAngle ) ) ) );
        const float h = cos( limitAngle );
        const float r = sin( limitAngle );
        glBegin( GL_LINE_STRIP );
        for( int i=0; i<numTheta; ++i )
        {
            float theta = i*thetaMaxY/(numTheta-1);
            glVertex3d( r*sin(theta), h, r*cos(theta) );
        }
        if( coneAngle < limitAngle + PI*0.5 )
        {
            glVertex3f( 0.0f, 0.0f, 0.0f );
        }
        glEnd();
    }
}


// Returns the twist between the two spaces in a rag doll constraint. Replicates the behaviour at run-time.
float hctRagDollConstraintNode::getTwist( const MMatrix& parentSpaceWorld, const MMatrix& childSpaceWorld)
{
    MVector twistParent = MVector::xAxis * parentSpaceWorld;
    MVector twistChild = MVector::xAxis * childSpaceWorld;

    MVector planeParent = MVector::yAxis * parentSpaceWorld;
    MVector planeChild = MVector::yAxis * childSpaceWorld;

    MVector twistAxisWS;
    {
        twistAxisWS = twistParent + twistChild;
        const float twistLength = (float)twistAxisWS.length();
        if (twistLength>1e-16f)
        {
            twistAxisWS = twistAxisWS/twistLength;
        }
        else
        {
            twistAxisWS = twistParent;
        }
    }

    const MVector perp1 = twistAxisWS ^ planeParent; // crossprod
    const MVector perp2 = perp1 ^ twistAxisWS; // crossprod

    const float d1 = float(perp1 * planeChild); // dotprod
    const float d2 = float(perp2 * planeChild); // dotprod
    const float currentTwistAngle  = atan2f(d1, d2);

    return currentTwistAngle;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
