// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Constraints/Prismatic/hctPrismaticConstraintNode.h>

#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>


// Attributes
MObject hctPrismaticConstraintNode::m_isLimitedMin;
MObject hctPrismaticConstraintNode::m_isLimitedMax;
MObject hctPrismaticConstraintNode::m_limitMin;
MObject hctPrismaticConstraintNode::m_limitMax;
MObject hctPrismaticConstraintNode::m_maxFrictionForce;
MObject hctPrismaticConstraintNode::m_motorType;


MStatus hctPrismaticConstraintNode::initialize()
{
    // Initialize the base class
    MStatus status = hctConstraintNode::initialize( "hkPrismaticConstraint", typeId() );
    if( status != MStatus::kSuccess )
    {
        return status;
    }

    // Add additional attributes
    MFnNumericAttribute numFn;
    MFnUnitAttribute    unitFn;
    MFnEnumAttribute    enumFn;

    m_isLimitedMin = numFn.create( "changeMinLinearLimit", "clmin", MFnNumericData::kBoolean, false );
    numFn.setKeyable( true );
    addAttribute( m_isLimitedMin );

    m_isLimitedMax = numFn.create( "changeMaxLinearLimit", "clmax", MFnNumericData::kBoolean, false );
    numFn.setKeyable( true );
    addAttribute( m_isLimitedMax );

    m_limitMin = unitFn.create( "minLinearLimit", "lmin", MFnUnitAttribute::kDistance, -1.0 );
    unitFn.setMax( 0.0f );
    unitFn.setSoftMin(-50.0f);
    unitFn.setKeyable( true );
    addAttribute( m_limitMin );

    m_limitMax = unitFn.create( "maxLinearLimit", "lmax", MFnUnitAttribute::kDistance, 1.0 );
    unitFn.setMin( 0.0f );
    unitFn.setSoftMax(50.0f);
    unitFn.setKeyable( true );
    addAttribute( m_limitMax );

    m_maxFrictionForce = numFn.create ("maxFrictionForce", "mft", MFnNumericData::kFloat, 0.0f);
    numFn.setMin(0.0f);
    numFn.setKeyable( true );
    addAttribute(m_maxFrictionForce);

    m_motorType = enumFn.create( "motorType", "motor", 0 );
    enumFn.addField( "None", 0 );
    enumFn.addField( "Position", 1 );
    enumFn.addField( "Velocity", 2);
    enumFn.addField( "Spring Damper", 3 );
    enumFn.setKeyable( true );
    addAttribute( m_motorType );

    return status;
}


static float parent_color[3];

void hctPrismaticConstraintNode::draw( const MMatrix& childSpaceToWorld, const MMatrix& parentSpaceToWorld, bool wireframe ) const
{
    // Get attribute values
    float limitMin, limitMax;

    MFnDependencyNode nodeFn( thisMObject() );
    bool isLimitedMin, isLimitedMax;
    nodeFn.findPlug( m_isLimitedMin ).getValue( isLimitedMin );
    nodeFn.findPlug( m_isLimitedMax ).getValue( isLimitedMax );

    nodeFn.findPlug( m_limitMin ).getValue( limitMin );
    nodeFn.findPlug( m_limitMax ).getValue( limitMax );

    // Draw base constraint representation
    hctConstraintNode::draw( childSpaceToWorld, parentSpaceToWorld, wireframe );

    MColor childColor( hctUtilities::getColor( "ConstraintChildSpace" ) );
    MColor parentColor( hctUtilities::getColor( "ConstraintParentSpace" ) );
    MColor volumeColor( hctUtilities::getColor( "ConstraintVolume" ) );
    volumeColor.a = 0.33f;

    parent_color[0] = parentColor.r;
    parent_color[1] = parentColor.g;
    parent_color[2] = parentColor.b;

    // Draw additional representation
    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_LINE_BIT | GL_POLYGON_BIT );
    glEnable( GL_LINE_SMOOTH );     glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );
    glEnable( GL_BLEND );           glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glDisable( GL_CULL_FACE );
    glLightModeli( GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE );
    glShadeModel( GL_SMOOTH );

    if( !wireframe )
    {
        glEnable( GL_LIGHTING );
        volumeColor.a = 0.5;
    }
    else
    {
        glLineWidth( 1.0 );
        glLineStipple( 4, 0xAAAA );
    }

    // Draw child space
    glPushMatrix();
    glMultMatrixd( &childSpaceToWorld.matrix[0][0] );
    glColor4fv( &childColor.r );
    drawTriangle();
    drawAxis();
    glPopMatrix();

    // Draw parent space
    glPushMatrix();
    glMultMatrixd( &parentSpaceToWorld.matrix[0][0] );
    {
        parentColor.a = 0.66f;
        glColor4fv( &parentColor.r );
        drawTriangle();

        drawAxis();

        // Draw the volume
        glColor4fv( &volumeColor.r );

        //modify scaling to keep z-axis world size fixed
        glPushMatrix();
        {
            MMatrix fixz;
            fixz.setToIdentity();
            if ( m_parentScale.matrix[2][2] != 0.0 )
                fixz.matrix[2][2] = 1.0 / m_parentScale.matrix[2][2];
            glMultMatrixd( &fixz.matrix[0][0] );
        }

        drawVolume( isLimitedMin, isLimitedMax,
                    limitMin, limitMax, wireframe );

        glPopMatrix();

    }
    glPopMatrix();

    glPopAttrib();
}

//triangle dimensions
static const float x = 0.5f;
static const float y = x;

//axis arrow dimensions
static const float axisbody_length = 0.5f*x;
static const float axisbody_radius = 0.1f*x;
static const float axiscone_length = 0.5f*x;
static const float axiscone_radius = 0.2f*x;

//prism volume vertices
static const float verts[3][2] = {{-0.5f*x, -y}, { -0.5f*x, y}, {0.5f*x, 0.0f}};

void hctPrismaticConstraintNode::drawAxis()
{
    glPushMatrix();
    glTranslatef( 0.0f, 0.0f, axisbody_length );
    hctUtilities::drawCone( axiscone_radius, axiscone_length, 16 );
    glPopMatrix();

    glPushMatrix();
    glTranslatef( 0.0f, 0.0f, 0.0f );
    hctUtilities::drawCylinder( axisbody_radius, axisbody_length, 16 );
    glPopMatrix();
}

void hctPrismaticConstraintNode::drawTriangle()
{
    glPushAttrib( GL_ENABLE_BIT | GL_POLYGON_BIT );

    glBegin( GL_TRIANGLES );
    glNormal3d( 0.0, 0.0, -1.0 );
    glVertex3d( verts[0][0], verts[0][1], 0.0 );
    glVertex3d( verts[1][0], verts[1][1], 0.0 );
    glVertex3d( verts[2][0], verts[2][1], 0.0 );
    glEnd();

    glPopAttrib();
}

//world size of "infinite" rails (before scaling)
static const float railsize = 2.0f;

void hctPrismaticConstraintNode::drawVolume( bool isLimitedMin, bool isLimitedMax,
                                            float limitMin, float limitMax, bool wireframe )
{
    // Cache the output of this function using display lists
    static GLuint solidDisplayList = glGenLists( 1 );
    static GLuint wireframeDisplayList = glGenLists( 1 );

    float color[4];
    glGetFloatv( GL_CURRENT_COLOR, color );

    // If the inputs haven't changed then just call the appropriate display list
    static float lastLimitMin = 0.0f;
    static float lastLimitMax = 0.0f;
    static bool lastIsLimitedMin = true;
    static bool lastIsLimitedMax = true;

    if( limitMin == lastLimitMin &&
        limitMax == lastLimitMax &&
        isLimitedMin == lastIsLimitedMin &&
        isLimitedMax == lastIsLimitedMax )
    {
        glCallList( wireframe? wireframeDisplayList: solidDisplayList );
        return;
    }
    else
    {
        lastLimitMin = limitMin;
        lastLimitMax = limitMax;
        lastIsLimitedMax = isLimitedMax;
        lastIsLimitedMin = isLimitedMin;
    }

    // Make sure the limits are the right way around
    if( limitMin > limitMax )
    {
        float temp = limitMin;
        limitMin = limitMax;
        limitMax = temp;
    }
    float limit;

    //
    // Compile the new display lists. Also execute the appropriate one
    //

    // Solid version
    {
        glNewList( solidDisplayList, wireframe? GL_COMPILE: GL_COMPILE_AND_EXECUTE );

        float normals[3][2] = {{1.0, 0.0},  {-y, -x}, {-y, x}};
        for (int i=1; i<=2; i++)
            for (int j=0; j<2; j++)
                normals[i][j] /= sqrtf(x*x+y*y);

        //minimum section
        if ( isLimitedMin)
        {
            limit = limitMin;
        }
        else
        {
            limit = -railsize;
        }

        //sides
        glBegin(GL_QUADS);
        for (int i=0; i<3; i++)
        {
            int j = (i+1)%3;
            glNormal3f(normals[i][0], normals[i][1], 0.0);

            glColor4f( color[0], color[1], color[2], isLimitedMin? color[3]: 0.0f );
            glVertex3f(verts[i][0], verts[i][1], limit);
            glVertex3f(verts[j][0], verts[j][1], limit);

            glColor4fv( color );
            glVertex3f(verts[j][0], verts[j][1], 0.0);
            glVertex3f(verts[i][0], verts[i][1], 0.0);
        }
        glEnd();

        //end cap
        if ( isLimitedMin)
        {
            glBegin( GL_TRIANGLES );
            glNormal3f(0.0f, 0.0f, -1.0f);
            for (int i=0; i<3; i++)
            {
                glVertex3d( verts[i][0], verts[i][1], limit );
            }
            glEnd();
        }

        //maximum section
        if ( isLimitedMax)
        {
            limit = limitMax;
        }
        else
        {
            limit = railsize;
        }

        //sides
        glBegin(GL_QUADS);
        for (int i=0; i<3; i++)
        {
            int j = (i+1)%3;
            glNormal3f(normals[i][0], normals[i][1], 0.0);

            glColor4f( color[0], color[1], color[2], isLimitedMax? color[3]: 0.0f );
            glVertex3f(verts[j][0], verts[j][1], limit);
            glVertex3f(verts[i][0], verts[i][1], limit);
            glColor4fv( color );
            glVertex3f(verts[i][0], verts[i][1], 0.0);
            glVertex3f(verts[j][0], verts[j][1], 0.0);

        }
        glEnd();

        //end cap
        if ( isLimitedMax)
        {
            glBegin( GL_TRIANGLES );
            glNormal3f(0.0f, 0.0f, 1.0f);
            for (int i=2; i>=0; i--)
            {
                glVertex3d( verts[i][0], verts[i][1], limit );
            }
            glEnd();
        }

        glEndList();

    }

    // Wireframe version
    {
        glNewList( wireframeDisplayList, wireframe? GL_COMPILE_AND_EXECUTE: GL_COMPILE );

        glColor4f( parent_color[0], parent_color[1], parent_color[2], 0.66f );

        //minimum section
        if ( isLimitedMin )
        {
            glDisable( GL_LINE_STIPPLE );
            limit = limitMin;
        }
        else
        {
            glEnable( GL_LINE_STIPPLE );
            limit = -railsize;
        }

        //sides
        glBegin(GL_LINE_LOOP);
        for (int i=0; i<3; i++)
        {
            int j = (i+1)%3;
            glVertex3f(verts[i][0], verts[i][1], limit);
            glVertex3f(verts[i][0], verts[i][1], 0.0);
            glVertex3f(verts[j][0], verts[j][1], 0.0);
            glVertex3f(verts[j][0], verts[j][1], limit);
        }
        glEnd();

        //end cap
        if ( isLimitedMin )
        {
            glBegin( GL_LINE_LOOP );
            for (int i=0; i<3; i++)
            {
                glVertex3d( verts[i][0], verts[i][1], limit );
            }
            glEnd();
        }

        //maximum section
        if ( isLimitedMax )
        {
            glDisable( GL_LINE_STIPPLE );
            limit = limitMax;
        }
        else
        {
            glEnable( GL_LINE_STIPPLE );
            limit = railsize;
        }

        //sides
        glBegin(GL_LINE_LOOP);
        for (int i=0; i<3; i++)
        {
            int j = (i+1)%3;
            glVertex3f(verts[i][0], verts[i][1], limit);
            glVertex3f(verts[i][0], verts[i][1], 0.0);
            glVertex3f(verts[j][0], verts[j][1], 0.0);
            glVertex3f(verts[j][0], verts[j][1], limit);
        }
        glEnd();

        //end cap
        if ( isLimitedMax )
        {
            glBegin( GL_LINE_LOOP );
            for (int i=0; i<3; i++)
            {
                glVertex3d( verts[i][0], verts[i][1], limit );
            }
            glEnd();
        }

        glEndList();

    }

}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
