// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Constraints/Hinge/hctHingeConstraintNode.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

// Attributes
MObject hctHingeConstraintNode::m_isLimited;
MObject hctHingeConstraintNode::m_limitMin;
MObject hctHingeConstraintNode::m_limitMax;
MObject hctHingeConstraintNode::m_maxFrictionTorque;
MObject hctHingeConstraintNode::m_motorType;


MStatus hctHingeConstraintNode::initialize()
{
    // Initialize the base class
    MStatus status = hctConstraintNode::initialize( "hkHingeConstraint", typeId() );
    if( status != MStatus::kSuccess )
    {
        return status;
    }

    // Add additional attributes
    MFnNumericAttribute numFn;
    MFnUnitAttribute    unitFn;
    MFnEnumAttribute    enumFn;

    m_isLimited = numFn.create( "isLimited", "limit", MFnNumericData::kBoolean, 0 );
    numFn.setKeyable( true );
    addAttribute( m_isLimited );

    m_limitMin = unitFn.create( "limitMin", "lmin", MFnUnitAttribute::kAngle, -PI*0.5f );
    unitFn.setMin( -PI );
    unitFn.setMax( PI );
    unitFn.setKeyable( true );
    addAttribute( m_limitMin );

    m_limitMax = unitFn.create( "limitMax", "lmax", MFnUnitAttribute::kAngle, PI*0.5f );
    unitFn.setMin( -PI );
    unitFn.setMax( PI );
    unitFn.setKeyable( true );
    addAttribute( m_limitMax );

    m_maxFrictionTorque = numFn.create ("maxFrictionTorque", "mft", MFnNumericData::kFloat, 0.0f);
    numFn.setMin(0.0f);
    numFn.setKeyable( true );
    addAttribute(m_maxFrictionTorque);

    m_motorType = enumFn.create( "motorType", "motor", 0 );
    enumFn.addField( "None", 0 );
    enumFn.addField( "Position", 1 );
    enumFn.addField( "Velocity", 2);
    enumFn.addField( "Spring Damper", 3 );
    enumFn.setKeyable( true );
    addAttribute( m_motorType );

    return status;
}


void hctHingeConstraintNode::draw( const MMatrix& childSpaceToWorld, const MMatrix& parentSpaceToWorld, bool wireframe ) const
{
    // Get attribute values
    float limitStart, limitEnd;
    {
        MFnDependencyNode nodeFn( thisMObject() );
        bool isLimited;
        nodeFn.findPlug( m_isLimited ).getValue( isLimited );
        nodeFn.findPlug( m_limitMin ).getValue( limitStart );
        nodeFn.findPlug( m_limitMax ).getValue( limitEnd );
        if( !isLimited )
        {
            limitStart = 0.0f;
            limitEnd = 2.0f*PI;
        }
    }


    // Draw base constraint representation
    hctConstraintNode::draw( childSpaceToWorld, parentSpaceToWorld, wireframe );

    MColor childColor( hctUtilities::getColor( "ConstraintChildSpace" ) );
    MColor parentColor( hctUtilities::getColor( "ConstraintParentSpace" ) );
    MColor volumeColor( hctUtilities::getColor( "ConstraintVolume" ) );

    // Draw additional representation
    glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_LINE_BIT | GL_POLYGON_BIT );
    glEnable( GL_LINE_SMOOTH );     glHint( GL_LINE_SMOOTH_HINT, GL_NICEST );
    glEnable( GL_BLEND );           glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glShadeModel( GL_SMOOTH );

    if( !wireframe )
    {
        glEnable( GL_LIGHTING );
        glEnable( GL_CULL_FACE );
        glCullFace( GL_BACK );
        volumeColor.a = 0.5;
    }

    // Draw child space
    glPushMatrix();
    glMultMatrixd( &childSpaceToWorld.matrix[0][0] );
    glColor4fv( &childColor.r );
    drawPlane();
    glScalef( 0.33f, 0.33f, 0.33f );
    drawAxis();
    glPopMatrix();

    // Draw parent space
    glPushMatrix();
    glMultMatrixd( &parentSpaceToWorld.matrix[0][0] );
    {
        parentColor.a = 0.66f;
        glColor4fv( &parentColor.r );
        drawPlane();

        glPushMatrix();
        glScalef( 0.33f, 0.33f, 0.33f );
        drawAxis();
        glPopMatrix();

        // Draw the volume
        glColor4fv( &volumeColor.r );
        drawVolume( limitStart, limitEnd, wireframe );
    }
    glPopMatrix();

    glPopAttrib();
}


void hctHingeConstraintNode::drawAxis()
{
    glPushMatrix();
    glTranslatef( 0.0f, 0.0f, 0.75f );
    hctUtilities::drawCone( 0.1f, 0.25f, 8 );
    glRotatef( 180.0f, 1.0f, 0.0f, 0.0f );
    hctUtilities::drawCone( 0.025f, 0.75f, 4 );
    glPopMatrix();
}

void hctHingeConstraintNode::drawPlane()
{
    const float length = 1.0f;
    const float width = 0.1f;

    glPushAttrib( GL_ENABLE_BIT | GL_POLYGON_BIT );
    glDisable( GL_CULL_FACE );
    glLightModeli( GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE );

    glBegin( GL_QUADS );
    glNormal3f( 0.0f, 1.0f, 0.0f );
    glVertex3f( 0.0f, 0.0f, -width );
    glVertex3f( 0.0f, 0.0f, width );
    glVertex3f( length, 0.0f, width );
    glVertex3f( length, 0.0f, -width );
    glEnd();

    glPopAttrib();
}

void hctHingeConstraintNode::drawVolume( float limitMin, float limitMax, bool wireframe )
{
    // Cache the output of this function using display lists
    static GLuint solidDisplayList = glGenLists( 1 );
    static GLuint wireframeDisplayList = glGenLists( 1 );

    // If the inputs haven't changed then just call the appropriate display list
    static float lastLimitMin = 0.0;
    static float lastLimitMax = 0.0;
    if( limitMin == lastLimitMin &&
        limitMax == lastLimitMax )
    {
        glCallList( wireframe? wireframeDisplayList: solidDisplayList );
        return;
    }
    else
    {
        lastLimitMin = limitMin;
        lastLimitMax = limitMax;
    }


    // Subdivision settings
    const int numSweepPoints = 64;
    const float sweepHalfWidth = 0.1f;
    bool limited = false;

    // Generate x & y values of sweep points
    MFloatVectorArray sweepPoints( numSweepPoints );
    {
        // Make sure the limits are the right way around
        if( limitMin > limitMax )
        {
            float temp = limitMin;
            limitMin = limitMax;
            limitMax = temp;
        }
        limited = (limitMin != 0.0 || limitMax != 2.0*PI );

        // Create the points
        for( int i=0; i<numSweepPoints; ++i )
        {
            float rad = ( i*(limitMax-limitMin)/(numSweepPoints-1) ) + limitMin;
            sweepPoints[i].x = cosf(rad);
            sweepPoints[i].y = sinf(rad);
        }
    }


    //
    // Compile the new display lists. Also execute the appropriate one
    //

    // Solid version
    {
        glNewList( solidDisplayList, wireframe? GL_COMPILE: GL_COMPILE_AND_EXECUTE );

        // First sweep
        glBegin( GL_TRIANGLE_FAN );
        glNormal3d( 0.0, 0.0, 1.0 );
        glVertex3f( 0.0f, 0.0f, sweepHalfWidth );
        for( int i=0; i<numSweepPoints; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, sweepHalfWidth );
        }
        glEnd();

        // Second sweep
        glBegin( GL_TRIANGLE_FAN );
        glNormal3d( 0.0, 0.0, -1.0 );
        glVertex3f( 0.0f, 0.0f, -sweepHalfWidth );
        for( int i=numSweepPoints-1; i>=0; --i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, -sweepHalfWidth );
        }
        glEnd();

        // Rim
        glBegin( GL_QUAD_STRIP );
        for( int i=0; i<numSweepPoints; ++i )
        {
            glNormal3d( sweepPoints[i].x, sweepPoints[i].y, 0.0 );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, sweepHalfWidth );
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, -sweepHalfWidth );
        }
        glEnd();

        // Cutaway
        if( limited )
        {
            glBegin( GL_QUADS );
            glNormal3d( sweepPoints[0].y, -sweepPoints[0].x, 0.0 );
            glVertex3d( sweepPoints[0].x, sweepPoints[0].y, -sweepHalfWidth );
            glVertex3d( sweepPoints[0].x, sweepPoints[0].y, sweepHalfWidth );
            glVertex3d( 0.0, 0.0, sweepHalfWidth );
            glVertex3d( 0.0, 0.0, -sweepHalfWidth );
            glNormal3d( -sweepPoints[numSweepPoints-1].y, sweepPoints[numSweepPoints-1].x, 0.0 );
            glVertex3d( sweepPoints[numSweepPoints-1].x, sweepPoints[numSweepPoints-1].y, sweepHalfWidth );
            glVertex3d( sweepPoints[numSweepPoints-1].x, sweepPoints[numSweepPoints-1].y, -sweepHalfWidth );
            glVertex3d( 0.0, 0.0, -sweepHalfWidth );
            glVertex3d( 0.0, 0.0, sweepHalfWidth );
            glEnd();
        }

        glEndList();
    }

    // Wireframe version
    {
        glNewList( wireframeDisplayList, wireframe? GL_COMPILE_AND_EXECUTE: GL_COMPILE );

        // First sweep
        glBegin( GL_LINE_LOOP );
        if( limited )
        {
            glVertex3f( 0.0f, 0.0f, sweepHalfWidth );
        }
        for( int i=0; i<numSweepPoints; ++i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, sweepHalfWidth );
        }
        glEnd();

        // Second sweep
        glBegin( GL_LINE_LOOP );
        if( limited )
        {
            glVertex3f( 0.0f, 0.0f, -sweepHalfWidth );
        }
        for( int i=numSweepPoints-1; i>=0; --i )
        {
            glVertex3d( sweepPoints[i].x, sweepPoints[i].y, -sweepHalfWidth );
        }
        glEnd();

        // Cutaway edges
        if( limited )
        {
            glBegin( GL_LINES );
            glVertex3f( 0.0f, 0.0f, sweepHalfWidth);
            glVertex3f( 0.0f, 0.0f, -sweepHalfWidth);
            glVertex3f( sweepPoints[0].x, sweepPoints[0].y, sweepHalfWidth);
            glVertex3f( sweepPoints[0].x, sweepPoints[0].y, -sweepHalfWidth);
            glVertex3f( sweepPoints[numSweepPoints-1].x, sweepPoints[numSweepPoints-1].y, sweepHalfWidth);
            glVertex3f( sweepPoints[numSweepPoints-1].x, sweepPoints[numSweepPoints-1].y, -sweepHalfWidth);
            glEnd();
        }

        glEndList();
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
