// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Channel/hctChannelNode.h>

#include <algorithm>
using std::vector;

#include <GL/glext.h>
#include <GL/wglext.h>

PFNGLBLENDCOLORPROC glBlendColor;

// Statics
MObject hctChannelNode::m_hkType;
MObject hctChannelNode::m_display;
MObject hctChannelNode::m_opacity;
MObject hctChannelNode::m_channelName;
MObject hctChannelNode::m_visualizationType;
MObject hctChannelNode::m_dimensionType;
MObject hctChannelNode::m_rescaleEnable;
MObject hctChannelNode::m_rescaleMin;
MObject hctChannelNode::m_rescaleMax;

void playbackCallback( bool state, void* clientData );

hctChannelNode::hctChannelNode()
{
    m_animationPlaybackCallbackId = 0;
}

hctChannelNode::~hctChannelNode()
{
    // We need to remove the old callback or it'll get called on stale data.
    MConditionMessage::removeCallback(this->m_animationPlaybackCallbackId);
}

MStatus hctChannelNode::initialize()
{
    MStatus status = MStatus::kSuccess;

    MFnTypedAttribute   typedFn;
    MFnMessageAttribute messageFn;
    MFnMatrixAttribute  matrixFn;
    MFnEnumAttribute    enumFn;
    MFnNumericAttribute numFn;
    MFnUnitAttribute    unitFn;

    // hkType
    {
        MFnStringData stringFn;
        m_hkType = typedFn.create( "hkType", "hkt", MFnData::kString, stringFn.create( "hkChannel" ), &status );
        if( !status ) MGlobal::displayError( "Couldn't create hkType attr" );
        typedFn.setWritable( false );
        typedFn.setHidden( true );
        addAttribute( m_hkType );
    }

    // Referenced channel (a string attribute containing the name of a channel in the associated mesh)
    {
        MFnStringData stringFn;
        m_channelName = typedFn.create( "channelName", "chan", MFnData::kString, stringFn.create( "NULL" ), &status );
        if( !status ) MGlobal::displayError( "Couldn't create channelName attr" );
        typedFn.setHidden( true );
        addAttribute( m_channelName );
    }

    // Visualization params
    {
        m_visualizationType = enumFn.create( "visualizationType", "vt", 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create visualizationType attr" );
        enumFn.addField( "radius",  0 );
        enumFn.addField( "normal distance", 1 );
        enumFn.setDefault( 0 );
        enumFn.setKeyable( true );
        addAttribute( m_visualizationType );

        m_display = numFn.create( "display", "disp", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create display attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        numFn.setDefault( false );
        addAttribute( m_display );

        m_opacity = numFn.create( "opacity", "op", MFnNumericData::kDouble, 1.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create opacity attr" );
        numFn.setKeyable( true );
        numFn.setMin( 0.0 );
        numFn.setMax( 1.0 );
        addAttribute( m_opacity );
    }

    // Rescaling
    {
        m_dimensionType = enumFn.create( "dimensionType", "dt", 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create dimensionType attr" );
        enumFn.addField( "float",    0 );
        enumFn.addField( "distance", 1 );
        enumFn.addField( "angle",    2 );
        enumFn.setDefault( 1 );
        enumFn.setKeyable( true );
        addAttribute( m_dimensionType );

        m_rescaleEnable = numFn.create( "rescaleEnable", "resceb", MFnNumericData::kBoolean, 0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create rescaleEnable attr" );
        numFn.setConnectable( false );
        numFn.setKeyable( true );
        numFn.setDefault( false );
        addAttribute( m_rescaleEnable );

        m_rescaleMin = numFn.create( "rescaleMin", "rescmin", MFnNumericData::kDouble, 1.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create rescaleMin attr" );
        numFn.setKeyable( true );
        numFn.setSoftMin( 0.0 );
        numFn.setSoftMax( 10.0 );
        numFn.setDefault( 0.0 );
        addAttribute( m_rescaleMin );

        m_rescaleMax = numFn.create( "rescaleMax", "rescmax", MFnNumericData::kDouble, 1.0, &status );
        if( !status ) MGlobal::displayError( "Couldn't create rescaleMax attr" );
        numFn.setKeyable( true );
        numFn.setSoftMin( 0.0 );
        numFn.setSoftMax( 10.0 );
        numFn.setDefault( 1.0 );
        addAttribute( m_rescaleMax );
    }

    // Allow manipulation
    MTypeId id = typeId();
    MPxManipContainer::addToManipConnectTable( id );

    // Needed to allow use of this OpenGL 2.0 function
    glBlendColor = (PFNGLBLENDCOLORPROC) wglGetProcAddress("glBlendColor");

    return status;
}


void hctChannelNode::readAttributes(bool& refreshAllCells, bool& refreshMainOnly)
{
    MStatus status;

    MObject node = thisMObject();
    MDagPath nodePath;
    MDagPath::getAPathTo( node, nodePath );
    MFnDependencyNode nodeFn( nodePath.node() );

    // refreshAllCells, refreshMainOnly false by default on entry

    MString channelName;
    {
        MFnAttribute attrFn( hctChannelNode::m_channelName );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( channelName );
            if (m_cachedChannelName != channelName)
            {
                refreshAllCells = true;
            }
            m_cachedChannelName = channelName;
        }
    }

    int rescaleEnable;
    {
        MFnAttribute attrFn( hctChannelNode::m_rescaleEnable );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( rescaleEnable );
            if (m_cachedRescaleEnable != rescaleEnable)
            {
                refreshAllCells = true;
            }
            m_cachedRescaleEnable = rescaleEnable;
        }
    }

    float rescaleMin;
    {
        MFnAttribute attrFn( hctChannelNode::m_rescaleMin );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( rescaleMin );
            if (m_cachedMin != rescaleMin)
            {
                refreshAllCells = true;
            }
            m_cachedMin = rescaleMin;
        }
    }

    float rescaleMax;
    {
        MFnAttribute attrFn( hctChannelNode::m_rescaleMax );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( rescaleMax );
            if (m_cachedMax != rescaleMax)
            {
                refreshAllCells = true;
            }
            m_cachedMax = rescaleMax;
        }
    }

    int visualizationType;
    {
        MFnAttribute attrFn( hctChannelNode::m_visualizationType );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( visualizationType );
            if (m_cachedVizTypeEnum != visualizationType)
            {
                refreshAllCells = true;
            }
            m_cachedVizTypeEnum = (VisualizationType)visualizationType;
        }
    }

    bool display;
    {
        MFnAttribute attrFn( hctChannelNode::m_display );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( display );
            if (display && !m_cachedDisplayEnable)
            {
                // Refresh geometry each time the visualization is toggled on
                m_refreshGeometry = true;
                refreshAllCells = true;
            }
            m_cachedDisplayEnable = display;
        }
    }

    int dimensions;
    {
        MFnAttribute attrFn( hctChannelNode::m_dimensionType );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( dimensions );
            if ( dimensions != 1 /*distance*/ )
            {
                m_cachedDisplayEnable = false;
            }
        }
    }

    double opacity;
    {
        MFnAttribute attrFn( hctChannelNode::m_opacity );
        MPlug plug = nodeFn.findPlug( attrFn.name(), &status );
        if ( status == MStatus::kSuccess )
        {
            plug.getValue( opacity );

            if (m_cachedOpacity != opacity)
            {
                // If opacity just became <1.0, we need to refresh all the rendering to get the transparency re-ordering
                if (opacity<1.0f && m_cachedOpacity==1.0f)
                {
                    refreshAllCells = true;
                }
                else
                {
                    refreshMainOnly = true;
                }
            }

            m_cachedOpacity = opacity;
        }
    }

    // Visualization colors
    for (int c=0; c<4; ++c)
    {
        const MString& colorAttrName = m_vizColorAttributeNames[c];
        MFloatVector& color = m_vizColors[c];

        MPlug rPlug = nodeFn.findPlug( colorAttrName + MString("R"), &status );
        if ( status != MStatus::kSuccess ) continue;
        MPlug gPlug = nodeFn.findPlug( colorAttrName + MString("G"), &status );
        if ( status != MStatus::kSuccess ) continue;
        MPlug bPlug = nodeFn.findPlug( colorAttrName + MString("B"), &status );
        if ( status != MStatus::kSuccess ) continue;

        float r, g, b;
        rPlug.getValue(r);
        gPlug.getValue(g);
        bPlug.getValue(b);
        MFloatVector newColor(r, g, b);

        if ( newColor != color )
        {
            refreshAllCells = true;
            color = newColor;
        }
    }
}


// Find the mesh (take the first shape node under the parent transform which has a channel with the cached name)
bool hctChannelNode::findMesh()
{
    MObject node = thisMObject();
    MDagPath nodePath;
    MDagPath::getAPathTo( node, nodePath );

    MString nodeName = nodePath.fullPathName();

    MDagPath parentPath = nodePath; parentPath.pop();
    MString parentName = parentPath.fullPathName();

    m_meshPath = parentPath;
    unsigned int numChildren = parentPath.childCount();

    for( unsigned int i=0; i<numChildren; ++i )
    {
        MFn::Type type = parentPath.child(i).apiType();
        if( type == MFn::kMesh )
        {
            MObject child = parentPath.child(i);

            MFnDagNode dagFn(child);
            MStatus status;
            MPlug attributePlug = dagFn.findPlug(m_cachedChannelName, status);

            if ( status == MStatus::kSuccess && attributePlug.isArray() )
            {
                m_meshPath.push( parentPath.child(i) );
                return true;
            }
        }
    }

    return false;
}

bool hctChannelNode::checkMesh()
{
    return ( m_meshPath.isValid() && m_meshPath.apiType() == MFn::kMesh );
}

void hctChannelNode::postConstructor()
{
    m_cachedFloatArray.clear();
    m_cachedVertexArray.clear();
    m_cachedNormalArray.clear();
    m_cachedVizTypeEnum = VIZ_INVALID;

    m_sphereDisplayList = 0;
    m_mainDisplayList = 0;
    for (int c=0; c<NCELL; ++c)
    {
        m_cellDisplayLists[c] = 0;
    }

    m_cachedObjToWorld.setToIdentity();
    m_cachedWorldToObj.setToIdentity();

    // Viz. color defaults
    float spherePos[3] = {0.479f, 0.323f, 0.938f};
    float sphereNeg[3] = {1.000f, 0.087f, 0.156f};
    float pinPos[3]    = {0.020f, 0.000f, 0.938f};
    float pinNeg[3]    = {1.000f, 0.798f, 0.000f};

    m_vizColors.setLength(4);
    m_vizColors.set( spherePos, SPHERE_POSITIVE );
    m_vizColors.set( sphereNeg, SPHERE_NEGATIVE );
    m_vizColors.set( pinPos, PIN_POSITIVE );
    m_vizColors.set( pinNeg, PIN_NEGATIVE );

    m_vizColorAttributeNames.setLength(4);
    m_vizColorAttributeNames.set( MString("spherePositiveColor"), SPHERE_POSITIVE );
    m_vizColorAttributeNames.set( MString("sphereNegativeColor"), SPHERE_NEGATIVE );
    m_vizColorAttributeNames.set( MString("pinPositiveColor"), PIN_POSITIVE );
    m_vizColorAttributeNames.set( MString("pinNegativeColor"), PIN_NEGATIVE );

    // Anim. playback callback
    MStatus status;
    m_animationPlaybackCallbackId = MConditionMessage::addConditionCallback("playingBack", playbackCallback, (void*)this, &status);
    requestContinualRefresh(false);
}


// Callback got when playing back an animation. In that case we want to continually refresh
// the mesh geometry and visualization display lists in order for the visualization to match the mesh.
void playbackCallback( bool playbackOn, void* clientData )
{
    hctChannelNode* channelNode = static_cast<hctChannelNode*>(clientData);

    channelNode->requestContinualRefresh(playbackOn);
}

void hctChannelNode::requestContinualRefresh(bool on)
{
    m_refreshAllContinually = on;
}

bool hctChannelNode::refreshGeometry()
{
    // Extract (object space) vertices and normals
    MPointArray vertexArray;
    MFloatVectorArray normalArray;
    extractObjectSpaceGeometry( vertexArray, normalArray );

    cacheVertexArray(vertexArray);
    cacheNormalArray(normalArray);

    // Compute and store the (object space) AABB of the mesh
    m_bbox = getMeshBoundingBox();

    return true;
}

void hctChannelNode::cacheVertexArray(const MPointArray& vertexArray)
{
    m_cachedVertexArray.clear();
    m_cachedVertexArray.setLength(vertexArray.length());
    m_cachedVertexArray.copy(vertexArray);
}

void hctChannelNode::cacheFloatArray(const MFloatArray& floatArray)
{
    m_cachedFloatArray.clear();
    m_cachedFloatArray.setLength(floatArray.length());
    m_cachedFloatArray.copy(floatArray);
}

void hctChannelNode::cacheNormalArray(const MFloatVectorArray& normalArray)
{
    m_cachedNormalArray.clear();
    m_cachedNormalArray.setLength(normalArray.length());
    m_cachedNormalArray.copy(normalArray);
}

MStatus hctChannelNode::rebuildSphereDisplayList()
{
    // Build sphere geometry
    if (m_sphereDisplayList != 0)
    {
        glDeleteLists(m_sphereDisplayList, 1);
    }
    m_sphereDisplayList = glGenLists(1);

    if ( m_sphereDisplayList == 0 )
    {
        MGlobal::displayError( "glGenLists() returned failure in hctChannelNode" );
        return MStatus::kFailure;
    }

    buildSphere();
    return MStatus::kSuccess;
}

void hctChannelNode::buildSphere()
{
    glNewList(m_sphereDisplayList, GL_COMPILE);

    const int numTheta = 8;
    const int numPhi = 16;

    MPointArray points;
    points.setLength( numTheta * numPhi );

    const float pi = 3.1415926f;

    for( int i=0; i<numTheta; ++i )
    {
        float theta = i * pi/(numTheta-1);
        for( int j=0; j<numPhi; ++j )
        {
            float phi = j * 2.0f * pi/(numPhi-1);
            MPoint& point = points[ i*numPhi + j ];
            point.x = cos(phi) * sin(theta);
            point.y = sin(phi) * sin(theta);
            point.z = cos(theta);
        }
    }

    for( int j=1; j<numPhi; ++j )
    {
        glBegin( GL_TRIANGLE_STRIP );
        glNormal3d( 0.0, 0.0, 1.0f );
        glVertex3d( 0.0, 0.0, 1.0f );
        for( int i=1; i<numTheta-1; ++i )
        {
            glNormal3dv( &points[ i*numPhi + j-1 ].x );
            glVertex3dv( &points[ i*numPhi + j-1 ].x );
            glNormal3dv( &points[ i*numPhi + j   ].x );
            glVertex3dv( &points[ i*numPhi + j   ].x );
        }
        glNormal3d( 0.0f, 0.0f, -1.0f );
        glVertex3d( 0.0f, 0.0f, -1.0f );
        glEnd();
    }

    glEndList();
}


bool hctChannelNode::extractFloats( MFloatArray& floatArray )
{
    MStatus status;
    floatArray.clear();

    MFnDagNode dagFn(m_meshPath.node());
    MPlug attributePlug = dagFn.findPlug(m_cachedChannelName, status);

    if ( status == MStatus::kSuccess && attributePlug.isArray() )
    {
        // Read the per-vertex values
        unsigned int vL = m_cachedVertexArray.length();

        float value;
        float range = m_cachedMax - m_cachedMin;

        for( unsigned int i=0; i<vL; ++i )
        {
            MPlug plugElement = attributePlug.elementByLogicalIndex(i, &status);
            if( status != MStatus::kSuccess )
            {
                MString msg = MString("Error, failed to read floats of channel ") + m_cachedChannelName;
                status.perror(msg);
                return false;
            }
            plugElement.getValue(value);

            if (m_cachedRescaleEnable)
            {
                value = m_cachedMin + range * value;
            }

            floatArray.append(value);
        }
    }

    return true;
}


// The input dagPath is the path of the transform node containing the channel node and its mesh
bool hctChannelNode::extractObjectSpaceGeometry( MPointArray& vertexArray, MFloatVectorArray& normalArray )
{
    MStatus status;

    // Get the mesh data
    MFnMesh mesh;
    status = mesh.setObject( m_meshPath );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Error creating MFnMesh in hctChannelNode::extractGeometry" );
        return false;
    }

    // Get the object space vertex positions
    status = mesh.getPoints( vertexArray, MSpace::kObject );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Error calling MFnMesh::getPoints in hctChannelNode::extractGeometry" );
        return false;
    }

    // Get the object space normals (flat list, may be multiple entries per vertex)
    status = mesh.getNormals( normalArray, MSpace::kObject );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Error calling MFnMesh::getNormals in hctChannelNode::extractGeometry" );
        return false;
    }

    // Determine which of the vertices each entry in the normals array corresponds to
    MIntArray normalVertexIndex;
    normalVertexIndex.setLength( normalArray.length() );

    MItMeshPolygon polyIter( m_meshPath.node(), &status );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Error constructing MItMeshPolygon in hctChannelNode::extractGeometry" );
        return false;
    }

    for( ; !polyIter.isDone(); polyIter.next() )
    {
        unsigned int polyVertexCount = polyIter.polygonVertexCount(&status);
        if( status != MStatus::kSuccess )
        {
            status.perror( "Error calling MItMeshPolygon::polygonVertexCount() in hctChannelNode::extractGeometry" );
            return false;
        }

        for (unsigned int localVertexIndx=0; localVertexIndx<polyVertexCount; ++localVertexIndx)
        {
            unsigned int globalVertexIndex = polyIter.vertexIndex(localVertexIndx, &status);
            if( status != MStatus::kSuccess )
            {
                status.perror( "Error calling MItMeshPolygon::vertexIndex() in hctChannelNode::extractGeometry" );
                return false;
            }

            unsigned int globalNormalIndex = polyIter.normalIndex (localVertexIndx, &status);
            if( status != MStatus::kSuccess )
            {
                status.perror( "Error calling MItMeshPolygon::normalIndex() in hctChannelNode::extractGeometry" );
                return false;
            }

            normalVertexIndex[globalNormalIndex] = globalVertexIndex;
        }
    }

    // Build an array mapping vertex indices into a list of normal indices
    m_perVertexNormals.resize(vertexArray.length());

    vector<VertexNormals>::iterator iter;
    for(iter = m_perVertexNormals.begin(); iter != m_perVertexNormals.end(); iter++)
    {
        VertexNormals& vn = *iter;
        vn.m_normalIndices.clear();
    }

    MIntArray vertexNormalIndex;
    vertexNormalIndex.setLength(normalVertexIndex.length());

    for (unsigned int ni=0; ni<normalVertexIndex.length(); ++ni)
    {
        int vi = normalVertexIndex[ni];

        VertexNormals& vn = m_perVertexNormals[vi];
        vn.m_normalIndices.push_back(ni);
    }

    // Geometry is fresh now
    m_refreshGeometry = false;

    return true;
}


/*
    To render reasonably fast, the mesh bounding box is divided uniformly into a grid of NGRID^3 cells (NGRID=4),
    and a separate display list is built for each cell, to render the visualization associated with
    the vertices contained in the cell.

    On painting, only display lists containing dirtied vertices (i.e. the ones whose float value has changed) are rebuilt.
*/

MBoundingBox hctChannelNode::getMeshBoundingBox()
{
    // Compute the AABB of the mesh (object space)
    MBoundingBox bbox;
    bbox.clear();
    for( unsigned int i=0; i<m_cachedVertexArray.length(); ++i )
    {
        const MPoint& point = m_cachedVertexArray[i];
        bbox.expand(point);
    }

    return bbox;
}


void hctChannelNode::getBoxDims(float* cellDims, MPoint& boxMin, int Ngrid)
{
    // Mesh bounding box was computed when geometry was refreshed and cached
    const double boxDimsTol = 1.0e-8;
    MPoint boxDims(m_bbox.width()+boxDimsTol, m_bbox.height()+boxDimsTol, m_bbox.depth()+boxDimsTol);

    // Compute cell dimensions (object space)
    for (int a=0; a<3; ++a)
    {
        cellDims[a] = static_cast<float>(boxDims[a])  / static_cast<float>(Ngrid);
    }

    MPoint center = m_bbox.center();
    boxMin = center - (boxDims*0.5);
}


static inline void getCellContaining(const MPoint& point, int& i, int& j, int& k,
                                     const MPoint& boxMin, const float* cellDims, const int Ngrid)
{
    i = static_cast<int>( (point(0)-boxMin(0))/cellDims[0] ); if (i >= Ngrid) i = Ngrid-1; if (i<0) i=0;
    j = static_cast<int>( (point(1)-boxMin(1))/cellDims[1]);  if (j >= Ngrid) j = Ngrid-1; if (j<0) j=0;
    k = static_cast<int>( (point(2)-boxMin(2))/cellDims[2]);  if (k >= Ngrid) k = Ngrid-1; if (k<0) k=0;
}


// Map cell grid indices -> flat NGRID^3 array index
static inline int getCellIndex(int ci, int cj, int ck, const int Ngrid)
{
    return ck + Ngrid*(cj + Ngrid*ci);
}


// Map flat NGRID^3 array index -> cell grid indices
static inline void getCellIndices(int cell, int& ci, int& cj, int& ck, int Ngrid)
{
    ck = cell % Ngrid;
    int m = (cell - ck)/Ngrid;
    cj = m % Ngrid;
    ci = (m - cj)/Ngrid;
}


// Return cell center (object space)
static inline MVector getCellCenter(int cell, const MPoint& boxMin, const float* cellDims, const int Ngrid)
{
    int ci, cj, ck;
    getCellIndices(cell, ci, cj, ck, Ngrid);

    double cX = boxMin(0) + ( 0.5f + static_cast<float>(ci) ) * cellDims[0];
    double cY = boxMin(1) + ( 0.5f + static_cast<float>(cj) ) * cellDims[1];
    double cZ = boxMin(2) + ( 0.5f + static_cast<float>(ck) ) * cellDims[2];

    MVector center(cX, cY, cZ);
    return center;
}


bool distancePredicate(const hctChannelNode::VertexInfo& lhs, const hctChannelNode::VertexInfo& rhs)
{
    return lhs.vDist > rhs.vDist;
}


void hctChannelNode::buildCellList(int cell)
{
    if ( m_cellVertexIndices[cell].length() == 0 )
    {
        m_listActive[cell] = false;
        return;
    }

    // Generate a new display list, deleting the old one. If we can't get one, just deactivate this cell.
    if ( m_cellDisplayLists[cell] > 0 ) glDeleteLists(m_cellDisplayLists[cell], 1);
    m_cellDisplayLists[cell] = glGenLists( 1 );
    if ( m_cellDisplayLists[cell]==0 )
    {
        m_listActive[cell] = false;
        return;
    }

    m_listActive[cell] = true;

    MIntArray& cellVertexIndices = m_cellVertexIndices[cell];

    // Set size to 0 but do not deallocate, to avoid reallocation overhead
    m_vertexInfo.resize(0);

    for( unsigned int i=0; i<cellVertexIndices.length(); ++i )
    {
        VertexInfo vinfo;
        vinfo.vIndex = cellVertexIndices[i];
        m_vertexInfo.push_back(vinfo);
    }

    // If object is non-opaque, sort vertices into descending order of distance from viewer in object space
    if (m_cachedOpacity<1.0f)
    {
        for( unsigned int i=0; i<cellVertexIndices.length(); ++i )
        {
            VertexInfo& vinfo = m_vertexInfo[i];
            MVector diff = m_cachedVertexArray[vinfo.vIndex] - m_cameraObjPos;
            vinfo.vDist = diff.length();
        }
        std::sort(m_vertexInfo.begin(), m_vertexInfo.end(), distancePredicate);
    }

    // Build display list (drawing more distance objects first, if non-opaque)
    vector<VertexInfo>::iterator iter;
    glNewList( m_cellDisplayLists[cell], GL_COMPILE );

    // Don't bother rendering objects with tiny/zero scale.
    const float MIN_DIST_TOL = 1.0e-5f;

    float spherePosColor[3], sphereNegColor[3];
    m_vizColors[SPHERE_POSITIVE].get(spherePosColor);
    m_vizColors[SPHERE_NEGATIVE].get(sphereNegColor);

    switch (m_cachedVizTypeEnum)
    {
        case VIZ_RADIUS:
        {
            for(iter = m_vertexInfo.begin(); iter != m_vertexInfo.end(); iter++)
            {
                VertexInfo& info = *iter;
                int vi = info.vIndex;

                // Transform object space points into world space
                MPoint& pObj = m_cachedVertexArray[vi];
                MPoint pWorld = pObj * m_cachedObjToWorld;

                float radius = m_cachedFloatArray[vi];
                if ( fabs(radius) > MIN_DIST_TOL )
                {
                    if (radius>0.0f)
                    {
                        drawSphere(pWorld, fabs(radius), spherePosColor);
                    }
                    else
                    {
                        drawSphere(pWorld, fabs(radius), sphereNegColor);
                    }
                }
            }
        }
        break;

        case VIZ_NORMAL_DISTANCE:
        {
            vector<int>::iterator nIter;

            for(iter = m_vertexInfo.begin(); iter != m_vertexInfo.end(); iter++)
            {
                VertexInfo& info = *iter;
                int vi = info.vIndex;

                // Transform object space points into world space
                MPoint& pObj = m_cachedVertexArray[info.vIndex];
                MPoint pWorld = pObj * m_cachedObjToWorld;

                VertexNormals& vn = m_perVertexNormals[info.vIndex];
                for(nIter = vn.m_normalIndices.begin(); nIter != vn.m_normalIndices.end(); nIter++)
                {
                    int ni = *nIter;

                    // Transform object space normals into world space
                    MVector nObj(m_cachedNormalArray[ni]);
                    MVector nWorld = nObj.transformAsNormal(m_cachedObjToWorld);

                    float length = m_cachedFloatArray[vi];
                    if ( fabs(length) > MIN_DIST_TOL )
                    {
                        drawNormal(pWorld, nWorld, length);
                    }
                }
            }
        }
        break;

        default:
            break;
    }

    glEndList();
}

bool hctChannelNode::vertexToCell(int i, int& cell)
{
    if ( i < (int)m_vertexCellIndices.length() )
    {
        cell = m_vertexCellIndices[i];
        return true;
    }
    return false;
}


void hctChannelNode::findDirtyCells(MIntArray& dirtyCells, const MFloatArray& floatArray)
{
    dirtyCells.clear();
    int cell;

    for( unsigned int c=0; c<NCELL; ++c)
    {
        m_dirty[c] = false;
    }

    for( unsigned int vi=0; vi<floatArray.length(); ++vi )
    {
        if ( vertexToCell(vi, cell) && m_cachedFloatArray[vi] != floatArray[vi] )
        {
            // Append the cell index if it is not already present (a cell index should appear at most once)
            if (!m_dirty[cell])
            {
                dirtyCells.append(cell);
                m_dirty[cell] = true;
            }
        }
    }
}


void hctChannelNode::rebuildAllCellLists()
{
    MStatus status = rebuildSphereDisplayList();
    if ( status != MStatus::kSuccess ) return;

    MFloatArray floatArray;
    if ( !extractFloats(floatArray) ) return;
    cacheFloatArray(floatArray);

    float cellDims[3];
    MPoint boxMin;
    getBoxDims(cellDims, boxMin, NGRID);

    // Assign each vertex to a cell
    m_vertexCellIndices.clear();
    m_vertexCellIndices.setLength( m_cachedVertexArray.length() );
    for (int c=0; c<NCELL; ++c)
    {
        m_cellVertexIndices[c].clear();
    }

    int ci, cj, ck, cell;
    for( unsigned int vi=0; vi<m_cachedVertexArray.length(); ++vi )
    {
        const MPoint& point = m_cachedVertexArray[vi];
        getCellContaining(point, ci, cj, ck, boxMin, cellDims, NGRID);

        cell = getCellIndex(ci, cj, ck, NGRID);
        m_vertexCellIndices[vi] = cell;
        m_cellVertexIndices[cell].append(vi);
    }

    // Build each cell's display list
    for (int c=0; c<NCELL; ++c)
    {
        buildCellList(c);
    }

    // Main display list needs to be rebuilt as well
    if (m_mainDisplayList != 0)
    {
        glDeleteLists(m_mainDisplayList, 1);
    }
    m_mainDisplayList = 0;
}


void hctChannelNode::rebuildDirtyCellLists()
{
    MStatus status = rebuildSphereDisplayList();
    if ( status != MStatus::kSuccess ) return;

    MFloatArray floatArray;
    extractFloats( floatArray );

    // Compare cached float values with new values
    MIntArray dirtyCells;
    findDirtyCells(dirtyCells, floatArray);

    // Overwrite the cached float values
    cacheFloatArray(floatArray);

    // Rebuild the display list of cells containing vertices with updated floats
    for (int c=0; c<(int)dirtyCells.length(); ++c)
    {
        buildCellList( dirtyCells[c] );
    }

    // If any cells were dirty, the main display list needs to be rebuilt
    if ( dirtyCells.length()>0 )
    {
        if (m_mainDisplayList != 0 )
        {
            glDeleteLists(m_mainDisplayList, 1);
        }
        m_mainDisplayList = 0;
    }
}


const float OPACITY_TOL = 1.0e-3f;

bool hctChannelNode::isTransparent() const
{
    MObject node = thisMObject();
    MDagPath nodePath;
    MDagPath::getAPathTo( node, nodePath );
    MFnDependencyNode nodeFn( nodePath.node() );

    double opacity;
    {
        MFnAttribute attrFn( hctChannelNode::m_opacity );
        nodeFn.findPlug( attrFn.name() ).getValue( opacity );
    }

    return (opacity < 1.0f-OPACITY_TOL);
}


void hctChannelNode::draw( M3dView& view, const MDagPath&, M3dView::DisplayStyle, M3dView::DisplayStatus )
{
    // Find the mesh and extract geometry, if we didn't already
    if (!checkMesh())
    {
        // Get stored channel name, so findMesh() can check we have a mesh containing that channel
        bool refreshAllCells, refreshMainOnly;
        readAttributes(refreshAllCells, refreshMainOnly);

        if ( !findMesh() )
        {
            //MGlobal::displayError( "Failed to find mesh associated with hctChannelNode" );
            // (If this error occurs, better to just not display the error message, since that spams Maya output).
            return;
        }

        if ( !refreshGeometry())
        {
            //MGlobal::displayError( "Failed to read mesh geometry in hctChannelNode" );
            // (If this error occurs, better to just not display the error message, since that spams Maya output).
            return;
        }
    }

    // Get object-To-World transform of the mesh
    MDagPath meshTransformPath = m_meshPath; meshTransformPath.pop();
    MMatrix nodeToWorld = meshTransformPath.inclusiveMatrix();
    MMatrix worldToNode = meshTransformPath.inclusiveMatrixInverse();

    bool refreshAllCells = false;

    if (m_cachedObjToWorld != nodeToWorld)
    {
        // If changed, we need to refresh all display lists
        refreshAllCells = true;
        m_cachedObjToWorld = nodeToWorld;
        m_cachedWorldToObj = worldToNode;
    }

    // Get camera position in object space of mesh (needed to determine rendering order of transparent objects)
    MDagPath cameraPath;
    view.getCamera( cameraPath );

    MDagPath cameraTransformPath( cameraPath );
    cameraTransformPath.pop();

    MStatus status;
    MFnTransform camTransformFn( cameraTransformPath, &status );
    bool validCam = ( status == MStatus::kSuccess );
    if( validCam )
    {
        MPoint CameraWorldPos( camTransformFn.getTranslation( MSpace::kWorld ) );
        m_cameraObjPos = worldToNode * CameraWorldPos;
    }

    // Read attributes, and determine if we need a complete display list refresh (e.g. scale changed),
    // or just a refresh of the main display list (e.g. opacity changed)
    bool refreshMainOnly = false;
    readAttributes(refreshAllCells, refreshMainOnly);

    // If display is not enabled, quit now, so incur negligible overhead if display disabled
    if (!m_cachedDisplayEnable) return;

    // Refresh geometry and regenerate all display lists
    // a) when display is toggled on, or
    // b) each frame while animation is playing back
    if (m_refreshGeometry || m_refreshAllContinually)
    {
        refreshGeometry();
        refreshAllCells = true;
    }

    // Rebuild display lists as needed
    if (refreshAllCells)
    {
        rebuildAllCellLists();
    }
    else
    {
        rebuildDirtyCellLists();
    }

    // Rebuild main display list if anything is dirty
    if ( refreshMainOnly || refreshAllCells || m_mainDisplayList == 0 || !glIsList(m_mainDisplayList) )
    {
        m_mainDisplayList = glGenLists(1);

        beginGL(view);
            glNewList( m_mainDisplayList, GL_COMPILE_AND_EXECUTE );
            renderMain();
            glEndList();
        endGL(view);
    }

    // Otherwise just call the existing main display list
    else
    {
        beginGL(view);
            glCallList( m_mainDisplayList );
        endGL(view);
    }
}


void hctChannelNode::renderMain()
{
    // Render subsequently in world space
    glMultMatrixd( &m_cachedWorldToObj.matrix[0][0] );

    glLightModeli( GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE );
    glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

    glEnable( GL_LIGHTING );
    glEnable( GL_DEPTH_TEST );
    glShadeModel( GL_SMOOTH );

    switch (m_cachedVizTypeEnum)
    {
        case VIZ_RADIUS:
        {
            if (m_cachedOpacity < 1.0f-OPACITY_TOL)
            {
                glEnable( GL_BLEND );
                glBlendFunc( GL_CONSTANT_ALPHA, GL_ONE_MINUS_CONSTANT_ALPHA );
                glBlendColor(1.f,1.f,1.f, (float)m_cachedOpacity);
            }

            glEnable( GL_CULL_FACE );
            glCullFace( GL_BACK );
        }
        break;

        case VIZ_NORMAL_DISTANCE:
        break;
    }

    float cellDims[3];
    MPoint boxMin;
    getBoxDims(cellDims, boxMin, NGRID);

    hkArray<VertexInfo> orderedCells;
    for (int cell=0; cell<NCELL; ++cell)
    {
        VertexInfo vinfo;
        vinfo.vIndex = cell;
        orderedCells.pushBack(vinfo);
    }

    if (m_cachedOpacity < 1.0f-OPACITY_TOL)
    {
        // Render cells in order of decreasing distance from cell center to observer (in object space)
        for (int cell=0; cell<NCELL; ++cell)
        {
            MVector diff = getCellCenter(cell, boxMin, cellDims, NGRID) - m_cameraObjPos;

            VertexInfo& vinfo = orderedCells[cell];
            vinfo.vDist = diff.length();
        }
        std::sort(orderedCells.begin(), orderedCells.end(), distancePredicate);
    }

    for (int i=0; i<orderedCells.getSize(); ++i)
    {
        int cell = orderedCells[i].vIndex;
        if (m_listActive[cell])
        {
            glCallList(m_cellDisplayLists[cell]);
        }
    }
}


void hctChannelNode::beginGL(M3dView& view)
{
    view.beginGL();
    glPushAttrib(GL_ALL_ATTRIB_BITS);
    glPushMatrix();

    //view.setDrawColor( MColor(0.698f, 0.612f, 0.859f) );
    //view.drawText( "CV", MPoint::origin, M3dView::kCenter );

    // Set the main OpenGL light position to the world camera position,
    glPushMatrix();
    glLoadIdentity();
    MDagPath cameraPath;
    view.getCamera( cameraPath );
    MFnTransform transformFn( cameraPath.node() );

    MStatus status;
    MVector camPosition = transformFn.getTranslation( MSpace::kWorld, &status );
    if (MStatus::kSuccess != status)
    {
        return;
    }

    GLfloat lightPosition[] = { (float)camPosition.x, (float)camPosition.y, (float)camPosition.z, 1.0f };
    glLightfv( GL_LIGHT0, GL_POSITION, lightPosition );
    glPopMatrix();

    // Set up the material properties
    glEnable( GL_COLOR_MATERIAL );
    glColorMaterial( GL_FRONT_AND_BACK, GL_DIFFUSE );

    GLfloat mat_specular[] = { 1.0f, 1.0f, 1.0f };
    GLfloat mat_shininess[] = { 60.0f };
    GLfloat mat_emission[] = { 0.3f, 0.3f, 0.3f };
    glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular );
    glMaterialfv( GL_FRONT_AND_BACK, GL_SHININESS, mat_shininess );
    glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, mat_emission );
}

void hctChannelNode::endGL(M3dView& view)
{
    glPopMatrix();
    glPopAttrib();
    view.endGL();
}



//
// Sphere visualization
//
void hctChannelNode::drawSphere( const MPoint& p, float radius, const float* color )
{
    glColor3fv(color);

    glPushMatrix();
    glTranslated(p.x, p.y, p.z);
    glScalef(radius, radius, radius);
    glCallList(m_sphereDisplayList);
    glPopMatrix();
}

//
// Normal distance visualization
//
void hctChannelNode::drawNormal( const MPoint& p, const MFloatVector& n, float dist )
{
    const float shaftColor[3] = {0.82f, 0.79f, 0.86f};
    glColor3fv(shaftColor);

    MPoint end = p + dist*n;

    glBegin( GL_LINES );
    glVertex3d(p.x, p.y, p.z);
    glVertex3d(end.x, end.y, end.z);
    glEnd();

    float pinPosColor[3], pinNegColor[3];
    m_vizColors[PIN_POSITIVE].get(pinPosColor);
    m_vizColors[PIN_NEGATIVE].get(pinNegColor);

    if (dist>0.0f)
    {
        drawSphere(end, 0.05f*fabs(dist), pinPosColor);
    }
    else
    {
        drawSphere(end, 0.05f*fabs(dist), pinNegColor);
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
