// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Nodes/Capsule/hctCapsuleNode.h>

MObject hctCapsuleNode::m_radius;
MObject hctCapsuleNode::m_height;
MObject hctCapsuleNode::m_taper;
MObject hctCapsuleNode::m_axis;
MObject hctCapsuleNode::m_outputMesh;


MStatus hctCapsuleNode::initialize()
{
    MStatus status = MStatus::kSuccess;

    MFnUnitAttribute    unitFn;
    MFnNumericAttribute numFn;
    MFnTypedAttribute   typedFn;

    m_radius = unitFn.create( "radius", "r", MFnUnitAttribute::kDistance, MDistance::uiToInternal( 1.0 ), &status );
    unitFn.setMin( 0.0 );
    unitFn.setSoftMax( 50.0f );
    unitFn.setKeyable(true);
    addAttribute( m_radius );

    m_height = unitFn.create( "height", "h", MFnUnitAttribute::kDistance, MDistance::uiToInternal( 1.0 ), &status );
    unitFn.setMin( 0.0 );
    unitFn.setSoftMax( 50.0f );
    unitFn.setKeyable(true);
    addAttribute( m_height );

    m_taper = numFn.create( "taper", "t", MFnNumericData::kDouble, 1.0, &status );
    numFn.setMin( 0.0 );
    numFn.setSoftMax( 5.0f );
    numFn.setKeyable(true);
    addAttribute( m_taper );

    MObject axisx = unitFn.create( "axisX", "aX", MFnUnitAttribute::kDistance, 0.0);
    MObject axisy = unitFn.create( "axisY", "aY", MFnUnitAttribute::kDistance, 1.0);
    MObject axisz = unitFn.create( "axisZ", "aZ", MFnUnitAttribute::kDistance, 0.0);
    m_axis = numFn.create( "axis", "a", axisx, axisy, axisz );
    addAttribute( m_axis );

    m_outputMesh = typedFn.create( "outputMesh", "mesh", MFnData::kMesh );
    typedFn.setHidden( true );
    addAttribute( m_outputMesh );

    attributeAffects( m_radius, m_outputMesh );
    attributeAffects( m_taper,  m_outputMesh );
    attributeAffects( m_height, m_outputMesh );
    attributeAffects( m_axis,   m_outputMesh );

    return status;
}


void hctCapsuleNode::postConstructor()
{
    MObject thisObj = thisMObject();
    MFnDependencyNode nodeFn( thisObj );

    nodeFn.findPlug( m_radius ).setValue( MDistance::uiToInternal( 1.0 ) );
    nodeFn.findPlug( m_taper ).setValue( 1.0 );
    nodeFn.findPlug( m_height ).setValue( MDistance::uiToInternal( 1.0 ) );
    nodeFn.findPlug( m_axis ).child(1).setValue( MDistance::uiToInternal( 1.0 ) );
}


// Return an inclusive range of linear values [phi0, phi1] given indices in inclusive range [i0, i1]
static inline double _interpPhi(double phi0, double phi1, int i, int i0, int i1)
{
    return phi0 + double(i - i0) * (phi1 - phi0) / double(i1 - i0);
}

MStatus hctCapsuleNode::compute( const MPlug& plug, MDataBlock& data )
{
    if( plug == m_outputMesh )
    {
        // Read input values (internal units)
#if MAYA_API_VERSION >= 600
        const double radius = data.inputValue( m_radius ).asDistance().value();
        const double taper = data.inputValue( m_taper ).asDouble();
        double height = data.inputValue( m_height ).asDistance().value();
#else
        const double radius = data.inputValue( m_radius ).asDouble();
        const double taper = data.inputValue( m_taper ).asDouble();
        double height = data.inputValue( m_height ).asDouble();
#endif

        MVector axis = data.inputValue( m_axis ).asVector();
        const short stacks = 2;
        const short slices = 16;

        double bigRadius;
        double smallRadius;

        double startRadius = radius;
        double endRadius = taper * startRadius;

        if (startRadius > endRadius)
        {
            bigRadius = startRadius;
            smallRadius = endRadius;
            axis *= -1.0;
        }
        else
        {
            bigRadius = endRadius;
            smallRadius = startRadius;
        }

        double TOL = 10.0 * double(HK_REAL_EPSILON);

        double radiusDiff = bigRadius - smallRadius + TOL;
        if (height < radiusDiff) height = radiusDiff;
        const double Delta = radiusDiff * bigRadius / height;

        // Create the mesh data
        MFnMeshData dataCreator;
        MObject newOutputData = dataCreator.create();
        {
            MPointArray slice( slices );
            {
                for( int i = 0; i < slices; ++i )
                {
                    slice[i] = MPoint( cos(2*PI*i / slices), 0, sin(2*PI*i / slices) );
                }
            }

            const int completeHemisphereStacks = hkMath::max2(4, int(double(slices)/2 * bigRadius / (bigRadius + Delta)));
            int partialHemisphereStacks = hkMath::max2(3, int(double(slices)/2 * Delta / (bigRadius + Delta)));
            if (fabs(taper-1.f) < TOL) partialHemisphereStacks = 0;

            const int capStacks = completeHemisphereStacks + partialHemisphereStacks;
            const int totalStacks = capStacks + stacks + capStacks;

            const int numPoints = (stacks + 1 + 2*capStacks) * slices;
            MPointArray points (numPoints);

            double sinTheta = (bigRadius - smallRadius) / height;
            double cosTheta = sqrt(1.0f - sinTheta*sinTheta);

            double theta = asin(sinTheta);
            double halfPi = 0.5f*PI;
            double halfPiPlusTheta = halfPi + theta;
            double halfPiMinusTheta = halfPi - theta;
            double dphi_big = theta / (double(partialHemisphereStacks) + TOL);
            double dphi_small = halfPiMinusTheta / capStacks;

            double h2 = 0.5f * height;
            double hmin = -h2 - smallRadius * sinTheta;
            double hmax = h2 - bigRadius * sinTheta;
            double rmax = bigRadius * cosTheta;
            double rmin = smallRadius * cosTheta;

            // Create the points
            {
                for( int i = 0; i < totalStacks+1; ++i )
                {
                    double r;
                    double h;

                    // big cap hemisphere
                    if( i < completeHemisphereStacks )
                    {
                        double phi = _interpPhi(0.0, halfPi, i, 0, completeHemisphereStacks-1);
                        r = bigRadius * sin(phi);
                        h = h2 + bigRadius * cos(phi);
                    }

                    // big cap partial hemisphere
                    else if ( i < capStacks )
                    {
                        double phi = _interpPhi(halfPi+dphi_big, halfPiPlusTheta-dphi_big, i, completeHemisphereStacks, capStacks-1);
                        r = bigRadius * sin(phi);
                        h = h2 + bigRadius * cos(phi);
                    }

                    // cone
                    else if( i < capStacks + stacks + 1 )
                    {
                        double alpha = double(i-capStacks) / stacks;
                        const double epsilon = 0.02f; // Needed to avoid degenerate triangles when radii are equal
                        double beta = epsilon + (1.0f - 2.0f*epsilon)* alpha;
                        r = beta * rmin + (1.0f - beta) * rmax;
                        h = beta * hmin + (1.0f - beta) * hmax;
                    }

                    // small cap partial hemisphere
                    else
                    {
                        int i0 = capStacks + stacks + 1;
                        int i1 = totalStacks;

                        double phi = _interpPhi(halfPiMinusTheta-dphi_small, 0.0f, i, i0, i1);
                        r = smallRadius * sin(phi);
                        h = -(h2 + smallRadius * cos(phi));
                    }

                    int cur = i*slices;
                    for( int j = 0; j < slices; ++j )
                    {
                        points[cur+j] = slice[j] * r;
                        points[cur+j].y = h;
                    }
                }
            }

            // Transform the points along the axis
            MQuaternion quat( MVector::yAxis, axis.normal() );
            MVector v;
            for( int i=0; i<numPoints; ++i )
            {
                v = points[i];
                points[i] = v.rotateBy( quat );
            }

            // Create the faces
            int numQuads = totalStacks * slices;
            MIntArray faceCounts( numQuads, 4 );
            MIntArray faceConnects( numQuads*4 );
            {
                for( int i = 0; i < totalStacks; ++i )
                {
                    int st = i*slices;
                    for( int j = 0; j < slices; ++j )
                    {
                        int cur = 4*(st + j);
                        faceConnects[cur] = st + j;
                        faceConnects[cur+1] = st + (j+1)%slices;
                        faceConnects[cur+2] = st + slices + (j+1)%slices;
                        faceConnects[cur+3] = st + slices + j;
                    }
                }
            }

            MFnMesh meshFn;
            meshFn.create( numPoints, numQuads, points, faceCounts, faceConnects, newOutputData );
        }

        // Set output value
        MDataHandle outputHandle = data.outputValue( m_outputMesh );
        outputHandle.set( newOutputData );
        data.setClean( plug );

        return MStatus::kSuccess;
    }

    return MStatus::kUnknownParameter;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
