// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <ContentTools/Maya/MayaSceneExport/Importer/MeshImport/hctMayaMeshBuilder.h>

// Additional includes
#include <maya/MFnSingleIndexedComponent.h>


hctMayaMeshBuilder::hctMayaMeshBuilder(const hkMeshBody& meshBody, MFnMesh& meshFn, MDagModifier& dagMod, const char* meshName, hkMatrix4& transform, MObject& parentNode, bool forceUseOfRotatePivot, const char* nodeName, hkxScene* sceneGraph)
:   m_meshFn(meshFn)
,   m_meshName(meshName)
,   m_parentNode(parentNode)
,   m_dagMod(dagMod)
,   m_meshBody(meshBody)
,   m_sceneGraph(sceneGraph)
{
    {
        m_reverseEffectOfRotatePivot = forceUseOfRotatePivot || (parentNode != MObject::kNullObj); // assume it is on (if parent node given)
        if ( m_reverseEffectOfRotatePivot )
        {
            // Double check with the scene options.
            if (!forceUseOfRotatePivot)
            {
                MCommandResult res;
                MStatus cmdStat = MGlobal::executeCommand( MString("hkProcPhysics_includePivotInTransforms()"), res );
                if (cmdStat == MStatus::kSuccess )
                {
                    int resInt;
                    if ( res.getResult(resInt) == MStatus::kSuccess)
                    {
                        m_reverseEffectOfRotatePivot = (resInt != 0);
                    }
                }
            }

            if (m_reverseEffectOfRotatePivot)
            {
                MStatus cmdStat;
                MFnTransform parentTransformFn( parentNode, &cmdStat );
                if( cmdStat == MStatus::kSuccess )
                {
                    m_rotatePivotOffset = parentTransformFn.rotatePivot( MSpace::kPreTransform );
                }
                else
                {
                    m_reverseEffectOfRotatePivot = false;
                }
            }
        }

        // When we export wrt rotatePivot, we alter the orig mesh transform
        // such that the local space is wrt to the pivot. So we need to reverse that to preserve the pivots
        // But we will only alter the current transform if one given, No transform == parent space, so already taken care of
        if (m_reverseEffectOfRotatePivot && !transform.isApproximatelyIdentity(hkSimdReal::fromFloat(1e-3f)))
        {
            //hkMatrix4 pivotMatrix; pivotMatrix.setIdentity();
            //pivotMatrix.getColumn(3).set( -(float)rotatePivotOffset(0), -(float)rotatePivotOffset(1), -(float)rotatePivotOffset(2), 1.0f );
            //transform.mul( pivotMatrix );
        }

        // If transform is identity, and we have a parentNode, then just attach direct to parent
        bool identityTransform = false; //CK: For now, always make a trans node.
        //transform.isApproximatelyEqual( hkMatrix4::getIdentity() );

        bool createTransformNode = (!identityTransform) || (parentNode == MObject::kNullObj);
        if ( createTransformNode )
        {
            MStatus status;
            m_transformNode = dagMod.createNode( MString("transform"), parentNode, &status );
            if ( status != MStatus::kSuccess)
            {
                return;
            }

            if ( nodeName )
            {
                dagMod.renameNode(m_transformNode, MString(nodeName) );
            }

            dagMod.doIt();

            // Set the transform
            {
                MFnTransform tFn(m_transformNode);

                HK_ALIGN16(float f[4][4]);
                transform.get4x4ColumnMajor((float*)f); // expects Translation in [3][0], [3][1] etc., so this is effectively a transpose
                MMatrix mt( f );
                tFn.set( mt );

                if ( m_reverseEffectOfRotatePivot )
                {
                    tFn.setRotatePivot(m_rotatePivotOffset, MSpace::kPreTransform, false );
                    tFn.setScalePivot(m_rotatePivotOffset, MSpace::kPreTransform, false );
                }

                // To compensate for scaling on the original node (or its hierarchy) we will "undo" the scaling on the newly
                // created children. By scaling with the inverse scaling factor of the parent node this essentially brings the
                // new node's scale back to 1/1/1.
                // Note that this only works as long as the grandparent nodes (or above) are uniformly scaled! The original node
                // can still be non-uniformly scaled, though.
                {
                    MDagPath parentNodePath = MDagPath::getAPathTo(parentNode);
                    MTransformationMatrix worldFromParentMatrix(parentNodePath.inclusiveMatrix());
                    double scaleFactors[3];
                    worldFromParentMatrix.getScale(scaleFactors, MSpace::kWorld);

                    hkReal scaleEps = 0.0001f; // HK_REAL_EPSILON won't work here
                    if ((hkMath::fabs(hkReal(scaleFactors[0]-scaleFactors[1])) > scaleEps) ||
                        (hkMath::fabs(hkReal(scaleFactors[0]-scaleFactors[2])) > scaleEps) ||
                        (hkMath::fabs(hkReal(scaleFactors[1]-scaleFactors[2])) > scaleEps)   )
                    {
                        const char* parentNodeName = "";
                        if (!parentNode.isNull() && parentNode.hasFn(MFn::kTransform))
                        {
                            MFnTransform parentTransformNode(parentNode);
                            parentNodeName = parentTransformNode.name().asChar();
                        }
                        else if (nodeName)
                        {
                            parentNodeName = nodeName;
                        }
                        HK_WARN_ALWAYS(0xaf41e152, "'" << parentNodeName << "' : The non-uniform scale (" << scaleFactors[0] << ", " << scaleFactors[1] << ", " << scaleFactors[2] << ") on this node can cause graphical distortions of the Fracture Pieces. If that's the case, you might consider baking the scaling into the mesh itself (by using 'Freeze Transformations > Scale' in the 'Modify' menu).");
                    }

                    if ((scaleFactors[0] > 0.0f) && (scaleFactors[1] > 0.0f) && (scaleFactors[2] > 0.0f))
                    {
                        scaleFactors[0] = 1.0f / scaleFactors[0];
                        scaleFactors[1] = 1.0f / scaleFactors[1];
                        scaleFactors[2] = 1.0f / scaleFactors[2];
                        tFn.setScale(scaleFactors);
                    }
                }
            }
        }
        else
        {
            //XX May not undo now..
            m_transformNode = parentNode;
        }
    }

    {
        MDagPath parentMeshPath = MDagPath::getAPathTo( parentNode );
        {
            unsigned int numChildren = parentMeshPath.childCount();
            for( unsigned int i=0; i<numChildren; ++i )
            {
                const MFn::Type apiType = parentMeshPath.child(i).apiType();
                if( apiType == MFn::kMesh || apiType==MFn::kNurbsSurface )
                {
                    parentMeshPath.push( parentMeshPath.child(i) );
                    break;
                }
            }
        }
        if( parentMeshPath.isValid() )
        {
            const MFn::Type apiType = parentMeshPath.apiType();
            if( apiType == MFn::kMesh)
            {
                MFnMesh meshFnParent(parentMeshPath);
                meshFnParent.getColorSetNames(m_parentColorSets);
                meshFnParent.getUVSetNames(m_parentUvSets);
            }
            else if ( apiType == MFn::kNurbsSurface )
            {
                // Can nurbs have named sets? API doesn't seem to expose them
            }
        }
    }

    m_vertices.clear();
    m_triangles.clear();
}


// ===============================================================================================
// MESH
// ===============================================================================================

hctModelerMeshBuilder::VertexId hctMayaMeshBuilder::addMeshVertexDelayed(const hkVector4& position)
{
    MPoint mPoint;
    {
        mPoint.x = double(position(0));
        mPoint.y = double(position(1));
        mPoint.z = double(position(2));
        mPoint.w = 1.0;

        if ( m_reverseEffectOfRotatePivot )
        {
            mPoint += m_rotatePivotOffset; // was subtracted on export, so we add it here
        }
    }

    int vertexId = m_vertices.length();

    m_vertices.append(mPoint);

    return vertexId;
}


int hctMayaMeshBuilder::addMeshTriangleDelayed(VertexId positionIndex0, VertexId positionIndex1, VertexId positionIndex2)
{
    int triangleId = m_vertices.length() / 3;

    m_triangles.append(positionIndex0);
    m_triangles.append(positionIndex1);
    m_triangles.append(positionIndex2);

    return triangleId;
}


void hctMayaMeshBuilder::realizeDelayedMesh()
{
    if ( m_meshFn.object() != MObject::kNullObj )
    {
        HK_WARN_ALWAYS(0xaf41e122, "Havok Maya Mesh Builder - Mesh has already been created.");
        return;
    }

    int numTriangles = m_triangles.length() / 3;

    MIntArray triangleVertexCounts(numTriangles, 3); // fill array with '3's (as 3 vertices per polygon/triangle)

    MStatus status;
    m_meshFn.create(m_vertices.length(), numTriangles, m_vertices, triangleVertexCounts, m_triangles, m_transformNode, &status);

#if defined(HK_DEBUG) && (MAYA_API_VERSION >= 200800)

    if ( status == MStatus::kSuccess )
    {
        MIntArray vertCounts;
        MIntArray vertIndices;
        m_meshFn.getVertices(vertCounts, vertIndices);

        if ((int)vertCounts.length() != numTriangles)
        {
            HK_WARN_ALWAYS(0x272ab59, "Havok Maya Mesh Builder - Maya changed the number of triangles on import from " << numTriangles << " to " << vertCounts.length() << ". Vertex attributes may be wrong on imported mesh.");
        }

        if ((int)vertIndices.length() != numTriangles*3)
        {
            HK_WARN_ALWAYS(0x272ab58, "Havok Maya Mesh Builder - Maya changed the number of face indices on import from " << numTriangles*3 << " to " << vertIndices.length() << ". Vertex attributes may be wrong on imported mesh.");
        }

        int j=0;
        for (unsigned int i=0; i<vertCounts.length(); i++)
        {
            if ( vertCounts[i] != 3 )
            {
                HK_WARN_ALWAYS(0x272ab5a, "Havok Maya Mesh Builder - Changed triangle " << i << " to have " << vertCounts[i] << " vertices:");
                for (int k=0; k<vertCounts[i]; k++)
                {
                    HK_WARN_ALWAYS(0x272ab5b, "Vertex " << k << " = " << vertIndices[j+k]);
                }
            }
            j += vertCounts[i];
        }

        MPointArray vertValues;
        m_meshFn.getPoints(vertValues);

        if ( int(vertValues.length()) != m_vertices.length() )
        {
            HK_WARN_ALWAYS(0x272ab57, "Havok Maya Mesh Builder - Maya changed the number of vertices on import from " <<  m_vertices.length() << " to " << vertValues.length() << ". Vertex attributes may be wrong on imported mesh.");
        }
    }

#endif

    if ( m_meshName )
    {
        m_dagMod.renameNode(m_meshFn.object(), m_meshName);
        m_dagMod.doIt();
    }

}


// ===============================================================================================
// VERTEX COLORS
// ===============================================================================================

void hctMayaMeshBuilder::selectColorChannel(int channel)
{
    m_vertexColors.clear();
    m_vertexColorTriangles.clear();

    if ( m_parentColorSets.length() > (unsigned int)channel )
    {
        // Inherit parent's Color Channel name.
        m_colorSet = m_parentColorSets[channel];
    }
    else
    {
        // Create a generic Color Channel name.
        MString channelAsString;
        channelAsString.set(channel+1);
        m_colorSet = "colorSet" + channelAsString;
    }

    MStatus status = m_meshFn.createColorSet(m_colorSet, &m_dagMod);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba651d, "Havok Maya Mesh Builder - createColorSet() failed. Maya reports : " << status.errorString().asChar());
        return;
    }

    status = m_meshFn.setCurrentColorSetName(m_colorSet, &m_dagMod);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba651e, "Havok Maya Mesh Builder - setCurrentColorSetName() failed. Maya reports : " << status.errorString().asChar());
        return;
    }
}


hctModelerMeshBuilder::ColorId hctMayaMeshBuilder::addVertexColorImmediate(float r, float g, float b)
{
    int colorId = m_meshFn.numColors();
    MColor color(r, g, b);
    MStatus status = m_meshFn.setColor(colorId, color, &m_colorSet);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba651c, "Havok Maya Mesh Builder - setColor() failed. Maya reports : " << status.errorString().asChar());
        return -1;
    }
    return colorId;
}


hctModelerMeshBuilder::ColorId hctMayaMeshBuilder::addVertexColorDelayed(float r, float g, float b)
{
    int colorId = m_vertexColors.length();

    MColor color(r, g, b);
    m_vertexColors.append(color);

    return colorId;
}


void hctMayaMeshBuilder::realizeDelayedVertexColors()
{
    MStatus status = m_meshFn.setColors(m_vertexColors, &m_colorSet);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba651b, "Havok Maya Mesh Builder - setColors() failed. Maya reports : " << status.errorString().asChar());
    }
}


int hctMayaMeshBuilder::addVertexColorTriangleImmediate(int triangleIndex, ColorId colorIndexVertex0, ColorId colorIndexVertex1, ColorId colorIndexVertex2)
{
    MStatus status0 = m_meshFn.assignColor(triangleIndex, 0, colorIndexVertex0, &m_colorSet);
    MStatus status1 = m_meshFn.assignColor(triangleIndex, 1, colorIndexVertex1, &m_colorSet);
    MStatus status2 = m_meshFn.assignColor(triangleIndex, 2, colorIndexVertex2, &m_colorSet);

    if ( status0 != MStatus::kSuccess || status1 != MStatus::kSuccess || status2 != MStatus::kSuccess )
    {
        MStatus status = (status0 != MStatus::kSuccess) ? status0 : ((status1 != MStatus::kSuccess) ? status1 : status2);
        HK_WARN_ALWAYS(0xabba651a, "Havok Maya Mesh Builder - assignColor() failed. Maya reports : " << status.errorString().asChar());
        return -1;
    }
    return triangleIndex;
}


int hctMayaMeshBuilder::addVertexColorTriangleDelayed(ColorId colorIndexVertex0, ColorId colorIndexVertex1, ColorId colorIndexVertex2)
{
    int triangleId = m_vertexColorTriangles.length() / 3;

    m_vertexColorTriangles.append(colorIndexVertex0);
    m_vertexColorTriangles.append(colorIndexVertex1);
    m_vertexColorTriangles.append(colorIndexVertex2);

    return triangleId;
}


void hctMayaMeshBuilder::realizeDelayedVertexColorTriangles()
{
    MStatus status = m_meshFn.assignColors(m_vertexColorTriangles, &m_colorSet);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba9af0, "Havok Maya Mesh Builder - assignColors() failed. Maya reports : " << status.errorString().asChar());
    }
}


// ===============================================================================================
// TEXTURES
// ===============================================================================================

void hctMayaMeshBuilder::selectUvChannel(int channel)
{
#if MAYA_API_VERSION < 850
    if ( channel > 0 )
    {
        return; // Only one texture channel supported in older versions of Maya. Abort...
    }
#endif

    m_uValues.clear();
    m_vValues.clear();
    m_vertexUvTriangles.clear();

    MStatus status;

    // Maya automatically names the UV channels 'mapX'. So we do the same.
    MString channelAsString; channelAsString.set(channel+1);
    MString defaultUvSet = "map" + channelAsString;

    if ( m_parentUvSets.length() > (unsigned int)channel )
    {
        // Inherit parent's UV Channel name.
        m_uvSet = m_parentUvSets[channel];

        // Rename first UV channel to the same name as the parent's first UV channel.
        // renameUVSet() only succeeds if oldName != newName. Hence the check.
        if ( (channel == 0) && (m_uvSet != defaultUvSet) )
        {
            status = m_meshFn.renameUVSet(defaultUvSet, m_uvSet, &m_dagMod);
            if ( status != MStatus::kSuccess )
            {
                HK_WARN_ALWAYS(0xabba46e7, "Havok Maya Mesh Builder - renameUVSet(" << defaultUvSet.asChar() << ", " << m_uvSet.asChar() << ") failed. Maya reports : " << status.errorString().asChar());
            }
        }
    }
    else
    {
        // Use a genetic name for the UV Channel, following Maya's standard.
        m_uvSet = defaultUvSet;
    }

#if MAYA_API_VERSION >= 850
    // No need to create channel 0 as it already exists.
    if ( channel > 0 )
    {
        m_uvSet = m_meshFn.createUVSetWithName(m_uvSet);
    }
#endif

    status = m_meshFn.setCurrentUVSetName(m_uvSet);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba621e, "Havok Maya Mesh Builder - setCurrentUVSetName() failed. Maya reports : " << status.errorString().asChar());
        return;
    }
}


hctModelerMeshBuilder::UvId hctMayaMeshBuilder::addVertexUvImmediate(float u, float v)
{
    int uvId = m_meshFn.numUVs();
    MStatus status = m_meshFn.setUV(uvId, u, v, &m_uvSet);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba3732, "Havok Maya Mesh Builder - setUV() failed. Maya reports : " << status.errorString().asChar());
        return -1;
    }
    return uvId;
}


hctModelerMeshBuilder::UvId hctMayaMeshBuilder::addVertexUvDelayed(float u, float v)
{
    int uvId = m_uValues.length();

    m_uValues.append(u);
    m_vValues.append(v);

    return uvId;
}


void hctMayaMeshBuilder::realizeDelayedVertexUvs()
{
    MStatus status = m_meshFn.setUVs(m_uValues, m_vValues, &m_uvSet);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba3731, "Havok Maya Mesh Builder - setUVs() failed. Maya reports : " << status.errorString().asChar());
    }
}


int hctMayaMeshBuilder::addVertexUvTriangleImmediate(int triangleIndex, UvId uvIndexVertex0, UvId uvIndexVertex1, UvId uvIndexVertex2)
{
    MStatus status0 = m_meshFn.assignUV(triangleIndex, 0, uvIndexVertex0, &m_uvSet);
    MStatus status1 = m_meshFn.assignUV(triangleIndex, 1, uvIndexVertex1, &m_uvSet);
    MStatus status2 = m_meshFn.assignUV(triangleIndex, 2, uvIndexVertex2, &m_uvSet);

    if ( status0 != MStatus::kSuccess || status1 != MStatus::kSuccess || status2 != MStatus::kSuccess )
    {
        MStatus status = (status0 != MStatus::kSuccess) ? status0 : ((status1 != MStatus::kSuccess) ? status1 : status2);
        HK_WARN_ALWAYS(0xabba6519, "Havok Maya Mesh Builder - assignUV() failed. Maya reports : " << status.errorString().asChar());
        return -1;
    }
    return triangleIndex;
}


int hctMayaMeshBuilder::addVertexUvTriangleDelayed(UvId uvIndexVertex0, UvId uvIndexVertex1, UvId uvIndexVertex2)
{
    int triangleId = m_vertexUvTriangles.length() / 3;

    m_vertexUvTriangles.append(uvIndexVertex0);
    m_vertexUvTriangles.append(uvIndexVertex1);
    m_vertexUvTriangles.append(uvIndexVertex2);

    return triangleId;
}


void hctMayaMeshBuilder::realizeDelayedVertexUvTriangles()
{
    int numTriangles = m_vertexUvTriangles.length() / 3;

    MIntArray triangleVertexCounts(numTriangles, 3); // fill array with '3's (as we have 3 vertices per polygon/triangle)

    MStatus status = m_meshFn.assignUVs(triangleVertexCounts, m_vertexUvTriangles, &m_uvSet);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabba3733, "Havok Maya Mesh Builder - assignUVs() failed. Maya reports : " << status.errorString().asChar());
    }
}


// ===============================================================================================
// NORMALS
// ===============================================================================================

hctModelerMeshBuilder::NormalId hctMayaMeshBuilder::addNormalDelayed(const hkVector4& normalIn)
{
    int normalId = m_normals.length();

    MVector normal(normalIn(0), normalIn(1), normalIn(2));
    m_normals.append(MVector(normalIn(0), normalIn(1), normalIn(2)));

    return normalId;
}


int hctMayaMeshBuilder::addNormalTriangleDelayed(NormalId triangleVertex0, NormalId triangleVertex1, NormalId triangleVertex2)
{
    if ( (triangleVertex0 >= int(m_normals.length())) || (triangleVertex1 >= int(m_normals.length())) || (triangleVertex2 >= int(m_normals.length())) )
    {
        int normalId = (triangleVertex0 >= int(m_normals.length())) ? triangleVertex0 : ((triangleVertex1 >= int(m_normals.length())) ? triangleVertex1 : triangleVertex2);
        HK_WARN_ALWAYS(0xabbabaef, "Havok Maya Mesh Builder - addNormalTriangleDelayed() : Referenced normal id " << normalId << " has not been registered.");
        return -1;
    }

    int triangleId = m_triangleVertexNormals.length() / 3;

    m_triangleVertexNormals.append(m_normals[triangleVertex0]);
    m_triangleVertexNormals.append(m_normals[triangleVertex1]);
    m_triangleVertexNormals.append(m_normals[triangleVertex2]);

    return triangleId;
}


void hctMayaMeshBuilder::realizeDelayedNormals()
{
    int numTriangles = m_triangleVertexNormals.length() / 3;

    MIntArray faceList(numTriangles*3);

    for (int ti = 0; ti < numTriangles; ti++)
    {
        faceList[ti*3  ] = ti;
        faceList[ti*3+1] = ti;
        faceList[ti*3+2] = ti;
    }

    MStatus status = m_meshFn.setFaceVertexNormals(m_triangleVertexNormals, faceList, m_triangles, MSpace::kObject);
    if ( status != MStatus::kSuccess )
    {
        HK_WARN_ALWAYS(0xabbabbef, "Havok Maya Mesh Builder - setFaceVertexNormals() failed. Maya reports : " << status.errorString().asChar());
    }
}


// ===============================================================================================
// MATERIALS
// ===============================================================================================

void hctMayaMeshBuilder::registerMaterial(const char* materialName)
{
    MString shadingGroupName("initialShadingGroup");
    if ( materialName )
    {
        shadingGroupName = materialName;
    }

    MObject shadingGroup;
    {
        MSelectionList shadingGroupSelection;
        shadingGroupSelection.add(shadingGroupName);
        shadingGroupSelection.getDependNode(0, shadingGroup);
    }

    if ( shadingGroup == MObject::kNullObj )
    {
        // Shading Group does not yet exist in scene and we need to create it.
        // Note that we will not add any triangles to the Shading Group yet.

        MStatus status;
        MSelectionList triangleIndicesList;
        MFnSet shadingGroupFn;
        shadingGroupFn.create(triangleIndicesList, MFnSet::kRenderableOnly, &status);
        m_dagMod.renameNode(shadingGroupFn.object(), shadingGroupName + MString("SG"));
        m_dagMod.doIt();

        HK_WARN_ALWAYS(0xabba1030, "Havok Maya Mesh Builder - Found a material that has no matching Maya Shading Group (yet). Creating a new one will not fully import material '" << materialName << "'.");

        
    }
}


//
//  Locate uvChooser by id, given a texture node. Returns true if found, false otherwise
//
static hkBool locateUvChooserByIdOnTexture(MObject& textureObject, const int uvChooserId, MObject& uvChooserOut)
{
    // Go back until we find an UV chooser
    MStatus status;
    MObject node = textureObject;
    while(node.apiType() != MFn::kUvChooser )
    {
        MFnDependencyNode dgFn(node);
        MPlug plug = dgFn.findPlug(MString("uvCoord"), &status);
        if(status != MStatus::kSuccess || !plug.isConnected())
        {
            break;  // Error, stop looking!
        }

        // Get the connection - there can be at most one input to a plug
        MPlugArray connections;
        plug.connectedTo(connections, true, false);
        if(connections.length() <= 0)
        {
            break;  // No more connections, stop looking!
        }

        // Go back one node
        node = connections[ 0].node();
    }

    // See if we got an UV chooser
    if (node.apiType() != MFn::kUvChooser)
    {
        return false;   // No uvChooser nodes found!
    }

    // Check the custom property
    const MString fullPropName("havokUniqueUvChooserId");
    MFnDependencyNode dgFn(node);
    if ( !dgFn.hasAttribute(fullPropName) )
    {
        return false;   // uvChooser hasn't got our custom property!
    }

    // Property found, get its value
    {
        MPlug plug = dgFn.findPlug(fullPropName, &status);
        if ( status != MStatus::kSuccess )
        {
            return false;   // Failed to read property value
        }

        // Get value set on the attribute
        int plugIntValue = -1;
        status = plug.getValue(plugIntValue);
        if ( status != MStatus::kSuccess )
        {
            return false;   // Failed to get the plug value
        }
        if ( plugIntValue == uvChooserId )
        {
            // Found, initialize output
            uvChooserOut = node;
            return true;
        }
    }

    // Not found!
    return false;
}


//
//  Locate uvChooser by id, given a shader node. Returns true if found, false otherwise
//
static hkBool locateUvChooserByIdOnShader(MObject& shaderObject, const int uvChooserId, MObject& uvChooserOut)
{
    // Search for texture nodes
    MStatus status;
    MItDependencyGraph itDG(shaderObject, MFn::kFileTexture, MItDependencyGraph::kUpstream);

    for (; !itDG.isDone(); itDG.next())
    {
        MObject shaderTextureNode = itDG.thisNode();
        MFnDependencyNode mayaTexture(shaderTextureNode);

        MString textureFile;
        MPlug texFilePlug = mayaTexture.findPlug("fileTextureName", &status);
        if( status == MStatus::kSuccess )
        {
            // Locate uvChooser in the texture node
            MObject textureObject = itDG.thisNode();
            if ( locateUvChooserByIdOnTexture(textureObject, uvChooserId, uvChooserOut) )
            {
                return true;    // Found!
            }
        }
    }

    // Not found
    return false;
}


//
//  Attempts to locate an uvChooser DG node by looking for a custom property, holding a unique int ID
//
static hkBool locateUvChooserById(MObject& shadingEngine, const int uvChooserId, MObject& uvChooserOut)
{
    MFnDependencyNode shadingEngineFn(shadingEngine);

    // Get the shader nodes
    MObject surfaceShaderNode;
    MObject displacementShaderNode;
    MStatus status;
    {
        MPlug surfaceShaderPlug = shadingEngineFn.findPlug("surfaceShader");
        if( !surfaceShaderPlug.isNull() )
        {
            MPlugArray connectedPlugs;
            surfaceShaderPlug.connectedTo(connectedPlugs, true, false, &status);
            if( status == MStatus::kSuccess && connectedPlugs.length() == 1 )
            {
                surfaceShaderNode = connectedPlugs[0].node();
            }
        }

        MPlug displacementShaderPlug = shadingEngineFn.findPlug("displacementShader");
        if( !displacementShaderPlug.isNull() )
        {
            MPlugArray connectedPlugs;
            displacementShaderPlug.connectedTo( connectedPlugs, true, false, &status );
            if( status == MStatus::kSuccess && connectedPlugs.length() == 1 )
            {
                displacementShaderNode = connectedPlugs[0].node();
            }
        }
    }

    // Need at least one of the shaders
    if( surfaceShaderNode.isNull() && displacementShaderNode.isNull() )
    {
        return false;
    }

    // Try to locate it on each valid shader
    if ( !surfaceShaderNode.isNull() )
    {
        if ( locateUvChooserByIdOnShader(surfaceShaderNode, uvChooserId, uvChooserOut) )
        {
            return true;    // Found!
        }
    }

    if ( !displacementShaderNode.isNull() )
    {
        if ( locateUvChooserByIdOnShader(displacementShaderNode, uvChooserId, uvChooserOut) )
        {
            return true;    // Found!
        }
    }

    // Not found. Should not happen!
    return false;
}


//
//  Locate the given uvSet plug
//
static hkBool getUvSetPlugIndex(int uvSetIndex, MObject& meshObject, int& wantedPlugIndexOut)
{
    MFnMesh fnMesh(meshObject);
    MFnDependencyNode depFnMesh(meshObject);

    // Get the array of uv sets from the imported mesh
    MStatus status;
    MStringArray setNames;
    status = fnMesh.getUVSetNames(setNames);
    if ( (status != MStatus::kSuccess) || (uvSetIndex < 0) || (uvSetIndex >= (int)setNames.length()) )
    {
        return false;   // Could not retrieve uv set name
    }
    const char* wantedUvSetName = setNames[uvSetIndex].asChar();

    // Get the plug for the imported mesh uvSet
    MPlug uvSetPlugs = depFnMesh.findPlug("uvSet");
    if( uvSetPlugs.isNull() )
    {
        return false;   // Could not locate uvSet plug
    }

    // Compute the number of elements in the uvSet
    int numUvSetPlugs = uvSetPlugs.evaluateNumElements();
    for (int pi = 0; pi < numUvSetPlugs; pi++)
    {
        // Get .uvSet[i]
        MPlug uvSetPlug = uvSetPlugs[pi];

        // Get .uvSet[i].uvSetName
        MPlug uvSetNamePlug = uvSetPlug.child(0);

        // Check if the value of this plug matches the uvSet name we are looking for
        MString mstrCurrentUvSetName;
        uvSetNamePlug.getValue(mstrCurrentUvSetName);

        const char* currentUvSetName = mstrCurrentUvSetName.asChar();
        if ( hkString::strCmp(currentUvSetName, wantedUvSetName) == 0 )
        {
            // We found our plug
            wantedPlugIndexOut = pi;
            return true;
        }
    }

    // Not found!
    return false;
}


//
//  Attempts to bind the material texture to the proper UV set
//
static hkBool bindToProperUVSet(hkxScene* sceneGraph, MObject& parentMesh, MObject& importedMesh, MObject& shadingEngine, const char* materialName)
{
    // Ensure we have a valid material name
    if ( !materialName )
    {
        return false;   // No material name, no way we can match the hkxMaterial in the scene!
    }

    // Locate the hkxMaterial in the sceneGraph by name
    hkxMaterial* parentMtl = HK_NULL;
    for (int mi = sceneGraph->m_materials.getSize() - 1; mi >= 0; mi--)
    {
        // Get material with valid name
        hkxMaterial* mtl = sceneGraph->m_materials[mi];
        if ( !mtl || !mtl->m_name )
        {
            continue;
        }

        // Check if it matches our material name
        if ( mtl->m_name.compareTo(materialName) == 0 )
        {
            parentMtl = mtl;
            break;  // Found the material!
        }
    }
    if ( !parentMtl )
    {
        return false;
    }

    // Get mesh functions for the imported mesh
//  MFnMesh fnImportedMesh(importedMesh);
    MFnDependencyNode depFnImportedMesh(importedMesh);

    // Retrieve the uvChooser Ids if any
    const int numTexStages = parentMtl->m_stages.getSize();
    for (int ti = 0; ti < numTexStages; ti++)
    {
        const int propKey = (hkxMaterial::PROPERTY_MTL_UV_ID_STAGE0 + ti);
        int uvChooserId = -1;
        if ( parentMtl->hasProperty(propKey) )
        {
            uvChooserId = (int)parentMtl->getProperty(propKey);
        }

        if ( uvChooserId < 0 )
        {
            continue;   // Nothing to do, no uvChooser Id specified
        }

        // Locate the uvChooser by Id, from the parent mesh
        MObject uvChooserObject;
        if ( !locateUvChooserById(shadingEngine, uvChooserId, uvChooserObject) )
        {
            continue;   // Failed to locate the uvChooser
        }
        if ( uvChooserObject.isNull() )
        {
            continue;   // Returned uvChooser is invalid!
        }

        // Locate the imported mesh uvSet
        const int uvSetIndex = parentMtl->m_stages[ti].m_tcoordChannel;
        int meshUvSetPlugIndex = -1;
        if ( !getUvSetPlugIndex(uvSetIndex, importedMesh, meshUvSetPlugIndex) )
        {
            continue;
        }

        // Finally, bind the two plugs together
        {
            // Get the uvChooser plug for uvSets
            MFnDependencyNode fnUvChooser(uvChooserObject);
            MStatus status;
            MPlug uvChooserSetsPlug = fnUvChooser.findPlug("uvSets", status);
            if ( status != MStatus::kSuccess )
            {
                continue;
            }

            // Get the number of uvSets in the uvChooser
            int numUvSetsInChooser = uvChooserSetsPlug.evaluateNumElements();
            MString mstrUvChooserName = fnUvChooser.name();
            const char* uvChooserName = mstrUvChooserName.asChar();

            // Get the mesh full DAG path name
            MFnDagNode fnMeshDag(importedMesh);
            MDagPath meshDagPath;
            fnMeshDag.getPath(meshDagPath);
            MString mstrMeshName = meshDagPath.fullPathName();
            const char* meshName = mstrMeshName.asChar();

            // Create MEL command
            hkStringBuf strBuf;
            strBuf.printf("connectAttr \"%s.uvst[%d].uvsn\" \"%s.uvs[%d]\"", meshName, meshUvSetPlugIndex, uvChooserName, numUvSetsInChooser);

            // Run MEL command
            MString commandString (strBuf.cString());
            status = MGlobal::executeCommand(commandString);
            if ( status != MStatus::kSuccess )
            {
                HK_WARN_ALWAYS(0x7f59bf3c, "Havok Maya Mesh Builder - Error linking mesh uvSet to uvChooser. Maya report : " << status.errorString().asChar());
            }
        }
    }

    // No error
    return true;
}


void hctMayaMeshBuilder::setTriangleMaterialImmediate(int triangleIndex, const char* materialName)
{
    // Only one entry in the array: the triangle's global index.
    MIntArray triangleIndicesArray(1);
    triangleIndicesArray[0] = triangleIndex;

    MFnSingleIndexedComponent triangleIndicesFn;
    triangleIndicesFn.create(MFn::kMeshPolygonComponent);
    triangleIndicesFn.addElements(triangleIndicesArray);

    MSelectionList triangleIndicesList;
    {
        MDagPath generatedMeshPath;
        m_meshFn.getPath(generatedMeshPath);
        triangleIndicesList.add(generatedMeshPath, triangleIndicesFn.object());
    }

    MObject shadingGroup;
    {
        MString shadingGroupName("initialShadingGroup");
        if ( materialName )
        {
            shadingGroupName = materialName;
        }

        MSelectionList shadingGroupSelection;
        shadingGroupSelection.add(shadingGroupName);
        shadingGroupSelection.getDependNode(0, shadingGroup);
    }

    HK_ASSERT_NO_MSG(0xaf4ee132, shadingGroup != MObject::kNullObj);

    // Assign Shading Group to all triangles in current section.
    MFnSet shadingGroupFn(shadingGroup);
    shadingGroupFn.addMembers(triangleIndicesList);

    // Link material with the proper uvSet if case
    MObject mesh = m_meshFn.object();
    bindToProperUVSet(m_sceneGraph, m_parentNode, mesh, shadingGroup, materialName);
}


void hctMayaMeshBuilder::setTriangleMaterialDelayed (int triangleIndex, const char* materialName)
{
    if ( materialName == HK_NULL )
    {
        materialName = "initialShadingGroup";
    }

    MaterialMap* materialMapEntry = HK_NULL;
    for (int i = 0; i < m_materialMap.getSize(); i++)
    {
        if ( hkString::strCasecmp(m_materialMap[i].m_materialName, materialName) == 0 )
        {
            materialMapEntry = &m_materialMap[i];
            break;
        }
    }

    if ( materialMapEntry == HK_NULL )
    {
        MaterialMap newMaterialMapEntry;
        newMaterialMapEntry.m_materialName = materialName;
        newMaterialMapEntry.m_triangleIndicesArray.append(triangleIndex);
        m_materialMap.pushBack(newMaterialMapEntry);
    }
    else
    {
        materialMapEntry->m_triangleIndicesArray.append(triangleIndex);
    }
}


void hctMayaMeshBuilder::realizeDelayedTriangleMaterials()
{
    for (int i = 0; i < m_materialMap.getSize(); i++)
    {
        MaterialMap& materialMapEntry = m_materialMap[i];

        MFnSingleIndexedComponent triangleIndicesFn;
        triangleIndicesFn.create(MFn::kMeshPolygonComponent);
        triangleIndicesFn.addElements(materialMapEntry.m_triangleIndicesArray);

        MSelectionList triangleIndicesList;
        {
            MDagPath generatedMeshPath;
            m_meshFn.getPath(generatedMeshPath);
            triangleIndicesList.add(generatedMeshPath, triangleIndicesFn.object());
        }

        MObject shadingGroup;
        bool found = false;
        {
            // This is either a surface of displacement shader node
            MString surfaceShaderName = materialMapEntry.m_materialName;

            MObject surfaceShaderNode;
            MSelectionList surfaceShaderSelection;
            surfaceShaderSelection.add(surfaceShaderName);
            surfaceShaderSelection.getDependNode(0, surfaceShaderNode);

            // Find the shading engine
            MObject shadingGroupNode;
            MFnDependencyNode surfaceShaderFn( surfaceShaderNode );
            {
                MPlugArray plugArray;
                surfaceShaderFn.getConnections(plugArray);

                for (int plugIndex = 0; (plugIndex < (int)plugArray.length()) && !found; plugIndex++)
                {
                    const MPlug& plug = plugArray[plugIndex];

                    // Check incoming plugs and locate a shading engine
                    MPlugArray incomingPlugs;
                    plug.connectedTo(incomingPlugs, false, true);
                    for (int j = 0; j < (int)incomingPlugs.length(); j++)
                    {
                        MFnDependencyNode incomingNodeFn(incomingPlugs[j].node());
                        const MString incomingNodeType = incomingNodeFn.typeName();
                        const MString shadingEngine("shadingEngine");
                        if ( incomingNodeType == shadingEngine )
                        {
                            shadingGroup = incomingPlugs[j].node();
                            found = true;
                            break;
                        }
                    }
                }
            }
        }

        if ( !found )
        {
            HK_WARN_ALWAYS(0xabba578e, "Failed to locate a shading engine for " << materialMapEntry.m_materialName);
            return;
        }

        // Assign Shading Group to all triangles in current section.
        MFnSet shadingGroupFn(shadingGroup);
        shadingGroupFn.addMembers(triangleIndicesList);

        // Link material with the proper uvSet if case
        MObject mesh = m_meshFn.object();
        bindToProperUVSet(m_sceneGraph, m_parentNode, mesh, shadingGroup, materialMapEntry.m_materialName);
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
