// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Exporter/hctMayaSceneExporter.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>

#include <Common/SceneData/Skin/hkxSkinBinding.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>

#include <ContentTools/Common/Filters/FilterManager/hctFilterManagerImpl.h>

// Loading the filter manager
bool hctMayaSceneExporter::loadFilterManager()
{
    MString filterManagerPath;
    if ( hctMayaUtilities::getFilterManagerPath( m_pluginPath.asChar(), filterManagerPath ) )
    {
        return m_filters.load( filterManagerPath.asChar() );
    }
    else
    {
        return m_filters.load( HK_NULL );
    }
}


bool hctMayaSceneExporter::loadFilterSet()
{
    MStatus status;

    // Create an empty array for the option data
    hkArray<char> optionData;
    hkBool found = false;

    // Check if the user specified an options file to use
    if( m_options.m_hkoFile.length() > 0 )
    {
        // Check if it is valid
        if( hkLoadUtil(m_options.m_hkoFile.asChar()).toArray(optionData) )
        {
            found = true;
        }
        else
        {
            // Asset filename
            hkStringBuf assetDir;
            {
                MString assetFilePath;
                status = MGlobal::executeCommand( "file -q -sceneName", assetFilePath );
                if( status == MStatus::kSuccess && assetFilePath.length() > 0 )
                {
                    assetDir = assetFilePath.asChar();
                    assetDir.replace( '/', '\\' );
                    assetDir.pathDirname();
                }
            }

            hkStringBuf searchloc;
            searchloc.printf("%s/%s", assetDir.cString(), m_options.m_hkoFile.asChar());
            if( hkLoadUtil(searchloc.cString()).toArray(optionData))
            {
                found = true;
            }
            else
            {
                HK_ERROR(0x444ecd41, "The specified .hko options file could not be found. Export cancelled." );
                return false;
            }
        }
    }

    // Check for a Havok options node in the Maya scene
    if( !found )
    {
        // Iterate through the DAG.
        MItDependencyNodes depIterator( MFn::kInvalid, &status );
        if( status != MStatus::kSuccess )
        {
            status.perror("MItDag constructor error.");
        }
        for( ; !depIterator.isDone(); depIterator.next() )
        {
            // Check the node type
            MObject depNodeObject = depIterator.item();
            MFnDependencyNode depNodeFn( depNodeObject );
            if( depNodeFn.typeId() == hkNodeOptionsID )
            {
                // Found the options node. Read the 'filter setup' attribute.
                MPlug filterPlug = depNodeFn.findPlug( "FilterSetup", &status );
                if (status != MStatus::kSuccess)
                {
                    status.perror( "Error obtaining filterSetup plug." );
                }

                // Copy the option data from the node
                MObject filterSetOptionsObj;
                filterPlug.getValue( filterSetOptionsObj );
                MFnIntArrayData intArrayFn( filterSetOptionsObj, &status );
                if( status != MStatus::kSuccess )
                {
                    status.perror( "Error getting node array data." );
                }
                MIntArray intArray = intArrayFn.array();
                optionData.setSize( intArray.length() );
                for( unsigned int i=0; i < intArray.length(); ++i )
                {
                    optionData[i] = (char)intArray[i];
                }

                found = true;
                break;
            }
        }
    }

    // Check for a filter setup stored in the registry
    if( !found )
    {
        HKEY animReg;
        DWORD dispos;

        RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

        // Read the buffer size first
        DWORD bufSize = 128*1024;
        RegQueryValueEx( animReg, TEXT("FilterSetup"), NULL, NULL, NULL, &bufSize );

        hkArray<char> optionsBuf;
        optionsBuf.setSize(bufSize);
        if ( RegQueryValueEx( animReg, TEXT("FilterSetup"), NULL, NULL, (LPBYTE)optionsBuf.begin(), &bufSize ) == ERROR_SUCCESS )
        {
            optionData.setSize( bufSize );
            hkString::memCpy( optionData.begin(), optionsBuf.begin(), bufSize );
            found = true;
        }
        else
        {
            HK_WARN_ALWAYS(0x444ecd41, "Could not read filter options from registry." );
        }

        RegCloseKey( animReg );
    }

    // Load options into the filter manager
    m_filters.setOptions( optionData.begin(), optionData.getSize() );

    if (m_options.m_configurationSet != "")
    {
        hctFilterManagerImpl *manager = (hctFilterManagerImpl *)((hctFilterManagerInterface *)m_filters.getFilterManagerInterface());
        if (manager != NULL && manager->m_configurationSet != NULL)
        {
            for (int curConfig = 0; curConfig < manager->m_configurationSet->m_configurations.getSize(); ++curConfig)
            {
                const hctFilterConfigurationSet::Configuration& config = manager->m_configurationSet->m_configurations[curConfig];
                if (m_options.m_configurationSet == config.m_configName.cString())
                {
                    m_configurationSetIndex = curConfig;
                }
            }
        }
    }

    return ( status == MStatus::kSuccess );
}


void hctMayaSceneExporter::saveFilterSet()
{
    // Get the new option data from the filter manager
    hkArray<char> options;
    int optionSize = m_filters.getOptionsSize();
    options.setSize( optionSize );
    if( optionSize > 0 )
    {
        m_filters.getOptions( options.begin() );
    }

    // Save the option data to the Havok options node
    {
        MObject optionsNode;

        // Iterate through the DAG.
        MStatus status;
        MItDependencyNodes depIterator( MFn::kInvalid, &status );
        if( status != MStatus::kSuccess )
        {
            status.perror( "MItDag constructor error." );
        }
        for( ; !depIterator.isDone(); depIterator.next() )
        {
            // Check the node type
            MObject depNodeObject = depIterator.item();
            MFnDependencyNode depNodeFn( depNodeObject );
            if( depNodeFn.typeId() == hkNodeOptionsID )
            {
                optionsNode = depNodeObject;
                break;
            }
        }

        // Create a new options node if necessary
        MFnDependencyNode optionsNodeFn( optionsNode );
        if( optionsNode.isNull() )
        {
            optionsNode = optionsNodeFn.create( hkNodeOptionsID, &status );
            if( status != MStatus::kSuccess )
            {
                status.perror( "Error creating hkNodeOptions node" );
            }
        }

        // Get the filter setup plug
        MObject filterSetupObj;
        status = optionsNodeFn.findPlug( "FilterSetup" ).getValue( filterSetupObj );
        if( status != MStatus::kSuccess )
        {
            status.perror( "Error obtaining filterSetup plug." );
        }

        // Set the data from the options
        MFnIntArrayData mayaDataFn( filterSetupObj, &status );
        if( status != MStatus::kSuccess )
        {
            status.perror( "Error getting filterSetup function set." );
        }
        MIntArray mayaData = mayaDataFn.array();
        mayaData.setLength( options.getSize() );
        for( int b=0; b<options.getSize(); ++b )
        {
            mayaData[b] = options[b];
        }
    }

    // Save the option data to the registry
    {
        HKEY animReg;
        DWORD dispos;
        if( RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL,
            REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos) == ERROR_SUCCESS )
        {
            RegSetValueEx( animReg, TEXT("FilterSetup"), 0, REG_BINARY, LPBYTE( options.begin() ), optionSize );
        }
        RegCloseKey(animReg);
    }
}



// Get a DAG node's transform matrix taking account of the optional rotation pivot(s)
MStatus hctMayaSceneExporter::getLocalPivotTransform( const MDagPath& path, MMatrix& out )
{
    MStatus status;

    // Begin as regular node transform (without pivot info)
    MFnTransform transformFn( path, &status );
    if( status != MStatus::kSuccess )
    {
        status.perror( "getLocalPivotTransform() called on a non-transform node" );
        return status;
    }
    out = transformFn.transformation().asMatrix();

    // We may have to include the rotate pivots
    if( m_options.m_useRotatePivot )
    {
        // Premultiply this node's pivot offset
        MPoint pivot = transformFn.rotatePivot( MSpace::kPreTransform );
        MMatrix pivotMatrix;
        pivotMatrix(3,0) = pivot.x;
        pivotMatrix(3,1) = pivot.y;
        pivotMatrix(3,2) = pivot.z;
        out = pivotMatrix * out;

        // Postmultiply parent node's inverse pivot offset, if there is one
        MDagPath parentPath = path;
        parentPath.pop();
        MFnTransform parentTransformFn( parentPath, &status );
        if( status == MStatus::kSuccess )
        {
            pivot = parentTransformFn.rotatePivot( MSpace::kPreTransform );
            pivotMatrix(3,0) = pivot.x;
            pivotMatrix(3,1) = pivot.y;
            pivotMatrix(3,2) = pivot.z;
            out = out * pivotMatrix.inverse();
        }
        else
        {
            return status;
        }
    }
    return MStatus::kSuccess;
}


MStatus hctMayaSceneExporter::findBindPoses( MFnSkinCluster& skin, hkArray<MDagPath>& dagPaths, hkArray<MMatrix>& bindPoses )
{
    MStatus status;

    // for each influence object extract the bind pose
    MDagPathArray dagPathArray;
    unsigned int numInfluenceObjects = skin.influenceObjects( dagPathArray, &status );

    if( status == MStatus::kFailure )
    {
        status.perror( "Failed to retrieve influence objects!" );
        return MStatus::kFailure;
    }

    for( unsigned int i = 0; i < numInfluenceObjects; i++ )
    {
        // get the index for this joint
        unsigned int influenceIndex = skin.indexForInfluenceObject( dagPathArray[i], &status );
        if( status == MStatus::kFailure )
        {
            status.perror( "Failed to retrieve joint influence index!" );
        }

        // find the 'bindPreMatrix' plug
        MPlug bindPreMatrixPlug = skin.findPlug( "bindPreMatrix", &status );
        if( status == MStatus::kFailure )
        {
            status.perror( "Failed to retrieve 'bindPreMatrix' plug!" );
        }

        // find the correct plug for this joint
        MPlug jointBindPreMatrixPlug = bindPreMatrixPlug.elementByLogicalIndex( influenceIndex, &status );
        if( status == MStatus::kFailure )
        {
            status.perror( "Failed to retrieve joint's specific 'bindPreMatrix' plug!" );
        }

        // retrieve the value
        MObject matrixData;
        status = jointBindPreMatrixPlug.getValue( matrixData );
        if( status == MStatus::kFailure )
        {
            status.perror( "Failed to retrieve matrixData from 'jointBindPreMatrixPlug'!" );
        }

        // retrieve actual matrix
        MFnMatrixData matrixDataFn( matrixData, &status );
        if( status == MStatus::kFailure )
        {
            status.perror( "Failed to create matrix function set!" );
        }

        // set the world transform to be the inverse
        MMatrix worldTransform = matrixDataFn.matrix().inverse();

        // store the dagpath and the bind pose
        dagPaths.pushBack( dagPathArray[i] );
        bindPoses.pushBack( worldTransform );
    }

    return MStatus::kSuccess;
}

MStatus hctMayaSceneExporter::findBindPosesGeneric( MFnGeometryFilter& skin, hkArray<MDagPath>& dagPaths, hkArray<MMatrix>& bindPoses )
{
    MObject skinObject = skin.object();
    for(MItDependencyGraph jointIt(skinObject, MFn::kJoint, MItDependencyGraph::kUpstream); !jointIt.isDone(); jointIt.next())
    {
        // Add first upstream joint (only one that is attached to this cluster).
        MDagPath dagPath;
        MStatus status;
        if(MFnDagNode(jointIt.thisNode()).getPath(dagPath) == MS::kSuccess)
        {
            MFnIkJoint fnJoint(dagPath);
            MPlug bindPoseMatrixPlug = fnJoint.findPlug("bindPose", &status);
            MObject matrixObject;
            if(status == MStatus::kSuccess)
            {
                status = bindPoseMatrixPlug.getValue(matrixObject);
            }

            // In the worst case, fall back to just grabbing the current world matrix.
            if(status != MStatus::kSuccess)
            {
                MPlug jointWorldMatrixPlug = fnJoint.findPlug("worldMatrix").elementByLogicalIndex(0);
                jointWorldMatrixPlug.getValue(matrixObject);
            }

            if(matrixObject != MObject::kNullObj)
            {
                dagPaths.pushBack(dagPath);
                bindPoses.pushBack(MFnMatrixData(matrixObject).matrix());
            }
        }
    }

    return MStatus::kSuccess;
    }


void hctMayaSceneExporter::verifyScene()
{
    // Verify skins
    int processedSkinBindingsCount = 0;
    for( int skinBindingIndex = 0; skinBindingIndex < m_currentScene.m_skinBindings.getSize(); ++skinBindingIndex )
    {
        hkxSkinBinding* curSkin = m_currentScene.m_skinBindings[skinBindingIndex];
        processedSkinBindingsCount++;

        for( int mappingIndex = 0; mappingIndex < curSkin->m_nodeNames.getSize(); ++mappingIndex )
        {
            hkxNode* node = m_currentScene.m_rootNode->findDescendantByName( curSkin->m_nodeNames[mappingIndex] );
            if( !node )
            {
                // Remove any references to the soon-to-be-removed skin binding node from the nodes in the scene.
                m_currentScene.m_rootNode->replaceAllObjects(m_currentScene.m_skinBindings[skinBindingIndex], curSkin->m_mesh);

                // Remove the skin binding, because not all of the necessary bones are in the exported scene.
                // The mesh will still be exported, but will not be skinned to the skeleton.
                m_currentScene.m_skinBindings.removeAtAndCopy( skinBindingIndex );
                --skinBindingIndex;

                HK_WARN_ALWAYS( 0xabba6655, "Skin binding " << processedSkinBindingsCount << " will not be created because not all of the required bones have been included for export." );
                break;
            }
        }
    }
}


//
// Maya Node <-> HKX node mappings
//

MDagPath hctMayaSceneExporter::getMayaNode( const hkxNode* node )
{
    int numNodes = m_exportMappings.getSize();
    for( int i=0; i<numNodes; ++i )
    {
        if( m_exportMappings[i].m_hkxNode == node )
        {
            return m_exportMappings[i].m_node;
        }
    }
    return MDagPath();
}

const hkxNode* hctMayaSceneExporter::getHkxNode( const MDagPath& path ) const
{
    int numNodes = m_exportMappings.getSize();
    for( int i=0; i<numNodes; ++i )
    {
        if( m_exportMappings[i].m_node == path)
        {
            return m_exportMappings[i].m_hkxNode;
        }
    }
    return HK_NULL;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
