// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Exporter/hctMayaSceneExporter.h>

#include <maya/MFnCamera.h>
#include <maya/MFnSpotLight.h>
#include <maya/MFNNurbsCurve.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>

// EXP-862
hkxMesh* hctMayaSceneExporter::lookForAlreadyExportedMesh (const MDagPath& shapeDagPath)
{
    // We only do this for instanced shapes - ignore otherwise
    if (!shapeDagPath.isInstanced())
    {
        return HK_NULL;
    }

    MPoint pivotOffset (0.0f,0.0f,0.0f);

    // may need to make them relative to the transform's rotate pivot
    if( m_options.m_useRotatePivot )
    {
        MStatus status;
        MDagPath transformDagPath = shapeDagPath; transformDagPath.pop();
        MFnTransform transformFn( transformDagPath, &status );
        if( status == MStatus::kSuccess )
        {
            pivotOffset = transformFn.rotatePivot( MSpace::kPreTransform );
        }
    }

    MObject obj = shapeDagPath.node();

    for (int i=0; i<m_exportedMeshesInstanceData.getSize(); i++)
    {
        if ( (obj == m_exportedMeshesInstanceData[i].m_mayaShape) &&
            (pivotOffset== m_exportedMeshesInstanceData[i].m_pivotOffset)
            )
        {
            return m_exportedMeshesInstanceData[i].m_havokMesh;
        }
    }

    return HK_NULL;
}

// EXP-862
void hctMayaSceneExporter::registerExportedMesh (const MDagPath& shapeDagPath, hkxMesh* mesh)
{
    // We only do this for instanced shapes - ignore otherwise
    if (!shapeDagPath.isInstanced())
    {
        return;
    }

    MPoint pivotOffset (0.0f,0.0f,0.0f);

    // may need to make them relative to the transform's rotate pivot
    if( m_options.m_useRotatePivot )
    {
        MStatus status;
        MDagPath transformDagPath = shapeDagPath; transformDagPath.pop();
        MFnTransform transformFn( transformDagPath, &status );
        if( status == MStatus::kSuccess )
        {
            pivotOffset = transformFn.rotatePivot( MSpace::kPreTransform );
        }
    }

    MObject obj = shapeDagPath.node();

    ExportedMeshInstanceData exportedMeshData;
    exportedMeshData.m_havokMesh = mesh;
    exportedMeshData.m_mayaShape = obj;
    exportedMeshData.m_pivotOffset = pivotOffset;

    m_exportedMeshesInstanceData.pushBack(exportedMeshData);
}

// Mesh - use the meshWriter
MStatus hctMayaSceneExporter::processMesh( const MDagPath& shapeDagPath, hkRefVariant& variant, bool forceSkinned )
{
    MStatus status;

    // EXP-862 : See if we exported it already
    {
        hkxMesh* alreadyExported = lookForAlreadyExportedMesh(shapeDagPath);

        if (alreadyExported)
        {
            // just a mesh
            variant = alreadyExported;
            return MStatus::kSuccess;
        }
    }

    // create mesh/skin objects using the meshwriter
    hkxMesh* newMesh = HK_NULL;
    hkxSkinBinding* newSkin = HK_NULL;
    status = m_meshWriter.write( shapeDagPath, newMesh, newSkin, forceSkinned  );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Error writing with meshWriter!" );
        return status;
    }

    // add to scene
    if( newSkin )
    {
        // skinned mesh
        variant = newSkin;
        m_currentScene.m_skinBindings.pushBack(newSkin);
        m_currentScene.m_meshes.pushBack(newMesh);

        newSkin->removeReference();
        newMesh->removeReference();

    }
    else if( newMesh )
    {
        // just a mesh
        variant = newMesh;
        m_currentScene.m_meshes.pushBack(newMesh);
        // EXP-862 : register the mesh in case it's instanced
        registerExportedMesh(shapeDagPath, newMesh);

        newMesh->removeReference();
    }

    return MStatus::kSuccess;
}

MStatus hctMayaSceneExporter::processNurbs( const MDagPath& nurbsDagPath, hkRefVariant& variant, bool forceSkinned )
{
    // Save the selection list so that we can restore it later
    MSelectionList selection;
    MGlobal::getActiveSelectionList( selection );

    //
    // We use the nurbsToPoly command since it handles connections to materials, etc..
    // We could use MFnNurbsSurface::tesselate() but then we'd lose those connections
    //

    MDagPath convertedNodePath;
    MDagPath convertedMeshPath;
    {
        // EXP-1147 : NurbsToPoly will collapse the transform into the converted polygon. We don't want that, so we start by
        // instancing the nurbs in an "identity" transform node
        MString identityNodeName;
        {
            // Create identity transform node
            MStringArray resultArray;
            MString cmd = "createNode transform";
            MStatus status = MGlobal::executeCommand(cmd, resultArray, false, false);

            identityNodeName = resultArray[0];
        }

        // instance the nurbs into that node
        MDagPath newNurbsPath;
        {
            MStringArray names;
            MString nurbsName = nurbsDagPath.fullPathName();
            MString cmd = "parent -add -s "+nurbsName+" "+identityNodeName+";";

            MStatus status = MGlobal::executeCommand(cmd, names, false, false);

            MString reparentedName = names[0];
            hctMayaSceneExportUtilities::nameToDagPath(reparentedName, newNurbsPath);
        }

        // Convert it to a mesh
        {
            MStringArray resultArray;
            MString cmd = "nurbsToPoly -chr 0.9 -uss 1 "+newNurbsPath.fullPathName()+";" ;
            MStatus status = MGlobal::executeCommand( cmd, resultArray, false, false );

            MString convertedNodeName = resultArray[0];
            hctMayaSceneExportUtilities::nameToDagPath(convertedNodeName, convertedNodePath);

            // We  safely assume the first child of the new transform node is the converted mesh
            convertedMeshPath = convertedNodePath;
            convertedMeshPath.push(convertedNodePath.child(0));
        }

        // Remove the identity node
        {
            MString cmd = "delete "+identityNodeName+";";
            MCommandResult res;
            MStatus status = MGlobal::executeCommand(cmd, res, false, false);
        }
    }

    // Reparent the mesh to the original node
    // We do this in order to use the same node transform (pivot, etc) during export
    {
        MDagPath nurbsNodePath = nurbsDagPath; nurbsNodePath.pop();
        MString nurbsNodeName = nurbsNodePath.fullPathName();
        MString convertedMeshName = convertedMeshPath.fullPathName();

        MStringArray names;
        MString cmd = "parent -add -s "+convertedMeshName+" "+nurbsNodeName+";";

        MStatus status = MGlobal::executeCommand(cmd, names, false, false);

        // The path has changed - update it
        MString reparentedName = names[0];
        hctMayaSceneExportUtilities::nameToDagPath(reparentedName, convertedMeshPath);
    }

    // Remove the extra node
    {
        MString convertedNodeName = convertedNodePath.fullPathName();
        MString cmd = "delete "+convertedNodeName+";";
        MCommandResult res;
        MStatus status = MGlobal::executeCommand(cmd, res, false, false);
    }

    // Export
    MStatus mwStatus = processMesh(convertedMeshPath, variant, forceSkinned );

    // Remove the extra mesh we created
    {
        MString convertedMeshName = convertedMeshPath.fullPathName();
        MString cmd = "delete "+convertedMeshName+";";
        MCommandResult res;
        MStatus status = MGlobal::executeCommand(cmd, res, false, false);
    }

    // Restore the original selection list
    MGlobal::setActiveSelectionList( selection );

    return MStatus::kSuccess;
}



// Lights - straightforward
hkxLight* hctMayaSceneExporter::processLight( const MDagPath& dagPath, MStatus& status )
{
    // check if the objects supports light function sets
    if( !dagPath.hasFn( MFn::kLight ) )
    {
        // dagPath does not represent a light
        status.perror("The 'dagPath' is not compatible with MFn::kLight!");
        status = MStatus::kFailure;
        return HK_NULL;
    }

    // create a new hkxLight node
    hkxLight* newLight = new hkxLight();

    MFnLight light (dagPath, &status);
    if ( !status )
    {
        status.perror("MFnLight constructor");
        status = MStatus::kFailure;
        return HK_NULL;
    }

    MObject transformNode = dagPath.transform( &status );

    // This node has no transform .. (has to though as a light)
    if( status == MStatus::kInvalidParameter )
    {
        status = MStatus::kFailure;
        return HK_NULL;
    }

    MFnDagNode  transform( transformNode, &status );
    if (!status)
    {
        status.perror("MFnDagNode constructor");
        status = MStatus::kFailure;
        return HK_NULL;
    }

    // get the position
    const MMatrix& matrix = transform.transformationMatrix();
    const double* pos = matrix[3]; // row 3 == trans
    newLight->m_position.set( (hkReal)pos[0], (hkReal)pos[1], (hkReal)pos[2] );

    // get the direction
    MFloatVector dir = light.lightDirection(0, MSpace::kWorld);
    dir.normalize();
    newLight->m_direction.set( dir[0], dir[1], dir[2] );

    // get the colour
    MColor color = light.color();
    newLight->m_color = hkSceneExportUtils::floatsToARGB( color.r, color.g, color.b, color.a );

    // is the light a spot light or directional light?
    if ( dagPath.hasFn( MFn::kSpotLight ))
    {
        MFnSpotLight spotLight( dagPath, &status);
        if (status)
        {
            newLight->m_innerAngle = static_cast<hkReal>(spotLight.coneAngle() * 0.75);
            newLight->m_outerAngle = static_cast<hkReal>(spotLight.coneAngle());
            newLight->m_type = hkxLight::SPOT_LIGHT;
        }
    }
    else if ( dagPath.hasFn( MFn::kDirectionalLight ) )
    {
        newLight->m_type = hkxLight::DIRECTIONAL_LIGHT;
    }
    else if ( dagPath.hasFn( MFn::kPointLight ) )
    {
        newLight->m_type = hkxLight::POINT_LIGHT;
    }
    else
    {
        char buf[200];
        hkString::sprintf( buf, "Lights of type %s are not supported", dagPath.node().apiTypeStr() );
        status.perror( buf );
        status = MStatus::kFailure;
        return HK_NULL;
    }

    MFnNonAmbientLight fnNonAmbientLight(dagPath, &status);

    if (status == MS::kSuccess)
    {
        newLight->m_intensity = light.intensity();

        newLight->m_decayRate = (hkInt16)fnNonAmbientLight.decayRate();

        if (newLight->m_type == hkxLight::POINT_LIGHT)
        {
            if (newLight->m_decayRate)
            {
                float cutOff = 0.01f;

                // calculate range of new light
                newLight->m_range = hkMath::pow((hkReal)(newLight->m_intensity / cutOff), (hkReal)(1.f / newLight->m_decayRate));

                newLight->m_fadeStart = newLight->m_range * 2;
                newLight->m_fadeEnd = newLight->m_range * 3;
            }
            else
            {
                // no decay
                char buf[200];
                hkString::sprintf(buf, "Point lights with no decay are not supported. Please use a directional light, or ambient lighting instead.");
                status.perror(buf);
                status = MStatus::kFailure;
                return HK_NULL;
            }
        }
    }
    else
    {
        // creation of non ambient light failed
        status.perror("Creation of non ambient light failed");
    }

    newLight->m_shadowCaster = true; // not set in vMaya2..

    // pushBack the light
    m_currentScene.m_lights.pushBack( newLight );

    // return successfully with the light
    status = MStatus::kSuccess;
    return newLight;
}

hkxSpline* hctMayaSceneExporter::processSpline( const MDagPath& dagPath, MStatus& status )
{
    if (!dagPath.hasFn(MFn::kNurbsCurve))
    {
        // dagPath does not represent a camera
        status.perror("The 'dagPath' is not compatible with MFn::kNurbsCurve!");
        status = MStatus::kFailure;
        return HK_NULL;
    }

    MFnNurbsCurve fnNurbsCurve(dagPath, &status);

    hkxSpline* newSpline = new hkxSpline;

    // Try to get bezier curve data out of the fucntion set (there is no specific MFn for bezier curves).
    int nc = fnNurbsCurve.numCVs();
    int nk = fnNurbsCurve.numKnots()/3;
    if(nk*3-2 == nc)  // Number of knots * 3 - 2 is all control points we get from Maya (Maya only stores 1 handle per spline endpoint).
    {
        // Create path, gather info, ...
        MDagPath curvePath;
        fnNurbsCurve.getPath(curvePath);

        newSpline->m_isClosed = (fnNurbsCurve.form() == MFnNurbsCurve::kClosed);

        for(int numCVs =fnNurbsCurve.numCVs(), cvIndex=1; cvIndex<=numCVs; cvIndex+=3)
        {
            MPoint cvPtL, cvPtM, cvPtR;
            fnNurbsCurve.getCV((cvIndex==0  && newSpline->m_isClosed) ? numCVs-2 : hkMath::max2(0,  cvIndex-2), cvPtL);  // If spline is closed, get 'in' from last knot for first i==0.
            fnNurbsCurve.getCV(cvIndex-1,                                              cvPtM);
            fnNurbsCurve.getCV((cvIndex==numCVs && newSpline->m_isClosed) ?    1 : hkMath::min2(cvIndex, numCVs-1), cvPtR);  // If spline is closed, get 'in' from last knot for first i==0.

            hkxSpline::ControlPoint& controlpoint = newSpline->m_controlPoints.expandOne();

            controlpoint.m_tangentIn.set ( (float)cvPtL.x, (float)cvPtL.y, (float)cvPtL.z );
            controlpoint.m_position.set ( (float)cvPtM.x, (float)cvPtM.y, (float)cvPtM.z );
            controlpoint.m_tangentOut.set ( (float)cvPtR.x, (float)cvPtR.y, (float)cvPtR.z );
      controlpoint.m_inType = hkxSpline::CUSTOM; //
      controlpoint.m_outType = hkxSpline::CUSTOM; // Unknown (xx check somehow if it is a bezier )
        }
    }
    else
    {
        // ?
    }

    m_currentScene.m_splines.pushBack( newSpline );

    status = MStatus::kSuccess;

    return newSpline;
}

// Cameras - straightforward
hkxCamera* hctMayaSceneExporter::processCamera( const MDagPath& dagPath, MStatus& status )
{
    // check if the object supports camera function sets
    if( !dagPath.hasFn( MFn::kCamera ) )
    {
        // dagPath does not represent a camera
        status.perror("The 'dagPath' is not compatible with MFn::kCamera!");
        status = MStatus::kFailure;
        return HK_NULL;
    }

    // create a new hkxCamera node
    hkxCamera* newCamera = new hkxCamera();

    MFnCamera camera(dagPath, &status);
    if ( !status )
    {
        status.perror("MFnCamera constructor");
        status = MStatus::kFailure;
        return HK_NULL;
    }

    // store camera properties
    float info[4];

    // from point
    MPoint eye = camera.eyePoint( MSpace::kWorld, &status );
    eye.get( info );
    newCamera->m_from.set( info[0], info[1], info[2] );

    // up direction
    MPoint up = camera.upDirection( MSpace::kWorld, &status );
    up.get( info );
    newCamera->m_up.set( info[0], info[1], info[2] );

    // to point
    MPoint poi = camera.centerOfInterestPoint( MSpace::kWorld, &status );
    poi.get( info );
    newCamera->m_focus.set( info[0], info[1], info[2] );

    // field of view
    newCamera->m_fov = static_cast<hkReal>( camera.horizontalFieldOfView( &status ) );

    // near clip plane
    newCamera->m_near = static_cast<hkReal>( camera.nearClippingPlane( &status ) );

    // far clip plane
    newCamera->m_far = static_cast<hkReal>( camera.farClippingPlane( &status ) );

    // Cameras in maya are right-handed
    newCamera->m_leftHanded = false;

    // pushBack the camera
    m_currentScene.m_cameras.pushBack( newCamera );

    // return successfully the camera
    status = MStatus::kSuccess;
    return newCamera;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
